/* some helper functions for selection lists,
 * (used e.g. by filelist.c) */

#include "int.h"

#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include <string.h>
#include <stdlib.h>

#include "main.h"
#include "preferences.h"
#include "dndsetup.h"
#include "selectlist.h"
#include "doubleclick.h"
#include "menusys.h"
#include "helpings.h"
#include "fileman.h"
#include "tracks.h"
#include "stdfiletrack.h"
#include "streams.h"
#include "vfs.h"
#include "main.h"
#include "clipboard.h"

/* uncomment for debugging */
//#define DEBUG
//
/* non-public callback functions */

const char *selectlist_clipboard_obtain(ObtainContentMode mode, void *data)
{
   const char *result = NULL;
   selectlist_info *info=(selectlist_info*)data;
   if (info)
     result = selectlist_getselection(info);
   return result;
};

int selectlist_clipboard_paste(void *data, const char *pastedata)
{
   int result = 0;
   selectlist_info *info=(selectlist_info*)data;
#ifdef DEBUG
   printf("Received paste request\n");
#endif
   if (info&&info->receiver&&pastedata)
     result = info->receiver(pastedata,0,0,DNDSETUP_LINK, info->callback_data);
   return result;
};

void selectlist_clipboard_activate(selectlist_info *info)
{
   clipboard_handlerinfo cinfo;
   cinfo.obtain  = selectlist_clipboard_obtain;
   cinfo.paste = selectlist_clipboard_paste;
   cinfo.completion = NULL; // we do not care right now
   cinfo.data  = (gpointer)info;
   clipboard_setcurrenthandler(&cinfo);
#ifdef DEBUG
   printf("Set clipboard focus\n");
#endif
};

void selectlist_clipboard_deactivate(selectlist_info *info)
{
   clipboard_handlerinfo cinfo;
   cinfo.obtain  = selectlist_clipboard_obtain;
   cinfo.paste = selectlist_clipboard_paste;
   cinfo.completion = NULL; // we do not care right now
   cinfo.data  = (gpointer)info;
   clipboard_removeifcurrent(&cinfo);
};

char *selectlist_drag(selectlist_info *info)
{
   selectlist_clipboard_activate(info);
   info->is_drag_source=1;
   return selectlist_getselection(info);
};

void selectlist_dragdone(int dragtype,
			 selectlist_info *info)
{
   gtk_clist_unselect_all(info->self);
   info->is_drag_source=0;
}
;

void selectlist_appendselection(char *s,char *h,char *f) // selectionlist,header,file
{
   char *entry;
   char *cmpcopy;

/* #ifdef DEBUG
	printf ("adding selection \"%s%s\"\n to \"%s\"\n",h,f,s);
#endif
 */
   if (f&&h&&s)
     {
	cmpcopy=(char*)malloc(SELECTLIST_MAXSELSIZE);
	strcpy(cmpcopy,CRLF);
	strcat(cmpcopy,s);

	entry=(char*)malloc(1024);
	strcpy(entry,CRLF);
	strcat(entry,h);
	strcat(entry,f);
	strcat(entry,CRLF); // entry=CRLF+header+file+CRLF

	if ((strstr(cmpcopy,entry)==NULL)&&
	    (strlen(h)+strlen(f)+strlen(CRLF)+strlen(s)<SELECTLIST_MAXSELSIZE))
	  {
	     strcat(s,h);
	     strcat(s,f);
	     strcat(s,CRLF);
	  }
	;
	free(entry);free(cmpcopy);
     };
}
;

/* this function is currently unused - but it is a nice piece of code
 * which we may need again later,so let's just keep it for a while */
void selectlist_removeselection(char *s,char *h,char *f)  // see above
{
   char *entry;
   char *cmpcopy;
   char *foundpos;

/*#ifdef DEBUG
	printf ("removing selection %s%s\n",h,f);
#endif
*/
   cmpcopy=(char*)malloc(SELECTLIST_MAXSELSIZE);
   strcpy(cmpcopy,CRLF);
   strcat(cmpcopy,s);

   entry=(char*)malloc(1024);
   strcpy(entry,CRLF);
   strcat(entry,h);
   strcat(entry,f);
   strcat(entry,CRLF); // entry=CRLF+header+file+CRLF

   foundpos=strstr(cmpcopy,entry);
   while (foundpos!=NULL)
     {
	strcpy(foundpos,(char*)((int)foundpos+strlen(entry)-2));
	foundpos=strstr(cmpcopy,entry);
     }
   ;
   strcpy(s,&cmpcopy[2]);        // cut leading cr+lf
   free(entry);free(cmpcopy);
}
;

void selectlist_select(GtkCList *clist,gint row,gint column,
		       GdkEventButton *event, selectlist_info *info)
{
   info->kbsearchinfo[0]=0;

   /* remember this selectlist has been used */
   selectlist_clipboard_activate(info);

   /* only mark row as selected if this has not already happened */
   if (g_list_find(info->selected_lines,(gpointer)row)==NULL)
     info->selected_lines=g_list_prepend(info->selected_lines,(gpointer)row);
#ifdef DEBUG
   printf ("selectlist_select: selected row %i\n",row);
   printf ("selectlist.c: selectlist_select current selection:\n\"%s\"\n",
	   selectlist_getselection(info));
#endif
}
;

void selectlist_unselect(GtkCList *clist,gint row,gint column,
			 GdkEventButton *event, selectlist_info *info)
{
   info->kbsearchinfo[0]=0;

   /* remember this selectlist has been used */
   selectlist_clipboard_activate(info);

#ifdef DEBUG
   printf ("selectlist_unselect: removing row %i\n",
	   row);
#endif
   info->selected_lines=g_list_remove(info->selected_lines,
				      (gpointer)row);
#ifdef DEBUG
   printf ("selectlist.c: selectlist_unselect current selection:\n\"%s\"\n",
	   selectlist_getselection(info));
#endif
}
;

selectlist_info *selectlist_info_create(char *header,int significant_column,
					selectlist_dropcallback receiver,
					int send,menusys_menu *popup,
					selectlist_doubleclick_cb_t doubleclick,
					gpointer callback_data,
					const char *headings[]
					)
{
   selectlist_info *sel;

   sel=(selectlist_info*)malloc(sizeof(selectlist_info));
   sel->selection[0]=0;
   sel->selected_lines=NULL; /* currently,no lines are selected */
   strcpy(sel->header,header);
   sel->significant_column=significant_column;
   sel->receiver=receiver;
   sel->send=send;
   sel->popup=popup;

   sel->doubleclick=(gpointer)doubleclick;
   sel->callback_data=callback_data;
   sel->headings=helpings_translatestringlist(headings);

   sel->kbsearchinfo[0]=0;

   return sel;
}
;

/* destroy the selectlist's info struct */
void selectlist_info_destroy(selectlist_info *info)
{
   int i;

   /* Avoid a system crash if a select list has been destroyed.
    * Reset any references it may still have */
   selectlist_clipboard_deactivate(info);

   /* free all externally allocated data */
   if (info->selected_lines!=NULL)
     g_list_free(info->selected_lines);
   for (i=0;info->headings[i]!=NULL;i++)
     free(info->headings[i]);
   free(info->headings);

   free(info);
};

void selectlist_doubleclick(GtkWidget *w,GdkEventButton *event,
			    selectlist_info *info)
{
   /* react to first mouse button only */
   if ((info->doubleclick!=NULL)&&(event->button==1))
     {
#ifdef DEBUG
	printf ("selectlist_doubleclick: callback() with selection \"%s\"\n",
		selectlist_getselection(info));
#endif
	((selectlist_doubleclick_cb_t)info->doubleclick)(info,info->callback_data);
     }
   ;
}
;

void selectlist_focusin(GtkWidget *w, GtkWidget *child, gpointer data)
{
   selectlist_info *info = (selectlist_info*)data;
   // See if the widget that came into focus is ours
   if (GTK_WIDGET(info->self)==child)
     {
	selectlist_clipboard_activate(info);
#ifdef DEBUG
	printf("selectlist_focusin: got focus widget %p\n", child);
#endif
     };
};

void selectlist_getfocus(GtkWidget *w, GdkEventButton *event, gpointer data)
{
   selectlist_info *info = (selectlist_info*)data;
   selectlist_clipboard_activate(info);
};

void selectlist_map(GtkWidget *w,gpointer data)
{
   if (w->parent)
     {
#ifdef DEBUG
	printf("selectlist_show: connecting to parent, data is %p\n",data);
#endif
	gtk_signal_connect(GTK_OBJECT(w->parent),"set-focus-child",
			   GTK_SIGNAL_FUNC(selectlist_focusin),data);
     };
};

void selectlist_unmap(GtkWidget *w,gpointer data)
{
   if (w->parent)
     {
#ifdef DEBUG
	printf("selectlist_show: disconnecting from parent, data is %p\n",data);
#endif
	gtk_signal_disconnect_by_data(GTK_OBJECT(w->parent),data);
     };
};

// Check if a specific row is selected.
// This is used internally for the new selection scheme
int selectlist_isselected(selectlist_info *info, int row)
{
   GList *i=NULL;
   int result = 0;
   for (i=info->selected_lines;(i!=NULL)&&(!result);i=i->next)
     if (row==(int)i->data)
       result = 1;
   return result;
};

void selectlist_setfocus(selectlist_info *info,gint row)
{
   if (info->self)
     {
	// Ripped from gtk source code,
	// focus the row the user clicked on
	info->self->focus_row = row;
	if (GTK_WIDGET_HAS_FOCUS(GTK_WIDGET(info->self)))
	  {
	     int width =0;
	     int height = 0;
	     GdkRectangle rec;
	     rec.x=0;rec.y=0;
	     gdk_window_get_size(GTK_WIDGET(info->self)->window,&width,&height);
	     rec.width = width; rec.height = height;
	     gtk_widget_draw(GTK_WIDGET(info->self),&rec);
	  };
     };
};

void selectlist_modifiedselect_internal(GtkWidget *widget, int row, gpointer data);
void selectlist_modifiedselect(GtkWidget *widget, GdkEventButton *event, gpointer data)
{
   if (event->button == 1)
     {
	int row = 0;
	int column = 0;
	gtk_clist_get_selection_info(GTK_CLIST(widget),
				     event->x, event->y,
				     &row, &column);

	selectlist_modifiedselect_internal(widget,row,data);
     };
};

void selectlist_modifiedselect_internal(GtkWidget *widget, int row, gpointer data)
{
   selectlist_info *info = (selectlist_info*)data;
   if (info)
     {
	int x = 0;
	int upper_bound = 0;
	int lower_bound = 0;

	int is_selected = 0;

	is_selected = selectlist_isselected(info, row);

	switch (info->modifier)
	  {
	   case None:
	     if (!is_selected)
	       gtk_clist_unselect_all(GTK_CLIST(widget));
	     gtk_clist_select_row(GTK_CLIST(widget), row, 0);
	     break;
	   case Control:
	     if (!is_selected)
	       gtk_clist_select_row(GTK_CLIST(widget), row, 0);
	     else
	       gtk_clist_unselect_row(GTK_CLIST(widget), row, 0);
	     break;
	   case Shift:
	     x = ((info&&info->selected_lines)?(int)info->selected_lines->data:row);
	     upper_bound = MAX(x,row);
	     lower_bound = MIN(x,row);
	     for (x = lower_bound; x<=upper_bound; ++x)
	       gtk_clist_select_row(GTK_CLIST(widget), x, 0);
	     break;
	  };
	selectlist_setfocus(info,row);
     };
};

void selectlist_modifiedselect_setmodifier(selectlist_info *info, int pressed, SelectModifier modifier)
{
   if ((modifier!=None)&&info)
     {
	if (pressed)
	  info->modifier = modifier;
	else
	  if (info->modifier==modifier)
	    info->modifier = None;
     };
};

static SelectModifier GetModifier(int keyval)
{
   SelectModifier key = None;
   switch (keyval)
     {
      case GDK_Shift_L:
      case GDK_Shift_R:
	key = Shift;
	break;
      case GDK_Control_L:
      case GDK_Control_R:
	key = Control;
	break;
     };
   return key;
};

void selectlist_modifiedselect_keyboard_press(GtkWidget *widget, GdkEventKey *event, gpointer data)
{
   selectlist_info *info = (selectlist_info*)data;
   selectlist_modifiedselect_setmodifier(info, 1, GetModifier(event->keyval));
};

void selectlist_modifiedselect_keyboard_release(GtkWidget *widget, GdkEventKey *event, gpointer data)
{
   selectlist_info *info = (selectlist_info*)data;
   selectlist_modifiedselect_setmodifier(info, 0, GetModifier(event->keyval));
};

int selectlist_getrowmatching(selectlist_info *info, const char *name)
{
   int result = -1;
   if (info&&name&&info->self)
     {
	int currentrow = 0;
	for (currentrow = 0;(currentrow<info->self->rows)&&(result==-1);++currentrow)
	  {
	     char *currentname = NULL;
	     int column = info->significant_column;
	     if (column==-1)
	       column = 0;
	     gtk_clist_get_text(info->self, currentrow, column, &currentname);
	     if (currentname&&(!strncasecmp(name,currentname,strlen(name))))
	       result = currentrow;
	  };
     };
   return result;
};

void selectlist_standardkeyboardhandler(GtkWidget *widget, GdkEventKey *event, gpointer data)
{
   selectlist_info *info = (selectlist_info*)data;
   /* ENTER or SPACE will select an entry according to the new selection rules */
   if ((event->keyval==GDK_Return)||(event->keyval==GDK_space))
     selectlist_modifiedselect_internal(widget, info->self->focus_row, data);
   /* Map ENTER to doubleclick */
   if ((info->doubleclick!=NULL)&&(event->keyval==GDK_Return))
     {
#ifdef DEBUG
	printf ("selectlist_doubleclick: callback() with selection \"%s\"\n",
		selectlist_getselection(info));
#endif
	((selectlist_doubleclick_cb_t)info->doubleclick)(info,info->callback_data);
     }
   ;
   if (info->modifier==None)
     {
	if (((event->keyval>=GDK_a)&&(event->keyval<=GDK_z))||
	    ((event->keyval>=GDK_0)&&(event->keyval<=GDK_9))||
	    (event->keyval==GDK_BackSpace))
	  {
	     int row = -1;
	     if (event->keyval!=GDK_BackSpace)
	       {
		  char newchar[2];
		  newchar[1]=0;
		  newchar[0]=0;
		  if ((event->keyval>=GDK_a)&&(event->keyval<=GDK_z))
		    newchar[0] = event->keyval-GDK_a+'a';
		  if ((event->keyval>=GDK_0)&&(event->keyval<=GDK_9))
		    newchar[0] = event->keyval-GDK_0+'0';
		  strcat(info->kbsearchinfo,newchar);
	       }
	     else
	       if (strlen(info->kbsearchinfo))
		 info->kbsearchinfo[strlen(info->kbsearchinfo)-1]=0;

	     row = selectlist_getrowmatching(info, info->kbsearchinfo);
	     if (row!=-1)
	       {
		  selectlist_setfocus(info, row);
		  gtk_clist_moveto(GTK_CLIST(info->self),row,0,-1,-1);
	       };
	  };
     }
   else
     // Reset keyboard search informations
     info->kbsearchinfo[0]=0;
};

GtkWidget *selectlist_create(selectlist_info *sel,int columns)
{
   GtkWidget *selectlist;

   if (sel->headings) /* headings available? */
     {
	selectlist=gtk_clist_new_with_titles(columns,
					     sel->headings);
     }
   else
     selectlist=gtk_clist_new(columns);
   gtk_clist_column_titles_passive(GTK_CLIST(selectlist));

   // Apply modified drag and drop scheme
   if (selectlist&&
       (!strcasecmp(varman_getvar(global_defs,"dnd_gnome"),"true")))
     {
	// This will make drag and drop only processed button action
	// Selecting/deselecting rows will be done manually
	gtk_clist_set_button_actions(GTK_CLIST(selectlist),0,GTK_BUTTON_DRAGS);
	// Now implement our own button press handler
	gtk_signal_connect(GTK_OBJECT(selectlist),"button_press_event",
			   GTK_SIGNAL_FUNC(selectlist_modifiedselect),sel);
	gtk_signal_connect(GTK_OBJECT(selectlist),"key_press_event",
			   GTK_SIGNAL_FUNC(selectlist_modifiedselect_keyboard_press),sel);
	gtk_signal_connect(GTK_OBJECT(selectlist),"key_release_event",
			   GTK_SIGNAL_FUNC(selectlist_modifiedselect_keyboard_release),sel);
     };

   sel->self=GTK_CLIST(selectlist);
   sel->modifier = None;

   if (sel->send)           /* let the user specify if the current window should be sourced */
     dndsetup_drag(selectlist,
		   (dndsetup_dragcallback)selectlist_drag,
		   (dndsetup_dragdonecallback)selectlist_dragdone,
		   sel);
   if (sel->receiver!=NULL) /* do not install drop handler if no handler is specified*/
     dndsetup_drop(selectlist,sel->receiver,sel);
   if (sel->popup!=NULL)
     menusys_attachpopupmenu(selectlist,
			     sel->popup,
			     sel);

   gtk_signal_connect(GTK_OBJECT(selectlist),"select_row",
		      GTK_SIGNAL_FUNC(selectlist_select),sel);
   gtk_signal_connect(GTK_OBJECT(selectlist),"unselect_row",
		      GTK_SIGNAL_FUNC(selectlist_unselect),sel);

   gtk_signal_connect(GTK_OBJECT(selectlist),"key_press_event",
		      GTK_SIGNAL_FUNC(selectlist_standardkeyboardhandler),sel);
   gtk_signal_connect(GTK_OBJECT(selectlist),"button_press_event",
		      GTK_SIGNAL_FUNC(selectlist_getfocus),sel);
   gtk_signal_connect(GTK_OBJECT(selectlist),"map",
		      GTK_SIGNAL_FUNC(selectlist_map),sel);
   gtk_signal_connect(GTK_OBJECT(selectlist),"unmap",
		      GTK_SIGNAL_FUNC(selectlist_unmap),sel);
   doubleclick_connect(GTK_WIDGET(selectlist),
		       GTK_SIGNAL_FUNC(selectlist_doubleclick),
		       sel);

   return selectlist;
}
;

char *selectlist_getselection(selectlist_info *info)
{
   GList *i;
   int row;
   char *filename;

   info->selection[0]=0; /* clear the selection string */
   /* create the selection string of all rows marked as selected */
   for (i=info->selected_lines;i!=NULL;i=i->next)
     {
	row=(int)i->data; /* get row */
   	/* if significant_column is set to -1,a string addressed by the
	 * hidden gpointer data field of the clist is taken as selection
	 * information */
	if (info->significant_column!=-1)
	  {
	     gtk_clist_get_text(info->self,
				row,
				info->significant_column,
				&filename);
	  }
	else
	  filename=(char*)gtk_clist_get_row_data(info->self,row);
	/* and finally append new selection entry */
	selectlist_appendselection(info->selection,
				   info->header,filename);
     };
#ifdef DEBUG
   printf("selectlist_getselection: Our selection is %s\n",info->selection);
#endif
   /* return pointer to selection */
   return (info->selection);
};

void selectlist_insert(selectlist_info *info,gint row,gchar *text[])
{
   GList *current;
   /* insert the new item */
   gtk_clist_insert(info->self,row,text);
   /* step through the list of selections */
   for (current=info->selected_lines;
	current!=NULL;
	current=current->next)
     {
	if ((int)current->data>=row)
	  current->data++;
     };
};

void selectlist_remove(selectlist_info *info,gint row)
{
   GList *current;
   /* remove an item */
   gtk_clist_remove(info->self,row);
   /* remove item from selection list */
   info->selected_lines=g_list_remove(info->selected_lines,(gpointer)row);
   /* step through the list of selections */
   for (current=info->selected_lines;
	current!=NULL;
	current=current->next)
     {
	if ((int)current->data>row)
	  current->data--;
     };
};

fileman_addstreamglobal selectlist_streamdropinfo;

void selectlist_filetranscode(char *i,
			      gpointer generic_userdata,
			      gpointer user_data,

			      fileman_additems_continuecb cb,
			      fileman_additems_state *cbdata)
{
   int success=0;
   if (i)
     {
	tracks_trackinfo *newtrack=stdfiletrack_create(i);
	if (newtrack)
	  {
	     char *newname;

	     char *str=streams_getstreamid(newtrack);
	     char *stream=malloc(strlen(str)+8); /* 'stream:str'\0 */

	     strcpy(stream,"stream:");
	     strcat(stream,str);
	     free(str);

	     /* we're dealing with a file here, so let's hack the track
	      * identifier to match the filename. This will avoid confusion
	      * and generate the same filename with a different suffix */
	     newname=strrchr(i,'/');
	     if (!newname)
	       newname=i;
	     else
	       newname++;
	     newname=strdup(newname);
	     if (strrchr(newname,'.'))
	       *strrchr(newname,'.')=0;
	     strcpy(newtrack->name,newname);
	     free(newname);

	     tracks_unclaim(newtrack);
	     if (stream)
	       {
		  fileman_addstream(stream,
				    generic_userdata,
				    user_data,
				    cb,
				    cbdata);
		  tracks_unclaim(newtrack);
		  free(stream);
		  success=1;
	       };
	  };
     };
   /* if we couldn't call addstream we have to handle the continue callback
    * ourselves */
   if (!success)
     cb(cbdata);
};

void selectlist_encodetofile_listprocessed_cb(int status,gpointer data)
{
   // free destination uri
   free(data);
};

void selectlist_encodetofile_dirselected(GtkWidget *w,selectlist_info *info)
{
   char *items=selectlist_getselection(info);

   if (items)
     {
	char *destination=vfs_buildvfsuri(unixtree,
					  gtk_file_selection_get_filename(GTK_FILE_SELECTION(w->parent->parent->parent)));
	GList *itemlist=helpings_convertdndlist(items);
#ifdef DEBUG
	printf("selectlist_encodetofile_dirselected: Encoding the following items: %s\n",items);
#endif

	selectlist_streamdropinfo.encodermapping=NULL;
	selectlist_streamdropinfo.dragtype=DNDSETUP_COPY;
	/* we don't need a "done" handler */
	fileman_additemlist(itemlist,
			    selectlist_encodetofile_listprocessed_cb,
			    (gpointer)destination,

			    (gpointer)destination,
			    2,

			    "",
			    selectlist_filetranscode,
			    (gpointer)&selectlist_streamdropinfo,

			    "stream:",
			    fileman_addstream,
			    (gpointer)&selectlist_streamdropinfo);
     };
};

void selectlist_encodetofile_destroyfs(GtkWidget *w,gpointer data)
{
   gtk_widget_destroy(w->parent->parent->parent);
};

/* you can directly refer to this handler from any popup menu attached to
 * a selectlist. It will provide transparent encoding capability for
 * both tracks and files */
void selectlist_encodetofile(GtkWidget *w,selectlist_info *info)
{
   GtkWidget *d=gtk_file_selection_new(_("Select destination directory"));
   gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(d)->ok_button),"clicked",
		      selectlist_encodetofile_dirselected,
		      info);
   gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(d)->ok_button),"clicked",
		      selectlist_encodetofile_destroyfs,NULL);
   gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(d)->cancel_button),"clicked",
		      selectlist_encodetofile_destroyfs,NULL);
   gtk_widget_show(d);
};

