/* Recording controls */

#include <gtk/gtk.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/wait.h>
#include <sys/types.h>
#include <stdlib.h>
#include <errno.h>

#include "int.h"

#include "preferences.h"
#include "dependencies.h"
#include "main.h"
#include "colors.h"
#include "record.h"
#include "piping.h"
#include "tracks.h"
#include "tracklist.h"
#include "trackedit.h"
#include "rterm.h"
#include "datacopydlg.h"
#include "varman.h"
#include "varmanwidgets.h"
#include "cddrives.h"
#include "preview.h"
#include "layoutconfig.h"
#include "fileman.h"
#include "updatehandlers.h"
#include "fsedit.h"
#include "dirlow.h"
#include "multisession.h"

/* uncomment for debugging */
// #define DEBUG
//
int record_active;  /* becomes true whenever access to the cdwriter is being performed*/

/* The following vars are not being exported to other objects,
 * so they do not have the leading "record_" ... */

/* controls for the recordingclient run macro */
int STDINWAIT=0;

/* End of local var list */

/* return name of the named pipe to be used for singlecall cdrecord.
 * has to be freed with free() when no longer needed */
char *record_getpipename()
{
   /* create filename for the FIFO used for writing */
   char *pipename=varman_getvar_copy(global_defs,"rec_cdrecordnpipe");
   /* take care of trailing slashes */
   char *tempdir=varman_getvar_copy(global_defs,"tempdir");
   if (!strlen(tempdir))
     tempdir=strdup("/tmp");
   if (tempdir[strlen(tempdir)-1]=='/')
     tempdir[strlen(tempdir)-1]=0;
   varman_replacestring(pipename,"$tempdir",tempdir);
   free(tempdir);
   return pipename;
};

void record_unexpected_childexit()
{
   rterm_printcln(Blue,_("\nChild exited unexpectedly."));
}
;

/* eject the cd if the user chose to do so */
void record_eject(record_recordingstate *state)
{
   char *call;
   int status;

	/* if eject is enabled */
   if (!strcmp(varman_getvar(global_defs,"rec_doeject"),"true"))
     {
	call=varman_getvar_copy(global_defs,"rec_ejectdisc");
	varman_replacestring(call,"$scsiid",
			     cddrives_getrecorder()->scsiid);
	varman_replacevars(global_defs,call);
	state->inpipe=-1;
	state->outpipe=-1;
	state->errpipe=-1;
	state->cdrecordpid=piping_create(call,
					 &state->inpipe,
					 &state->outpipe,
					 &state->errpipe);
	free(call);
	state->outtag=rterm_connectpipe(&Black,state->outpipe);
	state->errtag=rterm_connectpipe(&Red,state->errpipe);

		     /* wait for the eject client to finish */
	waitpid(state->cdrecordpid,&status,0);

	rterm_disconnectpipe(state->outtag);
	rterm_disconnectpipe(state->errtag);
	close(state->inpipe);
	close(state->outpipe);
	close(state->errpipe);
     }
   ;
	/* if reload is enabled */
   if (!strcmp(varman_getvar(global_defs,"rec_reloaddisc"),"true"))
     {
	call=varman_getvar_copy(global_defs,"rec_reloaddisc_cmd");
	varman_replacestring(call,"$device",
			     cddrives_getrecorder()->device);
	varman_replacevars(global_defs,call);
	state->inpipe=-1;
	state->outpipe=-1;
	state->errpipe=-1;
	state->cdrecordpid=piping_create(call,
					 &state->inpipe,
					 &state->outpipe,
					 &state->errpipe);
	free(call);
	state->outtag=rterm_connectpipe(&Black,state->outpipe);
	state->errtag=rterm_connectpipe(&Red,state->errpipe);

		     /* wait for the eject client to finish */
	waitpid(state->cdrecordpid,&status,0);

	rterm_disconnectpipe(state->outtag);
	rterm_disconnectpipe(state->errtag);
	close(state->inpipe);
	close(state->outpipe);
	close(state->errpipe);
     }
   ;
}
;

/* This function gets called by either
 *  * the recording high level control function (record_trackdonecb)
 *    if a fatal error occurred
 *  * by the record_finish function if the recording process was
 *    completed correctly but the "Fixate" option was deselected
 *    by the user
 * */
void record_complete(record_recordingstate *state)
{
   char *pipename;

   /* eject the cd if selected */
   record_eject(state);

   rterm_printcln(Blue,_("CD recording process finished."));

   layoutconfig_widget_hide(state->dlg->messagebox,"recordprogressdialog");
   record_active=0;
   cddrives_updateenable();

   /* delete cdrdao tocfile when done.
    * This is not a driver based issue at the moment.
    * Any driver using a TOC file would store it's name in state->tocfile.
    * this routine will then automatically take care of it's destruction */
   if (state->tocfile!=NULL)
     {
	remove(state->tocfile);
	free(state->tocfile);
	state->tocfile=NULL;
     };
   datacopydlg_destroy(state->dlg);

   /* remove FIFO file.
    * No error checking as we wouldn't be able to do anything about it
    * anyway */
   pipename=record_getpipename();
   remove(pipename);
   free(pipename);

   /* free info struct */
   free(state);
}
;

int record_fixating_finished(int status,gpointer data)
{
   record_recordingstate *state=(record_recordingstate*)data;

   /* this is finally done for all backends.
    * It's just the normal way of life I think */
   rterm_disconnectpipe(state->outtag);
   rterm_disconnectpipe(state->errtag);
   close(state->inpipe);
   close(state->outpipe);
   close(state->errpipe);

   gtk_timeout_remove(state->measure_timeout_handle);
   varman_setvar_value(global_defs,
		       "statistics_leadoutsize",
		       state->measure_size);
   record_complete(state);

   return 0;
};

gint record_measure_updateprogress(gpointer data)
{
   record_recordingstate *state=(record_recordingstate*)data;
   int readsize;

   readsize=(int)varman_getvar_value(global_defs,"rec_speed")*170*1024;
   datacopydlg_newdatablock(state->dlg,readsize);
   state->measure_size+=readsize;

   return 1;
};

/* this macro draws all the common parts of instantiating a recording
 * process together.
 * it can be used for both cdrdao and cdrecord by merely specifying
 * the appropriate client varman var*/
#define record_callrecordingclient \
   varman_replacevars(global_defs,call); \
   state->inpipe=-1;state->errpipe=-1;state->outpipe=-1; \
   if (piping_isvalid_command(call)) \
   { \
   rterm_printcln(Blue, _("GnomeToaster is calling the following command:")); \
   rterm_printcln(Blue, call); \
   if (STDINWAIT) \
   state->cdrecordpid=piping_create_wait(call,&state->inpipe,&state->outpipe,&state->errpipe); \
   else \
   state->cdrecordpid=piping_create(call,&state->inpipe,&state->outpipe,&state->errpipe); \
   } \
   else { state->cdrecordpid=-1;dependencies_showmissingexec(call);} \
   free(call); \
   state->outtag=rterm_connectpipe(&Black,state->outpipe); \
   state->errtag=rterm_connectpipe(&Red,state->errpipe)

void record_finishrecord(record_recordingstate *state)
{
   piping_watch *fixrom;
   char *call;

   /* only if fixating is enabled */
   if (!strcmp(varman_getvar(global_defs,"rec_dofixate"),"true"))
     {
	rterm_printcln (Blue,_("\nfixating CDROM."));
	switch (state->writing_mode)
	  {
	   case RECORD_MODE_TAO_MULTI:
	     /* only our call cdrecord per track mode needs to do anything
	      * here. All other backends will enter fixating mode
	      * transparently */
	     call=varman_getvar_copy(global_defs,"rec_fixate");
	     varman_replacestring(call,"$scsiid",
				  cddrives_getrecorder()->scsiid);
	     record_callrecordingclient;
	     break;
	  };
	/* add the pidwatch for cdrdao as well */
	fixrom=piping_pidwatch_add(state->cdrecordpid,
				   record_fixating_finished,
				   state);
	state->measure_size=0;
	state->measure_timeout_handle=gtk_timeout_add(1000,
						      record_measure_updateprogress,
						      state);

	/* update progress dialog */
	/* this is all about updating the progress dialog */
	datacopydlg_restartthread(state->dlg,
				  1,
				  (int)varman_getvar_value(global_defs,
							   "statistics_leadoutsize")
				  );
	datacopydlg_configurethread(state->dlg,
				    1,
				    DATACOPYDLG_SHOWFRAME|
				    DATACOPYDLG_SHOWLABEL|
				    DATACOPYDLG_SHOWTIME_REMAINING|
				    DATACOPYDLG_SHOWTIME_ELAPSED|
				    DATACOPYDLG_SHOWPROGRESS,
				    NULL,
				    _("fixating cdrom"),
				    -1);
	/* there's no copy thread while fixating */
	state->cs=NULL;
     }
   else
     record_complete(state);
}
;

void record_trackdonecb(streamcopy_state *cs,
			record_recordingstate *state);

void record_nexttrack(record_recordingstate *state)
{
   char *call;
   int trackpipe;
   unsigned int tracksize;

   int streamcopyoptions=0; /* filled out by the backend drivers
			     * and passed to streamcopy to fulfill the
			     * specific requirements of the different
			     * backends and control methods */

   /* do record only tracks that are available */
   if (tracks_valid(tracklist_gettrack(trackedit,state->currenttrack)))
     {
	trackpipe=tracks_openpipe(tracklist_gettrack(trackedit,state->currenttrack));
	tracksize=tracks_tracksize(tracklist_gettrack(trackedit,state->currenttrack));

	switch (state->writing_mode)
	  {
	   case RECORD_MODE_TAO_MULTI:
	     /* restart cdrecord when in _TAO_MULTI mode */
	     call=varman_getvar_copy(global_defs,"rec_writetrack");
	     varman_replacestring(call,"$tracktype",
				  tracklist_gettrack(trackedit,state->currenttrack)->tracktype);
	     varman_replacestring(call,"$scsiid",
				  cddrives_getrecorder()->scsiid);
	     varman_replacestring(call,"$buprotect",
				  (!strcmp(varman_getvar(global_defs,
							 "rec_buprotect"),
					   "true"))
				  ?" driveropts=burnproof":"");
	     varman_replacestringbyfloat(call,"$tracksize",
					 tracksize);
	     /* use makro to start the actual recording process
	      * wait for data to come in at stdin before calling backend */
	     STDINWAIT=1;
	     record_callrecordingclient;
	     STDINWAIT=0;
	     streamcopyoptions=(STREAMCOPY_OPT_CLOSEDEST|
				STREAMCOPY_OPT_WAITEXIT);
	     break;
	   case RECORD_MODE_TAO_SINGLE:
	   case RECORD_MODE_DAO_SINGLE:
	   case RECORD_MODE_DAO_CDRDAO:
	     /* when recording process is aborted by the user,
	      * send SIGTERM to the client */
	     streamcopyoptions=STREAMCOPY_OPT_SIGTERMONUSEREXT;
	     break;
	     /* no default handler, other recording clients don't need any
	      * control intervention here. */
	  };

	/* this is all about updating the progress dialog */
	datacopydlg_restartthread(state->dlg,
				  1,
				  tracks_tracksize(tracklist_gettrack(trackedit,state->currenttrack))
				  );
	call=(char*)malloc(MAXSTRINGSIZE);
	sprintf(call,
		_("Track %i/%i"),
		state->currenttrack+1,
		trackedit->entries);
	datacopydlg_configurethread(state->dlg,
				    0,
				    0,
				    NULL,
				    call,
				    -1);
	free(call);
#ifdef DEBUG
	printf ("record_nexttrack: starting data copy thread\n");
#endif
	state->cs=streamcopy_create(state->inpipe,
				    trackpipe,
				    state->cdrecordpid,
				    tracksize,
				    state->dlg,
				    // generic options
				    STREAMCOPY_OPT_PAD|
				    STREAMCOPY_OPT_CUT|
				    // backend specific options
				    streamcopyoptions,
				    (streamcopy_callback)&record_trackdonecb,
				    state);
     }
   else
     {
	state->currenttrack++;
	if (state->currenttrack<trackedit->entries)
	  record_nexttrack(state);
	else
	  record_finishrecord(state);
     };
};

/* this gets called when the streamcopy routine is done copying
 * the current track */
void record_trackdonecb(streamcopy_state *cs,
			record_recordingstate *state)
{
   /* do error processing */
   switch(cs->writingstate)
     {
      case STREAMCOPY_STATE_BACKENDEXT:
      case STREAMCOPY_STATE_PTHREADERR:
	record_unexpected_childexit();
	break;
     };
   /* using those switch statements consequently throughout record.c now.
    * This will make it particularly easy to switch to a "plugin-style"
    * backend architecture flawlessly */
   switch (state->writing_mode)
     {
      case RECORD_MODE_TAO_MULTI:
	rterm_disconnectpipe(state->outtag);
	rterm_disconnectpipe(state->errtag);
	close(state->outpipe);
	close(state->errpipe);
	break;
      case RECORD_MODE_DAO_CDRDAO:
      case RECORD_MODE_TAO_SINGLE:
      case RECORD_MODE_DAO_SINGLE:
	/* we only have to interfere with the normal course of things
	 * if there was a problem.
	 *
	 * This is the same for all our backends expect the one starting
	 * a new instance of cdrecord for every fart */
	if (cs->fatalerror)
	  {
	     rterm_disconnectpipe(state->outtag);
	     rterm_disconnectpipe(state->errtag);
	     close(state->outpipe);
	     close(state->errpipe);
	     /* additionally to the above, we have to manually close ->inpipe
	      * here.
	      * In cdrecord single-call mode a SIGTERM needs to be sent on
	      * user abortion. This has already been done by the streamcopy
	      * handler at this stage */
	     close(state->inpipe);
	  };
	break;
     };
   tracks_closepipe(tracklist_gettrack(trackedit,state->currenttrack));
#ifdef DEBUG
   printf ("record_trackdonecb: finished writing track.\n");
#endif
   if (!cs->fatalerror)
     {
	state->currenttrack++;
	if (state->currenttrack<trackedit->entries)
	  {
	     record_nexttrack(state);
	  }
	else
	  record_finishrecord(state);
     }
   else
     {
	/* a fatal error occured. Stop the writing process */
	record_complete(state);
     };
};

void record_stoprecording(GtkWidget *w,
			  gpointer data)
{
   record_recordingstate *state=(record_recordingstate*)data;

#ifdef DEBUG
   printf("record_stoprecording: request to cancel recording process received\n");
#endif
   if (state->cs!=NULL)
     streamcopy_cancel(state->cs);
};

/* startup cdrdao */
void record_startdao_cdrdao(record_recordingstate *state)
{
   char *call;

   call=varman_getvar_copy(global_defs,"rec_daowrite");
   varman_replacestring(call,"$tocfile",
			state->tocfile);
   varman_replacestring(call,"$scsiid",
			cddrives_getrecorder()->scsiid);
   varman_replacestring(call,"$cdtextflag",
			((!strcmp(varman_getvar(global_defs,"rec_writecdtext"),
				  "true"))
			 ?"0x10":"0x00"));
   /* use makro to start the actual recording process */
   record_callrecordingclient;
};

/* startup cdrecord in single-instance mode.
 * The code here is the same for DAO and TAO mode except for the
 * -dao parameter passed to cdrecord in the former case */
void record_startsinglecdrecord(record_recordingstate *state)
{
   int x;
   char *call;
   char *singular;
   char writelist[MAXSTRINGSIZE];
   char *pipename;
   int status;

#ifdef DEBUG
   printf("record_startsinglecdrecord: preparing commandline\n");
#endif
   pipename=record_getpipename();
   /* FIXME: do error checking here ?!? */
   mkfifo(pipename,0700);

   /* build cdrecord call */
   call=varman_getvar_copy(global_defs,"rec_writecd");
   varman_replacestring(call,"$scsiid",
			cddrives_getrecorder()->scsiid);
   varman_replacestring(call,"$daomode",
			(!strcmp(varman_getvar(global_defs,
					       "rec_dao"),
				 "true"))
			?" -dao":"");
   varman_replacestring(call,"$buprotect",
			(!strcmp(varman_getvar(global_defs,
					       "rec_buprotect"),
				 "true"))
			?" driveropts=burnproof":"");
   strcpy(writelist,"");
   for (x=0;x<trackedit->entries;x++)
     {
	singular=varman_getvar_copy(global_defs,"rec_trackitem");
	varman_replacestring(singular,"$tracktype",
			     tracklist_gettrack(trackedit,x)->tracktype);
	varman_replacestringbyfloat(singular,"$tracksize",
				    tracks_tracksize(tracklist_gettrack(trackedit,x)));
	varman_replacestring(singular,"$filename",pipename);

	if (!((strlen(writelist)+strlen(singular)+1)>MAXSTRINGSIZE))
	  {
	     strcat(writelist,singular);
	     if (x<(trackedit->entries-1))
	       strcat(writelist," ");
	  };
	free(singular);
     };
   varman_replacestring(call,"$writelist",writelist);
   /* use makro to start the actual recording process */
#ifdef DEBUG
   printf("record_startsinglecdrecord: running '%s'\n",call);
#endif
   /* replaces global vars and frees call afterwards */
   record_callrecordingclient;

   /* when opening a FIFO, open() might
    * block Gnometoaster indefinitely in case
    * cdrecord doesn't start up. avoid this by opening in
    * O_NONBLOCK mode and checking both the result and
    * the state of the client in a loop */
   while (((state->inpipe=open(pipename,O_WRONLY|O_NONBLOCK))==-1)
	  &&(!waitpid(state->cdrecordpid,&status,WNOHANG))
	  &&(errno==ENXIO));
   /* clear O_NONBLOCK flag */
   fcntl(state->inpipe,
	 F_SETFL,
	 0);
#ifdef DEBUG
   perror("result of open was");
   printf("record_startsinglecdrecord: opened named pipe '%s' to cdrecord: %i\n",
	  pipename,
	  state->inpipe);
#endif
   /* free temporary string */
   free(pipename);
};

/* a few macros for determining recorder settings */
#define RECORD_USEDAOMODE (!strcmp(varman_getvar(global_defs,"rec_dao"),"true"))
#define RECORD_USECDRDAO  (!strcmp(varman_getvar(global_defs,"rec_usecdrdao"),"true"))
#define RECORD_USESCDREC  (!strcmp(varman_getvar(global_defs,"rec_singlecallcdrecord"),"true"))

/* this is where the recording process continues after
 * all tracks with the precache flag set have been precached */
void record_afterprecache()
{
   int x,discsize;
   char buf[256];
   int tracksvalid;
   record_recordingstate *state;

   state=(record_recordingstate*)malloc(sizeof(record_recordingstate));

   /* calculate recording mode depending on the user's settings */
   if (RECORD_USEDAOMODE)
     {
	if (RECORD_USECDRDAO)
	  state->writing_mode=RECORD_MODE_DAO_CDRDAO;
	else
	  state->writing_mode=RECORD_MODE_DAO_SINGLE;
     }
   else
     {
	if (RECORD_USESCDREC)
	  state->writing_mode=RECORD_MODE_TAO_SINGLE;
	else
	  state->writing_mode=RECORD_MODE_TAO_MULTI;
     };

   state->tocfile=NULL;

   record_active=1;
   cddrives_updatedisable();
   discsize=0;
   tracksvalid=1;
   for (x=0;((x<trackedit->entries)&&(tracksvalid));x++)
     {
	discsize+=tracks_tracksize(tracklist_gettrack(trackedit,x));
	/* if only one track is invalid we won't record */
	if (!tracks_valid(tracklist_gettrack(trackedit,x)))
	  tracksvalid=0;
     }
   ;

   /* create cdrdao TOC file
    * FIXME: does this have to be hardcoded ?
    * (CORBA Backend ?!?) */

   /* write cdrdao TOC file if in RECORD_MODE_DAO_CDRDAO  */
   if ((state->writing_mode==RECORD_MODE_DAO_CDRDAO)&&(tracksvalid))
     {
	FILE *toc;
	int mixed_mode=0;
	int tempfd=-1;

	/* create a temporary filename */
	state->tocfile=helpings_fileinpath(TEMPDIR,"tocfileXXXXXX");
	tempfd=mkstemp(state->tocfile);
	toc=fdopen(tempfd,"w");
	if (toc!=NULL)
	  {
	     /* if there's at least one data track set the session type
	      * to mixed mode
	      * cdrdao wants this as kind of a security concept checking
	      * if you're really saying what you mean */
	     for (x=0;x<trackedit->entries;x++)
	       if (!strcmp(tracklist_gettrack(trackedit,x)->tracktype,"data"))
		 mixed_mode=1;
	     fprintf(toc,mixed_mode?"CD_ROM\n":"CD_DA\n");

	     if (!strcmp(varman_getvar(global_defs,"rec_writecdtext"),"true"))
	       {
		  const char *title = ((trackedit->title)?trackedit->title:_("Unnamed disc"));
		  const char *author = ((trackedit->author)?trackedit->author:_("No Artist"));
		  fprintf(toc,"CD_TEXT {\nLANGUAGE_MAP {\n0 : EN\n}\nLANGUAGE 0 {\nTITLE \"%s\"\nPERFORMER \"%s\"\n}\n}\n", title, author);
	       };
	     for (x=0;x<trackedit->entries;x++)
	       {
		  int is_data;
		  int tsize;

		  /* FIXME: write data tracks in mode2form1 if they're multisession
		   * tracks,not too important as dao is probably gonna be used
		   * for 1:1 copies and audio cds only anyway
		   * currently,we support only two tracktypes: audio and data */
		  is_data=
		    !strcmp(tracklist_gettrack(trackedit,x)->tracktype,"data");
		  tsize=
		    tracks_tracksize(tracklist_gettrack(trackedit,x));
		  fprintf(toc,"TRACK ");
		  fprintf(toc,
			  is_data
			  ?"MODE1\n"
			  :"AUDIO\n");
		  if (!strcmp(varman_getvar(global_defs,"rec_writecdtext"),"true"))
		    fprintf(toc,"CD_TEXT {\nLANGUAGE 0 {\nTITLE \"%s\"\nPERFORMER \"%s\"\n}\n}\n",
			    tracklist_gettrack(trackedit,x)->title,
			    tracklist_gettrack(trackedit,x)->performer);
		  fprintf(toc,
			  is_data
			  ?"DATA"
			  :"AUDIO");
		  fprintf(toc,"FILE \"-\" ");
		  if (is_data)
		    fprintf(toc,"%i\n",tsize);
		  else
		    {
		       int sectors=tsize/tracks_sectorsize("audio");
		       fprintf(toc,"0:0:0 %i:%i:%i\n",
			       sectors/(75*60),
			       (sectors/75)%60,
			       sectors%75);
		    };
	       };
	     fclose(toc);
	  };
     }; /* done writing toc file */

   sprintf(buf,_("Recording %i bytes to CD"),discsize);
   rterm_printcln(Blue,buf);

   state->currenttrack=0;
   /* no copy thread yet */
   state->cs=NULL;

   /* don't do actual recording in dao yet */
   if (!preview_player_active)
     {
	/* only continue if
	 *  a) there are tracks to record
	 *  b) all tracks can send their data immediately */
	if ((trackedit->entries>0) &&
	    (discsize>0) &&
	    (tracksvalid))
	  {
	     state->dlg=datacopydlg_create(_("Writing CD"),
					   record_stoprecording,(gpointer)state,
					   2,

					   DATACOPYDLG_SHOWFRAME|
					   DATACOPYDLG_SHOWLABEL|
					   DATACOPYDLG_SHOWTIME_REMAINING|
					   DATACOPYDLG_SHOWTIME_ELAPSED|
					   DATACOPYDLG_SHOWPROGRESS|
					   DATACOPYDLG_SHOWPROGRESSINTITLE,
					   _("CD status"),
					   _("Please wait"),
					   discsize+
					   (!strcmp(varman_getvar(global_defs,"rec_dofixate"),"true"))*
					   (int)varman_getvar_value(global_defs,
								    "statistics_leadoutsize")
					   ,

					   DATACOPYDLG_SHOWFRAME|
					   DATACOPYDLG_SHOWTHROUGHPUT|
					   DATACOPYDLG_SHOWTIME_REMAINING|
					   DATACOPYDLG_SHOWTIME_ELAPSED|
					   DATACOPYDLG_SHOWPROGRESS,
					   _("Track Status"),
					   "",
					   0

					   );
	     layoutconfig_widget_show(state->dlg->messagebox,
				      "recordprogressdialog");

	     /* start the disc at once recording client
	      * we can install an on-write handler and configure
	      * the vars required by the handler *afterwards* because
	      * gtk doesn't execute the handler before we return to
	      * gtk_main() */
	     switch (state->writing_mode)
	       {
		case RECORD_MODE_DAO_CDRDAO:
		  record_startdao_cdrdao(state);
		  break;
		case RECORD_MODE_TAO_SINGLE:
		case RECORD_MODE_DAO_SINGLE:
		  record_startsinglecdrecord(state);
		  break;
	       };
#ifdef DEBUG
	     printf("preparing to record first track...\n");
#endif

	     /* start recording with the next track
	      * This will call cdrecord in TAO mode
	      * and initialize the copy thread in any mode */
	     record_nexttrack(state);
	  }
	else
	  {
	     if (tracksvalid)
	       {
		  rterm_printcln(Blue,_("No recordable Tracks found."));
	       }
	     else
	       {
		  rterm_printcln(Blue,_("One or more tracks are unable to deliver their content."));
	       };
	     record_active=0;
	     cddrives_updateenable();
	  }
	;
     }
   else
     {
	rterm_printcln(Blue,_("Preview Player active,\nrecording process cancelled."));
	record_active=0;
	cddrives_updateenable();
     }
   ;
}
;

void record_startrecord(GtkWidget *w)
{
   /* if recording process is active or no recorder device could be found */
   if ((record_active)||(cddrives_getrecorder()==NULL)) return;

   gtk_notebook_set_page(GTK_NOTEBOOK(destination),2);
   /*So it can be called
    *from menubar or toolbar*/

   trackedit_doprereading(atwrite,
			  record_afterprecache);
};

int record_clear_finished(int status,
			  gpointer data)
{
   record_recordingstate *state=(record_recordingstate*)data;

   rterm_disconnectpipe(state->outtag);
   rterm_disconnectpipe(state->errtag);

   close(state->inpipe);
   close(state->outpipe);
   close(state->errpipe);

   record_eject(state);

   rterm_printcln(Blue,_("CDRW blanking complete."));

   record_active=0;
   cddrives_updateenable();

   layoutconfig_widget_hide(state->dlg->messagebox,
			    "blankingprogressdialog");
   datacopydlg_destroy(state->dlg);
   gtk_timeout_remove(state->measure_timeout_handle);
   varman_setvar_value(global_defs,
		       (!strcmp(varman_getvar(global_defs,"rec_blankingmode"),
				"fast"))
		       ?"statistics_fastblank"
		       :"statistics_allblank",
		       state->measure_size);

   /* free info struct */
   free(state);

   return 0;
}
;

void record_clear(GtkWidget *w)
{
   piping_watch *cdrecordwatch;
   char *call;
   record_recordingstate *state;

   state=(record_recordingstate*)malloc(sizeof(record_recordingstate));

   /* if recording process is running of no recorder could be found */
   if ((record_active)||(cddrives_getrecorder()==NULL))
     return;

   gtk_notebook_set_page(GTK_NOTEBOOK(destination),2);
   /*So it can be called
    *from menubar or toolbar*/

   record_active=1;
   cddrives_updatedisable();

   call=varman_getvar_copy(global_defs,"rec_cleardisc");
   varman_replacestring(call,"$scsiid",
			cddrives_getrecorder()->scsiid);
   varman_replacevars(global_defs,call);
   rterm_printcln (Blue,_("blanking CDRW."));
   state->inpipe=-1;state->errpipe=-1;state->outpipe=-1;
   if (piping_isvalid_command(call))
     {
	state->cdrecordpid=piping_create(call,&state->inpipe,&state->outpipe,&state->errpipe);
     }
   else
     {
	state->cdrecordpid=-1;
	dependencies_showmissingexec(call);
     };
   free(call);
   state->outtag=rterm_connectpipe(&Black,state->outpipe);
   state->errtag=rterm_connectpipe(&Red,state->errpipe);

   cdrecordwatch=piping_pidwatch_add(state->cdrecordpid,
				     record_clear_finished,
				     state);

   state->dlg=datacopydlg_create(_("Blanking CD/RW"),
				 NULL,NULL,
				 1,

				 DATACOPYDLG_SHOWTIME_REMAINING|
				 DATACOPYDLG_SHOWTIME_ELAPSED|
				 DATACOPYDLG_SHOWPROGRESS|
				 DATACOPYDLG_SHOWPROGRESSINTITLE,
				 "",
				 "",
				 /* different values for fast and normal blank */
				 (!strcmp(varman_getvar(global_defs,"rec_blankingmode"),
					  "fast"))
				 ?
				 (int)varman_getvar_value(global_defs,
							  "statistics_fastblank")
				 :
				 (int)varman_getvar_value(global_defs,
							  "statistics_allblank")
				 );
   state->measure_size=0;
   state->measure_timeout_handle=gtk_timeout_add(1000,
						 record_measure_updateprogress,
						 state);
   layoutconfig_widget_show(state->dlg->messagebox,
			    "blankingprogressdialog");

}
;

varmanwidgets_widget *makebootable;
varmanwidgets_widget *button_daomode;
varmanwidgets_widget *button_cdtext;
varmanwidgets_widget *button_buprotect;
varmanwidgets_widget *button_fixate;

void record_daopossible_changed(VARMAN_VAR_ENTRY *var,gpointer data)
{
   int ok;

   ok=!strcmp(varman_getvar(global_defs,"rec_multisession"),
	      "");
   if ((gpointer)multisession_lastsession!=NULL) ok=0;
   /* if cdrecord is to be used for DAO recording,
    * it should be safe to allow multisession discs in DAO mode */
   if (!RECORD_USECDRDAO) ok=1;

   gtk_widget_set_sensitive(button_daomode->visual,
			    ok);
   if (!ok)
     varman_setvar(global_defs,"rec_dao","false");
};

void record_fixatenecessary_changed(VARMAN_VAR_ENTRY *var,gpointer data)
{
   int required=(strcmp(varman_getvar(global_defs,"rec_multisession"),"")!=0);
   gtk_widget_set_sensitive(button_fixate->visual,(!required));
   if (required)
     varman_setvar(global_defs,"rec_dofixate","true");
};

void record_buppossible_changed(VARMAN_VAR_ENTRY *var,gpointer data)
{
   int ok=1;

   /* No Buffer-Underrun protection for cdrdao */
   if ((RECORD_USECDRDAO)&&(RECORD_USEDAOMODE)) ok=0;

   gtk_widget_set_sensitive(button_buprotect->visual,
			    ok);
   if (!ok)
     varman_setvar(global_defs,"rec_buprotect","false");
};

void record_cdtextpossible_changed(VARMAN_VAR_ENTRY *var,gpointer data)
{
   int ok;

   ok=((!strcmp(varman_getvar(global_defs,"rec_dao"),
		"true"))&&RECORD_USECDRDAO);
   gtk_widget_set_sensitive(button_cdtext->visual,
			    ok);
   if (!ok)
     varman_setvar(global_defs,"rec_writecdtext","false");
};

void record_bootimage_changed(VARMAN_VAR_ENTRY *var,gpointer data)
{
   char *bootfile;
   vfs_direntry *entry=NULL;

   int ok=0;

   bootfile=varman_getvar(global_defs,"isotrack_bootimage");

   entry=vfs_filestat(edited_fs,bootfile,0);
   ok= (entry&&(!entry->isdir));
   if (entry)
     free(entry);
   gtk_widget_set_sensitive(makebootable->edit,
			    ok);
   if (!ok)
     varman_setvar(global_defs,"isotrack_makebootable","false");
};

void record_setdaodummymode(VARMAN_VAR_ENTRY *var,gpointer data)
{
   if (!strcmp(varman_getvar(global_defs,"rec_dummymode")," -dummy"))
     varman_setvar(global_defs,"rec_daodummymode"," --simulate");
   else
     varman_setvar(global_defs,"rec_daodummymode","");
};

GtkWidget *record_create()
{
   GtkWidget *options;
   GtkWidget *rec;

   GtkWidget *hbox_record;
   GtkWidget *vbox_record;
   GtkWidget *frame_record;
   GtkWidget *button_record;
   varmanwidgets_widget *button_multisession;

   GtkWidget *hbox_cdrw;
   GtkWidget *frame_cdrw;
   GtkWidget *button_clear;
   varmanwidgets_widget *button_cleartype;

   GtkWidget *hbox_general;
   GtkWidget *frame_general;
   varmanwidgets_widget *speed;
   varmanwidgets_widget *button_doeject;
   varmanwidgets_widget *button_reload;
   varmanwidgets_widget *button_dummywrite;

   GtkWidget *hbox_iso;
   GtkWidget *frame_iso;
   varmanwidgets_widget *bootimage;

   GtkWidget *terminal;

   GtkWidget *label;
   GtkWidget *record_prefs;

   varmanwidgets_widget *edit;
   GtkWidget *pref_hbox;

   rec=gtk_hbox_new(0,0);
   gtk_widget_show(rec);

	/* General recording options */

   frame_general=gtk_frame_new(_("General"));
   hbox_general=gtk_vbox_new(0,0);

   speed=varmanwidgets_spin_new(_("Recorder Speed"),
				"rec_speed",
				global_defs,
				APPLYMODE_ALWAYS,
				0,
				GTK_ADJUSTMENT(gtk_adjustment_new(0,1,24,1,10,0))
				);
   gtk_box_pack_start(GTK_BOX(hbox_general),
		      speed->visual,0,0,0);
   gtk_widget_show(speed->visual);

   button_doeject=varmanwidgets_checkbox_new(_("Eject disc"),
					     "rec_doeject",
					     global_defs,
					     APPLYMODE_ALWAYS,
					     0,
					     "true",
					     "false"
					     );
   gtk_widget_show(button_doeject->visual);
   gtk_box_pack_start(GTK_BOX(hbox_general),button_doeject->visual,0,0,0);

   button_reload=varmanwidgets_checkbox_new(_("Reload disc"),
					    "rec_reloaddisc",
					    global_defs,
					    APPLYMODE_ALWAYS,
					    0,
					    "true",
					    "false"
					    );
   gtk_widget_show(button_reload->visual);
   gtk_box_pack_start(GTK_BOX(hbox_general),button_reload->visual,0,0,0);

   button_dummywrite=varmanwidgets_checkbox_new(_("Dummy Write"),
						"rec_dummymode",
						global_defs,
						APPLYMODE_ALWAYS,
						0,
						" -dummy",
						""
						);
   gtk_widget_show(button_dummywrite->visual);
   gtk_box_pack_start(GTK_BOX(hbox_general),button_dummywrite->visual,0,0,0);
   varman_install_handler(global_defs,
			  "rec_dummymode",
			  record_setdaodummymode,
			  NULL);

   button_daomode=varmanwidgets_checkbox_new(_("Disc at once"),
					     "rec_dao",
					     global_defs,
					     APPLYMODE_ALWAYS,
					     0,
					     "true",
					     "false"
					     );
   gtk_widget_show(button_daomode->visual);
   gtk_box_pack_start(GTK_BOX(hbox_general),button_daomode->visual,0,0,0);
/* make it impossible to even try to record a multisession cd in
 * disc at once mode.
 * Issues that currently make dao multisession recording impossible:
 *  * commandline for cdrdao multisesson leadout is --multi rather than -multi
 *    so a dynamic var in dynamic_def needs to be introduced and kept uptodate
 *  * we only call cdrdao once so we can't wait for isotrack to deliver
 *    it's data once we're in mid-writing. Besides,the scsi device
 *    will be blocked in this case.
 *    So essentially a linked datatrack session has to be the first track
 *    in the list, a situation we'll have to ensure before we allow
 *    dao writing */
   updatehandlers_register(&multisession_updatehandlers,
			   (updatehandlers_handler)record_daopossible_changed,
			   NULL);
   varman_install_handler(global_defs,
			  "rec_multisession",
			  record_daopossible_changed,
			  NULL);
   varman_install_handler(global_defs,
			  "rec_usecdrdao",
			  record_daopossible_changed,
			  NULL);
   record_daopossible_changed((VARMAN_VAR_ENTRY*)NULL,NULL);

   button_cdtext=varmanwidgets_checkbox_new(_("CD-Text"),
					    "rec_writecdtext",
					    global_defs,
					    APPLYMODE_ALWAYS,
					    0,
					    "true",
					    "false"
					    );
   gtk_widget_show(button_cdtext->visual);
   gtk_box_pack_start(GTK_BOX(hbox_general),button_cdtext->visual,0,0,0);
   varman_install_handler(global_defs,
			  "rec_dao",
			  record_cdtextpossible_changed,
			  NULL);
   varman_install_handler(global_defs,
			  "rec_usecdrdao",
			  record_cdtextpossible_changed,
			  NULL);
   // force update of cdtext flag   // 
   record_cdtextpossible_changed(NULL,NULL);

   gtk_container_add(GTK_CONTAINER(frame_general),hbox_general);
   gtk_widget_show(hbox_general);
   gtk_widget_show(frame_general);

   gtk_box_pack_start(GTK_BOX(rec),
		      frame_general,
		      0,0,0);

   options=gtk_vbox_new(0,0);
   gtk_widget_show(options);
   gtk_box_pack_start(GTK_BOX(rec),
		      options,
		      0,0,0);

   	/* recording frame */

   frame_record=gtk_frame_new(_("Recording"));
   hbox_record=gtk_hbox_new(0,0);
   vbox_record=gtk_vbox_new(0,0);

   button_record=gtk_button_new_with_label(_("Record"));
   gtk_signal_connect (GTK_OBJECT(button_record),"clicked",
		       GTK_SIGNAL_FUNC (record_startrecord),NULL);
   gtk_box_pack_start(GTK_BOX(hbox_record),button_record,0,0,0);
   gtk_widget_show(button_record);

   button_multisession=varmanwidgets_checkbox_new(_("Multisession"),
						  "rec_multisession",
						  global_defs,
						  APPLYMODE_ALWAYS,
						  0,
						  " -multi",
						  ""
						  );
   gtk_widget_show(button_multisession->visual);
   gtk_box_pack_start(GTK_BOX(hbox_record),button_multisession->visual,0,0,0);

   button_buprotect=varmanwidgets_checkbox_new(_("Buffer-Underrun Protection"),
						  "rec_buprotect",
						  global_defs,
						  APPLYMODE_ALWAYS,
						  0,
						  "true",
						  "false"
						  );
   gtk_widget_show(button_buprotect->visual);
   gtk_box_pack_end(GTK_BOX(vbox_record),button_buprotect->visual,0,0,0);
   varman_install_handler(global_defs,
			  "rec_usecdrdao",
			  record_buppossible_changed,
			  NULL);
   varman_install_handler(global_defs,
			  "rec_dao",
			  record_buppossible_changed,
			  NULL);
   record_buppossible_changed((VARMAN_VAR_ENTRY*)NULL,NULL);
   
   button_fixate=varmanwidgets_checkbox_new(_("Fixate"),
					    "rec_dofixate",
					    global_defs,
					    APPLYMODE_ALWAYS,
					    0,
					    "true",
					    "false"
					    );
   gtk_widget_show(button_fixate->visual);
   gtk_box_pack_start(GTK_BOX(hbox_record),button_fixate->visual,0,0,0);
   varman_install_handler(global_defs,
			  "rec_multisession",
			  record_fixatenecessary_changed,
			  NULL);
   record_fixatenecessary_changed((VARMAN_VAR_ENTRY*)NULL,NULL);

   gtk_box_pack_start(GTK_BOX(vbox_record),hbox_record,0,0,0);
   gtk_container_add(GTK_CONTAINER(frame_record),vbox_record);

   gtk_widget_show(vbox_record);
   gtk_widget_show(hbox_record);
   gtk_widget_show(frame_record);

   gtk_box_pack_start(GTK_BOX(options),
		      frame_record,
		      0,0,0);

	/* cd rw commands frame */

   frame_cdrw=gtk_frame_new("CD/RW");
   hbox_cdrw=gtk_hbox_new(0,0);

   button_clear=gtk_button_new_with_label(_("Clear Disc"));
   gtk_signal_connect (GTK_OBJECT(button_clear),"clicked",
		       GTK_SIGNAL_FUNC (record_clear),NULL);
   gtk_box_pack_start(GTK_BOX(hbox_cdrw),button_clear,0,0,0);
   gtk_widget_show(button_clear);

   button_cleartype=varmanwidgets_checkbox_new(_("TOC only"),
					       "rec_blankingmode",
					       global_defs,
					       APPLYMODE_ALWAYS,
					       0,
					       "fast",
					       "all"
					       );
   gtk_box_pack_start(GTK_BOX(hbox_cdrw),button_cleartype->visual,0,0,0);
   gtk_widget_show(button_cleartype->visual);

   gtk_container_add(GTK_CONTAINER(frame_cdrw),hbox_cdrw);
   gtk_widget_show(hbox_cdrw);
   gtk_widget_show(frame_cdrw);

   gtk_box_pack_start(GTK_BOX(options),
		      frame_cdrw,
		      0,0,0);

   /* common iso options */

   frame_iso=gtk_frame_new(_("ISO Options"));
   hbox_iso=gtk_hbox_new(0,0);

   bootimage=varmanwidgets_entry_new(_("Boot Image"),
				     "isotrack_bootimage",
				     global_defs,
				     APPLYMODE_ALWAYS,
				     0,64);
   gtk_box_pack_start(GTK_BOX(hbox_iso),bootimage->visual,0,0,0);
   gtk_widget_show(bootimage->visual);
   /* make checkbox below insensitive if path to bootimage is invalid */
   varman_install_handler(global_defs,
			  "isotrack_bootimage",
			  record_bootimage_changed,
			  NULL);
   updatehandlers_register(&fsedit_updatehandlers,
			   (updatehandlers_handler)record_bootimage_changed,
			   NULL);

   makebootable=varmanwidgets_checkbox_new(_("Make Bootable"),
					   "isotrack_makebootable",
					   global_defs,
					   APPLYMODE_ALWAYS,
					   0,
					   "true",
					   "false");
   gtk_box_pack_start(GTK_BOX(hbox_iso),makebootable->visual,0,0,0);
   gtk_widget_show(makebootable->visual);

   gtk_container_add(GTK_CONTAINER(frame_iso),hbox_iso);
   gtk_widget_show(hbox_iso);
   gtk_widget_show(frame_iso);

   gtk_box_pack_start(GTK_BOX(options),
		      frame_iso,
		      0,0,0);

   /* recording terminal */

   terminal=rterm_create();
   gtk_widget_show(terminal);

   gtk_box_pack_end(GTK_BOX(rec),
		    terminal,
		    1,1,0);

   /* Create Prefernces page for recording options */
   record_prefs=gtk_vbox_new(0,0);
   gtk_widget_show(record_prefs);
   label=gtk_label_new(_("Recorder"));
   gtk_widget_show(label);
   preferences_append_page(record_prefs,label);

   pref_hbox=gtk_hbox_new(0,0);
   edit=varmanwidgets_entry_new(_("recording client"),
				"rec_client",
				global_defs,APPLYMODE_BUTTON,160,160);
   gtk_box_pack_start(GTK_BOX(pref_hbox),edit->visual,0,0,0);
   edit=varmanwidgets_checkbox_new(_("Use single call"),
				   "rec_singlecallcdrecord",
				   global_defs,APPLYMODE_BUTTON,160,
				   "true","false");
   gtk_box_pack_start(GTK_BOX(pref_hbox),edit->visual,0,0,0);
   gtk_box_pack_start(GTK_BOX(record_prefs),pref_hbox,0,0,0);
   gtk_widget_show(pref_hbox);

   edit=varmanwidgets_entry_new(_("Single call FIFO"),
				"rec_cdrecordnpipe",
				global_defs,APPLYMODE_BUTTON,160,320);
   gtk_box_pack_start(GTK_BOX(record_prefs),edit->visual,0,0,0);

   pref_hbox=gtk_hbox_new(0,0);
   edit=varmanwidgets_entry_new(_("Write whole CD with"),
				"rec_writecd",
				global_defs,APPLYMODE_BUTTON,160,320);
   gtk_box_pack_start(GTK_BOX(pref_hbox),edit->visual,0,0,0);
   edit=varmanwidgets_entry_new(_("Track Item"),
				"rec_trackitem",
				global_defs,APPLYMODE_BUTTON,80,160);
   gtk_box_pack_start(GTK_BOX(pref_hbox),edit->visual,0,0,0);
   gtk_box_pack_start(GTK_BOX(record_prefs),pref_hbox,0,0,0);
   gtk_widget_show(pref_hbox);

   edit=varmanwidgets_entry_new(_("Fixating command"),
				"rec_fixate",
				global_defs,APPLYMODE_BUTTON,160,320);
   gtk_box_pack_start(GTK_BOX(record_prefs),edit->visual,0,0,0);

   edit=varmanwidgets_entry_new(_("Trackwrite command"),
				"rec_writetrack",
				global_defs,APPLYMODE_BUTTON,160,320);
   gtk_box_pack_start(GTK_BOX(record_prefs),edit->visual,0,0,0);

   edit=varmanwidgets_entry_new(_("Blank disc command"),
				"rec_cleardisc",
				global_defs,APPLYMODE_BUTTON,160,320);
   gtk_box_pack_start(GTK_BOX(record_prefs),edit->visual,0,0,0);

   edit=varmanwidgets_entry_new(_("Eject disc command"),
				"rec_ejectdisc",
				global_defs,APPLYMODE_BUTTON,160,320);
   gtk_box_pack_start(GTK_BOX(record_prefs),edit->visual,0,0,0);

   edit=varmanwidgets_entry_new(_("Reload Disc with"),
				"rec_reloaddisc_cmd",
				global_defs,APPLYMODE_BUTTON,160,320);
   gtk_box_pack_start(GTK_BOX(record_prefs),edit->visual,0,0,0);

   edit=varmanwidgets_entry_new(_("Multisession Info"),
				"rec_getmsinfo",
				global_defs,APPLYMODE_BUTTON,160,320);
   gtk_box_pack_start(GTK_BOX(record_prefs),edit->visual,0,0,0);

   pref_hbox=gtk_hbox_new(0,0);
   edit=varmanwidgets_entry_new(_("DAO Recording Client"),
				"rec_daoclient",
				global_defs,APPLYMODE_BUTTON,160,160);
   gtk_box_pack_start(GTK_BOX(pref_hbox),edit->visual,0,0,0);
   edit=varmanwidgets_checkbox_new(_("Use alternative client"),
				   "rec_usecdrdao",
				   global_defs,APPLYMODE_BUTTON,160,
				   "true","false");
   gtk_box_pack_start(GTK_BOX(pref_hbox),edit->visual,0,0,0);
   gtk_box_pack_start(GTK_BOX(record_prefs),pref_hbox,0,0,0);
   gtk_widget_show(pref_hbox);

   edit=varmanwidgets_entry_new(_("DAO Write Command"),
				"rec_daowrite",
				global_defs,APPLYMODE_BUTTON,160,320);
   gtk_box_pack_start(GTK_BOX(record_prefs),edit->visual,0,0,0);

   edit=varmanwidgets_entry_new(_("Scsi bus scan command"),
				"rec_scanbus",
				global_defs,APPLYMODE_BUTTON,160,320);
   gtk_box_pack_start(GTK_BOX(record_prefs),edit->visual,0,0,0);
   
   edit=varmanwidgets_entry_new(_("Scsi parse scan output"),
				"rec_sbgetid",
				global_defs,APPLYMODE_BUTTON,160,320);
   gtk_box_pack_start(GTK_BOX(record_prefs),edit->visual,0,0,0);
   
   edit=varmanwidgets_entry_new(_("Scsi drive inquiry"),
				"rec_inqdrive",
				global_defs,APPLYMODE_BUTTON,160,320);
   gtk_box_pack_start(GTK_BOX(record_prefs),edit->visual,0,0,0);

   edit=varmanwidgets_entry_new(_("Get drive type"),
				"rec_inqdrive_getdrivetype",
				global_defs,APPLYMODE_BUTTON,160,320);
   gtk_box_pack_start(GTK_BOX(record_prefs),edit->visual,0,0,0);

   edit=varmanwidgets_entry_new(_("Get vendor name"),
				"rec_inqdrive_getvendor",
				global_defs,APPLYMODE_BUTTON,160,320);
   gtk_box_pack_start(GTK_BOX(record_prefs),edit->visual,0,0,0);

   edit=varmanwidgets_entry_new(_("Get drive identification"),
				"rec_inqdrive_getmodel",
				global_defs,APPLYMODE_BUTTON,160,320);
   gtk_box_pack_start(GTK_BOX(record_prefs),edit->visual,0,0,0);

   edit=varmanwidgets_entry_new(_("Get cdrom type"),
				"rec_inqdrive_gettype",
				global_defs,APPLYMODE_BUTTON,160,320);
   gtk_box_pack_start(GTK_BOX(record_prefs),edit->visual,0,0,0);

   edit=varmanwidgets_entry_new(_("Mountpoint lookup file"),
				"rec_inqdrive_getmountpoint_file",
				global_defs,APPLYMODE_BUTTON,160,320);
   gtk_box_pack_start(GTK_BOX(record_prefs),edit->visual,0,0,0);

   edit=varmanwidgets_entry_new(_("Get mountpoint with"),
				"rec_inqdrive_getmountpoint_exp",
				global_defs,APPLYMODE_BUTTON,160,320);
   gtk_box_pack_start(GTK_BOX(record_prefs),edit->visual,0,0,0);

   /* call record_bootimage_changed callback to set the initial state of
    * the "Make bootable" Checkbox */
   record_bootimage_changed(NULL,NULL);

   record_active=0; // cdwriter is initially not active
   return rec;
}
;

