/* This file is responsible for handling compilations, that is
 * the current layout of a CD. It will handle both filesystems and tracklists
 * and can load them as well as store them.
 * Respective handlers have to install themselves by specifying their
 * data structures. Only one data structure of a certain type can be registered
 * at a particular time. */

#include <stdlib.h>
#include <glib.h>
#include <string.h>

#include "xml.h"
#include "compilation.h"

GList *sectionhandlers=NULL; /* a list of section handlers */

typedef struct
{
   xml_taghandler_t *handlers;
   FILE *input;
   compilation_callback_t donecb;
   void *data;
} compilation_load_info;   

void compilation_loaddone(int status,
			  xml_context_t *context,
			  void *data)
{
   compilation_load_info *info=(compilation_load_info*)data;
   if (info->input)
     fclose(info->input);
   free(info->handlers);
   info->donecb(status,data);
   free(info);
};

/* load a compilation, callback may be NULL */
void compilation_load(const char *filename,
		      compilation_callback_t donecb,
		      void *data)
{
   compilation_load_info *info=(compilation_load_info*)malloc(sizeof(compilation_load_info));
   int i;
   GList *current=sectionhandlers;
   
   /* Fill out callback informations */
   info->donecb=donecb;
   info->data=data;
   
   /* Open Source file */
   info->input=fopen(filename,"r");
   
   /* Create xml handler list from our section handler GList */
   info->handlers=(xml_taghandler_t*)
     malloc(sizeof(xml_taghandler_t)*(g_list_length(sectionhandlers)+1));
   for (i=0;current!=NULL;current=current->next,++i)
     {      
	if (current->data)
	  memcpy(&info->handlers[i],current->data,sizeof(xml_taghandler_t));
	else
	  {	     
	     printf("compilation.c: Internal error loading compilation. Please send a bugreport to A.Eckleder@bigfoot.com\n");
	     /* Limit damage. Don't let the program crash.
	      * By decreasing our counter we make sure that 
	      * the current (invalid) handler entry will be overwritten
	      * by the next one and terminated if no more handler
	      * entries are defined.
	      * Nevertheless there's a problem with the program if this happens
	      * so let's hope that this code will never get executed */
	     --i;
	  };
     };
   /* Terminate list of handlers */   
   info->handlers[i].tag=NULL;
   info->handlers[i].handler=NULL;
   info->handlers[i].data=NULL;

   /* And off we go */
   xml_parse(info->input,
	     info->handlers,
	     compilation_loaddone,
	     (void*)info);
};
  
/* The two most certainly defined entries */
#define COMPILATION_FILESYSTEM "filesystem"
#define COMPILATION_TRACKLIST  "tracklist"

/* register a compilation component */
void compilation_registercomponent(const char *section,
				   xml_itemhandler_t handler,
				   void *data)
{
   xml_taghandler_t *info=(xml_taghandler_t*)malloc(sizeof(xml_taghandler_t));
   info->tag=strdup(section);
   info->handler=handler;
   info->data=data;
   sectionhandlers=g_list_prepend(sectionhandlers,(void*)info);
};
