/*
 * radauth.cxx
 *
 * RADIUS protocol authenticator module for GNU Gatekeeper. 
 * Please see docs/radauth.txt for more details.
 *
 * Copyright (c) 2003, Quarcom FHU, Michal Zygmuntowicz
 *
 * This work is published under the GNU Public License (GPL)
 * see file COPYING for details.
 * We also explicitely grant the right to link this code
 * with the OpenH323 library.
 *
 * $Log: radauth.cxx,v $
 * Revision 1.1.2.38  2004/05/25 19:54:05  zvision
 * Some minor warnings removed
 *
 * Revision 1.1.2.37  2004/05/21 11:40:37  zvision
 * Do not declare H.235 Auth Procedure I support as it is not implemented
 *
 * Revision 1.1.2.36  2004/05/12 14:00:47  zvision
 * Header file usage more consistent. Solaris std::map problems fixed.
 * Compilation warnings removed. VSNET2003 project files added. ANSI.h removed.
 *
 * Revision 1.1.2.35  2004/04/14 21:52:57  zvision
 * Fixed referencing RRQ terminalAlias field without checking for its presence
 *
 * Revision 1.1.2.34  2004/03/31 11:13:36  zvision
 * New CheckSetupUnregisteredOnly option for RadAliasAuth module
 *
 * Revision 1.1.2.33  2004/02/18 20:45:16  zvision
 * Prototype changed for GkAuthenticator::Check ARQ method
 *
 * Revision 1.1.2.32  2004/01/16 13:05:01  zvision
 * Better h323-ivr-in attrbiute handling
 *
 * Revision 1.1.2.31  2004/01/08 11:46:47  zvision
 * Fixed semicolon processing for h323-ivr-in reply attribute
 *
 * Revision 1.1.2.30  2004/01/07 20:37:26  zvision
 * RRQ endpoint alias control (add/remove) through RadAuth/RadAliasAuth modules
 *
 * Revision 1.1.2.29  2003/12/26 13:59:31  zvision
 * Fixed destination call signaling address handling
 *
 * Revision 1.1.2.28  2003/12/21 12:20:35  zvision
 * Fixed conditional compilation
 *
 * Revision 1.1.2.27  2003/12/02 12:47:37  zvision
 * Q.931 Setup authenticator now returns (and documents) Q.931 cause values instead of H225_ReleaseCompleteReason
 *
 * Revision 1.1.2.26  2003/11/22 13:43:48  zvision
 * Q.931 Setup Check now accepts const callptr to prevent it from modifications
 *
 * Revision 1.1.2.25  2003/11/18 23:38:51  zvision
 * Q.931 Setup authentication optimized
 *
 * Revision 1.1.2.24  2003/11/12 13:34:55  zvision
 * Fixed Session-Timeout handling in RadAliasAuth Setup check
 *
 * Revision 1.1.2.23  2003/10/03 00:59:49  zvision
 * Fixed bus errors on some system with dword alignment checking enabled.
 * Small optimizations and RadiusPDU API changes.
 *
 * Revision 1.1.2.22  2003/09/30 11:08:16  zvision
 * Fixed non-working Q.931 authenticator due to previous Check(...) signature
 * changes. Many thanks to Oleg Ustinov!
 *
 * Revision 1.1.2.21  2003/09/29 16:23:45  zvision
 * Cisco h323-xxx response attributes handling improved
 *
 * Revision 1.1.2.20  2003/09/25 11:12:46  zvision
 * Fixed Session-Timeout bug and more flexible h323-credit-time processing
 *
 * Revision 1.1.2.19  2003/09/24 10:19:44  zvision
 * Call duration limit for registered endpoints (through ARQ authenticators)
 *
 * Revision 1.1.2.18  2003/08/25 12:18:35  zvision
 * Added h323-ivr-out attribute with an alias list (thanks Mark Lipscombe)
 *
 * Revision 1.1.2.17  2003/07/31 22:59:24  zvision
 * Fixed IP address retrieval for unregistered endpoints
 *
 * Revision 1.1.2.16  2003/07/31 13:09:15  zvision
 * Added Q.931 Setup message authentication and call duration limit feature
 *
 * Revision 1.1.2.15  2003/07/17 14:40:39  zvision
 * Conditional compilation of features available only when HAS_ACCT is defined.
 *
 * Revision 1.1.2.14  2003/07/16 22:13:21  zvision
 * Fixed Radius attributes for answer call ARQs.
 *
 * Revision 1.1.2.13  2003/07/07 14:28:30  zvision
 * Added missing NAS-Identifier attribute in RadAliasAuth. Thanks Julius Stavaris.
 *
 * Revision 1.1.2.12  2003/07/07 12:02:55  zvision
 * Improved H.235 handling.
 *
 * Revision 1.1.2.11  2003/06/19 15:33:29  zvision
 * Removed static modifier from GetConferenceIDString function.
 *
 * Revision 1.1.2.10  2003/06/11 13:06:57  zvision
 * Added gk_const.h include directive (OPENH323_NEWVERSION macro definition)
 *
 * Revision 1.1.2.9  2003/06/11 12:14:35  zvision
 * Cosmetic changes
 *
 * Revision 1.1.2.8  2003/06/05 10:03:04  zvision
 * Small fix to h323-gw-id attribute.
 *
 * Revision 1.1.2.7  2003/05/29 17:21:22  zvision
 * Fixed compilation errors with OpenH323 versions prior to 1.11.5 (no H235AuthCAT)
 *
 * Revision 1.1.2.6  2003/05/28 13:25:19  zvision
 * Added alias based authentication (RadAliasAuth)
 *
 * Revision 1.1.2.5  2003/05/27 00:13:05  zvision
 * Smart Calling and Called -Station-Id selection (dialedDigits and partyNumber alias types preferred)
 *
 * Revision 1.1.2.4  2003/05/26 23:09:59  zvision
 * Added new OnSend and OnReceive hooks. LocalInterface config parameter introduced.
 *
 * Revision 1.1.2.3  2003/05/13 17:49:49  zvision
 * Removed acctPort. New includeFramedIP feature. Better tracing. Bug-fixes
 *
 * Revision 1.1.2.2  2003/04/29 14:56:27  zvision
 * Added H.235 capability matching
 *
 * Revision 1.1.2.1  2003/04/23 20:15:37  zvision
 * Initial revision
 *
 */
#ifdef HAS_RADIUS

#if (_MSC_VER >= 1200)
#pragma warning( disable : 4786 ) // warning about too long debug symbol off
#endif

#include <ptlib.h>
#include <h225ras.h>
#include <h323pdu.h>
#include "gk_const.h"
#include "gkauth.h"
#include "Toolkit.h"
#include "RasTbl.h"
#include "h323util.h"
#include "ProxyChannel.h"
#include "radauth.h"

namespace {
// Settings for H.235 based module will be stored inside [RadAuth] config section
const char* const RadAuthConfigSectionName = "RadAuth";
// Settings for alias based module will be stored inside [RadAliasAuth] config section
const char* const RadAliasAuthConfigSectionName = "RadAliasAuth";
// append RADIUS H.235 authenticator to the global list of authenticators
GkAuthInit<RadAuth> RAD_A("RadAuth");
// append RADIUS Alias authenticator to the global list of authenticators
GkAuthInit<RadAliasAuth> RAD_A_A("RadAliasAuth");
}

// OID for CAT (Cisco Access Token) algorithm
PString RadAuth::OID_CAT( "1.2.840.113548.10.1.2.1" );

PString GetConferenceIDString( const H225_ConferenceIdentifier& id )
{
	if( id.GetSize() < 16 )
		return PString::Empty();
		
	PString h323ConfId;
					
	for( int j = 0, i = 0; j < 4; j++ )
	{
		unsigned hex = (unsigned)(id[i++])<<24;
		hex |= (unsigned)(id[i++])<<16;
		hex |= (unsigned)(id[i++])<<8;
		hex |= (unsigned)(id[i++]);
							
		h323ConfId += PString( PString::Unsigned, (long)hex, 16 );
		if( j < 3 )
			h323ConfId += ' ';
	}

	return h323ConfId;
}

RadAuth::RadAuth( 
	PConfig* cfg, 
	const char* authName 
	)
	:
	GkAuthenticator( cfg, authName ),
	radiusServers( cfg->GetString(
			RadAuthConfigSectionName,"Servers",""
			).Tokenise( ";, |\t", FALSE ) 
		),
	sharedSecret( cfg->GetString(
			RadAuthConfigSectionName,"SharedSecret",""
			)
		),
	authPort( (WORD)(cfg->GetInteger(
			RadAuthConfigSectionName,"DefaultAuthPort"
			))
		),
	portBase( 1024 ),
	portMax( 65535 ),
	requestTimeout( cfg->GetInteger(
			RadAuthConfigSectionName,"RequestTimeout"
			)
		),
	idCacheTimeout( cfg->GetInteger(
			RadAuthConfigSectionName,"IdCacheTimeout"
			)
		),
	socketDeleteTimeout( cfg->GetInteger(
			RadAuthConfigSectionName,"SocketDeleteTimeout"
			)
		),
	numRequestRetransmissions( cfg->GetInteger(
			RadAuthConfigSectionName,"RequestRetransmissions"
			)
		),
	roundRobin( cfg->GetBoolean(
			RadAuthConfigSectionName,"RoundRobinServers", TRUE
			)
		),
	appendCiscoAttributes( cfg->GetBoolean(
			RadAuthConfigSectionName,"AppendCiscoAttributes", TRUE
			)
		),
	includeFramedIp( cfg->GetBoolean(
			RadAuthConfigSectionName, "IncludeEndpointIP", TRUE
			)
		),
	includeTerminalAliases( cfg->GetBoolean(
			RadAuthConfigSectionName, "IncludeTerminalAliases", TRUE
			)
		),
	localInterface( cfg->GetString(
			RadAuthConfigSectionName, "LocalInterface", ""
			)
		),
	radiusClient( NULL )
{
	if( radiusServers.GetSize() < 1 )
	{
		PTRACE(1,"RADAUTH\tCannot build RADIUS authenticator"
			" - no RADIUS servers specified in the config"
			);
		return;
	}

	if( (!localInterface.IsEmpty()) && (!PIPSocket::IsLocalHost(localInterface)) )
	{
		PTRACE(1,"RADAUTH\tSpecified local interface - "<<localInterface
			<<" - does not belong to this machine"
			);
		localInterface = PString::Empty();
	}
	/// build RADIUS client
	radiusClient = new RadiusClient( 
		radiusServers[0],
		(radiusServers.GetSize() > 1) ? radiusServers[1] : PString::Empty(),
		localInterface
		);

	/// if there were specified more than two RADIUS servers, append them
	for( int i = 2; i < radiusServers.GetSize(); i++ )
		radiusClient->AppendServer( radiusServers[i] );	
		
	radiusClient->SetSharedSecret( sharedSecret );
	radiusClient->SetRoundRobinServers( roundRobin );
		
	if( authPort > 0 )
		radiusClient->SetAuthPort( authPort );
		
	if( requestTimeout > 0 )
		radiusClient->SetRequestTimeout( requestTimeout );
	if( idCacheTimeout > 0 )
		radiusClient->SetIdCacheTimeout( idCacheTimeout );
	if( socketDeleteTimeout > 0 )
		radiusClient->SetSocketDeleteTimeout( socketDeleteTimeout );
	if( numRequestRetransmissions > 0 )
		radiusClient->SetRetryCount( numRequestRetransmissions );
	
	PStringArray s = cfg->GetString(
		RadAuthConfigSectionName,"RadiusPortRange",""
		).Tokenise( "-" );

	// parse port range		
	if( s.GetSize() >= 2 )
	{ 
		unsigned p1 = s[0].AsUnsigned();
		unsigned p2 = s[1].AsUnsigned();
	
		// swap if base is greater than max
		if( p2 < p1 )
		{
			const unsigned temp = p1;
			p1 = p2;
			p2 = temp;
		}
		
		if( p1 > 65535 )
			p1 = 65535;
		if( p2 > 65535 )
			p2 = 65535;
	
		if( (p1 > 0) && (p2 > 0) )
		{
			portBase = (WORD)p1;
			portMax = (WORD)p2;
		}
	}
	
	radiusClient->SetClientPortRange( portBase, portMax-portBase+1 );

	if( localInterface.IsEmpty() )
		localInterfaceAddr = Toolkit::Instance()->GetRouteTable()->GetLocalAddress();
	else
		localInterfaceAddr = PIPSocket::Address(localInterface);

	NASIdentifier = Toolkit::Instance()->GKName();

	if( h235Authenticators == NULL )
		h235Authenticators = new H235Authenticators;
		
	H235AuthCAT* authenticator = new H235AuthCAT;
	authenticator->SetLocalId("dummy");
	authenticator->SetRemoteId("dummy");
	authenticator->SetPassword("dummy");
	h235Authenticators->Append(authenticator);
}

RadAuth::~RadAuth()
{
	delete radiusClient;
}

BOOL RadAuth::CheckAliases( 
	const H225_ArrayOf_AliasAddress& aliases, 
	const PString& id 
	) const
{
	BOOL result = FALSE;
	
	for( PINDEX i = 0; i < aliases.GetSize(); i++ )
		if( H323GetAliasAddressString(aliases[i]) == id ) {
			result = TRUE;
			break;
		}
	
	return result;
}

int RadAuth::Check(
	H225_RegistrationRequest& rrq, 
	unsigned& rejectReason
	)
{
	if( radiusClient == NULL )
	{
		PTRACE(3,"RADAUTH\tRRQ Auth failed - NULL Radius client");
		if( defaultStatus == e_fail )
			rejectReason = H225_RegistrationRejectReason::e_undefinedReason;
		return defaultStatus;
	}

	// RRQ has to carry at least one terminalAlias
	if( !rrq.HasOptionalField(H225_RegistrationRequest::e_terminalAlias) )
	{
		if( defaultStatus == e_fail )
			rejectReason = H225_RegistrationRejectReason::e_securityDenial;
		return defaultStatus;
	}
		
	const H225_ArrayOf_AliasAddress& aliases = rrq.m_terminalAlias;
	
	// check for ClearTokens (CAT uses ClearTokens)
	if( rrq.HasOptionalField(H225_RegistrationRequest::e_tokens) )
	{
		const H225_ArrayOf_ClearToken& tokens = rrq.m_tokens;
		BOOL foundCAT = FALSE;
		
		// scan ClearTokens and find CATs
		for( PINDEX i = 0; i < tokens.GetSize(); i++ )
		{
			const H235_ClearToken& token = tokens[i];
			
			// is it CAT?
			if( token.m_tokenOID == OID_CAT )
			{
				foundCAT = TRUE;
				
				// these field are required for CAT
			  	if( !(token.HasOptionalField(H235_ClearToken::e_generalID)
					&& token.HasOptionalField(H235_ClearToken::e_random)
					&& token.HasOptionalField(H235_ClearToken::e_timeStamp)
					&& token.HasOptionalField(H235_ClearToken::e_challenge)) )
				{
					PTRACE(4,"RADAUTH\tRRQ Auth failed - CAT without all required fields");
					rejectReason = H225_RegistrationRejectReason::e_securityDenial;
					return e_fail;
				}
				
				// generalID should be present in the list of terminal aliases
				const PString id = token.m_generalID;
				if( !CheckAliases(aliases,id) )
				{
					PTRACE(4,"RADAUTH\tRRQ Auth failed - CAT m_generalID is not a valid alias");
					rejectReason = H225_RegistrationRejectReason::e_invalidTerminalAliases;
					return e_fail;
				}
					
				// CAT pseudo-random has to be one byte only
				const int randomInt = token.m_random;
					
				if( (randomInt < -127) || (randomInt > 255) )
				{
					PTRACE(4,"RADAUTH\tRRQ Auth failed - CAT m_random out of range");
					rejectReason = H225_RegistrationRejectReason::e_securityDenial;
					return e_fail;
				}
					
				// CAT challenge has to be 16 bytes
				if( token.m_challenge.GetValue().GetSize() < 16 )
				{
					PTRACE(4,"RADAUTH\tRRQ Auth failed - m_challenge less than 16 bytes");
					rejectReason = H225_RegistrationRejectReason::e_securityDenial;
					return e_fail;
				}
					
				// build RADIUS Access-Request
				RadiusPDU* pdu = radiusClient->BuildPDU();
				if( pdu == NULL )
				{
					PTRACE(3,"RADAUTH\tRRQ auth failed - could not to create Access-Request PDU");
					rejectReason = H225_RegistrationRejectReason::e_undefinedReason;
					continue;
				}

				pdu->SetCode( RadiusPDU::AccessRequest );
				
				// append User-Name
				*pdu += new RadiusAttr( RadiusAttr::UserName, id );

				// build CHAP-Password
				char password[17];
				password[0] = (BYTE)randomInt;
				memcpy(password+1,(const BYTE*)(token.m_challenge),16);
				
				*pdu += new RadiusAttr( RadiusAttr::ChapPassword,
					password,
					sizeof(password)
					);
				
				// Gk works as NAS point, so append NAS IP
				*pdu += new RadiusAttr( RadiusAttr::NasIpAddress, 
					localInterfaceAddr
					);
				// NAS-Identifier as Gk name
				*pdu += new RadiusAttr( RadiusAttr::NasIdentifier,
					NASIdentifier
					);
				// Gk does not have a concept of physical ports,
				// so define port type as NAS-Port-Virtual
				*pdu += new RadiusAttr( RadiusAttr::NasPortType, 
					RadiusAttr::NasPort_Virtual 
					);
				// RRQ service type is Login-User
				*pdu += new RadiusAttr( RadiusAttr::ServiceType,
					RadiusAttr::ST_Login
					);
					
				*pdu += new RadiusAttr( RadiusAttr::ChapChallenge,
					(int)(DWORD)token.m_timeStamp
					);
					
				if( includeFramedIp )
				{
					PIPSocket::Address addr;
				
					if( rrq.m_callSignalAddress.GetSize() > 0 )
					{
						if( GetIPFromTransportAddr(rrq.m_callSignalAddress[0],addr)
							&& addr.IsValid() )
							*pdu += new RadiusAttr( 
								RadiusAttr::FramedIpAddress,
								addr
								);
					}
					else if( rrq.m_rasAddress.GetSize() > 0 )
					{
						if( GetIPFromTransportAddr(rrq.m_rasAddress[0],addr) 
							&& addr.IsValid() )
							*pdu += new RadiusAttr( 
								RadiusAttr::FramedIpAddress,
								addr
								);
					}
				}

				if( appendCiscoAttributes && includeTerminalAliases )
				{
					PString aliasList( "terminal-alias:" );
					
					for( PINDEX i = 0; i < aliases.GetSize(); i++ )
					{
						if( i > 0 )
							aliasList += ",";
						aliasList += H323GetAliasAddressString(aliases[i]);
					}
					*pdu += new RadiusAttr(
						PString("h323-ivr-out=") + aliasList + PString(";"),
						9,	// Cisco
						1	// Cisco-AV-Pair
						);
				}
				
				// send request and wait for response
				RadiusPDU* response = NULL;
				BOOL result = OnSendPDU(*pdu,rrq,rejectReason)
					&& radiusClient->MakeRequest( *pdu, response ) 
					&& (response != NULL);
			
				delete pdu;
			
				if( !result )
				{
					delete response;
					return defaultStatus;
				}
				
				result = (response->GetCode() == RadiusPDU::AccessAccept);
				
				// process h323-ivr-in=terminal-alias attribute
				if( result ) {
					PINDEX index = response->FindVsaAttr( 9, 1 );
					bool found = false;
					while( index != P_MAX_INDEX && !found ) {
						PString h323ivrin;
						const RadiusAttr* attr = response->GetAttrAt(index);
						if( attr && attr->IsValid() )
							h323ivrin = attr->AsVsaString();
						if( h323ivrin.Find("h323-ivr-in=") == 0 
							&& ((index = h323ivrin.Find("terminal-alias:")) != P_MAX_INDEX) ) {
							found = true;
							index += strlen("terminal-alias:");
							const PINDEX semicolonpos = h323ivrin.Find(';',index);
							h323ivrin = h323ivrin.Mid(
								index, semicolonpos == P_MAX_INDEX
									? P_MAX_INDEX : (semicolonpos-index)
								);
							PStringArray aliases = h323ivrin.Tokenise(",");
							if( aliases.GetSize() > 0 
								&& rrq.HasOptionalField(H225_RegistrationRequest::e_terminalAlias) ) {
								PINDEX i = 0;
								while( i < rrq.m_terminalAlias.GetSize() ) {
									PINDEX j = aliases.GetStringsIndex(H323GetAliasAddressString(rrq.m_terminalAlias[i]));
									if( j == P_MAX_INDEX )
										rrq.m_terminalAlias.RemoveAt(i);
									else {
										i++;
										aliases.RemoveAt(j);
									}
								}
							}
							for( PINDEX i = 0; i < aliases.GetSize(); i++ ) {
								if( rrq.HasOptionalField(H225_RegistrationRequest::e_terminalAlias) )
									rrq.m_terminalAlias.SetSize(rrq.m_terminalAlias.GetSize()+1);
								else {
									rrq.IncludeOptionalField(H225_RegistrationRequest::e_terminalAlias);
									rrq.m_terminalAlias.SetSize(1);
								}
								H323SetAliasAddress(aliases[i],rrq.m_terminalAlias[rrq.m_terminalAlias.GetSize()-1]);
							}
						}
						if( !found )
							index = response->FindVsaAttr( 9, 1, index + 1 );
					}
				}

				if( result )
					result = OnReceivedPDU(*response,rrq,rejectReason);
				else
					rejectReason = H225_RegistrationRejectReason::e_securityDenial;
					
				delete response;
				return result ? e_ok : e_fail;
			}
		}
		
		if( !foundCAT )
			if( defaultStatus == e_fail )
				PTRACE(3,"RADAUTH\tRRQ Auth failed - m_tokens without CAT");
			else if( defaultStatus != e_ok )
				PTRACE(4,"RADAUTH\tRRQ Auth undetermined - m_tokens without CAT");
	}	
	else
		if( defaultStatus == e_fail )
			PTRACE(3,"RADAUTH\tRRQ Auth failed - no m_tokens");
		else if( defaultStatus != e_ok )
			PTRACE(4,"RADAUTH\tRRQ Auth undetermined - no m_tokens");

	if( defaultStatus == e_fail )
		rejectReason = H225_RegistrationRejectReason::e_securityDenial;
		
	return defaultStatus;
}

int RadAuth::Check(
	H225_AdmissionRequest& arq, 
	unsigned& rejectReason,
	long& callDurationLimit
	)
{
	if( radiusClient == NULL )
	{
		PTRACE(3,"RADAUTH\tARQ Auth failed - NULL Radius client");
		if( defaultStatus == e_fail )
			rejectReason = H225_AdmissionRejectReason::e_undefinedReason;
		return defaultStatus;
	}
	
	// check for ClearTokens
	if( arq.HasOptionalField(H225_AdmissionRequest::e_tokens) )
	{
		const H225_ArrayOf_ClearToken& tokens = arq.m_tokens;
		BOOL foundCAT = FALSE;
		
		// scan ClearTokens for CATs
		for( PINDEX i = 0; i < tokens.GetSize(); i++ )
		{
			const H235_ClearToken& token = tokens[i];
				
			// is it CAT?
			if( token.m_tokenOID == OID_CAT )
			{
				foundCAT = TRUE;
				
				// these field are required for CAT
			  	if( !(token.HasOptionalField(H235_ClearToken::e_generalID)
					&& token.HasOptionalField(H235_ClearToken::e_random)
					&& token.HasOptionalField(H235_ClearToken::e_timeStamp)
					&& token.HasOptionalField(H235_ClearToken::e_challenge)) )
				{
					PTRACE(4,"RADAUTH\tARQ Auth failed - CAT without all required fields");
					rejectReason = H225_AdmissionRejectReason::e_securityDenial;
					return e_fail;
				}
			
				const PString id = token.m_generalID;
					
				// CAT random has to be one byte only				
				const int randomInt = token.m_random;
					
				if( (randomInt < -127) || (randomInt > 255) )
				{
					PTRACE(4,"RADAUTH\tARQ Auth failed - CAT m_random out of range");
					rejectReason = H225_AdmissionRejectReason::e_securityDenial;
					return e_fail;
				}
					
				// CAT challenge has to be 16 bytes
				if( token.m_challenge.GetValue().GetSize() < 16 )
				{
					PTRACE(4,"RADAUTH\tARQ Auth failed - CAT m_challenge less than 16 bytes");
					rejectReason = H225_AdmissionRejectReason::e_securityDenial;
					return e_fail;
				}
					
				// build RADIUS Access-Request packet
				RadiusPDU* pdu = radiusClient->BuildPDU();
				if( pdu == NULL )
				{
					PTRACE(3,"RADAUTH\tARQ auth failed - could not to create Access-Request PDU");
					continue;
				}

				pdu->SetCode( RadiusPDU::AccessRequest );
				
				*pdu += new RadiusAttr( RadiusAttr::UserName, id );
				
				// build CHAP-Password
				char password[17];
				password[0] = (BYTE)randomInt;
				memcpy(password+1,(const BYTE*)(token.m_challenge),16);
				
				*pdu += new RadiusAttr( RadiusAttr::ChapPassword,
					password,
					sizeof(password)
					);
				
				// append CHAP-Challenge 
				*pdu += new RadiusAttr( RadiusAttr::ChapChallenge,
					(int)(DWORD)token.m_timeStamp
					);
				
				// Gk acts as NAS, so include NAS IP
				*pdu += new RadiusAttr( RadiusAttr::NasIpAddress, 
					localInterfaceAddr
					);
					
				// NAS-Identifier as Gk name
				*pdu += new RadiusAttr( RadiusAttr::NasIdentifier,
					NASIdentifier
					);
					
				// NAS-Port-Type as Virtual, since Gk does
				// not care about physical ports concept
				*pdu += new RadiusAttr( RadiusAttr::NasPortType, 
					RadiusAttr::NasPort_Virtual 
					);
					
				// Service-Type is Login-User if originating the call
				// and Call Check if answering the call
				*pdu += new RadiusAttr( RadiusAttr::ServiceType,
					arq.m_answerCall ? 
						RadiusAttr::ST_CallCheck : RadiusAttr::ST_Login
					);
				
				endptr ep = RegistrationTable::Instance()->FindByEndpointId(
					arq.m_endpointIdentifier
					);
					
				if( includeFramedIp )
				{
					PIPSocket::Address addr;
				
					if( arq.HasOptionalField( arq.m_answerCall 
							? arq.e_destCallSignalAddress : arq.e_srcCallSignalAddress ) )
					{
						if( (arq.m_answerCall 
							? GetIPFromTransportAddr(arq.m_destCallSignalAddress,addr)
							: GetIPFromTransportAddr(arq.m_srcCallSignalAddress,addr)) 
							&& addr.IsValid() )
							*pdu += new RadiusAttr( 
								RadiusAttr::FramedIpAddress,
								addr
								);
					}
					else if( ep && GetIPFromTransportAddr( ep->GetCallSignalAddress(), addr)
							&& addr.IsValid() )
						*pdu += new RadiusAttr( 
							RadiusAttr::FramedIpAddress,
							addr
							);
				}
				
				// fill Calling-Station-Id and Called-Station-Id fields
				
				// Calling-Station-Id
				// Priority:
				//	ARQ.m_srcInfo[0] (first alias)
				//	generalID from CAT ClearToken (if the call originator)
					
				PString stationId;
				
				if( arq.m_srcInfo.GetSize() > 0 )
					stationId = GetBestAliasAddressString(
						arq.m_srcInfo,
						H225_AliasAddress::e_dialedDigits,
						H225_AliasAddress::e_partyNumber,
						H225_AliasAddress::e_h323_ID
						);
				
				if( stationId.IsEmpty() && ep && (!arq.m_answerCall) )
					stationId = GetBestAliasAddressString(
						ep->GetAliases(),
						H225_AliasAddress::e_dialedDigits,
						H225_AliasAddress::e_partyNumber,
						H225_AliasAddress::e_h323_ID
						);
					
				if( stationId.IsEmpty() && (!arq.m_answerCall) )
					stationId = id;
				
				if( stationId.IsEmpty() && arq.m_answerCall )
				{
					callptr call;
					if( arq.HasOptionalField(arq.e_callIdentifier ) )
						call = CallTable::Instance()->FindCallRec(arq.m_callIdentifier);
					else
					{
						H225_CallReferenceValue crv;
						crv.SetValue((WORD)arq.m_callReferenceValue & 0x7fffu);
						call = CallTable::Instance()->FindCallRec(crv);
					}
					if( call )
					{
						endptr callingEP = call->GetCallingParty();
						if( callingEP && (callingEP->GetAliases().GetSize() > 0) )
							stationId = GetBestAliasAddressString(
								callingEP->GetAliases(),
								H225_AliasAddress::e_dialedDigits,
								H225_AliasAddress::e_partyNumber,
								H225_AliasAddress::e_h323_ID
								);
#ifdef HAS_ACCT
						if( stationId.IsEmpty() )
						{
							stationId = call->GetSourceInfo();
							const PINDEX index = stationId.FindOneOf(":");
							if( index != P_MAX_INDEX )
								stationId = stationId.Left(index);
						}
#endif
						if( stationId.IsEmpty() && call->HasSrcSignalAddr() )
						{
							PIPSocket::Address addr;
							WORD port = 0;
							if( call->GetSrcSignalAddr(addr,port) && addr.IsValid() )
								stationId = AsString(addr,port);
						}
					}
				}
				
				if( !stationId.IsEmpty() )		
					*pdu += new RadiusAttr( 
						RadiusAttr::CallingStationId, 
						stationId			
						);
						
				stationId = PString::Empty();
					
				// Called-Station-Id
				// Priority:
				//	ARQ.m_destinationInfo[0] (first alias)
				//	ARQ.m_destCallSignalAddress::e_ipAddress
				//  generalID from CAT token (if answering the call)
				if( arq.HasOptionalField(H225_AdmissionRequest::e_destinationInfo) 
					&& (arq.m_destinationInfo.GetSize() > 0) )
					stationId = GetBestAliasAddressString(
						arq.m_destinationInfo,
						H225_AliasAddress::e_dialedDigits,
						H225_AliasAddress::e_partyNumber,
						H225_AliasAddress::e_h323_ID
						);
				
				if( stationId.IsEmpty() && ep && arq.m_answerCall )
					stationId = GetBestAliasAddressString(
						ep->GetAliases(),
						H225_AliasAddress::e_dialedDigits,
						H225_AliasAddress::e_partyNumber,
						H225_AliasAddress::e_h323_ID
						);
						
				if( stationId.IsEmpty() 
					&& arq.HasOptionalField(H225_AdmissionRequest::e_destCallSignalAddress) )
				{
					const H225_TransportAddress& tsap = arq.m_destCallSignalAddress;
					stationId = AsDotString(tsap);
				}

				if( stationId.IsEmpty() && arq.m_answerCall )
					stationId = id;
					
				if( stationId.IsEmpty() )
				{
					delete pdu;
					PTRACE(3,"RADAUTH\tARQ Auth failed - neither m_destinationInfo"
						" nor m_destCallSignalAddress found"
						);
					return e_fail; 	// ARQ without one of the above fields
									// should be rejected
				}
				
				*pdu += new RadiusAttr( RadiusAttr::CalledStationId,
					stationId
					);
			
				if( appendCiscoAttributes )
				{
					*pdu += new RadiusAttr(
						PString("h323-conf-id=") 
							+ GetConferenceIDString(arq.m_conferenceID),
						9, // VendorId = Cisco
						24 // Connection-ID (h323-conf-id)
						);
					
					*pdu += new RadiusAttr(
						PString(arq.m_answerCall 
							? "h323-call-origin=answer" 
							: "h323-call-origin=originate"),
						9, // VendorId = Cisco
						26 // Call-Origin (h323-call-origin)
						);
					*pdu += new RadiusAttr(
						PString("h323-call-type=VoIP"),
						9, // VendorId = Cisco
						27 // Call-Type (h323-call-type)
						);
					*pdu += new RadiusAttr(
						PString("h323-gw-id=") + NASIdentifier,
						9, // VendorId = Cisco
						33 // Call-Type (h323-gw-id)
						);
				}
					
				// send the request and wait for a response
				RadiusPDU* response = NULL;
				BOOL result = OnSendPDU(*pdu,arq,rejectReason) 
					&& radiusClient->MakeRequest( *pdu, response ) 
					&& (response != NULL);
			
				delete pdu;
			
				if( !result )
				{
					delete response;
					return defaultStatus;
				}
				
				// authenticated?
				result = (response->GetCode() == RadiusPDU::AccessAccept);
				if( result ) {
					const PINDEX index = response->FindVsaAttr( 9, 103 );
					if( index != P_MAX_INDEX ) {
						const RadiusAttr* attr = response->GetAttrAt(index);
						BOOL valid = FALSE;
		
						if( attr && attr->IsValid() ) {
							PString s = attr->AsVsaString();
							if( s.Find("h323-return-code=") == 0 )
								s = s.Mid( s.FindOneOf("=") + 1 );
							if( s.GetLength() > 0
								&& strspn((const char*)s,"0123456789") == (size_t)s.GetLength() ) {
								unsigned retcode = s.AsUnsigned();
								if( retcode != 0 ) {
									PTRACE(5,"RADAUTH\t"<<GetName()<<" ARQ check failed - return code "<<retcode);
									result = FALSE;
								}
								valid = TRUE;
							}
						}
					
						if( !valid ) {
							PTRACE(5,"RADAUTH\t"<<GetName()<<" check failed - invalid h323-return-code attribute");
							result = FALSE;
						}
					}
				}
				
				if( result ) {
					BOOL found = FALSE;
					
					const PINDEX index = response->FindVsaAttr( 9, 102 );
					if( index != P_MAX_INDEX ) {
						const RadiusAttr* attr = response->GetAttrAt(index);
						if( attr && attr->IsValid() ) {
							PString s = attr->AsVsaString();
							if( s.Find("h323-credit-time=") == 0 )
								s = s.Mid( s.FindOneOf("=") + 1 );
							if( s.GetLength() > 0
								&& strspn((const char*)s,"0123456789") == (size_t)s.GetLength() ) {
								found = TRUE;
								callDurationLimit = s.AsInteger();
								PTRACE(5,"RADAUTH\t"<<GetName()<<" ARQ check set duration limit set: "<<callDurationLimit);
								if( callDurationLimit == 0 )
									result = FALSE;
							}
						}
						
						if( !found ) {
							PTRACE(5,"RADAUTH\t"<<GetName()<<" ARQ check failed - invalid h323-credit-time attribute: ");
							result = FALSE;
						}
					}
				}
			
				if( result ) {
					BOOL found = FALSE;
					
					const PINDEX index = response->FindAttr( RadiusAttr::SessionTimeout );
					if( index != P_MAX_INDEX ) {
						const RadiusAttr* attr = response->GetAttrAt(index);
						if( attr && attr->IsValid() ) {
							found = TRUE;
							const long sessionTimeout = attr->AsInteger();
							if( (callDurationLimit < 0) || (callDurationLimit > sessionTimeout) )
							{
								callDurationLimit = sessionTimeout;
								PTRACE(5,"RADAUTH\t"<<GetName()<<" ARQ check set duration limit set: "<<callDurationLimit);
							}
							if( callDurationLimit == 0 )
								result = FALSE;
						}
					
						if( !found ) {
							PTRACE(5,"RADAUTH\t"<<GetName()<<" ARQ check failed - invalid Session-Timeout attribute");
							result = FALSE;
						}
					}
				}
				if( result )
					result = OnReceivedPDU(*response,arq,rejectReason,callDurationLimit);
				else
					rejectReason = H225_AdmissionRejectReason::e_securityDenial;
					
				delete response;
				return result ? e_ok : e_fail;
			}
		}
		
		if( !foundCAT )
			if( defaultStatus == e_fail )
				PTRACE(3,"RADAUTH\tARQ Auth failed - m_tokens without CAT");
			else if( defaultStatus != e_ok )
				PTRACE(4,"RADAUTH\tARQ Auth undetermined - m_tokens without CAT");
	}	
	else
		if( defaultStatus == e_fail )
			PTRACE(3,"RADAUTH\tARQ Auth failed - no m_tokens");
		else if( defaultStatus != e_ok )
			PTRACE(4,"RADAUTH\tARQ Auth undetermined - no m_tokens");

	if( defaultStatus == e_fail )
		rejectReason = H225_AdmissionRejectReason::e_securityDenial;
	return defaultStatus;
}

BOOL RadAuth::OnSendPDU(
	RadiusPDU& /*pdu*/,
	const H225_RegistrationRequest& /*rrq*/,
	unsigned& /*rejectReason*/
	)
{
	return TRUE;
}

BOOL RadAuth::OnSendPDU(
	RadiusPDU& /*pdu*/,
	const H225_AdmissionRequest& /*rrq*/,
	unsigned& /*rejectReason*/
	)
{
	return TRUE;
}

BOOL RadAuth::OnReceivedPDU(
	RadiusPDU& /*pdu*/,
	H225_RegistrationRequest& /*rrq*/,
	unsigned& /*rejectReason*/
	)
{
	return TRUE;
}

BOOL RadAuth::OnReceivedPDU(
	RadiusPDU& /*pdu*/,
	H225_AdmissionRequest& /*arq*/,
	unsigned& /*rejectReason*/
	)
{
	return TRUE;
}

BOOL RadAuth::OnReceivedPDU(
	RadiusPDU& pdu,
	H225_AdmissionRequest& arq,
	unsigned& rejectReason,
	long& /*durationLimit*/
	)
{
	return OnReceivedPDU(pdu,arq,rejectReason);
}


RadAliasAuth::RadAliasAuth( 
	PConfig* cfg, 
	const char* authName 
	)
	:
	GkAuthenticator( cfg, authName ),
	radiusServers( cfg->GetString(
			RadAliasAuthConfigSectionName,"Servers",""
			).Tokenise( ";, |\t", FALSE ) 
		),
	sharedSecret( cfg->GetString(
			RadAliasAuthConfigSectionName,"SharedSecret",""
			)
		),
	authPort( (WORD)(cfg->GetInteger(
			RadAliasAuthConfigSectionName,"DefaultAuthPort"
			))
		),
	portBase( 1024 ),
	portMax( 65535 ),
	requestTimeout( cfg->GetInteger(
			RadAliasAuthConfigSectionName,"RequestTimeout"
			)
		),
	idCacheTimeout( cfg->GetInteger(
			RadAliasAuthConfigSectionName,"IdCacheTimeout"
			)
		),
	socketDeleteTimeout( cfg->GetInteger(
			RadAliasAuthConfigSectionName,"SocketDeleteTimeout"
			)
		),
	numRequestRetransmissions( cfg->GetInteger(
			RadAliasAuthConfigSectionName,"RequestRetransmissions"
			)
		),
	roundRobin( cfg->GetBoolean(
			RadAliasAuthConfigSectionName,"RoundRobinServers", TRUE
			)
		),
	appendCiscoAttributes( cfg->GetBoolean(
			RadAliasAuthConfigSectionName,"AppendCiscoAttributes", TRUE
			)
		),
	includeFramedIp( cfg->GetBoolean(
			RadAliasAuthConfigSectionName, "IncludeEndpointIP", TRUE
			)
		),
	includeTerminalAliases( cfg->GetBoolean(
			RadAliasAuthConfigSectionName,"IncludeTerminalAliases", TRUE
			)
		),
	checkSetupUnregisteredOnly( Toolkit::AsBool(
		cfg->GetString(
			RadAliasAuthConfigSectionName, "CheckSetupUnregisteredOnly", "0"
			))
		),
	localInterface( cfg->GetString(
			RadAliasAuthConfigSectionName, "LocalInterface", ""
			)
		),
	fixedUsername( cfg->GetString(
			RadAliasAuthConfigSectionName, "FixedUsername", ""
			)
		),
	fixedPassword( cfg->GetString(
			RadAliasAuthConfigSectionName, "FixedPassword", ""
			)
		),
	radiusClient( NULL )
{
	if( radiusServers.GetSize() < 1 )
	{
		PTRACE(1,"RADAUTH\tCannot build RADIUS Alias authenticator"
			" - no RADIUS servers specified in the config"
			);
		return;
	}

	if( (!localInterface.IsEmpty()) && (!PIPSocket::IsLocalHost(localInterface)) )
	{
		PTRACE(1,"RADAUTH\tSpecified local interface - "<<localInterface
			<<" - does not belong to this machine"
			);
		localInterface = PString::Empty();
	}
	/// build RADIUS client
	radiusClient = new RadiusClient( 
		radiusServers[0],
		(radiusServers.GetSize() > 1) ? radiusServers[1] : PString::Empty(),
		localInterface
		);

	/// if there were specified more than two RADIUS servers, append them
	for( int i = 2; i < radiusServers.GetSize(); i++ )
		radiusClient->AppendServer( radiusServers[i] );	
		
	radiusClient->SetSharedSecret( sharedSecret );
	radiusClient->SetRoundRobinServers( roundRobin );
		
	if( authPort > 0 )
		radiusClient->SetAuthPort( authPort );
		
	if( requestTimeout > 0 )
		radiusClient->SetRequestTimeout( requestTimeout );
	if( idCacheTimeout > 0 )
		radiusClient->SetIdCacheTimeout( idCacheTimeout );
	if( socketDeleteTimeout > 0 )
		radiusClient->SetSocketDeleteTimeout( socketDeleteTimeout );
	if( numRequestRetransmissions > 0 )
		radiusClient->SetRetryCount( numRequestRetransmissions );
	
	PStringArray s = cfg->GetString(
		RadAliasAuthConfigSectionName,"RadiusPortRange",""
		).Tokenise( "-" );

	// parse port range		
	if( s.GetSize() >= 2 )
	{ 
		unsigned p1 = s[0].AsUnsigned();
		unsigned p2 = s[1].AsUnsigned();
	
		// swap if base is greater than max
		if( p2 < p1 )
		{
			const unsigned temp = p1;
			p1 = p2;
			p2 = temp;
		}
		
		if( p1 > 65535 )
			p1 = 65535;
		if( p2 > 65535 )
			p2 = 65535;
	
		if( (p1 > 0) && (p2 > 0) )
		{
			portBase = (WORD)p1;
			portMax = (WORD)p2;
		}
	}
	
	if( localInterface.IsEmpty() )
		localInterfaceAddr = Toolkit::Instance()->GetRouteTable()->GetLocalAddress();
	else
		localInterfaceAddr = PIPSocket::Address(localInterface);

	NASIdentifier = Toolkit::Instance()->GKName();
	
	radiusClient->SetClientPortRange(portBase, (WORD)(portMax-portBase+1));
}

RadAliasAuth::~RadAliasAuth()
{
	delete radiusClient;
}

BOOL RadAliasAuth::CheckAliases( 
	const H225_ArrayOf_AliasAddress& aliases, 
	const PString& id 
	) const
{
	BOOL result = FALSE;
	
	for( PINDEX i = 0; i < aliases.GetSize(); i++ )
		if( H323GetAliasAddressString(aliases[i]) == id )
		{
			result = TRUE;
			break;
		}
	
	return result;
}

int RadAliasAuth::Check(
	H225_RegistrationRequest& rrq, 
	unsigned& rejectReason
	)
{
	if( radiusClient == NULL )
	{
		PTRACE(3,"RADAUTH\tRRQ AliasAuth failed - NULL RADIUS Client");
		if( defaultStatus != e_ok )
			rejectReason = H225_ReleaseCompleteReason::e_undefinedReason;
		return defaultStatus;
	}

	PString username;				
	PIPSocket::Address addr;

	if( rrq.HasOptionalField(H225_RegistrationRequest::e_terminalAlias) 
		&& (rrq.m_terminalAlias.GetSize() > 0) )
		username = GetBestAliasAddressString(
			rrq.m_terminalAlias,
			H225_AliasAddress::e_h323_ID,
			H225_AliasAddress::e_dialedDigits
			);

	if( username.IsEmpty() && (rrq.m_callSignalAddress.GetSize() > 0) )
		if( GetIPFromTransportAddr(rrq.m_callSignalAddress[0],addr) 
			&& addr.IsValid() )
			username = addr.AsString();
	
	if( username.IsEmpty() && (rrq.m_rasAddress.GetSize() > 0) )
		if( GetIPFromTransportAddr(rrq.m_rasAddress[0],addr) 
			&& addr.IsValid() )
			username = addr.AsString();
							
	if( username.IsEmpty() 
		&& rrq.HasOptionalField(H225_RegistrationRequest::e_endpointIdentifier) )
		username = rrq.m_endpointIdentifier;

	if( username.IsEmpty() && fixedUsername.IsEmpty() )
	{
		PTRACE(2,"RADAUTH\tRRQ AliasAuth failed - neither FixedUsername"
			" nor alias inside RRQ were found"
			);
		rejectReason = H225_RegistrationRejectReason::e_securityDenial;
		return defaultStatus;
	}

	// build RADIUS Access-Request
	RadiusPDU* pdu = radiusClient->BuildPDU();
	if( pdu == NULL )
	{
		if( defaultStatus == e_fail )
			PTRACE(3,"RADAUTH\tRRQ AliasAuth failed - could not to create Access-Request PDU");
		else if( defaultStatus != e_ok )
			PTRACE(4,"RADAUTH\tRRQ AliasAuth undetermined - could not to create Access-Request PDU");
		return defaultStatus;
	}

	pdu->SetCode( RadiusPDU::AccessRequest );
	
	// append User-Name
    *pdu += new RadiusAttr( RadiusAttr::UserName, 
		fixedUsername.IsEmpty() ? username : fixedUsername 
		);
	*pdu += new RadiusAttr( RadiusAttr::UserPassword, 
		fixedPassword.IsEmpty() 
			? (fixedUsername.IsEmpty() ? username : fixedUsername)
			: fixedPassword  
			);
			
	// Gk works as NAS point, so append NAS IP
	*pdu += new RadiusAttr( RadiusAttr::NasIpAddress, localInterfaceAddr );
	// NAS-Identifier as Gk name
	*pdu += new RadiusAttr( RadiusAttr::NasIdentifier, NASIdentifier );
	*pdu += new RadiusAttr( RadiusAttr::NasPortType, RadiusAttr::NasPort_Virtual );
	*pdu += new RadiusAttr( RadiusAttr::ServiceType, RadiusAttr::ST_Login );
	
	if( includeFramedIp )
	{
		if( rrq.m_callSignalAddress.GetSize() > 0 )
		{
			if( GetIPFromTransportAddr(rrq.m_callSignalAddress[0],addr)
				&& addr.IsValid() )
				*pdu += new RadiusAttr( 
					RadiusAttr::FramedIpAddress,
					addr
					);
		}
		else if( rrq.m_rasAddress.GetSize() > 0 )
		{
			if( GetIPFromTransportAddr(rrq.m_rasAddress[0],addr) 
				&& addr.IsValid() )
				*pdu += new RadiusAttr( 
					RadiusAttr::FramedIpAddress,
					addr
					);
		}
	}

	if( appendCiscoAttributes && includeTerminalAliases
		&& rrq.HasOptionalField(H225_RegistrationRequest::e_terminalAlias) )
	{
		PString aliasList( "terminal-alias:" );
		for( PINDEX i = 0; i < rrq.m_terminalAlias.GetSize(); i++ )
		{
			if( i > 0 )
				aliasList += ",";
			aliasList += H323GetAliasAddressString(rrq.m_terminalAlias[i]);
		}
		*pdu += new RadiusAttr(
			PString("h323-ivr-out=") + aliasList + PString(";"),
			9,      // Cisco
			1       // Cisco-AV-Pair
			);
	}
					
	// send request and wait for response
	RadiusPDU* response = NULL;
	BOOL result = OnSendPDU(*pdu,rrq,rejectReason)
		&& radiusClient->MakeRequest( *pdu, response ) 
		&& (response != NULL);
			
	delete pdu;
			
	if( !result )
	{
		delete response;
		if( defaultStatus == e_fail )
			PTRACE(3,"RADAUTH\tRRQ AliasAuth failed - could not send Access-Request PDU");
		else if( defaultStatus != e_ok )
			PTRACE(4,"RADAUTH\tRRQ AliasAuth undetermined - could not send Access-Request PDU");
		return defaultStatus;
	}
				
	result = (response->GetCode() == RadiusPDU::AccessAccept);
	
	// process h323-ivr-in=terminal-alias attribute
	if( result ) {
		PINDEX index = response->FindVsaAttr( 9, 1 );
		bool found = false;
		while( index != P_MAX_INDEX && !found ) {
			PString h323ivrin;
			const RadiusAttr* attr = response->GetAttrAt(index);
			if( attr && attr->IsValid() )
				h323ivrin = attr->AsVsaString();
			if( h323ivrin.Find("h323-ivr-in=") == 0 
				&& ((index = h323ivrin.Find("terminal-alias:")) != P_MAX_INDEX) ) {
				found = true;
				index += strlen("terminal-alias:");
				const PINDEX semicolonpos = h323ivrin.Find(';',index);
				h323ivrin = h323ivrin.Mid(
					index, semicolonpos == P_MAX_INDEX
						? P_MAX_INDEX : (semicolonpos-index)
					);
				PStringArray aliases = h323ivrin.Tokenise(",");
				if( aliases.GetSize() > 0 
					&& rrq.HasOptionalField(H225_RegistrationRequest::e_terminalAlias) ) {
					PINDEX i = 0;
					while( i < rrq.m_terminalAlias.GetSize() ) {
						PINDEX j = aliases.GetStringsIndex(H323GetAliasAddressString(rrq.m_terminalAlias[i]));
						if( j == P_MAX_INDEX )
							rrq.m_terminalAlias.RemoveAt(i);
						else {
							i++;
							aliases.RemoveAt(j);
						}
					}
				}
				for( PINDEX i = 0; i < aliases.GetSize(); i++ ) {
					if( rrq.HasOptionalField(H225_RegistrationRequest::e_terminalAlias) )
						rrq.m_terminalAlias.SetSize(rrq.m_terminalAlias.GetSize()+1);
					else {
						rrq.IncludeOptionalField(H225_RegistrationRequest::e_terminalAlias);
						rrq.m_terminalAlias.SetSize(1);
					}
					H323SetAliasAddress(aliases[i],rrq.m_terminalAlias[rrq.m_terminalAlias.GetSize()-1]);
				}
			}
			if( !found )
				index = response->FindVsaAttr( 9, 1, index + 1 );
		}
	}
	
	if( result )
		result = OnReceivedPDU(*response,rrq,rejectReason);
	else
		rejectReason = H225_RegistrationRejectReason::e_securityDenial;
					
	delete response;
	return result ? e_ok : e_fail;
}

int RadAliasAuth::Check(
	H225_AdmissionRequest& arq, 
	unsigned& rejectReason,
	long& callDurationLimit
	)
{
	if( radiusClient == NULL )
	{
		PTRACE(3,"RADAUTH\tARQ AliasAuth failed - NULL Radius client");
		if( defaultStatus == e_fail )
			rejectReason = H225_AdmissionRejectReason::e_undefinedReason;
		return defaultStatus;
	}
	
	PString username;				
	PIPSocket::Address addr;

	endptr ep = RegistrationTable::Instance()->FindByEndpointId(
		arq.m_endpointIdentifier
		);
		
	if( ep && (ep->GetAliases().GetSize() > 0) )
		username = GetBestAliasAddressString(
			ep->GetAliases(),
			H225_AliasAddress::e_h323_ID,
			H225_AliasAddress::e_dialedDigits
			);

	if( username.IsEmpty() && (arq.m_srcInfo.GetSize() > 0) )
		username = GetBestAliasAddressString(
			arq.m_srcInfo,
			H225_AliasAddress::e_h323_ID,
			H225_AliasAddress::e_dialedDigits
			);
							
	if( username.IsEmpty() 
		&& arq.HasOptionalField(H225_AdmissionRequest::e_srcCallSignalAddress) )
	{
		if( GetIPFromTransportAddr(arq.m_srcCallSignalAddress,addr) 
			&& addr.IsValid() )
			username = addr.AsString();
	}
	
	if( username.IsEmpty() && ep )
	{
		if( GetIPFromTransportAddr(ep->GetCallSignalAddress(),addr) 
			&& addr.IsValid() )
			username = addr.AsString();
	}
	
	if( username.IsEmpty() )
		username = arq.m_endpointIdentifier;

	if( username.IsEmpty() && fixedUsername.IsEmpty() )
	{
		PTRACE(2,"RADAUTH\tARQ AliasAuth failed - neither FixedUsername"
			" nor alias inside ARQ were found"
			);
		rejectReason = H225_AdmissionRejectReason::e_securityDenial;
		return defaultStatus;
	}

	// build RADIUS Access-Request packet
	RadiusPDU* pdu = radiusClient->BuildPDU();
	if( pdu == NULL )
	{
		if( defaultStatus == e_fail )
			PTRACE(3,"RADAUTH\tARQ AliasAuth failed - could not to create Access-Request PDU");
		else if( defaultStatus != e_ok )
			PTRACE(4,"RADAUTH\tARQ AliasAuth undetermined - could not to create Access-Request PDU");
		return defaultStatus;
	}

	pdu->SetCode( RadiusPDU::AccessRequest );
	
	// append User-Name
    *pdu += new RadiusAttr( RadiusAttr::UserName, 
		fixedUsername.IsEmpty() ? username : fixedUsername 
		);
		
	*pdu += new RadiusAttr( RadiusAttr::UserPassword, 
		fixedPassword.IsEmpty() 
			? (fixedUsername.IsEmpty() ? username : fixedUsername)
			: fixedPassword  
			);
			
	// Gk works as NAS point, so append NAS IP
	*pdu += new RadiusAttr( RadiusAttr::NasIpAddress, localInterfaceAddr );
	// NAS-Identifier as Gk name
	*pdu += new RadiusAttr( RadiusAttr::NasIdentifier, NASIdentifier );
	*pdu += new RadiusAttr( RadiusAttr::NasPortType, RadiusAttr::NasPort_Virtual );
	// Service-Type is Login-User if originating the call
	// and Call Check if answering the call
	*pdu += new RadiusAttr( RadiusAttr::ServiceType,
		arq.m_answerCall ? 
			RadiusAttr::ST_CallCheck : RadiusAttr::ST_Login
		);
				
	if( includeFramedIp )
	{
		if( arq.HasOptionalField( arq.m_answerCall 
				? arq.e_destCallSignalAddress : arq.e_srcCallSignalAddress ) )
		{
			if( (arq.m_answerCall 
				? GetIPFromTransportAddr(arq.m_destCallSignalAddress,addr)
				: GetIPFromTransportAddr(arq.m_srcCallSignalAddress,addr))
				&& addr.IsValid() )
				*pdu += new RadiusAttr( 
					RadiusAttr::FramedIpAddress,
					addr
					);
		}
		else if( ep && GetIPFromTransportAddr( ep->GetCallSignalAddress(), addr)
				&& addr.IsValid() )
			*pdu += new RadiusAttr( 
				RadiusAttr::FramedIpAddress,
				addr
				);
	}
				
	// fill Calling-Station-Id and Called-Station-Id fields
				
	// Calling-Station-Id
	// Priority:
	//	ARQ.m_srcInfo[0] (first alias)
	//	generalID from CAT ClearToken (if the call originator)
					
	PString stationId;
				
	if( arq.m_srcInfo.GetSize() > 0 )
		stationId = GetBestAliasAddressString(
			arq.m_srcInfo,
			H225_AliasAddress::e_dialedDigits,
			H225_AliasAddress::e_partyNumber,
			H225_AliasAddress::e_h323_ID
			);
			
	if( stationId.IsEmpty() && ep && (!arq.m_answerCall) )
		stationId = GetBestAliasAddressString(
			ep->GetAliases(),
			H225_AliasAddress::e_dialedDigits,
			H225_AliasAddress::e_partyNumber,
			H225_AliasAddress::e_h323_ID
			);
			
	if( stationId.IsEmpty() && (!arq.m_answerCall) )
		stationId = fixedUsername.IsEmpty() ? username : fixedUsername;
				
	if( stationId.IsEmpty() && arq.m_answerCall )
	{
		callptr call;
		if( arq.HasOptionalField(arq.e_callIdentifier ) )
			call = CallTable::Instance()->FindCallRec(arq.m_callIdentifier);
		else
		{
			H225_CallReferenceValue crv;
			crv.SetValue((WORD)arq.m_callReferenceValue & 0x7fffu);
			call = CallTable::Instance()->FindCallRec(crv);
		}
		if( call )
		{
			endptr callingEP = call->GetCallingParty();
			if( callingEP && (callingEP->GetAliases().GetSize() > 0) )
				stationId = GetBestAliasAddressString(
					callingEP->GetAliases(),
					H225_AliasAddress::e_dialedDigits,
					H225_AliasAddress::e_partyNumber,
					H225_AliasAddress::e_h323_ID
					);
#ifdef HAS_ACCT
			if( stationId.IsEmpty() )
			{
				stationId = call->GetSourceInfo();
				const PINDEX index = stationId.FindOneOf(":");
				if( index != P_MAX_INDEX )
					stationId = stationId.Left(index);
			}
#endif
			if( stationId.IsEmpty() && call->HasSrcSignalAddr() )
			{
				PIPSocket::Address addr;
				WORD port;
				if( call->GetSrcSignalAddr(addr,port) && addr.IsValid() )
					stationId = AsString(addr,port);
			}
		}
	}
	
	if( !stationId.IsEmpty() )		
		*pdu += new RadiusAttr( 
			RadiusAttr::CallingStationId, 
			stationId			
			);
						
	stationId = PString::Empty();
				
	// Called-Station-Id
	// Priority:
	//	ARQ.m_destinationInfo[0] (first alias)
	//	ARQ.m_destCallSignalAddress::e_ipAddress
	//  generalID from CAT token (if answering the call)
	if( arq.HasOptionalField(H225_AdmissionRequest::e_destinationInfo) 
		&& (arq.m_destinationInfo.GetSize() > 0) )
		stationId = GetBestAliasAddressString(
			arq.m_destinationInfo,
			H225_AliasAddress::e_dialedDigits,
			H225_AliasAddress::e_partyNumber,
			H225_AliasAddress::e_h323_ID
			);
				
	if( stationId.IsEmpty() 
		&& arq.HasOptionalField(H225_AdmissionRequest::e_destCallSignalAddress) )
	{
		const H225_TransportAddress& tsap = arq.m_destCallSignalAddress;
		stationId = AsDotString(tsap);
	}
				
	if( stationId.IsEmpty() && arq.m_answerCall )
		stationId = fixedUsername.IsEmpty() ? username : fixedUsername;
					
	if( stationId.IsEmpty() )
	{
		delete pdu;
		PTRACE(3,"RADAUTH\tARQ AliasAuth failed - neither m_destinationInfo"
			" nor m_destCallSignalAddress found"
			);
		return e_fail; 	// ARQ without one of the above fields
						// should be rejected
	}
				
	*pdu += new RadiusAttr( RadiusAttr::CalledStationId, stationId );
			
	if( appendCiscoAttributes )
	{
		*pdu += new RadiusAttr(
			PString("h323-conf-id=") 
				+ GetConferenceIDString(arq.m_conferenceID),
			9, // VendorId = Cisco
			24 // Connection-ID (h323-conf-id)
			);
					
		*pdu += new RadiusAttr(
			PString(arq.m_answerCall 
				? "h323-call-origin=answer" 
				: "h323-call-origin=originate"),
			9, // VendorId = Cisco
			26 // Call-Origin (h323-call-origin)
			);
		*pdu += new RadiusAttr(
			PString("h323-call-type=VoIP"),
			9, // VendorId = Cisco
			27 // Call-Type (h323-call-type)
			);
		*pdu += new RadiusAttr(
			PString("h323-gw-id=") + NASIdentifier,
			9, // VendorId = Cisco
			33 // Call-Type (h323-gw-id)
			);
	}
					
	// send the request and wait for a response
	RadiusPDU* response = NULL;
	BOOL result = OnSendPDU(*pdu,arq,rejectReason) 
		&& radiusClient->MakeRequest( *pdu, response ) 
		&& (response != NULL);
			
	delete pdu;
			
	if( !result )
	{
		delete response;
		return defaultStatus;
	}
				
	// authenticated?
	result = (response->GetCode() == RadiusPDU::AccessAccept);
	if( result ) {
		const PINDEX index = response->FindVsaAttr( 9, 103 );
		if( index != P_MAX_INDEX ) {
			const RadiusAttr* attr = response->GetAttrAt(index);
			BOOL valid = FALSE;
		
			if( attr && attr->IsValid() ) {
				PString s = attr->AsVsaString();
				if( s.Find("h323-return-code=") == 0 )
					s = s.Mid( s.FindOneOf("=") + 1 );
				if( s.GetLength() > 0
					&& strspn((const char*)s,"0123456789") == (size_t)s.GetLength() ) {
					unsigned retcode = s.AsUnsigned();
					if( retcode != 0 ) {
						PTRACE(5,"RADAUTH\t"<<GetName()<<" ARQ check failed - return code "<<retcode);
						result = FALSE;
					}
					valid = TRUE;
				}
			}
		
			if( !valid ) {
				PTRACE(5,"RADAUTH\t"<<GetName()<<" check failed - invalid h323-return-code attribute");
				result = FALSE;
			}
		}
	}
	
	if( result ) {
		BOOL found = FALSE;
		
		const PINDEX index = response->FindVsaAttr( 9, 102 );
		if( index != P_MAX_INDEX ) {
			const RadiusAttr* attr = response->GetAttrAt(index);
			if( attr && attr->IsValid() ) {
				PString s = attr->AsVsaString();
				if( s.Find("h323-credit-time=") == 0 )
					s = s.Mid( s.FindOneOf("=") + 1 );
				if( s.GetLength() > 0
					&& strspn((const char*)s,"0123456789") == (size_t)s.GetLength() ) {
					found = TRUE;
					callDurationLimit = s.AsInteger();
					PTRACE(5,"RADAUTH\t"<<GetName()<<" ARQ check set duration limit set: "<<callDurationLimit);
					if( callDurationLimit == 0 )
						result = FALSE;
				}
			}
			
			if( !found ) {
				PTRACE(5,"RADAUTH\t"<<GetName()<<" ARQ check failed - invalid h323-credit-time attribute: ");
				result = FALSE;
			}
		}
	}
	
	if( result ) {
		BOOL found = FALSE;
		
		const PINDEX index = response->FindAttr( RadiusAttr::SessionTimeout );
		if( index != P_MAX_INDEX ) {
			const RadiusAttr* attr = response->GetAttrAt(index);
			if( attr && attr->IsValid() ) {
				found = TRUE;
				const long sessionTimeout = attr->AsInteger();
				if( (callDurationLimit < 0) || (callDurationLimit > sessionTimeout) )
				{
					callDurationLimit = sessionTimeout;
					PTRACE(5,"RADAUTH\t"<<GetName()<<" ARQ check set duration limit set: "<<callDurationLimit);
				}
				if( callDurationLimit == 0 )
					result = FALSE;
			}
			
			if( !found ) {
				PTRACE(5,"RADAUTH\t"<<GetName()<<" ARQ check failed - invalid Session-Timeout attribute");
				result = FALSE;
			}
		}
	}
	
	if( result )
		result = OnReceivedPDU(*response,arq,rejectReason,callDurationLimit);
	else
		rejectReason = H225_AdmissionRejectReason::e_securityDenial;
					
	delete response;
	return result ? e_ok : e_fail;
}

int RadAliasAuth::Check(
	Q931& q931pdu,
	H225_Setup_UUIE& setup, 
	callptr& call,
	unsigned& releaseCompleteCause,
	long& durationLimit
	)
{
	if( radiusClient == NULL ) {
		PTRACE(4,"RADAUTH\t"<<GetName()<<" Setup check failed - NULL RADIUS Client");
		releaseCompleteCause = Q931::TemporaryFailure;
		return defaultStatus;
	}

	CallRec* callrec = call.operator->();
	callptr tempcall;
	
	if( (!callrec) && setup.HasOptionalField(H225_Setup_UUIE::e_callIdentifier) ) {
		tempcall = CallTable::Instance()->FindCallRec(setup.m_callIdentifier);
		callrec = tempcall.operator->();
	}
	
	endptr callingEP;

	if( callrec )
		callingEP = callrec->GetCallingParty();

	if (checkSetupUnregisteredOnly && callingEP) {
		PTRACE(4,"RADAUTH\t"<<GetName()<<" Setup unconditionally accepted for registered endpoint");
		return e_ok;
	}
	
	if( (!callingEP) && setup.HasOptionalField(H225_Setup_UUIE::e_endpointIdentifier) )
		callingEP = RegistrationTable::Instance()->FindByEndpointId( 
				setup.m_endpointIdentifier
				);

	// get the IP address for Framed-IP-Address attribute
	PIPSocket::Address framedIP(0);
	WORD framedPort = 0;
	
	if( !(callrec && callrec->HasSrcSignalAddr() && callrec->GetSrcSignalAddr(framedIP,framedPort)) )
		if( !((setup.HasOptionalField(setup.e_sourceCallSignalAddress) 
				&& GetIPFromTransportAddr(setup.m_sourceCallSignalAddress,framedIP))
			|| (callingEP 
				&& GetIPFromTransportAddr(callingEP->GetCallSignalAddress(),framedIP))) )
			PTRACE(2,"RADAUTH\t"<<GetName()<<" Setup check could not determine Framed-IP-Address");
					 
	// get the username for User-Name attribute		
	PString username;				
	
	if(	callingEP && (callingEP->GetAliases().GetSize() > 0) )
		username = GetBestAliasAddressString(callingEP->GetAliases(),
				H225_AliasAddress::e_h323_ID,
				H225_AliasAddress::e_dialedDigits
				);
	
	if( username.IsEmpty() && setup.HasOptionalField(setup.e_sourceAddress) 
		&& (setup.m_sourceAddress.GetSize() > 0) )
		username = GetBestAliasAddressString( setup.m_sourceAddress,
			H225_AliasAddress::e_h323_ID,
			H225_AliasAddress::e_dialedDigits
			);
	
	if( username.IsEmpty() && framedIP.IsValid() )
		username = framedIP.AsString();
	
	if( username.IsEmpty() && fixedUsername.IsEmpty() ) {
		PTRACE(2,"RADAUTH\t"<<GetName()<<" Setup check failed - neither FixedUsername"
			" nor alias and IP address inside Setup were found"
			);
		releaseCompleteCause = Q931::CallRejected;
		return defaultStatus;
	}

	// build RADIUS Access-Request
	RadiusPDU* pdu = radiusClient->BuildPDU();
	if( pdu == NULL ) {
		if( defaultStatus == e_fail )
			PTRACE(3,"RADAUTH\t"<<GetName()<<" Setup check failed - could not to create Access-Request PDU");
		else if( defaultStatus != e_ok )
			PTRACE(4,"RADAUTH\t"<<GetName()<<" Setup check undetermined - could not to create Access-Request PDU");
		releaseCompleteCause = Q931::TemporaryFailure;
		return defaultStatus;
	}

	pdu->SetCode( RadiusPDU::AccessRequest );
	
	// append User-Name
    *pdu += new RadiusAttr( RadiusAttr::UserName, 
		fixedUsername.IsEmpty() ? username : fixedUsername 
		);
		
	*pdu += new RadiusAttr( RadiusAttr::UserPassword, 
		fixedPassword.IsEmpty() 
			? (fixedUsername.IsEmpty() ? username : fixedUsername)
			: fixedPassword  
			);
			
	// Gk works as NAS point, so append NAS IP
	*pdu += new RadiusAttr( RadiusAttr::NasIpAddress, localInterfaceAddr );
	// NAS-Identifier as Gk name
	*pdu += new RadiusAttr( RadiusAttr::NasIdentifier, NASIdentifier );
	*pdu += new RadiusAttr( RadiusAttr::NasPortType, RadiusAttr::NasPort_Virtual );
	// Service-Type is Login-User if originating the call
	*pdu += new RadiusAttr( RadiusAttr::ServiceType, RadiusAttr::ST_Login );

	if( includeFramedIp && framedIP.IsValid() )
		*pdu += new RadiusAttr( RadiusAttr::FramedIpAddress, framedIP );
	
	PString callingStationId;
	
	if( q931pdu.GetCallingPartyNumber( callingStationId ) )
		if(	callingEP ) {
			if( !CheckAliases(callingEP->GetAliases(),callingStationId) )
				callingStationId = PString::Empty();
		} else if( setup.HasOptionalField(setup.e_sourceAddress) ) {
			if( !CheckAliases(setup.m_sourceAddress,callingStationId) )
				callingStationId = PString::Empty();
		} else
			callingStationId = PString::Empty();
	
	if(	callingStationId.IsEmpty() && callingEP 
		&& (callingEP->GetAliases().GetSize() > 0) )
		callingStationId = GetBestAliasAddressString(
			callingEP->GetAliases(),
			H225_AliasAddress::e_dialedDigits,
			H225_AliasAddress::e_partyNumber,
			H225_AliasAddress::e_h323_ID
			);
	
	if( callingStationId.IsEmpty() 
		&& setup.HasOptionalField(setup.e_sourceAddress) 
		&& (setup.m_sourceAddress.GetSize() > 0) )
		callingStationId = GetBestAliasAddressString(
			setup.m_sourceAddress,
			H225_AliasAddress::e_dialedDigits,
			H225_AliasAddress::e_partyNumber,
			H225_AliasAddress::e_h323_ID
			);
	
	if( callingStationId.IsEmpty() )
		callingStationId = username.IsEmpty() ? fixedUsername : username;

#ifdef HAS_ACCT		
	if( (!callingStationId.IsEmpty()) && callrec 
		&& callrec->GetCallingStationId().IsEmpty() )
		callrec->SetCallingStationId(callingStationId);
#endif

	if( !callingStationId.IsEmpty() )
		*pdu += new RadiusAttr( RadiusAttr::CallingStationId, callingStationId );

	PString calledStationId;
	
	if( q931pdu.GetCalledPartyNumber(calledStationId) )
		if( setup.HasOptionalField(setup.e_destinationAddress) ) {
			if( !CheckAliases(setup.m_destinationAddress,calledStationId) )
				calledStationId = PString::Empty();
		} else
			calledStationId = PString::Empty();
	
	if( calledStationId.IsEmpty() 
		&& setup.HasOptionalField(setup.e_destinationAddress) 
		&& (setup.m_destinationAddress.GetSize() > 0) ) {
		calledStationId = GetBestAliasAddressString(
			setup.m_destinationAddress,
			H225_AliasAddress::e_dialedDigits,
			H225_AliasAddress::e_partyNumber,
			H225_AliasAddress::e_h323_ID
			);
	}
	
	if( calledStationId.IsEmpty() ) {
		PIPSocket::Address addr;
		WORD port;
		
		if( callrec && callrec->HasDestSignalAddr() 
			&& callrec->GetDestSignalAddr(addr,port) )
			calledStationId = AsString(addr,port);
		else if( setup.HasOptionalField(setup.e_destCallSignalAddress) 
			&& GetIPAndPortFromTransportAddr(setup.m_destCallSignalAddress,addr,port) 
			&& addr.IsValid() )
			calledStationId = AsString(addr,port);
	}
	
	if( calledStationId.IsEmpty() ) {
		delete pdu;
		PTRACE(3,"RADAUTH\t"<<GetName()<<" Setup check failed - no called station id found");
		releaseCompleteCause = Q931::CallRejected;
		return e_fail;
	}

#ifdef HAS_ACCT	
	if( callrec && callrec->GetCalledStationId().IsEmpty() )
		callrec->SetCalledStationId(calledStationId);
#endif

	*pdu += new RadiusAttr( RadiusAttr::CalledStationId, calledStationId );
	
	if( appendCiscoAttributes ) {
		*pdu += new RadiusAttr(
			PString("h323-conf-id=") 
				+ GetConferenceIDString(setup.m_conferenceID),
			9, // VendorId = Cisco
			24 // Connection-ID (h323-conf-id)
			);
		*pdu += new RadiusAttr(
			PString("h323-call-origin=originate"),
			9, // VendorId = Cisco
			26 // Call-Origin (h323-call-origin)
			);
		*pdu += new RadiusAttr(
			PString("h323-call-type=VoIP"),
			9, // VendorId = Cisco
			27 // Call-Type (h323-call-type)
			);
		*pdu += new RadiusAttr(
			PString("h323-gw-id=") + NASIdentifier,
			9, // VendorId = Cisco
			33 // Call-Type (h323-gw-id)
			);
	}
					
	// send the request and wait for a response
	RadiusPDU* response = NULL;
	BOOL result = OnSendPDU(*pdu,q931pdu,setup,releaseCompleteCause,durationLimit) 
		&& radiusClient->MakeRequest( *pdu, response ) 
		&& (response != NULL);
			
	delete pdu;
			
	if( !result ) {
		delete response;
		if( defaultStatus == e_fail )
			PTRACE(3,"RADAUTH\t"<<GetName()<<" Setup check failed - could not send Access-Request PDU");
		else if( defaultStatus != e_ok )
			PTRACE(4,"RADAUTH\t"<<GetName()<<" Setup check undetermined - could not send Access-Request PDU");
		releaseCompleteCause = Q931::TemporaryFailure;
		return defaultStatus;
	}
				
	// authenticated?
	result = (response->GetCode() == RadiusPDU::AccessAccept);
	
	// process h323-return-code reply attribute
	if( result ) {
		const PINDEX index = response->FindVsaAttr( 9, 103 );
		if( index != P_MAX_INDEX ) {
			const RadiusAttr* attr = response->GetAttrAt(index);
			BOOL valid = FALSE;
		
			if( attr && attr->IsValid() ) {
				PString s = attr->AsVsaString();
				if( s.Find("h323-return-code=") == 0 )
					s = s.Mid( s.FindOneOf("=") + 1 );
				if( s.GetLength() > 0
					&& strspn((const char*)s,"0123456789") == (size_t)s.GetLength() ) {
					unsigned retcode = s.AsUnsigned();
					if( retcode != 0 ) {
						PTRACE(5,"RADAUTH\t"<<GetName()<<" Setup check failed - return code "<<retcode);
						result = FALSE;
					}
					valid = TRUE;
				}
			}
		
			if( !valid ) {
				PTRACE(5,"RADAUTH\t"<<GetName()<<" check failed - invalid h323-return-code attribute");
				result = FALSE;
			}
		}
	}

	// process h323-credit-time attribute	
	if( result ) {
		BOOL found = FALSE;
		
		const PINDEX index = response->FindVsaAttr( 9, 102 );
		if( index != P_MAX_INDEX ) {
			const RadiusAttr* attr = response->GetAttrAt(index);
			if( attr && attr->IsValid() ) {
				PString s = attr->AsVsaString();
				if( s.Find("h323-credit-time=") == 0 ) 
					s = s.Mid( s.FindOneOf("=") + 1 );
				if( s.GetLength() > 0
					&& strspn((const char*)s,"0123456789") == (size_t)s.GetLength() ) {
					found = TRUE;
					durationLimit = s.AsInteger();
					PTRACE(5,"RADAUTH\t"<<GetName()<<" Setup check set duration limit set: "<<durationLimit);
					if( durationLimit == 0 )
						result = FALSE;
				}
			}
			
			if( !found ) {
				PTRACE(5,"RADAUTH\t"<<GetName()<<" Setup check failed - invalid h323-credit-time attribute: ");
				result = FALSE;
			}
		}
	}

	// process Session-Timeout attribute	
	if( result ) {
		BOOL found = FALSE;
		
		const PINDEX index = response->FindAttr( RadiusAttr::SessionTimeout );
		if( index != P_MAX_INDEX ) {
			const RadiusAttr* attr = response->GetAttrAt(index);
			if( attr && attr->IsValid() ) {
				found = TRUE;
				const long sessionTimeout = attr->AsInteger();
				if( (durationLimit < 0) || (durationLimit > sessionTimeout) ) {
					durationLimit = sessionTimeout;
					PTRACE(5,"RADAUTH\t"<<GetName()<<" Setup check set duration limit set: "<<durationLimit);
				}
				if( durationLimit == 0 )
					result = FALSE;
			}
			
			if( !found ) {
				PTRACE(5,"RADAUTH\t"<<GetName()<<" Setup check failed - invalid Session-Timeout attribute");
				result = FALSE;
			}
		}
	}
	
	if( result )
		result = OnReceivedPDU(*response,q931pdu,setup,releaseCompleteCause,durationLimit);
	else
		releaseCompleteCause = Q931::CallRejected;
	
	delete response;
	return result ? e_ok : e_fail;
}

BOOL RadAliasAuth::OnSendPDU(
	RadiusPDU& /*pdu*/,
	const H225_RegistrationRequest& /*rrq*/,
	unsigned& /*rejectReason*/
	)
{
	return TRUE;
}

BOOL RadAliasAuth::OnSendPDU(
	RadiusPDU& /*pdu*/,
	const H225_AdmissionRequest& /*rrq*/,
	unsigned& /*rejectReason*/
	)
{
	return TRUE;
}

BOOL RadAliasAuth::OnSendPDU(
	RadiusPDU& /*pdu*/,
	const Q931& /*q931pdu*/,
	const H225_Setup_UUIE& /*setup*/,
	unsigned& /*releaseCompleteCause*/,
	long& /*durationLimit*/
	)
{
	return TRUE;
}

BOOL RadAliasAuth::OnReceivedPDU(
	RadiusPDU& /*pdu*/,
	H225_RegistrationRequest& /*rrq*/,
	unsigned& /*rejectReason*/
	)
{
	return TRUE;
}

BOOL RadAliasAuth::OnReceivedPDU(
	RadiusPDU& /*pdu*/,
	H225_AdmissionRequest& /*arq*/,
	unsigned& /*rejectReason*/
	)
{
	return TRUE;
}

BOOL RadAliasAuth::OnReceivedPDU(
	RadiusPDU& pdu,
	H225_AdmissionRequest& arq,
	unsigned& rejectReason,
	long& /*durationLimit*/
	)
{
	return OnReceivedPDU(pdu,arq,rejectReason);
}

BOOL RadAliasAuth::OnReceivedPDU(
	RadiusPDU& /*pdu*/,
	Q931& /*q931pdu*/,
	H225_Setup_UUIE& /*setup*/,
	unsigned& /*releaseCompleteCause*/,
	long& /*durationLimit*/
	)
{
	return TRUE;
}

#endif /* HAS_RADIUS */
