/////////////////////////////////////////////////////////////////////////////
//
// gk_wldap_link.cxx
//
// $Id: gk_wldap_link.cxx,v 1.1.2.4 2004/05/12 17:46:40 zvision Exp $
//
// Copyright (C) 2003 Franz J Ehrengruber <franz@iptelenet.com>
//  
// PURPOSE OF THIS FILE: 
//   Provides the LDAP search functions based on the RFC 1823 MS LDAP-API
//   Windows LDAP integration (wldap.h/wldap32.lib)
//  
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//  
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//  
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//  
//
// History:
//      2003/10/01      initial version (Franz J Ehrengruber)
//                      based on ldaplink.cxx (Martin Froehlich)
//
/////////////////////////////////////////////////////////////////////////////

#if defined(HAS_WLDAP) 

#if (_MSC_VER >= 1200)
#pragma warning( disable : 4800 ) // warning about forcing value to bool
#endif

#include <ptlib.h>
#include <ptlib/sockets.h>
#include <stdlib.h>           // ANSI C: C standard library
#include <math.h>		      // ANSI C: C math library
#include "GkStatus.h"	      // gatekeeper status port for error handling
#include "gk_wldap_link.h"	  // First of includes: own interface

// simplified output
#if !defined(LDAP_DBG_LVL)
#  define LDAP_DBG_LVL 2
#endif

#if !defined(LDAP_DBG_LINEEND)
#  if defined(WIN32)
#    define LDAP_DBG_LINEEND "\r\n"
#  endif
#endif

// NOTE: Do not use the ldap_perror function! This environment provides its
//       own error handling:
#define ERRORPRINT(strpar) GkStatus::Instance()->SignalStatus(PString(strpar) + LDAP_DBG_LINEEND);
#define DEBUGPRINT(stream) PTRACE(LDAP_DBG_LVL, "LDAP\t" << stream << endl);

// list of names (keys) as used in config file, keep in sync with LDAPAttributeNamesEnum
//
const char *  lctn::LDAPAttrTags[lctn::MAX_ATTR_NO] = {"DN", 
													   "CommonName", 
													   "AccountStatus", 
													   "H323ID", 
													   "TelephonNo", 
													   "H235PassWord"}; 

// CLASS: LDAPAnswer
//
LDAPAnswer::LDAPAnswer()
{
	// initialize LDAPMessage * result - GPF at 
	// destructor below if result is NULL
	result = 0;
}

LDAPAnswer::~LDAPAnswer()
{
	// !! important - memory leak !!
	if( result )
		gk_ldap_msgfree( result );
}

// CLASS: LDAPCtrl
//
LDAPCtrl::LDAPCtrl(LDAPAttributeNamesClass * AttrNames,
		           LDAP_TIMEVAL default_timeout,
		           PString & ServerName,
		           PString & SearchBaseDN,
		           PString & BindUserDN,
		           PString & BindUserPW,
		           unsigned int timelimit = LDAP_NO_LIMIT,
		           int ServerPort = LDAP_PORT):AttributeNames(AttrNames), 
				                               timeout(default_timeout), 
											   ServerName(ServerName),
	                                           ServerPort(ServerPort), 
											   SearchBaseDN(SearchBaseDN), 
											   BindUserDN(BindUserDN),
	                                           BindUserPW(BindUserPW), 
											   timelimit(timelimit), 
	                                           ldap(NULL), 
											   known_to_be_bound(false)
{
	// do a parameter sanity check here
	// if we fail - return


	// initialize and bind with LDAP
	int ldap_ret = LDAP_SUCCESS;
	if (LDAP_SUCCESS != (ldap_ret = Initialize())) {
		ERRORPRINT("LDAPCtrl::LDAPCtrl() Initialize() result - " + PString(gk_ldap_err2string(ldap_ret)));
		DEBUGPRINT("LDAPCtrl::LDAPCtrl() Initialize() result - " + PString(gk_ldap_err2string(ldap_ret)));
		ldap = NULL;
		return;
	}
	DEBUGPRINT("LDAPCtrl::LDAPCtrl() Initialize() result - " + PString(gk_ldap_err2string(ldap_ret)));

} // constructor: LDAPCtrl

LDAPCtrl::~LDAPCtrl() {
	// this space left blank intentionally
}

// private: initially called from constructors
int
LDAPCtrl::Initialize(void) {
	if(m_bindLock.WillBlock()){
		PTRACE(1, "Will not Bind, because I'm blocked with Mutex: " );
		known_to_be_bound=false;
		return LDAP_UNAVAILABLE;
	}
	PWaitAndSignal lock(m_bindLock);

	int ldap_ret = LDAP_SUCCESS; // assume success

	// when called other then then from constructors
	if(ldap != NULL)
		Close();

	// set ldap c-object
	GK_LDAP * ld = NULL;
	ldap = NULL;

	if (NULL == (ld = gk_ldap_init(ServerName, ServerPort))) {
		ldap_ret = gk_LdapGetLastError();
		ERRORPRINT("LDAPCtrl::Initialize() gk_ldap_init() result - " + PString(gk_ldap_err2string(ldap_ret)));
		DEBUGPRINT("LDAPCtrl::Initialize() gk_ldap_init() result - " + PString(gk_ldap_err2string(ldap_ret)));
		ldap = NULL;
		return ldap_ret;
	} else {
		DEBUGPRINT("LDAPCtrl::Initialize() gk_ldap_init() result - " + PString(gk_ldap_err2string(ldap_ret)));
		ldap = ld;
	}

	if (LDAP_SUCCESS != (ldap_ret = Connect())) {
		ERRORPRINT("LDAPCtrl::Initialize() Connect() result - " + PString(gk_ldap_err2string(ldap_ret)));
		DEBUGPRINT("LDAPCtrl::Initialize() Connect() result - " + PString(gk_ldap_err2string(ldap_ret)));
		Close();
		return ldap_ret;
	}
	DEBUGPRINT("LDAPCtrl::Initialize Connect() result - " + PString(gk_ldap_err2string(ldap_ret)));

	if(LDAP_SUCCESS != (ldap_ret = Bind(true))) { // bind (enforced)
		ERRORPRINT("LDAPCtrl::Initialize() Bind() result - " + PString(gk_ldap_err2string(ldap_ret)));
		DEBUGPRINT("LDAPCtrl::Initialize() Bind() result - " + PString(gk_ldap_err2string(ldap_ret)));
		Close();
		return ldap_ret;
	}
	DEBUGPRINT("LDAPCtrl::Initialize Bind() result - " + PString(gk_ldap_err2string(ldap_ret)));

	// success
	return ldap_ret;

} // privat: Initialize

// connecting to LDAP server before binding
int
LDAPCtrl::Connect(void) {
	int ldap_ret = LDAP_OTHER;

	DEBUGPRINT("LDAPCtrl::Connect() Connecting to " << ServerName << ":" << ServerPort << "");
	if (LDAP_SUCCESS == (ldap_ret = gk_ldap_connect(ldap, &timeout))) {
		DEBUGPRINT("LDAPCtrl::Connect() gk_ldap_connect() result - " + PString(gk_ldap_err2string(ldap_ret)));
	} else {
		ERRORPRINT("LDAPCtrl::Connect() gk_ldap_connect() result - " + PString(gk_ldap_err2string(ldap_ret)));
		DEBUGPRINT("LDAPCtrl::Connect() gk_ldap_connect() result - " + PString(gk_ldap_err2string(ldap_ret)));
		return ldap_ret;
	}
	return ldap_ret;
}

// binding, may be enforced by passing true
int
LDAPCtrl::Bind(bool force = false) {
	int ldap_ret = LDAP_SUCCESS;
	if(known_to_be_bound && force) {
		DEBUGPRINT("LDAPCtrl::Bind() - I think I'm already bound, but action is forced");
	}

	if(!known_to_be_bound || force) {
		// this is the local bind version
		DEBUGPRINT("LDAPCtrl::Bind() Binding with " << BindUserDN << " pw length:" << BindUserPW.GetLength());
		if (LDAP_SUCCESS == 
		    (ldap_ret = gk_ldap_simple_bind_s(ldap, BindUserDN, BindUserPW))) {
			known_to_be_bound = true;
			DEBUGPRINT("LDAPCtrl::Bind() gk_ldap_simple_bind_s() result - " + PString(gk_ldap_err2string(ldap_ret)));
		} else {
			DEBUGPRINT("LDAPCtrl::Bind() gk_ldap_simple_bind_s() result - " + PString(gk_ldap_err2string(ldap_ret)));
			ERRORPRINT("LDAPCtrl::Bind() gk_ldap_simple_bind_s() result - " + PString(gk_ldap_err2string(ldap_ret)));
			return ldap_ret;
		}
	}
	return ldap_ret;
}

// unbinding, may be enforced by passing true
int
LDAPCtrl::Unbind(bool force) {
	int ldap_ret = LDAP_SUCCESS;

	if(!known_to_be_bound && force) {
		DEBUGPRINT("LDAPCtrl::Unbind() I think I'm already unbound, but action is forced");
	}

	if((NULL != ldap) && (known_to_be_bound || force)) {
		if(LDAP_SUCCESS != (ldap_ret = gk_ldap_unbind(ldap))) {
			ERRORPRINT("LDAPCtrl::Unbind() gk_ldap_unbind() result - " + PString(gk_ldap_err2string(ldap_ret)));
		} else {
			known_to_be_bound = false;
		}
	}

	DEBUGPRINT("LDAPCtrl::Unbind() gk_ldap_unbind() result - " + PString(gk_ldap_err2string(ldap_ret)));
	
	ldap = NULL;
	return ldap_ret;
}

void 
LDAPCtrl::Close(void) {
	if(m_bindLock.WillBlock()) {
		PTRACE(1, "Will not unbind because Blocked Thread by Mutex");
		return;
	}
	PWaitAndSignal lock(m_bindLock);

	int ldap_ret = Unbind(true);
	DEBUGPRINT("LDAPCtrl::Close() Unbind() result - " + PString(gk_ldap_err2string(ldap_ret)));
}

//                            ---  end of LDAP init ---

int 
LDAPCtrl::LdapUserLookup(const PString & common_name, const PStringArray & attr_names, LDAPAnswer * answer) {
	int entriesFound = 0;
	int ldap_ret;

	// invalid LDAP object - initialize LDAP
	// on failure - return 0
	if(ldap == NULL) {
		DEBUGPRINT("LDAPCtrl::LdapUserLookup() Invalid LDAP object - initialize LDAP");
		ERRORPRINT("LDAPCtrl::LdapUserLookup() Invalid LDAP object - initialize LDAP");
		if(LDAP_SUCCESS != Initialize())
			return 0;
	}

	if(attr_names.IsEmpty()) {
		// warning: all attributes will be returned
		// this can be a performance issue.
		DEBUGPRINT("LDAPCtrl::LdapUserLookup() Warning: No attribute names specified - All attributes will be returned.");
		ERRORPRINT("LDAPCtrl::LdapUserLookup() Warning: No attribute names specified - All attributes will be returned.");
	}

	// establish NULL terminated array of 
	// attribute names, for values to be returned
	unsigned int pos = 0;
	using namespace lctn;
	const char * (attrs[MAX_ATTR_NO]);
	PString filter;
	LDAPAttributeNamesClass::iterator iter = AttributeNames->begin();
	while((iter != AttributeNames->end()) && (MAX_ATTR_NO >= pos)) {
		PTRACE(5, "LDAP\tLDAPCtrl::LdapUserLookup() (*iter).first '" << (*iter).first << "' (*iter).second '" << (const char *)((*iter).second) << "'");
		for(int i = 0; i < attr_names.GetSize(); i++) {
			if((*iter).first == attr_names[i]) {
				// This cast is directly from hell, 
				// but pwlib is not nice to C APIs - could not agree more fje
				attrs[pos++] = (const char *)((*iter).second);
				PTRACE(5, "LDAP\tLDAPCtrl::LdapUserLookup() attrs[" << pos << "] '" << (const char *)((*iter).second) << "'");
			}
		
		} // for

		// simple search filter. Example e164/h323id: (cn=1234567890) or (cn=ABCDEFGHIJKLM)
		if((*iter).first == LDAPAttrTags[CommonName]) {
			filter+="(";
			filter+=(*iter).second;
			filter+="=";
			filter+=common_name;
			filter+=")";
			PTRACE(5, "LDAP\tLDAPCtrl::LdapUserLookup() filter   '" << filter << "'");
		}		
		iter++;
	
	} // while
	attrs[pos] = NULL;		// C construct: array of unknown size 
	                        // terminated by NULL-pointer 

	// search LDAP
	ldap_ret = LdapDirectorySearch(filter, attrs, answer);

	// success - get number of entries found
	if(ldap_ret == LDAP_SUCCESS) {
		if (0 > (ldap_ret = gk_ldap_count_entries(ldap, answer->result))) {
			ERRORPRINT("ldap_search_st: " + PString(gk_ldap_err2string(ldap_ret)));
			DEBUGPRINT("ldap_search_st: " + PString(gk_ldap_err2string(ldap_ret)));
		} else {
			DEBUGPRINT("ldap_search_st: " << ldap_ret << " results");
			entriesFound   = ldap_ret;
		}
	}

	// return the number of attributes returned
	return entriesFound;
}

int
LDAPCtrl::LdapDirectorySearch(const PString & filter, const char * attrs[], LDAPAnswer * answer) {
	int ldap_ret;

	unsigned int retry_count = 0;
	do {
		DEBUGPRINT("ldap_search_st(" << SearchBaseDN << ", " << filter << ", " << timeout.tv_sec << ":" << 
			   timeout.tv_usec << ")");

		if (LDAP_SUCCESS == (ldap_ret = gk_ldap_search_st(ldap, 
														  SearchBaseDN, 
														  LDAP_SCOPE_SUBTREE, 
														  filter, 
														  (char **)attrs, 
														  0, 
														  &timeout, 
														  &answer->result))) {
			DEBUGPRINT("ldap_search_st: OK " << PString(gk_ldap_err2string(ldap_ret)));
		} 
		else {		
			DEBUGPRINT("ldap_search_st: " + PString(gk_ldap_err2string(ldap_ret)));
			ERRORPRINT("ldap_search_st: " + PString(gk_ldap_err2string(ldap_ret)));

			switch (ldap_ret) {
			// success
			case LDAP_SUCCESS:
				break;

			// retry
			case LDAP_BUSY:
			case LDAP_OPERATIONS_ERROR:
			case LDAP_TIMELIMIT_EXCEEDED:
			case LDAP_TIMEOUT:
				continue;
			
			// re-initialize LDAP
			case LDAP_SERVER_DOWN:
			case LDAP_CONNECT_ERROR:
			case LDAP_LOCAL_ERROR: // frequnetly associated with local network problems
			case LDAP_UNAVAILABLE:
				if(LDAP_SUCCESS != (ldap_ret = Initialize()))
					return ldap_ret;
				continue;

			// no retry - don't waste your time
			case LDAP_FILTER_ERROR:
			case LDAP_INAPPROPRIATE_AUTH:
			case LDAP_INSUFFICIENT_RIGHTS:
			case LDAP_INVALID_CREDENTIALS:
			case LDAP_INVALID_DN_SYNTAX:
			case LDAP_INVALID_SYNTAX:
			case LDAP_NAMING_VIOLATION:
			case LDAP_NO_SUCH_OBJECT:
			case LDAP_OTHER:
			case LDAP_PARAM_ERROR:
			case LDAP_PROTOCOL_ERROR:
			case LDAP_UNWILLING_TO_PERFORM:
			case LDAP_USER_CANCELLED:
				return ldap_ret;
		
			// no retry - don't waste your time
			default:
				return ldap_ret;
			
			} // switch
		
		} // LDAP_SUCCESS

	} while((LDAP_SUCCESS != ldap_ret)&&(retry_count++ < 4));
	
	return ldap_ret;
}

int
LDAPCtrl::LdapCollectAttributes(LDAPAnswer * answer) {
	LDAPMessage *e;
	BerElement  *ber;
	char *a;
	int attribute_count = 0;

	// get entries/attributes
	for ( e = gk_ldap_first_entry( ldap, answer->result ); e != NULL; e = gk_ldap_next_entry( ldap, e ) ) {
		// get DN
		char *dn;
		if(NULL == (dn = gk_ldap_get_dn(ldap, e))) {
				ERRORPRINT("LDAP\tLDAPCtrl::LdapCollectAttributes() ldap_get_dn: Could not get distinguished name.");
		}
		DEBUGPRINT("found DN: " << dn);
		
		LDAPAttributeValueClass AV;
		for( a = gk_ldap_first_attribute( ldap, e, &ber ); a != NULL; a = gk_ldap_next_attribute( ldap, e, ber ) ) {
			// retrieve member attributes
			char **vals;
			if((vals = gk_ldap_get_values( ldap, e, a)) != NULL ) {
				int valc = gk_ldap_count_values(vals);
				if(0 == valc) { 
					DEBUGPRINT("LDAP\tLDAPCtrl::LdapCollectAttributes() Value handling: No values returned");
				}

				// put list of values (of this particular attribute) into PStringList
				// which can be accessed by a STL map, indexed by attribute names.
				// This implies, that the data is not bit- or octet-string, 
				// because it may NOT contain \0.
				AV.insert(LDAPAVValuePair(PString(a), PStringList(valc, vals, false)));
				PTRACE(5, "LDAP\tLDAPCtrl::LdapCollectAttributes() found value for attribute '" << PString(a) << "'");

				ldap_value_free( vals );
				attribute_count++;
			} // if

			gk_ldap_berfree(ber, 0);    // second parameter must be zero here
			gk_ldap_memfree( a );
		
		} // for

		// copy to LDAPAnswer result
		answer->LDAPec.insert(LDAPECValuePair(dn,AV));	
		gk_ldap_memfree( dn );

	} // for
   
	gk_ldap_msgfree( e );

	// Note: memory for 'answer->result'
	// is freed inside the destructor 'virtual ~LDAPAnswer()'
 
	return attribute_count;
}

#endif // HAS_WLDAP
//
// End of gk_wldap_link.cxx
//


