//////////////////////////////////////////////////////////////////
//
// Toolkit base class for the GnuGK
//
// $Id: Toolkit.cxx,v 1.22.2.35 2004/05/25 19:54:04 zvision Exp $
//
// This work is published under the GNU Public License (GPL)
// see file COPYING for details.
// We also explicitely grant the right to link this code
// with the OpenH323 library.
//
// History:
// 	991227  initial version (Torsten Will, mediaWays)
//
//////////////////////////////////////////////////////////////////

#if (_MSC_VER >= 1200)
#pragma warning( disable : 4786 ) // warning about too long debug symbol off
#pragma warning( disable : 4800 ) // warning about forcing value to bool
#endif

#include <ptlib.h>
#include <map>
#include <ptlib/sockets.h>
#include <ptclib/cypher.h>
#include <h225.h>
#include <h323pdu.h>
#include "gk_const.h"
#include "stl_supp.h"
#include "gksql.h"
#include "gktimer.h"
#include "Toolkit.h"

// class Toolkit::RouteTable::RouteEntry
Toolkit::RouteTable::RouteEntry::RouteEntry(
	const PString & net
) : PIPSocket::RouteEntry(0)
{
	destination = net.Tokenise("/", FALSE)[0];
	GetNetworkFromString(net, network, net_mask);
}

Toolkit::RouteTable::RouteEntry::RouteEntry(
	const PIPSocket::RouteEntry & re,
	const InterfaceTable & it
) : PIPSocket::RouteEntry(re)
{
	PINDEX i;
	for (i = 0; i < it.GetSize(); ++i) {
		const Address & ip = it[i].GetAddress();
		if (Compare(&ip)) {
			destination = ip;
			return;
		}
	}
	for (i = 0; i < it.GetSize(); ++i)
		if (it[i].GetName() == interfaceName) {
			destination = it[i].GetAddress();
			return;
		}
}

inline bool Toolkit::RouteTable::RouteEntry::Compare(const Address *ip) const
{
	return (*ip == destination) || ((*ip & net_mask) == network);
}

// class Toolkit::RouteTable
void Toolkit::RouteTable::InitTable()
{
	// Workaround for OS doesn't support GetRouteTable
	PIPSocket::GetHostAddress(defAddr);

	ClearTable();

	PString networks = GkConfig()->GetString("NetworkInterfaces", "");
	if (networks.IsEmpty())
		CreateTable();
	else
		CreateTable(networks);

	// Set default IP according to route table
	PIPSocket::Address defGW;
	PIPSocket::GetGatewayAddress(defGW);
	defAddr = GetLocalAddress(defGW);

#if PTRACING
	for (RouteEntry *entry = rtable_begin; entry != rtable_end; ++entry)
		PTRACE(2, "Network=" << entry->GetNetwork() << '/' << entry->GetNetMask() <<
			  ", IP=" << entry->GetDestination());
	PTRACE(2, "Default IP=" << defAddr);
#endif
}

void Toolkit::RouteTable::ClearTable()
{
	if (rtable_begin) {
		for (RouteEntry *r = rtable_begin; r != rtable_end; ++r)
			r->~RouteEntry();
		::free(rtable_begin);
		rtable_begin = 0;
	}
}

// can't pass a reference of Address, or STL complains...
PIPSocket::Address Toolkit::RouteTable::GetLocalAddress(const Address & addr) const
{
	RouteEntry *entry = find_if(rtable_begin, rtable_end,
			bind2nd(mem_fun_ref(&RouteEntry::Compare), &addr));
	return (entry != rtable_end) ? entry->GetDestination() : defAddr;
}

void Toolkit::RouteTable::CreateTable()
{
	InterfaceTable if_table;
	if (!PIPSocket::GetInterfaceTable(if_table)) {
		PTRACE(1, "Error: Can't get interface table");
		return;
	}
	PTRACE(4, "InterfaceTable:\n" << setfill('\n') << if_table << setfill(' '));
	PIPSocket::RouteTable r_table;
	if (!PIPSocket::GetRouteTable(r_table)) {
		PTRACE(1, "Error: Can't get route table");
		return;
	}

	int i = r_table.GetSize();
	rtable_end = rtable_begin = static_cast<RouteEntry *>(::malloc(i * sizeof(RouteEntry)));
	for (PINDEX r = 0; r < i ; ++r) {
		PIPSocket::RouteEntry & r_entry = r_table[r];
		if (r_entry.GetNetMask() != INADDR_ANY)
			// placement operator
			::new (rtable_end++) RouteEntry(r_entry, if_table);
	}
}

void Toolkit::RouteTable::CreateTable(const PString & nets)
{
	PStringArray networks(nets.Tokenise(" ,;\t", FALSE));
	int i = networks.GetSize();
	if (i > 0) {
		rtable_end = rtable_begin = static_cast<RouteEntry *>(::malloc(i * sizeof(RouteEntry)));
		for (PINDEX r = 0; r < i ; ++r)
			::new (rtable_end++) RouteEntry(networks[r]);
	}
}

// class Toolkit::ProxyCriterion
void Toolkit::ProxyCriterion::LoadConfig(PConfig *config)
{
	ClearTable();
	if (!AsBool(config->GetString(ProxySection, "Enable", "0"))) {
		PTRACE(2, "GK\tH.323 Proxy disabled");
		size = -1;
		return;
	}

	PTRACE(2, "GK\tH.323 Proxy enabled");

	PStringArray networks(config->GetString(ProxySection, "InternalNetwork", "").Tokenise(" ,;\t", FALSE));
	if ((size = networks.GetSize()) == 0) {
		// no internal networks specified, always use proxy
		return;
	}

	network = new Address[size * 2];
	netmask = network + size;
	for (int i = 0; i < size; ++i) {
		GetNetworkFromString(networks[i], network[i], netmask[i]);
		PTRACE(2, "GK\tInternal Network " << i << " = " <<
			   network[i] << '/' << netmask[i]);
	}
}

void Toolkit::ProxyCriterion::ClearTable()
{
	size = 0;
	delete [] network;
	network = 0;
}

bool Toolkit::ProxyCriterion::Required(const Address & ip1, const Address & ip2) const
{
	return (size >= 0) ? ((size == 0) || (IsInternal(ip1) != IsInternal(ip2))) : false;
}

bool Toolkit::ProxyCriterion::IsInternal(const Address & ip) const
{
	for (int i = 0; i < size; ++i)
		if ((ip & netmask[i]) == network[i])
			return true;
	return false;
}

// class Toolkit::RewriteTool

static const char *RewriteSection = "RasSrv::RewriteE164";

Toolkit::RewriteData::RewriteData(PConfig *config, const PString & section) 
	: m_RewriteKey(0)
{
	PStringToString cfgs(config->GetAllKeyValues(section));
	m_size = cfgs.GetSize();
	if (m_size > 0) {
		std::map<PString, PString> rules;
		for (PINDEX i = 0; i < m_size; ++i) {
			PString key = cfgs.GetKeyAt(i);
			if (!key && (isdigit(key[0]) || key[0]=='!'))
				rules[key] = cfgs.GetDataAt(i);
		}
		// now the rules are ascendantly sorted by the keys
		if ((m_size = rules.size()) > 0) {
//			m_RewriteKey = new PString[m_size * 2];
			m_RewriteKey = (PString*)(new BYTE[sizeof(PString) * m_size * 2]);
			m_RewriteValue = m_RewriteKey + m_size;
			std::map<PString, PString>::iterator iter = rules.begin();
			// reverse the order
			for (PINDEX i = m_size; i-- > 0 ; ++iter) {
//				m_RewriteKey[i] = iter->first;
				::new(m_RewriteKey +i) PString(iter->first);
//				m_RewriteValue[i] = iter->second;
				::new(m_RewriteValue+i) PString(iter->second);
			}
		}
	}
}

void Toolkit::RewriteData::AppendRules(
	PConfig* config, 
	const PString& section
	) 
{
	PStringToString cfgs(config->GetAllKeyValues(section));
	PINDEX newsize = cfgs.GetSize();
	std::map<PString, PString> rules;
	
	for (PINDEX i = 0; i < newsize; ++i) {
		PString key = cfgs.GetKeyAt(i);
		if (!key && (isdigit(key[0]) || key[0]=='!'))
			rules[key] = cfgs.GetDataAt(i);
	}
	
	newsize = rules.size();
	
	if (newsize > 0) {

		// append existing rules to the new ones		
		if (m_RewriteKey) {
			for (PINDEX i = 0; i < m_size; i++) {
				rules[m_RewriteKey[i]] = m_RewriteValue[i];
				(m_RewriteKey+i)->~PString();
				(m_RewriteValue+i)->~PString();
			}
			delete[] (BYTE*)m_RewriteKey;
			m_RewriteKey = NULL;
			m_RewriteValue = NULL;
		}
		
		// now the rules are ascendantly sorted by the keys
		if ((m_size = rules.size()) > 0) {
//			m_RewriteKey = new PString[m_size * 2];
			m_RewriteKey = (PString*)(new BYTE[sizeof(PString) * m_size * 2]);
			m_RewriteValue = m_RewriteKey + m_size;
			std::map<PString, PString>::iterator iter = rules.begin();
			// reverse the order
			for (PINDEX i = m_size; i-- > 0 ; ++iter) {
//				m_RewriteKey[i] = iter->first;
				::new(m_RewriteKey +i) PString(iter->first);
//				m_RewriteValue[i] = iter->second;
				::new(m_RewriteValue+i) PString(iter->second);
			}
		}
	}
}

Toolkit::RewriteData::~RewriteData() 
{
	if (m_RewriteKey)
		for (PINDEX i = 0; i < m_size * 2; i++)
			(m_RewriteKey+i)->~PString();
	delete[] (BYTE*)m_RewriteKey; 
}

void Toolkit::RewriteTool::LoadConfig(
	PConfig* config,
	bool erase
	)
{
	m_RewriteFastmatch = config->GetString(RewriteSection, "Fastmatch", "");
	m_TrailingChar = config->GetString("RasSrv::ARQFeatures", "RemoveTrailingChar", " ")[0];
	if (erase) {
		delete m_Rewrite;
		m_Rewrite = new RewriteData(config, RewriteSection);
	} else
		m_Rewrite->AppendRules(config, RewriteSection);
}

bool Toolkit::RewriteTool::RewritePString(PString & s) const
{
	bool changed = false;
	bool do_rewrite = false; // marker if a rewrite has to be done.

	// remove trailing character
	if (s.GetLength() > 1 && s[s.GetLength() - 1] == m_TrailingChar) {
		s = s.Left(s.GetLength() - 1);
		changed = true;
	}
	// startsWith?
	if (strncmp(s, m_RewriteFastmatch, m_RewriteFastmatch.GetLength()) != 0)
		return changed;

	PString t;
	for (PINDEX i = 0; i < m_Rewrite->Size(); ++i) {
		PString key = m_Rewrite->Key(i);
		const bool inverted = (key[0] == '!');
		if (inverted)
			key = key.Mid(1);
		// try a prefix match through all keys
		if ((strncmp(s, key, key.GetLength()) == 0) ^ inverted) {
			// Rewrite to #t#. Append the suffix, too.
			// old:  01901234999
			//               999 Suffix
			//       0190        Fastmatch
			//       01901234    prefix, Config-Rule: 01901234=0521321
			// new:  0521321999
			t = m_Rewrite->Value(i);
			// multiple targets possible
			if (!t) {
				const PStringArray ts = t.Tokenise(",:;&|\t ", FALSE);
				if (ts.GetSize() > 1) {
					PINDEX j = rand() % ts.GetSize();
					PTRACE(5, "GK\tRewritePString: randomly chosen [" << j << "] of " << t << "");
					t = ts[j];
				}
			}

			// append the suffix
			int striplen = (inverted) ? 0 : key.GetLength();
			t += s.Mid(striplen);

			do_rewrite = true;
			break;
		}
	}
	
	// 
	// Do the rewrite. 
	// @param #t# will be written to #s#
	//
	if (do_rewrite) {
		PTRACE(2, "\tRewritePString: " << s << " to " << t);
		s = t;
		changed = true;
	}
	
	return changed;
}

Toolkit::Toolkit() : m_Config(NULL), m_extConfig(NULL), m_ConfigDirty(false),
#ifdef HAS_ACCT
	m_acctList(NULL), m_acctSessionBase((long)time(NULL)),
	m_acctSessionCounter(0),
#endif
	m_timerManager(new GkTimerManager())
{
	srand(time(0));
}

Toolkit::~Toolkit()
{
#ifdef HAS_ACCT
	try {
		if( m_acctList != NULL )
		{
			callptr dummycall;
			m_acctList->LogAcctEvent( GkAcctLogger::AcctOff, dummycall );
		}
		delete m_acctList;	
	} catch( ... ) {
		PTRACE(0,"GKACCT\tException caught during Accounting-Off event logging");
	}
#endif

	if (m_Config) {
		delete m_Config;
		PFile::Remove(m_tmpconfig);
	}
	
	if (m_extConfig) {
		delete m_extConfig;
		PFile::Remove(m_extConfigFilePath);
	}
	delete m_timerManager;
}

PConfig* Toolkit::Config()
{
	// Make sure the config would not be called before SetConfig
	PAssert(!m_ConfigDefaultSection, "Error: Call Config() before SetConfig()!");
	return (m_Config == NULL) ? ReloadConfig() : m_Config;
}

PConfig* Toolkit::ExtConfig()
{
	// Make sure the config would not be called before SetConfig
	PAssert(!m_ConfigDefaultSection, "Error: Call Config() before SetConfig()!");
	return (m_extConfig == NULL) ? ReloadConfig() : m_extConfig;
}

PConfig* Toolkit::Config(const char *section)
{
	Config()->SetDefaultSection(section);
	return m_Config;
}

PConfig* Toolkit::SetConfig(const PFilePath &fp, const PString &section)
{ 
	m_ConfigFilePath = fp;
	m_ConfigDefaultSection = section;

	return ReloadConfig();
}

void Toolkit::SetConfig(int act, const PString & sec, const PString & key, const PString & value)
{
	// the original config
	PConfig cfg(m_ConfigFilePath, m_ConfigDefaultSection);
	switch (act)
	{
		case 1:
			cfg.SetString(sec, key, value);
			m_Config->SetString(sec, key, value);
			break;
		case 2:
			cfg.DeleteKey(sec, key);
			m_Config->DeleteKey(sec, key);
			break;
		case 3:
			cfg.DeleteSection(sec);
			m_Config->DeleteSection(sec);
			break;
	}

	m_ConfigDirty = true;
}

PString Toolkit::GetTempDir() const
{
	PString tmpdir;
	
#ifndef WIN32
	// check if the directory exists and is accessible (access rights)
	if (PFile::Exists("/tmp") && PFile::Access("/tmp", PFile::ReadWrite))
		tmpdir = "/tmp";
	else 
#endif
	{
		PConfig cfg(PConfig::Environment);
		
		if (cfg.HasKey("TMP"))
			tmpdir = cfg.GetString("TMP");
		else if (cfg.HasKey("TEMP"))
			tmpdir = cfg.GetString("TEMP");
		else if (cfg.HasKey("TMPDIR"))
			tmpdir = cfg.GetString("TMPDIR");
	}
	
	if (!tmpdir.IsEmpty()) {
		// strip trailing separator
		if (tmpdir[tmpdir.GetLength()-1] == PDIR_SEPARATOR)
			tmpdir = tmpdir.Left(tmpdir.GetLength()-1);
			
		// check if the directory exists and is accessible (access rights)
		if (!(PFile::Exists(tmpdir) && PFile::Access(tmpdir, PFile::ReadWrite)))
			tmpdir = PString();
	}
	
	return tmpdir;
}

void Toolkit::CreateConfig()
{
	if (m_Config != NULL)
		PFile::Remove(m_tmpconfig);

	PString tmpdir = GetTempDir();

#ifdef WIN32
	if (tmpdir.IsEmpty())
		if (PFile::Access(".", PFile::ReadWrite))
			tmpdir = ".";
		else {
			const PFilePath fpath(m_ConfigFilePath);
			tmpdir = fpath.GetDirectory();
		}
#else
	if (tmpdir.IsEmpty())
		tmpdir = ".";
#endif

	// generate a unique name
	do {
		m_tmpconfig = tmpdir + PDIR_SEPARATOR + "gnugk.ini-" + PString(PString::Unsigned, rand()%10000);
		PTRACE(5, "GK\tTrying file name "<< m_tmpconfig << " for temp config");
	} while (PFile::Exists(m_tmpconfig));

#ifdef WIN32
	// Does WIN32 support symlink?
	if (PFile::Copy(m_ConfigFilePath, m_tmpconfig)) {
#else
	if (symlink(m_ConfigFilePath, m_tmpconfig)==0) {
#endif
		delete m_Config;
		m_Config = new PConfig(m_tmpconfig, m_ConfigDefaultSection);
	} else { // Oops! Create temporary config file failed, use the original one
		delete m_Config;
		m_Config = new PConfig(m_ConfigFilePath, m_ConfigDefaultSection);
	}
	
	if (m_extConfig != NULL)
		PFile::Remove(m_extConfigFilePath);
	
	// generate a unique name
	do {
		m_extConfigFilePath = tmpdir + PDIR_SEPARATOR + "gnugk.ini-" + PString(PString::Unsigned, rand()%10000);
		PTRACE(5, "GK\tTrying file name "<< m_extConfigFilePath << " for external config");
	} while (PFile::Exists(m_extConfigFilePath));

	delete m_extConfig;
	m_extConfig = new PConfig(m_extConfigFilePath, m_ConfigDefaultSection);
}

void Toolkit::ReloadSQLConfig()
{
	GkSQLConnection* sqlConn;
	const PString driverName = m_Config->GetString("SQLConfig", "Driver", "");
	if (driverName.IsEmpty())
		return;
		
	GkSQLCreator* creator = GkSQLCreator::FindDriver(driverName);
	if (creator == NULL) {
		PTRACE(1, "GK\tFailed to read config settings from SQL: no driver found for "
				<< driverName << " database"
				);
		return;
	}
	
	sqlConn = creator->CreateConnection("SQLCONF");
	if (!sqlConn->Initialize(m_Config, "SQLConfig")) {
		delete sqlConn;
		sqlConn = NULL;
		PTRACE(1, "GK\tFailed to read config settings from SQL: could not connect to the database");
		return;
	}

	PTRACE(3, "GK\tSQL config connection established");
	
	PString query;
	GkSQLResult* queryResult;
			
	query = m_Config->GetString("SQLConfig", "RewriteE164Query", "");
	if (!query.IsEmpty()) {
		PTRACE(4, "GK\tLoading rewrite rules from SQL database");
		PStringArray params;
		params += GKName();
		queryResult = sqlConn->ExecuteQuery(query, &params);
		if (queryResult == NULL)
			PTRACE(1, "GK\tFailed to load rewrite rules from SQL database: timeout or fatal error");
		else if (!queryResult->IsValid())
			PTRACE(1, "GK\tFailed to load rewrite rules from SQL database ("
				<< queryResult->GetErrorCode() << "): " << queryResult->GetErrorMessage()
				);
		else if (queryResult->GetNumFields() < 2)
			PTRACE(1, "GK\tFailed to load rewrite rules from SQL database: at least two columns must be present in the result set");
		else {
			while (queryResult->FetchRow(params))
				m_extConfig->SetString("RasSrv::RewriteE164", params[0], params[1]);
			PTRACE(4, "GK\t" << queryResult->GetNumRows() << " rewrite rules loaded from SQL database");
		}
		delete queryResult;
		queryResult = NULL;
	}

	query = m_Config->GetString("SQLConfig", "NeighborsQuery", "");
	if (!query.IsEmpty()) {
		PTRACE(4, "GK\tLoading neighbors from SQL database");
		PStringArray params;
		params += GKName();
		queryResult = sqlConn->ExecuteQuery(query, &params);
		if (queryResult == NULL)
			PTRACE(1, "GK\tFailed to load neighbors from SQL database: timeout or fatal error");
		else if (!queryResult->IsValid())
			PTRACE(1, "GK\tFailed to load neighbors from SQL database ("
				<< queryResult->GetErrorCode() << "): " << queryResult->GetErrorMessage()
				);
		else if (queryResult->GetNumFields() < 6)
			PTRACE(1, "GK\tFailed to load neighbors from SQL database: at least 6 columns must be present in the result set");
		else {
			while (queryResult->FetchRow(params)) {
				PString value;
				if (!params[5])
					value = ";" + params[5];
				if (!(params[4].IsEmpty() && value.IsEmpty()))
					value = ";" + params[4] + value;
				if (!(params[3].IsEmpty() && value.IsEmpty()))
					value = ";" + params[3] + value;
				if (!params[2])
					value = params[1] + ":" + params[2] + value;
				else
					value = params[1] + value;
				m_extConfig->SetString("RasSrv::Neighbors", params[0], value);
			}
			PTRACE(4, "GK\t" << queryResult->GetNumRows() << " neighbor entries loaded from SQL database");
		}
		delete queryResult;
		queryResult = NULL;
	}

	query = m_Config->GetString("SQLConfig", "PermanentEndpointsQuery", "");
	if (!query.IsEmpty()) {
		PTRACE(4, "GK\tLoading permanent endpoints from SQL database");
		PStringArray params;
		params += GKName();
		queryResult = sqlConn->ExecuteQuery(query, &params);
		if (queryResult == NULL)
			PTRACE(1, "GK\tFailed to load permanent endpoints from SQL database: timeout or fatal error");
		else if (!queryResult->IsValid())
			PTRACE(1, "GK\tFailed to load permanent endpoints from SQL database ("
				<< queryResult->GetErrorCode() << "): " << queryResult->GetErrorMessage()
				);
		else if (queryResult->GetNumFields() < 4)
			PTRACE(1, "GK\tFailed to load permanent endpoints from SQL database: at least 4 columns must be present in the result set");
		else {
			PString key;
			PString value;
			while (queryResult->FetchRow(params)) {
				key = params[0];
				if (!params[1])
					key += ":" + params[1];
				value = params[2];
				if (!params[3])
					value += ";" + params[3];
				m_extConfig->SetString("RasSrv::PermanentEndpoints", key, value);
			}
			PTRACE(4, "GK\t" << queryResult->GetNumRows() << " permanent endpoints loaded from SQL database");
		}
		delete queryResult;
		queryResult = NULL;
	}

	query = m_Config->GetString("SQLConfig", "GWPrefixesQuery", "");
	if (!query.IsEmpty()) {
		PTRACE(4, "GK\tLoading gateway prefixes from SQL database");
		PStringArray params;
		params += GKName();
		queryResult = sqlConn->ExecuteQuery(query, &params);
		if (queryResult == NULL)
			PTRACE(1, "GK\tFailed to load gateway prefixes from SQL database: timeout or fatal error");
		else if (!queryResult->IsValid())
			PTRACE(1, "GK\tFailed to load gateway prefixes from SQL database ("
				<< queryResult->GetErrorCode() << "): " << queryResult->GetErrorMessage()
				);
		else if (queryResult->GetNumFields() < 2)
			PTRACE(1, "GK\tFailed to load gateway prefixes from SQL database: at least 4 columns must be present in the result set");
		else {
			while (queryResult->FetchRow(params))
				m_extConfig->SetString("RasSrv::GWPrefixes", 
					params[0], params[1]
					);
			PTRACE(4, "GK\t" << queryResult->GetNumRows() << " gateway prefixes loaded from SQL database");
		}
		delete queryResult;
		queryResult = NULL;
	}
	
	delete sqlConn;
	sqlConn = NULL;
	PTRACE(3, "GK\tSQL config connection closed");
}

PConfig* Toolkit::ReloadConfig()
{
	if (!m_ConfigDirty)
		CreateConfig();
	else // the config have been changed via status port, use it directly
		m_ConfigDirty = false;

	ReloadSQLConfig();
	
	m_RouteTable.InitTable();
	m_ProxyCriterion.LoadConfig(m_Config);
	m_Rewrite.LoadConfig(m_Config);
	m_Rewrite.LoadConfig(m_extConfig, false);

#ifdef HAS_ACCT
	const BOOL nasStart = (m_acctList == NULL);
	try { 
		delete m_acctList; 
	} catch( ... ) {
		PTRACE(0,"GKACCT\tException caught during accounting modules cleanup");
	}
	m_acctList = new GkAcctLoggers(*m_Config);
	try {
		if( nasStart && (m_acctList != NULL) )
		{
			callptr dummycall;
			m_acctList->LogAcctEvent( GkAcctLogger::AcctOn, dummycall );
		}
	} catch( ... ) {
		PTRACE(0,"GKACCT\tException caught during Accounting-On event logging");
	}
#endif
	return m_Config; 
}

#ifdef HAS_ACCT
PString Toolkit::GenerateAcctSessionId()
{
	PWaitAndSignal lock(m_acctSessionMutex);
	return psprintf("%08x%08x", m_acctSessionBase, ++m_acctSessionCounter);
}
#endif

BOOL Toolkit::MatchRegex(const PString &str, const PString &regexStr)
{
	PINDEX pos=0;
	PRegularExpression regex(regexStr, PRegularExpression::Extended);
	if(regex.GetErrorCode() != PRegularExpression::NoError) {
		PTRACE(2, "Errornous '"<< regex.GetErrorText() <<"' compiling regex: " << regexStr);
		return FALSE;
	}
	if(!regex.Execute(str, pos)) {
		PTRACE(5, "Gk\tRegex '"<<regexStr<<"' did not match '"<<str<<"'");
		return FALSE;
	}
	return TRUE;
}



bool Toolkit::RewriteE164(H225_AliasAddress &alias)
{ 
	if (alias.GetTag() != H225_AliasAddress::e_dialedDigits) 
		return FALSE;
	
	PString E164 = H323GetAliasAddressString(alias);

	bool changed = RewritePString(E164);
	if (changed)
		H323SetAliasAddress(E164, alias);
	
	return changed;
}

bool Toolkit::RewriteE164(H225_ArrayOf_AliasAddress & aliases)
{
	bool changed = false;
	for (PINDEX n = 0; n < aliases.GetSize(); ++n)
		changed |= RewriteE164(aliases[n]);
	return changed;
}

const PString 
Toolkit::GKName() 
{
  return GkConfig()->GetString("Name", "OpenH323GK"); //use default section (MM 06.11.01)
}



// a int to print
#ifdef P_PTHREADS
static const int INT_PTHREADS = 1;
#else
static const int INT_PTHREADS = 0;
#endif

const PString 
Toolkit::GKVersion() 
{
	return PString(PString::Printf,
		"Gatekeeper(%s) Version(%s) Ext(pthreads=%d,acct=%d,radius=%d,mysql=%d,pgsql=%d,ldap=%d,large_fdset=%d) Build(%s, %s) Sys(%s %s %s)\r\n",
		(const unsigned char*)(PProcess::Current().GetManufacturer()),
		(const unsigned char*)(PProcess::Current().GetVersion(TRUE)),
		INT_PTHREADS,
#ifdef HAS_ACCT
		1,
#else
		0,
#endif
#ifdef HAS_RADIUS
		1,
#else
		0,
#endif
#if HAS_MYSQL
		1,
#else
		0,
#endif
#if HAS_PGSQL
		1,
#else
		0,
#endif
#if defined(HAS_LDAP) || defined(HAS_WLDAP)
		1,
#else
		0,
#endif
#ifdef LARGE_FDSET
		(int)LARGE_FDSET,
#else
		0,
#endif
		__DATE__, __TIME__,
		(const unsigned char*)(PProcess::GetOSName()),
		(const unsigned char*)(PProcess::GetOSHardware()),
		(const unsigned char*)(PProcess::GetOSVersion())
		);
}



int
Toolkit::GetInternalExtensionCode( const unsigned &country, 
								   const unsigned &extension, 
								   const unsigned &manufacturer) const 
{
	switch(country) {
	case t35cOpenOrg: 
		switch(manufacturer) {
		case t35mOpenOrg:
			switch(extension) {
				case t35eFailoverRAS: return iecFailoverRAS;
			}
		}
	}

	// default for all other cases
	return iecUnknown;
}


bool Toolkit::AsBool(const PString & str) 
{
	if (str.IsEmpty())
		return false;
	const unsigned char c = (unsigned char)tolower(str[0]);
	return ( c=='t' || c=='1' || c=='y' || c=='a' );
}

void Toolkit::GetNetworkFromString(const PString & cfg, PIPSocket::Address & network, PIPSocket::Address & netmask)
{
	if (cfg *= "ALL") {
		network = netmask = INADDR_ANY;
		return;
	}
	PStringArray net = cfg.Tokenise("/", FALSE);
	if (net.GetSize() < 2) {
		netmask = (DWORD(~0));
	} else if (net[1].Find('.') == P_MAX_INDEX) {
		// CIDR notation
		DWORD n = (DWORD(~0) >> net[1].AsInteger());
		netmask = PIPSocket::Host2Net(~n);
	} else {
		// decimal dot notation
		netmask = net[1];
	}
	network = PIPSocket::Address(net[0]) & netmask; // normalize
}

PString Toolkit::CypherDecode(const PString & key, const PString & crypto, int s)
{
	size_t sz = key.GetLength();
	if (sz > sizeof(PTEACypher::Key))
		sz = sizeof(PTEACypher::Key);
	PTEACypher::Key thekey;
	memset(&thekey, s, sizeof(PTEACypher::Key));
	memcpy(&thekey, (const char *)key, sz);
	PTEACypher cypher(thekey);

	return cypher.Decode(crypto);
}
