//////////////////////////////////////////////////////////////////
//
// RAS-Server for H.323 gatekeeper
//
// $Id: RasSrv.cxx,v 1.65.2.129 2004/05/25 19:54:03 zvision Exp $
//
// This work is published under the GNU Public License (GPL)
// see file COPYING for details.
// We also explicitely grant the right to link this code
// with the OpenH323 library.
//
// History:
//  990500  initial version (Xiang Ping Chen, Rajat Todi, Joe Metzger)
//  990600  ported to OpenH323 V. 1.08 (Jan Willamowius)
//  990620  bugfix for unregistration (Jan Willamowius)
//  990702  code cleanup, small fixes for compilation under Visual C++ 5 (Jan Willamowius)
//  990924  bugfix for registrations without terminalAlias (Jan Willamowius)
//  991016  clean shutdown (Jan Willamowius)
//  991027  added support for LRQ (Ashley Unitt)
//  991100  new call table (Jan Willamowius)
//  991100  status messages (Henrik Joerring)
//
//////////////////////////////////////////////////////////////////

#if (_MSC_VER >= 1200)  
#pragma warning( disable : 4800 ) // one performance warning off
#pragma warning( disable : 4786 ) // warning about too long debug symbol off
#define snprintf _snprintf
#endif

#ifndef ARJREASON_ROUTECALLTOGATEKEEPER
#define ARJREASON_ROUTECALLTOGATEKEEPER 1
#endif

#include <ptlib.h>
#include <string>
#include <set>
#include <ptlib/sockets.h>
#include <h225.h>
#include <h323pdu.h>
#include "gk_const.h"
#include "gk.h"
#include "SoftPBX.h"
#include "GkStatus.h"
#include "GkClient.h"
#include "ProxyThread.h"
#include "gkauth.h"
#include "gkDestAnalysis.h"
#include "WaitingARQ.h"
#if defined(HAS_LDAP)		// shall use LDAP
#include "gkldap.h"
#endif
#if defined(HAS_WLDAP)		// shall use WLDAP
#include "gk_wldap.h"
#endif
#include "RasSrv.h"

H323RasSrv *RasThread = 0;
const char* const NeighborSection = "RasSrv::Neighbors";
const char* const LRQFeaturesSection = "RasSrv::LRQFeatures";
const char* const H235_OID_MD5 = "1.2.840.113549.2.5";
const char* const H235_OID_U = "0.0.8.235.0.2.6";

namespace {
// prevent re-registration of end points on shutdown
bool bOnRasSrvCloseRejectRRQ = false;
const char* const H235_OIDS[] = { H235_OID_MD5, H235_OID_U };
}


class NBPendingList : public PendingList {
public:
	NBPendingList(H323RasSrv *rs, int ttl) : PendingList(rs, ttl) {}

	bool Insert(const H225_AdmissionRequest & obj_arq, const endptr & reqEP, long callDurationLimit = -1);
	void ProcessLCF(const H225_RasMessage & obj_ras);
	void ProcessLRJ(const H225_RasMessage & obj_ras);
};

class NeighborList {
	class Neighbor {
	public:
		Neighbor(const PString &, const PString &);
		bool SendLRQ(int seqNum, const H225_AdmissionRequest &, H323RasSrv *, const endptr &) const;
		bool ForwardLRQ(PIPSocket::Address, const H225_LocationRequest &, H323RasSrv *) const;
		bool CheckIP(PIPSocket::Address ip) const;
		PString GetPassword() const;
		int PrefixMatch(const H225_ArrayOf_AliasAddress &);
		bool AlwaysSendLRQ;

		typedef std::vector<std::string>::iterator prefix_iterator;
		typedef std::vector<std::string>::const_iterator const_prefix_iterator;
private:
		bool InternalSendLRQ(int seqNum, const H225_AdmissionRequest &, H323RasSrv *, const endptr &) const;
		bool InternalForwardLRQ(const H225_LocationRequest &, H323RasSrv *, int) const;
		bool ResolveName(PIPSocket::Address & ip) const;

		PString m_gkid;
		PString m_name;
		PString m_password;
		bool m_dynamic;
		mutable PIPSocket::Address m_ip;
		WORD m_port;
		std::vector<std::string> Prefixes;
	};

public:
	class NeighborPasswordAuth : public SimplePasswordAuth {
	public:
		NeighborPasswordAuth(PConfig *cfg, const char *n) : SimplePasswordAuth(cfg, n) {}

	private:
		virtual int Check(const H225_GatekeeperRequest &, unsigned &)     { return e_next; }
		virtual int Check(H225_RegistrationRequest &, unsigned &)   { return e_next; }
		virtual int Check(const H225_UnregistrationRequest &, unsigned &) { return e_next; }
		virtual int Check(const H225_AdmissionRequest &, unsigned &)      { return e_next; }
		virtual int Check(const H225_BandwidthRequest &, unsigned &)      { return e_next; }
		virtual int Check(const H225_DisengageRequest &, unsigned &)      { return e_next; }
		virtual int Check(const H225_LocationRequest &, unsigned &);
		virtual int Check(const H225_InfoRequest &, unsigned &)		  { return e_next; }

		virtual PString GetPassword(const PString &);
	};

	typedef std::list<Neighbor>::iterator iterator;
	typedef std::list<Neighbor>::const_iterator const_iterator;

	NeighborList(H323RasSrv *);
	void LoadConfig(
		PConfig* cfg,
		bool erase = true
		);
	int SendLRQ(int seqNum, const H225_AdmissionRequest &, const endptr &);
	int ForwardLRQ(PIPSocket::Address, const H225_LocationRequest &);
	bool CheckIP(PIPSocket::Address ip) const;
	// only valid after calling CheckIP
	PString GetPassword() const { return tmppasswd; }
	void InsertSiblingIP(PIPSocket::Address ip) { siblingIPs.insert(ip); }

	class InvalidNeighbor {};

private:
	std::list<Neighbor> nbList;
	H323RasSrv *RasSrv;
	std::set<PIPSocket::Address> siblingIPs;
	mutable PString tmppasswd;
};

bool PendingList::PendingARQ::DoACF(H323RasSrv *RasSrv, const endptr & called) const
{
	callptr call = RasSrv->ReplyARQ(m_reqEP, called, m_arq);
	if( call && m_durationLimit > 0 )
		call->SetDurationLimit(m_durationLimit);
	return true;
}

void PendingList::Check()
{
	PWaitAndSignal lock(usedLock);
	iterator Iter = find_if(arqList.begin(), arqList.end(), not1(bind2nd(mem_fun(&PendingARQ::IsStaled), pendingTTL)));
	for_each(arqList.begin(), Iter, bind2nd(mem_fun(&PendingARQ::DoARJ), RasSrv));
	for_each(arqList.begin(), Iter, delete_arq);
	arqList.erase(arqList.begin(), Iter);
}

bool PendingList::ProcessRIP(
	const H225_RequestInProgress& rip
	)
{
	PWaitAndSignal lock(usedLock);
	iterator i = FindBySeqNum(rip.m_requestSeqNum.GetValue());
	if (i != arqList.end()) {
		(*i)->SetRequestTime(PTime());
		return true;
	} else
		return false;
}


bool NBPendingList::Insert(const H225_AdmissionRequest & obj_arq, const endptr & reqEP, long callDurationLimit)
{
	// TODO: check if ARQ duplicate
	int seqNumber = RasSrv->GetRequestSeqNum();
	int nbCount = RasSrv->GetNeighborsGK()->SendLRQ(seqNumber, obj_arq, reqEP);
	if (nbCount > 0) {
		PWaitAndSignal lock(usedLock);
		arqList.push_back(new PendingARQ(seqNumber, obj_arq, reqEP, nbCount,callDurationLimit));
		return true;
	}
	return false;
}

void NBPendingList::ProcessLCF(const H225_RasMessage & obj_ras)
{
	const H225_LocationConfirm & obj_lcf = obj_ras;

	// TODO: check if the LCF is sent from my neighbors
	PWaitAndSignal lock(usedLock);
	iterator Iter = FindBySeqNum(obj_lcf.m_requestSeqNum.GetValue());
	if (Iter == arqList.end()) {
		PTRACE(2, "GK\tUnknown LCF, ignore!");
		return;
	}
	if (endptr called = RegistrationTable::Instance()->InsertRec(const_cast<H225_RasMessage &>(obj_ras))) {
		(*Iter)->DoACF(RasSrv, called);
	} else {
		PTRACE(2, "GK\tUnable to add EP for this LCF!");
	}

	Remove(Iter);
}

void NBPendingList::ProcessLRJ(const H225_RasMessage & obj_ras)
{
	const H225_LocationReject & obj_lrj = obj_ras;

	// TODO: check if the LRJ is sent from my neighbors
	PWaitAndSignal lock(usedLock);
	iterator Iter = FindBySeqNum(obj_lrj.m_requestSeqNum.GetValue());
	if (Iter == arqList.end()) {
		PTRACE(2, "GK\tUnknown LRJ, ignore!");
		return;
	}
	if ((*Iter)->DecCount() == 0) {
		(*Iter)->DoARJ(RasSrv);
		Remove(Iter);
	}
}

NeighborList::Neighbor::Neighbor(const PString & gkid, const PString & cfgs) : m_gkid(gkid)
{
	Prefixes.reserve(8);
	AlwaysSendLRQ = false;
	PStringArray cfg(cfgs.Tokenise(";", TRUE));
	PString ipAddr = cfg[0].Trim();
	PINDEX p = ipAddr.Find(':');
	m_name = ipAddr.Left(p);
	m_port = (WORD)((p != P_MAX_INDEX) ? ipAddr.Mid(p+1).AsUnsigned() : GK_DEF_UNICAST_RAS_PORT);
	if (cfg.GetSize() > 1) {
		PStringArray p = cfg[1].Tokenise(",", false);
		for (PINDEX i = 0; i < p.GetSize(); ++i) {
			if (p[i] == "*")
				AlwaysSendLRQ = TRUE;
			else
				Prefixes.push_back((const char *)p[i]);
		}
	} else {
		AlwaysSendLRQ = true;
	}
	if (cfg.GetSize() > 2)
		m_password = cfg[2];
	m_dynamic = (cfg.GetSize() > 3) ? Toolkit::AsBool(cfg[3]) : false;

	if (!m_dynamic && !PIPSocket::GetHostAddress(m_name, m_ip))
		throw InvalidNeighbor();
	PTRACE(1, "Add neighbor " << m_gkid << '(' << (m_dynamic ? m_name : m_ip.AsString()) << ':' << m_port << ')' << ((cfg.GetSize() > 1) ? (" for prefixes " + cfg[1]) : PString()));
}

inline bool NeighborList::Neighbor::CheckIP(PIPSocket::Address ip) const
{
	return (!m_dynamic || ResolveName(m_ip)) ? ip == m_ip : false;
}

inline PString NeighborList::Neighbor::GetPassword() const
{
	return m_password;
}

int NeighborList::Neighbor::PrefixMatch(const H225_ArrayOf_AliasAddress & aliases)
{
	int maxlen = -1;
	for (PINDEX i = 0; i < aliases.GetSize(); ++i) {
		const unsigned tag = aliases[i].GetTag();
		if (tag == H225_AliasAddress::e_dialedDigits
			|| tag == H225_AliasAddress::e_partyNumber
			|| tag == H225_AliasAddress::e_h323_ID) {
			const PString alias = H323GetAliasAddressString(aliases[i]);
			if( tag == H225_AliasAddress::e_h323_ID )
				if( strspn(alias,"1234567890#*") != strlen(alias) )
					continue;
			const_prefix_iterator Iter = Prefixes.begin(), eIter= Prefixes.end();
			while (Iter != eIter) {
				const int len = Iter->length();
				if ((maxlen < len) && (strncmp(alias, Iter->c_str(), len)==0)) {
					maxlen = len;
				}
				Iter++;
			}
		}
	}
	return maxlen;
}

bool NeighborList::Neighbor::SendLRQ(int seqNum, const H225_AdmissionRequest & obj_arq, H323RasSrv *RasSrv, const endptr & e) const
{
	return InternalSendLRQ(seqNum, obj_arq, RasSrv, e);
}

bool NeighborList::Neighbor::InternalSendLRQ(int seqNum, const H225_AdmissionRequest & obj_arq, H323RasSrv *RasSrv, const endptr & e) const
{
	PIPSocket::Address ip = m_ip;
	if (m_dynamic && !ResolveName(ip))
		return false;
	H225_RasMessage obj_ras;
	obj_ras.SetTag(H225_RasMessage::e_locationRequest);
	H225_LocationRequest & obj_lrq = obj_ras;
	obj_lrq.m_requestSeqNum.SetValue(seqNum);
	obj_lrq.m_replyAddress = RasSrv->GetRasAddress(ip);
	obj_lrq.m_destinationInfo = obj_arq.m_destinationInfo;

	// tell the neighbor who I am
	obj_lrq.IncludeOptionalField(H225_LocationRequest::e_sourceInfo);
	obj_lrq.m_sourceInfo = e->GetAliases();
	obj_lrq.IncludeOptionalField(H225_LocationRequest::e_gatekeeperIdentifier);
	obj_lrq.m_gatekeeperIdentifier = RasSrv->GetGKName();
	RasSrv->GetGkClient()->SetPassword(obj_lrq, Toolkit::GKName());

	int hopCount = GkConfig()->GetInteger(LRQFeaturesSection, "ForwardHopCount", 0);
	if (hopCount > 1) { // what if set hopCount = 1?
		obj_lrq.IncludeOptionalField(H225_LocationRequest::e_hopCount);
		obj_lrq.m_hopCount = hopCount;
	}

	// workaround for Cisco GK
	if (Toolkit::AsBool(GkConfig()->GetString(LRQFeaturesSection, "CiscoGKCompatible", "0"))) {
		obj_lrq.IncludeOptionalField(H225_LocationRequest::e_nonStandardData);
		obj_lrq.m_nonStandardData.m_data.SetValue("gnugk");
		obj_lrq.m_nonStandardData.m_nonStandardIdentifier.SetTag(H225_NonStandardIdentifier::e_h221NonStandard);
		H225_H221NonStandard & h221 = obj_lrq.m_nonStandardData.m_nonStandardIdentifier;
		h221.m_manufacturerCode = 18;
		h221.m_t35CountryCode = 181;
		h221.m_t35Extension = 0;
		// I don't think this is the right solution, but...
		obj_lrq.IncludeOptionalField(H225_LocationRequest::e_canMapAlias);
		obj_lrq.m_canMapAlias = TRUE;
	}

	RasSrv->SendRas(obj_ras, ip, m_port);
	return true;
}

bool NeighborList::Neighbor::ForwardLRQ(PIPSocket::Address ip, const H225_LocationRequest & obj_lrq, H323RasSrv *RasSrv) const
{
	if ((m_dynamic && !ResolveName(m_ip)) || ip == m_ip)
		return false; // don't forward to GK that sent the LRQ

	int hopCount = 0;
	if (obj_lrq.HasOptionalField(H225_LocationRequest::e_hopCount))
		hopCount = obj_lrq.m_hopCount - 1;
	else
		hopCount = GkConfig()->GetInteger(LRQFeaturesSection, "AlwaysForwardLRQ", 0);
	if (hopCount < 1)
		return false; // don't forward LRQ without hopCount unless otherwise specified

	return InternalForwardLRQ(obj_lrq, RasSrv, hopCount);
}

bool NeighborList::Neighbor::InternalForwardLRQ(const H225_LocationRequest & obj_lrq, H323RasSrv *RasSrv, int hopCount) const
{
	H225_RasMessage obj_ras;
	obj_ras.SetTag(H225_RasMessage::e_locationRequest);
	H225_LocationRequest & lrq = obj_ras;
	lrq = obj_lrq; // copy and modify
	lrq.IncludeOptionalField(H225_LocationRequest::e_hopCount);
	lrq.m_hopCount = hopCount;
	RasSrv->SendRas(obj_ras, m_ip, m_port);
	return true;
}

bool NeighborList::Neighbor::ResolveName(PIPSocket::Address & ip) const
{
	PIPSocket::ClearNameCache();
	// Retrieve the ip address at this time
	if (PIPSocket::GetHostAddress(m_name, ip)) {
		PTRACE(3, "Retrieve neighbor ip for " << m_name << '=' << ip);
		return true;
	} else {
		PTRACE(1, "Can't get neighbor ip for " << m_name);
		return false;
	}
}

NeighborList::NeighborList(H323RasSrv* rassrv) : RasSrv(rassrv)
{
}

void NeighborList::LoadConfig(
	PConfig* config,
	bool erase
	)
{
	if (erase)
		nbList.clear();
		
	PStringToString cfgs(config->GetAllKeyValues(NeighborSection));
	for (PINDEX i=0; i < cfgs.GetSize(); i++) {
		try {
			nbList.push_back(Neighbor(cfgs.GetKeyAt(i), cfgs.GetDataAt(i)));
		} catch (InvalidNeighbor) {
			PTRACE(1, "Bad neighbor " << cfgs.GetKeyAt(i));
			// ignore it :p
		}
	}
}
 
#if HAS_WAITARQ
BOOL H323RasSrv::RouteToAlias(PString TargetAlias, PString SourceEpId, PString CallRef)
{
	return wArqList->RouteToAlias(TargetAlias, SourceEpId, CallRef);
}

BOOL H323RasSrv::RouteReject(PString SourceEpId, PString CallRef)
{
	// send ARJ and remove ARQ from list
	return wArqList->RouteReject(SourceEpId, CallRef);
}
#endif
 
int NeighborList::SendLRQ(int seqNum, const H225_AdmissionRequest & obj_arq, const endptr & e)
{
	int nbCount = 0;
	int maxlen = 0;
	std::list<Neighbor> List;
	iterator Iter, eIter = nbList.end();
	for (Iter = nbList.begin(); Iter != eIter; ++Iter) {
		if (Iter->AlwaysSendLRQ) {
			if (Iter->SendLRQ(seqNum, obj_arq, RasSrv, e))
				++nbCount;
		} else {
			int len = (Iter)->PrefixMatch(obj_arq.m_destinationInfo);
			if (maxlen < len) {
				List.clear();
				maxlen = len;
			}
			if (maxlen == len) {
				List.push_back(*Iter);
			}
		}
	}
	
	eIter = List.end();
	for(Iter = List.begin(); Iter != eIter; ++Iter) {
		if (Iter->SendLRQ(seqNum, obj_arq, RasSrv, e))
			++nbCount;
	}

	PTRACE_IF(2, nbCount, "GK\tSend LRQ to " << nbCount << " neighbor(s)");
	return nbCount;
}

int NeighborList::ForwardLRQ(PIPSocket::Address ip, const H225_LocationRequest & obj_lrq)
{
	int nbCount = 0;
	int maxlen = 0;
	std::list<Neighbor> List;
	iterator Iter, eIter = nbList.end();
	for (Iter = nbList.begin(); Iter != eIter; ++Iter) {
		if (Iter->AlwaysSendLRQ) {
			if (Iter->ForwardLRQ(ip, obj_lrq, RasSrv))
				++nbCount;
		} else {
			int len = (Iter)->PrefixMatch(obj_lrq.m_destinationInfo);
			if (maxlen < len) {
				List.clear();
				maxlen = len;
			}
			if (maxlen == len) {
				List.push_back(*Iter);
			}
		}
	}
	
	eIter = List.end();
	for(Iter = List.begin(); Iter != eIter; ++Iter) {
		if (Iter->ForwardLRQ(ip, obj_lrq, RasSrv))
			++nbCount;
	}

	PTRACE_IF(2, nbCount, "GK\tForward LRQ to " << nbCount << " neighbor(s)");
	return nbCount;
}

bool NeighborList::CheckIP(PIPSocket::Address ip) const
{
	const_iterator Iter = find_if(
		nbList.begin(), nbList.end(),
		bind2nd(mem_fun_ref(&Neighbor::CheckIP), ip)
	);
	if (Iter != nbList.end()) {
		tmppasswd = Iter->GetPassword();
		return true;
	}
	std::set<PIPSocket::Address>::const_iterator it = siblingIPs.find(ip);
	return (it != siblingIPs.end());
}

int NeighborList::NeighborPasswordAuth::Check(const H225_LocationRequest & lrq, unsigned & rsn)
{
	PString nullid;
	return (!GetPassword(nullid)) ? SimplePasswordAuth::Check(lrq, rsn) : e_next;
}

PString NeighborList::NeighborPasswordAuth::GetPassword(const PString &)
{
	return RasThread->GetNeighborsGK()->GetPassword();
}

static GkAuthInit<NeighborList::NeighborPasswordAuth> N_B_A("NeighborPasswordAuth");

H323RasSrv::H323RasSrv(PIPSocket::Address _GKHome)
      : PThread(10000, NoAutoDeleteThread), requestSeqNum(0)
{
	GKHome = _GKHome;
	
	GKRasPort = (WORD)GkConfig()->GetInteger("UnicastRasPort", GK_DEF_UNICAST_RAS_PORT);

	EndpointTable = RegistrationTable::Instance(); //initialisation is done in LoadConfig
	CallTbl = CallTable::Instance();

	sigHandler = 0;

	gkClient = 0;
	authList = 0;
	destAnalysisList = 0;
	NeighborsGK = 0;

	// default routed mode
	GKRoutedSignaling = GKRoutedH245 = false;

	GkStatusThread = GkStatus::Instance();
	GkStatusThread->Initialize(GKHome);

#if HAS_WAITARQ
	wArqList = new WaitingARQlist();
#endif
 
	LoadConfig();

	arqPendingList = new NBPendingList(this, GkConfig()->GetInteger(LRQFeaturesSection, "NeighborTimeout", 2));

	// initialize the handler map
	for (unsigned i = 0; i <= H225_RasMessage::e_serviceControlResponse; ++i)
		rasHandler[i] = &H323RasSrv::OnUnknown;

	rasHandler[H225_RasMessage::e_gatekeeperRequest] =	    &H323RasSrv::OnGRQ;
	rasHandler[H225_RasMessage::e_registrationRequest] =	    &H323RasSrv::OnRRQ;
	rasHandler[H225_RasMessage::e_registrationConfirm] =	    &H323RasSrv::OnRCF;
	rasHandler[H225_RasMessage::e_registrationReject] =	    &H323RasSrv::OnRRJ;
	rasHandler[H225_RasMessage::e_unregistrationRequest] =	    &H323RasSrv::OnURQ;
	rasHandler[H225_RasMessage::e_unregistrationConfirm] =	    &H323RasSrv::OnIgnored;
	rasHandler[H225_RasMessage::e_unregistrationReject] =	    &H323RasSrv::OnIgnored;
	rasHandler[H225_RasMessage::e_admissionRequest] =	    &H323RasSrv::OnARQ;
	rasHandler[H225_RasMessage::e_admissionConfirm] =	    &H323RasSrv::OnACF;
	rasHandler[H225_RasMessage::e_admissionReject] =	    &H323RasSrv::OnARJ;
	rasHandler[H225_RasMessage::e_bandwidthRequest] =	    &H323RasSrv::OnBRQ;
	rasHandler[H225_RasMessage::e_bandwidthConfirm] =	    &H323RasSrv::OnIgnored;
	rasHandler[H225_RasMessage::e_bandwidthReject] =	    &H323RasSrv::OnIgnored;
	rasHandler[H225_RasMessage::e_disengageRequest] =	    &H323RasSrv::OnDRQ;
	rasHandler[H225_RasMessage::e_disengageConfirm] =	    &H323RasSrv::OnIgnored;
	rasHandler[H225_RasMessage::e_disengageReject] =	    &H323RasSrv::OnIgnored;
	rasHandler[H225_RasMessage::e_locationRequest] =	    &H323RasSrv::OnLRQ;
	rasHandler[H225_RasMessage::e_locationConfirm] =	    &H323RasSrv::OnLCF;
	rasHandler[H225_RasMessage::e_locationReject] =		    &H323RasSrv::OnLRJ;
	rasHandler[H225_RasMessage::e_requestInProgress] =	    &H323RasSrv::OnRIP;
	rasHandler[H225_RasMessage::e_infoRequestResponse] =	    &H323RasSrv::OnIRR;
	rasHandler[H225_RasMessage::e_resourcesAvailableIndicate] = &H323RasSrv::OnRAI;
	rasHandler[H225_RasMessage::e_serviceControlIndication] =   &H323RasSrv::OnSCI;
	rasHandler[H225_RasMessage::e_serviceControlResponse] =	    &H323RasSrv::OnSCR;
}

H323RasSrv::~H323RasSrv()
{
	delete gkClient;
	delete authList;
	delete destAnalysisList;
	delete NeighborsGK;
	delete arqPendingList;
#if HAS_WAITARQ
	delete wArqList ;
#endif
 
}

void H323RasSrv::SetRoutedMode(bool routedSignaling, bool routedH245)
{
	GKRoutedSignaling = routedSignaling;
	if (GKRoutedSignaling) {
		if (sigHandler)
			sigHandler->LoadConfig();
		else
			sigHandler = new HandlerList(GKHome);
		GKCallSigPort = sigHandler->GetCallSignalPort();
	} else {
		delete sigHandler;
		sigHandler = 0;
	}
	GKRoutedH245 = routedH245;

#if PTRACING
	const char *modemsg = GKRoutedSignaling ? "Routed" : "Direct";
	const char *h245msg = GKRoutedH245 ? "Enabled" : "Disabled";
	PTRACE(2, "GK\tUsing " << modemsg << " Signalling");
	PTRACE(2, "GK\tH.245 Routed " << h245msg);
#endif
}

// set the signaling mode according to the config file
// don't change it if not specified in the config
void H323RasSrv::SetRoutedMode()
{
	if (GkConfig()->HasKey(RoutedSection, "GKRouted"))
		GKRoutedSignaling = Toolkit::AsBool(GkConfig()->GetString(RoutedSection, "GKRouted", "0"));
	if (GkConfig()->HasKey(RoutedSection, "H245Routed"))
		GKRoutedH245 = Toolkit::AsBool(GkConfig()->GetString(RoutedSection, "H245Routed", "0"));
	SetRoutedMode(GKRoutedSignaling, GKRoutedH245);
}

PString H323RasSrv::GetParent() const
{
	return gkClient->GetParent();
}

bool H323RasSrv::AcceptUnregisteredCalls(PIPSocket::Address ip, bool & fp) const
{
	fp = (gkClient->IsRegistered() && gkClient->CheckGKIP(ip));
	return AcceptUnregCalls || (AcceptNBCalls ? (fp || NeighborsGK->CheckIP(ip)) : false);
}

void H323RasSrv::ClearAltGKsTable()
{
	redirectGK = e_noRedirect;
	altGKs.SetSize(0);
	altGKsAddr.clear();
	skipAddr.clear();
	altGKsPort.clear();
	altGKsSize = 0;
	epLimit = callLimit = P_MAX_INDEX;
}

void H323RasSrv::GetAlternateGK()
{
	ClearAltGKsTable();

	PString redirect(GkConfig()->GetString("RedirectGK", ""));
	if (redirect *= "temporary")
		redirectGK = e_temporaryRedirect;
	else if (redirect *= "permanent")
		redirectGK = e_permanentRedirect;
	else {
		PStringArray limits(redirect.Tokenise("|,;&", FALSE));
		for (PINDEX i = 0; i < limits.GetSize(); ++i) {
			PINDEX gr = limits[i].Find('>');
			if (gr != P_MAX_INDEX) {
				PCaselessString tkn(limits[i].Left(gr));
				if (tkn.Find("endpoints") != P_MAX_INDEX) {
					epLimit = limits[i].Mid(gr + 1).AsInteger();
					PTRACE(2, "GK\tSet registration limit to " << epLimit);
				} else if (tkn.Find("calls") != P_MAX_INDEX) {
					callLimit = limits[i].Mid(gr + 1).AsInteger();
					PTRACE(2, "GK\tSet call limit to " << callLimit);
				}
			}
		}
	}

	PString skips(GkConfig()->GetString("SkipForwards", ""));
	PStringArray skipips(skips.Tokenise(" ,;\t", FALSE));
	PINDEX skipSize = skipips.GetSize();
	if (skipSize > 0)
		for (PINDEX i = 0; i < skipSize; ++i)
			skipAddr.push_back(PIPSocket::Address(skipips[i]));

	PString param(GkConfig()->GetString("AlternateGKs", ""));
	if (param.IsEmpty())
		return;

	PStringArray altgks(param.Tokenise(" ,;\t", FALSE));
	altGKs.SetSize(altgks.GetSize());

	for (PINDEX idx = 0; idx < altgks.GetSize(); ++idx) {
		const PStringArray tokens = altgks[idx].Tokenise(":", FALSE);
		if (tokens.GetSize() < 4) {
			PTRACE(1,"GK\tFormat error in AlternateGKs");
			continue; 
		}

		H225_AlternateGK & A = altGKs[idx];
		A.m_rasAddress = SocketToH225TransportAddr(PIPSocket::Address(tokens[0]), (WORD)(tokens[1].AsUnsigned()));
		A.m_needToRegister = Toolkit::AsBool(tokens[2]);
		A.m_priority = tokens[3].AsInteger();
		if (tokens.GetSize() > 4) {
			A.IncludeOptionalField(H225_AlternateGK::e_gatekeeperIdentifier);
			A.m_gatekeeperIdentifier = tokens[4];
		}
	}

	PString sendto(GkConfig()->GetString("SendTo", ""));
	PStringArray svrs(sendto.Tokenise(" ,;\t", FALSE));
	if ((altGKsSize = svrs.GetSize()) > 0)
		for (PINDEX i = 0; i < altGKsSize; ++i) {
			PStringArray tokens(svrs[i].Tokenise(":", FALSE));
			altGKsAddr.push_back(PIPSocket::Address(tokens[0]));
			altGKsPort.push_back((tokens.GetSize() > 1) ? WORD(tokens[1].AsUnsigned()) : GK_DEF_UNICAST_RAS_PORT);
		}
}

void H323RasSrv::LoadConfig()
{
	PWaitAndSignal lock(loadLock);

	AcceptNBCalls = Toolkit::AsBool(GkConfig()->GetString(RoutedSection, "AcceptNeighborsCalls", "1"));
	AcceptUnregCalls = Toolkit::AsBool(GkConfig()->GetString(RoutedSection, "AcceptUnregisteredCalls", "0"));
	bRemoveCallOnDRQ = Toolkit::AsBool(GkConfig()->GetString(RoutedSection, "RemoveCallOnDRQ", 1));

	GetAlternateGK();

	// add authenticators
	delete authList;
	authList = new GkAuthenticatorList(GkConfig());

	// add destination analysis
	delete destAnalysisList;
	destAnalysisList = new GkDestAnalysisList(GkConfig());
	EndpointTable->Initialize(*destAnalysisList);
	
#if defined(HAS_LDAP)		// shall use LDAP
	
	// initialize LDAP
	GkLDAP::Instance()->Initialize(*GkConfig());
	
#endif // HAS_LDAP

#if defined(HAS_WLDAP)		// shall use WLDAP
	#ifdef WIN32
		// initialize LDAP
		GkLDAP::Instance()->Initialize(*GkConfig());
	#endif
#endif // HAS_WLDAP

#if defined(HAS_WAITARQ)		// supports VirtualQueues
	// (re) load config
	wArqList->LoadConfig();
#endif
	
	// add neighbors
	delete NeighborsGK;
	NeighborsGK = new NeighborList(this);
	NeighborsGK->LoadConfig(GkConfig());
	NeighborsGK->LoadConfig(GkExtConfig(), false);
	
	if (gkClient) { // don't create GkClient object at the first time
		delete gkClient;
		gkClient = new GkClient(this);
	}
}

void H323RasSrv::Close(void)
{
	PTRACE(2, "GK\tClosing RasThread");

	// disconnect all calls
	CallTbl->ClearTable();

	delete sigHandler;
	sigHandler = 0;

	// prevent re-registation attempts - set flag bOnRasSrvCloseRejectRRQ
	bOnRasSrvCloseRejectRRQ = TRUE;

	if (gkClient->IsRegistered())
		gkClient->SendURQ();

	// send all registered clients a URQ
	cout << "\nUnregister endpoints  . . ." << endl;
	UnregisterAllEndpoints();
	cout << "Endpoints unregistered" << endl;

	// close the listener thread
	listener.Close();

	// terminate status thread
	if (GkStatusThread) {
		GkStatusThread->Close();
		GkStatusThread->WaitForTermination();
		delete GkStatusThread;
		GkStatusThread = 0;
	}
 
	PTRACE(1, "GK\tRasSrv closed");
}


void H323RasSrv::UnregisterAllEndpoints(void)
{
	SoftPBX::UnregisterAllEndpoints();
}

bool H323RasSrv::IsForwardedMessage(const H225_NonStandardParameter * nonStandardParam, const PIPSocket::Address & rx_addr) const
{
	// mechanism 1: forwarding detection per "flag"
	if (nonStandardParam && nonStandardParam->m_nonStandardIdentifier.GetTag() == H225_NonStandardIdentifier::e_h221NonStandard) {
		const H225_H221NonStandard & nonStandard = nonStandardParam->m_nonStandardIdentifier;
		if (Toolkit::Instance()->GetInternalExtensionCode(nonStandard) == Toolkit::iecFailoverRAS) {
			PTRACE(6, "GK\tForwarded RAS detected!");
			return true;
		}
	// mechanism 2: forwarding detection per "from"
	}
	if (find(skipAddr.begin(), skipAddr.end(), rx_addr) != skipAddr.end()) {
		PTRACE(6, "GK\tSkip forwarding RAS to other GK");
		return true;
	}
	return false;
}

bool H323RasSrv::ForwardRasMsg(H225_RasMessage msg) // not passed as const, ref or pointer!
{
	if (altGKsSize <= 0)
		return false;

	H225_NonStandardParameter *nonStandardParam = 0;
	switch (msg.GetTag())
	{
		case H225_RasMessage::e_gatekeeperRequest: {
			H225_GatekeeperRequest & o = msg;
			o.IncludeOptionalField(H225_GatekeeperRequest::e_nonStandardData);
			nonStandardParam = &o.m_nonStandardData;
			break;
		}
		case H225_RasMessage::e_registrationRequest: {
			H225_RegistrationRequest & o = msg;
			o.IncludeOptionalField(H225_RegistrationRequest::e_nonStandardData);
			nonStandardParam = &o.m_nonStandardData;
			if (o.HasOptionalField(H225_RegistrationRequest::e_endpointIdentifier))
				nonStandardParam->m_data = o.m_endpointIdentifier;
			break;
		}
		case H225_RasMessage::e_unregistrationRequest: {
			H225_UnregistrationRequest & o = msg;
			o.IncludeOptionalField(H225_UnregistrationRequest::e_nonStandardData);
			nonStandardParam = &o.m_nonStandardData;
			break;
		}
		default:
			PTRACE(2,"Warning: unsupported RAS message type for forwarding; field 'forwarded' not included in msg.");
			break;
	}

	// include the "this is a forwared message" tag (could be a static variable to increase performance)
	if (nonStandardParam) {
		H225_NonStandardIdentifier & id = nonStandardParam->m_nonStandardIdentifier;
		id.SetTag(H225_NonStandardIdentifier::e_h221NonStandard);
		H225_H221NonStandard & h221 = id;
		h221.m_t35CountryCode   = Toolkit::t35cOpenOrg;
		h221.m_t35Extension     = Toolkit::t35eFailoverRAS;
		h221.m_manufacturerCode = Toolkit::t35mOpenOrg;
	}

	for (PINDEX i = 0; i < altGKsSize; ++i) {
		PTRACE(4, "Forwarding RAS to " << altGKsAddr[i] << ':' << altGKsPort[i]);
		SendReply(msg, altGKsAddr[i], altGKsPort[i], listener);
	}

	return true;
}

template<class ORAS, class DRAS> void CopyNonStandardData(const ORAS & omsg, DRAS & dmsg)
{
	if (omsg.HasOptionalField(ORAS::e_nonStandardData)) {
		dmsg.IncludeOptionalField(DRAS::e_nonStandardData);
		dmsg.m_nonStandardData = omsg.m_nonStandardData;
	}
}

// Set the cryptoTokens field if ORAS has it. This is just a workaround.
// It avoids an OpenH323 client (like ohphone)
// refuse to accept our confirm messages.
template<class ORAS, class DRAS> void SetCryptoTokens(const ORAS & /*omsg*/, DRAS & /*dmsg*/)
{
// cryptoTokens should not be copied - these should be generated
// by the gatekeeper, to allow authentication of reply messages
// H.235 Auth Procedure I should be implemented here instead to
// generate nestedCryptoTokens for each reply message
/* 
	if (omsg.HasOptionalField(ORAS::e_cryptoTokens))
		for (PINDEX i = 0; i < omsg.m_cryptoTokens.GetSize(); ++i)
			if (omsg.m_cryptoTokens[i].GetTag() == H225_CryptoH323Token::e_cryptoEPPwdHash) {
				// only copy H235AuthSimpleMD5
				dmsg.IncludeOptionalField(DRAS::e_cryptoTokens);
				dmsg.m_cryptoTokens.SetSize(1);
				dmsg.m_cryptoTokens[0] = omsg.m_cryptoTokens[i];
				return;
			}
*/
}

/* Gatekeeper request */
BOOL H323RasSrv::OnGRQ(const PIPSocket::Address & rx_addr, H225_RasMessage & obj_grq, H225_RasMessage & obj_rpl)
{
	PTRACE(1, "GK\tGRQ Received");

	const H225_GatekeeperRequest & obj_gr = obj_grq;

	// reply only if gkID matches
	if ( obj_gr.HasOptionalField ( H225_GatekeeperRequest::e_gatekeeperIdentifier ) )
		if (obj_gr.m_gatekeeperIdentifier.GetValue() != GetGKName()) {
			PTRACE(2, "GK\tGRQ is not meant for this gatekeeper");
			return FALSE;
		}

	BOOL bShellSendReply = !(IsForwardedMessage(obj_gr.HasOptionalField(H225_GatekeeperRequest::e_nonStandardData) ? &obj_gr.m_nonStandardData : 0, rx_addr));

	PString msg;
	PString aliasListString(obj_gr.HasOptionalField(H225_GatekeeperRequest::e_endpointAlias) ?
		AsString(obj_gr.m_endpointAlias) : PString());

	unsigned rsn = H225_GatekeeperRejectReason::e_securityDenial;
	if ((redirectGK != e_noRedirect) || !authList->Check(obj_gr, rsn)) {
		obj_rpl.SetTag(H225_RasMessage::e_gatekeeperReject); 
		H225_GatekeeperReject & obj_grj = obj_rpl;
		obj_grj.m_protocolIdentifier = obj_gr.m_protocolIdentifier;
		if (redirectGK != e_noRedirect) {
			rsn = H225_GatekeeperRejectReason::e_resourceUnavailable;
			SetAltGKInfo(obj_grj);
		}
		obj_grj.m_requestSeqNum = obj_gr.m_requestSeqNum;
		obj_grj.m_rejectReason.SetTag(rsn);
		obj_grj.IncludeOptionalField(obj_grj.e_gatekeeperIdentifier);
		obj_grj.m_gatekeeperIdentifier.SetValue( GetGKName() );
		msg = PString(PString::Printf, "GRJ|%s|%s|%s|%s;\r\n",
			inet_ntoa(rx_addr),
			(const unsigned char *) aliasListString,
			(const unsigned char *) AsString(obj_gr.m_endpointType),
			(const unsigned char *) obj_grj.m_rejectReason.GetTagName()
			);
	} else {
		obj_rpl.SetTag(H225_RasMessage::e_gatekeeperConfirm); 
		H225_GatekeeperConfirm & obj_gcf = obj_rpl;

		obj_gcf.m_requestSeqNum = obj_gr.m_requestSeqNum;
		obj_gcf.m_protocolIdentifier = obj_gr.m_protocolIdentifier;
		obj_gcf.m_rasAddress = GetRasAddress(rx_addr);
		obj_gcf.IncludeOptionalField(obj_gcf.e_gatekeeperIdentifier);
		obj_gcf.m_gatekeeperIdentifier.SetValue( GetGKName() );
		if (obj_gr.HasOptionalField(H225_GatekeeperRequest::e_supportsAltGK))
			SetAlternateGK(obj_gcf);
/*		if (obj_gr.HasOptionalField(H225_GatekeeperRequest::e_authenticationCapability) && obj_gr.HasOptionalField(H225_GatekeeperRequest::e_algorithmOIDs)) {
			size_t j = 0;
			bool found = false;
			for (int i = 0; !found && i < obj_gr.m_authenticationCapability.GetSize(); ++i)
				if (obj_gr.m_authenticationCapability[i].GetTag() == H235_AuthenticationMechanism::e_pwdHash)
					for (j = 0; !found && j < sizeof(H235_OIDS); ++j)
						for (int k = 0; k < obj_gr.m_algorithmOIDs.GetSize(); ++k)
							if ((found = (obj_gr.m_algorithmOIDs[k].AsString() == H235_OIDS[j])))
								break;
			if (found) {
				obj_gcf.IncludeOptionalField(H225_GatekeeperConfirm::e_authenticationMode);
				obj_gcf.m_authenticationMode.SetTag(H235_AuthenticationMechanism::e_pwdHash);
				obj_gcf.IncludeOptionalField(H225_GatekeeperConfirm::e_algorithmOID);
				obj_gcf.m_algorithmOID = H235_OIDS[--j];
			}
		}
*/
		SelectH235Capability( obj_gr, obj_gcf  );
		
		//if (bShellSendReply)cw	  need to forward GRQ?
		//	ForwardRasMsg(obj_grq);

		msg = PString(PString::Printf, "GCF|%s|%s|%s;\r\n", 
			inet_ntoa(rx_addr),
			(const unsigned char *) aliasListString,
			(const unsigned char *) AsString(obj_gr.m_endpointType) );
	}

	PTRACE(2, msg);
	GkStatusThread->SignalStatus(msg);
		
	return bShellSendReply;
}

void H323RasSrv::BuildRCF(H225_RasMessage & obj_ras, const H225_RegistrationRequest & rrq, const endptr & ep, const PIPSocket::Address & rx_addr)
{
	obj_ras.SetTag(H225_RasMessage::e_registrationConfirm); 
	H225_RegistrationConfirm & rcf = obj_ras;
	rcf.m_requestSeqNum = rrq.m_requestSeqNum;
	rcf.m_protocolIdentifier = rrq.m_protocolIdentifier;
	rcf.m_callSignalAddress.SetSize(1);
	rcf.m_callSignalAddress[0] = GetCallSignalAddress(rx_addr);
	rcf.m_endpointIdentifier = ep->GetEndpointIdentifier();
	rcf.IncludeOptionalField(H225_RegistrationConfirm::e_gatekeeperIdentifier);
	rcf.m_gatekeeperIdentifier = GetGKName();
	if (ep->GetTimeToLive() > 0) {
		rcf.IncludeOptionalField(H225_RegistrationConfirm::e_timeToLive);
		rcf.m_timeToLive = ep->GetTimeToLive();
	}
	SetCryptoTokens(rrq, rcf);
}

/* Registration Request */
BOOL H323RasSrv::OnRRQ(const PIPSocket::Address & rx_addr, H225_RasMessage & obj_rrq, H225_RasMessage & obj_rpl)
{
	PTRACE(1, "GK\tRRQ Received");

	H225_RegistrationRequest & obj_rr = obj_rrq;
	BOOL bReject = FALSE;		// RRJ with any other reason from #rejectReason#
	unsigned rejectReason = 0; // avoid warning

	PString alias;

	H225_TransportAddress SignalAdr;

	BOOL bShellSendReply = TRUE;
	BOOL bShellForwardRequest = (altGKsSize > 0);

	// On shutdown we don't accept RRQ's	
	if(bOnRasSrvCloseRejectRRQ)
	{
		PTRACE(1, "GK\tRRQ bOnRasSrvCloseRejectRRQ == TRUE");
		bReject = TRUE;
		rejectReason = H225_RegistrationRejectReason::e_undefinedReason;
		bool nated = false;

		obj_rpl.SetTag(H225_RasMessage::e_registrationReject); 
		H225_RegistrationReject & rrj = obj_rpl;

		rrj.m_requestSeqNum = obj_rr.m_requestSeqNum;
		rrj.m_protocolIdentifier =  obj_rr.m_protocolIdentifier;
		rrj.IncludeOptionalField(rrj.e_gatekeeperIdentifier);
		rrj.m_gatekeeperIdentifier.SetValue( GetGKName() );
		rrj.m_rejectReason = rejectReason;
		if (rejectReason == H225_RegistrationRejectReason::e_resourceUnavailable || nated)
			SetAltGKInfo(rrj);
//		CopyNonStandardData(obj_rr, rrj);
		SetCryptoTokens(obj_rr, rrj);

		PString aliasListString;
		if (obj_rr.HasOptionalField(H225_RegistrationRequest::e_terminalAlias))
			aliasListString = AsString(obj_rr.m_terminalAlias);
		else
			aliasListString = " ";
	
		PString msg(PString::Printf, "RRJ|%s|%s|%s|%s;\r\n", 
			    inet_ntoa(rx_addr),
			    (const unsigned char *) aliasListString,
				(const unsigned char *) AsString(obj_rr.m_terminalType),
			    (const unsigned char *) rrj.m_rejectReason.GetTagName()
			    );
		PTRACE(2, msg);
		GkStatusThread->SignalStatus(msg);
		return TRUE; // always send reject
	}

	if (IsForwardedMessage(obj_rr.HasOptionalField(H225_RegistrationRequest::e_nonStandardData) ? &obj_rr.m_nonStandardData : 0, rx_addr))
		bShellSendReply = bShellForwardRequest = FALSE;

	// lightweight registration update
	if (obj_rr.HasOptionalField(H225_RegistrationRequest::e_keepAlive) && obj_rr.m_keepAlive.GetValue()) {
		endptr ep = obj_rr.HasOptionalField(H225_RegistrationRequest::e_endpointIdentifier) ?
			EndpointTable->FindByEndpointId(obj_rr.m_endpointIdentifier) :
			(obj_rr.m_callSignalAddress.GetSize() >= 1) ?
			EndpointTable->FindBySignalAdr(obj_rr.m_callSignalAddress[0], rx_addr) : endptr(0);
		bReject = !ep;
		// check if the RRQ was sent from the registered endpoint
		if (ep && bShellSendReply) { // not forwarded RRQ
			if (ep->IsNATed())
				// for nated endpoint, only check rx_addr
			       	bReject = (ep->GetNATIP() != rx_addr);
			else {
				PIPSocket::Address oaddr, raddr;
				WORD oport, rport;
				if (obj_rr.m_callSignalAddress.GetSize() >= 1)
					GetIPAndPortFromTransportAddr(ep->GetCallSignalAddress(), oaddr, oport),
					GetIPAndPortFromTransportAddr(obj_rr.m_callSignalAddress[0], raddr, rport);
				else if (obj_rr.m_rasAddress.GetSize() >= 1)
					GetIPAndPortFromTransportAddr(ep->GetRasAddress(), oaddr, oport),
					GetIPAndPortFromTransportAddr(obj_rr.m_rasAddress[0], raddr, rport);
				else
					GetIPAndPortFromTransportAddr(ep->GetCallSignalAddress(), oaddr, oport),
					raddr = oaddr, rport = oport;
				bReject = (oaddr != raddr) || (oport != rport) || (raddr != rx_addr);
			}
		}

		if (bReject) {
			PTRACE_IF(1, ep && bShellSendReply, "GK\tWarning: Possibly endpointId collide or security attack!!");
			// endpoint was NOT registered
			rejectReason = H225_RegistrationRejectReason::e_fullRegistrationRequired;
		} else {
			// endpoint was already registered
			if (bShellSendReply)
				BuildRCF(obj_rpl, obj_rr, ep, rx_addr);
			// forward lightweights, too
			if (bShellForwardRequest) 
				ForwardRasMsg(obj_rrq);

			ep->Update(obj_rrq);
			return bShellSendReply;
		}
	}

	if (redirectGK != e_noRedirect || EndpointTable->Size() >= epLimit) {
		bReject = TRUE;
		rejectReason = H225_RegistrationRejectReason::e_resourceUnavailable;
		PTRACE(1, "GK\tWarning: Exceed registration limit!!");
	}

	bool nated = false, validaddress = false;
	if (obj_rr.m_callSignalAddress.GetSize() >= 1) {
		PIPSocket::Address ipaddr;
		for (PINDEX s = 0; s < obj_rr.m_callSignalAddress.GetSize(); ++s) {
			SignalAdr = obj_rr.m_callSignalAddress[s];
			if (GetIPFromTransportAddr(SignalAdr, ipaddr))
				if ((validaddress = (rx_addr == ipaddr || rx_addr == GKHome)))
					break;
		}
		if (!bShellSendReply) // is forwarded RRQ?
			validaddress = true;
		else if (!validaddress && (ipaddr != PIPSocket::Address(127,0,0,1))) // do not allow nated from private IP 127.0.0.1
			nated = true, validaddress = Toolkit::AsBool(GkConfig()->GetString(RoutedSection, "SupportNATedEndpoints", "0"));
	}
	if (!bReject && !validaddress) {
		bReject = TRUE;
		rejectReason = H225_RegistrationRejectReason::e_invalidCallSignalAddress;
	}
	if (!bReject && obj_rr.m_rasAddress.GetSize() == 0) {
		bReject = TRUE;
		rejectReason = H225_RegistrationRejectReason::e_invalidRASAddress;
	}

	// Check if the endpoint has specified the EndpointIdentifier.
	// The GK will accept the EndpointIdentifier if
	// the EndpointIdentifier doesn't exist in the RegistrationTable,
	// or the request is sent from the original endpoint that has
	// this EndpointIdentifier. Otherwise the request will be rejected.
	if (!bReject && obj_rr.HasOptionalField(H225_RegistrationRequest::e_endpointIdentifier)) {
		endptr ep = EndpointTable->FindByEndpointId(obj_rr.m_endpointIdentifier);
		if (ep && ep->GetCallSignalAddress() != SignalAdr) {
			bReject = TRUE;
			// no reason named invalidEndpointIdentifier? :(
			rejectReason = H225_RegistrationRejectReason::e_securityDenial;
		}
	}
	
	if (!bReject) {
		unsigned rsn = H225_RegistrationRejectReason::e_securityDenial;
		if (!authList->Check(obj_rr, rsn, rx_addr)) {
			bReject = TRUE;
			rejectReason = rsn;
		}
	}

	if (!bReject) {
		if (obj_rr.HasOptionalField(H225_RegistrationRequest::e_terminalAlias) &&
			(obj_rr.m_terminalAlias.GetSize() >= 1)) {
			const H225_ArrayOf_AliasAddress & NewAliases = obj_rr.m_terminalAlias;
			const endptr ep = EndpointTable->FindByAliases(NewAliases);
			if (ep && ep->GetCallSignalAddress() != SignalAdr) {
				if (Toolkit::AsBool(GkConfig()->GetString("RasSrv::RRQFeatures", "OverwriteEPOnSameAddress", "0"))) {
					// If the operators policy allows this case:
					// 1) unregister the active ep - sends URQ and
					// 2) remove the ep from the EndpointTable, then
					// 3) allow the new ep to register - see below
					ep->Unregister();
					EndpointTable->RemoveByEndptr(ep);
				} else {
					bReject = TRUE;
					rejectReason = H225_RegistrationRejectReason::e_duplicateAlias;
				}	
			}

			// reject the empty string
			for (PINDEX AliasIndex=0; AliasIndex < NewAliases.GetSize(); ++AliasIndex) {
				const PString & s = AsString(NewAliases[AliasIndex], FALSE);
			//	if (s.GetLength() < 1 || !(isalnum(s[0]) || s[0]=='#') ) {
				if (s.GetLength() < 1) {
					bReject = TRUE;
					rejectReason = H225_RegistrationRejectReason::e_invalidAlias;
				}
				if (!nated)
					nated = GkConfig()->HasKey("NATedEndpoints", s);
			}
		} else {
			// reject gw without alias
			switch (obj_rr.m_terminalType.GetTag()) {
			case H225_EndpointType::e_gatekeeper:
			case H225_EndpointType::e_gateway:
			case H225_EndpointType::e_mcu:
				bReject = TRUE;
				rejectReason = H225_RegistrationRejectReason::e_invalidAlias;
				break;
			/* only while debugging
			default:  
				bReject = TRUE;
				rejectReason.SetTag(H225_RegistrationRejectReason::e_invalidAlias);
				break;
			*/
			}
		}
	}

	if (!bReject) {
		// make a copy for modifying
		H225_RasMessage store_rrq = obj_rrq;
		H225_RegistrationRequest & rrq = store_rrq;
		rrq.m_callSignalAddress.SetSize(1);
		rrq.m_callSignalAddress[0] = SignalAdr;

		endptr ep = EndpointTable->InsertRec(store_rrq, nated ? rx_addr : PIPSocket::Address(INADDR_ANY));
		if ( ep ) {
			if (nated)
				ep->SetNATAddress(rx_addr);
			else
				ep->SetNAT(false);
			if (bShellSendReply) {
				//	
				// OK, now send RCF
				//	
				BuildRCF(obj_rpl, obj_rr, ep, rx_addr);
				H225_RegistrationConfirm & rcf = obj_rpl;
				rcf.IncludeOptionalField(H225_RegistrationConfirm::e_terminalAlias);
				rcf.m_terminalAlias = ep->GetAliases();
//				CopyNonStandardData(obj_rr, rcf);
				if (nated) {
					// tell the endpoint its translated address
					rcf.IncludeOptionalField(H225_RegistrationConfirm::e_nonStandardData);
					rcf.m_nonStandardData.m_data = "NAT=" + rx_addr.AsString();
				}

				// Alternate GKs
				if (obj_rr.HasOptionalField(H225_RegistrationRequest::e_supportsAltGK))
					SetAlternateGK(rcf);
			} else {
				PIPSocket::Address rasip, sigip;
				if (GetIPFromTransportAddr(obj_rr.m_rasAddress[0], rasip) && GetIPFromTransportAddr(SignalAdr, sigip) && rasip != sigip)
					// this is an nated endpoint
					ep->SetNATAddress(rasip);
			}
			// forward heavyweight
			if (bShellForwardRequest) {
				rrq.IncludeOptionalField(H225_RegistrationRequest::e_endpointIdentifier);
				rrq.m_endpointIdentifier = ep->GetEndpointIdentifier();
				if (nated) {
					rrq.m_rasAddress.SetSize(1);
					rrq.m_rasAddress[0] = ep->GetRasAddress(); // translated address
				}
				ForwardRasMsg(store_rrq);
			}

			// Note that the terminalAlias is not optional here as we pass the auto generated alias if not were provided from
			// the endpoint itself
			PString msg(PString::Printf, "RCF|%s|%s|%s|%s;\r\n", 
				    (const unsigned char *) AsDotString(ep->GetCallSignalAddress()),
				    (const unsigned char *) AsString(ep->GetAliases()),
				    (const unsigned char *) AsString(obj_rr.m_terminalType),
				    (const unsigned char *) ep->GetEndpointIdentifier().GetValue()
				    );
			PTRACE(2, msg);
			GkStatusThread->SignalStatus(msg);

			return bShellSendReply;
		} else { // Oops! Should not happen...
			bReject = TRUE;
			rejectReason = H225_RegistrationRejectReason::e_undefinedReason;
			PTRACE(3, "Gk\tRRQ rejected by unknown reason " << alias);
		}
	}
	//
	// final rejection handling
	//
	obj_rpl.SetTag(H225_RasMessage::e_registrationReject); 
	H225_RegistrationReject & rrj = obj_rpl;

	rrj.m_requestSeqNum = obj_rr.m_requestSeqNum;
	rrj.m_protocolIdentifier =  obj_rr.m_protocolIdentifier;
	rrj.IncludeOptionalField(rrj.e_gatekeeperIdentifier);
	rrj.m_gatekeeperIdentifier.SetValue( GetGKName() );
	rrj.m_rejectReason = rejectReason;
	if (rejectReason == H225_RegistrationRejectReason::e_resourceUnavailable ||  (rejectReason == H225_RegistrationRejectReason::e_invalidCallSignalAddress && nated))
		SetAltGKInfo(rrj);
//	CopyNonStandardData(obj_rr, rrj);
	SetCryptoTokens(obj_rr, rrj);

	PString aliasListString;
	if (obj_rr.HasOptionalField(H225_RegistrationRequest::e_terminalAlias))
		aliasListString = AsString(obj_rr.m_terminalAlias);
	else
		aliasListString = " ";
	
	PString msg(PString::Printf, "RRJ|%s|%s|%s|%s;\r\n", 
		    inet_ntoa(rx_addr),
		    (const unsigned char *) aliasListString,
		    (const unsigned char *) AsString(obj_rr.m_terminalType),
		    (const unsigned char *) rrj.m_rejectReason.GetTagName()
		    );
	PTRACE(2, msg);
	GkStatusThread->SignalStatus(msg);
	return TRUE; // always send reject
}


/* Registration Confirm */
BOOL H323RasSrv::OnRCF(const PIPSocket::Address & rx_addr, H225_RasMessage & obj_rcf, H225_RasMessage &)
{
	PTRACE(1, "GK\tRCF Received");

	gkClient->OnRCF(obj_rcf, rx_addr);
	return FALSE;
}


/* Registration Reject */
BOOL H323RasSrv::OnRRJ(const PIPSocket::Address & rx_addr, H225_RasMessage & obj_rrj, H225_RasMessage & obj_rpl)
{
	PTRACE(1, "GK\tRRJ Received");

	const H225_RegistrationReject & rrj = obj_rrj;
	if (rrj.HasOptionalField(H225_RegistrationReject::e_nonStandardData)) {
		if (rrj.m_nonStandardData.m_nonStandardIdentifier.GetTag() == H225_NonStandardIdentifier::e_h221NonStandard) {
			const H225_H221NonStandard & nonStandard = rrj.m_nonStandardData.m_nonStandardIdentifier;
			if (Toolkit::Instance()->GetInternalExtensionCode(nonStandard) == Toolkit::iecFailoverRAS) {
				// RRJ from alternateGKs
				H225_EndpointIdentifier id;
				id = rrj.m_nonStandardData.m_data.AsString();
				if (endptr ep = EndpointTable->FindByEndpointId(id)) {
					obj_rpl = ep->GetCompleteRegistrationRequest();
					if (obj_rpl.GetTag() == H225_RasMessage::e_registrationRequest) {
						CopyNonStandardData(rrj, (H225_RegistrationRequest &)obj_rpl);
						return TRUE;
					}
				}
			}
		}
	}
	gkClient->OnRRJ(obj_rrj, rx_addr);
	return FALSE;
}


BOOL H323RasSrv::CheckForIncompleteAddress(const H225_ArrayOf_AliasAddress & alias) const
{
	// since this routine is only called when the routing decision has been made,
	// finding a prefix that is longer than our dialled number implies the number is incomplete

	BOOL DoCheck = Toolkit::AsBool(GkConfig()->GetString
		("RasSrv::ARQFeatures", "IncompleteAddresses", "TRUE"));

	if (!DoCheck)
		return FALSE;


	// find gateway with longest prefix matching our dialled number
	endptr GW = EndpointTable->FindByAliases(alias);

	if (!GW)
		return FALSE;
// TODO: how to port?
/*
	const PString & aliasStr = AsString (alias, FALSE);
	const PString GWAliasStr = H323GetAliasAddressString(GW->GetAliases()[0]);
	const PStringArray *prefixes = EndpointTable->GetGatewayPrefixes(GWAliasStr);


	PINDEX max = prefixes->GetSize();
	for (PINDEX i = 0; i < max; i++) {
		const PString &prefix = (*prefixes)[i];

		// is this prefix matching the dialled number and is it longer ?
		if (aliasStr.Find(prefix) == 0 && prefix.GetLength() > aliasStr.GetLength()) {
			PTRACE(4,"Gk\tConsidered to be an incomplete address: " << aliasStr << "\n" <<
				GWAliasStr << " has a longer matching prefix " << prefix);
			return TRUE;
		}
	}
*/
	return FALSE;
}

bool H323RasSrv::SendLRQ(const H225_AdmissionRequest & arq, const endptr & reqEP)
{
	return arqPendingList->Insert(arq, reqEP);
}


BOOL H323RasSrv::DoARQ(
	const PIPSocket::Address& rx_addr, 
	H225_RasMessage& obj_arq
	)
{
	H225_RasMessage obj_rpl;
	const H225_AdmissionRequest & obj_rr = obj_arq; 
	
	// find the caller
	const endptr RequestingEP = EndpointTable->FindByEndpointId(obj_rr.m_endpointIdentifier);
	if (!RequestingEP) {
		// TODO: signal error on status thread
		return false;
	};
	 
	BOOL ShallSendReply = OnARQ(rx_addr, obj_arq, obj_rpl);

	if (ShallSendReply) { 
		const H225_TransportAddress_ipAddress & ip = RequestingEP->GetRasAddress();
		PIPSocket::Address ipaddress(ip.m_ip.GetSize(), (const BYTE*)ip.m_ip);
		PTRACE(1, "GK\tDoARQ  sendReply " << ipaddress << " " << ip.m_port );
		SendReply(obj_rpl, ipaddress, (WORD)ip.m_port, listener);
	}
	return true;
}

 
/* Admission Request */
BOOL H323RasSrv::OnARQ(const PIPSocket::Address & rx_addr, H225_RasMessage & obj_arq, H225_RasMessage & obj_rpl)
{    
	PTRACE(1, "GK\tARQ Received");

	H225_AdmissionRequest & obj_rr = obj_arq;
 
	BOOL bReject = FALSE;
	long callDurationLimit = -1;
	unsigned rejectReason = 0;
	unsigned rsn = H225_AdmissionRejectReason::e_securityDenial;

	// find the caller
	const endptr RequestingEP = EndpointTable->FindByEndpointId(obj_rr.m_endpointIdentifier);
	
	endptr CalledEP(0);
	PString calledPartyNumber;
	 
	if (CallTbl->Size() >= callLimit && !obj_rr.m_answerCall) {
		bReject = TRUE;
		rejectReason = H225_AdmissionRejectReason::e_resourceUnavailable;
		PTRACE(1, "GK\tWarning: Exceed call limit!!");
	} else if (RequestingEP) { // Is the ARQ from a registered endpoint?
		bool bHasDestInfo = (obj_rr.HasOptionalField(H225_AdmissionRequest::e_destinationInfo) && obj_rr.m_destinationInfo.GetSize() >= 1);
		if (bHasDestInfo) // apply rewriting rules
			Toolkit::Instance()->RewriteE164(obj_rr.m_destinationInfo[0]);

		if (!authList->Check(obj_rr,rsn,callDurationLimit)) {
			bReject = TRUE;
			rejectReason = rsn;
		} else if (obj_rr.m_answerCall) {
			// don't search endpoint table for an answerCall ARQ
			CalledEP = RequestingEP;
		} else {
#if HAS_WAITARQ
			if (bHasDestInfo) {
				// check if call is for a virtual queue
				PString alias = H323GetAliasAddressString (obj_rr.m_destinationInfo[0]);
				if (wArqList->IsDestinationVirtualQueue(alias)) {
					PString src = (RequestingEP) ? AsDotString(RequestingEP->GetCallSignalAddress()) : PString(" ");
					PString msg(PString::Printf, "RouteRequest|%s|%s|%u|%s|%s;\r\n", 
							(const unsigned char *) src,
							(const unsigned char *) RequestingEP->GetEndpointIdentifier().GetValue(),
							(unsigned) obj_rr.m_callReferenceValue,
							(const unsigned char *) alias,
							(const unsigned char *) AsString(obj_rr.m_srcInfo)
						   );
					PTRACE(2,msg);
					if ( wArqList->InsertWaitingARQ (obj_arq , rx_addr , alias , RequestingEP ) ) {
						GkStatusThread->SignalStatus(msg);
					}
					return false;	// don't send a reply, wait for external application
				}
			}
#endif

			// if a destination address is provided, we check if we know it
			if (obj_rr.HasOptionalField(H225_AdmissionRequest::e_destCallSignalAddress) &&
			    GetCallSignalAddress(rx_addr) != obj_rr.m_destCallSignalAddress) { // if destAddr is the GK, ignore it
				PIPSocket::Address destAddr;
				WORD destPort = 0;
				if (GetIPAndPortFromTransportAddr(obj_rr.m_destCallSignalAddress, destAddr, destPort)
					&& destAddr.IsValid() && destPort != 0) {
					CalledEP = EndpointTable->FindBySignalAdr(obj_rr.m_destCallSignalAddress);
					if (!CalledEP && Toolkit::AsBool(GkConfig()->GetString("RasSrv::ARQFeatures", "CallUnregisteredEndpoints", "1"))) { // oh, unknown address, accept it?
						H225_RasMessage arq = obj_arq;
						CalledEP = EndpointTable->InsertRec(arq);
					}
				} else
					PTRACE(3, "RAS\tInvalid destination address found in the ARQ, ignored: "
						<< obj_rr.m_destCallSignalAddress
						);
			}
			if (!CalledEP && bHasDestInfo) {
#if WITH_DEST_ANALYSIS_LIST
				CalledEP = EndpointTable->getMsgDestination(obj_rr, rsn);
				if (!CalledEP && 
						rsn == H225_AdmissionRejectReason::e_incompleteAddress) {
					bReject = TRUE;
					rejectReason = rsn;
				}
#else
				int matchedalias = -1;
				CalledEP = EndpointTable->FindEndpoint(obj_rr.m_destinationInfo, true, true, &matchedalias);
				if( CalledEP && matchedalias >= 0 && matchedalias < obj_rr.m_destinationInfo.GetSize() )
					calledPartyNumber = H323GetAliasAddressString(obj_rr.m_destinationInfo[matchedalias]);
#endif
				if (!bReject && !CalledEP && Toolkit::AsBool(
					GkConfig()->GetString("RasSrv::ARQFeatures", "ParseEmailAliases", "0"))) {
					H225_TransportAddress destAddr;
					PString destAlias;
					PINDEX aliasIndex;
					if (GetEmailAlias(obj_rr.m_destinationInfo, destAddr, 
							&destAlias, &aliasIndex)) {
						CalledEP = EndpointTable->FindBySignalAdr(destAddr);
						if (CalledEP) {
							// remove the domain name part
							H323SetAliasAddress(destAlias, 
								obj_rr.m_destinationInfo[aliasIndex]
								);
						} else if (Toolkit::AsBool(
							GkConfig()->GetString("RasSrv::ARQFeatures",
								"CallUnregisteredEndpoints", "1"))) { 
							// call to an unregistered endpoint
							H225_RasMessage arq = obj_arq;
							H225_AdmissionRequest & arq_rr = arq;
							arq_rr.IncludeOptionalField(H225_AdmissionRequest::e_destCallSignalAddress);
							arq_rr.m_destCallSignalAddress = destAddr;
							CalledEP = EndpointTable->InsertRec(arq);
							// remove the domain name part
							H323SetAliasAddress(destAlias, 
								obj_rr.m_destinationInfo[aliasIndex]
								);
						}
					}
				}
				
				if (!bReject && ((!CalledEP) || CalledEP->IsFromParent())) {
					bool pending = false;
					if (gkClient->IsRegistered()) {
						if( !CalledEP ) {
							H225_ArrayOf_AliasAddress dest = obj_rr.m_destinationInfo;
							if (gkClient->RewriteE164(dest, false)) {
								matchedalias = -1;
								CalledEP = EndpointTable->FindEndpoint(dest, true, true, &matchedalias);
								if( CalledEP && matchedalias >= 0 && matchedalias < obj_rr.m_destinationInfo.GetSize() )
									calledPartyNumber = H323GetAliasAddressString(obj_rr.m_destinationInfo[matchedalias]);
							}
						}
						if ((!CalledEP) || CalledEP->IsFromParent()) {
							gkClient->SendARQ(obj_rr, RequestingEP,callDurationLimit);
							pending = true;
						}
					} else if ( (!CalledEP) && arqPendingList->Insert(obj_rr, RequestingEP,callDurationLimit))
						pending = true;
					if (pending) {
						if (RequestingEP->IsNATed())
							RequestingEP->SetRasAddress(SocketToH225TransportAddr(rx_addr, m_rx_port));
						return FALSE;
					}
				}
			}
		}
	}

	// remove dest call signaling address if it is the gk signaling address	
	if (obj_rr.HasOptionalField(H225_AdmissionRequest::e_destCallSignalAddress) &&
	    GetCallSignalAddress(rx_addr) == obj_rr.m_destCallSignalAddress)
		obj_rr.RemoveOptionalField(H225_AdmissionRequest::e_destCallSignalAddress);
		
	if (bReject) {
		obj_rpl.SetTag(H225_RasMessage::e_admissionReject); 
		H225_AdmissionReject & arj = obj_rpl; 
		arj.m_rejectReason.SetTag(rejectReason);
		if (rejectReason == H225_AdmissionRejectReason::e_resourceUnavailable)
			SetAltGKInfo(arj);
	}
	callptr pExistingCallRec = ProcessARQ(rx_addr, RequestingEP, CalledEP, obj_rr, obj_rpl, bReject);
	if( pExistingCallRec ) {
#ifdef HAS_ACCT
		if( pExistingCallRec->GetCalledStationId().IsEmpty() 
			&& !calledPartyNumber.IsEmpty() )
			pExistingCallRec->SetCalledStationId(calledPartyNumber);
#endif
		if( callDurationLimit > 0 )
			pExistingCallRec->SetDurationLimit(callDurationLimit);
	}
	return TRUE;
}

callptr H323RasSrv::ProcessARQ(const PIPSocket::Address & rx_addr, const endptr & RequestingEP, const endptr & CalledEP, const H225_AdmissionRequest & obj_arq, H225_RasMessage & obj_rpl, BOOL bReject)
{
	unsigned rejectReason = 0; // aviod warning
	if (obj_rpl.GetTag() == H225_RasMessage::e_admissionReject) {
		H225_AdmissionReject & arj = obj_rpl;
		rejectReason = arj.m_rejectReason.GetTag();
	}

	// check if the endpoint requesting is registered with this gatekeeper
	if (!bReject && !RequestingEP) {
		bReject = TRUE;
		rejectReason = H225_AdmissionRejectReason::e_callerNotRegistered/*was :e_invalidEndpointIdentifier*/;
	}

/*
	// allow overlap sending for incomplete prefixes
	if (!CalledEP && obj_arq.m_destinationInfo.GetSize() >= 1) {
		const PString alias = AsString(obj_arq.m_destinationInfo[0], FALSE);
		if (CheckForIncompleteAddress(obj_arq.m_destinationInfo)) {
			bReject = TRUE;
			rejectReason = H225_AdmissionRejectReason::e_incompleteAddress;
		}
	}
*/

	if (!bReject && !CalledEP) {
		bReject = TRUE;
		rejectReason = H225_AdmissionRejectReason::e_calledPartyNotRegistered;
	}

#ifdef ARJREASON_ROUTECALLTOSCN
 	//
 	// call from one GW to itself? 
 	// generate ARJ-reason: 'routeCallToSCN'
 	//
 	if(!bReject && 
 	   Toolkit::AsBool(GkConfig()->GetString("RasSrv::ARQFeatures", "ArjReasonRouteCallToSCN", "1")))
 	{
 		// are the endpoints the same (GWs of course)?
 		if( (CalledEP) && (RequestingEP) && (CalledEP == RequestingEP) && 
 			(!obj_arq.m_answerCall) // only first ARQ may be rejected with with 'routeCallToSCN'
 			) 
 		{
 			// we have to extract the SCN from the destination. only EP-1 will be rejected this way
 			if ( obj_arq.m_destinationInfo.GetSize() >= 1 ) {
 				// PN will be the number that is set in the arj reason
 				H225_PartyNumber PN;
 				PN.SetTag(H225_PartyNumber::e_publicNumber);
 				H225_PublicPartyNumber &PPN = PN;
 				// set defaults
 				PPN.m_publicTypeOfNumber.SetTag(H225_PublicTypeOfNumber::e_unknown);
 				PPN.m_publicNumberDigits = "";
 				
 				// there can be diffent information in the destination info
 				switch(obj_arq.m_destinationInfo[0].GetTag()) {
 				case H225_AliasAddress::e_dialedDigits: 
 					// normal number, extract only the digits
 					PPN.m_publicNumberDigits = AsString(obj_arq.m_destinationInfo[0], FALSE);
 					break;
 				case H225_AliasAddress::e_partyNumber: 
 					// ready-to-use party number
 					PN = obj_arq.m_destinationInfo[0];
 					break;
 				default:
 					PTRACE(1,"Unsupported AliasAdress for ARQ reason 'routeCallToSCN': "
 						   << obj_arq.m_destinationInfo[0]);
 				}
 				
 				// set the ARJ reason
 				bReject = TRUE;
 				rejectReason = H225_AdmissionRejectReason::e_routeCallToSCN;
 				H225_ArrayOf_PartyNumber &APN = arj.m_rejectReason;
 				APN.SetSize(1);
 				APN[0] = PN;
 			}
 			else { 
 				// missing destination info. is this possible at this point?
 			}
 		}
 	}
 	//:towi
#endif

	//
	// Do the reject or the confirm
	//
	PString srcInfoString = (RequestingEP) ? AsDotString(RequestingEP->GetCallSignalAddress()) : PString(" ");
	PString destinationInfoString = (obj_arq.HasOptionalField(H225_AdmissionRequest::e_destinationInfo)) ?
		AsString(obj_arq.m_destinationInfo) : PString("unknown");

	// CallRecs should be looked for using callIdentifier instead of callReferenceValue
	// callIdentifier is globally unique, callReferenceValue is just unique per-endpoint.
	callptr pExistingCallRec = (obj_arq.HasOptionalField(H225_AdmissionRequest::e_callIdentifier)) ?
		CallTbl->FindCallRec(obj_arq.m_callIdentifier) :
	// since callIdentifier is optional, we might have to look for the callReferenceValue as well
		CallTbl->FindCallRec(obj_arq.m_callReferenceValue);

	if (!bReject && GKRoutedSignaling && obj_arq.m_answerCall && !pExistingCallRec) {
		if (Toolkit::AsBool(GkConfig()->GetString("RasSrv::ARQFeatures", "ArjReasonRouteCallToGatekeeper", "1"))) {
			bool rej = true;
			if (obj_arq.HasOptionalField(H225_AdmissionRequest::e_srcCallSignalAddress)) {
				PIPSocket::Address ipaddress;
				if (GetIPFromTransportAddr(obj_arq.m_srcCallSignalAddress, ipaddress))
					if ((rej = !IsForwardedMessage(0, ipaddress)))
						rej = !(AcceptNBCalls && NeighborsGK->CheckIP(ipaddress));
			}
			if (rej) {
				bReject = TRUE;
				rejectReason = H225_AdmissionRejectReason::e_routeCallToGatekeeper;
			}
		}
	}

	//
	// Bandwidth 
	// and GkManager admission
	//
	int BWRequest = 1280;
	if (!bReject) {
		// hack for Netmeeting 3.0x
		if (obj_arq.m_bandWidth.GetValue() >= 100)
			BWRequest = obj_arq.m_bandWidth.GetValue();
		// check if it is the first arrived ARQ
		if (pExistingCallRec) {
			// request more bandwidth?
			if (BWRequest > pExistingCallRec->GetBandWidth())
				if (CallTbl->GetAdmission(BWRequest, pExistingCallRec))
					pExistingCallRec->SetBandwidth(BWRequest);
				else
					bReject = TRUE;
		} else {
			bReject = (!CallTbl->GetAdmission(BWRequest));
		}
		if (bReject)
			rejectReason = H225_AdmissionRejectReason::e_requestDenied; // what the spec says
		PTRACE(3, "GK\tARQ will request bandwith of " << BWRequest);
	}

	const char *answerCall = obj_arq.m_answerCall ? "true" : "false";
	if (bReject) {
		if (obj_rpl.GetTag() != H225_RasMessage::e_admissionReject)
			obj_rpl.SetTag(H225_RasMessage::e_admissionReject);
		H225_AdmissionReject & arj = obj_rpl;
		arj.m_rejectReason.SetTag(rejectReason);
		arj.m_requestSeqNum = obj_arq.m_requestSeqNum;
		SetCryptoTokens(obj_arq, arj);
		PString msg(PString::Printf, "ARJ|%s|%s|%s|%s|%s;\r\n", 
			    (const unsigned char *) srcInfoString,
			    (const unsigned char *) destinationInfoString,
			    (const unsigned char *) AsString(obj_arq.m_srcInfo),
			    answerCall,
			    (const unsigned char *) arj.m_rejectReason.GetTagName() );
		PTRACE(2, msg);
		GkStatusThread->SignalStatus(msg);
 
		if (CalledEP && CalledEP->IsFromParent()) {
			H225_RasMessage drq_ras;
			drq_ras.SetTag(H225_RasMessage::e_disengageRequest);
			H225_DisengageRequest & drq = drq_ras;
			drq.m_conferenceID = obj_arq.m_conferenceID;
			drq.m_callReferenceValue = obj_arq.m_callReferenceValue;
			drq.m_disengageReason.SetTag(H225_DisengageReason::e_normalDrop);
			if (obj_arq.HasOptionalField(H225_AdmissionRequest::e_callIdentifier))
				drq.m_callIdentifier = obj_arq.m_callIdentifier;
			gkClient->SendDRQ(drq_ras);
		}
	} else {
		// new connection admitted
		obj_rpl.SetTag(H225_RasMessage::e_admissionConfirm); // re-cast (see above)
		H225_AdmissionConfirm & acf = obj_rpl;

		acf.m_requestSeqNum = obj_arq.m_requestSeqNum;
		acf.m_bandWidth = BWRequest;

		if (pExistingCallRec) {
			if (obj_arq.m_answerCall) // the second ARQ
				pExistingCallRec->SetCalled(CalledEP, (WORD)obj_arq.m_callReferenceValue.GetValue());
			// else this may be a duplicate ARQ, ignore!
			PTRACE(3, "GK\tACF: found existing call no " << pExistingCallRec->GetCallNumber());
		} else {
			// the call is not in the table		
			CallRec *pCallRec = new CallRec(
				obj_arq, RequestingEP, CalledEP, BWRequest, GKRoutedH245,
				GKRoutedSignaling
				);
			pExistingCallRec = callptr(pCallRec);
			CallTbl->Insert(pCallRec);
		}
			
		if ( GKRoutedSignaling ) {
			acf.m_callModel.SetTag( H225_CallModel::e_gatekeeperRouted );
			acf.m_destCallSignalAddress = GetCallSignalAddress(rx_addr);
		} else {
			// direct signalling

			// Set ACF fields
			acf.m_callModel.SetTag( H225_CallModel::e_direct );
			if(obj_arq.HasOptionalField( H225_AdmissionRequest::e_destCallSignalAddress))
				acf.m_destCallSignalAddress = obj_arq.m_destCallSignalAddress;
			else
				acf.m_destCallSignalAddress = CalledEP->GetCallSignalAddress();
		}

		acf.IncludeOptionalField ( H225_AdmissionConfirm::e_irrFrequency );
		acf.m_irrFrequency.SetValue( 120 );

		SetCryptoTokens(obj_arq, acf);

		PString destinationInfoString = "";
		if (obj_arq.HasOptionalField(H225_AdmissionRequest::e_destinationInfo))
			destinationInfoString = AsString(obj_arq.m_destinationInfo);
		else if (obj_arq.HasOptionalField(H225_AdmissionRequest::e_destCallSignalAddress))
			if (const endptr pEPRec = EndpointTable->FindBySignalAdr(obj_arq.m_destCallSignalAddress))
				destinationInfoString = AsString(pEPRec->GetAliases());

#if HAS_WAITARQ
		if( (wArqList->IsWaitingARQ(obj_arq) || !GKRoutedSignaling) && !destinationInfoString.IsEmpty() ) {
#else
		if( !(GKRoutedSignaling || destinationInfoString.IsEmpty()) ) {
#endif
			// add destinationInfo field with the possibly changed destination
			bool canMapAlias = false;
			if (obj_arq.HasOptionalField(H225_AdmissionRequest::e_canMapAlias))
			if (canMapAlias) {
				const PStringArray alias = destinationInfoString.Tokenise(":");
				if( alias.GetSize() > 0 && !alias[0].IsEmpty() ) {
					// only add destinationInfo for endpoint who can handle it
					acf.IncludeOptionalField ( H225_AdmissionConfirm::e_destinationInfo );
					acf.m_destinationInfo.SetSize(1);
					H323SetAliasAddress(alias[0],acf.m_destinationInfo[0]);
#ifdef HAS_ACCT
					if( pExistingCallRec && pExistingCallRec->GetCalledStationId().IsEmpty() )
						pExistingCallRec->SetCalledStationId(alias[0]);
#endif
				}
			}
		}
		
		if (destinationInfoString.IsEmpty())
			destinationInfoString = "unknown";
		// always signal ACF
		PString msg(PString::Printf, "ACF|%s|%s|%u|%s|%s|%s;\r\n", 
				(const unsigned char *) srcInfoString,
				(const unsigned char *) RequestingEP->GetEndpointIdentifier().GetValue(),
				(unsigned) obj_arq.m_callReferenceValue,
				(const unsigned char *) destinationInfoString,
				(const unsigned char *) AsString(obj_arq.m_srcInfo),
				answerCall
				);
		PTRACE(2, msg);
		GkStatusThread->SignalStatus(msg);
	}
	
	return pExistingCallRec;
}

callptr H323RasSrv::ReplyARQ(const endptr & RequestingEP, const endptr & CalledEP, const H225_AdmissionRequest & obj_arq)
{
	callptr call(NULL);
	H225_RasMessage obj_rpl;
	if (!RequestingEP) {
		PTRACE(1, "Err: call ReplyARQ without RequestingEP!");
		return call;
	}
	PIPSocket::Address ipaddress;
	WORD port;
	if (!GetIPAndPortFromTransportAddr(RequestingEP->GetRasAddress(), ipaddress, port)) {
		PTRACE(1, "Err: RequestingEP doesn't have valid ras address!");
		return call;
	}
	call = ProcessARQ(ipaddress, RequestingEP, CalledEP, obj_arq, obj_rpl);
	SendReply(obj_rpl, ipaddress, port, listener);
	return call;
}

/* Admission Confirm */
BOOL H323RasSrv::OnACF(const PIPSocket::Address & rx_addr, H225_RasMessage &obj_rr, H225_RasMessage &)
{
	PTRACE(1, "GK\tACF Received");

	if (gkClient->IsRegistered())
		gkClient->OnACF(obj_rr, rx_addr);
	return FALSE;
}

/* Admission Reject */
BOOL H323RasSrv::OnARJ(const PIPSocket::Address & rx_addr, H225_RasMessage &obj_rr, H225_RasMessage &)
{
	PTRACE(1, "GK\tARJ Received");

	if (gkClient->IsRegistered())
		gkClient->OnARJ(obj_rr, rx_addr);
	return FALSE;
}

/* Disengage Request */
BOOL H323RasSrv::OnDRQ(const PIPSocket::Address & rx_addr, H225_RasMessage & obj_drq, H225_RasMessage & obj_rpl)
{
	PTRACE(1, "GK\tDRQ Received");

	const H225_DisengageRequest & obj_rr = obj_drq;
	bool bReject = false;

	endptr ep;
	unsigned rsn = H225_DisengageRejectReason::e_securityDenial;
	if (gkClient->IsRegistered() && gkClient->OnDRQ(obj_rr, rx_addr)) {
		PTRACE(4,"GKC\tDRQ: from my GK");
	} else if ((ep = EndpointTable->FindByEndpointId(obj_rr.m_endpointIdentifier))) {
		PTRACE(4, "GK\tDRQ: closed conference");
		bReject = !authList->Check(obj_rr, rsn);
	} else {
		bReject = true;
		rsn = H225_DisengageRejectReason::e_notRegistered;
	}

	PString msg;
	if (bReject) {
		obj_rpl.SetTag(H225_RasMessage::e_disengageReject); 
		H225_DisengageReject & drj = obj_rpl;
		drj.m_requestSeqNum = obj_rr.m_requestSeqNum;
		drj.m_rejectReason.SetTag(rsn);
		SetCryptoTokens(obj_rr, drj);

		msg = PString(PString::Printf, "DRJ|%s|%s|%u|%s;\r\n", 
				inet_ntoa(rx_addr),
				(const unsigned char *) obj_rr.m_endpointIdentifier.GetValue(),
				(unsigned) obj_rr.m_callReferenceValue,
				(const unsigned char *) drj.m_rejectReason.GetTagName());
	} else {
		obj_rpl.SetTag(H225_RasMessage::e_disengageConfirm); 
		H225_DisengageConfirm & dcf = obj_rpl;
		dcf.m_requestSeqNum = obj_rr.m_requestSeqNum;    
		SetCryptoTokens(obj_rr, dcf);

		// always signal DCF
		msg = PString(PString::Printf, "DCF|%s|%s|%u|%s;\r\n", 
				inet_ntoa(rx_addr),
				(const unsigned char *) obj_rr.m_endpointIdentifier.GetValue(),
				(unsigned) obj_rr.m_callReferenceValue,
				(const unsigned char *) obj_rr.m_disengageReason.GetTagName());

		if (!GKRoutedSignaling || bRemoveCallOnDRQ)
			CallTbl->RemoveCall(obj_rr, ep);
	}

	PTRACE(2, msg);
	GkStatusThread->SignalStatus(msg);
	return TRUE;
}


/* Unregistration Request */
BOOL H323RasSrv::OnURQ(const PIPSocket::Address & rx_addr, H225_RasMessage &obj_urq, H225_RasMessage &obj_rpl)
{ 
	PTRACE(1, "GK\tURQ Received");

	const H225_UnregistrationRequest & obj_rr = obj_urq;
	PString msg;

	BOOL bShellSendReply = TRUE;
	BOOL bShellForwardRequest = TRUE;

	// check first if it comes from my GK
	if (gkClient->IsRegistered() && gkClient->OnURQ(obj_rr, rx_addr)) {
		// Return UCF
		obj_rpl.SetTag(H225_RasMessage::e_unregistrationConfirm);
		H225_UnregistrationConfirm & ucf = obj_rpl;
		ucf.m_requestSeqNum = obj_rr.m_requestSeqNum;
		return TRUE;
	}
	// OK, it comes from my endpoints
	if (IsForwardedMessage(obj_rr.HasOptionalField(H225_UnregistrationRequest::e_nonStandardData) ? &obj_rr.m_nonStandardData : 0, rx_addr))
		bShellSendReply = bShellForwardRequest = FALSE;

	PString endpointIdentifierString(obj_rr.HasOptionalField(H225_UnregistrationRequest::e_endpointIdentifier) ? obj_rr.m_endpointIdentifier.GetValue() : PString(" "));
	endptr ep = obj_rr.HasOptionalField(H225_UnregistrationRequest::e_endpointIdentifier) ?
		EndpointTable->FindByEndpointId(obj_rr.m_endpointIdentifier) :
		obj_rr.m_callSignalAddress.GetSize() ?  EndpointTable->FindBySignalAdr(obj_rr.m_callSignalAddress[0], rx_addr) : endptr(0);
	if (ep) {
		// Disconnect all calls of the endpoint
		SoftPBX::DisconnectEndpoint(ep);
		// Remove from the table
//		EndpointTable->RemoveByEndpointId(obj_rr.m_endpointIdentifier);
		EndpointTable->RemoveByEndptr(ep);

		// Return UCF
		obj_rpl.SetTag(H225_RasMessage::e_unregistrationConfirm);
		H225_UnregistrationConfirm & ucf = obj_rpl;
		ucf.m_requestSeqNum = obj_rr.m_requestSeqNum;
//		CopyNonStandardData(obj_rr, ucf);
		SetCryptoTokens(obj_rr, ucf);

		msg = PString(PString::Printf, "UCF|%s|%s;", 
			inet_ntoa(rx_addr),
			(const unsigned char *) endpointIdentifierString) ;
	} else {
		// Return URJ	
		obj_rpl.SetTag(H225_RasMessage::e_unregistrationReject);
		H225_UnregistrationReject & urj = obj_rpl;
		urj.m_requestSeqNum = obj_rr.m_requestSeqNum;
		urj.m_rejectReason.SetTag(H225_UnregRejectReason::e_notCurrentlyRegistered);
//		CopyNonStandardData(obj_rr, urj);

		msg = PString(PString::Printf, "URJ|%s|%s|%s;", 
			inet_ntoa(rx_addr),
			(const unsigned char *) endpointIdentifierString,
			(const unsigned char *) urj.m_rejectReason.GetTagName() );
	}

	PTRACE(2, msg);
	GkStatusThread->SignalStatus(msg + "\r\n");

	if (bShellForwardRequest) 
		ForwardRasMsg(obj_urq);

	return bShellSendReply;
}


/* Bandwidth Request */
BOOL H323RasSrv::OnBRQ(const PIPSocket::Address & rx_addr, H225_RasMessage & obj_rr, H225_RasMessage & obj_rpl)
{ 
	PTRACE(1, "GK\tBRQ Received");

	const H225_BandwidthRequest & obj_brq = obj_rr;
	obj_rpl.SetTag(H225_RasMessage::e_bandwidthReject);
	H225_BandwidthReject & brj = obj_rpl;

	int bandwidth;
	if (obj_brq.m_bandWidth.GetValue() < 100) {
		/* hack for Netmeeting 3.0 */
		bandwidth = 1280;
	} else {
		bandwidth = obj_brq.m_bandWidth;
	}

	callptr pCall = obj_brq.HasOptionalField(H225_BandwidthRequest::e_callIdentifier) ?
		CallTbl->FindCallRec(obj_brq.m_callIdentifier) :
		CallTbl->FindCallRec(obj_brq.m_callReferenceValue);

	PString msg;
	unsigned rsn = H225_BandRejectReason::e_securityDenial;
	bool bReject = !authList->Check(obj_brq, rsn);
	if (!bReject && !pCall) {
		bReject = true;
		rsn = H225_BandRejectReason::e_invalidConferenceID;
	} else if (!CallTbl->GetAdmission(bandwidth, pCall)) {
		bReject = true;
		rsn = H225_BandRejectReason::e_insufficientResources;
		brj.m_allowedBandWidth = CallTbl->GetAvailableBW();
		// ask the endpoint to try alternate gatekeepers
		SetAltGKInfo(brj);
	}
	if (bReject) {
		brj.m_rejectReason.SetTag(rsn);
//		CopyNonStandardData(obj_brq, brj);
		msg = PString(PString::Printf, "BRJ|%s|%s|%u|%s;\r\n", 
			inet_ntoa(rx_addr),
			(const unsigned char *)obj_brq.m_endpointIdentifier.GetValue(),
			bandwidth,
			(const unsigned char *)brj.m_rejectReason.GetTagName());
	} else {
		pCall->SetBandwidth(bandwidth);
		obj_rpl.SetTag(H225_RasMessage::e_bandwidthConfirm); // re-cast
		H225_BandwidthConfirm & bcf = obj_rpl;
		bcf.m_requestSeqNum = obj_brq.m_requestSeqNum;
		bcf.m_bandWidth = bandwidth;
//		CopyNonStandardData(obj_brq, bcf);
		msg = PString(PString::Printf, "BCF|%s|%s|%u;\r\n", 
			inet_ntoa(rx_addr),
			(const unsigned char *)obj_brq.m_endpointIdentifier.GetValue(),
			bandwidth);
	}
	PTRACE(2, msg);
	GkStatusThread->SignalStatus(msg);

	return TRUE;
}


/* Location Request */
BOOL H323RasSrv::OnLRQ(const PIPSocket::Address & rx_addr, H225_RasMessage & obj_rr, H225_RasMessage & obj_rpl)
{
	PTRACE(1, "GK\tLRQ Received");

	PString msg;
	endptr WantedEndPoint;

	const H225_LocationRequest & obj_lrq = obj_rr;
	// we need to put the original destinationInfo in LCF, optionally
	// filtered out to contain only the alias that matched endpoint table search
	H225_ArrayOf_AliasAddress lcfDestinationInfo = obj_lrq.m_destinationInfo;
	if (obj_lrq.m_destinationInfo.GetSize() > 0)
		Toolkit::Instance()->RewriteE164(obj_lrq.m_destinationInfo[0]);

	unsigned rsn = H225_LocationRejectReason::e_securityDenial;
	PIPSocket::Address ipaddr;
	WORD port;
	bool fromRegEndpoint = false;
	bool replyAddrMatch = GetIPAndPortFromTransportAddr(obj_lrq.m_replyAddress, ipaddr, port) && (ipaddr == rx_addr);
	if (obj_lrq.HasOptionalField(H225_LocationRequest::e_endpointIdentifier))
		if (endptr ep = EndpointTable->FindByEndpointId(obj_lrq.m_endpointIdentifier))
			fromRegEndpoint = replyAddrMatch ? true : ep->IsNATed();
	bool bReject = (!(fromRegEndpoint || NeighborsGK->CheckIP(rx_addr)) || !authList->Check(obj_lrq, rsn));
	if (!(fromRegEndpoint || replyAddrMatch || Toolkit::AsBool(GkConfig()->GetString(LRQFeaturesSection, "AcceptForwardedLRQ", "1"))))
		bReject = true;

	PString sourceInfoString(fromRegEndpoint ? obj_lrq.m_endpointIdentifier.GetValue() : obj_lrq.HasOptionalField(H225_LocationRequest::e_gatekeeperIdentifier) ? obj_lrq.m_gatekeeperIdentifier.GetValue() : rx_addr.AsString());
	if (!bReject) {
		int matchedalias = -1;
		if (
		// only search registered endpoints
#if WITH_DEST_ANALYSIS_LIST
		(WantedEndPoint = EndpointTable->getMsgDestination(obj_lrq, rsn, false))
#else
		(WantedEndPoint = EndpointTable->FindEndpoint(obj_lrq.m_destinationInfo, false, false,&matchedalias))
#endif
		) {
			// Alias found
			obj_rpl.SetTag(H225_RasMessage::e_locationConfirm);
			H225_LocationConfirm & lcf = obj_rpl;
			lcf.m_requestSeqNum = obj_lrq.m_requestSeqNum;
//			CopyNonStandardData(obj_lrq, lcf);

			WantedEndPoint->BuildLCF(obj_rpl);
			if (GKRoutedSignaling && AcceptNBCalls) {
				lcf.m_callSignalAddress = GetCallSignalAddress(rx_addr);
				lcf.m_rasAddress = GetRasAddress(rx_addr);
			}

			// BuildLCF filled destinationInfo field (if IncludeDestinationInfoInLCF is set)
			if( lcf.HasOptionalField(H225_LocationConfirm::e_destinationInfo)
				&& matchedalias >= 0 && matchedalias < lcfDestinationInfo.GetSize() ) {
				// overwrite destination info to include the alias 
				// that matched the destination endpoint
				lcf.m_destinationInfo.SetSize(1);
				// in GK routed mode we can return the original alias
				// in direct mode we have to return the rewritten one
				if( GKRoutedSignaling )
					lcf.m_destinationInfo[0] = lcfDestinationInfo[matchedalias];
				else
					lcf.m_destinationInfo[0] = obj_lrq.m_destinationInfo[matchedalias];
			}
			
			msg = PString(PString::Printf, "LCF|%s|%s|%s|%s;\r\n", 
					inet_ntoa(rx_addr),
					(const unsigned char *) WantedEndPoint->GetEndpointIdentifier().GetValue(),
					(const unsigned char *) AsString(
						lcf.HasOptionalField(H225_LocationConfirm::e_destinationInfo)
							?lcf.m_destinationInfo:obj_lrq.m_destinationInfo
							),
					(const unsigned char *) sourceInfoString);
		} else {
			if (NeighborsGK->ForwardLRQ(rx_addr, obj_lrq) > 0)
				return FALSE; // forward successful, nothing for us :)
			bReject = true;
			rsn = H225_LocationRejectReason::e_requestDenied;
		}
	}
	if (bReject) {
		// Alias not found
		obj_rpl.SetTag(H225_RasMessage::e_locationReject);
		H225_LocationReject & lrj = obj_rpl;
		lrj.m_requestSeqNum = obj_lrq.m_requestSeqNum;
		lrj.m_rejectReason.SetTag(rsn);
//		CopyNonStandardData(obj_lrq, lrj);

		msg = PString(PString::Printf, "LRJ|%s|%s|%s|%s;\r\n", 
			     inet_ntoa(rx_addr),
			     (const unsigned char *) AsString(obj_lrq.m_destinationInfo),
			     (const unsigned char *) sourceInfoString,
			     (const unsigned char *) lrj.m_rejectReason.GetTagName() );
	}

	PTRACE(2, msg);
	GkStatusThread->SignalStatus(msg);

	if (!(fromRegEndpoint || replyAddrMatch)) {
		if (!bReject)
			NeighborsGK->InsertSiblingIP(ipaddr);
		// reply to replyAddress instead of rx_addr
		SendReply(obj_rpl, ipaddr, port, listener);
		return FALSE;
	}
	// for a regsistered endpoint, reply to rx_addr
	return TRUE;
}

/* Location Confirm */
BOOL H323RasSrv::OnLCF(
	const PIPSocket::Address& /*rx_addr*/, 
	H225_RasMessage& obj_rr, 
	H225_RasMessage& /*obj_rpl*/
	)
{
	PTRACE(1, "GK\tLCF Received");

//	if (NeighborsGK->CheckIP(rx_addr)) // may send from sibling
	if (!gkClient->OnLCF(obj_rr))
		arqPendingList->ProcessLCF(obj_rr);
	return FALSE;
}

/* Location Reject */
BOOL H323RasSrv::OnLRJ(const PIPSocket::Address & rx_addr, H225_RasMessage &obj_rr, H225_RasMessage &)
{
	PTRACE(1, "GK\tLRJ Received");
	// we should ignore LRJ from sibling
	if (NeighborsGK->CheckIP(rx_addr))
		arqPendingList->ProcessLRJ(obj_rr);
	return FALSE;
}


/* Information Request Response */
BOOL H323RasSrv::OnIRR(const PIPSocket::Address & rx_addr, H225_RasMessage &obj_rr, H225_RasMessage &obj_rpl)
{ 
	PTRACE(1, "GK\tIRR Received");

	const H225_InfoRequestResponse & obj_irr = obj_rr;

	if (endptr ep = EndpointTable->FindByEndpointId(obj_irr.m_endpointIdentifier)) {
		ep->Update(obj_rr);
		if (obj_irr.HasOptionalField( H225_InfoRequestResponse::e_needResponse) && obj_irr.m_needResponse) {
			obj_rpl.SetTag(H225_RasMessage::e_infoRequestAck);
			H225_InfoRequestAck & ira = obj_rpl;
			ira.m_requestSeqNum = obj_irr.m_requestSeqNum;
//			CopyNonStandardData(obj_irr, ira);

			PString msg(PString::Printf, "IACK|%s;", inet_ntoa(rx_addr));
			PTRACE(2, msg);
			GkStatusThread->SignalStatus(msg + "\r\n");
			return TRUE;
		}
	}
	// otherwise don't respond
	return FALSE;
}

/* Admission Confirm */
BOOL H323RasSrv::OnRIP(const PIPSocket::Address& rx_addr, H225_RasMessage &obj_rr, H225_RasMessage &)
{
	PTRACE(1, "GK\tRIP Received");

	if (!gkClient->OnRIP(obj_rr, rx_addr))
		arqPendingList->ProcessRIP(obj_rr);

	return FALSE;
}

/* Resource Availability Indicate */
BOOL H323RasSrv::OnRAI(
	const PIPSocket::Address& /*rx_addr*/, 
	H225_RasMessage& obj_rr, 
	H225_RasMessage& obj_rpl
	)
{ 
	PTRACE(1, "GK\tRAI Received");

	const H225_ResourcesAvailableIndicate & obj_rai = obj_rr;

	/* accept all RAIs */
	obj_rpl.SetTag(H225_RasMessage::e_resourcesAvailableConfirm);
	H225_ResourcesAvailableConfirm & rac = obj_rpl;
	rac.m_requestSeqNum = obj_rai.m_requestSeqNum;
	rac.m_protocolIdentifier =  obj_rai.m_protocolIdentifier;
//	CopyNonStandardData(obj_rai, rac);
    
	return TRUE;
}

/* Service Control Indication */
BOOL H323RasSrv::OnSCI(
	const PIPSocket::Address& /*rx_addr*/, 
	H225_RasMessage& /*obj_rr*/, 
	H225_RasMessage& /*obj_rpl*/
	)
{
	PTRACE(1, "GK\tSCI Received");
	return FALSE;
}
 
BOOL H323RasSrv::OnSCR(
	const PIPSocket::Address& /*rx_addr*/, 
	H225_RasMessage& /*obj_rr*/, 
	H225_RasMessage& /*obj_rpl*/
	)
{
	PTRACE(1, "GK\tSCR Received");
	return FALSE;
}
 
BOOL H323RasSrv::OnIgnored(
	const PIPSocket::Address& /*rx_addr*/, 
	H225_RasMessage& obj_rr, 
	H225_RasMessage& /*obj_rpl*/
	)
{
	PTRACE(2, "GK\t" << obj_rr.GetTagName() << " received and safely ignored");
	return FALSE;
}

BOOL H323RasSrv::OnUnknown(
	const PIPSocket::Address& /*rx_addr*/, 
	H225_RasMessage& /*obj_rr*/, 
	H225_RasMessage& /*obj_rpl*/
	)
{
	PTRACE(1, "GK\tUnknown RAS message received");
	return FALSE;
}

bool H323RasSrv::Check()
{
	PWaitAndSignal lock(loadLock);
	if (sigHandler)
		sigHandler->Check();
	gkClient->CheckRegistration();
	arqPendingList->Check();
#if HAS_WAITARQ
	wArqList->CheckWaitingARQ();
#endif
 
	return !IsTerminated();
}

void H323RasSrv::SendRas(const H225_RasMessage & obj_ras, const H225_TransportAddress & dest)
{
	SendRas(obj_ras, dest, listener.IsOpen() ? listener : udpForwarding);
}

void H323RasSrv::SendRas(const H225_RasMessage & obj_ras, const H225_TransportAddress & dest, PUDPSocket & BoundSocket)
{
	PIPSocket::Address ip;
	WORD port;
	if (GetIPAndPortFromTransportAddr(dest, ip, port)) {
		SendReply(obj_ras, ip, port, BoundSocket);
	} else {
		PTRACE(2, "No IP address to send!" );
	}
}

void H323RasSrv::SendRas(const H225_RasMessage & obj_ras, const PIPSocket::Address & rx_addr, WORD rx_port)
{
	SendReply(obj_ras, rx_addr, rx_port, listener.IsOpen() ? listener : udpForwarding);
}

void H323RasSrv::SendReply(const H225_RasMessage & obj_rpl, const PIPSocket::Address & rx_addr, WORD rx_port, PUDPSocket & BoundSocket)
{
#if PTRACING
	if (PTrace::CanTrace(3))
		PTRACE(3, "GK\tSend to " << rx_addr << ':' << rx_port << '\n' << setprecision(2) << obj_rpl);
	else
		PTRACE(2, "GK\tSend " << obj_rpl.GetTagName() << " to " << rx_addr << ':' << rx_port);
#endif

	PBYTEArray wtbuf(4096);
	PPER_Stream wtstrm(wtbuf);
	obj_rpl.Encode(wtstrm);
	wtstrm.CompleteEncoding();

	PWaitAndSignal lock(writeMutex);
	if(!BoundSocket.WriteTo(wtstrm.GetPointer(), wtstrm.GetSize(), rx_addr, rx_port) ) {
		PTRACE(4, "GK\tRAS thread: Write error: " << BoundSocket.GetErrorText());
	} else {
		PTRACE(5, "GK\tSent Successful");
	}
}


void H323RasSrv::Main(void)
{
	ReadLock cfglock(ConfigReloadMutex);

	loadLock.Wait(); // lock to avoid Check
	const int buffersize = 4096;
	BYTE buffer[buffersize];

	PTRACE(1, "GK\tRasThread " << getpid() << " started");

	PString err_msg("ERROR: Request received by gatekeeper: ");   
	PTRACE(2, "GK\tEntering connection handling loop");

	// queueSize is useless for UDPSocket
	listener.Listen(GKHome, 0, GKRasPort, PSocket::CanReuseAddress);
	PTRACE_IF(1, !listener.IsOpen(), "GK\tBind to RAS port failed!");

	PTimeInterval wtimeout(500);
	listener.SetWriteTimeout(wtimeout);

	gkClient = new GkClient(this);
	loadLock.Signal(); // OK, the thread is ready

	while (listener.IsOpen()) { 
		PIPSocket::Address rx_addr;
		H225_RasMessage obj_req;   
		H225_RasMessage obj_rpl;

		// large mutex! only allow the reloadhandler be executed
		// in the small block
		ConfigReloadMutex.EndRead();
		int iResult = listener.ReadFrom(buffer, buffersize, rx_addr, m_rx_port);
		// not allow to reload below here
		ConfigReloadMutex.StartRead();

		if (!iResult) {
			PTRACE(1, "GK\tRAS thread: Read error: " << listener.GetErrorText());

			// TODO: "return" (terminate) on some errors (like the one at shutdown)
			continue;
		}
		PTRACE(2, "GK\tRead from " << rx_addr << ':' << m_rx_port);

		// get only bytes which are really read
		PPER_Stream rawPDU(buffer, listener.GetLastReadCount());
		// set rawPDU for authentication methods
		authList->setLastReceivedRawPDU(rawPDU);

		if (!obj_req.Decode(rawPDU)) {
			PTRACE(1, "GK\tCouldn't decode message!");
			continue;
		}
#if PTRACING
		if (PTrace::CanTrace(3))
			PTRACE(3, "GK\n" << setprecision(2) << obj_req);
		else
			PTRACE(2, "GK\tReceived " << obj_req.GetTagName());
#endif          

		if (obj_req.GetTag() <= H225_RasMessage::e_serviceControlResponse) {
			if ((this->*rasHandler[obj_req.GetTag()])(rx_addr, obj_req, obj_rpl))
				SendReply( obj_rpl, rx_addr, m_rx_port, listener );
		} else {
			PTRACE(1, "GK\tWarning: unknown RAS message tag");
		}
	}

	// after listener closed, use this socket to send RAS
	udpForwarding.Listen(GKHome, 0, GKRasPort, PSocket::CanReuseAddress);
	udpForwarding.SetWriteTimeout(wtimeout);

	PTRACE(1, "GK\tRasThread terminated!");
}

void H323RasSrv::SelectH235Capability( 
	const H225_GatekeeperRequest& grq,
	H225_GatekeeperConfirm& gcf
	)
{
	if( authList == NULL )
		return;
		
	// if GRQ does not contain a list of authentication mechanisms simply return
	if( !(grq.HasOptionalField(H225_GatekeeperRequest::e_authenticationCapability)
			&& grq.HasOptionalField(H225_GatekeeperRequest::e_algorithmOIDs)
			&& grq.m_authenticationCapability.GetSize() > 0
			&& grq.m_algorithmOIDs.GetSize() > 0) )
		return;

	H225_ArrayOf_AuthenticationMechanism mechanisms;
	H225_ArrayOf_PASN_ObjectId algorithmOIDs;

	authList->GetH235Capabilities(mechanisms,algorithmOIDs);
	
	// And now match H.235 capabilities found with those from GRQ
	// to find the one to be returned in GCF		
	for( int i = 0; i < grq.m_authenticationCapability.GetSize(); i++ )
		for( int j = 0; j < mechanisms.GetSize(); j++ )	
			if( grq.m_authenticationCapability[i].GetTag() 
				== mechanisms[j].GetTag() )
				for( int l = 0; l < algorithmOIDs.GetSize(); l++ )
					for( int k = 0; k < grq.m_algorithmOIDs.GetSize(); k++ )
						if( grq.m_algorithmOIDs[k] == algorithmOIDs[l] )
						{
							GkAuthenticator* authenticator = authList ? authList->GetHead() : NULL;
							while( authenticator )
							{
								if( authenticator->IsH235Capable() 
									&& authenticator->IsH235Capability(
										mechanisms[j], algorithmOIDs[l]
										) )
								{
									gcf.IncludeOptionalField(H225_GatekeeperConfirm::e_authenticationMode);
									gcf.m_authenticationMode = mechanisms[j];
									gcf.IncludeOptionalField(H225_GatekeeperConfirm::e_algorithmOID);
									gcf.m_algorithmOID = algorithmOIDs[l];
									
									PTRACE(4,"GK\tGCF will select authentication mechanism: "
										<<mechanisms[j]<<" and algorithm OID: "<<algorithmOIDs[l]
										);
									return;
								}
								authenticator = authenticator->GetNext();
							}
							
							PTRACE(5,"GK\tAuthentication mechanism: "
								<<mechanisms[j]<<" and algorithm OID: "
								<<algorithmOIDs[l]<<" dropped"
								);
						}
}
