//////////////////////////////////////////////////////////////////
//
// ProxyThread.cxx
//
// $Id: ProxyThread.cxx,v 1.15.2.45 2004/05/25 19:54:03 zvision Exp $
//
// Copyright (c) Citron Network Inc. 2001-2002
//
// This work is published under the GNU Public License (GPL)
// see file COPYING for details.
// We also explicitely grant the right to link this code
// with the OpenH323 library.
//
// initial author: Chih-Wei Huang <cwhuang@linux.org.tw>
// initial version: 12/7/2001
//
//////////////////////////////////////////////////////////////////

#if (_MSC_VER >= 1200)
#pragma warning( disable : 4355 ) // warning about using 'this' in initializer
#pragma warning( disable : 4786 ) // warning about too long debug symbol off
#pragma warning( disable : 4800 ) // warning about forcing value to bool
#endif

#include <ptlib.h>
#include <ptlib/sockets.h>
#include "h323util.h"
#include "stl_supp.h"
#include "rwlock.h"
#include "gk.h"
#include "ProxyThread.h"


class ProxyConnectThread : public MyPThread {
public:
	PCLASSINFO ( ProxyConnectThread, MyPThread )

	ProxyConnectThread(ProxyHandleThread *);
	bool Acquire();
	void Release() { available = true; }
	bool Connect(ProxySocket *);

	// override from class MyPThread
	virtual void Exec();

private:
	ProxyHandleThread *handler;
	bool available;

	ProxySocket *calling;
	PMutex amutex;
};

struct TPKTV3 {
	TPKTV3() {}
	TPKTV3(WORD);

	BYTE header, padding;
	WORD length;
};

inline TPKTV3::TPKTV3(WORD len)
	: header(3), padding(0)
{
	length = PIPSocket::Host2Net(WORD(len + sizeof(TPKTV3)));
}


// class ProxySocket
ProxySocket::ProxySocket(IPSocket *s, const char *t) : self(s), type(t)
{
	wbufsize = 1536; // 1.5KB
	wbuffer = new BYTE[wbufsize];
	qsize = buflen = 0;
	blocked = connected = deletable = false;
}

ProxySocket::~ProxySocket()
{
	PWaitAndSignal lock(writeMutex);
	delete [] wbuffer;
	DeleteObjectsInContainer<PBYTEArray>(queue);
	queue.clear();
	qsize = 0;
	PTRACE(3, type << "\tDelete socket " << name);
}

ProxySocket::Result ProxySocket::ReceiveData()
{
	if (!self->Read(wbuffer, wbufsize)) {
		ErrorHandler(PSocket::LastReadError);
		return NoData;
	}
	PTRACE(6, type << "\tReading from " << Name());
	buflen = (WORD)self->GetLastReadCount();
	return Forwarding;
}

bool ProxySocket::ForwardData()
{
	return WriteData(wbuffer, buflen);
}

bool ProxySocket::TransmitData(const PBYTEArray & buf)
{
	return WriteData(buf, (WORD)buf.GetSize());
}

bool ProxySocket::TransmitData(const BYTE *buf, WORD len)
{
	return WriteData(buf, len);
}

bool ProxySocket::EndSession()
{
	MarkBlocked(false);
//	SetConnected(false);
	return CloseSocket();
}

bool ProxySocket::Flush()
{
	while (qsize > 0) {
		PBYTEArray* const packet = PopQueuedPacket();
		if (packet == NULL)
			break;
		PBYTEArray data(*packet);
		delete packet;
		if (!FlushData(data, (WORD)data.GetSize()))
			return false;
		PTRACE(4, type << '\t' << data.GetSize() << " bytes flushed to " << Name());
	}
	return true;
}

bool ProxySocket::WriteData(const BYTE *buf, WORD len)
{
	if (!IsSocketOpen())
		return false;

	if (qsize > 0 || writeMutex.WillBlock()) {
		QueuePacket(buf, len);
		PTRACE(3, Name() << " socket is busy, " << len << " bytes queued");
		return false;
	}

	PWaitAndSignal lock(writeMutex);

	if (self->Write(buf, len)) {
		PTRACE(6, Name() << ' ' << len << " bytes sent");
		return true;
	}

	const WORD wcount = (WORD)self->GetLastWriteCount();
	buf += wcount;
	len -= wcount;

	PutBackPacket(buf, len);
	PTRACE(4, Name() << " blocked, " << wcount << " bytes written, " << len << " bytes queued");
	return (wcount > 0) ? false : ErrorHandler(PSocket::LastWriteError);
}

bool ProxySocket::FlushData(const BYTE *buf, WORD len)
{
	if (!IsSocketOpen())
		return false;

	if (!writeMutex.WillBlock()) {
		PWaitAndSignal lock(writeMutex);

		if (self->Write(buf, len)) {
			PTRACE(6, Name() << ' ' << len << " bytes sent");
			return true;
		}

		const WORD wcount = (WORD)self->GetLastWriteCount();
		buf += wcount;
		len -= wcount;

		PutBackPacket(buf, len);

		PTRACE(4, Name() << " blocked, " << wcount << " bytes written, " << len << " bytes queued");
		return (wcount > 0) ? false : ErrorHandler(PSocket::LastWriteError);
	} else {
		PutBackPacket(buf, len);
		PTRACE(3, Name() << " socket is busy, " << len << " bytes queued");
		return false;
	}
}

bool ProxySocket::ErrorHandler(PSocket::ErrorGroup group)
{
	PSocket::Errors e = self->GetErrorCode(group);

	PString msg(PString(type) + "\t" + Name());
	switch (e)
	{
	//	case PSocket::NoError:
	//	// I don't know why there is error with code NoError
	//		PTRACE(4, msg << " Error(" << group << "): No error?");
	//		break;
		case PSocket::Timeout:
			PTRACE(4, msg << " Error(" << group << "): Timeout");
			break;
		default:
			PTRACE(3, msg << " Error(" << group << "): " << PSocket::GetErrorText(e) << " (" << e << ')');
			CloseSocket();
			break;
	}
	return false;
}

void ProxySocket::SetName(const PIPSocket::Address & ip, WORD pt)
{
	name = AsString(ip, pt);
}


// class TCPProxySocket
TCPProxySocket::TCPProxySocket(const char *t, TCPProxySocket *s, WORD p)
      : TCPSocket(p), ProxySocket(this, t), remote(s)
{
}

TCPProxySocket::~TCPProxySocket()
{
	if (remote) {
		remote->remote = 0; // detach myself from remote
		remote->SetDeletable();
	}
}

bool TCPProxySocket::ForwardData()
{
	return (remote) ? remote->InternalWrite(buffer) : false;
}

bool TCPProxySocket::TransmitData(const PBYTEArray & buf)
{
	return InternalWrite(buf);
}

#ifdef LARGE_FDSET
BOOL TCPProxySocket::Accept(PSocket & socket)
{
	while (true) {
		int fd = socket.GetHandle();
		if (fd < 0) { // socket closed
			errno = ENOTSOCK;
			break;
		}

		YaSelectList::large_fd_set fdset;
		fdset.add(fd);
		struct timeval tval = { 1, 0 };
		int r = ::select(fd + 1, fdset, 0, 0, &tval);
		if (r < 0)
			break;
		else if (r == 0)
			continue;

		socklen_t addrsize = sizeof(peeraddr);
		os_handle = ::accept(fd, (struct sockaddr *)&peeraddr, &addrsize);
		if (os_handle < 0)
			break;

		SetNonBlockingMode();
		SetWriteTimeout(PTimeInterval(10));
		SetName(peeraddr.sin_addr, ntohs(peeraddr.sin_port));
		return TRUE;
	}
	return ConvertOSError(-1);
}

BOOL TCPProxySocket::Connect(const Address & iface, WORD localPort, const Address & addr)
{
	if (os_handle < 0) {
		os_handle = ::socket(PF_INET, SOCK_STREAM, 0);
		if (!ConvertOSError(os_handle))
			return FALSE;
	}

	SetNonBlockingMode();
	SetWriteTimeout(PTimeInterval(10));
	peeraddr.sin_addr = addr;
	SetName(addr, ntohs(peeraddr.sin_port));

	int optval;
	socklen_t optlen = sizeof(optval);

	// bind local port
	if (iface != INADDR_ANY || localPort != 0) {
		optval = 1;
		::setsockopt(os_handle, SOL_SOCKET, SO_REUSEADDR, &optval, optlen);
		sockaddr_in inaddr;
		inaddr.sin_family = AF_INET;

		inaddr.sin_addr.s_addr = iface;

		inaddr.sin_port = htons(localPort);
		if (!ConvertOSError(::bind(os_handle, (struct sockaddr *)&inaddr, sizeof(inaddr))))
			return FALSE;
	}

	// connect in non-blocking mode
	int r = ::connect(os_handle, (struct sockaddr *)&peeraddr, sizeof(peeraddr));
	if (r == 0 || errno != EINPROGRESS)
		return ConvertOSError(r);

	YaSelectList::large_fd_set fdset;
	fdset.add(os_handle);
	YaSelectList::large_fd_set exset = fdset;
	struct timeval tval = { 6, 0 }; // TODO: read from config...
	if (::select(os_handle + 1, 0, fdset, exset, &tval) > 0) {
		optval = -1;
		::getsockopt(os_handle, SOL_SOCKET, SO_ERROR, &optval, &optlen);
		if (optval == 0) // connected
			return TRUE;
		errno = optval;
	}
	return ConvertOSError(-1);
}

#else

BOOL TCPProxySocket::Accept(PSocket & socket)
{
//	SetReadTimeout(PMaxTimeInterval);
	BOOL result = PTCPSocket::Accept(socket);
	PTimeInterval timeout(100);
	SetReadTimeout(timeout);
	SetWriteTimeout(timeout);
	// since GetName() may not work if socket closed,
	// we save it for reference
	Address ip;
	WORD pt;
	GetPeerAddress(ip, pt);
	SetName(ip, pt);
	return result;
}

BOOL TCPProxySocket::Connect(const Address & iface, WORD localPort, const Address & addr)
{
	SetName(addr, GetPort());
	SetReadTimeout(PTimeInterval(6000)); // TODO: read from config...
	BOOL result = PTCPSocket::Connect(iface, localPort, addr);
	PTimeInterval timeout(100);
	SetReadTimeout(timeout);
	SetWriteTimeout(timeout);
	return result;
}

#endif

BOOL TCPProxySocket::Connect(WORD localPort, const Address & addr)
{
	return Connect(INADDR_ANY, localPort, addr);
}

BOOL TCPProxySocket::Connect(const Address & addr)
{
	return Connect(INADDR_ANY, 0, addr);
}

bool TCPProxySocket::ReadTPKT()
{
	PTRACE(5, Type() << "\tReading from " << Name());
	if (buflen == 0) {
		TPKTV3 tpkt;
		if (!ReadBlock(&tpkt, sizeof(TPKTV3)))
			return ErrorHandler(PSocket::LastReadError);
		//if (tpkt.header != 3 || tpkt.padding != 0)
		// some bad endpoints don't set padding to 0, e.g., Cisco AS5300
		if (tpkt.header != 3) {
			PTRACE(2, "Proxy\t" << Name() << " NOT TPKT PACKET!");
			return false; // Only support version 3
		}
		buflen = (WORD)(PIPSocket::Net2Host(tpkt.length) - sizeof(TPKTV3));
		if (buflen < 1) {
			PTRACE(3, "Proxy\t" << Name() << " PACKET TOO SHORT!");
			buflen = 0;
			return false;
		}
		if (!SetMinBufSize(buflen))
			return false;
		buffer = PBYTEArray(bufptr = wbuffer, buflen, false);
	}

#ifdef LARGE_FDSET
	// some endpoints may send TPKT header and payload in separate
	// packets, so we have to check again if data available
	if (!CanRead(0))
		return false;
#endif
	if (!Read(bufptr, buflen))
		return ErrorHandler(PSocket::LastReadError);

	if (!(buflen -= GetLastReadCount()))
		return true;

	bufptr += GetLastReadCount();
	PTRACE(3, "Proxy\t" << Name() << " read timeout?");
	return false;
}

bool TCPProxySocket::InternalWrite(const PBYTEArray & buf)
{
	const WORD len = (WORD)buf.GetSize();
	const WORD tlen = (WORD)(len + sizeof(TPKTV3));
	PBYTEArray tbuf(tlen);
	BYTE *bptr = tbuf.GetPointer();
	new (bptr) TPKTV3(len); // placement operator
	memcpy(bptr + sizeof(TPKTV3), buf, len);
	return WriteData(bptr, tlen);
}

bool TCPProxySocket::SetMinBufSize(WORD len)
{
	if (wbufsize < len) {
		delete [] wbuffer;
		wbuffer = new BYTE[wbufsize = len];
	}
	return (wbuffer != 0);
}


// class ProxyConnectThread
ProxyConnectThread::ProxyConnectThread(ProxyHandleThread *h)
	: handler(h), available(true)
{
	Resume();
}

bool ProxyConnectThread::Acquire()
{
	PWaitAndSignal lock(amutex);
	if (!available)
		return false;
	available = false;
	return true;
}

bool ProxyConnectThread::Connect(ProxySocket *socket)
{
	if (available) {
		PTRACE(1, "ProxyC\tWarning: not acquired yet");
		return false;
	}

	socket->MarkBlocked(true);
	calling = socket;
	Go();
	return true;
}

void ProxyConnectThread::Exec()
{
	if (!Wait())
		return;

	TCPProxySocket *socket = dynamic_cast<TCPProxySocket *>(calling);
	TCPProxySocket *remote = socket->ConnectTo();
	if (remote) {
		handler->Insert(remote);
		socket->MarkBlocked(false);
	}
	// else
	// 	Note: socket may be invalid

	available = true;
}

// class ProxyListener
ProxyListener::ProxyListener(HandlerList *h, const PIPSocket::Address & i, WORD p, unsigned qs)
      : m_listener(0), m_interface(i), m_port(p), m_handler(h)
{
	Open(qs);
}

ProxyListener::~ProxyListener()
{
	delete m_listener;
}

bool ProxyListener::Open(unsigned queueSize)
{
	m_listener = new PTCPSocket(m_port);
	isOpen = m_listener->Listen(m_interface, queueSize, m_port, PSocket::CanReuseAddress);
	m_port = m_listener->GetPort(); // get the listen port
	if (isOpen) {
		PTRACE(2, "ProxyL\tListen to " << m_interface << ':' << m_port);
		Resume();
#if PTRACING
	} else {
		PTRACE(1, "ProxyL\tCan't listen port " << m_port);
#endif
	}
	return isOpen;
}

void ProxyListener::Close()
{
	PTRACE(3, "ProxyL\tClosing ProxyListener");
	m_listener->Close();
	isOpen = false;
}

void ProxyListener::Exec()
{
	if (!m_listener->IsOpen()) {
		isOpen = false;
		return;
	}

	TCPProxySocket *socket = CreateSocket();
	if (socket->Accept(*m_listener)) { // incoming connection
		PTRACE(3, "ProxyL\tConnected from " << socket->Name());
		m_handler->Insert(socket);
	} else {
		PTRACE(3, "ProxyL\tError: " << socket->GetErrorText(PSocket::LastGeneralError));
		delete socket;  // delete unused socket
	}
}

// class ProxyHandleThread
ProxyHandleThread::ProxyHandleThread(PINDEX i)
{
	SetID(PString(PString::Printf, "ProxyH(%u)", i));
	Resume();

	lcHandler = new ProxyHandleThread;
	lcHandler->SetID(PString(PString::Printf, "ProxyLC(%u)", i));
	lcHandler->SetPriority(HighPriority);
	lcHandler->Resume();
}

ProxyHandleThread::~ProxyHandleThread()
{
	ForEachInContainer(connList, mem_fun(&MyPThread::Destroy));
	DeleteObjectsInContainer<ProxySocket>(sockList);
	// The RTP/RTCP sockets should be deleted after
	// call signalling sockets being deleted
	if (lcHandler)
		lcHandler->Destroy();
}

void ProxyHandleThread::Insert(ProxySocket *socket)
{
	socket->SetHandler(this);
	mutex.StartWrite();
	sockList.push_back(socket);
	PTRACE(5, id << " add a socket, total " << sockList.size());
	mutex.EndWrite();
	Go();
}

void ProxyHandleThread::FlushSockets()
{
	SocketSelectList wlist;
	mutex.StartRead();
	iterator i = sockList.begin(), j = sockList.end();
	while (i != j) {
		if ((*i)->CanFlush())
			(*i)->AddToSelectList(wlist);
		++i;
	}
	mutex.EndRead();
	if (wlist.IsEmpty())
		return;

	if (!wlist.Select(SocketSelectList::Write, PTimeInterval(10)))
	       return;
	
	PTRACE(5, "Proxy\t" << wlist.GetSize() << " sockets to flush...");
	const int wsize = wlist.GetSize();
	for (int k = 0; k < wsize; ++k) {
		ProxySocket *socket = dynamic_cast<ProxySocket *>(wlist[k]);
		if (socket->Flush()) {
			PTRACE(4, "Proxy\t" << socket->Name() << " flush ok");
		}
	}
}

void ProxyHandleThread::MoveTo(ProxyHandleThread *dest, ProxySocket *socket)
{
	mutex.StartWrite();
	sockList.remove(socket);
	mutex.EndWrite();
	dest->Insert(socket);
}

void ProxyHandleThread::Remove(iterator i)
{
	PWaitAndSignal lock(removedMutex);
	removedList.push_back(*i);
	removedTime.push_back(new PTime);
	sockList.erase(i); // list already be locked
}

void ProxyHandleThread::Remove(ProxySocket *socket)
{
	PWaitAndSignal lock(removedMutex);
	removedList.push_back(socket);
	removedTime.push_back(new PTime);
	sockList.remove(socket); // list already be locked
}

void ProxyHandleThread::RemoveSockets()
{
	PTime now;
	PWaitAndSignal lock(removedMutex);
	iterator i = removedList.begin();
	std::list<PTime *>::iterator ti = removedTime.begin();
	while ((ti != removedTime.end()) && ((now - **ti) > 5*1000)) {
		delete *i++;
		delete *ti++;
	}
	removedList.erase(removedList.begin(), i);
	removedTime.erase(removedTime.begin(), ti);
}

void ProxyHandleThread::BuildSelectList(SocketSelectList & result)
{
	WriteLock lock(mutex);
	iterator i = sockList.begin(), j = sockList.end();
	while (i != j) {
		iterator k=i++;
		ProxySocket *socket = *k;
		if (!socket->IsBlocked()) {
			if (socket->IsSocketOpen())
				socket->AddToSelectList(result);
			else if (!socket->IsConnected()) {
				Remove(k);
				continue;
			}
			if (socket->IsDeletable())
				Remove(k);
#if PTRACING
	//	} else {
	//		PTRACE(5, socket->Name() << " is busy!");
#endif
		}
	}
}

void ProxyHandleThread::Exec()
{
	int ss = 0;
	ReadLock cfglock(ConfigReloadMutex);
	SocketSelectList sList;
	while (true) {
		FlushSockets();
		BuildSelectList(sList);
		ss = sList.GetSize();
		if (ss > 0) {
			ReadUnlock cfglock(ConfigReloadMutex);
			if (sList.Select(SocketSelectList::Read, PTimeInterval(100)))
				break;
			RemoveSockets();
			return;
		}
		RemoveSockets();
		PTRACE(3, id << " waiting...");

		ReadUnlock unlock(ConfigReloadMutex);
		if (!Wait())
			return;
	}

#if PTRACING
	if (PTrace::CanTrace(4)) {
		ReadLock lock(mutex);
		int slistsize = sockList.size(), rlistsize = removedList.size();
		PString msg(PString::Printf, " %u sockets selected from %u, total %u/%u", sList.GetSize(), ss, slistsize, rlistsize);
		PTRACE(4, id + msg);
	}
#endif
	ss = sList.GetSize();
	for (int i = 0; i < ss; ++i) {
		ProxySocket *socket = dynamic_cast<ProxySocket *>(sList[i]);
		switch (socket->ReceiveData())
		{
			case ProxySocket::Connecting:
				ConnectTo(socket);
				break;
			case ProxySocket::Forwarding:
				if (!socket->ForwardData()) {
					PTRACE(3, "Proxy\t" << socket->Name() << " forward blocked");
				}
				break;
			case ProxySocket::Closing:
				socket->ForwardData();
				// then close the socket
			case ProxySocket::Error:
				socket->CloseSocket();
				break;
			default:
				break;
		}
	}
	RemoveSockets();
}

ProxyConnectThread *ProxyHandleThread::FindConnectThread()
{
	connMutex.StartRead();
	for (citerator i = connList.begin(); i != connList.end(); ++i)
		if ((*i)->Acquire()) {
			connMutex.EndRead();
			return (*i);
		}

	connMutex.EndRead();
	ProxyConnectThread *ct = new ProxyConnectThread(this);
	connMutex.StartWrite();
	connList.push_back(ct);
	PTRACE(2, "Proxy\tCreate a new ConnectThread, total " << connList.size());
	connMutex.EndWrite();
	ct->Acquire();
	return ct;
}

void ProxyHandleThread::ConnectTo(ProxySocket *socket)
{
	FindConnectThread()->Connect(socket);
}

bool ProxyHandleThread::CloseUnusedThreads()
{
	static unsigned idx = 0;
	if (connList.size() <= 1 || (++idx % 60))
		return true;

	ProxyConnectThread *ct = 0;
	connMutex.StartRead();
	citerator i = connList.end();
	do {
		if ((*--i)->Acquire()) {
			if (!ct) {
				ct = (*i);
				continue;
			}
			connMutex.EndRead();
			connMutex.StartWrite();
			connList.remove(ct);
			PTRACE(1, id << " Close one unused ConnectThread, left " << connList.size());
			connMutex.EndWrite();
			(*i)->Release();
			ct->Destroy();
			return true;
		}
	} while (i != connList.begin());
	connMutex.EndRead();
	if (ct)
		ct->Release();
	return true; // useless, workaround for VC
}


HandlerList::HandlerList(PIPSocket::Address home) : GKHome(home), GKPort(0)
{
	currentHandler = 0;
	listenerThread = 0;
	LoadConfig();
#ifdef LARGE_FDSET
	PTRACE(1, "GK\tLarge fd_set(" << LARGE_FDSET << ") enabled");
#endif
}

HandlerList::~HandlerList()
{
	CloseListener();
	ForEachInContainer(handlers, mem_fun(&MyPThread::Destroy));
}

void HandlerList::Insert(ProxySocket *socket)
{
	// only lister thread will call this method, no mutex required
	handlers[currentHandler]->Insert(socket);
	if (++currentHandler >= handlers.size())
		currentHandler = 0;
}

void HandlerList::Check()
{
	ForEachInContainer(handlers, mem_fun(&ProxyHandleThread::CloseUnusedThreads));
}

void HandlerList::CloseListener()
{
	if (listenerThread) {
		listenerThread->Destroy();
		listenerThread = 0;
	}
}

