//////////////////////////////////////////////////////////////////
//
// ProxyChannel.cxx
//
// $Id: ProxyChannel.cxx,v 1.39.2.127 2004/05/25 19:54:02 zvision Exp $
//
// Copyright (c) Citron Network Inc. 2001-2002
//
// This work is published under the GNU Public License (GPL)
// see file COPYING for details.
// We also explicitely grant the right to link this code
// with the OpenH323 library.
//
// initial author: Chih-Wei Huang <cwhuang@linux.org.tw>
// initial version: 12/19/2001
//
//////////////////////////////////////////////////////////////////

#if (_MSC_VER >= 1200)
#pragma warning( disable : 4355 ) // warning about using 'this' in initializer
#pragma warning( disable : 4786 ) // warning about too long debug symbol off
#pragma warning( disable : 4800 ) // warning about forcing value to bool
#endif

#include <ptlib.h>
#include <q931.h>
#include <h245.h>
#include <h323pdu.h>
#include "h323util.h"
#include "gk_const.h"
#include "stl_supp.h"
#include "Toolkit.h"
#include "gk.h"
#include "RasSrv.h"
#include "GkClient.h"
#include "gkauth.h"
#ifdef HAS_ACCT
#include "gkacct.h"
#endif
#include "ProxyChannel.h"

const char* const RoutedSection = "RoutedMode";
const char* const ProxySection = "Proxy";
const char* const H225_ProtocolID = "0.0.8.2250.0.2";

class PortRange {
public:
	WORD GetPort();
	void LoadConfig(const char *, const char *, const char * = "");

private:
	WORD port, minport, maxport;
	PMutex mutex;
};

WORD PortRange::GetPort()
{
	if (port == 0)
		return 0;
	PWaitAndSignal lock(mutex);
	WORD result = port++;
	if (port > maxport)
		port = minport;
	return result;
}

void PortRange::LoadConfig(const char *sec, const char *setting, const char *def)
{
	PStringArray cfgs = GkConfig()->GetString(sec, setting, def).Tokenise(",.:-/'", FALSE);
	if (cfgs.GetSize() >= 2) // no such a setting in config
		port = minport = (WORD)cfgs[0].AsUnsigned(), maxport = (WORD)cfgs[1].AsUnsigned();
	else
		port = 0;
	PTRACE_IF(2, port, setting << ": " << minport << '-' << maxport);
}

static PortRange Q931PortRange;
static PortRange H245PortRange;
static PortRange T120PortRange;
static PortRange RTPPortRange;

class H245Socket : public TCPProxySocket {
public:
#ifndef LARGE_FDSET
	PCLASSINFO ( H245Socket, TCPProxySocket )
#endif

	H245Socket(CallSignalSocket *);
	H245Socket(H245Socket *, CallSignalSocket *);
	~H245Socket();

	// override from class ProxySocket
        virtual Result ReceiveData();
	virtual bool EndSession();

	// override from class TCPProxySocket
	virtual TCPProxySocket *ConnectTo();

	void SendEndSessionCommand();
	H225_TransportAddress GetH245Address(const Address &);
	bool SetH245Address(H225_TransportAddress & h245addr, const Address &);
	bool Reverting(const H225_TransportAddress &);
	void OnSignalingChannelClosed() { sigSocket = 0; }
	void SetSigSocket(CallSignalSocket *socket) { sigSocket = socket; }

protected:
	// override from class TCPProxySocket
	virtual BOOL Accept(PSocket &);
	// new virtual function
	virtual bool ConnectRemote();

	CallSignalSocket *sigSocket;
	H225_TransportAddress *peerH245Addr;
	PTCPSocket *listener;
};

class NATH245Socket : public H245Socket {
public:
#ifndef LARGE_FDSET
	PCLASSINFO ( NATH245Socket, H245Socket )
#endif
	NATH245Socket(CallSignalSocket *sig) : H245Socket(sig) {}

private:
	// override from class H245Socket
	virtual bool ConnectRemote();
};

class UDPProxySocket : public UDPSocket, public ProxySocket {
public:
#ifndef LARGE_FDSET
	PCLASSINFO( UDPProxySocket, UDPSocket )
#endif

	UDPProxySocket(const char *);
	virtual ~UDPProxySocket() {}

	void SetDestination(H245_UnicastAddress_iPAddress &);
	void SetForwardDestination(const Address &, WORD, const H245_UnicastAddress_iPAddress &);
	void SetReverseDestination(const Address &, WORD, const H245_UnicastAddress_iPAddress &);
	typedef void (UDPProxySocket::*pMem)(const Address &, WORD, const H245_UnicastAddress_iPAddress &);

	bool Bind(WORD pt);
	bool Bind(const Address& addr, WORD pt);
	void SetNAT(bool);
	void OnHandlerSwapped() { std::swap(fnat, rnat); }

	// override from class ProxySocket
	virtual Result ReceiveData();

protected:
	virtual bool WriteData(const BYTE* buffer, WORD len);
	virtual bool FlushData(const BYTE* buffer, WORD len);
		
private:
	// these should not be used
	UDPProxySocket();
	UDPProxySocket( const UDPProxySocket& );
	UDPProxySocket& operator=( const UDPProxySocket& );
	
	Address fSrcIP, fDestIP, rSrcIP, rDestIP;
	WORD fSrcPort, fDestPort, rSrcPort, rDestPort;
	bool fnat, rnat;
};

class T120ProxySocket : public TCPProxySocket {
public:
#ifndef LARGE_FDSET
	PCLASSINFO ( T120ProxySocket, TCPProxySocket )
#endif

	T120ProxySocket(T120ProxySocket * = 0, WORD = 0);

	void SetDestination(const Address & ip, WORD pt) { peerAddr = ip, peerPort = pt; }

	// override from class ProxySocket
	virtual bool ForwardData();

	// override from class TCPProxySocket
	virtual TCPProxySocket *ConnectTo();

private:
	Address peerAddr;
	WORD peerPort;
};

class LogicalChannel {
public:
	LogicalChannel(WORD flcn = 0) : channelNumber(flcn), used(false) {}
	virtual ~LogicalChannel() {}

	bool IsUsed() const { return used; }
	bool Compare(WORD lcn) const { return channelNumber == lcn; }
	WORD GetPort() const { return port; }
	WORD GetChannelNumber() const { return channelNumber; }
	void SetChannelNumber(WORD cn) { channelNumber = cn; }

	virtual bool SetDestination(H245_OpenLogicalChannelAck &, H245Handler *) = 0;
	virtual void StartReading(ProxyHandleThread *) = 0;

protected:
	WORD channelNumber;
	WORD port;
	bool used;
};

class RTPLogicalChannel : public LogicalChannel {
public:
	RTPLogicalChannel(WORD, bool);
	RTPLogicalChannel(RTPLogicalChannel *, WORD, bool);
	virtual ~RTPLogicalChannel();

	void SetSource(const H245_UnicastAddress_iPAddress &);
	void HandleMediaChannel(H245_UnicastAddress_iPAddress *, H245_UnicastAddress_iPAddress *, const PIPSocket::Address &, bool);
	bool OnLogicalChannelParameters(H245_H2250LogicalChannelParameters &, const PIPSocket::Address &, bool);

	// override from class LogicalChannel
	virtual bool SetDestination(H245_OpenLogicalChannelAck &, H245Handler *);
	virtual void StartReading(ProxyHandleThread *);
	bool IsAttached() const { return (peer != 0); }
	void OnHandlerSwapped(bool);

	class NoPortAvailable {};

private:
	void SetNAT(bool);

	bool reversed;
	RTPLogicalChannel *peer;
	UDPProxySocket *rtp, *rtcp;
	PIPSocket::Address SrcIP;
	WORD SrcPort;

	static WORD GetPortNumber();
};

class T120LogicalChannel : public LogicalChannel {
public:
	T120LogicalChannel(WORD);
	virtual ~T120LogicalChannel();

	// override from class LogicalChannel
	virtual bool SetDestination(H245_OpenLogicalChannelAck &, H245Handler *);
	virtual void StartReading(ProxyHandleThread *);

	void AcceptCall();
	bool OnSeparateStack(H245_NetworkAccessParameters &, H245Handler *);

private:
	class T120Listener : public MyPThread {
	public:
		PCLASSINFO ( T120Listener, MyPThread )

		T120Listener(T120LogicalChannel *lc) : t120lc(lc) { Resume(); }
		// override from class MyPThread
		virtual void Exec() { t120lc->AcceptCall(); }

	private:
		T120LogicalChannel *t120lc;
	};

	static void delete_s(T120ProxySocket *s) { s->SetDeletable(); }

	std::list<T120ProxySocket *> sockets;
	T120Listener *listenThread;
	PTCPSocket listener;
	ProxyHandleThread *handler;
	PIPSocket::Address peerAddr;
	WORD peerPort;
};

class NATHandler {
public:
	NATHandler(const PIPSocket::Address & remote) : remoteAddr(remote) {}

	void TranslateH245Address(H225_TransportAddress &);
	bool HandleOpenLogicalChannel(H245_OpenLogicalChannel &);
	bool HandleOpenLogicalChannelAck(H245_OpenLogicalChannelAck &);

private:
	bool SetAddress(H245_UnicastAddress_iPAddress *);
	PIPSocket::Address remoteAddr;
};

// class CallSignalSocket
CallSignalSocket::CallSignalSocket() : TCPProxySocket("Q931s")
{
	InternalInit();
	localAddr = peerAddr = INADDR_ANY;
	m_h245Tunneling = true;
}

/*
CallSignalSocket::CallSignalSocket(CallSignalSocket *socket) : TCPProxySocket("Q931d", socket)
{
	InternalInit();
	remote = socket;
	m_call = socket->m_call;
}
*/

CallSignalSocket::CallSignalSocket(CallSignalSocket *socket, WORD port) : TCPProxySocket("Q931d", socket, port)
{
	InternalInit();
	SetRemote(socket);
}

void CallSignalSocket::InternalInit()
{
	m_h245handler = 0;
	m_h245socket = 0;
	m_isnatsocket = false;
	m_lastQ931 = new Q931;
	m_setupUUIE = 0;
	m_crv = 0;
}

void CallSignalSocket::SetRemote(CallSignalSocket *socket)
{
	remote = socket;
	m_call = socket->m_call;
	m_call->SetSocket(socket, this);
	m_crv = (WORD)(socket->m_crv & 0x7fffu);
	m_h245Tunneling = socket->m_h245Tunneling;
	socket->GetPeerAddress(peerAddr, peerPort);
	localAddr = RasThread->GetLocalAddress(peerAddr);
	SetHandler(socket->GetHandler());
	ProxySocket::SetName(socket->peerAddr, GetPort());

	Address calling = INADDR_ANY, called = INADDR_ANY;
	int type = m_call->GetNATType(calling, called);
	if (type & CallRec::calledParty)
		socket->peerAddr = called;

	if (type == CallRec::both && calling == called)
		if (!Toolkit::AsBool(GkConfig()->GetString(ProxySection, "ProxyForSameNAT", "0")))
			return;

	// enable proxy if required, no matter whether H.245 routed
	if (Toolkit::Instance()->ProxyRequired(peerAddr, socket->peerAddr) ||
		((type != CallRec::none) && Toolkit::AsBool(GkConfig()->GetString(ProxySection, "ProxyForNAT", "1")))) {
		H245ProxyHandler *proxyhandler = new H245ProxyHandler(socket, socket->localAddr, calling);
		socket->m_h245handler = proxyhandler;
		m_h245handler = new H245ProxyHandler(this, localAddr, called, proxyhandler);
		PTRACE(3, "GK\tCall " << m_call->GetCallNumber() << " proxy enabled");
	} else if (m_call->IsH245Routed()) {
		socket->m_h245handler = new H245Handler(socket->localAddr, calling);
		m_h245handler = new H245Handler(localAddr, called);
	}
}

CallSignalSocket::~CallSignalSocket()
{
	if (m_h245socket) {
		if (CallSignalSocket *ret = dynamic_cast<CallSignalSocket *>(remote)) {
			if (!m_h245handler->IsSessionEnded() && ret->m_h245socket)
				ret->m_h245socket->SendEndSessionCommand();
			if (!ret->m_h245handler->IsSessionEnded())
				m_h245socket->SendEndSessionCommand();
		}
		m_h245socket->OnSignalingChannelClosed();
		m_h245socket->EndSession();
		m_h245socket->SetDeletable();
	}
	delete m_h245handler;
	delete m_lastQ931;
	delete m_setupUUIE;
}

namespace { // anonymous namespace
#if PTRACING
void PrintQ931(int tlevel, const PString & msg, const Q931 *q931, const H225_H323_UserInformation *uuie)
{
	PStringStream pstrm;
	pstrm << "Q931\t" << msg << " {\n  q931pdu = " << setprecision(2) << *q931;
	if (uuie)
		pstrm << "\n  h225pdu = " << setprecision(2) << *uuie;
	pstrm << "\n}";
	PTRACE(tlevel, pstrm);
}
#else
inline void PrintQ931(int, const PString &, const Q931 *, const H225_H323_UserInformation *)
{
	// nothing to do
}
#endif

bool GetUUIE(const Q931 & q931, H225_H323_UserInformation & uuie, const PString & name)
{
	if (q931.HasIE(Q931::UserUserIE)) {
		PPER_Stream strm(q931.GetIE(Q931::UserUserIE));
		if (uuie.Decode(strm))
			return true;
		PTRACE(3, "Q931\t" << name << " ERROR DECODING UUIE!");
	}
	return false;
}

void SetUUIE(Q931 & q931, const H225_H323_UserInformation & uuie)
{
	PPER_Stream strm;
	uuie.Encode(strm);
	strm.CompleteEncoding();
	q931.SetIE(Q931::UserUserIE, strm);
}

} // end of anonymous namespace

ProxySocket::Result CallSignalSocket::ReceiveData()
{
	if (!ReadTPKT())
		return NoData;

	Q931* q931pdu = new Q931();
	
	if (!q931pdu->Decode(buffer)) {
		PTRACE(2, "Q931\t" << Name() << " ERROR DECODING Q.931!");
		delete q931pdu;
		return Error;
	}

	bool changed = false;
	PTRACE(3, Type() << "\tReceived: " << q931pdu->GetMessageTypeName() << " CRV=" << q931pdu->GetCallReference() << " from " << Name());

	m_result = Forwarding;

	H225_H323_UserInformation signal, *psignal = 0;
	if (GetUUIE(*q931pdu, signal, Name())) {
		H225_H323_UU_PDU & pdu = signal.m_h323_uu_pdu;
		H225_H323_UU_PDU_h323_message_body & body = pdu.m_h323_message_body;

		PrintQ931(4, "Received:", q931pdu, psignal = &signal);

		if( remote && body.GetTag() == H225_H323_UU_PDU_h323_message_body::e_setup ) {
			const WORD newcrv = (WORD)q931pdu->GetCallReference();
			if( m_crv && (newcrv == (m_crv & 0x7fffu)) ) {
				PTRACE(3, "Q931\tWarning: duplicate Setup - ignored!");
			} else {
				PTRACE(4, "Q931\tMultiple calls over single signalling channel not supported - new connection needed");
				
				Q931 releasePDU;
				H225_H323_UserInformation userInfo;
				H225_H323_UU_PDU_h323_message_body& msgBody = userInfo.m_h323_uu_pdu.m_h323_message_body;
				msgBody.SetTag(H225_H323_UU_PDU_h323_message_body::e_releaseComplete);
				H225_ReleaseComplete_UUIE& uuie = msgBody;
				uuie.m_protocolIdentifier.SetValue(H225_ProtocolID);
				uuie.IncludeOptionalField(H225_ReleaseComplete_UUIE::e_reason);
				uuie.m_reason.SetTag(H225_ReleaseCompleteReason::e_newConnectionNeeded);
				if (((H225_Setup_UUIE&)body).HasOptionalField(H225_Setup_UUIE::e_callIdentifier))
					uuie.m_callIdentifier = ((H225_Setup_UUIE&)body).m_callIdentifier;
				releasePDU.BuildReleaseComplete(newcrv, TRUE);
				SetUUIE(releasePDU, userInfo);
				PrintQ931(5, "Send to " + Name(), &releasePDU, &userInfo);
				
				PBYTEArray buf;
				if( releasePDU.Encode(buf) )
					TransmitData(buf);
				else
					PTRACE(3,"Q931\tFailed to encode message "<<releasePDU);
			}
			delete q931pdu;
			return NoData;
		} else {
			delete m_lastQ931;
			m_lastQ931 = q931pdu;
		}
		
		if (m_h245Tunneling)
			m_h245Tunneling = (pdu.HasOptionalField(H225_H323_UU_PDU::e_h245Tunneling) && pdu.m_h245Tunneling.GetValue());
			
		switch (body.GetTag())
		{
		case H225_H323_UU_PDU_h323_message_body::e_setup:
			m_crv = (WORD)(m_lastQ931->GetCallReference() | 0x8000u);
			m_setupUUIE = new H225_H323_UserInformation(signal);
			changed = OnSetup(body);
			break;
		case H225_H323_UU_PDU_h323_message_body::e_callProceeding:
			changed = OnCallProceeding(body);
			break;
		case H225_H323_UU_PDU_h323_message_body::e_connect:
			changed = OnConnect(body);
			break;
		case H225_H323_UU_PDU_h323_message_body::e_alerting:
			changed = OnAlerting(body);
			break;
		case H225_H323_UU_PDU_h323_message_body::e_information:
			changed = OnInformation(body);
			break;
		case H225_H323_UU_PDU_h323_message_body::e_releaseComplete:
			changed = OnReleaseComplete(body);
			break;
		case H225_H323_UU_PDU_h323_message_body::e_facility:
			changed = OnFacility(body);
			break;
		case H225_H323_UU_PDU_h323_message_body::e_progress:
			changed = OnProgress(body);
			break;
		case H225_H323_UU_PDU_h323_message_body::e_empty:
			changed = OnEmpty(body);
			break;
		case H225_H323_UU_PDU_h323_message_body::e_status:
			changed = OnStatus(body);
			break;
		case H225_H323_UU_PDU_h323_message_body::e_statusInquiry:
			changed = OnStatusInquiry(body);
			break;
		case H225_H323_UU_PDU_h323_message_body::e_setupAcknowledge:
			changed = OnSetupAcknowledge(body);
			break;
		case H225_H323_UU_PDU_h323_message_body::e_notify:
			changed = OnNotify(body);
			break;
		default:
			PTRACE(4, "Q931\t" << Name() << " UNKNOWN Q.931");
			break;
		}
/*
		Buggy routine. Most probably it rewrites useful vendor info into trashes
		and crashes the gk.
		if (pdu.HasOptionalField(H225_H323_UU_PDU::e_nonStandardControl)) {
			for (PINDEX n = 0; n < pdu.m_nonStandardControl.GetSize(); ++n)
				if (OnNonStandardData(pdu.m_nonStandardControl[n].m_data))
					changed = true;
			PTRACE(5, "Q931\t" << Name() << " Rewriting nonStandardControl");
		}
*/
		if (pdu.HasOptionalField(H225_H323_UU_PDU::e_h245Control) && m_h245handler)
			if (OnTunneledH245(pdu.m_h245Control))
				changed = true;

		if (changed)
			SetUUIE(*m_lastQ931, signal);
	} else { // not have UUIE
		delete m_lastQ931;
		m_lastQ931 = q931pdu;
		PrintQ931(4, "Received:", m_lastQ931, 0);
	}
/*
   Note: Openh323 1.7.9 or later required.
   The older version has an out of memory bug in Q931::GetCalledPartyNumber.
*/
	if (m_lastQ931->HasIE(Q931::CalledPartyNumberIE)) {
		unsigned plan, type;
		PString calledNumber;
		if (m_lastQ931->GetCalledPartyNumber(calledNumber, &plan, &type) &&
		    Toolkit::Instance()->RewritePString(calledNumber)) {
			m_lastQ931->SetCalledPartyNumber(calledNumber, plan, type);
			changed = true;
		}
	}

	if (m_lastQ931->HasIE(Q931::DisplayIE)) {
		PString display = GkConfig()->GetString(RoutedSection, "ScreenDisplayIE", "");
		if (!display) {
			m_lastQ931->SetDisplayName(display);
			changed = true;
		}
	}
	if (m_lastQ931->HasIE(Q931::CallingPartyNumberIE)) {
		PString newnumber = GkConfig()->GetString(RoutedSection, "ScreenCallingPartyNumberIE", "");
		if (!newnumber) {
			unsigned plan, type;
			PString oldnumber;
			m_lastQ931->GetCallingPartyNumber(oldnumber, &plan, &type);
			m_lastQ931->SetCallingPartyNumber(newnumber, plan, type);
			changed = true;
		}
	}

	if (changed) {
		m_lastQ931->Encode(buffer);
#if PTRACING
		if (remote)
			PrintQ931(5, "Send to " + remote->Name(), m_lastQ931, psignal);
#endif
	}

	switch (m_lastQ931->GetMessageType())
	{
		case Q931::SetupMsg:
			if (m_result == Error) {
				EndSession();
				SetDeletable();
			} else if (IsConnected()) // remote is NAT socket
				MarkBlocked(false);
			break;
		/* why don't forward Status? - by cwhuang 16.04.2002
		case Q931::StatusMsg:
			// don't forward status messages mm-30.04.2001
			m_result = NoData;
		*/
		case Q931::InformationMsg:
			OnInformationMsg(*m_lastQ931);
			break;
		case Q931::ReleaseCompleteMsg:
			CallTable::Instance()->RemoveCall(m_call);
			m_result = Closing;
			break;
		default:
			break;
	}
	return m_result;
}

void CallSignalSocket::BuildReleasePDU(Q931 & ReleasePDU, const H225_CallTerminationCause *cause) const
{
	H225_H323_UserInformation signal;
	H225_H323_UU_PDU_h323_message_body & body = signal.m_h323_uu_pdu.m_h323_message_body;
	body.SetTag(H225_H323_UU_PDU_h323_message_body::e_releaseComplete);
	H225_ReleaseComplete_UUIE & uuie = body;
	uuie.m_protocolIdentifier.SetValue(H225_ProtocolID);
	if (m_call) {
		uuie.IncludeOptionalField(H225_ReleaseComplete_UUIE::e_callIdentifier);
		uuie.m_callIdentifier = m_call->GetCallIdentifier();
	}
	if (cause) {
		if (cause->GetTag() == H225_CallTerminationCause::e_releaseCompleteReason) {
			uuie.IncludeOptionalField(H225_ReleaseComplete_UUIE::e_reason);
			uuie.m_reason = *cause;
		} else { // H225_CallTerminationCause::e_releaseCompleteCauseIE
			PPER_Stream strm;
			cause->Encode(strm);
			strm.CompleteEncoding();
			ReleasePDU.SetIE(Q931::CauseIE, strm);
		}
	}
	ReleasePDU.BuildReleaseComplete(m_crv & 0x7fffu, m_crv & 0x8000u);
	SetUUIE(ReleasePDU, signal);

	PrintQ931(5, "Send to " + Name(), &ReleasePDU, &signal);
}

void CallSignalSocket::SendReleaseComplete(const H225_CallTerminationCause *cause)
{
	if (IsOpen()) {
		Q931 ReleasePDU;
		BuildReleasePDU(ReleasePDU, cause);
		PBYTEArray buf;
		ReleasePDU.Encode(buf);
		TransmitData(buf);
	}
}

void CallSignalSocket::SendReleaseComplete(H225_ReleaseCompleteReason::Choices reason)
{
	H225_CallTerminationCause cause;
	cause.SetTag(H225_CallTerminationCause::e_releaseCompleteReason);
	H225_ReleaseCompleteReason & releaseReason = cause;
	releaseReason.SetTag(reason);
	SendReleaseComplete(&cause);
}

void CallSignalSocket::BuildFacilityPDU(Q931 & FacilityPDU, int reason, const PObject *parm)
{
	H225_H323_UserInformation signal;
	H225_H323_UU_PDU_h323_message_body & body = signal.m_h323_uu_pdu.m_h323_message_body;
	body.SetTag(H225_H323_UU_PDU_h323_message_body::e_facility);
	H225_Facility_UUIE & uuie = body;
	// Don't set protocolID intentionally so the remote
	// can determine whether this is a message generate by GnuGK
	// uuie.m_protocolIdentifier.SetValue(H225_ProtocolID);
	if (m_call) {
		uuie.IncludeOptionalField(H225_Facility_UUIE::e_conferenceID);
		uuie.m_conferenceID = m_call->GetConferenceIdentifier();
		uuie.IncludeOptionalField(H225_Facility_UUIE::e_callIdentifier);
		uuie.m_callIdentifier = m_call->GetCallIdentifier();
	}
	uuie.m_reason.SetTag(reason);
	switch (reason)
	{
		case H225_FacilityReason::e_startH245:
			uuie.IncludeOptionalField(H225_Facility_UUIE::e_h245Address);
			if (CallSignalSocket *ret = dynamic_cast<CallSignalSocket *>(remote))
				uuie.m_h245Address = m_h245socket->GetH245Address(ret->localAddr);
			else
				PTRACE(2, "Warning: " << Name() << " has no remote party?");
			break;

		case H225_FacilityReason::e_callForwarded:
			uuie.m_protocolIdentifier.SetValue(H225_ProtocolID);
			if (const H225_TransportAddress *addr = dynamic_cast<const H225_TransportAddress *>(parm)) {
				uuie.IncludeOptionalField(H225_Facility_UUIE::e_alternativeAddress);
				uuie.m_alternativeAddress = *addr;
			} else if (const PString *dest = dynamic_cast<const PString *>(parm)) {
				uuie.IncludeOptionalField(H225_Facility_UUIE::e_alternativeAliasAddress);
				uuie.m_alternativeAliasAddress.SetSize(1);
				H323SetAliasAddress(*dest, uuie.m_alternativeAliasAddress[0]);
			}
			if (m_call) {
				uuie.IncludeOptionalField(H225_Facility_UUIE::e_callIdentifier);
				uuie.m_callIdentifier = m_call->GetCallIdentifier();
			}
			break;
	}

	FacilityPDU.BuildFacility(m_crv & 0x7fffu, m_crv & 0x8000u);
	SetUUIE(FacilityPDU, signal);

	PrintQ931(5, "Send to " + Name(), &FacilityPDU, &signal);
}

bool CallSignalSocket::EndSession()
{
	SendReleaseComplete();

	return TCPProxySocket::EndSession();
}

TCPProxySocket *CallSignalSocket::ConnectTo()
{
	return (m_call && m_call->IsForwarded()) ? ForwardCall() : InternalConnectTo();
}

TCPProxySocket *CallSignalSocket::InternalConnectTo()
{
	if (remote->Connect(peerAddr)) {
		PTRACE(3, "Q931(" << getpid() << ") Connect to " << remote->Name() << " successful");
		SetConnected(true);
		remote->SetConnected(true);
		ForwardData();
	} else {
		PTRACE(3, "Q931\t" << peerAddr << ':' << peerPort << " DIDN'T ACCEPT THE CALL");
		if( m_call )
			m_call->SetSocket(this,NULL);
		delete remote;
		remote = NULL;
		SendReleaseComplete(H225_ReleaseCompleteReason::e_unreachableDestination);
		MarkBlocked(true);
		CallTable::Instance()->RemoveCall(m_call);
		MarkBlocked(false);
	}
	return remote;
}

BOOL CallSignalSocket::Connect(const Address & addr)
{
        PIPSocket::Address GKHome = RasThread->GetGKHome();
	return TCPProxySocket::Connect(GKHome, Q931PortRange.GetPort(), addr);
}

TCPProxySocket *CallSignalSocket::ForwardCall()
{
	// disconnect from forwarder
	SendReleaseComplete(H225_ReleaseCompleteReason::e_facilityCallDeflection);
	Close();

	CallSignalSocket *ret = dynamic_cast<CallSignalSocket *>(remote);
	if (!ret) {
		PTRACE(2, "Warning: " << Name() << " has no remote party?");
		return 0;
	}
	if (!ret->m_setupUUIE) {
		PTRACE(1, "Logical Error: " << Name() << " no SetupUUIE!");
		return 0;
	}

	ret->MarkBlocked(true);

	Q931 fakeSetup, *Setup = ret->m_lastQ931;
	H225_H323_UserInformation fuuie, suuie = *ret->m_setupUUIE;
	if (Setup->GetMessageType() != Q931::SetupMsg) {
		fakeSetup.BuildSetup(m_crv & 0x7fffu);
		Setup = &fakeSetup;
	}
	H225_Setup_UUIE & SetupUUIE = suuie.m_h323_uu_pdu.m_h323_message_body;

	GetUUIE(*m_lastQ931, fuuie, Name());
	H225_Facility_UUIE & FacilityUUIE = fuuie.m_h323_uu_pdu.m_h323_message_body;
	if (FacilityUUIE.HasOptionalField(H225_Facility_UUIE::e_alternativeAliasAddress)) {
		const H225_ArrayOf_AliasAddress & a = FacilityUUIE.m_alternativeAliasAddress;
		for (PINDEX n = 0; n < a.GetSize(); ++n)
			if (a[n].GetTag() == H225_AliasAddress::e_dialedDigits) {
				Setup->SetCalledPartyNumber(AsString(a[n], FALSE));
				break;
			}
		SetupUUIE.IncludeOptionalField(H225_Setup_UUIE::e_destinationAddress);
		SetupUUIE.m_destinationAddress = a;
	}
	if (Toolkit::AsBool(GkConfig()->GetString(RoutedSection, "ShowForwarderNumber", "0")))
		if (endptr forwarder = m_call->GetForwarder()) {
			const H225_ArrayOf_AliasAddress & a = forwarder->GetAliases();
			for (PINDEX n = 0; n < a.GetSize(); ++n)
				if (a[n].GetTag() == H225_AliasAddress::e_dialedDigits) {
					PString callingNumber(AsString(a[n], FALSE));
					Setup->SetCallingPartyNumber(callingNumber);
					SetupUUIE.IncludeOptionalField(H225_Setup_UUIE::e_sourceAddress);
					SetupUUIE.m_sourceAddress.SetSize(1);
					H323SetAliasAddress(callingNumber, SetupUUIE.m_sourceAddress[0]);
					break;
				}
		}

	// detach from the call
	m_call->SetSocket(0, 0);
	remote = ret->remote = 0;
	delete ret->m_h245handler;
	ret->m_h245handler = 0;

	CallSignalSocket *result = 0;
	if (ret->OnSetup(SetupUUIE)) {
		SetUUIE(*Setup, suuie);
		Setup->Encode(ret->buffer);
		PrintQ931(5, "Forward Setup to " + ret->remote->Name(), Setup, &suuie);
		if (ret->m_result == Forwarding || ret->InternalConnectTo()) {
			result = dynamic_cast<CallSignalSocket *>(ret->remote);
			if (m_h245socket) {
				m_h245socket->SetSigSocket(result);
				result->m_h245socket = m_h245socket;
				m_h245socket = 0;
			}
			if (ret->m_result == Forwarding) {
				ret->ForwardData();
				// avoid the socket be inserted again
				result = 0;
			}
			ret->MarkBlocked(false);
		}
	} else {
		ret->EndSession();
		ret->SetConnected(false);
		CallTable::Instance()->RemoveCall(m_call);
	}

	// let the socket be deletable
	SetConnected(false);
	MarkBlocked(false);
	return result;
}

bool CallSignalSocket::OnSetup(H225_Setup_UUIE & Setup)
{
	const time_t setupTime = time(NULL);

	m_result = Error;

	if (!Setup.HasOptionalField(H225_Setup_UUIE::e_destinationAddress) || (Setup.m_destinationAddress.GetSize() < 1)) {
		unsigned plan, type;
		PString destination;
		if (m_lastQ931->GetCalledPartyNumber(destination, &plan, &type)) {
			// Setup_UUIE doesn't contain any destination information, but Q.931 has CalledPartyNumber
			// We create the destinationAddress according to it
			Setup.IncludeOptionalField(H225_Setup_UUIE::e_destinationAddress);
			Setup.m_destinationAddress.SetSize(1);
			H323SetAliasAddress(destination, Setup.m_destinationAddress[0]);
		}
	}
	if (Setup.HasOptionalField(H225_Setup_UUIE::e_destinationAddress))
		Toolkit::Instance()->RewriteE164(Setup.m_destinationAddress);

#if PTRACING
	PString callid;
#endif
	if (Setup.HasOptionalField(H225_Setup_UUIE::e_callIdentifier)) {
		m_call = CallTable::Instance()->FindCallRec(Setup.m_callIdentifier);
#if PTRACING
		callid = AsString(Setup.m_callIdentifier.m_guid);
#endif
	} else { // try CallReferenceValue
		PTRACE(3, "Q931\tSetup_UUIE doesn't contain CallIdentifier!");
		H225_CallIdentifier callIdentifier; // empty callIdentifier
		H225_CallReferenceValue crv;
		crv.SetValue(m_crv & 0x7fffu);
		m_call = CallTable::Instance()->FindCallRec(crv);
#if PTRACING
		callid = AsString(callIdentifier.m_guid);
#endif
	}
	GkClient *gkClient = RasThread->GetGkClient();
	Address fromIP;
	WORD fromPort = 0;
	GetPeerAddress(fromIP,fromPort);
	if (m_call) {
		// set calling station IP:PORT, if not yet set
		if( !m_call->HasSrcSignalAddr() )
			if( Setup.HasOptionalField(H225_Setup_UUIE::e_sourceCallSignalAddress) )
				m_call->SetSrcSignalAddr(Setup.m_sourceCallSignalAddress);
			else
				m_call->SetSrcSignalAddr(fromIP,fromPort);
				
#ifdef HAS_ACCT
		if( m_call->GetCalledStationId().IsEmpty() )
			if( Setup.HasOptionalField(H225_Setup_UUIE::e_destinationAddress) 
				&& (Setup.m_destinationAddress.GetSize() > 0) )
			{
				const PString calledPartyNumber = GetBestAliasAddressString(
					Setup.m_destinationAddress,
					H225_AliasAddress::e_dialedDigits,
					H225_AliasAddress::e_partyNumber,
					H225_AliasAddress::e_h323_ID
					);
				if( !calledPartyNumber.IsEmpty() )
					m_call->SetCalledStationId(calledPartyNumber);
			}
			
		if (m_call->GetCallingStationId().IsEmpty()
			&& m_lastQ931->HasIE(Q931::CallingPartyNumberIE)) {
			PString cli;
			if (m_lastQ931->GetCallingPartyNumber(cli)) {
				if (Toolkit::AsBool(GkConfig()->GetString("Accounting", "AlwaysUseCLID", "0")))
					m_call->SetCallingStationId(cli);
				else {
					endptr callingEP = m_call->GetCallingParty();
					if (callingEP) {
						H225_ArrayOf_AliasAddress cliAliases;
						cliAliases.SetSize(1);
						H323SetAliasAddress(cli,cliAliases[0]);
						if (callingEP->CompareAlias(&cliAliases))
							m_call->SetCallingStationId(cli);
						else if (callingEP->IsGateway() 
							&& dynamic_cast<GatewayRec*>(callingEP.operator->())->PrefixMatch(cliAliases) > 0)
							m_call->SetCallingStationId(cli);
					} else if (Setup.HasOptionalField(H225_Setup_UUIE::e_sourceAddress)) {
						for (PINDEX i = 0; i < Setup.m_sourceAddress.GetSize(); i++)
							if (H323GetAliasAddressString(Setup.m_sourceAddress[i]) == cli) {
								m_call->SetCallingStationId(cli);
								break;
							}
					}
				}
			}
		}
		
#endif
		m_call->SetSetupTime(setupTime);
		
		if (m_call->IsSocketAttached()) {
			PTRACE(2, "Q931\tWarning: socket already attached for callid " << callid);
			return false;
		} else if (m_call->IsRegistered() && !m_call->IsForwarded()) {
			if (gkClient->CheckGKIP(fromIP)) {
				// looped call
				PTRACE(2, "Q931\tWarning: a registered call from my GK(" << Name() << ')');
				return false;
			}
			gkClient->RewriteE164(*m_lastQ931, Setup, true);
		}
		/*
		if (endptr ep = CallTable::Instance()->IsForwardedCall(m_call)) {
			// change the calling party number for the forwarded call
			const H225_ArrayOf_AliasAddress & a = ep->GetAliases();
			for (PINDEX n = 0; n < a.GetSize(); ++n)
				if (a[n].GetTag() == H225_AliasAddress::e_dialedDigits) {
					PString callingNumber(AsString(a[n], FALSE));
					//m_lastQ931->SetDisplayName(callingNumber);
					m_lastQ931->SetCallingPartyNumber(callingNumber);
					if (Setup.HasOptionalField(H225_Setup_UUIE::e_sourceAddress)) {
						Setup.m_sourceAddress.SetSize(1);
						H323SetAliasAddress(callingNumber, Setup.m_sourceAddress[0]);
					}
					break;
				}
			// billing for the transfering party
			m_call->SetForwarded(ep);
		}
		*/
		{
//			already locked by proxy handler
//			ReadLock cfgLock(ConfigReloadMutex);
		
			GkAuthenticatorList* authList = RasThread->GetAuthList();
			if( authList )
			{
				unsigned reason = 0;
				long limit = -1;
				if( !authList->CheckSig(*m_lastQ931,Setup,m_call,reason,limit) )
				{
					PTRACE(3,"GKAUTH\tH.225 Setup authentication failed: "<<reason);
					return false;
				}
			
				if( limit > 0 )
					m_call->SetDurationLimit( limit );
			}

#ifdef HAS_ACCT
			GkAcctLoggers* acct = Toolkit::Instance()->GetAcctList();
			if( acct != NULL )
				if( m_call->GetAcctEventsLogged() & GkAcctLogger::AcctStart )
					PTRACE(3,"GKACCT\tAcct-Start already logged for call no "
						<<m_call->GetCallNumber()
						);
				else {
					m_call->SetAcctEventsLogged(GkAcctLogger::AcctStart);
					if( !acct->LogAcctEvent( GkAcctLogger::AcctStart, m_call ) )
					{
						PTRACE(3,"GKACCT\tAcctStart event log failed for call no. "
							<<m_call->GetCallNumber()
							);
						return false;
					}
				}
#endif
		}
	} else {
		bool fromParent;
		if (!RasThread->AcceptUnregisteredCalls(fromIP, fromParent)) {
			PTRACE(3, "Q931\tNo CallRec found in CallTable for callid " << callid);
			return false;
		}
		if (fromParent)
			gkClient->RewriteE164(*m_lastQ931, Setup, false);

		endptr called;

		if (Setup.HasOptionalField(H225_Setup_UUIE::e_destCallSignalAddress)) {
			PIPSocket::Address destAddr;
			WORD destPort = 0;
			if (GetIPAndPortFromTransportAddr(Setup.m_destCallSignalAddress, destAddr, destPort)
				&& destAddr.IsValid() && destPort != 0) {
				called = RegistrationTable::Instance()->FindBySignalAdr(Setup.m_destCallSignalAddress);
				// if the destination signalling address is not the gatekeeper
				// signalling address, add this address as an outer zone endpoint
				if ((!called) && Toolkit::AsBool(GkConfig()->GetString("RasSrv::ARQFeatures", "CallUnregisteredEndpoints", "1"))) {
					PIPSocket::Address _localAddr(0);
					WORD _localPort = 0;
					GetLocalAddress(_localAddr, _localPort);
					if (_localPort != 0 && _localAddr.IsValid())
						if (destPort != _localPort || destAddr != _localAddr)
							called = RegistrationTable::Instance()->InsertRec(Setup);
				}
			} else
				PTRACE(3, "Q931\tInvalid destination address found in the Q.931 Setup, ignored: "
					<< Setup.m_destCallSignalAddress
					);
		}
#ifdef HAS_ACCT
		PString calledPartyNumber;
		PString callingPartyNumber;
				
		if( m_lastQ931->HasIE(Q931::CallingPartyNumberIE) 
			&& Setup.HasOptionalField(H225_Setup_UUIE::e_sourceAddress) ) {
			PString cli;
			if( m_lastQ931->GetCallingPartyNumber(cli) ) {
				for( PINDEX i = 0; i < Setup.m_sourceAddress.GetSize(); i++ )
					if( H323GetAliasAddressString(Setup.m_sourceAddress[i]) == cli ) {
						callingPartyNumber = cli;
						break;
					}
			}
		}
		
		if( called && Setup.HasOptionalField(H225_Setup_UUIE::e_destinationAddress) ) {
			int matchedalias = -1;
			if( called->MatchAlias(Setup.m_destinationAddress,matchedalias) 
				&& matchedalias >= 0 && matchedalias < Setup.m_destinationAddress.GetSize() )
				calledPartyNumber = H323GetAliasAddressString(
					Setup.m_destinationAddress[matchedalias]
					);
			else if( called->IsGateway() 
				&& dynamic_cast<GatewayRec*>(called.operator->())->PrefixMatch(Setup.m_destinationAddress,matchedalias)
				&& matchedalias >= 0 && matchedalias < Setup.m_destinationAddress.GetSize() )
				calledPartyNumber = H323GetAliasAddressString(
					Setup.m_destinationAddress[matchedalias]
					);
		}
#endif
		if (!called && Setup.HasOptionalField(H225_Setup_UUIE::e_destinationAddress)) {
#ifdef HAS_ACCT
			int matchedalias = -1;
			called = RegistrationTable::Instance()->FindEndpoint(Setup.m_destinationAddress, true, true, &matchedalias);
			if( called && matchedalias >= 0 && matchedalias < Setup.m_destinationAddress.GetSize() )
				calledPartyNumber = H323GetAliasAddressString(
					Setup.m_destinationAddress[matchedalias]
					);
#else
			called = RegistrationTable::Instance()->FindEndpoint(Setup.m_destinationAddress, true);
#endif
			if (!called && Toolkit::AsBool(
				GkConfig()->GetString("RasSrv::ARQFeatures", "ParseEmailAliases", "0"))) {
				H225_TransportAddress destAddr;
				PString destAlias;
				PINDEX aliasIndex;
				if (GetEmailAlias(Setup.m_destinationAddress, destAddr, 
						&destAlias, &aliasIndex)) {
					called = RegistrationTable::Instance()->FindBySignalAdr(destAddr);
					if (called) {
#ifdef HAS_ACCT
						calledPartyNumber = H323GetAliasAddressString(
							Setup.m_destinationAddress[aliasIndex]
							);
#endif
						// remove the domain name part
						H323SetAliasAddress(destAlias, 
							Setup.m_destinationAddress[aliasIndex]
							);
					} else if (Toolkit::AsBool(
						GkConfig()->GetString("RasSrv::ARQFeatures",
							"CallUnregisteredEndpoints", "1"))) { 
						// call to an unregistered endpoint
						Setup.IncludeOptionalField(H225_Setup_UUIE::e_destCallSignalAddress);
						Setup.m_destCallSignalAddress = destAddr;
						called = RegistrationTable::Instance()->InsertRec(Setup);
#ifdef HAS_ACCT
						calledPartyNumber = H323GetAliasAddressString(
							Setup.m_destinationAddress[aliasIndex]
							);
#endif
						// remove the domain name part
						H323SetAliasAddress(destAlias, 
							Setup.m_destinationAddress[aliasIndex]
							);
					}
				}
			}
		}

		if (!called) {
			PTRACE(3, "Q931\tDestination not found for the unregistered call " << callid);
			return false;
		}

		CallRec* call = new CallRec( 
			*m_lastQ931, Setup, setupTime, this, called, 100000, 
			RasThread->IsGKRoutedH245() || (fromParent && RasThread->GetGkClient()->IsNATed())
#ifdef HAS_ACCT
			, callingPartyNumber,
			calledPartyNumber
#endif
			);
		
		m_call = callptr(call);
		CallTable::Instance()->Insert(call);

		long durationLimit = -1;
		
		{
//			already locked by proxy handler
//			ReadLock cfgLock(ConfigReloadMutex);
		
			GkAuthenticatorList* authList = RasThread->GetAuthList();
			if( authList )
			{
				unsigned reason = 0;
				if( !authList->CheckSig(*m_lastQ931,Setup,m_call,reason,durationLimit) )
				{
					PTRACE(3,"GKAUTH\tH.225 Setup authentication failed for call no. "
						<<m_call->GetCallNumber()<<", reason: "<<reason
						);
					CallTable::Instance()->RemoveCall(m_call);
					return false;
				}
			}

#ifdef HAS_ACCT
			GkAcctLoggers* acct = Toolkit::Instance()->GetAcctList();
			if( acct != NULL )
				if( m_call->GetAcctEventsLogged() & GkAcctLogger::AcctStart )
					PTRACE(3,"GKACCT\tAcct-Start already logged for call no "
						<<m_call->GetCallNumber()
						);
				else {
					m_call->SetAcctEventsLogged(GkAcctLogger::AcctStart);
					if( !acct->LogAcctEvent( GkAcctLogger::AcctStart, m_call ) )
					{
						PTRACE(3,"GKACCT\tAcctStart event log failed for call no. "
							<<m_call->GetCallNumber()
							);
						CallTable::Instance()->RemoveCall(m_call);
						return false;
					}
				}
#endif
		}
		
		if( durationLimit > 0 )
			m_call->SetDurationLimit( durationLimit );
		
		if (fromParent) {
			m_call->SetRegistered(true);
			gkClient->SendARQ(Setup, m_crv & 0x7fffu, m_call);
		}
	}

	const H225_TransportAddress *addr = m_call->GetCalledAddress();
	if (!addr || !GetIPAndPortFromTransportAddr(*addr, peerAddr, peerPort)) {
		CallTable::Instance()->RemoveCall(m_call);
		PTRACE(3, "Q931\t" << Name() << " INVALID ADDRESS");
		return false;
	}
	Address calling = INADDR_ANY;
	int type = m_call->GetNATType(calling, peerAddr);

	Setup.IncludeOptionalField(H225_Setup_UUIE::e_sourceCallSignalAddress);
	Setup.m_sourceCallSignalAddress = RasThread->GetCallSignalAddress(peerAddr);
	GetIPFromTransportAddr(Setup.m_sourceCallSignalAddress, localAddr);

	PTRACE(3, "GK\tCall " << m_call->GetCallNumber() << " is NAT type " << type);
	if (type & CallRec::calledParty) {
		if (CallSignalSocket *socket = m_call->GetCalledParty()->GetSocket()) {
			PTRACE(3, "Q931\tUsing NAT socket " << socket->Name());
			// it's dangerous if the remote socket has
			// different handler than this
			// so we move this socket to other handler
			MarkBlocked(true);
			GetHandler()->MoveTo(socket->GetHandler(), this);

			socket->SetRemote(this);
			socket->SetConnected(true);
			SetConnected(true);
			remote = socket;
			m_result = Forwarding;
		}
	}
	if (!remote) {
		remote = new CallSignalSocket(this, peerPort);
		m_result = Connecting;
	}

	// in routed mode the caller may have put the GK address in destCallSignalAddress
	// since it is optional, we just remove it (we could alternativly insert the real destination SignalAdr)
	if (Setup.HasOptionalField(H225_Setup_UUIE::e_destCallSignalAddress))
		Setup.RemoveOptionalField(H225_Setup_UUIE::e_destCallSignalAddress);

	// to compliance to MediaRing VR, we have to setup our H323ID
	PString H323ID = GkConfig()->GetString("H323ID");
	if (!H323ID) {
		PINDEX s = 0;
		if (Setup.HasOptionalField(H225_Setup_UUIE::e_sourceAddress))
			s = Setup.m_sourceAddress.GetSize();
		else
			Setup.IncludeOptionalField(H225_Setup_UUIE::e_sourceAddress);
		Setup.m_sourceAddress.SetSize(s+1);
		H323SetAliasAddress(H323ID, Setup.m_sourceAddress[s], H225_AliasAddress::e_h323_ID);
	} else if (Setup.HasOptionalField(H225_Setup_UUIE::e_sourceAddress)) {
		const PString display = GkConfig()->GetString(RoutedSection, "ScreenSourceAddress", "");
		if (!display) {
			Setup.m_sourceAddress.SetSize(1);
			H323SetAliasAddress(display, Setup.m_sourceAddress[0]);
		}
	}

	HandleH245Address(Setup);
	HandleFastStart(Setup, true);
	
#if H225_PROTOCOL_VERSION >= 4
	if (Setup.HasOptionalField(H225_Setup_UUIE::e_parallelH245Control) && m_h245handler)
		OnTunneledH245(Setup.m_parallelH245Control);
#endif
	return true;
}

bool CallSignalSocket::OnCallProceeding(H225_CallProceeding_UUIE & CallProceeding)
{
	bool changed = HandleH245Address(CallProceeding);
	return HandleFastStart(CallProceeding, false) || changed;
}

bool CallSignalSocket::OnConnect(H225_Connect_UUIE & Connect)
{
	if (m_call) // hmm... it should not be null
		m_call->SetConnected();
	
#ifndef NDEBUG
	if (!Connect.HasOptionalField(H225_Connect_UUIE::e_callIdentifier)) {
		PTRACE(1, "Q931\tConnect_UUIE doesn't contain CallIdentifier!");
	} else if (m_call->GetCallIdentifier() != Connect.m_callIdentifier) {
		PTRACE(1, "Q931\tWarning: CallIdentifier doesn't match?");
	}
#endif
	bool changed = HandleH245Address(Connect);
	return HandleFastStart(Connect, false) || changed;
}

bool CallSignalSocket::OnAlerting(H225_Alerting_UUIE & Alerting)
{
	bool changed = HandleH245Address(Alerting);
	return HandleFastStart(Alerting, false) || changed;
}

bool CallSignalSocket::OnInformation(H225_Information_UUIE &)
{
	return false; // do nothing
}

bool CallSignalSocket::OnReleaseComplete(H225_ReleaseComplete_UUIE & /*ReleaseComplete*/)
{
	if( m_call ) {
		m_call->SetDisconnectTime(time(NULL));
		if( m_lastQ931->HasIE(Q931::CauseIE) )
			m_call->SetDisconnectCause(m_lastQ931->GetCause());
	}
	return false; // do nothing
}

bool CallSignalSocket::OnFacility(H225_Facility_UUIE & Facility)
{
	bool changed = false;
	CallSignalSocket *fsocket = 0;
	switch (Facility.m_reason.GetTag())
	{
		case H225_FacilityReason::e_startH245:
			if (Facility.HasOptionalField(H225_Facility_UUIE::e_h245Address) && Facility.m_protocolIdentifier.GetValue().IsEmpty())
				if (m_h245socket && m_h245socket->Reverting(Facility.m_h245Address))
					m_result = NoData;
			break;
		case H225_FacilityReason::e_callForwarded:
			if (!Toolkit::AsBool(GkConfig()->GetString(RoutedSection, "ForwardOnFacility", "1")))
				break;
			fsocket = this;
		case H225_FacilityReason::e_routeCallToGatekeeper:
		case H225_FacilityReason::e_routeCallToMC:
			// to avoid complicated handling of H.245 channel on forwarding,
			// we only do forward if H.245 channel is not established yet
			if (m_setupUUIE || (m_h245socket && m_h245socket->IsConnected()))
				break;
			if (m_call && CallTable::Instance()->FindCallRec(m_call->GetCallNumber())) {
				endptr forwarded;
				PString altDestInfo;
				if (Facility.HasOptionalField(H225_Facility_UUIE::e_alternativeAddress)) {
					forwarded = RegistrationTable::Instance()->FindBySignalAdr(Facility.m_alternativeAddress);
					altDestInfo = AsDotString(Facility.m_alternativeAddress);
				}
				if (Facility.HasOptionalField(H225_Facility_UUIE::e_alternativeAliasAddress)) {
					// copy it to avoid being rewritten twice
					H225_ArrayOf_AliasAddress aliases = Facility.m_alternativeAliasAddress;
					Toolkit::Instance()->RewriteE164(aliases);
					altDestInfo = AsString(aliases);
					// only transfer to registered endpoint
					if (!forwarded)
						forwarded = RegistrationTable::Instance()->FindEndpoint(aliases, true, false);
				}
				if (forwarded) {
					PString forwarder;
					if (Facility.HasOptionalField(H225_Facility_UUIE::e_featureSet) && Facility.m_featureSet.HasOptionalField(H225_FeatureSet::e_neededFeatures)) {
						// get the forwarder
						H225_ArrayOf_FeatureDescriptor & fd = Facility.m_featureSet.m_neededFeatures;
						if ((fd.GetSize() > 0) && fd[0].HasOptionalField(H225_FeatureDescriptor::e_parameters))
							if (fd[0].m_parameters.GetSize() > 0) {
								H225_EnumeratedParameter & parm = fd[0].m_parameters[0];
								if (parm.HasOptionalField(H225_EnumeratedParameter::e_content))
									if (parm.m_content.GetTag() == H225_Content::e_alias)
										forwarder = AsString(parm.m_content, FALSE) + ":forward";
							}
					}
					m_call->SetForward(fsocket, forwarded, forwarder, altDestInfo);
					m_result = Connecting;
					PTRACE(3, "Q931\tCall " << m_call->GetCallNumber() << " is forwarded to " << altDestInfo << ", EP " << forwarded->GetEndpointIdentifier().GetValue() << (!forwarder ? (" by " + forwarder) : PString()));
					//CallTable::Instance()->AddForwardedCall(m_call);
				}
			}
			break;
	}
	if (m_result != NoData)
		changed = HandleH245Address(Facility);
	return HandleFastStart(Facility, false) || changed;
}

bool CallSignalSocket::OnProgress(H225_Progress_UUIE & Progress)
{
	bool changed = HandleH245Address(Progress);
	return HandleFastStart(Progress, false) || changed;
}

bool CallSignalSocket::OnEmpty(H225_H323_UU_PDU_h323_message_body &)
{
	return false; // do nothing
}

bool CallSignalSocket::OnStatus(H225_Status_UUIE &)
{
	return false; // do nothing
}

bool CallSignalSocket::OnStatusInquiry(H225_StatusInquiry_UUIE &)
{
	return false; // do nothing
}

bool CallSignalSocket::OnSetupAcknowledge(H225_SetupAcknowledge_UUIE &)
{
	return false; // do nothing
}

bool CallSignalSocket::OnNotify(H225_Notify_UUIE &)
{
	return false; // do nothing
}

/*
bool CallSignalSocket::OnNonStandardData(PASN_OctetString & octs)
{
	bool changed = false;
	BYTE buf[5000];
	BYTE *pBuf  = buf;               // write pointer
	BYTE *pOcts = octs.GetPointer(); // read pointer
	BYTE *mOcts = pOcts + octs.GetSize();
	PString *CalledPN;

	while (pOcts < mOcts) {
		BYTE type  = pOcts[0];
		BYTE len   = pOcts[1];
		switch (type) {
		case 0x70: // called party
			CalledPN = new PString( (char*) (&(pOcts[3])), len-1);
			if (Toolkit::Instance()->RewritePString(*CalledPN)) {
				// change
				const char* s = *CalledPN;
				pBuf[0] = type;
				pBuf[1] = strlen(s)+1;
				pBuf[2] = pOcts[2];  // type of number, numbering plan id
				memcpy(&(pBuf[3]), s, strlen(s));
				pBuf += strlen(s)+3;
				changed = true;
			} else {
				// leave unchanged
				memcpy(pBuf, pOcts, (len+2)*sizeof(BYTE));
				pBuf += len+2;  // incr write pointer
			}
			delete CalledPN;
			break;
		case 0x6c: // calling party
		default: // copy through
			memcpy(pBuf, pOcts, (len+2)*sizeof(BYTE));
			pBuf += len+2;  // incr write pointer
		}

		// increment read pointer
		pOcts += len+2;
	}

	// set new value if necessary
	if (changed)
		octs.SetValue(buf, pBuf-buf);
	return changed;
}
*/

bool CallSignalSocket::OnTunneledH245(H225_ArrayOf_PASN_OctetString & h245Control)
{
	bool changed = false;
	for (PINDEX i = 0; i < h245Control.GetSize(); ++i) {
		PPER_Stream strm = h245Control[i].GetValue();
		if (HandleH245Mesg(strm)) {
			h245Control[i].SetValue(strm);
			changed = true;
		}
	}
	return changed;
}

bool CallSignalSocket::OnFastStart(H225_ArrayOf_PASN_OctetString & fastStart, bool fromCaller)
{
	bool changed = false;
	PINDEX sz = fastStart.GetSize();
	for (PINDEX i = 0; i < sz; ++i) {
		PPER_Stream strm = fastStart[i].GetValue();
		H245_OpenLogicalChannel olc;
		if (!olc.Decode(strm)) {
			PTRACE(4, "Q931\t" << Name() << " ERROR DECODING FAST START ELEMENT " << i);
			return false;
		}
		PTRACE(4, "Q931\nfastStart[" << i << "] received: " << setprecision(2) << olc);
		H245Handler::pMem handlefs = (fromCaller) ? &H245Handler::HandleFastStartSetup : &H245Handler::HandleFastStartResponse;
		if ((m_h245handler->*handlefs)(olc)) {
			PPER_Stream wtstrm;
			olc.Encode(wtstrm);
			wtstrm.CompleteEncoding();
			fastStart[i].SetValue(wtstrm);
			changed = true;
			PTRACE(5, "Q931\nfastStart[" << i << "] to send " << setprecision(2) << olc);
		}
	}
	return changed;
}

bool CallSignalSocket::OnInformationMsg(Q931 & q931pdu)
{
	PBYTEArray buf = q931pdu.GetIE(Q931::FacilityIE);
	if (!remote && buf.GetSize() > 0) {
		H225_EndpointIdentifier id;
		PString epid((const char *)buf.GetPointer(), buf.GetSize());
		id = epid;
		PTRACE(3, "Q931\tEPID = " << epid);
		endptr ep = RegistrationTable::Instance()->FindByEndpointId(id);
		buf = q931pdu.GetIE(Q931::CallStateIE);
		if (buf.GetSize() > 0 && buf[0] == Q931::CallState_DisconnectRequest) {
			if (ep) {
				ep->GetSocket();
				SetConnected(false);
			}
			PTRACE(3, "Q931\tClose NAT socket" << Name());
			CloseSocket();
		} else if (ep) {
			m_isnatsocket = true;
			ep->SetSocket(this);
			SetConnected(true); // avoid the socket be deleted
		}
		m_result = NoData;
	}
	return false; // unchanged
}

bool CallSignalSocket::SetH245Address(H225_TransportAddress & h245addr)
{
	if (m_h245Tunneling && Toolkit::AsBool(GkConfig()->GetString(RoutedSection, "RemoveH245AddressOnTunneling", "0")))
		return false;
	if (!m_h245handler) // no H245 routed
		return true;

	CallSignalSocket *ret = dynamic_cast<CallSignalSocket *>(remote);
	if (!ret) {
		PTRACE(2, "Warning: " << Name() << " has no remote party?");
		return false;
	}
	m_h245handler->OnH245Address(h245addr);
	if (m_h245socket) {
		if (m_h245socket->IsConnected()) {
			PTRACE(4, "H245\t" << Name() << " H245 channel already established");
			return false;
		} else {
			if (m_h245socket->SetH245Address(h245addr, localAddr))
				std::swap(m_h245socket, ret->m_h245socket);
			return true;
		}
	}
	m_h245socket = ((2 - bool(m_crv & 0x8000u)) & m_call->GetNATType()) ? new NATH245Socket(this) : new H245Socket(this);
	ret->m_h245socket = new H245Socket(m_h245socket, ret);
	m_h245socket->SetH245Address(h245addr, localAddr);
	GetHandler()->ConnectTo(m_h245socket);
	return true;
}


// class H245Handler
H245Handler::H245Handler(const PIPSocket::Address & local, const PIPSocket::Address & remote)
      : localAddr(local), remoteAddr(remote), isH245ended(false)
{
	hnat = (remoteAddr != INADDR_ANY) ? new NATHandler(remoteAddr) : 0;
}

H245Handler::~H245Handler()
{
	delete hnat;
}

void H245Handler::OnH245Address(H225_TransportAddress & addr)
{
	if (hnat)
		hnat->TranslateH245Address(addr);
}

bool H245Handler::HandleMesg(PPER_Stream & mesg)
{
	H245_MultimediaSystemControlMessage h245msg;
	if (!h245msg.Decode(mesg)) {
		PTRACE(4, "H245\tERROR DECODING H.245");
		return false;
	}
	PTRACE(4, "H245\tReceived: " << setprecision(2) << h245msg);

	bool changed = false;
	switch (h245msg.GetTag())
	{
		case H245_MultimediaSystemControlMessage::e_request:
			changed = HandleRequest(h245msg);
			break;
		case H245_MultimediaSystemControlMessage::e_response:
			changed = HandleResponse(h245msg);
			break;
		case H245_MultimediaSystemControlMessage::e_command:
			changed = HandleCommand(h245msg);
			break;
		case H245_MultimediaSystemControlMessage::e_indication:
			changed = HandleIndication(h245msg);
			break;
		default:
			PTRACE(2, "H245\tUnknown H245 message: " << h245msg.GetTag());
			break;
	}
	if (changed) {
		mesg.BeginEncoding();
		h245msg.Encode(mesg);
		mesg.CompleteEncoding();
		PTRACE(5, "H245\tTo send: " << setprecision(2) << h245msg);
	}

	return changed;
}

bool H245Handler::HandleFastStartSetup(H245_OpenLogicalChannel & olc)
{
	return hnat ? hnat->HandleOpenLogicalChannel(olc) : false;
}

bool H245Handler::HandleFastStartResponse(H245_OpenLogicalChannel & olc)
{
	return hnat ? hnat->HandleOpenLogicalChannel(olc) : false;
}

bool H245Handler::HandleRequest(H245_RequestMessage & Request)
{
	PTRACE(4, "H245\tRequest: " << Request.GetTagName());
	if (hnat && Request.GetTag() == H245_RequestMessage::e_openLogicalChannel)
		return hnat->HandleOpenLogicalChannel(Request);
	else
		return false;
}

bool H245Handler::HandleResponse(H245_ResponseMessage & Response)
{
	PTRACE(4, "H245\tResponse: " << Response.GetTagName());
	if (hnat && Response.GetTag() == H245_ResponseMessage::e_openLogicalChannelAck)
		return hnat->HandleOpenLogicalChannelAck(Response);
	else
		return false;
}

bool H245Handler::HandleIndication(H245_IndicationMessage & Indication)
{
	PTRACE(4, "H245\tIndication: " << Indication.GetTagName());
	return false;
}

bool H245Handler::HandleCommand(H245_CommandMessage & Command)
{
	PTRACE(4, "H245\tCommand: " << Command.GetTagName());
	if (Command.GetTag() == H245_CommandMessage::e_endSessionCommand)
		isH245ended = true;
	return false;
}

// class H245Socket
H245Socket::H245Socket(CallSignalSocket *sig)
      : TCPProxySocket("H245d"), sigSocket(sig), listener(new PTCPSocket)
{
	peerH245Addr = 0;
	listener->Listen(1, H245PortRange.GetPort());
	SetHandler(sig->GetHandler());
}

H245Socket::H245Socket(H245Socket *socket, CallSignalSocket *sig)
      : TCPProxySocket("H245s", socket), sigSocket(sig), listener(0)
{
	peerH245Addr = 0;
	socket->remote = this;
}

H245Socket::~H245Socket()
{
	delete listener;
	delete peerH245Addr;
	if (sigSocket)
		sigSocket->OnH245ChannelClosed();
}

ProxySocket::Result H245Socket::ReceiveData()
{
	if (!ReadTPKT())
		return NoData;

	PPER_Stream strm(buffer);

	if (sigSocket && sigSocket->HandleH245Mesg(strm))
		buffer = strm;

	return Forwarding;
}

bool H245Socket::EndSession()
{
	if (listener)
		listener->Close();
	return TCPProxySocket::EndSession();
}

TCPProxySocket *H245Socket::ConnectTo()
{
	if (remote->Accept(*listener)) {
		if (ConnectRemote()) {
			GetHandler()->Insert(this);
			SetConnected(true);
			remote->SetConnected(true);
			return remote;
		}
	}
	// establish H.245 channel failed, disconnect the call
	if (sigSocket)
		sigSocket->EndSession();
	if (H245Socket *ret = dynamic_cast<H245Socket *>(remote))
		if (ret->sigSocket)
			ret->sigSocket->EndSession();
	GetHandler()->Insert(remote);
	GetHandler()->Insert(this);
	return 0;
}

void H245Socket::SendEndSessionCommand()
{
	if (!IsConnected())
		return;
	// generate EndSessionCommand
	H245_MultimediaSystemControlMessage h245msg;
	h245msg.SetTag(H245_MultimediaSystemControlMessage::e_command);
	H245_CommandMessage & h245cmd = h245msg;
	h245cmd.SetTag(H245_CommandMessage::e_endSessionCommand);
	H245_EndSessionCommand & endcmd = h245cmd;
	endcmd.SetTag(H245_EndSessionCommand::e_disconnect);
	PPER_Stream wtstrm;
	h245msg.Encode(wtstrm);
	wtstrm.CompleteEncoding();
	TransmitData(wtstrm);
	PTRACE(4, "H245\tSend endSessionCommand to " << Name());
}

BOOL H245Socket::Accept(PSocket & socket)
{
	bool result = TCPProxySocket::Accept(socket);
	if (result) {
		PTRACE(3, "H245\tConnected from " << Name());
	} else if (peerH245Addr) {
		result = H245Socket::ConnectRemote();
	}
	return result;
}

bool H245Socket::ConnectRemote()
{
	if (listener)
		listener->Close(); // don't accept other connection
	PIPSocket::Address peerAddr;
	WORD peerPort;
	if (!peerH245Addr || !GetIPAndPortFromTransportAddr(*peerH245Addr, peerAddr, peerPort)) {
		PTRACE(3, "H245\tINVALID ADDRESS");
		return false;
	}
	SetPort(peerPort);
	bool result = Connect(RasThread->GetGKHome(), H245PortRange.GetPort(), peerAddr);
	if (result) {
		PTRACE(3, "H245(" << getpid() << ") Connect to " << Name() << " successful");
	} else {
		PTRACE(3, "H245\t" << peerAddr << ':' << peerPort << " DIDN'T ACCEPT THE CALL");
	}
	return result;
}

H225_TransportAddress H245Socket::GetH245Address(const Address & myip)
{
	return SocketToH225TransportAddr(myip, (WORD)(listener ? listener->GetPort() : 0));
}

bool H245Socket::SetH245Address(H225_TransportAddress & h245addr, const Address & myip)
{
	bool swapped;
	H245Socket *socket;
	if (listener) {
		socket = this;
		swapped = false;
	} else {
		socket = dynamic_cast<H245Socket *>(remote);
		swapped = true;
		std::swap(this->sigSocket, socket->sigSocket);
	}
	if (socket->peerH245Addr)
		*socket->peerH245Addr = h245addr;
	else
		socket->peerH245Addr = new H225_TransportAddress(h245addr);
	h245addr = SocketToH225TransportAddr(myip, socket->listener->GetPort());
	PTRACE(3, "H245\tSet h245Address to " << AsDotString(h245addr));
	return swapped;
}

bool H245Socket::Reverting(const H225_TransportAddress & h245addr)
{
	PTRACE(3, "H245\tH.245 Reverting detected");
	PSocket *socket = dynamic_cast<H245Socket *>(remote)->listener;
	if (socket && socket->IsOpen()) {
		peerH245Addr = new H225_TransportAddress(h245addr);
		socket->Close();
		return true;
	}
	return false;
}

// class NATH245Socket
bool NATH245Socket::ConnectRemote()
{
	if (!sigSocket || !listener)
		return false;

	Q931 q931;
	sigSocket->BuildFacilityPDU(q931, H225_FacilityReason::e_startH245);
	q931.Encode(buffer);
	sigSocket->TransmitData(buffer);
	bool result = Accept(*listener);
	PTRACE_IF(3, result, "H245\tChannel established for NAT EP");
	listener->Close();
	return result;
}

namespace { // anonymous namespace

H245_UnicastAddress_iPAddress *GetH245UnicastAddress(H245_TransportAddress & tsap)
{
	if (tsap.GetTag() == H245_TransportAddress::e_unicastAddress) {
		H245_UnicastAddress & uniaddr = tsap;
		if (uniaddr.GetTag() == H245_UnicastAddress::e_iPAddress)
			return &((H245_UnicastAddress_iPAddress &)uniaddr);
	}
	return 0;
}

bool IsSeparateLANStack(const H245_DataType & data)
{
	if (data.GetTag() == H245_DataType::e_data ) {
		const H245_DataApplicationCapability & cap = data;
		if (cap.m_application.GetTag() == H245_DataApplicationCapability_application::e_t120) {
			const H245_DataProtocolCapability & proto_cap = cap.m_application;
			return (proto_cap.GetTag() == H245_DataProtocolCapability::e_separateLANStack);
		}
	}
	return false;
}

bool IsT120Channel(const H245_OpenLogicalChannel & olc)
{
	return  IsSeparateLANStack(olc.m_forwardLogicalChannelParameters.m_dataType) &&
		olc.HasOptionalField(H245_OpenLogicalChannel::e_reverseLogicalChannelParameters) &&
		IsSeparateLANStack(olc.m_reverseLogicalChannelParameters.m_dataType);
}

H245_H2250LogicalChannelParameters *GetLogicalChannelParameters(H245_OpenLogicalChannel & olc, bool & isReverseLC)
{
	if (olc.HasOptionalField(H245_OpenLogicalChannel::e_reverseLogicalChannelParameters)) {
		if (!olc.m_reverseLogicalChannelParameters.HasOptionalField(H245_OpenLogicalChannel_reverseLogicalChannelParameters::e_multiplexParameters))
			return 0;
		H245_OpenLogicalChannel_reverseLogicalChannelParameters_multiplexParameters & params = olc.m_reverseLogicalChannelParameters.m_multiplexParameters;
		isReverseLC = true;
		return (params.GetTag() == H245_OpenLogicalChannel_reverseLogicalChannelParameters_multiplexParameters::e_h2250LogicalChannelParameters) ?  &((H245_H2250LogicalChannelParameters &)params) : 0;
	} else {
		H245_OpenLogicalChannel_forwardLogicalChannelParameters_multiplexParameters & params = olc.m_forwardLogicalChannelParameters.m_multiplexParameters;
		isReverseLC = false;
		return (params.GetTag() == H245_OpenLogicalChannel_forwardLogicalChannelParameters_multiplexParameters::e_h2250LogicalChannelParameters) ?  &((H245_H2250LogicalChannelParameters &)params) : 0;
	}
}

bool GetChannelsFromOLCA(H245_OpenLogicalChannelAck & olca, H245_UnicastAddress_iPAddress * & mediaControlChannel, H245_UnicastAddress_iPAddress * & mediaChannel)
{
	if (!olca.HasOptionalField(H245_OpenLogicalChannelAck::e_forwardMultiplexAckParameters))
		return false;
	H245_OpenLogicalChannelAck_forwardMultiplexAckParameters & ackparams = olca.m_forwardMultiplexAckParameters;
	if (ackparams.GetTag() != H245_OpenLogicalChannelAck_forwardMultiplexAckParameters::e_h2250LogicalChannelAckParameters)
		return false;
	H245_H2250LogicalChannelAckParameters & h225Params = ackparams;
	if (!h225Params.HasOptionalField(H245_H2250LogicalChannelAckParameters::e_mediaControlChannel))
		return false;
	mediaControlChannel = GetH245UnicastAddress(h225Params.m_mediaControlChannel);
	if (!mediaControlChannel)
		return false;
	mediaChannel = h225Params.HasOptionalField(H245_H2250LogicalChannelAckParameters::e_mediaChannel) ? GetH245UnicastAddress(h225Params.m_mediaChannel) : 0;
	return true;
}

inline H245_UnicastAddress_iPAddress & operator<<(H245_UnicastAddress_iPAddress & addr, const PIPSocket::Address & ip)
{
	for (int i = 0; i < 4; ++i)
		addr.m_network[i] = ip[i];
	return addr;
}

inline H245_UnicastAddress_iPAddress & operator<<(H245_UnicastAddress_iPAddress & addr, WORD port)
{
	addr.m_tsapIdentifier = port;
	return addr;
}

inline const H245_UnicastAddress_iPAddress & operator>>(const H245_UnicastAddress_iPAddress & addr, PIPSocket::Address & ip)
{
	ip = PIPSocket::Address(addr.m_network[0], addr.m_network[1], addr.m_network[2], addr.m_network[3]);
	return addr;
}

inline const H245_UnicastAddress_iPAddress & operator>>(const H245_UnicastAddress_iPAddress & addr, WORD & port)
{
	port = (WORD)addr.m_tsapIdentifier;
	return addr;
}

inline bool compare_lc(std::pair<const WORD, RTPLogicalChannel *> p, LogicalChannel *lc)
{
	return p.second == lc;
}

inline void delete_lc(std::pair<const WORD, LogicalChannel *> & p)
{
	delete p.second;
}

inline void delete_rtplc(std::pair<const WORD, RTPLogicalChannel *> & p)
{
	delete p.second;
}

} // end of anonymous namespace


#ifndef IPTOS_PREC_CRITIC_ECP
#define IPTOS_PREC_CRITIC_ECP (5 << 5)
#endif

#ifndef IPTOS_LOWDELAY
#define IPTOS_LOWDELAY 0x10
#endif

// class UDPProxySocket
UDPProxySocket::UDPProxySocket(const char *t) : ProxySocket(this, t)
{
	SetReadTimeout(PTimeInterval(50));
	SetWriteTimeout(PTimeInterval(50));
	fnat = rnat = false;
}

bool UDPProxySocket::Bind(WORD pt)
{
	if (!Listen(0, pt))
		return false;

	// Set the IP Type Of Service field for prioritisation of media UDP packets
#ifdef WIN32
	// Windows MultMedia stuff seems to need greater depth due to enormous
	// latencies in its operation, need to use DirectSound maybe?
	int rtpIpTypeofService = IPTOS_PREC_CRITIC_ECP | IPTOS_LOWDELAY;
#else
	// Don't use IPTOS_PREC_CRITIC_ECP on Unix platforms as then need to be root
	int rtpIpTypeofService = IPTOS_LOWDELAY;
#endif
	if (!ConvertOSError(::setsockopt(os_handle, IPPROTO_IP, IP_TOS, (char *)&rtpIpTypeofService, sizeof(int)))) {
		PTRACE(1, Type() << "\tCould not set TOS field in IP header: " << GetErrorText(PSocket::LastGeneralError));
	}
	return true;
}

bool UDPProxySocket::Bind(const Address& addr, WORD pt)
{
	if (!Listen(addr, 0, pt))
		return false;

	// Set the IP Type Of Service field for prioritisation of media UDP packets
#ifdef WIN32
	// Windows MultMedia stuff seems to need greater depth due to enormous
	// latencies in its operation, need to use DirectSound maybe?
	int rtpIpTypeofService = IPTOS_PREC_CRITIC_ECP | IPTOS_LOWDELAY;
#else
	// Don't use IPTOS_PREC_CRITIC_ECP on Unix platforms as then need to be root
	int rtpIpTypeofService = IPTOS_LOWDELAY;
#endif
	if (!ConvertOSError(::setsockopt(os_handle, IPPROTO_IP, IP_TOS, (char *)&rtpIpTypeofService, sizeof(int)))) {
		PTRACE(1, Type() << "\tCould not set TOS field in IP header: " << GetErrorText(PSocket::LastGeneralError));
	}
	return true;
}

void UDPProxySocket::SetNAT(bool rev)
{
	// if the handler of lc is NATed,
	// the destination of reverse direction should be changed
	(rev ? fnat : rnat) = true;
	PTRACE(5, Type() << "\tfnat=" << fnat << " rnat=" << rnat);
}

void UDPProxySocket::SetForwardDestination(const Address & srcIP, WORD srcPort, const H245_UnicastAddress_iPAddress & addr)
{
	fSrcIP = srcIP, fSrcPort = srcPort;
	addr >> fDestIP >> fDestPort;

	SetName(srcIP, srcPort);
	PTRACE(5, Type() << "\tForward " << Name() << " to " << fDestIP << ':' << fDestPort);
	SetConnected(true);
}

void UDPProxySocket::SetReverseDestination(const Address & srcIP, WORD srcPort, const H245_UnicastAddress_iPAddress & addr)
{
	rSrcIP = srcIP, rSrcPort = srcPort;
	addr >> rDestIP >> rDestPort;

	PTRACE(5, Type() << "\tReverse " << srcIP << ':' << srcPort << " to " << rDestIP << ':' << rDestPort);
	SetConnected(true);
}

ProxySocket::Result UDPProxySocket::ReceiveData()
{
	if (!Read(wbuffer, wbufsize)) {
		ErrorHandler(PSocket::LastReadError);
		return NoData;
	}
	Address fromIP;
	WORD fromPort;
	GetLastReceiveAddress(fromIP, fromPort);
	buflen = (WORD)GetLastReadCount();

	// Workaround: some bad endpoints don't send packets from the specified port
	if ((fromIP == fSrcIP || fromIP == rDestIP) && (fromIP != rSrcIP || fromPort == fSrcPort)) {
		PTRACE(6, Type() << "\tforward " << fromIP << ':' << fromPort << " to " << fDestIP << ':' << fDestPort);
		SetSendAddress(fDestIP, fDestPort);
		if (rnat)
			rDestIP = fromIP, rDestPort = fromPort;
	} else {
		PTRACE(6, Type() << "\tforward " << fromIP << ':' << fromPort << " to " << rDestIP << ':' << rDestPort);
		SetSendAddress(rDestIP, rDestPort);
		if (fnat)
			fDestIP = fromIP, fDestPort = fromPort;
	}
	return Forwarding;
}

bool UDPProxySocket::WriteData(const BYTE *buffer, WORD len)
{
	if (!IsSocketOpen())
		return false;

	if (GetQueueSize() > 0) {
		QueuePacket(buffer, len);
		PTRACE(3, Name() << " socket is busy, " << len << " bytes queued");
		return false;
	}
	
	if (Self()->Write(buffer, len)) {
		PTRACE(6, Name() << ' ' << len << " bytes sent");
		return true;
	}

	const WORD wcount = (WORD)Self()->GetLastWriteCount();
	buffer += wcount;
	len -= wcount;

	PTRACE(4, Name() << " blocked, " << wcount << " bytes written, " << len << " bytes queued");
	PutBackPacket(buffer, len);
	return (wcount > 0) ? false : ErrorHandler(PSocket::LastWriteError);
}

bool UDPProxySocket::FlushData(const BYTE *buf, WORD len)
{
	if (!IsSocketOpen())
		return false;

	if (Self()->Write(buf, len)) {
		PTRACE(6, Name() << ' ' << len << " bytes sent");
		return true;
	}

	const WORD wcount = (WORD)Self()->GetLastWriteCount();
	buf += wcount;
	len -= wcount;

	PutBackPacket(buf, len);

	PTRACE(4, Name() << " blocked, " << wcount << " bytes written, " << len << " bytes queued");
	return (wcount > 0) ? false : ErrorHandler(PSocket::LastWriteError);
}


// class T120ProxySocket
T120ProxySocket::T120ProxySocket(T120ProxySocket *socket, WORD pt)
      : TCPProxySocket("T120", socket, pt)
{
}

bool T120ProxySocket::ForwardData()
{
	return (remote) ? remote->ProxySocket::TransmitData(wbuffer, buflen) : false;
}

TCPProxySocket *T120ProxySocket::ConnectTo()
{
	remote = new T120ProxySocket(this, peerPort);
	if (remote->Connect(RasThread->GetGKHome(), T120PortRange.GetPort(), peerAddr)) {
		PTRACE(3, "T120\tConnect to " << remote->Name() << " successful");
		SetConnected(true);
		remote->SetConnected(true);
	} else {
		PTRACE(3, "T120\t" << peerAddr << ':' << peerPort << " DIDN'T ACCEPT THE CALL");
		delete remote; // would close myself
	}
	return remote;
}

// class RTPLogicalChannel
RTPLogicalChannel::RTPLogicalChannel(WORD flcn, bool nated) : LogicalChannel(flcn), reversed(false), peer(0)
{
	rtp = new UDPProxySocket("RTP");
	rtcp = new UDPProxySocket("RTCP");
	SetNAT(nated);

	const PIPSocket::Address addr = RasThread->GetGKHome();
	// try to get an available port 10 times
	for (int i = 0; i < 10; ++i) {
		port = (WORD)GetPortNumber();
		// try to bind rtp to an even port and rtcp to the next one port
		if ((addr == INADDR_ANY) ? (rtp->Bind(port) && rtcp->Bind(port + 1))
			: (rtp->Bind(addr,port) && rtcp->Bind(addr,port + 1)))
			return;

		PTRACE(3, "RTP\tPort " << port << " not available");
		rtp->Close(), rtcp->Close();
		// try next...
	}
	delete rtp;
	delete rtcp;
	// Oops...
	throw NoPortAvailable();
}

RTPLogicalChannel::RTPLogicalChannel(RTPLogicalChannel *flc, WORD flcn, bool nated)
{
	memcpy(this, flc, sizeof(RTPLogicalChannel)); // bitwise copy :)
	reversed = !flc->reversed;
	peer = flc, flc->peer = this;
	SetChannelNumber(flcn);
	SetNAT(nated);
}

RTPLogicalChannel::~RTPLogicalChannel()
{
	if (peer) {
		peer->peer = 0;
	} else {
		if (used) {
			// the sockets will be deleted by ProxyHandler,
			// so we don't need to delete it here
			// don't close the sockets, or it causes crashing
			rtp->SetDeletable();
			rtcp->SetDeletable();
		} else {
			delete rtp;
			delete rtcp;
		}
	}
	PTRACE(4, "RTP\tDelete logical channel " << channelNumber);
}

void RTPLogicalChannel::SetSource(const H245_UnicastAddress_iPAddress & addr)
{
	addr >> SrcIP >> SrcPort;
	--SrcPort; // get the RTP port
}

void RTPLogicalChannel::HandleMediaChannel(H245_UnicastAddress_iPAddress *mediaControlChannel, H245_UnicastAddress_iPAddress *mediaChannel, const PIPSocket::Address & local, bool rev)
{
	// mediaControlChannel should be non-zero.
	H245_UnicastAddress_iPAddress tmp, tmpmedia, *dest = mediaControlChannel;
	PIPSocket::Address tmpSrcIP = SrcIP;
	WORD tmpSrcPort = SrcPort + 1;
	if (rev) { // from a reverseLogicalChannelParameters
		tmp << tmpSrcIP << tmpSrcPort;
		dest = &tmp;
		*mediaControlChannel >> tmpSrcIP >> tmpSrcPort;
		if (!mediaChannel) {
			tmpmedia = *mediaControlChannel;
			tmpmedia.m_tsapIdentifier = tmpmedia.m_tsapIdentifier - 1;
			mediaChannel = &tmpmedia;
		}
	}
	UDPProxySocket::pMem SetDest = (reversed) ? &UDPProxySocket::SetReverseDestination : &UDPProxySocket::SetForwardDestination;
	(rtcp->*SetDest)(tmpSrcIP, tmpSrcPort, *dest);
	*mediaControlChannel << local << (WORD)(port + 1);

	if (mediaChannel) {
		if (rev)
			tmp.m_tsapIdentifier = tmp.m_tsapIdentifier - 1;
		else
			dest = mediaChannel;
		(rtp->*SetDest)(tmpSrcIP, tmpSrcPort - 1, *dest);
		*mediaChannel << local << port;
	}
}

bool RTPLogicalChannel::OnLogicalChannelParameters(H245_H2250LogicalChannelParameters & h225Params, const PIPSocket::Address & local, bool rev)
{
	if (!h225Params.HasOptionalField(H245_H2250LogicalChannelParameters::e_mediaControlChannel))
		return false;
	H245_UnicastAddress_iPAddress *mediaControlChannel = GetH245UnicastAddress(h225Params.m_mediaControlChannel);
	H245_UnicastAddress_iPAddress *mediaChannel = h225Params.HasOptionalField(H245_H2250LogicalChannelParameters::e_mediaChannel) ? GetH245UnicastAddress(h225Params.m_mediaChannel) : 0;
	HandleMediaChannel(mediaControlChannel, mediaChannel, local, rev);
	return true;
}

bool RTPLogicalChannel::SetDestination(H245_OpenLogicalChannelAck & olca, H245Handler *handler)
{
	H245_UnicastAddress_iPAddress *mediaControlChannel, *mediaChannel;
	if (GetChannelsFromOLCA(olca, mediaControlChannel, mediaChannel)) {
		HandleMediaChannel(mediaControlChannel, mediaChannel, handler->GetLocalAddr(), false);
		return true;
	}
	return false;
}

void RTPLogicalChannel::StartReading(ProxyHandleThread *handler)
{
	if (!used) {
		handler->InsertLC(rtp);
		handler->InsertLC(rtcp);
		used = true;
		if (peer)
			peer->used = true;
	}
}

void RTPLogicalChannel::OnHandlerSwapped(bool nated)
{
	rtp->OnHandlerSwapped();
	rtcp->OnHandlerSwapped();
	SetNAT(nated);
}

void RTPLogicalChannel::SetNAT(bool nated)
{
	if (nated)
		rtp->SetNAT(reversed), rtcp->SetNAT(reversed);
}

WORD RTPLogicalChannel::GetPortNumber()
{
	WORD port = RTPPortRange.GetPort();
	if (port & 1) // make sure it is even
		port = RTPPortRange.GetPort();
	RTPPortRange.GetPort(); // skip one
	return port;
}

// class T120LogicalChannel
T120LogicalChannel::T120LogicalChannel(WORD flcn) : LogicalChannel(flcn)
{
	listener.Listen(1, T120PortRange.GetPort());
	port = listener.GetPort();
	PTRACE(4, "T120\tOpen logical channel " << flcn << " port " << port);
}

T120LogicalChannel::~T120LogicalChannel()
{
	if (used) {
		listener.Close();
		listenThread->Destroy();
		for_each(sockets.begin(), sockets.end(), delete_s);
	}
	PTRACE(4, "T120\tDelete logical channel " << channelNumber);
}

bool T120LogicalChannel::SetDestination(H245_OpenLogicalChannelAck & olca, H245Handler *handler)
{
	return (olca.HasOptionalField(H245_OpenLogicalChannelAck::e_separateStack)) ?
		OnSeparateStack(olca.m_separateStack, handler) : false;
}

void T120LogicalChannel::StartReading(ProxyHandleThread *h)
{
	if (!used) {
		handler = h;
		used = true;
		listenThread = new T120Listener(this);
	}
}

void T120LogicalChannel::AcceptCall()
{
	if (!listener.IsOpen())
		return;

	T120ProxySocket *socket = new T120ProxySocket;
	if (socket->Accept(listener)) {
		PTRACE(4, "T120\tConnected from " << socket->Name());
		socket->SetDestination(peerAddr, peerPort);
		TCPProxySocket *remote = socket->ConnectTo();
		if (remote) {
			handler->InsertLC(socket);
			handler->InsertLC(remote);
			sockets.push_back(socket);
			return;
		}
	}

	PSocket::Errors err = socket->GetErrorCode(PSocket::LastGeneralError);
	PTRACE_IF(3, err != PSocket::Interrupted,
		  "T120\tError: " << PSocket::GetErrorText(err));
	delete socket;
}

bool T120LogicalChannel::OnSeparateStack(H245_NetworkAccessParameters & sepStack, H245Handler *handler)
{
	bool changed = false;
	if (sepStack.m_networkAddress.GetTag() == H245_NetworkAccessParameters_networkAddress::e_localAreaAddress) {
		H245_UnicastAddress_iPAddress *addr = GetH245UnicastAddress(sepStack.m_networkAddress);
		if (addr) {
			*addr >> peerAddr >> peerPort;
			*addr << handler->GetLocalAddr() << port;
			changed = true;
		}
	}
	return changed;
}

// class H245ProxyHandler
H245ProxyHandler::H245ProxyHandler(CallSignalSocket *sig, const PIPSocket::Address & local, const PIPSocket::Address & remote, H245ProxyHandler *pr)
      : H245Handler(local, remote), handler(sig->GetHandler()), peer(pr)
{
	if (peer)
		peer->peer = this;
}

H245ProxyHandler::~H245ProxyHandler()
{
	for_each(logicalChannels.begin(), logicalChannels.end(), delete_lc);
	for_each(fastStartLCs.begin(), fastStartLCs.end(), delete_rtplc);
	if (peer)
		peer->peer = 0;
}

bool H245ProxyHandler::HandleRequest(H245_RequestMessage & Request)
{
	PTRACE(4, "H245\tRequest: " << Request.GetTagName());
	if (peer)
		switch (Request.GetTag())
		{
			case H245_RequestMessage::e_openLogicalChannel:
				return HandleOpenLogicalChannel(Request);
			case H245_RequestMessage::e_closeLogicalChannel:
				return HandleCloseLogicalChannel(Request);
			default:
				break;
		}
	return false;
}

bool H245ProxyHandler::HandleResponse(H245_ResponseMessage & Response)
{
	PTRACE(4, "H245\tResponse: " << Response.GetTagName());
	if (peer)
		switch (Response.GetTag())
		{
			case H245_ResponseMessage::e_openLogicalChannelAck:
				return HandleOpenLogicalChannelAck(Response);
			case H245_ResponseMessage::e_openLogicalChannelReject:
				return HandleOpenLogicalChannelReject(Response);
			default:
				break;
		}
	return false;
}

bool H245ProxyHandler::OnLogicalChannelParameters(H245_H2250LogicalChannelParameters *h225Params, WORD flcn)
{
	if (!h225Params->HasOptionalField(H245_H2250LogicalChannelParameters::e_mediaControlChannel))
		return false;
	H245_UnicastAddress_iPAddress *addr = GetH245UnicastAddress(h225Params->m_mediaControlChannel);
	if (!addr)
		return false;
	RTPLogicalChannel *lc = (flcn) ?
		CreateRTPLogicalChannel((WORD)h225Params->m_sessionID, flcn) :
		CreateFastStartLogicalChannel((WORD)h225Params->m_sessionID);
	if (!lc)
		return false;
	lc->SetSource(*addr);
	*addr << GetLocalAddr() << (WORD)(lc->GetPort() + 1);
	if (h225Params->HasOptionalField(H245_H2250LogicalChannelParameters::e_mediaChannel)) {
		H245_UnicastAddress_iPAddress *addr = GetH245UnicastAddress(h225Params->m_mediaChannel);
		if (addr)
			*addr << GetLocalAddr() << lc->GetPort();
	}
	return true;
}

bool H245ProxyHandler::HandleOpenLogicalChannel(H245_OpenLogicalChannel & olc)
{
	if (hnat)
		hnat->HandleOpenLogicalChannel(olc);
	const WORD flcn = (WORD)olc.m_forwardLogicalChannelNumber;
	if (IsT120Channel(olc)) {
		T120LogicalChannel *lc = CreateT120LogicalChannel(flcn);
		if (olc.HasOptionalField(H245_OpenLogicalChannel::e_separateStack)
			&& lc && lc->OnSeparateStack(olc.m_separateStack, this)) {
			lc->StartReading(handler);
			return true;
		}
		return false;
	} else {
		bool nouse;
		H245_H2250LogicalChannelParameters *h225Params = GetLogicalChannelParameters(olc, nouse);
		return (h225Params) ? OnLogicalChannelParameters(h225Params, flcn) : false;
	}
}

bool H245ProxyHandler::HandleOpenLogicalChannelReject(H245_OpenLogicalChannelReject & olcr)
{
	peer->RemoveLogicalChannel((WORD)olcr.m_forwardLogicalChannelNumber);
	return false; // nothing changed :)
}

bool H245ProxyHandler::HandleOpenLogicalChannelAck(H245_OpenLogicalChannelAck & olca)
{
	if (hnat)
		hnat->HandleOpenLogicalChannelAck(olca);
	const WORD flcn = (WORD)olca.m_forwardLogicalChannelNumber;
	LogicalChannel *lc = peer->FindLogicalChannel(flcn);
	if (!lc) {
		PTRACE(2, "Proxy\tWarning: logical channel " << flcn << " not found");
		return false;
	}
	bool result = lc->SetDestination(olca, this);
	if (result)
		lc->StartReading(handler);
	return result;
}

bool H245ProxyHandler::HandleCloseLogicalChannel(H245_CloseLogicalChannel & clc)
{
	// due to bad implementation of some endpoints, we check the
	// forwardLogicalChannelNumber on both sides
	H245ProxyHandler *first, *second;
	if (clc.m_source.GetTag() == H245_CloseLogicalChannel_source::e_lcse)
		first = this, second = peer;
	else
		first = peer, second = this;
	first->RemoveLogicalChannel((WORD)clc.m_forwardLogicalChannelNumber) 
		|| second->RemoveLogicalChannel((WORD)clc.m_forwardLogicalChannelNumber);
	return false; // nothing changed :)
}

bool H245ProxyHandler::HandleFastStartSetup(H245_OpenLogicalChannel & olc)
{
	if (!peer)
		return false;
	if (hnat)
		hnat->HandleOpenLogicalChannel(olc);
	bool nouse;
	H245_H2250LogicalChannelParameters *h225Params = GetLogicalChannelParameters(olc, nouse);
	return (h225Params) ? OnLogicalChannelParameters(h225Params, 0) : false;
}

bool H245ProxyHandler::HandleFastStartResponse(H245_OpenLogicalChannel & olc)
{
	if (!peer)
		return false;
	if (hnat)
		hnat->HandleOpenLogicalChannel(olc);
	const WORD flcn = (WORD)olc.m_forwardLogicalChannelNumber;
	bool changed = false, isReverseLC;
	H245_H2250LogicalChannelParameters *h225Params = GetLogicalChannelParameters(olc, isReverseLC);
	if (!h225Params)
		return false;
	WORD id = (WORD)h225Params->m_sessionID;
	siterator iter = peer->fastStartLCs.find(id);
	RTPLogicalChannel *lc = (iter != peer->fastStartLCs.end()) ? iter->second : 0;
	if (isReverseLC) {
		if (lc) {
			if (!FindLogicalChannel(flcn)) {
				logicalChannels[flcn] = sessionIDs[id] = lc;
				lc->SetChannelNumber(flcn);
				lc->OnHandlerSwapped(hnat != 0);
				peer->fastStartLCs.erase(iter);
			}
		} else if ((lc = peer->FindRTPLogicalChannelBySessionID(id))) {
			LogicalChannel *akalc = FindLogicalChannel(flcn);
			if (akalc)
				lc = dynamic_cast<RTPLogicalChannel *>(akalc);
			else
				logicalChannels[flcn] = sessionIDs[id] = lc = new RTPLogicalChannel(lc, flcn, hnat != 0);
		}
	} else {
		if (lc) {
			if (!peer->FindLogicalChannel(flcn)) {
				peer->logicalChannels[flcn] = peer->sessionIDs[id] = lc;
				lc->SetChannelNumber(flcn);
				peer->fastStartLCs.erase(iter);
			}
		} else if ((lc = FindRTPLogicalChannelBySessionID(id))) {
			LogicalChannel *akalc = peer->FindLogicalChannel(flcn);
			if (akalc)
				lc = dynamic_cast<RTPLogicalChannel *>(akalc);
			else
				peer->logicalChannels[flcn] = peer->sessionIDs[id] = lc = new RTPLogicalChannel(lc, flcn, hnat != 0);
		}
	}
	if (lc && (changed = lc->OnLogicalChannelParameters(*h225Params, GetLocalAddr(), isReverseLC)))
		lc->StartReading(handler);
	return changed;
}

LogicalChannel *H245ProxyHandler::FindLogicalChannel(WORD flcn)
{
	iterator iter = logicalChannels.find(flcn);
	return (iter != logicalChannels.end()) ? iter->second : 0;
}

RTPLogicalChannel *H245ProxyHandler::FindRTPLogicalChannelBySessionID(WORD id)
{
	siterator iter = sessionIDs.find(id);
	return (iter != sessionIDs.end()) ? iter->second : 0;
}

RTPLogicalChannel *H245ProxyHandler::CreateRTPLogicalChannel(WORD id, WORD flcn)
{
	if (FindLogicalChannel(flcn)) {
		PTRACE(3, "Proxy\tRTP logical channel " << flcn << " already exist?");
		return 0;
	}
	RTPLogicalChannel *lc = peer->FindRTPLogicalChannelBySessionID(id);
	if (lc && !lc->IsAttached()) {
		lc = new RTPLogicalChannel(lc, flcn, hnat != 0);
	// if H.245 OpenLogicalChannel is received, the fast connect procedure
	// should be disable. So we reuse the fast start logical channel here
	} else if (!fastStartLCs.empty()) {
		siterator iter = fastStartLCs.begin();
		(lc = iter->second)->SetChannelNumber(flcn);
		fastStartLCs.erase(iter);
	} else if (!peer->fastStartLCs.empty()){
		siterator iter = peer->fastStartLCs.begin();
		(lc = iter->second)->SetChannelNumber(flcn);
		lc->OnHandlerSwapped(hnat != 0);
		peer->fastStartLCs.erase(iter);
	} else {
		try {
			lc = new RTPLogicalChannel(flcn, hnat != 0);
		} catch (RTPLogicalChannel::NoPortAvailable) {
			PTRACE(2, "Proxy\tError: Can't create RTP logical channel " << flcn);
			return 0;
		}
	}

	logicalChannels[flcn] = sessionIDs[id] = lc;
	PTRACE(4, "RTP\tOpen logical channel " << flcn << " id " << id << " port " << lc->GetPort());
	return lc;
}

RTPLogicalChannel *H245ProxyHandler::CreateFastStartLogicalChannel(WORD id)
{
	siterator iter = fastStartLCs.find(id);
	RTPLogicalChannel *lc = (iter != fastStartLCs.end()) ? iter->second : 0;
	if (!lc) {
		try {
			// the LogicalChannelNumber of a fastStart logical channel is irrelevant
			// it may be set later
			lc = new RTPLogicalChannel(0, hnat != 0);
		} catch (RTPLogicalChannel::NoPortAvailable) {
			PTRACE(2, "Proxy\tError: Can't create fast start logical channel id " << id);
			return 0;
		}
		fastStartLCs[id] = lc;
		PTRACE(4, "RTP\tOpen fast start logical channel id " << id << " port " << lc->GetPort());
	}
	return lc;
}

T120LogicalChannel *H245ProxyHandler::CreateT120LogicalChannel(WORD flcn)
{
	if (FindLogicalChannel(flcn)) {
		PTRACE(3, "Proxy\tT120 logical channel " << flcn << " already exist?");
		return 0;
	}
	T120LogicalChannel *lc = new T120LogicalChannel(flcn);
	logicalChannels[flcn] = lc;
	return lc;
}

bool H245ProxyHandler::RemoveLogicalChannel(WORD flcn)
{
	iterator iter = logicalChannels.find(flcn);
	if (iter == logicalChannels.end()) {
		PTRACE(3, "Proxy\tLogical channel " << flcn << " not found");
		return false;
	}
	LogicalChannel *lc = iter->second;
	siterator i = find_if(sessionIDs.begin(), sessionIDs.end(), bind2nd(ptr_fun(compare_lc), lc));
	if (i != sessionIDs.end())
		sessionIDs.erase(i);
	logicalChannels.erase(iter);
	delete lc;
	return true;
}

// class NATHandler
void NATHandler::TranslateH245Address(H225_TransportAddress & h245addr)
{
	if (h245addr.GetTag() == H225_TransportAddress::e_ipAddress) {
		H225_TransportAddress_ipAddress & addr = h245addr;
		for (int i = 0; i < 4; ++i)
			addr.m_ip[i] = remoteAddr[i];
	}
}

bool NATHandler::HandleOpenLogicalChannel(H245_OpenLogicalChannel & olc)
{
	bool changed = false;
	if (IsT120Channel(olc) && olc.HasOptionalField(H245_OpenLogicalChannel::e_separateStack)) {
		if (olc.m_separateStack.m_networkAddress.GetTag() == H245_NetworkAccessParameters_networkAddress::e_localAreaAddress)
			changed = SetAddress(GetH245UnicastAddress(olc.m_separateStack.m_networkAddress));
	} else {
		bool nouse;
		if (H245_H2250LogicalChannelParameters *h225Params = GetLogicalChannelParameters(olc, nouse)) {
			if (h225Params->HasOptionalField(H245_H2250LogicalChannelParameters::e_mediaControlChannel))
				changed = SetAddress(GetH245UnicastAddress(h225Params->m_mediaControlChannel));
			if (h225Params->HasOptionalField(H245_H2250LogicalChannelParameters::e_mediaChannel))
				changed |= SetAddress(GetH245UnicastAddress(h225Params->m_mediaChannel));
		}
	}
	return changed;
}

bool NATHandler::HandleOpenLogicalChannelAck(H245_OpenLogicalChannelAck & olca)
{
	if (olca.HasOptionalField(H245_OpenLogicalChannelAck::e_separateStack)) {
		H245_NetworkAccessParameters & sepStack = olca.m_separateStack;
		if (sepStack.m_networkAddress.GetTag() == H245_NetworkAccessParameters_networkAddress::e_localAreaAddress)
			return SetAddress(GetH245UnicastAddress(sepStack.m_networkAddress));
	} else {
		H245_UnicastAddress_iPAddress *mediaControlChannel, *mediaChannel;
		if (GetChannelsFromOLCA(olca, mediaControlChannel, mediaChannel)) {
			SetAddress(mediaChannel);
			return SetAddress(mediaControlChannel);
		}
	}
	return false;
}

bool NATHandler::SetAddress(H245_UnicastAddress_iPAddress * addr)
{
	return addr ? (*addr << remoteAddr, true) : false;
}

// to avoid ProxyThread.cxx to include the large h225.h,
// put the method here...
// class ProxyListener
TCPProxySocket *ProxyListener::CreateSocket()
{
	return new CallSignalSocket;
}

// class HandlerList
void HandlerList::LoadConfig()
{
	Q931PortRange.LoadConfig(RoutedSection, "Q931PortRange");
	H245PortRange.LoadConfig(RoutedSection, "H245PortRange");
	T120PortRange.LoadConfig(ProxySection, "T120PortRange");
	RTPPortRange.LoadConfig(ProxySection, "RTPPortRange", "10000-59999");

	WORD port = (WORD)GkConfig()->GetInteger(RoutedSection, "CallSignalPort", GK_DEF_CALL_SIGNAL_PORT);
	if (!listenerThread || (GKPort != port)) {
		CloseListener();
		unsigned queueSize = GkConfig()->GetInteger("ListenQueueLength", GK_DEF_LISTEN_QUEUE_LENGTH);
		listenerThread = new ProxyListener(this, GKHome, port ? port : Q931PortRange.GetPort(), queueSize);
		GKPort = port;
	}

	// the handler number can only be increased
	int s = GkConfig()->GetInteger(RoutedSection, "CallSignalHandlerNumber", 1);
	if (s < 1) { // sanity check
		PTRACE(3, "Proxy\tCallSignalHandlerNumber must be >= 1");
		s = 1;
	}
	for (int i = handlers.size(); i < s; ++i)
		handlers.push_back(new ProxyHandleThread(i));
}
