/*************************************************************************************************
 * Command line utility to convert a mbox file to mh files
 *                                                      Copyright (C) 2003-2004 Mikio Hirabayashi
 * This file is part of Estraier, a personal full-text search system.
 * Estraier is free software; you can redistribute it and/or modify it under the terms of the GNU
 * General Public License as published by the Free Software Foundation; either version 2 of the
 * License, or any later version.
 * Estraier is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License along with Estraier;
 * if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA 02111-1307 USA.
 *************************************************************************************************/


#include "estcommon.h"


/* global variables */
const char *progname = NULL;             /* program name */


/* function prototypes */
int main(int argc, char **argv);
void usage(void);
char *getl(FILE *ifp);
int dividefile(const char *file, const char *pre, const char *suf, int col);


/* main routine */
int main(int argc, char **argv){
  char *file, *pre, *suf;
  int i, col;
  estputenv("LANG", ESTLOCALE);
  estputenv("LC_ALL", ESTLOCALE);
  cbstdiobin();
  progname = argv[0];
  col = 1;
  pre = NULL;
  suf = NULL;
  file = NULL;
  for(i = 1; i < argc; i++){
    if(!file && argv[i][0] == '-'){
      if(!strcmp(argv[i], "-pre")){
        if(++i >= argc) usage();
        pre = argv[i];
      } else if(!strcmp(argv[i], "-suf")){
        if(++i >= argc) usage();
        suf = argv[i];
      } else if(!strcmp(argv[i], "-col")){
        if(++i >= argc) usage();
        col = atoi(argv[i]);
      } else {
        usage();
      }
    } else if(!file){
      file = argv[i];
    } else {
      usage();
    }
  }
  return dividefile(file, pre, suf, col);
}


/* divide a mbox file into mh files */
int dividefile(const char *file, const char *pre, const char *suf, int col){
  FILE *ifp, *ofp;
  char *line, *rp, nfmt[16], oname[2048];
  int num, err;
  ifp = stdin;
  if(file){
    if(!(ifp = fopen(file, "rb"))){
      fprintf(stderr, "%s: %s: could not open\n", progname, file);
      return 1;
    }
  }
  err = FALSE;
  num = 0;
  ofp = NULL;
  sprintf(nfmt, "%%s%%0%dd%%s", col);
  while((line = getl(ifp)) != NULL){
    if(cbstrfwmatch(line, "From ")){
      do {
        num++;
        sprintf(oname, nfmt, pre ? pre : "", num, suf ? suf : "");
        if(!cbfilestat(oname, NULL, NULL, NULL)) break;
        fprintf(stderr, "%s: %s: skipped\n", file, oname);
      } while(TRUE);
      if(ofp) fclose(ofp);
      if(!(ofp = fopen(oname, "wb"))){
        fprintf(stderr, "%s: %s: could not open\n", file, oname);
        err = TRUE;
        break;
      }
      printf("%s\t%s\n", oname, line);
    } else if(ofp){
      rp = line;
      while(TRUE){
        if(cbstrfwmatch(rp, "=46")){
          fputc('F', ofp);
          rp += 3;
        } else if(cbstrfwmatch(rp, "=3D")){
          fputc('=', ofp);
          rp += 3;
        } else {
          break;
        }
      }
      fprintf(ofp, "%s\r\n", rp);
    }
    free(line);
  }
  if(ofp) fclose(ofp);
  if(ifp != stdin) fclose(ifp);
  return err ? 1 : 0;
}


/* print the usage and exit */
void usage(void){
  fprintf(stderr, "%s: converter of a mbox file to mh files\n", progname);
  fprintf(stderr, "\n");
  fprintf(stderr, "usage:\n");
  fprintf(stderr, "  %s [-pre str] [-suf str] [-col num] [file]\n", progname);
  fprintf(stderr, "\n");
  exit(1);
}


/* read a line */
char *getl(FILE *ifp){
  char *buf;
  int c, len, blen;
  buf = NULL;
  len = 0;
  blen = 256;
  while((c = fgetc(ifp)) != EOF){
    if(blen <= len) blen *= 2;
    buf = cbrealloc(buf, blen + 1);
    if(c == '\n') c = '\0';
    buf[len++] = c;
    if(c == '\0') break;
  }
  if(!buf) return NULL;
  buf[len] = '\0';
  len--;
  if(len > 0 && buf[len-1] == '\r') buf[len-1] = '\0';
  return buf;
}



/* END OF FILE */
