
#include <stdlib.h>
#include <stdarg.h>
#include <stdio.h>
#include <newt.h>
#include <unistd.h>
#include <dirent.h>
#include <string.h>
#include <malloc.h>
#include <errno.h>
#include <limits.h>
#include <fcntl.h>

#define CIPE_VALUE_STRING	1
#define CIPE_VALUE_BOOLEAN	2
#define CIPE_VALUE_INTEGER	3
#define CIPE_VALUE_IP		CIPE_VALUE_STRING
#define CIPE_VALUE_UDP		CIPE_VALUE_STRING
#define CIPE_VALUE_TCP		CIPE_VALUE_UDP

#define CIPE_FROM_OPTIONS	1
#define CIPE_FROM_IFCFG		2
#define CIPE_FROM_USER		3

typedef struct cipeValue cipeValue;
struct cipeValue {
	cipeValue *	next;
	cipeValue *	prev;
	char *		name;
	int		type;
	int		from;
	union {
		char *			sval;
		char			bval;
		long			lval;
	} value;
	newtComponent	co;
};

typedef struct cipeInterface cipeInterface;
struct cipeInterface {
	char *		name;		/* Original interface name. */
	cipeValue *	config;
	newtComponent 	form;		/* Interface edit form for validation*/
};

static void			browseInterfaces	(void);
static void			loadInterfaces		(newtComponent iflist);
static void			deleteInterface		(const char *iface);
static void			editInterface		(const char *iface);
static void			doEditForm		(cipeInterface *iface);

/* Manipulate interface stuff. */
static cipeInterface*		cipeNew			(void);
static cipeInterface*		cipeLoad		(const char *ifname);
static void			cipeDestroy		(cipeInterface* iface);
static int			cipeSave		(cipeInterface* iface);

static cipeValue*		cipeGetValue		(cipeInterface* iface,
							 const char *name);
static cipeValue*		cipeSetValue		(cipeInterface* iface,
							 const char *name,
							 const char *value,
							 int from);
static int			cipeValidate		(cipeInterface *iface);
static char*			cipeGetEntryValue	(cipeInterface *iface,
							 const char *name);


static int			cipeGetType		(const char *name);

int main (int argc, char *argv[])
{
	newtInit ();
	newtCls ();
	newtPushHelpLine (NULL);
	newtDrawRootText (0,0,"cipecfg " VERSION " - CIPE interface configuration utility");
	newtDrawRootText (0,1,"--------------------------------------------------");

	browseInterfaces ();

	newtPopHelpLine ();
	newtFinished ();
	exit (0);
}

static void loadInterfaces (newtComponent iflist)
{
	DIR *dh;
	struct dirent *pent;

	/* XXX: Memory leak since we don't free the interface names. */
	newtListboxClear (iflist);

	/* XXX: We should sort the interface names, and accept IDEA and
	 * different protocol version interfaces. */
	if ((dh = opendir ("/etc/sysconfig/network-scripts")) == NULL) {
		newtFinished ();
		perror("/etc/sysconfig/network-scripts");
		exit (1);
	}
	while ((pent = readdir (dh)) != NULL) {
		char *scan;
		if (strlen(pent->d_name) < strlen ("ifcfg-cipcbx"))
			continue;
		if (memcmp (pent->d_name, "ifcfg-cipcb", strlen ("ifcfg-cipcb")))
			continue;
		scan = pent->d_name + strlen("ifcfg-cipcb");
		while (*scan && isdigit (*scan))
			scan++;
		if (*scan)
			continue; /* Likely a vi backup. */
		scan = strdup (pent->d_name + strlen("ifcfg-"));
		newtListboxAppendEntry (iflist, scan, scan);
	}
	closedir(dh);
}

static void browseInterfaces (void)
{
	newtComponent form;
	newtComponent iflist, addButton, editButton, deleteButton, closeButton;
	struct newtExitStruct es;
	char *iface;

	newtCenteredWindow (43, 15, "CIPE Interfaces");

	form = newtForm(NULL, NULL, 0);
	newtFormAddHotKey (form, NEWT_KEY_DELETE);

	iflist = newtListbox (1,1,6,NEWT_FLAG_RETURNEXIT);
	addButton = newtButton (1, 11, "Add");
	editButton = newtButton (10, 11, "Edit");
	deleteButton = newtButton (20, 11, "Delete");
	closeButton = newtButton (32, 11, "Close");

	newtFormAddComponents (form, iflist, addButton, editButton,
			deleteButton, closeButton, NULL);

	while (1) {
		loadInterfaces (iflist);
		newtFormRun (form, &es);

		switch (es.reason) {
		case NEWT_EXIT_HOTKEY:
			switch (es.u.key) {
			case NEWT_KEY_DELETE:
				iface = (char *)newtListboxGetCurrent (iflist);
				if (iface != NULL) {
					deleteInterface (iface);
				}
				continue;
			}
			break;
		case NEWT_EXIT_COMPONENT:
			if (es.u.co == addButton) {
				editInterface (NULL);
				continue;
			} else if (es.u.co == editButton || es.u.co == iflist) {
				iface = (char *)newtListboxGetCurrent (iflist);
				if (iface != NULL) {
					editInterface (iface);
				}
				continue;
			} else if (es.u.co == deleteButton) {
				iface = (char *)newtListboxGetCurrent (iflist);
				if (iface != NULL) {
					deleteInterface (iface);
				}
				continue;
			}
		}
		break;
	}

	newtPopWindow ();
	newtFormDestroy (form);
}

static void guiUnlink (const char *filen)
{
	if (unlink (filen) == -1 && errno != ENOENT)
		newtWinMessage ("Error Deleting File", "Ok", "%s: %s", filen,
				strerror (errno));
}

static void deleteInterface (const char *iface)
{
	char filen[256];
	int ans = newtWinChoice ("Delete Interface", "Yes", "No",
			"Are you sure you want to delete CIPE interface %s?",
			iface);
	if (ans != 1)
		return;

	sprintf (filen, "/etc/sysconfig/network-scripts/ifcfg-%s", iface);
	guiUnlink (filen);
	sprintf (filen, "/etc/cipe/options.%s", iface);
	guiUnlink (filen);
}

static void editInterface (const char *ifname)
{
	cipeInterface *iface;

	if (ifname != NULL)
		iface = cipeLoad (ifname);
	else
		iface = cipeNew ();

	doEditForm (iface);
	cipeDestroy (iface);
}

static void cipeGenerateKey(cipeInterface * iface)
{
	FILE *f;
	cipeValue *v;
	int failed = 0;
	char buf[36];
	if (system("mcookie > /etc/cipe/cipecfg.tmp") != 0) {
		if (system("ps awux |md5sum >/etc/cipe/cipecfg.tmp") != 0) {
			/* Oops!. */
			failed = 1;
		}
	}
	if (!failed && (f = fopen("/etc/cipe/cipecfg.tmp", "r")) != NULL) {
		fgets(buf, sizeof(buf), f);
		if (strchr(buf, '\n'))
			*strchr(buf, '\n') = '\0';
		fclose(f);
	}
	unlink("/etc/cipe/cipecfg.tmp");
	if (failed && iface->form != NULL) {
		newtWinMessage ("Error", "Ok", "Could not generate new key!");
		return;
	}

	/* Either set the control or the current value. */
	v = cipeGetValue (iface, "key");
	if (v != NULL && v->co != NULL)
		newtEntrySet (v->co, buf, 1);
	else
		cipeSetValue (iface, "key", buf, CIPE_FROM_OPTIONS);
}

static cipeInterface* cipeNew ()
{
	cipeInterface *iface;

	if ((iface = (cipeInterface*)malloc (sizeof (cipeInterface))) == NULL){
		newtFinished ();
		perror("malloc");
		exit (1);
	}

	iface->config = NULL;
	iface->form = NULL;
	iface->name = NULL;

	cipeSetValue (iface, "device", "cipcb", CIPE_FROM_OPTIONS);
	cipeGenerateKey (iface);
	return iface;
}

static cipeInterface* cipeLoad (const char *ifname)
{
	cipeInterface *iface;
	cipeValue *v;
	char fname[256], linebuf[256];
	FILE *f;
	char *scan;
	char *name;
	int i;

	if ((iface = (cipeInterface*)malloc (sizeof (cipeInterface))) == NULL){
		newtFinished ();
		perror("malloc");
		exit (1);
	}
	iface->config = NULL;

	/* Load the options file. */
	sprintf (fname, "/etc/cipe/options.%s", ifname);
	if ((f = fopen(fname, "r")) == NULL) {
		newtFinished ();
		perror(fname);
		exit (1);
	}
	while (fgets (linebuf, sizeof(linebuf)-2, f)) {
		linebuf[sizeof(linebuf)-1] = '\0';
		scan = linebuf;
		while (*scan && isspace (*scan))
			scan++;
		if (!*scan || *scan == '#')
			continue; /* Comments and blank lines */
		if (strchr (scan, '\n'))
			*strchr (scan, '\n') = '\0';
		name = scan;
		i = strspn (name, "abcdefghijklmnopqrstuvwxyz_");
		if (i == 0)
			continue; /* ? */
		scan = name + i;
		while (*scan && isspace (*scan))
			scan++;
		if (*scan == '\0') {
			name[i] = 0;
			cipeSetValue (iface, name, NULL, CIPE_FROM_OPTIONS);
		} else if (*scan == '=') {
			scan++;
			while (*scan && isspace (*scan))
				scan++;
			name[i] = 0;
			cipeSetValue (iface, name, scan, CIPE_FROM_OPTIONS);
		} else
			continue; /* ? */
	}
	fclose (f);

	/* Load the ifcfg file. */
	sprintf (fname, "/etc/sysconfig/network-scripts/ifcfg-%s", ifname);
	if ((f = fopen(fname, "r")) == NULL) {
		newtFinished ();
		perror(fname);
		exit (1);
	}
	while (fgets (linebuf, sizeof(linebuf)-2, f)) {
		linebuf[sizeof(linebuf)-1] = '\0';
		scan = linebuf;
		while (*scan && isspace (*scan))
			scan++;
		if (!*scan || *scan == '#')
			continue; /* Comments and blank lines */
		if (strchr (scan, '\n'))
			*strchr (scan, '\n') = '\0';
		if (!strchr (scan, '='))
			continue;
		name = scan;
		i = strcspn (name, "= \t\r\n\v\f\b\a");
		scan += i + 1;
		name[i] = 0;
		if (*scan == '"') {
			char *qscan = ++scan;
			while (*qscan && *qscan != '"') {
				if (*qscan == '\\')
					qscan++;
				qscan++;
			}
			*qscan = '\0';
		} else if (*scan == '\'') {
			scan++;
			i = strcspn (scan, "'");
			scan[i] = '\0';	
		} else {
			i = strcspn (scan, " \t\r\n\f\v\b\a");
			scan[i] = '\0';
		}
		cipeSetValue (iface, name, scan, CIPE_FROM_IFCFG);
	}
	fclose (f);

	/* IPADDR in the ifcfg file is the same as ipaddr in the options
	 * file. */
	if ((v = cipeGetValue (iface, "IPADDR")) != NULL)
		cipeSetValue (iface, "ipaddr", v->value.sval, CIPE_FROM_OPTIONS);
	if ((v = cipeGetValue (iface, "PTPADDR")) != NULL)
		cipeSetValue (iface, "ptpaddr", v->value.sval, CIPE_FROM_OPTIONS);
	if ((v = cipeGetValue (iface, "PEER")) != NULL)
		cipeSetValue (iface, "peer", v->value.sval, CIPE_FROM_OPTIONS);
	if ((v = cipeGetValue (iface, "ME")) != NULL) {
		cipeValue *p;
		if ((p = cipeGetValue (iface, "MYPORT")) != NULL) {
			char buf[256];
			sprintf (buf, "%s:%d", v->value.sval, p->value.lval);
			cipeSetValue (iface, "me", buf, CIPE_FROM_OPTIONS);
		} else
			cipeSetValue (iface, "me", v->value.sval, CIPE_FROM_OPTIONS);
	}
	if ((v = cipeGetValue (iface, "DEVICE")) != NULL)
		cipeSetValue (iface, "device", v->value.sval, CIPE_FROM_OPTIONS);

	if ((v = cipeGetValue (iface, "device")) != NULL) {
		if (v->value.sval != NULL) {
			iface->name = strdup (v->value.sval);
		}
	}

	return iface;
}

static void cipeDestroy	(cipeInterface* iface)
{
	cipeValue *iter, *next;
	if (iface->name != NULL)
		free (iface->name);
	if ((iter = iface->config) != NULL) {
		do {
			next = iter->next;
			if (iter->name != NULL)
				free (iter->name);
			if (iter->type == CIPE_VALUE_STRING &&
					iter->value.sval != NULL)
				free (iter->value.sval);
			free (iter);
			iter = next;
		} while (iter != iface->config);
	}
	free (iface);
}

static cipeValue* cipeGetValue (cipeInterface* iface,
				const char *name)
{
	cipeValue* iter;
	if ((iter = iface->config) != NULL) {
		do {
			if (iter->name && !strcmp (iter->name, name))
				return iter;
			iter = iter->next;
		} while (iter != iface->config);
	}
	return NULL;
}

static cipeValue* cipeSetValue (cipeInterface* iface,
			  	const char *name,
			  	const char *value,
				int from)
{
	cipeValue *val;

	if ((val = cipeGetValue (iface, name)) == NULL) {
		val = (cipeValue*)malloc (sizeof (cipeValue));
		if (val == NULL) {
			newtFinished ();
			perror("malloc");
			exit (1);
		}
		val->name = strdup (name);
		val->type = cipeGetType (name);
		val->from = from;
		val->co = NULL;
		if (iface->config == NULL)
			iface->config = val->next = val->prev = val;
		else {
			val->next = iface->config;
			val->prev = iface->config->prev;
			val->prev->next = val;
			val->next->prev = val;
		}
	} else {
		if (val->type == CIPE_VALUE_STRING && val->value.sval != NULL)
			free (val->value.sval);
	}

	if (val->type == CIPE_VALUE_STRING)
		val->value.sval = strdup (value == NULL ? "" : value);
	else if (val->type == CIPE_VALUE_BOOLEAN) {
		if (from == CIPE_FROM_OPTIONS)
			val->value.bval = 1;
		else if (!strcmp (value, "yes") || !strcmp (value, "on")
				|| !strcmp (value, "true") || !strcmp (value, "1"))
			val->value.bval = 1;
		else
			val->value.bval = 0;
	} else if (val->type == CIPE_VALUE_INTEGER) {
		char *endptr;
		if (value == NULL || !strcmp (value, ""))
			val->value.lval = LONG_MIN;
		else
			val->value.lval = strtol (value, &endptr, 10);
	}

	return val;
}

struct cipeValueType {
	char *name;
	int type;
};
static struct cipeValueType cipeValueTypes[] = {
	{ "device", 	CIPE_VALUE_STRING },
	{ "debug", 	CIPE_VALUE_BOOLEAN },
	{ "ipaddr", 	CIPE_VALUE_IP },
	{ "ptpaddr",	CIPE_VALUE_IP },
	{ "mtu",	CIPE_VALUE_INTEGER },
	{ "metric",	CIPE_VALUE_INTEGER },
	{ "cttl",	CIPE_VALUE_INTEGER },
	{ "me",		CIPE_VALUE_UDP },
	{ "peer",	CIPE_VALUE_UDP },
	{ "key",	CIPE_VALUE_STRING },
	{ "nokey",	CIPE_VALUE_BOOLEAN },
	{ "socks",	CIPE_VALUE_TCP },
	{ "tokxc",	CIPE_VALUE_INTEGER },
	{ "tokey",	CIPE_VALUE_INTEGER },
	{ "ipup",	CIPE_VALUE_STRING },
	{ "ipdown",	CIPE_VALUE_STRING },
	{ "arg",	CIPE_VALUE_STRING },
	{ "maxerr",	CIPE_VALUE_INTEGER },
	{ "tokxts",	CIPE_VALUE_INTEGER },
	{ "ping",	CIPE_VALUE_INTEGER },
	{ "toping",	CIPE_VALUE_INTEGER },
	{ "dynip",	CIPE_VALUE_BOOLEAN },
	{ "ONBOOT",	CIPE_VALUE_BOOLEAN },
	{ "USERCTL",	CIPE_VALUE_BOOLEAN },
	{ "DEVICE",	CIPE_VALUE_STRING },
	{ "IPADDR",	CIPE_VALUE_IP },
	{ "PTPADDR",	CIPE_VALUE_IP },
	{ "ME",		CIPE_VALUE_UDP },
	{ "MYPORT",	CIPE_VALUE_INTEGER },
	{ "PEER",	CIPE_VALUE_UDP },
	{ NULL, 0 }
};

static int cipeGetType (const char *name)
{
	struct cipeValueType* vtiter = cipeValueTypes;
	while (vtiter->name != NULL) {
		if (!strcmp (vtiter->name, name))
			return vtiter->type;
		vtiter++;
	}
	return CIPE_VALUE_STRING;
}

static int cipeIntFilter (newtComponent entry,
			  void *data,
			  int ch,
			  int cursor)
{
	if (cursor == 0 && ch == '-' && !strchr (newtEntryGetValue (entry),'-'))
		return '-';
	if (ch >= 127 || ch < ' ')
		return ch;
	if (!isdigit (ch))
		return 0;
	return ch;
}

static newtComponent cipeEntry (cipeInterface *iface, 
				const char *name,
				const char *labl,
				int row,
				int posflag,
				int from)
{
	newtComponent label;
	cipeValue *value = cipeGetValue (iface, name);
	char ivbuf[256];
	char *initial_value;
	int left, width;
	if (value == NULL)
		value = cipeSetValue (iface, name, "", from);

	if (labl != NULL) {
		label = newtLabel (2, row, labl);
		newtFormAddComponent (iface->form, label);
	}
	switch (posflag) {
	case 0:
		left = 25;
		width = 33;
		break;
	case 1:
		left = 25;
		width = 16;
		break;
	case 2:
		left = 25 + 17;
		width = 16;
		break;
	}
	switch (value->type) {
	case CIPE_VALUE_STRING:
		initial_value = value->value.sval;
		break;
	case CIPE_VALUE_INTEGER:
		if (value->value.lval == LONG_MIN)
			ivbuf[0] = '\0';
		else
			sprintf (ivbuf, "%ld", value->value.lval);
		initial_value = ivbuf;
		break;
	}
	value->co = newtEntry (left, row, initial_value, width, NULL, 0);
	switch (value->type) {
	case CIPE_VALUE_INTEGER:
		newtEntrySetFilter (value->co, cipeIntFilter, NULL);
		break;
	}
	newtFormAddComponent (iface->form, value->co);
	return value->co;
}

static newtComponent cipeCheckbox (cipeInterface *iface,
				   const char *name,
				   const char *labl,
				   int row,
				   int from)
{
	newtComponent cb;
	cipeValue *value = cipeGetValue (iface, name);
	if (value == NULL) {
		value = cipeSetValue (iface, name, "no", from);
		value->value.bval = 0;
	}

	value->co = newtCheckbox (12, row, labl, 
			value->value.bval ? '*' : ' ', NULL, NULL);
	newtFormAddComponent (iface->form, value->co);
	return value->co;
}

static void getNewtValues (cipeInterface *iface)
{
	cipeValue *iter;
	if ((iter = iface->config) == NULL)
		return;
	do {
		if (iter->co != NULL) {
			switch (iter->type) {
			case CIPE_VALUE_STRING:
			case CIPE_VALUE_INTEGER:
				cipeSetValue (iface, iter->name, newtEntryGetValue (iter->co), CIPE_FROM_USER);
				break;

			case CIPE_VALUE_BOOLEAN:
				iter->value.bval = (newtCheckboxGetValue (iter->co) == '*');
				break;
			}
		}
		iter = iter->next;
	} while (iter != iface->config);
}

static void doEditForm(cipeInterface * iface)
{
	newtComponent exitComp;
	newtComponent formOk, formCancel, timeLabel, timeLabel2, formNewKey;
	struct newtExitStruct es;

	newtCenteredWindow(60, 21, "Add/Edit CIPE Interface");
	iface->form = newtForm(NULL, NULL, 0);
	formOk = newtButton(27, 17, "Ok");
	formCancel = newtButton(35, 17, "Cancel");
	formNewKey = newtButton (47, 17, "New Key");

	cipeEntry(iface, "device", "Interface Name:", 1, 0, CIPE_FROM_OPTIONS);
	cipeEntry(iface, "key", "Key:", 2, 0, CIPE_FROM_OPTIONS);
	cipeEntry(iface, "ipaddr", "Interface Address:", 3, 0, CIPE_FROM_OPTIONS);
	cipeEntry(iface, "ptpaddr", "Point-to-Point Addr:", 4, 0, CIPE_FROM_OPTIONS);
	cipeEntry(iface, "me", "Local Tunnel Endpoint:", 5, 0, CIPE_FROM_OPTIONS);
	cipeEntry(iface, "peer", "Remt Tunnel Endpoint:", 6, 0, CIPE_FROM_OPTIONS);
	cipeEntry(iface, "socks", "SOCKS5 Proxy:", 7, 0, CIPE_FROM_OPTIONS);

	cipeEntry(iface, "tokey", "Key/Key Exchange T/O: *", 9, 1, CIPE_FROM_OPTIONS);
	cipeEntry(iface, "tokxc", NULL, 9, 2, CIPE_FROM_OPTIONS);
	cipeEntry(iface, "ping", "Ping Interval/Timeout:", 10, 1, CIPE_FROM_OPTIONS);
	cipeEntry(iface, "toping", NULL, 10, 2, CIPE_FROM_OPTIONS);
	cipeEntry(iface, "maxerr", "Maximum Error Count:", 11, 1, CIPE_FROM_OPTIONS);

	cipeCheckbox(iface, "ONBOOT",
		     "Activate at boot time          ", 13, CIPE_FROM_IFCFG);
	cipeCheckbox(iface, "USERCTL",
		     "Users can control interface    ", 14, CIPE_FROM_IFCFG);
	cipeCheckbox(iface, "dynip",
		     "Detect dynamic IP changes      ", 15, CIPE_FROM_OPTIONS);

	timeLabel = newtLabel (2, 17, "* All time values are");
	timeLabel2 =newtLabel (2, 18, "  in seconds.");
	newtFormAddComponents(iface->form, formOk, formCancel, 
			formNewKey, timeLabel, timeLabel2, NULL);

	while (1) {
		newtFormRun (iface->form, &es);
		switch (es.reason) {
		case NEWT_EXIT_HOTKEY:
			switch (es.u.key) {
			case NEWT_KEY_F12:
				if (!cipeValidate (iface))
					continue;
				getNewtValues (iface);
				if (!cipeSave (iface))
					continue;
			}
			break;
		case NEWT_EXIT_COMPONENT:
			if (es.u.co == formOk) {
				if (!cipeValidate (iface))
					continue;
				getNewtValues (iface);
				if (!cipeSave (iface))
					continue;
			} else if (es.u.co == formNewKey) {
				cipeGenerateKey (iface);
				continue;
			}
			break;
		}

		newtPopWindow();
		newtFormDestroy(iface->form);
		iface->form = NULL;
		break;
	}
}

static char* cipeGetEntryValue (cipeInterface *iface, const char *name)
{
	cipeValue* val = cipeGetValue (iface, name);
	return val == NULL ? NULL : newtEntryGetValue(val->co);
}

static int invalid (cipeInterface *iface,
		    const char *fld,
		    const char *fmt,
		    ...)
{
	va_list vl;
	cipeValue *value;

	va_start (vl, fmt);
	newtWinMessagev ("Validation Error", "Ok", (char *)fmt, vl);
	va_end (vl);
	value = cipeGetValue (iface, fld);
	if (value != NULL && value->co != NULL && iface->form != NULL)
		newtFormSetCurrent (iface->form, value->co);
	return 0;
}

static int cipeValidateIP (cipeInterface *iface,
			   const char *name,
			   int accept_port)
{
	char *str = cipeGetEntryValue (iface, name), *scan;
	char *addr;
	if (str == NULL || !strcmp (str, "")) {
		if (accept_port)
			return 1; /* Ok for me and peer to be blank. */
		else
			return invalid (iface, name, "%s address cannot be blank.", name);
	}
	addr = strdup (str);
	if (strchr (str, ':') != NULL) {
		char *p;
		if (!accept_port)
			return invalid (iface, name, "%s address cannot contain a port.", name);
		p = strchr (str, ':') + 1;
		scan = p;
		while (*scan && isdigit (*scan))
			scan++;
		if (*scan)
			return invalid (iface, name, "Invalid port number `%s'.", p);
		*strchr (addr, ':') = '\0';
	}

	/* We can't validate the address because it might be a name which
	 * is unresolveable at the moment.  (NT tries to do this and it
	 * really pisses me off...) */
	return 1;
}


static int cipeValidate (cipeInterface *iface)
{
	char *str;
	int flag;

	/* Validate the device name. */
	flag = 0;
	str = cipeGetEntryValue (iface, "device");
	if (str == NULL || memcmp (str, "cipcb", 5))
		flag = 1;
	else {
		str += 5;
		if (*str == '\0')
			flag = 1;
		else {
			while (*str && isdigit (*str))
				str++;
			if (*str != '\0')
				flag = 1;
		}
	}
	if (flag)
		return invalid (iface, "device", "The CIPE device name must be in the format `cipcbxx', where `xx' is the interface number.");

	/* Validate the key. */
	flag = 0;
	str = cipeGetEntryValue (iface, "key");
	if (str == NULL || !strcmp (str, ""))
		return invalid (iface, "key", "The key must not be blank.");
	else if (strlen (str) != 32)
		return invalid (iface, "key", "Invalid key length.");
	else {
		while (*str) {
			if (!isxdigit (*str)) {
				return invalid (iface, "key", "Invalid character '%c' in key.", *str);
			}
			str++;
		}
	}

	/* Validate ipaddr and ptpaddr. */
	if (!cipeValidateIP (iface, "ipaddr", 0))
		return 0;
	if (!cipeValidateIP (iface, "ptpaddr", 0))
		return 0;
	
	/* Validate me and peer. */
	if (!cipeValidateIP (iface, "me", 1))
		return 0;
	if (!cipeValidateIP (iface, "peer", 1))
		return 0;

	return 1;
}

struct cipeWriteAlias {
	char *name;
	char *alias_for;
};
static struct cipeWriteAlias aliases[] = {
	{ "DEVICE",	"device" },
	{ "IPADDR",	"ipaddr" },
	{ "PTPADDR",	NULL },
	{ "PEER",	NULL },
	{ "ME",		NULL },
	{ "MYPORT",	NULL },
	{ "device",	NULL },
	{ "ipaddr",	NULL },
	{ NULL, NULL }
};

static int cipeSave (cipeInterface *iface)
{
	FILE *options, *ifcfg;
	cipeValue *v;
	char *name, filen[256];
	int fd;

	/* unlink old files in case the interface has been renamed. */
	if (iface->name != NULL) {
		sprintf (filen, "/etc/cipe/options.%s", iface->name);
		unlink(filen);
		sprintf (filen, "/etc/sysconfig/network-scripts/ifcfg-%s", name);
		unlink(filen);
	}
	
	if ((v = cipeGetValue (iface, "device")) != NULL)
		name = v->value.sval;
	sprintf (filen, "/etc/cipe/options.%s", name);
	if ((fd = open (filen, O_CREAT | O_TRUNC | O_WRONLY, 0600)) == -1) {
		newtWinMessage ("Error", "Ok", "/etc/cipe/options.%s: %s",
				name, strerror(errno));
		return 0;
	}
	if ((options = fdopen (fd, "w")) == NULL) {
		newtWinMessage ("Error", "Ok", "fdopen: %s", strerror(errno));
		return 0;
	}
	sprintf (filen, "/etc/sysconfig/network-scripts/ifcfg-%s", name);
	ifcfg = fopen (filen, "w");

	if ((v = iface->config) != NULL) {
		do {
			if (!strcmp (v->name, "device"))
				fprintf (ifcfg, "DEVICE=\"%s\"\n", v->value.sval);
			else if (!strcmp (v->name, "ipaddr"))
				fprintf (ifcfg, "IPADDR=\"%s\"\n", v->value.sval);
			else if (!strcmp (v->name, "ptpaddr"))
				fprintf (ifcfg, "PTPADDR=\"%s\"\n", v->value.sval);
			if (!strcmp (v->name, "DEVICE")
					|| !strcmp (v->name, "IPADDR")
					|| !strcmp (v->name, "PTPADDR")
					|| !strcmp (v->name, "PEER")
					|| !strcmp (v->name, "ME")
					|| !strcmp (v->name, "MYPORT")
					|| !strcmp (v->name, "device")
					|| !strcmp (v->name, "ipaddr")
					|| !strcmp (v->name, "ptpaddr"))
				goto loop_end;

			if (v->from == CIPE_FROM_OPTIONS || v->from == CIPE_FROM_USER) {
				switch (v->type) {
				case CIPE_VALUE_INTEGER:
					if (v->value.lval != LONG_MIN)
						fprintf (options, "%s = %ld\n", v->name, v->value.lval);
					break;
				case CIPE_VALUE_STRING:
					if (v->value.sval != NULL && strcmp (v->value.sval, ""))
						fprintf (options, "%s = %s\n", v->name, v->value.sval);
					break;
				case CIPE_VALUE_BOOLEAN:
					if (v->value.bval)
						fprintf (options, "%s\n", v->name);
					break;
				}
			} else {
				switch (v->type) {
				case CIPE_VALUE_INTEGER:
					if (v->value.lval != LONG_MIN)
						fprintf (ifcfg, "%s=%ld\n", v->name, v->value.lval);
					break;
				case CIPE_VALUE_STRING:
					if (v->value.sval != NULL)
						fprintf (ifcfg, "%s=\"%s\"\n", v->name, v->value.sval);
					break;
				case CIPE_VALUE_BOOLEAN:
					if (v->value.bval)
						fprintf (ifcfg, "%s=yes\n", v->name);
					else
						fprintf (ifcfg, "%s=no\n", v->name);
					break;
				}
			}
loop_end:
			v = v->next;
		} while (v != iface->config);
	}

	fclose (ifcfg);
	fclose (options);
	return 1;
}
