/*
This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "combine_wave.h"


void running_info()
{
fprintf(stderr,\
"    RUNNNING COMMANDS\n\
b         toggles move both channels // move right channel delay mode.\n\
ESC       exits.\n\
'z'  'x'  1 sample forward / backward.\n\
'Z'  'X'  10 samples forward / backward.\n\
'['  ']'  1 ms forward / backward.\n\
'{'  '}'  10 ms forward / backward.\n\
','  '.'  100 ms forward / backward.\n\
'<'  '>'  1000 ms forward / backward.\n\
'-'  '='  10000 ms forward / backward.\n\
'_'  '+'  100000 ms forward / backward.\n\
");

} /* end function running info */



void usage()
{
fprintf(stderr,\
"Usage:\
combine_wave [-a] [-d milli seconds delay right channel relative to left]\n\
[-e samples delay right channel relative to left]\n\
[-k] -l filename_left [-m] -o output_filename -r filename_right [s start seek offset].\n\
-a  adjust mode, audio output via /dev/dsp, '[' decreases delay, ']' increases delay.\n\
-d  float   right channel delay in milli seconds (can be negative), use this or option w.\n\
-e  integer right channel delay in samples (can be negative), use this or option d.\n\
-k  seek in left channel if delay < 0,\n\
    do not use left channel start as reference.\n\
    (Normaly for d < 0 a silence would be inserted in the right channel).\n\
-l  left channel filename.\n\
-m  output is right channel only mono.\n\
-o  output filename (if none specified then stdout is used).\n\
-r  right channel filename.\n\
-s  offset in both files from start in bytes\n\
 TYPE '?' during running to view these comands.\n\
");

running_info();
} /* end function usage */



char *strsave(char *s)
/* save string s somewhere */
{
char *p;

p = malloc(strlen(s) + 1);
if(!p) return 0;
strcpy(p, s);

return p;
} /* end function strsave */


int main(int argc, char **argv)
{
int a, b, i, l, r;
double da, db;
int error;
int le, re;
char *left_input_filename, *right_input_filename, *output_filename;
wave_header *left_header, *right_header, *out_header;
FILE *pfleft, *pfright, *pfout;
int dsp_speed;
int sample_size;
int dsp_stereo;
int header_size;
int out_bytes;
char *left_buffer, *right_buffer, *out_buffer;
char *ptr;
double delay_right, delay_man;
off_t offset, roffset, man_offset, total_offset;
off_t start_offset;
int right_insert_length, right_insert_pos;
int in_buffer_size, out_buffer_size;
int bytes_passed;
int keep_left_reference;
int right_channel_only_mono_flag;
int adjust_flag;
struct pollfd pfd[1];
int total;
int length;
int audiofd;
char *device;
int tmp;
int abuf_size;  
char *audiobuf;
char *pabuf;
char ca;
int both_flag;
fd_set readfs;
struct timeval timeout;
int samples_man;

// -Wall
l = 0;
r = 0;
da = 0.0;
audiofd = -1;
audiobuf = 0;
// end -Wall

/* select con io 1 byte at the time */
setbuf(stdout, NULL);
setbuf(stdin, NULL);

fprintf(stderr, "\ncombine_wave-%s Copyright 2004-always Jan Panteltje.\n\n", VERSION);

/* defaults */
debug_flag = 0;
left_input_filename = NULL;
right_input_filename = NULL;
output_filename = NULL;
delay_right = 0.0;
keep_left_reference = 1;
right_channel_only_mono_flag = 0;
start_offset = 0;
adjust_flag = 0;
device = strsave(DSP_DEVICE);
if(! device)
	{
	fprintf(stderr, "could not allocate memory for device, aborting.\n");

	exit(1);
	}

both_flag = 0;
samples_man = 0;
offset = 0;
/* end defaults */

while(1)
	{
	a = getopt(argc, argv, "ad:e:kl:mo:r:s:");
	if(a == -1) break;

	switch(a)
		{
        case -1:
        	break;
		case 'a':
			adjust_flag = 1;
			break;
		case 'd':
			delay_right = atof(optarg);
			break;
		case 'e':
			offset = atoi(optarg);	
			break;
		case 'k':
			keep_left_reference = 0;
			break;
		case 'l':
			left_input_filename = strsave(optarg);
			if(! left_input_filename)
				{
				fprintf(stderr,\
				"combine_wave: could not allocate space for left_input_filename.\n");

				exit(1);
				}
			break;
		case 'm':
			right_channel_only_mono_flag = 1 - right_channel_only_mono_flag;
			break;
		case 'r':
			right_input_filename = strsave(optarg);
			if(! right_input_filename)
				{
				fprintf(stderr,\
				"combine_wave: could not allocate space for right_input_filename.\n");

				exit(1);
				}
			break;
		case 's':
			start_offset = atol(optarg);
			break;
		case 'o':
			output_filename = strsave(optarg);
			if(! output_filename)
				{
				fprintf(stderr,\
		"combine_wave: could not allocate space for output_filename.\n");

				exit(1);
				}
			break;
		default:
			fprintf(stderr, "combine_wave: unknown option.\n");
			usage();
			exit(1);
			break;
		}/* end switch a */
	}/* end while getopt() */

error = 0;
if(! left_input_filename)
	{
	fprintf(stderr,\
	"combine_wave: No left input filename specified.\n");
	error = 1;
	}

if(! right_input_filename)
	{
	fprintf(stderr,\
	"combine_wave: No right input filename specified.\n");
	error = 1;
	}

if(! output_filename)
	{
	fprintf(stderr,\
	"combine_wave: No output filename specified, using stdout\n");
//	error = 1;
	}

if(error)
	{
	usage();
	exit(1);
	}

/* allocate space for left input wave header */
left_header = malloc( sizeof(wave_header) );
if(! left_header)
	{
	fprintf(stderr,\
	"combine_wave: could not allocate space for left_header.\n");

	exit(1);
	}

/* allocate space for right input wave header */
right_header = malloc( sizeof(wave_header) );
if(! right_header)
	{
	fprintf(stderr,\
	"combine_wave: could not allocate space for right_header.\n");

	exit(1);
	}

/* allocate space for output wave header */
out_header = malloc( sizeof(wave_header) );
if(! out_header)
	{
	fprintf(stderr,\
	"combine_wave: could not allocate space for out_header.\n");

	exit(1);
	}

pfleft = fopen(left_input_filename, "r");
if(! pfleft)
	{
	fprintf(stderr,\
	"combine_wave: could not open left input file %s for read, aborting.\n", 
	left_input_filename);

	exit(1);
	}

pfright = fopen(right_input_filename, "r");
if(! pfright)
	{
	fprintf(stderr,\
	"combine_wave: could not open right input file %s for read, aborting.\n", 
	right_input_filename);

	exit(1);
	}

if(! output_filename)
	{
	pfout = stdout;
	}
else
	{
	pfout = fopen(output_filename, "w");
	if(! pfout)
		{
		fprintf(stderr,\
		"combine_wave: could not open output file %s for read, aborting.\n", 
		output_filename);

		exit(1);
		}
	}

header_size = sizeof(wave_header);

//fprintf(stderr, "combine_wave: header_size=%d\n", header_size);
//fprintf(stderr, "combine_wave: sizof(char)=%d\n", sizeof(char) );

/* read left input file header */
a = fread(left_header, sizeof(char), header_size, pfleft);

if(a != header_size)
	{
	fprintf(stderr,\
	"combine_wave: could only read %d bytes of left input file header, aborting.\n", a);
	exit(1);
	}
					
/* set up audio if in adjust mode */
if(adjust_flag)
	{
	dsp_speed = left_header -> sample_fq;   
	dsp_stereo = 1; //left_header -> modus - 1;
	sample_size = (left_header -> byte_p_spl * 8) / left_header -> modus;

	fprintf(stderr, "dsp_speed=%d dsp_stereo=%d sample_size=%d\n", dsp_speed, dsp_stereo, sample_size);
 
	audiofd = open (device, O_WRONLY, 0);
	if(audiofd < 0)
		{
		fprintf(stdout,\
		" could not open device %s for read, aborted\n",\
		device);

		return 0;
		}

	tmp = sample_size;
	ioctl(audiofd, SNDCTL_DSP_SAMPLESIZE, &tmp);
	if (tmp != sample_size)
		{
		close(audiofd);

		fprintf(stdout,\
		"play_file(): could not set sample_size %d for %s aborted\n",\
		sample_size, device);

		return 0;
		}

	if (ioctl (audiofd, SNDCTL_DSP_STEREO, &dsp_stereo) == -1)
		{
		close(audiofd);

		fprintf(stdout,\
		"play_file(): Could not set stereo / mono mode for %s aborted\n",\
		device);

		return 0;
		}

	tmp = dsp_speed;
	if (ioctl (audiofd, SNDCTL_DSP_SPEED, &tmp) == -1)
		{
		close(audiofd);

		fprintf(stdout,\
		"play_file(): could not set sample speed %d for %s aborted\n",\
		dsp_speed, device);

		return 0;
		}

	ioctl (audiofd, SNDCTL_DSP_GETBLKSIZE, &abuf_size);
	if (abuf_size < 1)
		{
		close(audiofd);

		fprintf(stdout,\
		"play_file(): could not get blocksize for %s aborted\n",\
		device);

		return 0;
		}

	if ((audiobuf = malloc (abuf_size)) == NULL)
		{
		close(audiofd);

		fprintf(stdout,\
		"play_file(): could not allocate buffer, aborted\n");

		return 0;
		}
	} /* end if adjust_flag */

/* read right input file header */
a = fread(right_header, sizeof(char), header_size, pfright);

if(a != header_size)
	{
	fprintf(stderr,\
	"combine_wave: could only read %d bytes of right input file header, aborting.\n", a);
	exit(1);
	}

/* check if left channel mono */
if(left_header -> modus != 1)
	{
	fprintf(stderr,\
	"combine_wave: left input channel is not mono, aborting.\n");

	exit(1);
	}

/* check if right channel mono */
if(right_header -> modus != 1)
	{
	fprintf(stderr,\
	"combine_wave: right input channel is not mono, aborting.\n");

	exit(1);
	}

/* check if sample rates are the same */
if(left_header -> sample_fq != right_header -> sample_fq)
	{
	fprintf(stderr,\
	"combine_wave: input sample rates differ! (l=%d r=%d) aborting.\n",\
	left_header -> sample_fq, right_header -> sample_fq);

	exit(1);
	}

/* check if bytes per sample is 2 (16 bit mono) */
if(left_header -> byte_p_spl != 2)
	{
	fprintf(stderr,\
	"combine_wave: left channel not 2 bytes per sample.\n");

	exit(1);
	}

if(right_header -> byte_p_spl != 2)
	{
	fprintf(stderr,\
	"combine_wave: right channel not 2 bytes per sample.\n");

	exit(1);
	}

/* check if bits per sample is 16 */
if(left_header -> bit_p_spl != 16)
	{
	fprintf(stderr,\
	"combine_wave: left channel not 16 bits per sample.\n");

	exit(1);
	}

if(right_header -> bit_p_spl != 16)
	{
	fprintf(stderr,\
	"combine_wave: right channel not 16 bits per sample.\n");

	exit(1);
	}

/* check if bytes per sample the same */
if(left_header -> byte_p_spl != right_header -> byte_p_spl)
	{
	fprintf(stderr,\
"combine_wave: input bytes per sample differ! (l=%d r=%d) aborting.\n",\
	left_header -> byte_p_spl, right_header -> byte_p_spl);

	exit(1);
	}

/* check if bits per sample the same */
if(left_header -> bit_p_spl != right_header -> bit_p_spl)
	{
	fprintf(stderr,\
"combine_wave: input bytes per sample differ! (l=%d r=%d) aborting.\n",\
	left_header -> bit_p_spl, right_header -> bit_p_spl);

	exit(1);
	}

/* allocate space for buffers */
/* this must be a multiple of 4 */
#define IN_BUFFER_SIZE 4*32768

if(delay_right != 0.0)
	{
	if(offset)
		{
		fprintf(stderr, "combine_wave: both -d and -e flag specified, use only one of these, aborting.\n");

		exit(1);
		}

	/* bytes per millisecond */
	da = (double)left_header -> byte_p_sec / 1000.0;

	/* bytes to shift in audio */

	db = da * delay_right;

	offset = abs( (off_t)db );

//fprintf(stderr, "combine_wave: uncorrected offset=%d\n", offset);
	}
else
	{
	offset *= (double)left_header -> byte_p_spl;

	delay_right = offset * (1000.0 / (double)left_header -> byte_p_sec);

	offset = abs(offset);
	}

/*
offset must be a  multiple of 2
*/
a = offset % 2;
if(a) offset += 2 - a;

fprintf(stderr,\
"combine_wave: delay_right=%.2f da=%.2f corrected offset=%ld\n", delay_right, da, (long)offset);

if( (delay_right > 0) && (keep_left_reference) )
	{
	/*
	the trick used is here that the inserted bytes are exactly buffer size,
	so one empty buffer is inserted...
	*/

	right_insert_length = offset;

	in_buffer_size = right_insert_length;
	}
else
	{
	right_insert_length = 0;

	in_buffer_size = IN_BUFFER_SIZE;
	}

if(right_channel_only_mono_flag) out_buffer_size = in_buffer_size;
else out_buffer_size = in_buffer_size * 2;

left_buffer = malloc(in_buffer_size);
if(! left_buffer)
	{
	fprintf(stderr,\
	"combine_wave: could allocate left_buffer, aborting.\n");

	exit(1);
	}

right_buffer = malloc(in_buffer_size);
if(! right_buffer)
	{
	fprintf(stderr,\
	"combine_wave: could allocate right_buffer, aborting.\n");

	exit(1);
	}

out_buffer = malloc(out_buffer_size);
if(! out_buffer)
	{
	fprintf(stderr,\
	"combine_wave: could allocate out_buffer, aborting.\n");

	exit(1);
	}

/*
copy left wave header parameters to out wave header, but change 4 things:
file length is updated,
modus = 2, stereo,
bytes_p_spl *= 2,
and data_length is the longest of the two input files.
*/
ptr = (char *)out_header;
for(i = 0; i < header_size; i++)
	{
	*ptr = 0;
	ptr++;
	}
strcpy(out_header -> main_chunk, "RIFF");
strcpy(out_header -> chunk_type, "WAVE"); 
strcpy(out_header -> sub_chunk, "fmt ");
out_header -> length_chunk = 16; //always 16
out_header -> format = 1; //PCM

if(! right_channel_only_mono_flag)
	{
	out_header -> modus = 2; // stereo
	out_header -> byte_p_sec = left_header -> byte_p_sec * 2;
	out_header -> byte_p_spl = 4; // 16 bit stereo
	}
else
	{
	out_header -> modus = 1; // mono
	out_header -> byte_p_sec = left_header -> byte_p_sec;
	out_header -> byte_p_spl = left_header -> byte_p_spl;
	}

out_header -> sample_fq = left_header -> sample_fq;
out_header -> bit_p_spl = 16; // left_header -> bit_p_spl;
strcpy(out_header -> data_chunk, "data");

if(right_header -> data_length > left_header -> data_length) 
	{
	a = right_header -> data_length;
	}
else
	{
	a = left_header -> data_length;
	}

a += right_insert_length;

if(! right_channel_only_mono_flag) a *= 2;

out_header -> data_length = a;

out_header -> length = out_header -> data_length + 36; // FIXME ?

/* seek start_offset bytes in left channel from header end */
a = fseek(pfleft, start_offset, SEEK_CUR);	
if(a)
	{
	fprintf(stderr,\
	"combine_wave: could not set left channel offset %ld, aborting.\n", (long)offset);

	exit(1);
	}

/* seek start_offset bytes in right channel from header end */
a = fseek(pfright, start_offset, SEEK_CUR);	
if(a)
	{
	fprintf(stderr,\
	"combine_wave: could not set right channel offset %ld, aborting.\n", (long)offset);
		
	exit(1);
	}

fprintf(stderr, "combine_wave: offset in both channels %ld bytes.\n", (long)start_offset);

if(delay_right > 0)
	{
	if(! keep_left_reference)
		{
		/* seek in left channel from header end */
		a = fseek(pfleft, offset, SEEK_CUR);	
		if(a)
			{
			fprintf(stderr,\
			"combine_wave: could not set left channel offset %ld, aborting.\n", (long)offset);

			exit(1);
			}

		fprintf(stderr,\
		"combine_wave: starting left channel %.2f milli seconds early.\n",\
		fabs(delay_right) );
		} /* end if ! keep_left_reference */
	else /* keep left reference */
		{
		if(delay_right > 0)
			{
			/* insert silence at start of left channel */

			fprintf(stderr,\
			"combine_wave: inserting %.4f milli seconds silence at start of right channel\n",\
			fabs(delay_right) );
			}
		} /* end if keep_left_reference */
	} /* end if delay_right > 0 */

if(delay_right < 0)
	{
	/* seek in right channel from header end */
	roffset = fseek(pfright, offset, SEEK_CUR);	
	if(roffset)
		{
		fprintf(stderr,\
		"combine_wave: could not set right channel offset %ld, aborting.\n", (long)offset);
		
		exit(1);
		}

	fprintf(stderr,\
	"combine_wave: starting right channel %.4f milli seconds early.\n",\
	fabs(delay_right) );
	} /* end if delay_right < 0 */

if(! adjust_flag)
	{
	/* write out header */
	a = fwrite(out_header, sizeof(char), header_size, pfout);
	if(a != header_size)
		{
		fprintf(stderr,\
		"combine_wave: could only write %d of %d bytes of out header, aborting.\n",\
		a, header_size);   

		exit(1); 
		}
	} /* end if ! adjust_flag */

pfd[0].fd = fileno(stdin);
pfd[0].events = POLLOUT;

bytes_passed = 0;
out_bytes = 0;
right_insert_pos = 0;
le = 0;
re = 0;
total_offset = offset;
while(1)
	{
	if(adjust_flag)
		{
		delay_man = 0.0;

		/* read a user key */
		system("stty raw");
	
		FD_ZERO(&readfs);
		FD_SET(fileno(stdin), &readfs);
		timeout.tv_sec = 0;
		timeout.tv_usec = 0;
		b = select(FD_SETSIZE, &readfs, NULL, NULL, &timeout);
		if(b == -1)
			{
			fprintf(stderr, "select returned -1, aborting\n");

			exit(1);
			}

		if( FD_ISSET(fileno(stdin), &readfs) )
			{
			a = read( fileno(stdin), &ca, 1);

			FD_CLR(fileno(stdin), &readfs);

			system("stty sane");

			if(ca == 27)
				{
				fprintf(stderr, "user abort\n");

				exit(1);
				}

			if(ca == 'b')
				{
				both_flag = 1 - both_flag;

				if(both_flag) fprintf(stderr, "moving both channels mode selected.\n");
				else fprintf(stderr, "moving only right channel mode selected.\n");
				}
			else if(
			(ca == '[') || (ca == ']') ||
			(ca == '{') || (ca == '}') ||
			(ca == ',') || (ca == '.') || 
			(ca == '<') || (ca == '>') ||
			(ca == '-') || (ca == '=') ||
			(ca == '_') || (ca == '+') ||
            (ca == 'z') || (ca == 'x') ||
            (ca == 'Z') || (ca == 'X')
			)
				{
				if(     ca == '[') delay_man = -1.0;
				else if(ca == ']') delay_man = +1.0;
				else if(ca == '{') delay_man = -10.0;
				else if(ca == '}') delay_man = +10.0;
				else if(ca == ',') delay_man = -100.0;
				else if(ca == '.') delay_man = +100.0;
				else if(ca == '<') delay_man = -1000.0;
				else if(ca == '>') delay_man = +1000.0;
				else if(ca == '-') delay_man = -10000.0;
				else if(ca == '=') delay_man = +10000.0;
				else if(ca == '_') delay_man = -100000.0;
				else if(ca == '+') delay_man = +100000.0;
				else if(ca == 'z') samples_man = -1;
				else if(ca == 'x') samples_man = +1;
				else if(ca == 'Z') samples_man = -10;
				else if(ca == 'X') samples_man = +10;

				fprintf(stderr, "ca=%d(%c)\n", ca, ca);
	
				/* bytes per millisecond */
				da = (double)left_header -> byte_p_sec / 1000.0;

				/* bytes to shift in audio */
				db = da * delay_man;

				man_offset = (off_t)db;

				/* offset must be a  multiple of 4 */

				a = man_offset % 2;
				if(a) man_offset += 2 - a;

				man_offset += (samples_man * left_header -> byte_p_spl);

				/* offset in time (ms)*/
				if(! both_flag) delay_right += delay_man;

				/* seek in right channel from position */
				a = fseek(pfright, man_offset, SEEK_CUR);	
				if(a)
					{
					fprintf(stderr,\
					"combine_wave: could not set right channel man_offset %ld, aborting.\n",\
					(long)man_offset);
		
					exit(1);
					}

				total_offset += man_offset;

				/* calculate total offset in float milliseconds */
				da = (double) total_offset / ( (double)left_header -> byte_p_sec / 1000.0);

				fprintf(stderr, "offset=%ld bytes  %ld samples  %.2fms\n",\
				(long)total_offset, (long)((long)total_offset / left_header -> byte_p_spl), da);

				if(both_flag)
					{
					/* seek in left channel from position */
					a = fseek(pfleft, man_offset, SEEK_CUR);	
					if(a)
						{
						fprintf(stderr,\
						"combine_wave: could not set left channel man_offset %ld, aborting.\n",\
						(long)man_offset);
		
						exit(1);
						}
					} /* end if both_flag */

				delay_man = 0.0;

				samples_man = 0;

				} /* end if man_offset change */
			else
				{
				running_info();
				}
			} /* end if key */
		else /* a = 0 */
			{
			system("stty sane");
			/* no data available */
			}

		} /* end if adjust_flag */

	if(! le)
		{
		l = fread(left_buffer, sizeof(char), in_buffer_size, pfleft);
		if(l != in_buffer_size)
			{
			if(feof(pfleft) )
				{
				fclose(pfleft);
				le = 1;
				}
			else
				{
				perror("left read: ");
				fprintf(stderr,\
				"combine_wave: error reading left file, aborting.\n"); 

				exit(1);
				}
			} /* end if l != requested */
		} /* end if ! le */

	if(! re)
		{
		if(right_insert_pos < right_insert_length)
			{
fprintf(stderr, "WAS inserting %d bytes silence\n", in_buffer_size);

			/*
			dummy read, insert an empty buffer, just the size of the
			silence.
			*/
			for(i = 0; i < in_buffer_size; i++) right_buffer[i] = 0;			
			right_insert_pos = right_insert_length;
			}
		else
			{
			r = fread(right_buffer, sizeof(char), in_buffer_size, pfright);
			if(r < in_buffer_size)
				{
				if(feof(pfright) )
					{
					fclose(pfright);
					re = 1;
					}
				else
					{
					perror("right read: ");
					fprintf(stderr,\
					"combine_wave: error reading right file, aborting.\n"); 

					exit(1);
					}
				} /* end if r != requested */
			} /* end if not in right insert mode */
		} /* end if ! re */

	if(right_channel_only_mono_flag)
		{
		/* add right channel to output buffer */
		for(i = 0; i < in_buffer_size; i++)
			{
			if(i < r)
				{
				out_buffer[i] = right_buffer[i];
				}
			else
				{
				out_buffer[i] = 0;
				}
			} /* end for i */
		} /* end if right_channel_only_mono_flag */
	else /* stereo 2 channels */
		{
		/* add left channel to output buffer */
		for(i = 0; i < in_buffer_size - 1; i += 2)
			{
			if(i < l)
				{		
				out_buffer[i * 2] = left_buffer[i];
				out_buffer[(i * 2) + 1] = left_buffer[i + 1];
				}
			else
				{
				out_buffer[i * 2] = 0;
				out_buffer[(i * 2) + 1] = 0;
				}
			} /* end for i */

		/* add right channel to output buffer */
		for(i = 0; i < in_buffer_size - 1; i += 2)
			{
			if(i < r)
				{
				out_buffer[(i * 2) + 2] = right_buffer[i];
				out_buffer[(i * 2) + 3] = right_buffer[i + 1];
				}
			else
				{
				out_buffer[(i * 2) + 2] = 0;
				out_buffer[(i * 2) + 3] = 0;
				}
			} /* end for i */
		} /* end if ! right_channel_only_mono_flag */

	/* test if to dsp device */
	if(adjust_flag)
		{
//fprintf(stderr, "requested=%d\n", out_buffer_size);
		total = out_buffer_size;
		ptr = out_buffer;
		while(1)
			{
			memcpy(audiobuf, ptr, abuf_size);

			pabuf = audiobuf;
			a = abuf_size;
			while(1)
				{
				if(total <= 0) break;
				errno = 0;
				length = write (audiofd, pabuf, a);

				if(length < 0)
					{
					if(errno == EAGAIN)
						{
						usleep(10000);					

						continue;
						}
					else if(errno == EINTR)
						{
						usleep(10000);

						continue;
						}	
					else
						{
						perror("write(): ");
						fprintf(stdout, "write failed to %s, aborted\n",  device);
	
						close(audiofd);

						exit(1);
						}
					} /* end if read returned < 0 */
				else if(length == 0) break;
				else
					{
					a -= length;
					if(a == 0) break;

					pabuf += length;
					}
				} /* end while try write  */

			
			total -= abuf_size;
			if(total == 0) break;
//fprintf(stderr, "left to send=%d\n", total);

			ptr += abuf_size;
			} /* end while output out_buffer to audio device */ 
		
		} /* end if adjust_flag (to dsp device) */
	else /* to file or stdout */
		{
		a = fwrite(out_buffer, sizeof(char), out_buffer_size, pfout);
		if(a != out_buffer_size)
			{
			fprintf(stderr,\
			"combine_wave: could only write %d of %d bytes, aborting.\n",\
			a, out_buffer_size);			

			exit(1);
			}
		} /* end if to file or stdout */

	out_bytes += out_buffer_size;

	/* print progress after each MB */
	bytes_passed += out_buffer_size;
	if(bytes_passed > 1000000)
		{
		fprintf(stderr,\
		"combine_wave: %d bytes (%d MB) written\r",\
		out_bytes, out_bytes / 1000000);

		bytes_passed = 0;
		}

	if( (le == 1) && (re == 1) ) break;
	} /* end while */

if(adjust_flag)
	{
	close(audiofd);
	}

fprintf(stderr, "\nReady\n");

exit(0);
} /* end function main */

