# SPDX-License-Identifier: Apache-2.0
#
# The OpenSearch Contributors require contributions made to
# this file be licensed under the Apache-2.0 license or a
# compatible open source license.
#
# Modifications Copyright OpenSearch Contributors. See
# GitHub history for details.
#
#  Licensed to Elasticsearch B.V. under one or more contributor
#  license agreements. See the NOTICE file distributed with
#  this work for additional information regarding copyright
#  ownership. Elasticsearch B.V. licenses this file to you under
#  the Apache License, Version 2.0 (the "License"); you may
#  not use this file except in compliance with the License.
#  You may obtain a copy of the License at
#
# 	http://www.apache.org/licenses/LICENSE-2.0
#
#  Unless required by applicable law or agreed to in writing,
#  software distributed under the License is distributed on an
#  "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
#  KIND, either express or implied.  See the License for the
#  specific language governing permissions and limitations
#  under the License.

from datetime import datetime, timedelta
from typing import Any

import pytest

from opensearchpy import Range


@pytest.mark.parametrize(  # type: ignore
    "kwargs, item",
    [
        ({}, 1),
        ({}, -1),
        ({"gte": -1}, -1),
        ({"lte": 4}, 4),
        ({"lte": 4, "gte": 2}, 4),
        ({"lte": 4, "gte": 2}, 2),
        ({"gt": datetime.now() - timedelta(seconds=10)}, datetime.now()),
    ],
)
def test_range_contains(kwargs: Any, item: Any) -> None:
    assert item in Range(**kwargs)


@pytest.mark.parametrize(  # type: ignore
    "kwargs, item",
    [
        ({"gt": -1}, -1),
        ({"lt": 4}, 4),
        ({"lt": 4}, 42),
        ({"lte": 4, "gte": 2}, 1),
        ({"lte": datetime.now() - timedelta(seconds=10)}, datetime.now()),
    ],
)
def test_range_not_contains(kwargs: Any, item: Any) -> None:
    assert item not in Range(**kwargs)


@pytest.mark.parametrize(  # type: ignore
    "args,kwargs",
    [
        (({},), {"lt": 42}),
        ((), {"not_lt": 42}),
        ((object(),), {}),
        ((), {"lt": 1, "lte": 1}),
        ((), {"gt": 1, "gte": 1}),
    ],
)
def test_range_raises_value_error_on_wrong_params(args: Any, kwargs: Any) -> None:
    with pytest.raises(ValueError):
        Range(*args, **kwargs)


@pytest.mark.parametrize(  # type: ignore
    "range,lower,inclusive",
    [
        (Range(gt=1), 1, False),
        (Range(gte=1), 1, True),
        (Range(), None, False),
        (Range(lt=42), None, False),
    ],
)
def test_range_lower(range: Any, lower: Any, inclusive: Any) -> None:
    assert (lower, inclusive) == range.lower


@pytest.mark.parametrize(  # type: ignore
    "range,upper,inclusive",
    [
        (Range(lt=1), 1, False),
        (Range(lte=1), 1, True),
        (Range(), None, False),
        (Range(gt=42), None, False),
    ],
)
def test_range_upper(range: Any, upper: Any, inclusive: Any) -> None:
    assert (upper, inclusive) == range.upper
