/*-------------------------------------------------------------------------
 *  sgerr.c 
 *  
 *  Module to report sense key errors for SCSI routines.
 *
 *  04/30/01 ARCress - created
 *  10/11/01 ARCress - minor changes and streamlining
 *  10/25/01 ARCress - took out conditional defines
 *  10/26/01 ARCress - moved strings & structures to scsierr.h
 *-------------------------------------------------------------------------*/
/*-------------------------------------------------------------------------
Copyright (c) 2002-2008, Intel Corporation
All rights reserved.

Redistribution and use in source and binary forms, with or without 
modification, are permitted provided that the following conditions are met:

  a.. Redistributions of source code must retain the above copyright notice, 
      this list of conditions and the following disclaimer. 
  b.. Redistributions in binary form must reproduce the above copyright notice,
      this list of conditions and the following disclaimer in the documentation 
      and/or other materials provided with the distribution. 
  c.. Neither the name of Intel Corporation nor the names of its contributors 
      may be used to endorse or promote products derived from this software 
      without specific prior written permission. 

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND 
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE 
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR 
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES 
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; 
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON 
ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS 
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
  -------------------------------------------------------------------------*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <scsi/sg.h>
#include <scsi/scsi.h>
#include "scsierr.h"
static int host_max = 0, driver_max = 0, suggest_max = 0;

/*
 * Local Subroutines
 */

void show_host_status(FILE * fdout, int host_status)
{
    int i;
    char *phost;
    if (!host_max) {
	for (i = 0; hostbyte_table[i]; i++);
	host_max = i - 1;
    }
    if (host_status > host_max)
	phost = (char *) unknown;
    else
	phost = (char *) hostbyte_table[host_status];
    fprintf(fdout, "Host_status=0x%02x (%s) ", host_status, phost);
    return;
}

void show_driver_status(FILE * fdout, int driver_status)
{
    char *pdr;
    char *psu;
    int i;
    int dr = driver_status & SG_ERR_DRIVER_MASK;
    int su = (driver_status & SG_ERR_SUGGEST_MASK) >> 4;
    if (!driver_max) {
	for (i = 0; driverbyte_table[i]; i++);
	driver_max = i - 1;
	for (i = 0; driversuggest_table[i]; i++);
	suggest_max = i - 1;
    }
    if (dr > driver_max)
	pdr = (char *) unknown;
    else
	pdr = (char *) driverbyte_table[dr];
    if (su > suggest_max)
	psu = (char *) unknown;
    else
	psu = (char *) driversuggest_table[su];
    fprintf(fdout, "Driver_status=0x%02x (%s,%s)", driver_status, pdr,
	    psu);
    return;
}

/* Show sense information */
void show_sense_err(FILE * fdout, const char *leadin,
		    const unsigned char *sense_buffer, int sb_len)
{
    int i, s, k;
    int sense_class, valid, code;
    int asc, ascq;
    const char *error = NULL;
    char bytestr[30];
    char bitstr[30];
    char estr[80];
    char *estr1;
    char fdumpbufr;
    char foundtxt;

    fdumpbufr = 0;
    if (fdout == NULL) fdout = stderr;
    sense_class = (sense_buffer[0] >> 4) & 0x07;
    code = sense_buffer[0] & 0x0f;
    valid = sense_buffer[0] & 0x80;
    if (sense_class == 7) {		/* extended sense data */
	k = sense_buffer[2] & 0x0f;	/* sense key */
	s = sense_buffer[7] + 8;    /* length = addtl sense + sense data(8) */
	if (s > sb_len)
	    s = sb_len;		/* truncate if buffer too small */
	if (s > 13) {		/* get ASC & ASCQ */
	    asc = sense_buffer[12];
	    ascq = sense_buffer[13];
	} else {
	    asc = 0;
	    ascq = 0;
	}
	if (leadin)
	    fprintf(fdout, "%s: ", leadin);
	fprintf(fdout, "sense error, key=%02x asc=%02x ascq=%02x %s",
		k, asc, ascq, snstext[k]);
	/* Check to see if additional sense information is valid. */
	if (asc == 0 && ascq == 0) {	
	    /* addtl sense not helpful, just dump out the buffer */
	    fprintf(fdout, "\n");
	    fdumpbufr = 1;
	}
	else {			/* show the additional sense fields */
	    foundtxt = 0;
	    for (i = 0; additional[i].text; i++)
		if (additional[i].code1 == asc &&
		    additional[i].code2 == ascq) {
			fprintf(fdout, ", %s\n", additional[i].text);
			foundtxt = 1;
		}
	    if (foundtxt == 0)  /* not found yet, check ascq ranges */
	    for (i = 0; additional2[i].text; i++)   /* additional2 messages */
		if (additional2[i].code1 == asc &&
		    additional2[i].code2_min >= ascq &&
		    additional2[i].code2_max <= ascq) {
		    fprintf(fdout, ", ");
		    fprintf(fdout, additional2[i].text, ascq);
		    fprintf(fdout, "\n");
		}
	    /* Illegal Request & Sense-Key Specific Field Valid bit is set */
	    /* Note that getting byte# requires up to 18 sense bytes. */
	    if (k == 0x05 && (sense_buffer[15] & 0x80) == 0x80) {
		/* Now find the invalid byte in parameter list or CDB */
		if ((sense_buffer[15] & 0x40) == 0x40)
		    estr1 = "Error in CDB: ";
		else
		    estr1 = "Error in Parameter List: ";
		if (s >= 18) {	/* enough sense data to get byte number */
		    sprintf(bytestr, "Byte #%d",
			    sense_buffer[16] * 0x100 + sense_buffer[17]);
		} else          /* not enough sense data */
		    strcpy(bytestr, "Byte #?");
		if ((sense_buffer[15] & 0x08) == 0x08) /*Bit Pointer Valid?*/
		    sprintf(bitstr, ", Bit #%d is in error.\n",
			    sense_buffer[15] & 0x07);
		else
		    sprintf(bitstr, " is in error.\n");
		fprintf(fdout, "%s%s%s", estr1, bytestr, bitstr);
	    }			/*endif */
	    else fprintf(fdout,", ");
	}			/* end-else show addtl sense */
	/* show the Info field, etc. */
	if (valid)
	    sprintf(estr, "Info fld=0x%08x, ",
		    (int) ((sense_buffer[3] << 24) |
			   (sense_buffer[4] << 16) | (sense_buffer[5] << 8)
			   | sense_buffer[6]));
	else
	    strcpy(estr, "Info valid=0, ");
	if (sense_buffer[2] & 0x80)
	    strcat(estr, "FMK ");    /* current command has read a filemark */
	if (sense_buffer[2] & 0x40)
	    strcat(estr, "EOM ");    /* end-of-medium condition exists */
	if (sense_buffer[2] & 0x20)
	    strcat(estr, "ILI ");    /* incorrect block length requested */
	switch (code) {
	case 0x0:
	    error = "Current ";	/* error concerns current command */
	    break;
	case 0x1:
	    error = "Deferred";	/* error concerns some earlier command */
	    /* e.g., an earlier write to disk cache succeeded, but
	       now the disk discovers that it cannot write the data */
	    break;
	case 0xf:
	    error = "Vendor  ";	/* vendor specific */
	    break;
	default:
	    error = "RsvdCode";	/* reserved response code */
	}
	fprintf(fdout, "%s%s\n", estr, error);
    } else {			/* non-extended sense data */
	/*
	 * Use SCSI-1 Standard, where:
	 *    sense_buffer[0] & 0x80 : address valid
	 *    sense_buffer[0] & 0x7F : sense code
	 *    sense_buffer[1..3] : 21-bit logical block address
	 * OR
	 *    sense_buffer[0] = 0x7F : vendor-specific error code
	 *    sense_buffer[1] & 0xF0 : vendor-specific code
	 */
	if (leadin)
	    fprintf(fdout, "%s: ", leadin);
	k = sense_buffer[0] & 0x0f;	/* sense key */
	fprintf(fdout, "old sense = %2x %2x %s\n",
		sense_buffer[0], sense_buffer[1], snstext[k]);
	if (valid)
	    fprintf(fdout, "LBA: %02x%02x%02x\n",
		    sense_buffer[1], sense_buffer[2], sense_buffer[3]);
	s = 4;
    }
    if (fdumpbufr) {
	fprintf(fdout, "Raw sense data (in hex):\n  ");
	for (i = 0; i < s; ++i) {
	    if ((i > 0) && (0 == (i % 24)))
		fprintf(fdout, "\n  ");
	    fprintf(fdout, "%02x ", sense_buffer[i]);
	}
	fprintf(fdout, "\n");
	/* TODO: set up a way to dump the buffer if debug is on. */
    }
    return;
}				/* end show_sense_err() */

int sg_chk_err(FILE * fdout, const char *leadin, int masked_status,
	       int host_status, int driver_status,
	       const unsigned char *sense_buffer, int sb_len)
{
    int ret = 0;
    if ((0 == masked_status) && (0 == host_status) && (0 == driver_status))
	return 0;		/* OK, No problems */
    if (fdout == NULL)
	return 1;		/* don't show anything */
    fprintf(fdout, "%s: ", leadin);
    if (0 != masked_status) {
	fprintf(fdout, "%s ", statuses[masked_status]);
	ret = 2;
	}
    if (0 != host_status) {
	show_host_status(fdout, host_status);
	ret = 3;
	}
    if (0 != driver_status) {
	show_driver_status(fdout, driver_status);
	ret = driver_status;
	}
    fprintf(fdout, "\n");
    if (sense_buffer && (sense_buffer[0] != 0)) {
	show_sense_err(fdout, 0, sense_buffer, sb_len);
	if (ret == 0) ret = -2;
        /* check for specific sense codes here? */
	}
    return(ret);
}				/* end sg_chk_err() */

#ifdef TEST
#define uchar unsigned char

static uchar htoi(uchar *inhex)
{
   // char rghex[16] = "0123456789ABCDEF";
   uchar val;
   uchar c;
   c = inhex[0] & 0x5f;  /* force cap */
   if (c > '9') c += 9;  /* c >= 'A' */
   val = (c & 0x0f) << 4;
   c = inhex[1] & 0x5f;  /* force cap */
   if (c > '9') c += 9;  /* c >= 'A' */
   val += (c & 0x0f);
   return(val);
}

int main(int argc, char **argv)
{
   uchar buf[32];
   int i, len;
   // char msg[132];
 
   argc--; argv++;
   if (argc > 18) len = 18;
   else len = argc;
   if (len < 16) {
      printf("Need 16-18 bytes for sense data, got %d bytes input\n",len);
      printf("Usage: sgerr 01 02 03 04 05 06 07 08 09 0a 0b 0c 0d 0e 0f 10\n");
      exit(1);
   }
   for (i = 0; i < len; i++)
   {
      buf[i] = htoi(argv[i]);
   }
   show_sense_err(stdout, 0, buf, len);
   exit(0);
}
#endif
/* end sgerr.c */
