/*-------------------------------------------------------------------------- 
  Filename        : sgdskfl.c
  Abstract        : SCSI Generic Disk Firmware Load tool
  Operation System: Linux 2.2 or greater
  Author  	  : Andy Cress   <arcress@users.sourceforge.net>
  Copyright (c) Intel Corporation 2001-2008
  Copyright (c) 2009, Kontron America, Inc.
  ----------- Change History -----------------------------------------------
  05/01/01 v0.70 ARCress  created 
  07/17/01 v0.91 ARCress  added code to download servo also
                          added write_buffers code to do multi-part dl.
  08/08/01 v0.92 ARCress  fixed write_buffers logic for IBM disks.
  09/05/01 v0.93 ARCress  message cleanup
  09/28/01 v0.94 ARCress  turn off sg_debug in afterdl
  10/15/01 v0.95 ARCress  fixed IBM chunk size
  10/30/01 v1.0  ARCress  cleanup includes & indent -kr 
  11/09/01 v1.0  ARCress  enlarged devstat2 from 80 to 120 for overrun 
  11/14/01 v1.0  ARCress  added default directory for image files.
  04/11/02 v1.1  ARCress  path changes for log & image files,
                          set do_numeric=1 default
  05/08/02 v1.3  ARCress  tune sg_debug level down to 5 by default
  08/15/02 v1.4  ARCress  moved common subroutines to sgcommon
                          added more usage messages
  09/03/02 v1.5  ARCress  streamline display for errors in get_scsi_info 
  01/08/03 v1.6  ARCress  include Quantum in getimage:fscanmodel.
  07/26/04 v1.7  ARCress  added other IBM disk model parsing
  05/11/05 v1.8  ARCress  included patch for larger devlist from Nate Dailey
  02/02/07 v1.9  ARCress  larger disk_buffer for messages during writeimage(), 
                          fixed segfault if stale dbgout pointer,
                          added -i option.
  ----------- Description --------------------------------------------------
  sgdskfl  
Sequence of events for this utility:
  * List each device on the system with firmware versions.
  * User selects a device for firmware load (automatic if using -m)
  * Read the firmware image file for the selected disk and verify that it is 
    valid.  
  * Verify that the disk is present and ready
  * Close all open files, flush the adapter, sync any data to the SCSI disks.  
  * Write the firmware image to the disk using a 'write buffer' SCSI command
    with large reserved buffer or scatter/gather.
  * Wait 5 (or specified number) seconds
  * Verify that the disk comes ready again using SCSI test_unit_ready commands,
    and start_unit or scsi_reset to recover if not.
  * If the '-m' option was used, repeat writing the firmware for each disk of 
    this model.
sgdskfl    [-e -m diskmodel -f imagefile -t secdelay -x] 
Options:
  -e  	Do not write to any files. Usually a log file (sgdskfl.log) is created
    	and written to, up until the firmware download begins on root.
  -m  	Automatically download all drives that match this model string.
        Note that this option will initiate a reboot after it is done, 
        unless option u is also specified.
  -f 	Specify this filename for the firmware image.  Normally, this option is 
	not used and the filename is formed using the first 8 characters of the 
	model, with the ".lod" extension.  For example:  "st39140w.lod".   
        Note that this utility uses the raw firmware image without any added 
	headers.
  -t 	Specifies the number of seconds to delay after the firmware is written 
	and the program attempts to test if the unit is ready again.  
	Default is 10 seconds.
  -r    Recover a non-ready drive by updating its firmware.
        Don't test if the drive is ready or not.
  -x    Outputs extra debug messages
  -d    Specify a unix device name
----------------------------------------------------------------------------*/
/*-------------------------------------------------------------------------
Copyright (c) 2002, Intel Corporation
Copyright (c) 2009, Kontron America, Inc.
All rights reserved.
Redistribution and use in source and binary forms, with or without 
modification, are permitted provided that the following conditions are met:
  a.. Redistributions of source code must retain the above copyright notice, 
      this list of conditions and the following disclaimer. 
  b.. Redistributions in binary form must reproduce the above copyright notice,
      this list of conditions and the following disclaimer in the documentation 
      and/or other materials provided with the distribution. 
  c.. Neither the name of Intel Corporation nor the names of its contributors 
      may be used to endorse or promote products derived from this software 
      without specific prior written permission. 
THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND 
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE 
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR 
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES 
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; 
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON 
ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS 
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
  -------------------------------------------------------------------------*/
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <errno.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <sys/param.h>  /*PATH_MAX = 4096*/
#include <time.h>
#include <ctype.h>
#include <fcntl.h>
#include "sgsub.h"
#include "sgcommon.h"

/* External Functions prototypes */
extern void setbuf(FILE * stream, char *buf);
extern int errno;
extern char fdebugsub;		/* used in sgsub.c */
extern FILE *dbgout;		/* used in sgsub.c */

#define  EBUFF_LEN   80
#define  MAX_ERRORS   4
#define  MAX_PATHSZ  PATH_MAX /* use param.h max if user-specified filename */
#define  MAX_PATHSM  132      /* scsi device name max, devfs could be long */
#define  MAX_MODELSZ 20

/* Global data definitions */
char *progver  = "1.66";		/* program version */
char *progname = "sgdskfl";	/* program name */
char logfile[] = "/var/log/sgdskfl.log";	/* log filename */
char fwsufx[] = ".lod";		/* suffix for firmware image filenames */
char model[MAX_MODELSZ] = "";	/* default model string */
char fsetfile = 0;
char fwfile[MAX_PATHSZ] = "default";    /* default firmware filename (8.lod) */
char svfile[MAX_PATHSZ] = "default";    /* default servo filename (8.tms) */
char devname[MAX_PATHSM] = "/dev/sda";  /* UNIX device name (w path) */
char defdir[] = "/usr/share/scsirastools/"; /*default fwfile directory*/
int sdelay = 10;		/* num sec to delay before TUR */
FILE *fdlog = NULL;		/* log file descriptor */
FILE *fdmsg;			/* file descriptor for messages  */
char fautomodel = 0;		/* =1 to automatically do all of a model */
char fautoall = 0;		/* =1 to automatically flash all models found */
char filesok = 1;		/* =1 if ok to write to files */
char fdebug = 0;		/* =1 for debug messages */
char finteractive = 0;		/* =1 for interactive mode */
char fdoreset = 0;		/* =1 if need to do reset for this drive */
char flogopen = 0;		/* =1 if log file is open */
char fnotur = 0;		/* =1 to not try TUR */
char fdevname = 0;		/* =1 if device name specified */
char fonedev = 0;		/* =1 to download to one device (-i) */
char forcematch = 0;		/* =1 to force image to match disk */
char fusermax = 0;		/* =1 user specified a max chunk size */
int  ionedev = 0;		/* index of the one device from -i */
int  do_numeric = 1;		/* NUMERIC_SCAN_DEF; */
uchar sense_buffer[80];		/* buffer for sense data */
char output[MSG_LEN];		/* message output buffer */
char output2[MSG_LEN];		/* extra message output buffer */
uchar devstat[80];		/* used for inquiry & device status */
uchar devstat2[120];		/* used for Seagate inquiry2 & Mylex stat */
uchar disk_buffer[0x400];	/* 1024 bytes, used for misc messages/data */
uchar *image_buffer;		/* malloc'd firmware image buffer */
uchar *servo_buffer;		/* malloc'd servo buffer */
ulong user_chunk = 0x8000;      /* user-specified max chunk size */
				/* get chunk sz from mode page C7, bytes 16-18*/
int idev = 0;
int ndev = 0;
DEVLIST_TYPE devlist[MAX_DEVLIST_DEVS];

/* Local subroutines */
int writeimage(int idx, uchar * imgbuf, ulong len, char fservo);
ulong getimage(int idx, uchar ** pbuf, char fservo);
int beforedl(int idx, int *pnumrdy);
int afterdl(void);

void msgbuf_add(char *format, ...)
{
   va_list vptr; 
   char outmsg[MSG_LEN];
   int i, n;

   outmsg[0] = 0;
   va_start(vptr, format);
   vsnprintf(outmsg, MSG_LEN, format, vptr);
   va_end(vptr);
   n = strlen(disk_buffer);
   i = strlen(outmsg);
   if ((n + i) >= sizeof(disk_buffer)) {
      i = (sizeof(disk_buffer) - n) - 1;
      if (i < 0) i = 0;
      outmsg[i] = 0;  /*truncate msg*/
   }
   strcat((char *) disk_buffer, outmsg);
}

int main(int argc, char **argv)
{
    int i, c, idx;
    char response = 0;
    ulong size;			/* size of fw */
    ulong size_sv;		/* size of servo */
    int runl;
    short sts, rc;
    ulong ltime1;
    char fname[MAX_PATHSZ];
    char ebuff[EBUFF_LEN];
    int sg_fd;
    int flags;
    int res, k;
    int eacces_err = 0;
    int num_errors = 0;
    int nloops = 0;
    DEVFS_LIST devfs_components = {0,0,0,0,0};

    // progname = argv[0];
    fdmsg = stdout;		/* set default fdmsg */
    while ((c = getopt(argc, argv, "aenrxId:f:i:m:s:t:y?")) != EOF)
	switch (c) {
	case 'e':
	    filesok = 0;
	    break;
	case 'x':
	    fdebug = 1;
	    break;
	case 'y':
	    forcematch = 1;
	    break;
	case 'r':
	    fnotur = 1;
	    break;
	case 's':
	    user_chunk = atoi(optarg);
	    fusermax = 1;
	    break;
	case 'n':
	    do_numeric = 0;
	    break;
	case 'I':
            finteractive = 1;
            break;
	case 'i':
            fonedev = 1;
	    ionedev = atoi(optarg);
            break;
	case 'f':
	    i = strlen(optarg);
	    if (i < MAX_PATHSZ) {
	       strcpy(fwfile, optarg);
	       fsetfile = 1;
	    } else printf("filename too long (%d > %d)\n",i,MAX_PATHSZ);
	    break;
	case 'm':
	    i = strlen(optarg);
	    if (i >= MAX_MODELSZ) {
		i = MAX_MODELSZ - 1;
		optarg[i] = 0;  /*truncate the model string*/
	    }
	    strcpy(model, optarg);
	    fautomodel = 1;	/* automatically do all of this model */
	    break;
	case 'a':
	    fautoall = 1;	/* automatically all models found */
	    break;
	case 't':
	    sdelay = atoi(optarg);	/* time to delay */
	    break;
	case 'd':
	    i = strlen(optarg);
	    if (i < MAX_PATHSM) {
	       strcpy(devname, optarg);	/* specify a unix device name */
	       fdevname = 1;
               fonedev = 1;   /*flash one device*/
               ionedev = 0;   /*use first/only device in list*/
	    } else printf("device name too long (%d > %d)\n",i,MAX_PATHSM);
	    break;
	case '?':
	default:
	    printf("Usage: %s [-m diskmodel -f imagefile -t secdelay -adeinrxyI]\n",
		   progname);
	    printf(" -a  Auto-flash all disks with images in %s\n",defdir);
	    printf(" -d  Only use a specified device name, like /dev/sg1.\n");
	    printf(" -n  Turn off Numeric, use alpha /dev/sga names instead\n");
	    printf(" -e  Do not write to any files, so don't use the log file\n");
	    printf(" -f  Specify filename for firmware image instead of default\n");
	    printf(" -iN Download to one device at Index N\n");
	    printf(" -m  Automatically download all drives that match this model string\n");
	    printf(" -n  Turn off Numeric, use alpha /dev/sga names instead\n");
	    printf(" -r  Recover a non-ready drive by updating its firmware (ignore sense errors)\n");
	    printf(" -sN Force split buffer method, with chunk size N\n");
	    printf(" -t  Time in seconds to delay after the firmware is written, default 10\n");
	    printf(" -x  Outputs extra debug messages\n");
	    printf(" -y  Yes, force the image to match the disk model\n");
	    printf(" -I  Interactive mode, waits for input\n");
	    exit(1);
	}
    /* only run this as superuser */
    i = geteuid();
    if (i > 1) {
	printf("Not superuser (%d)\n", i);
	exit(1);
    }
    /* check for run-level 1 */
    {
	runl = 1;
	/* how to check this? */
	if (fdebug) { /*DEBUG*/ 
            printf("sdelay = %d sec\n", sdelay);
	    printf("run-level = %d\n", runl);
	}
    }
    if (runl != 1) {		/* show warning if not run level 1 */
	printf
	    ("Warning: Make sure disks are inactive during firmware download\n");
	// dont exit, just give a warning.
    }
    /* open log file */
    if (filesok)
	fdlog = fopen(logfile, "a+");
    if (fdlog == NULL) {
	flogopen = 0;
	filesok = 0;
	fdlog = stderr;
    } else {			/* logfile open successful */
	flogopen = 1;
	time((time_t *) & ltime1);
	fprintf(fdlog, "\n--- %s v%s log started at %s", progname, progver,
		ctime((long *) &ltime1));	// ctime outputs a '\n' at end
	fprintf(fdlog, "model:   %s\n", model);
	fprintf(fdlog, "fw file: %s\n", fwfile);
    }
    fdebugsub = fdebug;
    dbgout = fdlog;
    nloops = 0;
    /* loop to display disk choices and do downloads */
    while (1) {
	/* display header */
	printf("\n");
	printf("                %s utility v%s for SCSI disks\n", 
		progname, progver);
	printf
	    ("              ******************************************\n");
	if (flogopen)
	    printf("Log file %s is open, debug=%d\n", logfile, fdebug);
	printf
	    ("\nNum Name [bus:ch:id:lun] Type Vendor Device_Model     FW__ Serial#_ Servo___\n");
	/* get SCSI Device Info */
	idev = 0;
	flags = O_RDWR;		/* could use OPEN_FLAG if read-only. */
	num_errors = 0;
        for (k = 0; (k < MAX_SG_DEVS) && (idev < MAX_DEVLIST_DEVS) &&
                    (num_errors < MAX_ERRORS); ++k) {
	    // if (fdebug) showlog("k=%d, num_errors=%d\n", k, num_errors);
	    if (fdevname) {	/* user-specified device name */
		strcpy(fname, devname);
		k = MAX_SG_DEVS;
	    } else {
		make_dev_name(fname, k, do_numeric, &devfs_components);
		if (devfs_components.all_leaves_searched != 0) {
		   devfs_components.all_leaves_searched=0;
		   break;
		}
	    }
	    sg_fd = open(fname, flags | O_NONBLOCK);
	    if (fdevname && fdebug)
		showmsg("k=%d, %s open, sg_fd = %d\n", k, fname, sg_fd);
	    if (sg_fd < 0) {
		if (EBUSY == errno) {
		    printf("%s: device busy (O_EXCL lock), skipping\n",
			   fname);
		    continue;
		} else if ((ENODEV == errno) || (ENOENT == errno) ||
			     (ENXIO == errno)) {
		    ++num_errors;
		    continue;
		} else {   /* open failed with other error */
		    if (EACCES == errno)
			eacces_err = 1;
		    sprintf(ebuff, "Error opening %s ", fname);
		    perror(ebuff);
		    ++num_errors;
		    continue;
		}
	    }
	    sts = get_scsi_info(sg_fd, idev, fname,1);
	    if (sts) {
		showmsg("device %s failed with sts = %d, skip\n",fname, sts);
	    } 
            if (sg_fd > 0) {
                    res = close(sg_fd);
                    if (res < 0)
                        fprintf(fdmsg, "Error closing %s, errno = %d\n",
                                fname, errno);
                    else
                        devlist[idev].sgfd = 0; /* set in get_scsi_info() */
            }
            idev++;
	}			/*end loop */
	ndev = idev;
	if (ndev == 0 && errno == ENODEV) {	/* CONFIG_CHR_DEV_SG != y ? */
	    sprintf(output,
		    "Cant open any SCSI devices.\nMake sure "
                    "CONFIG_CHR_DEV_SG is set in /usr/src/linux/.config\n");
	    showit(output);
	    quit(1);
	}
        else if (ndev == 0) {
            sprintf(output,"Cant open any sg SCSI devices, errno = %d\n",errno);
	    if (errno == ENOENT) 
	       strcat(output,"Try option -n to change /dev/sg* device names\n");
            showit(output);
        }
	if (finteractive) {
	    printf("\nEnter Selection ('d' to download, 'q' to quit) : ");
	    response = get_func();
	} else if (fautomodel || fonedev || fautoall) {
	    response = 'd';
	} else {
	    response = 'q';  /*quit*/
	}
	if (response == 'q' || response == 'Q') {
	    printf("\n");
	    quit(0);
	} else if (response == 'd' || response == 'D') {  /* do download */
	    printf("Download ...\n");
	    if (fautomodel) {
		i = get_mdev(model);	// get first matching dev
	    } else if (fautoall) {
                i = get_nextdev();
	    } else if (fonedev)
                i = ionedev;
	    else
		i = get_idev();	/* get the device index that the user picks */
	    if (i == 0xff) {
		if (finteractive) {
		    printf("Index out of range, try again.\n");
		} else { // if (fautomodel || fonedev || fautoall) 
		    if (nloops == 0) {
                       printf("No matching devices found\n");
		       quit(1);
		    } else break;
		} 
	    } else {		/*index ok */
		int numrdy;
		idx = i;	/* first disk to download */
                /* read in the firmware image */
		size = getimage(idx, &image_buffer, 0);
                if (size == 0) size_sv = 0;  /*error, no need to try servo*/
                else {
                   /* read in the servo image */
		   size_sv = getimage(idx, &servo_buffer, 1);
                }
		if (fdebug) { /*DEBUG*/ 
		    showlog( "size = %ld\n", size);
		    if (size != 0)
			showlog( "imgbuf: %02x %02x %02x %02x \n",
				image_buffer[0], image_buffer[1],
				image_buffer[2], image_buffer[3]);
		    if (finteractive) {	
			do_pause();
		    }
		}
		if (size == 0)
		    idx = ndev;	/* error, skip download */
		else {		/* image ok */
		    rc = beforedl(idx, &numrdy);
		    if (numrdy == 0)
			idx = ndev;
		    /* download the image to each matching disk */
		    for (i = idx; i < ndev; i++) {
			if (devlist[i].fdoit) {
			    struct SCSI_INQUIRY *scsi_inq;
			    char fservo;
			    /* decide which download method to use */
			    scsi_inq =
				(struct SCSI_INQUIRY *) devlist[i].inq;
			    if ((devlist[i].inq[0] & 0x1F) == 1) { 
				/* devtype == 1 (DEVT_TAPE) */
				if (strncmp((char *) scsi_inq->vendorid,
				     "SEAGATE", 7) != 0)
				    /*Seagate tapes use split buffer method 3*/
				    fservo = 3;	
				else
				    /*Tecmar, Travan, Wangtek, WangDAT tapes 
				      use method 2 */
				    fservo = 2;	/* use split buffer method 2 */
			    }
			    else if (strncmp((char *) scsi_inq->vendorid, 
					"IBM", 3) == 0) {
				  if (strncmp ((char *) scsi_inq->productid, 
				        "ST", 2) != 0)  fservo = 0;
				  if (strncmp ((char *) scsi_inq->productid, 
				        "NTP", 3) != 0) fservo = 0;
				  else /* for all IBM disks, except Seagate*/
				     /* IBM disks use split buffer method 4 */
				     fservo = 4;  
			    } /*end-else IBM */
			    else if (fusermax) fservo = 5;
			    else
				fservo = 0;	/* normal, all one buffer */
			    if (size > 0) {	/* flash firmware */
				rc = writeimage(i, image_buffer, size, fservo);
#ifdef DEBUG
				printf("writeimage rc=%d msg_buffer=%c%c%c%c%c%c%c%c\n",
					rc,disk_buffer[0],disk_buffer[1],
					disk_buffer[2],disk_buffer[3],
					disk_buffer[4],disk_buffer[5],
					disk_buffer[6],disk_buffer[7]);
#endif
				if (fdebug && rc != 0) {
				    showit((char *) disk_buffer);  /*writes saved messages */
				    quit(rc);
				}
			    }
			    if (size_sv > 0) {	/* flash servo */
				rc = writeimage(i, servo_buffer,size_sv,1);
			    }
			}
			if (finteractive || fonedev)
			    i = ndev;	/* only do one if interactive */
		    }		/*endfor each disk */
		    rc = afterdl();
		}		/*endif image ok */
		free(image_buffer);     /*allocated in getimage() */
		if (size_sv > 0)
		   free(servo_buffer); /*allocated in getimage() */
	    }			/*end else index ok */
	} /*end else do download */
	else {			/* response not d or q */
	    printf("Invalid input [%c], enter 'd' or 'q'.\n", response);
	}
	if (finteractive) {	/* interactive, go to menu again */
	    do_pause();
	} else if (fonedev) {   /* if (fonedev) * then done */
	    if (0) {	/* if root is down and likely to hang otherwise, */
		printf("Rebooting ...\n");
		sleep(3);
		/* could do reboot here, without shutdown */
	    } else
		quit(0);	/* normal exit, successful */
	}
	// else if (fautomodel || fautoall) ; /* keep going until i==0xff*/
	nloops++;
    }				/*end while(1) */
    quit(0);	/* normal exit */
    return(0);
}				/*end main() */

ulong getimage(int idx, uchar ** pbuf, char fservo)
{
    ulong filsz;
    FILE *fdimg;
    struct stat statbuf;
    struct SCSI_INQUIRY *scsi_inq;
    char modelx[20];
    char *filenm;
    char fullfilenm[80];
    char *buf;
    int i, rc;
    *pbuf = NULL;
    scsi_inq = (struct SCSI_INQUIRY *) devlist[idx].inq;
    if (fautomodel)
	strncpy(modelx, model, strlen(model));
    else
	strncpy(modelx, (char *) scsi_inq->productid, 16);
    for (i = 0; i < 16; i++)	/* eliminate trailing spaces */
	if (modelx[i] == ' ')
	    break;
    modelx[i] = 0;		/*stringify */
    if (fdebug) { /*DEBUG*/ 
        showlog( "get image idx=%d\n", idx);
	showlog( "fw file: %s\n", fwfile);
	showlog(
		"___________  0....+....1....+....2....+....3....+....4\n");
	showlog( "scsi_inq[8]: %s\n", &devlist[idx].inq[8]);
    }
    if (fservo != 1) {
	showmsg("Selected %d: [%d:%d] %s\n", idx,
		devlist[idx].chn, devlist[idx].tgt, scsi_inq->vendorid);
    }
    if (!fsetfile) {		/* build the default filename */
	for (i = 0; i < 8; i++) {	/*limit filename to 8 chars */
	    if (modelx[i] == ' ')
		break;
	    else
		fwfile[i] = tolower(modelx[i]);
	}
	fwfile[i] = 0;
	strcat(fwfile, fwsufx);
    }
    if (fservo == 1) {
	char *p;
	filenm = fwfile;
	strcpy(svfile, fwfile);	/* servo filename      */
	p = strrchr(svfile, '.');	/* find last '.' in filename */
	strcpy(++p, "tms");	/* skip '.', add new suffix */
	filenm = svfile;
    } else
	filenm = fwfile;
    if (fdebug)
	showlog( "fw filename: %s\n", filenm);
    /* open file */
    fdimg = fopen(filenm, "r");  /* try current directory */
    if (fdimg == NULL) {  /* if error, set default directory  */
	if (fdebug) 
	   showmsg("Cannot open %s, errno = %d\n", filenm, errno);
	strcpy(fullfilenm,defdir);
	strcat(fullfilenm,filenm);
	filenm = fullfilenm;
	}
    fdimg = fopen(filenm, "r");  /* try default directory */
    if (fdimg == NULL) {
	if (fservo == 1) {
	    if (fdebug)
		showlog( "Cannot open %s, errno = %d\n", filenm, errno);
	    sprintf(output, "No servo image\n");
	} else
	    sprintf(output, "Cannot open %s, errno = %d\n", filenm, errno);
	showit(output);
	return (0);		/* size = 0 */
    }
    /* get image file size */
    if (fstat(fileno(fdimg), &statbuf) < 0) {
	showmsg("Can't get size of %s, errno = %d\n", filenm, errno);
	fclose(fdimg);
	return (0);		/* size = 0 */
    }
    filsz = statbuf.st_size;
    buf = malloc(filsz);
    if (buf == NULL) {
	sprintf(output,
		"Cannot allocate %ld bytes of memory, errno = %d.\n",
		filsz, errno);
	showit(output);
	fclose(fdimg);
	return (0);		/* size = 0 */
    }
    /* read in the image file */
    sprintf(output, "Reading image file %s, size = %ld\n", filenm, filsz);
    showit(output);
    rc = fread(buf, filsz, 1, fdimg);
    if (rc == -1) {
	sprintf(output, "Error reading %s, errno = %d.\n", filenm, errno);
	showit(output);
	fclose(fdimg);
	free(buf);
	return (0);		/* size = 0 */
    }
    if (fservo != 1) {
	char fscanmodel;
	fscanmodel = 0;
	if (strncmp((char *) scsi_inq->vendorid, "SEAGATE", 7) == 0) {
	    if (modelx[7] == 'W')
		modelx[7] = 'N';	/*always match with narrow */
	    modelx[8] = 0;	/* truncate to 8 chars for search */
	    fscanmodel = 1;
	} else if (strncmp((char *) scsi_inq->vendorid, "IBM", 3) == 0) {
	    fscanmodel = 1;
	    modelx[14] = 0;  /* IBM max 14 chars, not 16 */
	    }
	else if (strncmp((char *) scsi_inq->vendorid, "MAXTOR", 6) == 0)
	    fscanmodel = 1;
	else if (strncmp((char *) scsi_inq->vendorid, "QUANTUM", 7) == 0)
	    fscanmodel = 1;
	if (forcematch == 1) fscanmodel = 0;  /*do not scan for match*/
	/* HITACHI and FUJITSU only have partial model strings to match */
	if (fscanmodel) {	/* scan image buffer for model(productid) */
	    /* A variable offset in the file has a list of the target models. */
	    rc = findmatch(buf, filsz, modelx, strlen(modelx), 1);	/*also ignore case */
	    if (rc == -1) {
		sprintf(output, "Image does not match model %s\n", modelx);
		showit(output);
		fclose(fdimg);
		free(buf);
		return (0);	/* size = 0 */
	    }
	    if (fdebug) {
		sprintf(output, "Image file matches model %s\n", modelx);
		showit(output);
	    }
	}			/*endif scanmodel */
    }
    /*endif not servo */
    *pbuf = buf;
    fclose(fdimg);
    return (filsz);
}				/*end getimage() */

/*
 * write_buffers:
 *   sgfd   = the file descriptor of the device (opened)
 *   buf    = the firmware image, read from getimage()
 *   len    = the length of the firmware image
 *   fservo = control flag, where  
 *         0 is normal firmware, all one writebuffer command,
 *         1 is servo, use all one command
 *         2 is for most tapes, split buffer (0x8000/0x04,0x05)
 *         3 is for Seagate tapes, split buffer (0x4000/0x04,0x07)
 *         4 is for IBM disks, split buffer (0x8000/0x05,0x05)
 *         5 is for user-specified max, split buffer (0x8000/0x04,0x05)
 *              get chunk size from mode page C7, bytes 16-18
 */
int write_buffers(int sgfd, uchar * buf, ulong len, uchar fservo)
{
    int sts;
    ulong j, sz_left;
    ulong max_chunk;
    uchar save_mode;
    uchar bufid;
    uchar ebuf[20];
    if (fdebug)
	printf("write_buffers(len=%ld,mode=%d)\n",len,fservo);
    if (fservo >= 2) {          /* use split buffers */
	if (fservo == 3) {	/* Seagate tapes */
	    max_chunk = 0x4000;
	    save_mode = 0x07;
	} else if (fservo == 2) {	/* most tapes */
	    max_chunk = 0x8000;
	    save_mode = 0x05;
	} else if (fservo == 4) {	/* IBM disks */
	    max_chunk = 0x8000;	/* get this from mode page C7, bytes 16-18 */
	    save_mode = 0x05;
	    if (fdebug)  /*fdlog closed here*/
		printf("IBM split buffer mode, chunk size = %ld\n",max_chunk);
	    j = 0;
	    sz_left = len;
	    bufid = 0;
	    while (sz_left > max_chunk) {
		/* Note that it uses mode 05 for every chunk, not just last one */
		sts = write_buffer(sgfd, &buf[j],max_chunk, save_mode,bufid);
		if (sts != 0) {
		    unsigned int k, a, q;
		    int rc;
		    rc = get_sense(sts, ebuf);
		    k = ebuf[2] & 0x0f;	/*Sense Key */
		    a = ebuf[12]; /*ASC*/ 
		    q = ebuf[13]; /*ASCQ*/
		    sprintf(output, "wb sense error: %x/%x/%x\n", k, a, q);
		    showit(output);
		    if (k == 03 && a == 0x30)
			printf("Overlay upload error\n");
		    else if ((k == 2 || k == 3 || k == 4) &&
			     (a == 0x40 && q <= 0x85))
			printf("Microcode Load Failure\n");
		    return (sts);
		}
		sz_left -= max_chunk;
		j += max_chunk;
		bufid++;
	    }
	    sts = write_buffer(sgfd, &buf[j], sz_left, save_mode, bufid);
	    return (sts);
	}			/*endelse IBM disks */
	else {   /* (fservo == 5) * user specified */
	    max_chunk = user_chunk;
	    save_mode = 0x05;
        }
	/* Handle the split buffer scheme */
	/* This assumes that the start address is zero for all. */
	bufid = 0;
	j = 0;
        if (fdebug)
  	    printf("split buffers: max_chunk=%ld,save_mode=%d\n",
			max_chunk,save_mode);
	sz_left = len;
	while (sz_left > max_chunk) {
	    sts = write_buffer(sgfd, &buf[j], max_chunk, 0x04, bufid);
	    if (sts != 0)
		return (sts);
	    sz_left -= max_chunk;
	    j += max_chunk;
	}
	sts = write_buffer(sgfd, &buf[j], sz_left, save_mode, bufid);
    }  /*endif fservo>=2, split buffer scheme */

    else {			/* fservo == 0, or fservo == 1 */
        if (fdebug)
  	    printf("one buffer: buf=%p, len=%ld, save_mode=0x05\n",buf,len);
	/* Send it all as one big buffer */
	sts = write_buffer(sgfd, buf, len, 0x05, 0L);
    }
    return (sts);
}				/* end write_buffers() */

/*
 * writeimage:
 *   idx = the index of the device in devlist
 *   imgbuf = the firmware image, read from getimage()
 *   len    = the length of the firmware image
 *   fservo = control flag, where  
 *         0 is normal firmware, all one writebuffer command,
 *         1 is servo, use all one command
 *         2 is for most tapes, split buffer (0x8000/0x04,0x05)
 *         3 is for Seagate tapes, split buffer (0x4000/0x04,0x07)
 *         4 is for IBM disks, split buffer (0x8000/0x05,0x05)
 *         5 is for user-specified max, split buffer (0x8000/0x04,0x05)
 */
int writeimage(int idx, uchar * imgbuf, ulong len, char fservo)
{
    int sts;
    int devfd;
    int isave;
    ulong ltime, ltime1, ltime2;
    ulong ltime3;
    int i, rc, rcdl;
    int k, a, q;
    char *tag;
    uchar ch, dv;
    isave = idx;
    rcdl = 0;
    /* Download firmware to this disk */
    {
	rcdl = 0;
	fdoreset = 0;
	if (!fautomodel)
	    idx = isave;	/* only do selected disk */
	devfd = devlist[idx].sgfd;
	ch = devlist[idx].chn;
	dv = devlist[idx].tgt;
	if (devlist[idx].fdoit == 1) {
	    if (fautomodel || fautoall) {
		/* Sometimes next device may have a unit attention/reset 
		 * check condition pending, so clear that first before the 
		 * write_buffer. 
		 */
		sts = test_unit_ready(devfd, 0);
		if (fdebug) {
		    sprintf(output, "Extra test_unit_ready sts = %x\n", sts);
	            showit(output);
		}
	    }
	    time((time_t *) & ltime1);	// get the start time
	    /* start the firmware download */
	    if (fservo == 1)
		tag = "Servo";
	    else
		tag = "Firmware";
	    sprintf(output, "Downloading %s image to disk %d %s",
		    tag, idx, ctime((long *) &ltime1));
	    // ctime outputs a '\n' at end
	    showit(output);
	    sts = write_buffers(devfd, imgbuf, len, fservo);
#ifdef DEBUG
            printf("write_buffers sts=%d\n",sts);
#endif
	    if (sts) {
		rcdl = sts;
		sprintf(output, "[%d] Error %d in writing buffer\n", idx,sts);
		showit(output);	// fdlog closed
		if (fautomodel || fautoall) {
		   /* Save error for logging later */
                   msgbuf_add("[%d] %s download error, status = 0x%x\n",
				idx, tag, rcdl);
		   /* Try to restart the drive, even if the download failed. */
		   return (rcdl);	/*skip to next one if error */
		} else if (fdebug) {
		    return (rcdl);	// (only return if debug on)
		}
	    }
	    /*
	     * At this point the firmware image has been transferred, 
	     * but the device may need some poking/patting as it starts
	     * to come ready again.  
	     * This loop waits up to 30 loops of 3 sec each for the 
             * device to return a UnitReady status. 
             */
	    showmsg("Waiting for disk to come ready again ");
	    sleep(sdelay);	/* wait for the drive to reset and come ready */
	    for (i = 0; i < 30 && sts; i++) {
		// sts = restart_device(devfd,devlist[idx].dstatus);
		sts = test_unit_ready(devfd, 0);
#ifdef DEBUG
                printf("test_unit_ready sts=%d\n",sts);
#endif
	 	if (sts == SCHECK_CND) {	
		    /* check for certain key/asc/ascq values */
		    rc = get_sense(sts, sense_buffer);
		    k = sense_buffer[2] & 0x0f;	/*Sense Key */
		    a = sense_buffer[12]; /*ASC*/ 
		    q = sense_buffer[13]; /*ASCQ*/ 
		    /* 02-04-02 means it needs a start command, so issue it. */
		    if (k == 2 && a == 4 && q == 2) {
			sts = start_unit(devfd);
			if (fdebug) { /*DEBUG*/
			    showmsg("\nStart Unit: sts=%d, ", sts);
			}
		    }		/*endif 02-04-02 */
		} /*endif sts */
		else {
		    k = 0;
		    a = 0;
		    q = 0;
		}
		if (fdebug) {
		    showmsg("[%d: %d %d %d]", sts, k, a, q);
		}
		fprintf(fdmsg, ".");
		sleep(3);
	    }			/* end for TUR loop */
	    showit("\n");
	    // filesok = 1;     /* afterdl allows files to be opened again */
	    if (sts < 0) {	/* still not ready, needs a reset */
		fdoreset = 1;
		sts = scsi_reset(devfd, 3);
#ifdef DEBUG
                printf("scsi_reset sts=%d\n",sts);
#endif
		if (sts) {
		    msgbuf_add("Error %d during SCSI reset of channel\n",sts);
		} else {
		    msgbuf_add("SCSI reset of channel %d successful\n", ch);
		}
		if (fdebug)
		    do_pause();
		sleep(1);
		sts = test_unit_ready(devfd, 0);	/*first may get check condition */
		sts = test_unit_ready(devfd, 1);
	    }
#ifdef DEBUG
            printf("last tur, sts=%d\n",sts);
#endif
	    if (sts) {		/* TUR status reports not ready */
		rcdl = 0x1000 | sts;
		showmsg("[%d] Did not come ready after download, (%d)\n",
			idx, sts);
	    } /*endif not ready */
	    else {		/* successful download */
		showmsg("[%d] Successful %s download\n", idx, tag);
	    }
	    if (fdebug) {
		time((time_t *) & ltime3);	// get the interval time into ltime3
		showmsg( "[%d] write_buffer complete, sts = 0x%x, elapsed secs %02ld\n",
			idx, rcdl, ltime3 - ltime1);
	    }
	    /* Set the timeout shorter. */
	    k = 500;
	    sts = ioctl(devfd, 0x2201, &k);	/* SG_SET_TIMEOUT */
#ifdef DEBUG
            printf("set_timeout1 ioctl, sts=%d\n",sts);
#endif
	    // if (sts < 0) just continue
	    // First SCSI command after this may time out, so send a dummy command.
	    sts = scsi_inquiry(devfd, devstat, 80);	// do a dummy inquiry
#ifdef DEBUG
            printf("scsi_inquiry, sts=%d\n",sts);
#endif
	    if (fdebug)
		fprintf(fdmsg, "dummy inq sts = %d\n", sts);	// should be ok now
	    if (sts != 0)
		rcdl = 0x3000 | sts;	// use status from inquiry 
	    /* Dont make a device online if it was offline before we started. */
	    sts = test_unit_ready(devfd, 0); /*1st cmd may get check condition*/
#ifdef DEBUG
            printf("test_unit_ready, sts=%d\n",sts);
#endif
	    if (fdebug)
		showmsg("dummy tur sts = %d\n", sts);	// should be ok now
	    /* may get 06/29/04 here */

	    /* Now the device is either ready, or not, so resume activity. */
	    /* Restore the timeout. */
	    k = 6000;
	    sts = ioctl(devfd, 0x2201, &k);	/* SG_SET_TIMEOUT */
#ifdef DEBUG
            printf("set_timeout2 ioctl, sts=%d\n",sts);
#endif
	    // if (sts < 0) just continue
	    time((time_t *) & ltime2);	// get the end time into ltime2
	    ltime = ltime2 - ltime1;	// total time in seconds
	    ltime1 = ltime / 3600;	// num hours
	    ltime = ltime % 3600;	// remainder, minutes & seconds
	    // fdlog closed, so ok
	    showmsg("\t\t elapsed time %02ld:%02ld:%02ld\n",
		    ltime1, ltime / 60, ltime % 60);
	    /* save result for logging later */
	    msgbuf_add("[%d] %s download complete, status = 0x%x, elapsed time %02ld:%02ld:%02ld\n",
		    idx, tag, rcdl, ltime1, ltime / 60, ltime % 60);
	    if ((rcdl & 0x0FFF) == SCHECK_CND) {
		msgbuf_add("   Sense data: %02x %02x %02x\n",
			sense_buffer[2] & 0x0f, sense_buffer[12],
			sense_buffer[13]);
	    }
	    if (fdebug && rcdl != 0) {
		return (rcdl);	/* return (only if debug on) */
	    }
	    // if (!fautomodel) idx = ndev;   *end loop (no longer needed)*
	}			/*endif fdoit */
    }				/*end download disk */
    return (rcdl);
}				/*end writeimage() */

int beforedl(int idx, int *pnumrdy)
{
    struct SCSI_INQUIRY *scsi_inq;
    int sts;
    int sgfd;
    int isave, nrdy;
    char *pl;
    int i, rc, rcdl;
    int k, a, q;
    int openflags;
    char *fname;
    int dbglvl;
    isave = idx;
    nrdy = 0;
    sts = 0;
    rcdl = 0;
    openflags = O_RDWR;		/* | O_NONBLOCK; */

    /* if fautomodel, use global model from optarg */
    if (!fautomodel) {  /* otherwise use model from scsi_inq */
	scsi_inq = (struct SCSI_INQUIRY *) devlist[idx].inq;
	strncpy(model, (char *) scsi_inq->productid, 8);
	model[8] = 0;		/*stringify */
    }
    for (idx = 0; idx < ndev; idx++) {
	if (fautomodel || fautoall) ;
	else idx = isave;	/* only do selected disk */
	fname = devlist[idx].fname;
	sgfd = open(fname, openflags);	/* open blocking */
	if (sgfd < 0) {
	    printf("%s: open error, errno = %d\n", fname, errno);
	    return (errno);
	}
	devlist[idx].sgfd = sgfd;
	if (fdebug) dbglvl = 10;
	else dbglvl = 5;
	sts = set_sg_debug(sgfd, dbglvl);
	if (sts) {
	    sprintf(output, "[%d] cant set debug level %d, sts = %d", 
		    idx, dbglvl, sts);
	    showit(output);
	}
	scsi_inq = (struct SCSI_INQUIRY *) devlist[idx].inq;
	if (fdebug) /*DEBUG*/
		showlog(
			"Doing Test Unit Ready on idx = %d, model=%s\n",
			idx, model);
	if (strncmp(model, (char *) scsi_inq->productid, 8) == 0) {
	    /* do only ones of the same disk product id */
	    sts = test_unit_ready(sgfd, 0);
	    if (sts == SCHECK_CND) {
                /* not ready initially, try to make it ready */
		i = get_sense(sts, sense_buffer);
		k = sense_buffer[2] & 0x0f;	/*Sense Key */
		a = sense_buffer[12];           /*ASC*/ 
		q = sense_buffer[13];           /*ASCQ*/ 
		if (k == 2 && a == 4 && q == 2) {
		    /* 02-04-02 means it needs a start command, so issue it. */
		    sts = start_unit(sgfd);
		    if (fdebug) {
			/*DEBUG*/ printf("\nStart Unit: sts=%d, ", sts);
			if (sts == SCHECK_CND) {
			    rc = get_sense(sts, sense_buffer);
			    printf("sense data: %x %x %x\n",
				   sense_buffer[2] & 0x0f,
				   sense_buffer[12], sense_buffer[13]);
			}
		    }
		}		/*endif 02-04-02 */
		/* else if (k == 6 && a == 29) {  *reset, try again below*/
		/* else if (k == 3 && a == 31) {  *medium error, use fnotur*/
		sts = test_unit_ready(sgfd, 1);	/* try again */
	    }
	    if (sts) {
		sprintf(output, "[%d] Error %d from Test Unit Ready\n",
			idx, sts);
		showit(output);
		devlist[idx].fdoit = 0;
		if (fnotur) {
		    devlist[idx].fdoit = 1;
		    nrdy++;
		}
	    } else {
		devlist[idx].fdoit = 1;
		sprintf(output, "Device [%d] is ready for download\n",
			idx);
		showit(output);
		nrdy++;		/* number ready to download */
	    }
	} else
	    devlist[idx].fdoit = 0;	/* not same product id */
	if (fautomodel || fautoall) ;
	else idx = ndev;		/* just one, end loop */
    }				/* end for devlist TUR loop */
    *pnumrdy = nrdy;		/* set return value */
    if (nrdy == 0) {
	sprintf(output, "There are no ready devices for model %s.\n", model);
        strcat(output,"Try 'modprobe sg' if not already loaded\n"); 
	showit(output);
	return (sts);
    } else {
	if (nrdy > 1)
	    pl = "s";
	else
	    pl = "";
	sprintf(output, "Starting download process for %d disk%s.\n", nrdy,
		pl);
	showit(output);
    }
    closelog(); 		/* sets flogopen to false */
    filesok = 0;		/* do not allow files to be opened  */
    fflush(fdmsg);		/* flush stdout/stderr for messages */
    setbuf(fdmsg, NULL);	/* set for unbuffered writes */
    sync();
    sync();			/* write all UNIX buffers to disk */
    sleep(2);
    /* can't have any more disk IOs from here on */
    if (0) {    /* was if(fumntroot), disabled for now */
        /* unmount root by default, skip it if user says not to. */
	fprintf(fdmsg, "Unmounting root ...");
	// rc = uadmin(4, 128, 0);             /* unmount root fs */
	if (rc == 0)
	    fprintf(fdmsg, "done\n");
	else
	    fprintf(fdmsg, "rc = %d\n", rc);
	// frootdown = 1;  
    }
    /* Take all drives offline first if specified. */
    disk_buffer[0] = 0;		/* used for messages later */
    return (rcdl);
}				/*end beforedl() */

int afterdl(void)
{
    int i, sts;
    sts = 0;
    /* Bring all drives online again if raid */
    /* done, all drives should be ready now */
    filesok = 1;		/* allows files to be opened again */
    showit((char *) disk_buffer);	/* writes saved messages */
    for (i = 0; i < ndev; i++) {
	if (devlist[i].sgfd > 0) {
	    sts = set_sg_debug(devlist[i].sgfd, 0);
	    if (sts && fdebug)
		showlog( "[%d] cant clear debug flag, sts=%d errno=%d\n",
			 i, sts, errno);
	    close(devlist[i].sgfd);	/*opened in beforedl() */
	    devlist[i].sgfd = 0;
	}
    }
    return (sts);
}				/*end afterdl() */

/* end sgdskfl.c */
