/*++

Copyright (C) 2018 Autodesk Inc. (Original Author)

All rights reserved.

Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

1. Redistributions of source code must retain the above copyright notice, this
list of conditions and the following disclaimer.
2. Redistributions in binary form must reproduce the above copyright notice,
this list of conditions and the following disclaimer in the documentation
and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

--*/

//////////////////////////////////////////////////////////////////////////////////////////////////////
// buildimplementationcpp.go
// functions to generate C++ interface classes, implementation stubs and wrapper code that maps to
// the C-header.
//////////////////////////////////////////////////////////////////////////////////////////////////////

package main

import (
	"fmt"
	"log"
	"path"
	"strings"
)

// BuildImplementationCPP builds C++ interface classes, implementation stubs and wrapper code that maps to the C-header
func BuildImplementationCPP(component ComponentDefinition, outputFolder string, stubOutputFolder string, projectOutputFolder string, implementation ComponentDefinitionImplementation) error {
	forceRecreation := false

	doJournal := len (component.Global.JournalMethod) > 0;
	
	NameSpace := component.NameSpace;
	ImplementationSubNameSpace := "Impl"
	LibraryName := component.LibraryName;
	BaseName := component.BaseName;

	indentString := getIndentationString(implementation.Indentation)
	stubIdentifier := ""
	if len(implementation.StubIdentifier) > 0 {
		stubIdentifier = "_" + strings.ToLower(implementation.StubIdentifier)
	}

	IntfExceptionHeaderName := path.Join(outputFolder, BaseName+"_interfaceexception.hpp");
	log.Printf("Creating \"%s\"", IntfExceptionHeaderName)
	hInternalExceptionHeaderFile, err :=  CreateLanguageFile (IntfExceptionHeaderName, indentString)
	if err != nil {
		return err
	}
	hInternalExceptionHeaderFile.WriteCLicenseHeader(component,
		fmt.Sprintf("This is an autogenerated C++ Header file with the basic internal\n exception type in order to allow an easy use of %s", LibraryName),
		true)

	IntfExceptionImplName := path.Join(outputFolder, BaseName+"_interfaceexception.cpp");
	log.Printf("Creating \"%s\"", IntfExceptionImplName)
	hInternalExceptionImplFile, err :=  CreateLanguageFile (IntfExceptionImplName, indentString)
	if err != nil {
		return err
	}
	hInternalExceptionImplFile.WriteCLicenseHeader(component,
		fmt.Sprintf("This is an autogenerated C++ Implementation file with the basic internal\n exception type in order to allow an easy use of %s", LibraryName),
		true)

	err = buildCPPInternalException(hInternalExceptionHeaderFile, hInternalExceptionImplFile, NameSpace, BaseName )
	if err != nil {
		return err
	}

	IntfHeaderName := path.Join(outputFolder, BaseName+"_interfaces.hpp");
	log.Printf("Creating \"%s\"", IntfHeaderName)
	interfaceshppfile, err := CreateLanguageFile (IntfHeaderName, indentString)
	if err != nil {
		return err
	}
	interfaceshppfile.WriteCLicenseHeader(component,
		fmt.Sprintf("This is an autogenerated C++ header file in order to allow easy\ndevelopment of %s. The implementer of %s needs to\nderive concrete classes from the abstract classes in this header.", LibraryName, LibraryName),
		true)
	err = buildCPPInterfaces(component, interfaceshppfile, ImplementationSubNameSpace, implementation.ClassIdentifier)
	if err != nil {
		return err
	}

	IntfWrapperImplName := path.Join(outputFolder, BaseName+"_interfacewrapper.cpp");
	log.Printf("Creating \"%s\"", IntfWrapperImplName)
	cppWrapperfile, err := CreateLanguageFile(IntfWrapperImplName, indentString)
	if err != nil {
		return err
	}
	cppWrapperfile.WriteCLicenseHeader(component,
		fmt.Sprintf("This is an autogenerated C++ implementation file in order to allow easy\ndevelopment of %s. The functions in this file need to be implemented. It needs to be generated only once.", LibraryName),
		true)
	err = buildCPPInterfaceWrapper(component, cppWrapperfile, NameSpace, ImplementationSubNameSpace, implementation.ClassIdentifier, BaseName, doJournal)
	if err != nil {
		return err
	}

	if (doJournal) {
		IntfJournalHeaderName := path.Join(outputFolder, strings.ToLower(BaseName)+"_interfacejournal.hpp");	
		log.Printf("Creating \"%s\"", IntfJournalHeaderName)
		interfacejournalhppfile, err := CreateLanguageFile (IntfJournalHeaderName, indentString)
		if err != nil {
			return err
		}
		interfacejournalhppfile.WriteCLicenseHeader(component,
			fmt.Sprintf("This is an autogenerated C++ header file in order to allow easy\ndevelopment of %s. It provides an automatic Journaling mechanism for the library implementation.", LibraryName),
			true)
		
		IntfJournalImplName := path.Join(outputFolder, strings.ToLower(BaseName)+"_interfacejournal.cpp");
		log.Printf("Creating \"%s\"", IntfJournalImplName)
		interfacejournalcppfile, err := CreateLanguageFile(IntfJournalImplName, indentString)
		if err != nil {
			return err
		}
		interfacejournalcppfile.WriteCLicenseHeader(component,
			fmt.Sprintf("This is an autogenerated C++ implementation file in order to allow easy\ndevelopment of %s. It provides an automatic Journaling mechanism for the library implementation.", LibraryName),
			true)
		
		err = buildJournalingCPP(component, interfacejournalhppfile, interfacejournalcppfile)
		if err != nil {
			return err
		}
	}

	err = buildCPPStub(component, NameSpace, ImplementationSubNameSpace, implementation.ClassIdentifier, BaseName, stubOutputFolder, indentString, stubIdentifier, forceRecreation)
	if err != nil {
		return err
	}

	IntfWrapperStubName := path.Join(stubOutputFolder, BaseName + stubIdentifier + ".cpp")
	if forceRecreation || (!FileExists(IntfWrapperStubName) ) {
		log.Printf("Creating \"%s\"", IntfWrapperStubName)
		stubfile, err := CreateLanguageFile (IntfWrapperStubName, indentString)
		if err != nil {
			return err
		}
		stubfile.WriteCLicenseHeader(component,
			fmt.Sprintf("This is an autogenerated C++ implementation file in order to allow easy\ndevelopment of %s. It needs to be generated only once.", LibraryName),
			true)
		
		err = buildCPPGlobalStubFile(component, stubfile, NameSpace, ImplementationSubNameSpace, implementation.ClassIdentifier, BaseName)
		if err != nil {
			return err
		}
	} else {
		log.Printf("Omitting recreation of implementation stub \"%s\"", IntfWrapperStubName)
	}

	if ( len(projectOutputFolder) > 0 ) {
		CMakeListsFileName := path.Join(projectOutputFolder, "CMakeLists.txt");
		if forceRecreation || !FileExists(CMakeListsFileName) {
			log.Printf("Creating CMake-Project \"%s\" for CPP Implementation", CMakeListsFileName)
			CMakeListsFile, err := CreateLanguageFile(CMakeListsFileName, indentString)
			if err != nil {
				return err
			}
			CMakeListsFile.WriteCMakeLicenseHeader(component,
				fmt.Sprintf("This is an autogenerated CMakeLists file for the development of %s.", LibraryName),
				true)
			buildCMakeForCPPImplementation(component, CMakeListsFile, doJournal)
		} else {
			log.Printf("Omitting recreation of CMake-Project \"%s\" for CPP Implementation", CMakeListsFileName)
		}
	}

	return nil
}

func buildCPPInternalException(wHeader LanguageWriter, wImpl LanguageWriter, NameSpace string, BaseName string) (error) {
	wHeader.Writeln("#ifndef __%s_INTERFACEEXCEPTION_HEADER", strings.ToUpper(NameSpace));
	wHeader.Writeln("#define __%s_INTERFACEEXCEPTION_HEADER", strings.ToUpper(NameSpace));
	wHeader.Writeln("");

	wHeader.Writeln("#include <exception>");
	wHeader.Writeln("#include <stdexcept>");

	wHeader.Writeln("#include \"%s_types.hpp\"", BaseName);
	wHeader.Writeln("");
		
	wHeader.Writeln("/*************************************************************************************************************************");
	wHeader.Writeln(" Class E%sInterfaceException", NameSpace);
	wHeader.Writeln("**************************************************************************************************************************/");
	wHeader.Writeln("");
	wHeader.Writeln("");
	wHeader.Writeln("class E%sInterfaceException : public std::exception {", NameSpace);
	wHeader.Writeln("protected:");
	wHeader.Writeln("  /**");
	wHeader.Writeln("  * Error code for the Exception.");
	wHeader.Writeln("  */");
	wHeader.Writeln("  %sResult m_errorCode;", NameSpace);
	wHeader.Writeln("  /**");
	wHeader.Writeln("  * Error message for the Exception.");
	wHeader.Writeln("  */");
	wHeader.Writeln("  std::string m_errorMessage;");
	wHeader.Writeln("")
	wHeader.Writeln("public:");
	wHeader.Writeln("  /**");
	wHeader.Writeln("  * Exception Constructor.");
	wHeader.Writeln("  */"); 
	wHeader.Writeln("  E%sInterfaceException(%sResult errorCode);", NameSpace, NameSpace);
	wHeader.Writeln("");
	wHeader.Writeln("  /**");
	wHeader.Writeln("  * Custom Exception Constructor.");
	wHeader.Writeln("  */");
	wHeader.Writeln("  E%sInterfaceException(%sResult errorCode, std::string errorMessage);", NameSpace, NameSpace);
	wHeader.Writeln("");
	wHeader.Writeln("  /**");
	wHeader.Writeln("  * Returns error code");
	wHeader.Writeln("  */");
	wHeader.Writeln("  %sResult getErrorCode();", NameSpace);
	wHeader.Writeln("  /**");
	wHeader.Writeln("  * Returns error message");
	wHeader.Writeln("  */");
	wHeader.Writeln("  const char* what() const noexcept override;");
	wHeader.Writeln("};");
	wHeader.Writeln("");
	
	wHeader.Writeln("#endif // __%s_INTERFACEEXCEPTION_HEADER", strings.ToUpper(NameSpace));


	wImpl.Writeln("");
	wImpl.Writeln("#include <string>");
	wImpl.Writeln("");
	wImpl.Writeln("#include \"%s_interfaceexception.hpp\"", BaseName);
	wImpl.Writeln("");

	wImpl.Writeln("/*************************************************************************************************************************");
	wImpl.Writeln(" Class E%sInterfaceException", NameSpace);
	wImpl.Writeln("**************************************************************************************************************************/");
	wImpl.Writeln("E%sInterfaceException::E%sInterfaceException(%sResult errorCode)", NameSpace, NameSpace, NameSpace);
	wImpl.Writeln("  : m_errorMessage(\"%s Error \" + std::to_string (errorCode))", NameSpace);
	wImpl.Writeln("{");
	wImpl.Writeln("  m_errorCode = errorCode;");
	wImpl.Writeln("}");
	wImpl.Writeln("");
	wImpl.Writeln("E%sInterfaceException::E%sInterfaceException(%sResult errorCode, std::string errorMessage)", NameSpace, NameSpace, NameSpace);
	wImpl.Writeln("  : m_errorMessage(errorMessage + \" (\" + std::to_string (errorCode) + \")\")");
	wImpl.Writeln("{");
	wImpl.Writeln("  m_errorCode = errorCode;");
	wImpl.Writeln("}");
	wImpl.Writeln("");
	wImpl.Writeln("%sResult E%sInterfaceException::getErrorCode ()", NameSpace, NameSpace);
	wImpl.Writeln("{");
	wImpl.Writeln("  return m_errorCode;");
	wImpl.Writeln("}");
	wImpl.Writeln("");
	wImpl.Writeln("const char * E%sInterfaceException::what () const noexcept", NameSpace);
	wImpl.Writeln("{");
	wImpl.Writeln("  return m_errorMessage.c_str();");
	wImpl.Writeln("}");
	wImpl.Writeln("");

	return nil;
}

func writeSharedPtrTemplate(component ComponentDefinition, w LanguageWriter, ClassIdentifier string) {
	IBaseClassName := "I" + ClassIdentifier + component.Global.BaseClassName
	w.Writeln("")
	w.Writeln("")
	w.Writeln("/**")
	w.Writeln(" Definition of a shared pointer class for %s", IBaseClassName)
	w.Writeln("*/")
	IBaseSharedPtrName := "I" + component.Global.BaseClassName + "SharedPtr"
	DeleteBaseMethodStr := ReleaseBaseClassInterfaceMethod(component.Global.BaseClassName).MethodName
	w.Writeln("template<class T>")
	w.Writeln("class %s : public std::shared_ptr<T>", IBaseSharedPtrName)
	w.Writeln("{")
	w.Writeln("public:")
	w.Writeln("  explicit %s(T* t = nullptr)", IBaseSharedPtrName)
	w.Writeln("    : std::shared_ptr<T>(t, %s::%s)", IBaseClassName, DeleteBaseMethodStr)
	w.Writeln("  {")
	w.Writeln("    t->%s();", IncRefCountMethod().MethodName)
	w.Writeln("  }")
	w.Writeln("")
	w.Writeln("  // Reset function, as it also needs to properly set the deleter.")
	w.Writeln("  void reset(T* t = nullptr)")
	w.Writeln("  {")
	w.Writeln("    std::shared_ptr<T>::reset(t, %s::%s);", IBaseClassName, DeleteBaseMethodStr)
	w.Writeln("  }")
	w.Writeln("")
	w.Writeln("  // Get-function that increases the Base class's reference count")
	w.Writeln("  T* getCoOwningPtr()")
	w.Writeln("  {")
	w.Writeln("    T* t = this->get();")
	w.Writeln("    t->%s();", IncRefCountMethod().MethodName)
	w.Writeln("    return t;")
	w.Writeln("  }")
	w.Writeln("};")
	w.Writeln("")
}

func writeCPPClassInterface(component ComponentDefinition, class ComponentDefinitionClass, w LanguageWriter, NameSpace string, NameSpaceImplementation string, ClassIdentifier string, BaseName string) (error) {
	w.Writeln("")
	w.Writeln("/*************************************************************************************************************************")
	w.Writeln(" Class interface for %s ", class.ClassName)
	w.Writeln("**************************************************************************************************************************/")
	w.Writeln("")
	parentClassString := " "
	if (!component.isBaseClass(class)) {
		parentClassString = " : public virtual "
		if (class.ParentClass == "") {
			parentClassString += fmt.Sprintf("I%s%s ", ClassIdentifier, component.Global.BaseClassName)
		} else {
			parentClassString += fmt.Sprintf("I%s%s ", ClassIdentifier, class.ParentClass)
		}
	}
	
	classInterfaceName := fmt.Sprintf("I%s%s", ClassIdentifier, class.ClassName)
	w.Writeln("class %s%s{", classInterfaceName, parentClassString)
	w.Writeln("public:")

	if (component.isBaseClass(class)) {
		w.Writeln("  /**")
		w.Writeln("  * %s::~%s - virtual destructor of %s", classInterfaceName, classInterfaceName, classInterfaceName)
		w.Writeln("  */")
		w.Writeln("  virtual ~%s() {};", classInterfaceName)
		w.Writeln("")

		releaseBaseClassInterfaceMethod := ReleaseBaseClassInterfaceMethod(component.Global.BaseClassName)
		methodstring, _, err := buildCPPInterfaceMethodDeclaration(releaseBaseClassInterfaceMethod, class.ClassName, NameSpace, ClassIdentifier, BaseName, w.IndentString, true, false, true)
		if err != nil {
			return err
		}
		w.Writeln("%s", methodstring)
		argument := "p"+releaseBaseClassInterfaceMethod.Params[0].ParamName
		w.Writeln("  {")
		w.Writeln("    if (%s) {", argument)
		w.Writeln("      %s->%s();", argument, DecRefCountMethod().MethodName)
		w.Writeln("    }")
		w.Writeln("  };")
		w.Writeln("")

		acquireBaseClassInterfaceMethod := AcquireBaseClassInterfaceMethod(component.Global.BaseClassName)
		methodstring, _, err = buildCPPInterfaceMethodDeclaration(acquireBaseClassInterfaceMethod, class.ClassName, NameSpace, ClassIdentifier, BaseName, w.IndentString, true, false, true)
		if err != nil {
			return err
		}
		w.Writeln("%s", methodstring)
		argument = "p"+acquireBaseClassInterfaceMethod.Params[0].ParamName
		w.Writeln("  {")
		w.Writeln("    if (%s) {", argument)
		w.Writeln("      %s->%s();", argument, IncRefCountMethod().MethodName)
		w.Writeln("    }")
		w.Writeln("  };")
		w.Writeln("")

		var methods [5]ComponentDefinitionMethod
		methods[0] = GetLastErrorMessageMethod()
		methods[1] = ClearErrorMessageMethod()
		methods[2] = RegisterErrorMessageMethod()
		methods[3] = IncRefCountMethod()
		methods[4] = DecRefCountMethod()
		for j := 0; j < len(methods); j++ {
			methodstring, _, err := buildCPPInterfaceMethodDeclaration(methods[j], class.ClassName, NameSpace, ClassIdentifier, BaseName, w.IndentString, false, true, true)
			if err != nil {
				return err
			}
			w.Writeln("")
			w.Writeln("%s;", methodstring)
		}
	}

	for j := 0; j < len(class.Methods); j++ {
		method := class.Methods[j]
		methodstring, _, err := buildCPPInterfaceMethodDeclaration(method, class.ClassName, NameSpace, ClassIdentifier, BaseName, w.IndentString, false, true, true)
		if err != nil {
			return err
		}
		w.Writeln("%s;", methodstring)
		w.Writeln("")
	}

	w.Writeln("};")

	if component.isBaseClass(class) {
		writeSharedPtrTemplate(component, w, ClassIdentifier)
	}

	w.Writeln("")
	w.Writeln("typedef I%s%sSharedPtr<%s> P%s;", ClassIdentifier, component.Global.BaseClassName, classInterfaceName, classInterfaceName)
	w.Writeln("")
	return nil
}

func writeClassDefinitions(component ComponentDefinition, w LanguageWriter, NameSpaceImplementation string, ClassIdentifier string) {
	NameSpace := component.NameSpace
	BaseName := component.BaseName

	for i := 0; i < len(component.Classes); i++ {
		class := component.Classes[i]
		writeCPPClassInterface(component, class, w, NameSpace, NameSpaceImplementation, ClassIdentifier, BaseName)
	}
}

func buildCPPInterfaces(component ComponentDefinition, w LanguageWriter, NameSpaceImplementation string, ClassIdentifier string) error {
	NameSpace := component.NameSpace
	BaseName := component.BaseName

	w.Writeln("")
	w.Writeln("#ifndef __%s_CPPINTERFACES", strings.ToUpper(NameSpace))
	w.Writeln("#define __%s_CPPINTERFACES", strings.ToUpper(NameSpace))
	w.Writeln("")

	w.Writeln("#include <string>")
	w.Writeln("#include <memory>")
	w.Writeln("")
	w.Writeln("#include \"%s_types.hpp\"", BaseName)
	w.Writeln("")
	w.Writeln("")
	for _, subComponent := range(component.ImportedComponentDefinitions) {
		w.Writeln("#include \"%s_dynamic.hpp\"", subComponent.BaseName)
	}
	w.Writeln("")

	w.Writeln("namespace %s {", NameSpace)
	w.Writeln("namespace %s {", NameSpaceImplementation)

	w.Writeln("")

	w.Writeln("/**")
	w.Writeln(" Forward declarations of class interfaces")
	w.Writeln("*/")
	for i := 0; i < len(component.Classes); i++ {
		class := component.Classes[i]
		w.Writeln("class I%s%s;", ClassIdentifier, class.ClassName);
	}
	w.Writeln("")
	w.Writeln("")

	writeClassDefinitions(component, w, NameSpaceImplementation, ClassIdentifier)

	w.Writeln("")
	w.Writeln("/*************************************************************************************************************************")
	w.Writeln(" Global functions declarations")
	w.Writeln("**************************************************************************************************************************/")

	w.Writeln("class C%sWrapper {", ClassIdentifier)
	w.Writeln("public:")

	if len(component.ImportedComponentDefinitions) > 0 {
		w.Writeln("  // Injected Components")
		for _, subComponent := range(component.ImportedComponentDefinitions) {
			subNameSpace := subComponent.NameSpace
			w.Writeln("  static %s::PWrapper sP%sWrapper;", subNameSpace, subNameSpace)
		}
		w.Writeln("")
	}

	global := component.Global;
	for j := 0; j < len(global.Methods); j++ {
		method := global.Methods[j]

		// Omit Journal Method
		isSpecialFunction, err := CheckHeaderSpecialFunction(method, global);
		if err != nil {
			return err
		}
		if (isSpecialFunction == eSpecialMethodJournal) || (isSpecialFunction == eSpecialMethodInjection) ||
			(isSpecialFunction == eSpecialMethodSymbolLookup) {
			continue
		}

		methodstring, _, err := buildCPPInterfaceMethodDeclaration(method, BaseName, NameSpace, ClassIdentifier, "Wrapper", w.IndentString, true, false, true)
		if err != nil {
			return err
		}
		w.Writeln("%s;", methodstring)
		w.Writeln("")
	}
	w.Writeln("};")
	w.Writeln("")
	w.Writeln("} // namespace %s", NameSpaceImplementation)
	w.Writeln("} // namespace %s", NameSpace)
	w.Writeln("")

	w.Writeln("#endif // __%s_CPPINTERFACES", strings.ToUpper(NameSpace))

	return nil
}

func buildCPPGlobalStubFile(component ComponentDefinition, stubfile LanguageWriter, NameSpace string, NameSpaceImplementation string, ClassIdentifier string, BaseName string) error {
	var defaultImplementation []string
	defaultImplementation = append(defaultImplementation, fmt.Sprintf("throw E%sInterfaceException(%s_ERROR_NOTIMPLEMENTED);", NameSpace, strings.ToUpper(NameSpace)))

	stubfile.Writeln("#include \"%s_abi.hpp\"", BaseName)
	stubfile.Writeln("#include \"%s_interfaces.hpp\"", BaseName)
	stubfile.Writeln("#include \"%s_interfaceexception.hpp\"", BaseName)
	stubfile.Writeln("")
	stubfile.Writeln("using namespace %s;", NameSpace)
	stubfile.Writeln("using namespace %s::%s;", NameSpace, NameSpaceImplementation)
	stubfile.Writeln("")

	if len(component.ImportedComponentDefinitions) > 0 {
		stubfile.Writeln("// Injected Components")
		for _, subComponent := range(component.ImportedComponentDefinitions) {
			subNameSpace := subComponent.NameSpace
			stubfile.Writeln("%s::PWrapper C%sWrapper::sP%sWrapper;", subNameSpace, ClassIdentifier, subNameSpace)
		}
		stubfile.Writeln("")
	}

	for j := 0; j < len(component.Global.Methods); j++ {
		method := component.Global.Methods[j]

		thisMethodDefaultImpl := defaultImplementation
		
		// Treat special functions
		isSpecialFunction, err := CheckHeaderSpecialFunction(method, component.Global);
		if err != nil {
			return err
		}
		if (isSpecialFunction == eSpecialMethodJournal) || (isSpecialFunction == eSpecialMethodInjection) ||
			(isSpecialFunction == eSpecialMethodSymbolLookup) {
			continue
		}
		if (isSpecialFunction == eSpecialMethodVersion) {
			var versionImplementation []string
			versionImplementation = append(versionImplementation,
				fmt.Sprintf("n%s = %s_VERSION_MAJOR;", method.Params[0].ParamName, strings.ToUpper(NameSpace)),
				fmt.Sprintf("n%s = %s_VERSION_MINOR;", method.Params[1].ParamName, strings.ToUpper(NameSpace)),
				fmt.Sprintf("n%s = %s_VERSION_MICRO;", method.Params[2].ParamName, strings.ToUpper(NameSpace)) )
			thisMethodDefaultImpl = versionImplementation
		}
		if (isSpecialFunction == eSpecialMethodRelease) {
			var releaseImplementation []string
			releaseImplementation = append(releaseImplementation,
				fmt.Sprintf("I%s%s::%s(p%s);", ClassIdentifier, component.Global.BaseClassName, ReleaseBaseClassInterfaceMethod(component.Global.BaseClassName).MethodName, method.Params[0].ParamName))
			thisMethodDefaultImpl = releaseImplementation
		}
		if (isSpecialFunction == eSpecialMethodAcquire) {
			var acquireImplementation []string
			acquireImplementation = append(acquireImplementation,
				fmt.Sprintf("I%s%s::%s(p%s);", ClassIdentifier, component.Global.BaseClassName, AcquireBaseClassInterfaceMethod(component.Global.BaseClassName).MethodName, method.Params[0].ParamName))
			thisMethodDefaultImpl = acquireImplementation
		}

		_, implementationdeclaration, err := buildCPPInterfaceMethodDeclaration(method, "Wrapper", NameSpace, ClassIdentifier, BaseName, stubfile.IndentString, true, false, false)
		if err != nil {
			return err
		}

		stubfile.Writeln("%s", implementationdeclaration)
		stubfile.Writeln("{")
		stubfile.Writelns("  ", thisMethodDefaultImpl)
		stubfile.Writeln("}")
		stubfile.Writeln("")
	}

	stubfile.Writeln("")

	return nil
}

func buildCPPInterfaceWrapperMethods(component ComponentDefinition, class ComponentDefinitionClass, w LanguageWriter, NameSpace string, NameSpaceImplementation string, ClassIdentifier string, BaseName string, doJournal bool) error {
	w.Writeln("")
	w.Writeln("/*************************************************************************************************************************")
	w.Writeln(" Class implementation for %s", class.ClassName)
	w.Writeln("**************************************************************************************************************************/")

	for j := 0; j < len(class.Methods); j++ {
		method := class.Methods[j]
		err := writeCImplementationMethod(component, method, w, BaseName, NameSpace, ClassIdentifier, class.ClassName, component.Global.BaseClassName, false, doJournal, eSpecialMethodNone)
		if err != nil {
			return err
		}
	}
	return nil
}

func buildCPPGetSymbolAddressMethod(component ComponentDefinition, w LanguageWriter) error {
	NameSpace := component.NameSpace
	w.Writeln("")
	w.Writeln("/*************************************************************************************************************************")
	w.Writeln(" Function table lookup implementation")
	w.Writeln("**************************************************************************************************************************/")
	w.Writeln("")
	w.Writeln("%sResult _%s_getprocaddress_internal(const char * pProcName, void ** ppProcAddress)", NameSpace, strings.ToLower (NameSpace))
	w.Writeln("{")

	w.AddIndentationLevel(1)
	w.Writeln("static bool sbProcAddressMapHasBeenInitialized = false;")
	w.Writeln("static std::map<std::string, void*> sProcAddressMap;")
	w.Writeln("if (!sbProcAddressMapHasBeenInitialized) {")

	w.AddIndentationLevel(1)
	
	global := component.Global;
	for i := 0; i < len(component.Classes); i++ {
		class := component.Classes[i]
		for j := 0; j < len(class.Methods); j++ {
			method := class.Methods[j]
			procName := strings.ToLower(class.ClassName + "_" + method.MethodName)
			w.Writeln(fmt.Sprintf("sProcAddressMap[\"%s_%s\"] = (void*)&%s_%s;", strings.ToLower(NameSpace), procName, strings.ToLower(NameSpace), procName))
		}
	}
	for j := 0; j < len(global.Methods); j++ {
		method := global.Methods[j]
		procName := strings.ToLower(method.MethodName)

		w.Writeln(fmt.Sprintf("sProcAddressMap[\"%s_%s\"] = (void*)&%s_%s;", strings.ToLower(NameSpace), procName, strings.ToLower(NameSpace), procName))
	}

	w.AddIndentationLevel(-1)

	w.Writeln("  ")
	w.Writeln("  sbProcAddressMapHasBeenInitialized = true;")
	w.Writeln("}")

	w.Writeln("if (pProcName == nullptr)")		
	w.Writeln("  return %s_ERROR_INVALIDPARAM;", strings.ToUpper(NameSpace))
	w.Writeln("if (ppProcAddress == nullptr)")		
	w.Writeln("  return %s_ERROR_INVALIDPARAM;", strings.ToUpper(NameSpace))
	w.Writeln("*ppProcAddress = nullptr;")
	w.Writeln("std::string sProcName (pProcName);")
	w.Writeln("")

	w.Writeln("auto procPair = sProcAddressMap.find(sProcName);")
	w.Writeln("if (procPair == sProcAddressMap.end()) {")
	w.Writeln("  return %s_ERROR_COULDNOTFINDLIBRARYEXPORT;", strings.ToUpper(NameSpace))
	w.Writeln("}")
	w.Writeln("else {")
	w.Writeln("  *ppProcAddress = procPair->second;")
	w.Writeln("  return %s_SUCCESS;", strings.ToUpper (NameSpace))
	w.Writeln("}")
	w.Writeln("")
	w.AddIndentationLevel(-1)
	w.Writeln("}")
	return nil;
}

func buildCPPInterfaceWrapper(component ComponentDefinition, w LanguageWriter, NameSpace string, NameSpaceImplementation string, ClassIdentifier string, BaseName string, doJournal bool) error {
	w.Writeln("#include \"%s_abi.hpp\"", strings.ToLower(BaseName))
	w.Writeln("#include \"%s_interfaces.hpp\"", strings.ToLower(BaseName))
	w.Writeln("#include \"%s_interfaceexception.hpp\"", strings.ToLower(BaseName))
	if (doJournal) {
		w.Writeln("#include \"%s_interfacejournal.hpp\"", strings.ToLower(BaseName))
	}
	w.Writeln("")
	w.Writeln("#include <map>")
	w.Writeln("")
	w.Writeln("using namespace %s::%s;", NameSpace, NameSpaceImplementation)
	w.Writeln("")

	if (doJournal) {
		w.Writeln("P%sInterfaceJournal m_GlobalJournal;", NameSpace)
		w.Writeln("")
	}

	
	journalParameter := "";
	if (doJournal) {
		journalParameter = fmt.Sprintf (", C%sInterfaceJournalEntry * pJournalEntry = nullptr", NameSpace);
	}

	IBaseClassName := "I" + ClassIdentifier + component.Global.BaseClassName
	registerErrorMethod := RegisterErrorMessageMethod()
	w.Writeln("%sResult handle%sException(%s * pIBaseClass, E%sInterfaceException & Exception%s)", NameSpace, NameSpace, IBaseClassName, NameSpace, journalParameter)
	w.Writeln("{")
	w.Writeln("  %sResult errorCode = Exception.getErrorCode();", NameSpace)
	w.Writeln("")
	if (doJournal) {
		w.Writeln("  if (pJournalEntry != nullptr)")
		w.Writeln("    pJournalEntry->writeError(errorCode);")
		w.Writeln("")
	}

	w.Writeln("  if (pIBaseClass != nullptr)")
	
	w.Writeln("    pIBaseClass->%s(Exception.what());", registerErrorMethod.MethodName)

	w.Writeln("")
	w.Writeln("  return errorCode;")
	w.Writeln("}")
	w.Writeln("")

	w.Writeln("%sResult handleStdException(%s * pIBaseClass, std::exception & Exception%s)", NameSpace, IBaseClassName, journalParameter)
	w.Writeln("{")
	w.Writeln("  %sResult errorCode = %s_ERROR_GENERICEXCEPTION;", NameSpace, strings.ToUpper(NameSpace))
	w.Writeln("")
	if (doJournal) {
		w.Writeln("  if (pJournalEntry != nullptr)")
		w.Writeln("    pJournalEntry->writeError(errorCode);")
		w.Writeln("")
	}

	w.Writeln("  if (pIBaseClass != nullptr)")
	w.Writeln("    pIBaseClass->%s(Exception.what());", registerErrorMethod.MethodName)

	w.Writeln("")
	w.Writeln("  return errorCode;")
	w.Writeln("}")
	w.Writeln("")

	w.Writeln("%sResult handleUnhandledException(%s * pIBaseClass%s)", NameSpace, IBaseClassName, journalParameter)
	w.Writeln("{")
	w.Writeln("  %sResult errorCode = %s_ERROR_GENERICEXCEPTION;", NameSpace, strings.ToUpper(NameSpace))
	w.Writeln("")
	if (doJournal) {
		w.Writeln("  if (pJournalEntry != nullptr)")
		w.Writeln("    pJournalEntry->writeError(errorCode);")
		w.Writeln("")
	}
	
	w.Writeln("  if (pIBaseClass != nullptr)")
	w.Writeln("    pIBaseClass->%s(\"Unhandled Exception\");", registerErrorMethod.MethodName)
	
	w.Writeln("")
	w.Writeln("  return errorCode;")
	w.Writeln("}")
	w.Writeln("")
	
	
	w.Writeln("")
	for i := 0; i < len(component.Classes); i++ {
		class := component.Classes[i]
		err := buildCPPInterfaceWrapperMethods(component, class, w, NameSpace, NameSpaceImplementation, ClassIdentifier, BaseName, doJournal)
		if (err != nil) {
			return err
		}
	}

	w.Writeln("")
	err := buildCPPGetSymbolAddressMethod(component, w);
	if err != nil {
		return err
	}

	w.Writeln("")
	w.Writeln("/*************************************************************************************************************************")
	w.Writeln(" Global functions implementation")
	w.Writeln("**************************************************************************************************************************/")
	global := component.Global;
	for j := 0; j < len(global.Methods); j++ {
		method := global.Methods[j]
				
		// Check for special functions
		isSpecialFunction, err := CheckHeaderSpecialFunction (method, global);
		if err != nil {
			return err
		}
		
		// Do not self-journal Journal special method
		doMethodJournal := doJournal;
		if (isSpecialFunction == eSpecialMethodJournal) {
			doMethodJournal = false;
		}

		// Write Static function implementation
		err = writeCImplementationMethod(component, method, w, BaseName, NameSpace, ClassIdentifier, "Wrapper", component.Global.BaseClassName, true, doMethodJournal, isSpecialFunction)
		if err != nil {
			return err
		}
	}

	w.Writeln("")

	return nil
}

func writeCImplementationMethod(component ComponentDefinition, method ComponentDefinitionMethod, w LanguageWriter, BaseName string, NameSpace string, ClassIdentifier string, ClassName string, BaseClassName string, isGlobal bool, doJournal bool, isSpecialFunction int) error {
	CMethodName := ""
	cParams, err := GenerateCParameters(method, ClassName, NameSpace)
	if err != nil {
		return err
	}

	cparameters := ""
	for _, cParam := range cParams {
		if cparameters != "" {
			cparameters = cparameters + ", "
		}
		cparameters = cparameters + cParam.ParamType + " " + cParam.ParamName
	}

	if isGlobal {
		CMethodName = fmt.Sprintf("%s_%s", strings.ToLower(NameSpace), strings.ToLower(method.MethodName))
	} else {
		CMethodName = fmt.Sprintf("%s_%s_%s", strings.ToLower(NameSpace), strings.ToLower(ClassName), strings.ToLower(method.MethodName))
		if cparameters != "" {
			cparameters = ", " + cparameters
		}
		cparameters = fmt.Sprintf("%s_%s p%s", NameSpace, ClassName, ClassName) + cparameters
	}

	callCPPFunctionCode := make([]string, 0)
	
	checkInputCPPFunctionCode, preCallCPPFunctionCode, postCallCPPFunctionCode, returnVariable, callParameters, err := generatePrePostCallCPPFunctionCode(component, method, NameSpace, ClassIdentifier, ClassName, BaseClassName)
	if err != nil {
		return err
	}
	
	if (isSpecialFunction == eSpecialMethodJournal) {
		callCPPFunctionCode = append(callCPPFunctionCode, "m_GlobalJournal = nullptr;")
		callCPPFunctionCode = append(callCPPFunctionCode, fmt.Sprintf("if (s%s != \"\") {", method.Params[0].ParamName))
		callCPPFunctionCode = append(callCPPFunctionCode, fmt.Sprintf("  m_GlobalJournal = std::make_shared<C%sInterfaceJournal> (s%s);", NameSpace, method.Params[0].ParamName))
		callCPPFunctionCode = append(callCPPFunctionCode, "}")
	} else if (isSpecialFunction == eSpecialMethodInjection) {
		callCPPFunctionCode = append(callCPPFunctionCode, "")
		callCPPFunctionCode = append(callCPPFunctionCode, "bool bNameSpaceFound = false;")
		callCPPFunctionCode = append(callCPPFunctionCode, "")
		for _, subComponent := range(component.ImportedComponentDefinitions) {
			theNameSpace := subComponent.NameSpace
			callCPPFunctionCode = append(callCPPFunctionCode, fmt.Sprintf("if (s%s == \"%s\") {", method.Params[0].ParamName, theNameSpace))
			wrapperName := "C" + ClassIdentifier + "Wrapper"
			callCPPFunctionCode = append(callCPPFunctionCode, fmt.Sprintf("  if (%s::sP%sWrapper.get() != nullptr) {", wrapperName, theNameSpace))
			callCPPFunctionCode = append(callCPPFunctionCode, fmt.Sprintf("    throw E%sInterfaceException(%s_ERROR_COULDNOTLOADLIBRARY);", NameSpace, strings.ToUpper(NameSpace)) )
			callCPPFunctionCode = append(callCPPFunctionCode, fmt.Sprintf("  }"))
			callCPPFunctionCode = append(callCPPFunctionCode, fmt.Sprintf("  %s::sP%sWrapper = %s::CWrapper::loadLibraryFromSymbolLookupMethod(p%s);", wrapperName, theNameSpace, theNameSpace, method.Params[1].ParamName))
			callCPPFunctionCode = append(callCPPFunctionCode, fmt.Sprintf("  bNameSpaceFound = true;"))
			callCPPFunctionCode = append(callCPPFunctionCode, fmt.Sprintf("}"))
		}
		callCPPFunctionCode = append(callCPPFunctionCode, "")
		callCPPFunctionCode = append(callCPPFunctionCode, "if (!bNameSpaceFound)")
		callCPPFunctionCode = append(callCPPFunctionCode, fmt.Sprintf("  throw E%sInterfaceException(%s_ERROR_COULDNOTLOADLIBRARY);", NameSpace, strings.ToUpper(NameSpace)) )
		callCPPFunctionCode = append(callCPPFunctionCode, "")
	} else if (isSpecialFunction == eSpecialMethodSymbolLookup) {
		callCPPFunctionCode = append(callCPPFunctionCode, fmt.Sprintf("*p%s = (void*)&_%s_getprocaddress_internal;", method.Params[0].ParamName, strings.ToLower(NameSpace)))
	} else {
		callCode, err := generateCallCPPFunctionCode(method, NameSpace, ClassIdentifier, ClassName, returnVariable, callParameters, isGlobal)
		if err != nil {
			return err
		}
		callCPPFunctionCode = append(callCPPFunctionCode, callCode)
	}
	
	journalInitFunctionCode := make([]string, 0)
	journalSuccessFunctionCode := make([]string, 0)
	if (doJournal) {
		journalInitFunctionCode, journalSuccessFunctionCode, err = generateJournalFunctionCode(method, NameSpace, ClassName, isGlobal)
		if err != nil {
			return err
		}
	}
	

	if !isGlobal {
		preCallCPPFunctionCode = append(preCallCPPFunctionCode, fmt.Sprintf("I%s%s* pI%s = dynamic_cast<I%s%s*>(pIBaseClass);", ClassIdentifier, ClassName, ClassName, ClassIdentifier, ClassName))
		preCallCPPFunctionCode = append(preCallCPPFunctionCode, fmt.Sprintf("if (!pI%s)", ClassName))
		preCallCPPFunctionCode = append(preCallCPPFunctionCode, fmt.Sprintf("  throw E%sInterfaceException(%s_ERROR_INVALIDCAST);", NameSpace, strings.ToUpper(NameSpace)) )
		preCallCPPFunctionCode = append(preCallCPPFunctionCode, "")
	}

	w.Writeln("%sResult %s(%s)", NameSpace, CMethodName, cparameters)
	w.Writeln("{")

	IBaseClassName := fmt.Sprintf("I%s%s", ClassIdentifier, BaseClassName)
	if !isGlobal {
		w.Writeln ("  %s* pIBaseClass = (%s *)p%s;\n", IBaseClassName, IBaseClassName, ClassName);
	} else {
		w.Writeln ("  %s* pIBaseClass = nullptr;\n", IBaseClassName);
	}
	
	
	if (doJournal) {
		w.Writeln("  P%sInterfaceJournalEntry pJournalEntry;", NameSpace);
	}
	
	w.Writeln("  try {")

	if (doJournal) {
		w.Writelns("    ", journalInitFunctionCode)
	}

	w.Writelns("    ", checkInputCPPFunctionCode)
	w.Writelns("    ", preCallCPPFunctionCode)
	w.Writelns("    ", callCPPFunctionCode)
	w.Writelns("    ", postCallCPPFunctionCode)

	journalHandleParam := "";
	
	if (doJournal) {
		w.Writelns("    ", journalSuccessFunctionCode)
		journalHandleParam = ", pJournalEntry.get()";
	}
	
	w.Writeln("    return %s_SUCCESS;", strings.ToUpper(NameSpace))
	w.Writeln("  }")
	w.Writeln("  catch (E%sInterfaceException & Exception) {", NameSpace)
	w.Writeln("    return handle%sException(pIBaseClass, Exception%s);", NameSpace, journalHandleParam)
	w.Writeln("  }")
	w.Writeln("  catch (std::exception & StdException) {")
	w.Writeln("    return handleStdException(pIBaseClass, StdException%s);", journalHandleParam)
	w.Writeln("  }")
	w.Writeln("  catch (...) {")
	w.Writeln("    return handleUnhandledException(pIBaseClass%s);", journalHandleParam)
	w.Writeln("  }")

	w.Writeln("}")
	w.Writeln("")
	return nil
}


func buildCPPStubClass(component ComponentDefinition, class ComponentDefinitionClass, NameSpace string, NameSpaceImplementation string, ClassIdentifier string, BaseName string, outputFolder string, indentString string, stubIdentifier string, forceRecreation bool) error {
		outClassName := "C" + ClassIdentifier + class.ClassName

		StubHeaderFileName := path.Join(outputFolder, BaseName + stubIdentifier + "_" +strings.ToLower(class.ClassName)+".hpp");
		StubImplFileName := path.Join(outputFolder, BaseName + stubIdentifier + "_" + strings.ToLower(class.ClassName)+".cpp");
		if !forceRecreation && ( FileExists(StubHeaderFileName) || FileExists(StubImplFileName) ) {
			log.Printf("Omitting recreation of Stub implementation for \"%s\"", outClassName)
			return nil
		}

		log.Printf("Creating \"%s\"", StubHeaderFileName)
		stubheaderw, err := CreateLanguageFile(StubHeaderFileName, indentString)
		if err != nil {
			return err
		}
		stubheaderw.WriteCLicenseHeader(component,
			fmt.Sprintf("This is the class declaration of %s", outClassName),
			false)
		
		log.Printf("Creating \"%s\"", StubImplFileName)
		stubimplw, err := CreateLanguageFile(StubImplFileName, indentString)
		if err != nil {
			return err
		}
		stubimplw.WriteCLicenseHeader(component,
			fmt.Sprintf("This is a stub class definition of %s", outClassName),
			false)

		stubheaderw.Writeln("")
		stubheaderw.Writeln("#ifndef __%s_%s", strings.ToUpper(NameSpace), strings.ToUpper(class.ClassName))
		stubheaderw.Writeln("#define __%s_%s", strings.ToUpper(NameSpace), strings.ToUpper(class.ClassName))
		stubheaderw.Writeln("")

		stubheaderw.Writeln("#include \"%s_interfaces.hpp\"", BaseName)
		if (component.isBaseClass(class)) {
			stubheaderw.Writeln("#include <vector>")
			stubheaderw.Writeln("#include <list>")
			stubheaderw.Writeln("#include <memory>")
		}
		stubheaderw.Writeln("")

		if (!component.isBaseClass(class)) {
			if (class.ParentClass == "") {
				class.ParentClass = component.Global.BaseClassName
			}
		}

		if class.ParentClass != "" {
			stubheaderw.Writeln("// Parent classes")
			stubheaderw.Writeln("#include \"%s%s_%s.hpp\"", BaseName, stubIdentifier, strings.ToLower(class.ParentClass))
			stubheaderw.Writeln("#ifdef _MSC_VER")
			stubheaderw.Writeln("#pragma warning(push)")
			stubheaderw.Writeln("#pragma warning(disable : 4250)")
			stubheaderw.Writeln("#endif")
		}
		stubheaderw.Writeln("")

		stubheaderw.Writeln("// Include custom headers here.")
		stubheaderw.Writeln("")

		stubheaderw.Writeln("")
		stubheaderw.Writeln("namespace %s {", NameSpace)
		stubheaderw.Writeln("namespace %s {", NameSpaceImplementation)
		stubheaderw.Writeln("")

		stubheaderw.Writeln("")
		stubheaderw.Writeln("/*************************************************************************************************************************")
		stubheaderw.Writeln(" Class declaration of %s ", outClassName)
		stubheaderw.Writeln("**************************************************************************************************************************/")
		stubheaderw.Writeln("")
		parentClassName := fmt.Sprintf("I%s%s", ClassIdentifier, class.ClassName)
		if "" != class.ParentClass {
			parentClassName = parentClassName + ", " + fmt.Sprintf("public virtual C%s%s", ClassIdentifier, class.ParentClass)
		}
		stubheaderw.Writeln("class %s : public virtual %s {", outClassName, parentClassName)
		stubheaderw.Writeln("private:")
		stubheaderw.Writeln("")
		
		if (component.isBaseClass(class)) {
			stubheaderw.Writeln("  std::unique_ptr<std::list<std::string>> m_pErrors;")
			stubheaderw.Writeln("  %s_uint32 m_nReferenceCount = 1;", NameSpace)
			stubheaderw.Writeln("")
		}
		stubheaderw.Writeln("  /**")
		stubheaderw.Writeln("  * Put private members here.")
		stubheaderw.Writeln("  */")
		stubheaderw.Writeln("")

		stubheaderw.Writeln("protected:")
		stubheaderw.Writeln("")
		stubheaderw.Writeln("  /**")
		stubheaderw.Writeln("  * Put protected members here.")
		stubheaderw.Writeln("  */")
		stubheaderw.Writeln("")
		stubheaderw.Writeln("public:")
		stubheaderw.Writeln("")
		stubheaderw.Writeln("  /**")
		stubheaderw.Writeln("  * Put additional public members here. They will not be visible in the external API.")
		stubheaderw.Writeln("  */")
		stubheaderw.Writeln("")


		stubimplw.Writeln("#include \"%s%s_%s.hpp\"", BaseName, stubIdentifier, strings.ToLower(class.ClassName))
		stubimplw.Writeln("#include \"%s_interfaceexception.hpp\"", BaseName)
		stubimplw.Writeln("")
		stubimplw.Writeln("// Include custom headers here.")
		stubimplw.Writeln("")


		stubimplw.Writeln("")
		stubimplw.Writeln("using namespace %s::%s;", NameSpace, NameSpaceImplementation)
		stubimplw.Writeln("")

		stubimplw.Writeln("/*************************************************************************************************************************")
		stubimplw.Writeln(" Class definition of %s ", outClassName)
		stubimplw.Writeln("**************************************************************************************************************************/")
		stubimplw.Writeln("")

		if (component.isBaseClass(class)) {
			var methods [5]ComponentDefinitionMethod
			methods[0] = GetLastErrorMessageMethod()
			methods[1] = ClearErrorMessageMethod()
			methods[2] = RegisterErrorMessageMethod()
			methods[3] = IncRefCountMethod()
			methods[4] = DecRefCountMethod()

			var implementations [5][]string
			implementations[0] = append(implementations[0], "if (m_pErrors && !m_pErrors->empty()) {")
			implementations[0] = append(implementations[0], "  sErrorMessage = m_pErrors->back();")
			implementations[0] = append(implementations[0], "  return true;")
			implementations[0] = append(implementations[0], "} else {")
			implementations[0] = append(implementations[0], "  sErrorMessage = \"\";")
			implementations[0] = append(implementations[0], "  return false;")
			implementations[0] = append(implementations[0], "}")

			implementations[1] = append(implementations[1], "m_pErrors.reset();")

			implementations[2] = append(implementations[2], "if (!m_pErrors) {")
			implementations[2] = append(implementations[2], "  m_pErrors.reset(new std::list<std::string>());")
			implementations[2] = append(implementations[2], "}")
			implementations[2] = append(implementations[2], "m_pErrors->clear();")
			implementations[2] = append(implementations[2], "m_pErrors->push_back(sErrorMessage);")

			implementations[3] = append(implementations[3], "++m_nReferenceCount;")

			implementations[4] = append(implementations[4], "m_nReferenceCount--;")
			implementations[4] = append(implementations[4], "if (!m_nReferenceCount) {")
			implementations[4] = append(implementations[4], "  delete this;")
			implementations[4] = append(implementations[4], "  return true;")
			implementations[4] = append(implementations[4], "}")
			implementations[4] = append(implementations[4], "return false;")

			for i := 0; i < len(methods); i++ {
				methodstring, implementationdeclaration, err := buildCPPInterfaceMethodDeclaration(methods[i], class.ClassName, NameSpace, ClassIdentifier, BaseName, stubimplw.IndentString, false, false, false)
				if (err!=nil) {
					return err
				}
				stubheaderw.Writeln("%s;", methodstring)
				stubheaderw.Writeln("")

				stubimplw.Writeln("%s", implementationdeclaration)
				stubimplw.Writeln("{")
				stubimplw.Writelns("  ", implementations[i])
				stubimplw.Writeln("}")
				stubimplw.Writeln("")
			}
		}

		stubheaderw.Writeln("")
		stubheaderw.Writeln("  /**")
		stubheaderw.Writeln("  * Public member functions to implement.")
		stubheaderw.Writeln("  */")
		stubheaderw.Writeln("")

		for j := 0; j < len(class.Methods); j++ {
			method := class.Methods[j]
			methodstring, implementationdeclaration, err := buildCPPInterfaceMethodDeclaration(method, class.ClassName, NameSpace, ClassIdentifier, BaseName, stubimplw.IndentString, false, false, false)
			if err != nil {
				return err
			}
			stubheaderw.Writeln("%s;", methodstring)
			stubheaderw.Writeln("")

			stubimplw.Writeln("%s", implementationdeclaration)
			stubimplw.Writeln("{")
			stubimplw.Writeln("  throw E%sInterfaceException(%s_ERROR_NOTIMPLEMENTED);", NameSpace, strings.ToUpper(NameSpace))
			stubimplw.Writeln("}")
			stubimplw.Writeln("")
		}

		stubheaderw.Writeln("};")
		stubheaderw.Writeln("")

		stubheaderw.Writeln("} // namespace %s", NameSpaceImplementation)
		stubheaderw.Writeln("} // namespace %s", NameSpace)
		stubheaderw.Writeln("")

		if class.ParentClass != "" {
			stubheaderw.Writeln("#ifdef _MSC_VER")
			stubheaderw.Writeln("#pragma warning(pop)")
			stubheaderw.Writeln("#endif")
		}
		stubheaderw.Writeln("#endif // __%s_%s", strings.ToUpper(NameSpace), strings.ToUpper(class.ClassName))
	return nil
}

func buildCPPStub(component ComponentDefinition, NameSpace string, NameSpaceImplementation string, ClassIdentifier string, BaseName string, outputFolder string, indentString string, stubIdentifier string, forceRecreation bool) error {

	for i := 0; i < len(component.Classes); i++ {
		class := component.Classes[i]
		err :=  buildCPPStubClass(component, class, NameSpace, NameSpaceImplementation, ClassIdentifier, BaseName, outputFolder, indentString, stubIdentifier, forceRecreation)
		if err != nil {
			return err
		}
	}

	return nil
}

func getCppVariableName (param ComponentDefinitionParam) (string) {
	switch (param.ParamType) {
		case "uint8", "uint16", "uint32", "uint64", "int8", "int16", "int32", "int64":
			return "n" + param.ParamName;
		case "string":
			return "s" + param.ParamName;
		case "bool":
			return "b" + param.ParamName;
		case "single":
			return "f" + param.ParamName;
		case "basicarray", "structarray":
			return "p" + param.ParamName + "Buffer";
		case "double":
			return "d" + param.ParamName;
		case "pointer":
			return "p" + param.ParamName;
		case "enum":
			return "e" + param.ParamName;
		case "struct":
			return param.ParamName;
		case "class", "optionalclass":
			return "p" + param.ParamName;
		case "functiontype":
			return "p" + param.ParamName;
	}
	
	log.Fatal ("Invalid parameter type: ", param.ParamType);
	
	return "";
}

func buildCPPInterfaceMethodDeclaration(method ComponentDefinitionMethod, className string, NameSpace string, ClassIdentifier string, BaseName string, indentString string, isGlobal bool, isVirtual bool, writeComment bool) (string, string, error) {
	parameters := ""
	returntype := "void"
	commentcode := ""

	templateimplementation := ""

	for k := 0; k < len(method.Params); k++ {

		param := method.Params[k]
		paramNameSpaceCPP, paramClassNameCPP, _ := decomposeParamClassNameCPP(param.ParamClass)

		switch param.ParamPass {
		case "in":

			cppParamType := getCppParamType(param, NameSpace, true)

			if parameters != "" {
				parameters = parameters + ", "
			}

			switch param.ParamType {
			case "uint8", "uint16", "uint32", "uint64", "int8", "int16", "int32", "int64":
				commentcode = commentcode + fmt.Sprintf(indentString + "* @param[in] n%s - %s\n", param.ParamName, param.ParamDescription)
				parameters = parameters + fmt.Sprintf("const %s n%s", cppParamType, param.ParamName)

			case "string":
				commentcode = commentcode + fmt.Sprintf(indentString + "* @param[in] s%s - %s\n", param.ParamName, param.ParamDescription)
				parameters = parameters + fmt.Sprintf("const std::string & s%s", param.ParamName)

			case "bool":
				commentcode = commentcode + fmt.Sprintf(indentString + "* @param[in] b%s - %s\n", param.ParamName, param.ParamDescription)
				parameters = parameters + fmt.Sprintf("const %s b%s", cppParamType, param.ParamName)

			case "single":
				commentcode = commentcode + fmt.Sprintf(indentString + "* @param[in] f%s - %s\n", param.ParamName, param.ParamDescription)
				parameters = parameters + fmt.Sprintf("const %s f%s", cppParamType, param.ParamName)

			case "double":
				commentcode = commentcode + fmt.Sprintf(indentString + "* @param[in] d%s - %s\n", param.ParamName, param.ParamDescription)
				parameters = parameters + fmt.Sprintf("const %s d%s", cppParamType, param.ParamName)

			case "pointer":
				commentcode = commentcode + fmt.Sprintf(indentString + "* @param[in] n%s - %s\n", param.ParamName, param.ParamDescription)
				parameters = parameters + fmt.Sprintf("const %s p%s", cppParamType, param.ParamName)

			case "enum":
				commentcode = commentcode + fmt.Sprintf(indentString + "* @param[in] e%s - %s\n", param.ParamName, param.ParamDescription)
				parameters = parameters + fmt.Sprintf("const %s e%s", cppParamType, param.ParamName)

			case "struct":
				commentcode = commentcode + fmt.Sprintf(indentString + "* @param[in] %s - %s\n", param.ParamName, param.ParamDescription)
				parameters = parameters + fmt.Sprintf("const %s %s", cppParamType, param.ParamName)

			case "class", "optionalclass":
				commentcode = commentcode + fmt.Sprintf(indentString + "* @param[in] p%s - %s\n", param.ParamName, param.ParamDescription)
				if len(paramNameSpaceCPP) > 0 {
					// TODO: ClassIdentifier is incorrect! get via // component.ImportedComponentDefinitions[paramNameSpace].Bindings
					parameters = parameters + fmt.Sprintf("%sP%s%s p%s", paramNameSpaceCPP, ClassIdentifier, paramClassNameCPP, param.ParamName)
				} else {
					parameters = parameters + fmt.Sprintf("I%s%s* p%s", ClassIdentifier, param.ParamClass, param.ParamName)
				}

			case "basicarray":
				commentcode = commentcode + fmt.Sprintf(indentString + "* @param[in] n%sBufferSize - Number of elements in buffer\n", param.ParamName)
				commentcode = commentcode + fmt.Sprintf(indentString + "* @param[in] p%sBuffer - %s\n", param.ParamName, param.ParamDescription)
				parameters = parameters + fmt.Sprintf("const %s_uint64 n%sBufferSize, const %s p%sBuffer", NameSpace, param.ParamName, cppParamType, param.ParamName)

			case "structarray":
				commentcode = commentcode + fmt.Sprintf(indentString + "* @param[in] n%sBufferSize - Number of elements in buffer\n", param.ParamName)
				commentcode = commentcode + fmt.Sprintf(indentString + "* @param[in] p%sBuffer - %s\n", param.ParamName, param.ParamDescription)
				parameters = parameters + fmt.Sprintf("const %s_uint64 n%sBufferSize, const %s p%sBuffer", NameSpace, param.ParamName, cppParamType, param.ParamName)

			case "functiontype":
				commentcode = commentcode + fmt.Sprintf(indentString + "* @param[in] p%s - callback function\n", param.ParamName)
				parameters = parameters + fmt.Sprintf("const %s p%s", cppParamType, param.ParamName)

			default:
				return "", "", fmt.Errorf("invalid method parameter type \"%s\" for %s.%s(%s)", param.ParamType, className, method.MethodName, param.ParamName)
			}

		case "out":

			cppParamType := getCppParamType(param, NameSpace, false)

			if parameters != "" {
				parameters = parameters + ", "
			}

			switch param.ParamType {
			case "uint8", "uint16", "uint32", "uint64", "int8", "int16", "int32", "int64":
				commentcode = commentcode + fmt.Sprintf(indentString + "* @param[out] n%s - %s\n", param.ParamName, param.ParamDescription)
				parameters = parameters + fmt.Sprintf("%s & n%s", cppParamType, param.ParamName)

			case "bool":
				commentcode = commentcode + fmt.Sprintf(indentString + "* @param[out] b%s - %s\n", param.ParamName, param.ParamDescription)
				parameters = parameters + fmt.Sprintf("%s & b%s", cppParamType, param.ParamName)

			case "single":
				commentcode = commentcode + fmt.Sprintf(indentString + "* @param[out] f%s - %s\n", param.ParamName, param.ParamDescription)
				parameters = parameters + fmt.Sprintf("%s & f%s", cppParamType, param.ParamName)

			case "double":
				commentcode = commentcode + fmt.Sprintf(indentString + "* @param[out] d%s - %s\n", param.ParamName, param.ParamDescription)
				parameters = parameters + fmt.Sprintf("%s & d%s", cppParamType, param.ParamName)

			case "pointer":
				commentcode = commentcode + fmt.Sprintf(indentString + "* @param[out] d%s - %s\n", param.ParamName, param.ParamDescription)
				parameters = parameters + fmt.Sprintf("%s & p%s", cppParamType, param.ParamName)

			case "string":
				commentcode = commentcode + fmt.Sprintf(indentString + "* @param[out] s%s - %s\n", param.ParamName, param.ParamDescription)
				parameters = parameters + fmt.Sprintf("std::string & s%s", param.ParamName)

			case "enum":
				commentcode = commentcode + fmt.Sprintf(indentString + "* @param[out] e%s - %s\n", param.ParamName, param.ParamDescription)
				parameters = parameters + fmt.Sprintf("%s & e%s", cppParamType, param.ParamName)

			case "struct":
				commentcode = commentcode + fmt.Sprintf(indentString + "* @param[out] s%s - %s\n", param.ParamName, param.ParamDescription)
				parameters = parameters + fmt.Sprintf("%s & s%s", cppParamType, param.ParamName)

			case "basicarray":
				commentcode = commentcode + fmt.Sprintf(indentString + "* @param[in] n%sBufferSize - Number of elements in buffer\n", param.ParamName)
				commentcode = commentcode + fmt.Sprintf(indentString + "* @param[out] p%sNeededCount - will be filled with the count of the written structs, or needed buffer size.\n", param.ParamName)
				commentcode = commentcode + fmt.Sprintf(indentString + "* @param[out] p%sBuffer - %s buffer of %s\n", param.ParamName, param.ParamClass, param.ParamDescription)
				parameters = parameters + fmt.Sprintf("%s_uint64 n%sBufferSize, %s_uint64* p%sNeededCount, %s p%sBuffer", NameSpace, param.ParamName, NameSpace, param.ParamName, cppParamType, param.ParamName)

			case "structarray":
				commentcode = commentcode + fmt.Sprintf(indentString + "* @param[in] n%sBufferSize - Number of elements in buffer\n", param.ParamName)
				commentcode = commentcode + fmt.Sprintf(indentString + "* @param[out] p%sNeededCount - will be filled with the count of the written structs, or needed buffer size.\n", param.ParamName)
				commentcode = commentcode + fmt.Sprintf(indentString + "* @param[out] p%sBuffer - %s buffer of %s\n", param.ParamName, param.ParamClass, param.ParamDescription)
				parameters = parameters + fmt.Sprintf("%s_uint64 n%sBufferSize, %s_uint64* p%sNeededCount, %s p%sBuffer", NameSpace, param.ParamName, NameSpace, param.ParamName, cppParamType, param.ParamName)

			case "class", "optionalclass":
				commentcode = commentcode + fmt.Sprintf(indentString + "* @param[out] p%s - %s\n",  param.ParamName, param.ParamDescription)
				if len(paramNameSpaceCPP) > 0 {
					// TODO: ClassIdentifier is incorrect! get via // component.ImportedComponentDefinitions[paramNameSpace].Bindings
					parameters = parameters + fmt.Sprintf("%sP%s%s p%s", paramNameSpaceCPP, ClassIdentifier, paramClassNameCPP, param.ParamName)
				} else {
					parameters = parameters + fmt.Sprintf("I%s%s*& p%s", ClassIdentifier, param.ParamClass, param.ParamName)
				}
				

			default:
				return "", "", fmt.Errorf("invalid method parameter type \"%s\" for %s.%s(%s)", param.ParamType, className, method.MethodName, param.ParamName)
			}

		case "return":
			currentReturnType := getCppParamType(param, NameSpace, false)
			switch param.ParamType {
			case "uint8", "uint16", "uint32", "uint64", "int8", "int16", "int32", "int64", "bool", "single", "double", "pointer", "string", "enum", "struct":
				returntype = currentReturnType
				commentcode = commentcode + fmt.Sprintf(indentString + "* @return %s\n", param.ParamDescription)

			case "class", "optionalclass":
				commentcode = commentcode + fmt.Sprintf(indentString + "* @return %s\n", param.ParamDescription)
				if len(paramNameSpaceCPP) > 0 {
					// TODO: ClassIdentifier is incorrect! get via // component.ImportedComponentDefinitions[paramNameSpace].Bindings
					returntype = fmt.Sprintf("%sP%s%s", paramNameSpaceCPP, ClassIdentifier, paramClassNameCPP)
				} else {
					returntype = fmt.Sprintf("I%s%s *", ClassIdentifier, param.ParamClass)
				}

			default:
				return "", "", fmt.Errorf("invalid method parameter type \"%s\" for %s.%s(%s)", param.ParamType, className, method.MethodName, param.ParamName)
			}

		default:
			return "", "", fmt.Errorf("invalid method parameter passing \"%s\" for %s.%s(%s)", param.ParamPass, className, method.MethodName, param.ParamName)
		}
	}
	outstring := "";
	if (writeComment) {
		outstring = indentString + "/**\n"
		outstring = outstring + fmt.Sprintf(indentString + "* I%s%s::%s - %s\n", ClassIdentifier, className, method.MethodName, method.MethodDescription)
		outstring = outstring + commentcode
		outstring = outstring + indentString + "*/\n"
	}
	
	if isGlobal {
		if isVirtual {
			return "", "", fmt.Errorf("Method \"%s\"can not be virtual static", method.MethodName)
		}
		outstring = outstring + fmt.Sprintf(indentString + "static %s %s(%s)", returntype, method.MethodName, parameters)
	} else {
		if isVirtual {
			outstring = outstring + fmt.Sprintf(indentString + "virtual %s %s(%s) = 0", returntype, method.MethodName, parameters)
		} else {
			outstring = outstring + fmt.Sprintf(indentString + "%s %s(%s) override", returntype, method.MethodName, parameters)
		}
	}

	if isGlobal {
		templateimplementation = fmt.Sprintf("%s C%s%s::%s(%s)", returntype, ClassIdentifier, className, method.MethodName, parameters)
	} else {
		templateimplementation = fmt.Sprintf("%s C%s%s::%s(%s)", returntype, ClassIdentifier, className, method.MethodName, parameters)
	}

	return outstring, templateimplementation, nil
}

func getCppParamType (param ComponentDefinitionParam, NameSpace string, isInput bool) (string) {
	cppClassPrefix := "C" + NameSpace;
	switch (param.ParamType) {
		case "uint8", "uint16", "uint32", "uint64", "int8", "int16", "int32", "int64", "single", "double":
			return fmt.Sprintf ("%s_%s", NameSpace, param.ParamType);
		case "string":
			return fmt.Sprintf ("std::string");
		case "bool":
			return fmt.Sprintf ("bool");
		case "pointer":
			return fmt.Sprintf ("%s_pvoid", NameSpace);

		case "basicarray":
			cppBasicType := "";
			switch (param.ParamClass) {
			case "uint8", "uint16", "uint32", "uint64", "int8", "int16", "int32", "int64", "single", "double":
				cppBasicType = fmt.Sprintf ("%s_%s", NameSpace, param.ParamClass);
			case "bool":
				cppBasicType = "bool";
			case "pointer":
				cppBasicType = fmt.Sprintf ("%s_pvoid", NameSpace);
			default:
				log.Fatal ("Invalid parameter type: ", param.ParamClass);
			}
			return fmt.Sprintf ("%s *", cppBasicType);
		case "structarray":
			return fmt.Sprintf ("%s::s%s *", NameSpace, param.ParamClass);
		case "enum":
			return fmt.Sprintf ("%s::e%s", NameSpace, param.ParamClass);
		case "struct":
			return fmt.Sprintf ("%s::s%s", NameSpace, param.ParamClass);
		case "class", "optionalclass":
			if (isInput) {
				return fmt.Sprintf ("%s%s *", cppClassPrefix, param.ParamClass);
			}
			return fmt.Sprintf ("P%s", param.ParamClass);
		case "functiontype":
			return fmt.Sprintf ("%s::%s", NameSpace, param.ParamClass);
	}
	
	log.Fatal ("Invalid parameter type: ", param.ParamType);
	return "";
}

func generatePrePostCallCPPFunctionCode(component ComponentDefinition, method ComponentDefinitionMethod, NameSpace string, ClassIdentifier string, ClassName string, BaseClassName string) ([]string, []string, []string, string, string, error) {
	preCallCode := make([]string, 0)
	postCallCode := make([]string, 0)
	callParameters := ""
	returnVariable := ""
	checkInputCode := make([]string, 0)
	IBaseClassName := fmt.Sprintf("I%s%s", ClassIdentifier, BaseClassName)
	for k := 0; k < len(method.Params); k++ {
		param := method.Params[k]
		variableName := getCppVariableName(param)

		switch param.ParamPass {
		case "in":

			if callParameters != "" {
				callParameters = callParameters + ", "
			}

			switch param.ParamType {
			case "bool", "uint8", "uint16", "uint32", "uint64", "int8", "int16", "int32", "int64", "single", "double", "pointer":
				callParameters = callParameters + variableName
			case "enum":
				callParameters = callParameters + "e" + param.ParamName
			case "struct":
				callParameters = callParameters + "*p" + param.ParamName
			case "basicarray":
				checkInputCode = append(checkInputCode, fmt.Sprintf("if ( (!p%sBuffer) && (n%sBufferSize>0))", param.ParamName, param.ParamName))
				checkInputCode = append(checkInputCode, fmt.Sprintf("  throw E%sInterfaceException (%s_ERROR_INVALIDPARAM);", NameSpace, strings.ToUpper(NameSpace)))
				callParameters = callParameters + fmt.Sprintf("n%sBufferSize, ", param.ParamName) + variableName
			case "structarray":
				checkInputCode = append(checkInputCode, fmt.Sprintf("if ( (!p%sBuffer) && (n%sBufferSize>0))", param.ParamName, param.ParamName))
				checkInputCode = append(checkInputCode, fmt.Sprintf("  throw E%sInterfaceException (%s_ERROR_INVALIDPARAM);", NameSpace, strings.ToUpper(NameSpace)))
				callParameters = callParameters + fmt.Sprintf("n%sBufferSize, ", param.ParamName) + variableName

			case "class", "optionalclass":
				paramNameSpace, paramClassName, _ := decomposeParamClassName(param.ParamClass)
				if len(paramNameSpace) > 0 {
					theWrapper := "C" + ClassIdentifier + "Wrapper::sP" + paramNameSpace + "Wrapper"
					preCallCode = append(preCallCode, fmt.Sprintf("%s::P%s pI%s = std::make_shared<%s::C%s>(%s.get(), p%s);", paramNameSpace, paramClassName, param.ParamName, paramNameSpace, paramClassName, theWrapper, param.ParamName))
					acqurireMethod := component.ImportedComponentDefinitions[paramNameSpace].Global.AcquireMethod
					preCallCode = append(preCallCode, fmt.Sprintf("%s->%s(pI%s.get());", theWrapper, acqurireMethod, param.ParamName))
				} else {
					preCallCode = append(preCallCode, fmt.Sprintf("%s* pIBaseClass%s = (%s *)p%s;", IBaseClassName, param.ParamName, IBaseClassName, param.ParamName))
					preCallCode = append(preCallCode, fmt.Sprintf("I%s%s* pI%s = dynamic_cast<I%s%s*>(pIBaseClass%s);", ClassIdentifier, param.ParamClass, param.ParamName, ClassIdentifier, param.ParamClass, param.ParamName))
				}
				
				if (param.ParamType == "class") {
					preCallCode = append(preCallCode, fmt.Sprintf("if (!pI%s)", param.ParamName))
					preCallCode = append(preCallCode, fmt.Sprintf("  throw E%sInterfaceException (%s_ERROR_INVALIDCAST);", NameSpace, strings.ToUpper(NameSpace)))
					preCallCode = append(preCallCode, "")
				}
				
				callParameters = callParameters + fmt.Sprintf("pI%s", param.ParamName)
			case "string":
				checkInputCode = append(checkInputCode, fmt.Sprintf("if (p%s == nullptr)", param.ParamName))
				checkInputCode = append(checkInputCode, fmt.Sprintf("  throw E%sInterfaceException (%s_ERROR_INVALIDPARAM);", NameSpace, strings.ToUpper(NameSpace)))
				preCallCode = append(preCallCode, fmt.Sprintf("std::string %s(p%s);", variableName, param.ParamName))
				callParameters = callParameters + variableName

			case "functiontype":
				callParameters = callParameters + variableName

			default:
				return checkInputCode, preCallCode, postCallCode, "", "", fmt.Errorf("method parameter type \"%s\" of param pass \"%s\" is not implemented for %s::%s(%s) )", param.ParamType, param.ParamPass, ClassName, method.MethodName, param.ParamName)
			}

		case "out":
			if callParameters != "" {
				callParameters = callParameters + ", "
			}

			switch param.ParamType {

			case "bool", "uint8", "uint16", "uint32", "uint64", "int8", "int16", "int32", "int64", "single", "double", "enum", "struct", "pointer":
				checkInputCode = append(checkInputCode, fmt.Sprintf("if (!p%s)", param.ParamName))
				checkInputCode = append(checkInputCode, fmt.Sprintf("  throw E%sInterfaceException (%s_ERROR_INVALIDPARAM);", NameSpace, strings.ToUpper(NameSpace)))
				callParameters = callParameters + "*p" + param.ParamName

			case "basicarray", "structarray":
				checkInputCode = append(checkInputCode, fmt.Sprintf("if ((!p%sBuffer) && !(p%sNeededCount))", param.ParamName, param.ParamName))
				checkInputCode = append(checkInputCode, fmt.Sprintf("  throw E%sInterfaceException (%s_ERROR_INVALIDPARAM);", NameSpace, strings.ToUpper(NameSpace)))
				callParameters = callParameters + fmt.Sprintf("n%sBufferSize, p%sNeededCount, ", param.ParamName, param.ParamName) + variableName

			case "string":
				checkInputCode = append(checkInputCode, fmt.Sprintf("if ( (!p%sBuffer) && !(p%sNeededChars) )", param.ParamName, param.ParamName))
				checkInputCode = append(checkInputCode, fmt.Sprintf("  throw E%sInterfaceException (%s_ERROR_INVALIDPARAM);", NameSpace, strings.ToUpper(NameSpace)))

				preCallCode = append(preCallCode, fmt.Sprintf("std::string %s(\"\");", variableName))
				callParameters = callParameters + variableName

				postCallCode = append(postCallCode, fmt.Sprintf("if (p%sNeededChars)", param.ParamName))
				postCallCode = append(postCallCode, fmt.Sprintf("  *p%sNeededChars = (%s_uint32) (%s.size()+1);", param.ParamName, NameSpace, variableName))
				postCallCode = append(postCallCode, fmt.Sprintf("if (p%sBuffer) {", param.ParamName))
				postCallCode = append(postCallCode, fmt.Sprintf("  if (%s.size() >= n%sBufferSize)", variableName, param.ParamName))
				postCallCode = append(postCallCode, fmt.Sprintf("    throw E%sInterfaceException (%s_ERROR_BUFFERTOOSMALL);", NameSpace, strings.ToUpper(NameSpace)))
				postCallCode = append(postCallCode, fmt.Sprintf("  for (size_t i%s = 0; i%s < %s.size(); i%s++)", param.ParamName, param.ParamName, variableName, param.ParamName))
				postCallCode = append(postCallCode, fmt.Sprintf("    p%sBuffer[i%s] = %s[i%s];", param.ParamName, param.ParamName, variableName, param.ParamName))
				postCallCode = append(postCallCode, fmt.Sprintf("  p%sBuffer[%s.size()] = 0;", param.ParamName, variableName))
				postCallCode = append(postCallCode, fmt.Sprintf("}"))

			case "class", "optionalclass":
				checkInputCode = append(checkInputCode, fmt.Sprintf("if (p%s == nullptr)", param.ParamName))
				checkInputCode = append(checkInputCode, fmt.Sprintf("  throw E%sInterfaceException (%s_ERROR_INVALIDPARAM);", NameSpace, strings.ToUpper(NameSpace)))

				paramNameSpace, paramClassName, _ := decomposeParamClassName(param.ParamClass)
				if len(paramNameSpace) > 0 {
					outVarName := fmt.Sprintf("p%s%s", paramNameSpace, param.ParamName)
					preCallCode = append(preCallCode, fmt.Sprintf("%s::P%s %s;", paramNameSpace, paramClassName, outVarName))
					theWrapper := "C" + ClassIdentifier + "Wrapper::sP" + paramNameSpace + "Wrapper"
					acqurireMethod := component.ImportedComponentDefinitions[paramNameSpace].Global.AcquireMethod
					postCallCode = append(postCallCode, fmt.Sprintf("%s->%s(%s.get());", theWrapper, acqurireMethod, outVarName))
					postCallCode = append(postCallCode, fmt.Sprintf("*%s = %s->GetHandle();", variableName, outVarName));
					callParameters = callParameters + outVarName
				} else {
					preCallCode = append(preCallCode, fmt.Sprintf("%s* pBase%s(nullptr);", IBaseClassName, param.ParamName))
					postCallCode = append(postCallCode, fmt.Sprintf("*%s = (%s*)(pBase%s);", variableName, IBaseClassName, param.ParamName));
					callParameters = callParameters + "pBase" + param.ParamName
				}
				

			default:
				return checkInputCode, preCallCode, postCallCode, "", "", fmt.Errorf("method parameter type \"%s\" of param pass \"%s\" is not implemented for %s::%s(%s) )", param.ParamType, param.ParamPass, ClassName, method.MethodName, param.ParamName)
			}

		case "return":

			switch param.ParamType {

			case "uint8", "uint16", "uint32", "uint64", "int8", "int16", "int32", "int64", "bool", "single", "double", "enum", "pointer":
				checkInputCode = append(checkInputCode, fmt.Sprintf("if (p%s == nullptr)", param.ParamName))
				checkInputCode = append(checkInputCode, fmt.Sprintf("  throw E%sInterfaceException (%s_ERROR_INVALIDPARAM);", NameSpace, strings.ToUpper(NameSpace)))

				returnVariable = fmt.Sprintf("*p%s", param.ParamName)

			case "struct":
				checkInputCode = append(checkInputCode, fmt.Sprintf("if (p%s == nullptr)", param.ParamName))
				checkInputCode = append(checkInputCode, fmt.Sprintf("throw E%sInterfaceException (%s_ERROR_INVALIDPARAM);", NameSpace, strings.ToUpper(NameSpace)))

				returnVariable = fmt.Sprintf("*p%s", param.ParamName)

			case "string":
				checkInputCode = append(checkInputCode, fmt.Sprintf("if ( (!p%sBuffer) && !(p%sNeededChars) )", param.ParamName, param.ParamName))
				checkInputCode = append(checkInputCode, fmt.Sprintf("  throw E%sInterfaceException (%s_ERROR_INVALIDPARAM);", NameSpace, strings.ToUpper(NameSpace)))

				preCallCode = append(preCallCode, fmt.Sprintf("std::string %s(\"\");", variableName))
				returnVariable = variableName

				postCallCode = append(postCallCode, fmt.Sprintf("if (p%sNeededChars)", param.ParamName))
				postCallCode = append(postCallCode, fmt.Sprintf("  *p%sNeededChars = (%s_uint32) (%s.size()+1);", param.ParamName, NameSpace, variableName))
				postCallCode = append(postCallCode, fmt.Sprintf("if (p%sBuffer) {", param.ParamName))
				postCallCode = append(postCallCode, fmt.Sprintf("  if (%s.size() >= n%sBufferSize)", variableName, param.ParamName))
				postCallCode = append(postCallCode, fmt.Sprintf("    throw E%sInterfaceException (%s_ERROR_BUFFERTOOSMALL);", NameSpace, strings.ToUpper(NameSpace)))
				postCallCode = append(postCallCode, fmt.Sprintf("  for (size_t i%s = 0; i%s < %s.size(); i%s++)", param.ParamName, param.ParamName, variableName, param.ParamName))
				postCallCode = append(postCallCode, fmt.Sprintf("    p%sBuffer[i%s] = %s[i%s];", param.ParamName, param.ParamName, variableName, param.ParamName))
				postCallCode = append(postCallCode, fmt.Sprintf("  p%sBuffer[%s.size()] = 0;", param.ParamName, variableName))
				postCallCode = append(postCallCode, fmt.Sprintf("}"))

			case "class", "optionalclass":
				checkInputCode = append(checkInputCode, fmt.Sprintf("if (p%s == nullptr)", param.ParamName))
				checkInputCode = append(checkInputCode, fmt.Sprintf("  throw E%sInterfaceException (%s_ERROR_INVALIDPARAM);", NameSpace, strings.ToUpper(NameSpace)))

				paramNameSpace, paramClassName, _ := decomposeParamClassName(param.ParamClass)
				if len(paramNameSpace) > 0 {
					preCallCode = append(preCallCode, fmt.Sprintf("%s::P%s p%s%s;", paramNameSpace, paramClassName, paramNameSpace, param.ParamName))
					theWrapper := "C" + ClassIdentifier + "Wrapper::sP" + paramNameSpace + "Wrapper"
					acqurireMethod := component.ImportedComponentDefinitions[paramNameSpace].Global.AcquireMethod
					returnVariable = fmt.Sprintf("p%s%s", paramNameSpace, param.ParamName)
					postCallCode = append(postCallCode, fmt.Sprintf("%s->%s(p%s%s.get());", theWrapper, acqurireMethod, paramNameSpace, param.ParamName))
					postCallCode = append(postCallCode, fmt.Sprintf("*%s = p%s%s->GetHandle();", variableName, paramNameSpace, param.ParamName));
				} else {
					preCallCode = append(preCallCode, fmt.Sprintf("%s* pBase%s(nullptr);", IBaseClassName, param.ParamName))
					returnVariable = fmt.Sprintf("pBase%s", param.ParamName)
					postCallCode = append(postCallCode, fmt.Sprintf("*%s = (%s*)(pBase%s);", variableName, IBaseClassName, param.ParamName));
				}

			default:
				return checkInputCode, preCallCode, postCallCode, "", "", fmt.Errorf("invalid method parameter type \"%s\" for %s.%s(%s)", param.ParamType, ClassName, method.MethodName, param.ParamName)
			}

		default:
			return checkInputCode, preCallCode, postCallCode, "", "", fmt.Errorf("invalid method parameter passing \"%s\" for %s.%s(%s)", param.ParamPass, ClassName, method.MethodName, param.ParamName)
		}
	}

	return checkInputCode, preCallCode, postCallCode, returnVariable, callParameters, nil
}

func generateCallCPPFunctionCode(method ComponentDefinitionMethod, NameSpace string, ClassIdentifier string, ClassName string, returnVariable string, callParameters string, isGlobal bool) (string, error) {
	returnValueCode := ""
	if returnVariable != "" {
		returnValueCode = returnVariable + " = "
	}
	callFunctionCode := ""
	if isGlobal {
		callFunctionCode = fmt.Sprintf("%sC%s%s::%s(%s);\n", returnValueCode, ClassIdentifier, ClassName, method.MethodName, callParameters)
	} else {
		callFunctionCode = fmt.Sprintf("%spI%s->%s(%s);\n", returnValueCode, ClassName, method.MethodName, callParameters)
	}
	return callFunctionCode, nil
}


func generateJournalFunctionCode(method ComponentDefinitionMethod, NameSpace string, ClassName string, isGlobal bool) ([]string, []string, error) {

	journalInitFunctionCode := make([]string,0)
	journalSuccessFunctionCode := make([]string,0)
	
	
	journalInitFunctionCode = append(journalInitFunctionCode, fmt.Sprintf("if (m_GlobalJournal.get() != nullptr)  {"))
	if (isGlobal) {
		journalInitFunctionCode = append(journalInitFunctionCode, fmt.Sprintf("  pJournalEntry = m_GlobalJournal->beginStaticFunction(\"%s\");", method.MethodName))
	} else {
		journalInitFunctionCode = append(journalInitFunctionCode, fmt.Sprintf("  pJournalEntry = m_GlobalJournal->beginClassMethod(p%s, \"%s\", \"%s\");", ClassName, ClassName, method.MethodName))
	}
	
	for k := 0; k < len(method.Params); k++ {
		param := method.Params[k]
		variableName := getCppVariableName(param)
		
		if (param.ParamPass == "in") {
			journalCall := "";
			
			switch (param.ParamType) {
				case "bool":
					journalCall = "addBooleanParameter(\"" + param.ParamName+ "\", " + variableName + ")";
				
				case "uint8":
					journalCall = "addUInt8Parameter(\"" + param.ParamName+ "\", " + variableName + ")";

				case "uint16":
					journalCall = "addUInt16Parameter(\"" + param.ParamName+ "\", " + variableName + ")";

				case "uint32":
					journalCall = "addUInt32Parameter(\"" + param.ParamName+ "\", " + variableName + ")";

				case "uint64":
					journalCall = "addUInt64Parameter(\"" + param.ParamName+ "\", " + variableName + ")";

				case "int8":
					journalCall = "addInt8Parameter(\"" + param.ParamName+ "\", " + variableName + ")";

				case "int16":
					journalCall = "addInt16Parameter(\"" + param.ParamName+ "\", " + variableName + ")";

				case "int32":
					journalCall = "addInt32Parameter(\"" + param.ParamName+ "\", " + variableName + ")";

				case "int64":
					journalCall = "addInt64Parameter(\"" + param.ParamName+ "\", " + variableName + ")";

				case "single":
					journalCall = "addSingleParameter(\"" + param.ParamName+ "\", " + variableName + ")";

				case "double":
					journalCall = "addDoubleParameter(\"" + param.ParamName+ "\", " + variableName + ")";
				
				case "pointer":
					journalCall = "addPointerParameter(\"" + param.ParamName+ "\", " + variableName + ")";

				case "string":
					journalCall = "addStringParameter(\"" + param.ParamName+ "\", p" + param.ParamName + ")";

				case "enum":
					journalCall = "addEnumParameter(\"" + param.ParamName+ "\", \"" + param.ParamClass + "\", (" + NameSpace + "_int32)(" + variableName + "))";
					
				case "class", "optionalclass":
					journalCall = "addHandleParameter(\"" + param.ParamName+ "\", " + variableName + ")";

				case "struct":
				case "basicarray":
				case "structarray":
				case "functiontype":
				
				default:
					return journalInitFunctionCode, journalSuccessFunctionCode, fmt.Errorf("invalid method parameter passing \"%s\" for %s.%s(%s)", param.ParamPass, ClassName, method.MethodName, param.ParamName)
				
			}
			if journalCall != "" {
				journalInitFunctionCode = append(journalInitFunctionCode, fmt.Sprintf("  pJournalEntry->%s;", journalCall))
			}
		}		
	}
	
	journalInitFunctionCode = append(journalInitFunctionCode, fmt.Sprintf("}"))

	journalSuccessFunctionCode = append(journalSuccessFunctionCode, fmt.Sprintf("if (pJournalEntry.get() != nullptr) {"))
	for k := 0; k < len(method.Params); k++ {
		param := method.Params[k]
		
		if (param.ParamPass == "out") || (param.ParamPass == "return") {
			journalCall := "";
			
			switch (param.ParamType) {
				case "bool":
					journalCall = "addBooleanResult(\"" + param.ParamName+ "\", *p" + param.ParamName + ")";
				
				case "uint8":
					journalCall = "addUInt8Result(\"" + param.ParamName+ "\", *p" + param.ParamName + ")";

				case "uint16":
					journalCall = "addUInt16Result(\"" + param.ParamName+ "\", *p" + param.ParamName + ")";

				case "uint32":
					journalCall = "addUInt32Result(\"" + param.ParamName+ "\", *p" + param.ParamName + ")";

				case "uint64":
					journalCall = "addUInt64Result(\"" + param.ParamName+ "\", *p" + param.ParamName + ")";

				case "int8":
					journalCall = "addInt8Result(\"" + param.ParamName+ "\", *p" + param.ParamName + ")";

				case "int16":
					journalCall = "addInt16Result(\"" + param.ParamName+ "\", *p" + param.ParamName + ")";

				case "int32":
					journalCall = "addInt32Result(\"" + param.ParamName+ "\", *p" + param.ParamName + ")";

				case "int64":
					journalCall = "addInt64Result(\"" + param.ParamName+ "\", *p" + param.ParamName + ")";

				case "single":
					journalCall = "addSingleResult(\"" + param.ParamName+ "\", *p" + param.ParamName + ")";

				case "double":
					journalCall = "addDoubleResult(\"" + param.ParamName+ "\", *p" + param.ParamName + ")";

				case "pointer":
					journalCall = "addPointerResult(\"" + param.ParamName+ "\", *p" + param.ParamName + ")";

				case "string":
					journalCall = "addStringResult(\"" + param.ParamName+ "\", s" + param.ParamName + ".c_str())";

				case "enum":
					journalCall = "addEnumResult(\"" + param.ParamName+ "\", \"" + param.ParamClass + "\", (" + NameSpace + "_int32)(*p" + param.ParamName + "))";
					
				case "class", "optionalclass":
					journalCall = "addHandleResult(\"" + param.ParamName+ "\", *p" + param.ParamName + ")";
				
				case "struct":
				case "basicarray":
				case "structarray":
				
				default:
					return journalInitFunctionCode, journalSuccessFunctionCode, fmt.Errorf("invalid method parameter passing \"%s\" for %s.%s(%s)", param.ParamPass, ClassName, method.MethodName, param.ParamName)
				
			}
			if journalCall != "" {
				journalSuccessFunctionCode = append(journalSuccessFunctionCode, fmt.Sprintf("  pJournalEntry->%s;", journalCall))
			}
		}
	}

	journalSuccessFunctionCode = append(journalSuccessFunctionCode, fmt.Sprintf("  pJournalEntry->writeSuccess();"))
	journalSuccessFunctionCode = append(journalSuccessFunctionCode, fmt.Sprintf("}"))
	
	return journalInitFunctionCode, journalSuccessFunctionCode, nil;

}

func buildCMakeForCPPImplementation(component ComponentDefinition, w LanguageWriter, doJournal bool) {
	NameSpace := component.NameSpace
	BaseName := component.BaseName
	
	w.Writeln("cmake_minimum_required(VERSION 3.5)")
	w.Writeln("")

	w.Writeln("### The implementation of the %s component", component.LibraryName)
	w.Writeln("project(%s)", NameSpace)
	w.Writeln("")
	w.Writeln("set (CMAKE_CXX_STANDARD 11)")
	w.Writeln("")
	
	w.Writeln("# The location of autogenerated interfaces")
	w.Writeln("set(CMAKE_CURRENT_AUTOGENERATED_DIR ${CMAKE_CURRENT_SOURCE_DIR}/Interfaces)",)
	w.Writeln("")

	w.Writeln("file(GLOB %s_SRC", strings.ToUpper(NameSpace))
	w.Writeln("  ${CMAKE_CURRENT_SOURCE_DIR}/Stub/*.cpp")
	w.Writeln(")")
	w.Writeln("file(GLOB %s_HDR", strings.ToUpper(NameSpace))
	w.Writeln("  ${CMAKE_CURRENT_SOURCE_DIR}/Stub/*.hpp")
	w.Writeln(")")

	w.Writeln("set(%s_SRC ${%s_SRC} ${%s_SRC}", strings.ToUpper(NameSpace), strings.ToUpper(NameSpace), strings.ToUpper(NameSpace))
	w.Writeln("  ${CMAKE_CURRENT_AUTOGENERATED_DIR}/%s_interfaceexception.cpp", strings.ToLower(BaseName))
	w.Writeln("  ${CMAKE_CURRENT_AUTOGENERATED_DIR}/%s_interfacewrapper.cpp", strings.ToLower(BaseName))
	if (doJournal) {
		w.Writeln("  ${CMAKE_CURRENT_AUTOGENERATED_DIR}/%s_interfacejournal.cpp", strings.ToLower(BaseName))
	}
	w.Writeln(")")
	w.Writeln("")

	targetName := strings.ToLower(NameSpace)
	w.Writeln("add_library(%s SHARED ${%s_SRC})", targetName, strings.ToUpper(NameSpace))
	w.Writeln("# Do not prefix the binary's name with \"lib\" on Unix systems:")
	w.Writeln("set_target_properties(%s PROPERTIES PREFIX \"\" IMPORT_PREFIX \"\" )", targetName)
	w.Writeln("# The following two properties are crucial to reduce the number of undesirably exported symbols")
	w.Writeln("set_target_properties(%s PROPERTIES CXX_VISIBILITY_PRESET hidden)", targetName)
	w.Writeln("set_target_properties(%s PROPERTIES VISIBILITY_INLINES_HIDDEN ON)", targetName)
	w.Writeln("# This makes sure symbols are exported")
	w.Writeln("target_compile_options(%s PRIVATE \"-D__%s_EXPORTS\")", targetName, strings.ToUpper(NameSpace))
	w.Writeln("target_include_directories(%s PRIVATE ${CMAKE_CURRENT_AUTOGENERATED_DIR})", targetName)
	w.Writeln("target_include_directories(%s PRIVATE ${CMAKE_CURRENT_SOURCE_DIR}/Stub)", targetName)
	for _, subComponent := range(component.ImportedComponentDefinitions) {
		w.Writeln("target_include_directories(%s PRIVATE \"${CMAKE_CURRENT_SOURCE_DIR}/../../../%s_component/Bindings/CppDynamic\")", targetName, subComponent.NameSpace)
	}
}

// buildJournalingCPP generates Declaration and Implementation of the Journaling class
func buildJournalingCPP(component ComponentDefinition, headerw LanguageWriter, implw LanguageWriter) error {
	NameSpace := component.NameSpace
	BaseName := component.BaseName

	headerw.Writeln("#ifndef __%s_INTERFACEJOURNAL_HEADER", strings.ToUpper (NameSpace));
	headerw.Writeln("#define __%s_INTERFACEJOURNAL_HEADER", strings.ToUpper (NameSpace));
	headerw.Writeln("");
	headerw.Writeln("#include <iostream>");
	headerw.Writeln("#include <fstream>");
	headerw.Writeln("#include <string>");
	headerw.Writeln("#include <memory>");
	headerw.Writeln("#include <list>");
	headerw.Writeln("#include <mutex>");
	headerw.Writeln("#include <chrono>");
	headerw.Writeln("#include \"%s_types.hpp\"", BaseName);
	headerw.Writeln("");
	headerw.Writeln("/*************************************************************************************************************************");
	headerw.Writeln(" Class C%sInterfaceJournal ", NameSpace);
	headerw.Writeln("**************************************************************************************************************************/");
	headerw.Writeln("");
	headerw.Writeln("class C%sInterfaceJournal;", NameSpace);
	headerw.Writeln("");
	headerw.Writeln("class C%sInterfaceJournalEntry {", NameSpace);
	headerw.Writeln("");
	headerw.Writeln("  protected:");
	headerw.Writeln("");
	headerw.Writeln("    C%sInterfaceJournal * m_pJournal;", NameSpace);
	headerw.Writeln("    %sResult m_ErrorCode;", NameSpace);
	headerw.Writeln("    std::string m_sClassName;");
	headerw.Writeln("    std::string m_sMethodName;");
	headerw.Writeln("    std::string m_sInstanceHandle;");
	headerw.Writeln("    %s_uint64 m_nInitTimeStamp;", NameSpace);
	headerw.Writeln("    %s_uint64 m_nFinishTimeStamp;", NameSpace);
	headerw.Writeln("    std::list<std::pair<std::pair<std::string, std::string>, std::string>> m_sParameters;");
	headerw.Writeln("    std::list<std::pair<std::pair<std::string, std::string>, std::string>> m_sResultValues;");
	headerw.Writeln("");
	headerw.Writeln("    std::string getXMLString();");
	headerw.Writeln("    void addParameter (const std::string & sName, const std::string & sParameterType, const std::string & sParameterValue);");
	headerw.Writeln("    void addResult (const std::string & sName, const std::string & sResultType, const std::string & sResultValue);");
	
	headerw.Writeln("");
	headerw.Writeln("  public:");
	headerw.Writeln("    C%sInterfaceJournalEntry(C%sInterfaceJournal * pJournal, std::string sClassName, std::string sMethodName, %sHandle pInstanceHandle);", NameSpace, NameSpace, NameSpace);
	headerw.Writeln("    ~C%sInterfaceJournalEntry();", NameSpace);
	headerw.Writeln("");
	headerw.Writeln("    void writeSuccess ();");
	headerw.Writeln("    void writeError (%sResult nErrorCode);", NameSpace);
	headerw.Writeln("");
	headerw.Writeln("    void addBooleanParameter(const std::string & sName, const bool bValue);");
	headerw.Writeln("    void addUInt8Parameter(const std::string & sName, const %s_uint8 nValue);", NameSpace);
	headerw.Writeln("    void addUInt16Parameter(const std::string & sName, const %s_uint16 nValue);", NameSpace);
	headerw.Writeln("    void addUInt32Parameter(const std::string & sName, const %s_uint32 nValue);", NameSpace);
	headerw.Writeln("    void addUInt64Parameter(const std::string & sName, const %s_uint64 nValue);", NameSpace);
	headerw.Writeln("    void addInt8Parameter(const std::string & sName, const %s_int8 nValue);", NameSpace);
	headerw.Writeln("    void addInt16Parameter(const std::string & sName, const %s_int16 nValue);", NameSpace);
	headerw.Writeln("    void addInt32Parameter(const std::string & sName, const %s_int32 nValue);", NameSpace);
	headerw.Writeln("    void addInt64Parameter(const std::string & sName, const %s_int64 nValue);", NameSpace);
	headerw.Writeln("    void addSingleParameter(const std::string & sName, const %s_single fValue);", NameSpace);
	headerw.Writeln("    void addDoubleParameter(const std::string & sName, const %s_double dValue);", NameSpace);
	headerw.Writeln("    void addPointerParameter(const std::string & sName, const %s_pvoid pValue);", NameSpace);
	headerw.Writeln("    void addStringParameter(const std::string & sName, const char * pValue);");
	headerw.Writeln("    void addHandleParameter(const std::string & sName, const %sHandle pHandle);", NameSpace);
	headerw.Writeln("    void addEnumParameter(const std::string & sName, const std::string & sEnumType, const %s_int32 nValue);", NameSpace);
	headerw.Writeln("");
	headerw.Writeln("    void addBooleanResult(const std::string & sName, const bool bValue);");
	headerw.Writeln("    void addUInt8Result(const std::string & sName, const %s_uint8 nValue);", NameSpace);
	headerw.Writeln("    void addUInt16Result(const std::string & sName, const %s_uint16 nValue);", NameSpace);
	headerw.Writeln("    void addUInt32Result(const std::string & sName, const %s_uint32 nValue);", NameSpace);
	headerw.Writeln("    void addUInt64Result(const std::string & sName, const %s_uint64 nValue);", NameSpace);
	headerw.Writeln("    void addInt8Result(const std::string & sName, const %s_int8 nValue);", NameSpace);
	headerw.Writeln("    void addInt16Result(const std::string & sName, const %s_int16 nValue);", NameSpace);
	headerw.Writeln("    void addInt32Result(const std::string & sName, const %s_int32 nValue);", NameSpace);
	headerw.Writeln("    void addInt64Result(const std::string & sName, const %s_int64 nValue);", NameSpace);
	headerw.Writeln("    void addSingleResult(const std::string & sName, const %s_single fValue);", NameSpace);
	headerw.Writeln("    void addDoubleResult(const std::string & sName, const %s_double dValue);", NameSpace);
	headerw.Writeln("    void addPointerResult(const std::string & sName, const %s_pvoid pValue);", NameSpace);
	headerw.Writeln("    void addStringResult(const std::string & sName, const char * pValue);");
	headerw.Writeln("    void addHandleResult(const std::string & sName, const %sHandle pHandle);", NameSpace);
	headerw.Writeln("    void addEnumResult(const std::string & sName, const std::string & sEnumType, const %s_int32 nValue);", NameSpace);
	headerw.Writeln("");
	headerw.Writeln("friend class C%sInterfaceJournal;", NameSpace);
	headerw.Writeln("");
	headerw.Writeln("};");
	headerw.Writeln("");
	headerw.Writeln("typedef std::shared_ptr<C%sInterfaceJournalEntry> P%sInterfaceJournalEntry;", NameSpace, NameSpace);
	headerw.Writeln("");
	headerw.Writeln("");
	headerw.Writeln("");
	

	headerw.Writeln("class C%sInterfaceJournal {", NameSpace);
	headerw.Writeln("");
	headerw.Writeln("  protected:");
	headerw.Writeln("");
	headerw.Writeln("    std::string m_sFileName;");
	headerw.Writeln("    std::mutex m_Mutex;");
	headerw.Writeln("    std::ofstream m_Stream;");
	headerw.Writeln("    std::chrono::time_point<std::chrono::high_resolution_clock> m_StartTime;");
	headerw.Writeln("    void writeEntry (C%sInterfaceJournalEntry * pEntry);", NameSpace);
	headerw.Writeln("    %s_uint64 getTimeStamp ();", NameSpace);
	headerw.Writeln("");
	headerw.Writeln("  public:");
	headerw.Writeln("");
	headerw.Writeln("    C%sInterfaceJournal (const std::string & sFileName);", NameSpace);
	headerw.Writeln("    ~C%sInterfaceJournal ();", NameSpace);
	headerw.Writeln("    P%sInterfaceJournalEntry beginClassMethod (const %sHandle pHandle, const std::string & sClassName, const std::string & sMethodName);", NameSpace, NameSpace);
	headerw.Writeln("    P%sInterfaceJournalEntry beginStaticFunction (const std::string & sMethodName);", NameSpace);
	headerw.Writeln("    friend class C%sInterfaceJournalEntry;", NameSpace);
	headerw.Writeln("};");
	headerw.Writeln("");
	headerw.Writeln("typedef std::shared_ptr<C%sInterfaceJournal> P%sInterfaceJournal;", NameSpace, NameSpace);
	headerw.Writeln("");
	headerw.Writeln("#endif // __%s_INTERFACEJOURNAL_HEADER", strings.ToUpper (NameSpace));
	headerw.Writeln("");


	implw.Writeln("");
	implw.Writeln("#include <string>");
	implw.Writeln("#include <sstream>");
	implw.Writeln("#include <iomanip>");
	implw.Writeln("");
	implw.Writeln("#include \"%s_interfacejournal.hpp\"", strings.ToLower(BaseName));
	implw.Writeln("#include \"%s_interfaceexception.hpp\"", strings.ToLower(BaseName));
	implw.Writeln("");
	implw.Writeln("");
	
	implw.Writeln("std::string %sHandleToHex (%sHandle pHandle)", NameSpace, NameSpace);
	implw.Writeln("{");
	implw.Writeln("  std::stringstream stream;");
	implw.Writeln("  stream << std::setfill('0') << std::setw(sizeof(%s_uint64) * 2)", NameSpace);
	implw.Writeln("    << std::hex << (%s_uint64) pHandle;", NameSpace);
	implw.Writeln("  return stream.str();");
	implw.Writeln("}");
	implw.Writeln("");
	
	implw.Writeln("C%sInterfaceJournalEntry::C%sInterfaceJournalEntry(C%sInterfaceJournal * pJournal, std::string sClassName, std::string sMethodName, %sHandle pInstanceHandle)", NameSpace, NameSpace, NameSpace, NameSpace);
	implw.Writeln("  : m_pJournal(pJournal), m_ErrorCode(%s_SUCCESS), m_sClassName(sClassName), m_sMethodName(sMethodName), m_nInitTimeStamp(0), m_nFinishTimeStamp(0)", strings.ToUpper(NameSpace))
	implw.Writeln("{");
	implw.Writeln("  if (pJournal == nullptr)");
	implw.Writeln("    throw E%sInterfaceException(%s_ERROR_INVALIDPARAM);", NameSpace, strings.ToUpper (NameSpace));
	implw.Writeln("  m_nInitTimeStamp = m_pJournal->getTimeStamp ();");
	implw.Writeln("  m_sInstanceHandle = %sHandleToHex (pInstanceHandle);", NameSpace);
	implw.Writeln("}");
	implw.Writeln("");
	implw.Writeln("C%sInterfaceJournalEntry::~C%sInterfaceJournalEntry()", NameSpace, NameSpace);
	implw.Writeln("{");
	implw.Writeln("}");
	implw.Writeln("");
	implw.Writeln("void C%sInterfaceJournalEntry::addParameter(const std::string & sName, const std::string & sParameterType, const std::string & sParameterValue)", NameSpace);
	implw.Writeln("{");
	implw.Writeln("  m_sParameters.push_back(std::make_pair(std::make_pair(sName, sParameterType), sParameterValue));");
	implw.Writeln("}");
	implw.Writeln("");
	implw.Writeln("void C%sInterfaceJournalEntry::addResult(const std::string & sName, const std::string & sResultType, const std::string & sResultValue)", NameSpace);
	implw.Writeln("{");
	implw.Writeln("  m_sResultValues.push_back(std::make_pair(std::make_pair(sName, sResultType), sResultValue));");
	implw.Writeln("}");
	implw.Writeln("");
	implw.Writeln("std::string C%sInterfaceJournalEntry::getXMLString()", NameSpace);
	implw.Writeln("{");
	implw.Writeln("  std::stringstream sStream;");
	implw.Writeln("  %s_uint64 nDuration = 0;", NameSpace);
	implw.Writeln("");
	implw.Writeln("  if (m_nFinishTimeStamp > m_nInitTimeStamp)");
	implw.Writeln("    nDuration = m_nFinishTimeStamp - m_nInitTimeStamp;");
	implw.Writeln("");
	implw.Writeln("  sStream << \"    <entry\";");
	implw.Writeln("  if (m_sClassName != \"\")");
	implw.Writeln("    sStream << \" class=\\\"\" << m_sClassName << \"\\\"\";");
	implw.Writeln("  sStream << \" method=\\\"\" << m_sMethodName << \"\\\"\";");
	implw.Writeln("  if (m_ErrorCode != %s_SUCCESS)", strings.ToUpper (NameSpace));
	implw.Writeln("    sStream << \" errorcode=\\\"\" << m_ErrorCode << \"\\\"\";");
	implw.Writeln("  sStream << \" timestamp=\\\"\" << m_nInitTimeStamp << \"\\\" duration=\\\"\" << nDuration << \"\\\">\\n\";");
	implw.Writeln("");
	implw.Writeln("  if (m_sClassName != \"\")");
	implw.Writeln("    sStream << \"        <instance handle=\\\"\" << m_sInstanceHandle << \"\\\" />\\n\";");
	implw.Writeln("");
	implw.Writeln("  auto iParamIter = m_sParameters.begin();");
	implw.Writeln("  while (iParamIter != m_sParameters.end()) {");
	implw.Writeln("    sStream << \"        <parameter name=\\\"\" << iParamIter->first.first << \"\\\" type=\\\"\" << iParamIter->first.second << \"\\\" value=\\\"\" << iParamIter->second <<\"\\\" />\\n\";");
	implw.Writeln("    iParamIter++;");
	implw.Writeln("  }");
	implw.Writeln("");
	implw.Writeln("  auto iResultIter = m_sResultValues.begin();",);
	implw.Writeln("  while (iResultIter != m_sResultValues.end()) {");
	implw.Writeln("    sStream << \"        <result name=\\\"\" << iResultIter->first.first << \"\\\" type=\\\"\" << iResultIter->first.second << \"\\\" value=\\\"\" << iResultIter->second << \"\\\" />\\n\";");
	implw.Writeln("    iResultIter++;");
	implw.Writeln("  }");
	implw.Writeln("");
	implw.Writeln("  sStream << \"    </entry>\\n\";");
	implw.Writeln("  return sStream.str ();");
	implw.Writeln("}");
	implw.Writeln("");
	implw.Writeln("void C%sInterfaceJournalEntry::writeSuccess()", NameSpace);
	implw.Writeln("{");
	implw.Writeln("  writeError(%s_SUCCESS);", strings.ToUpper (NameSpace));
	implw.Writeln("}");
	implw.Writeln("");
	implw.Writeln("void C%sInterfaceJournalEntry::writeError(%sResult nErrorCode)", NameSpace, NameSpace);
	implw.Writeln("{");
	implw.Writeln("  m_ErrorCode = nErrorCode;");
	implw.Writeln("  m_nFinishTimeStamp = m_pJournal->getTimeStamp();");
	implw.Writeln("  m_pJournal->writeEntry(this);");
	implw.Writeln("}");
	implw.Writeln("");
	implw.Writeln("void C%sInterfaceJournalEntry::addBooleanParameter(const std::string & sName, const bool bValue)", NameSpace);
	implw.Writeln("{");
	implw.Writeln("  addParameter (sName, \"bool\", std::to_string((int)bValue));");
	implw.Writeln("}");
	implw.Writeln("");
	implw.Writeln("void C%sInterfaceJournalEntry::addUInt8Parameter(const std::string & sName, const %s_uint8 nValue)", NameSpace, NameSpace);
	implw.Writeln("{");
	implw.Writeln("  addParameter(sName, \"uint8\", std::to_string(nValue));");
	implw.Writeln("}");
	implw.Writeln("");
	implw.Writeln("void C%sInterfaceJournalEntry::addUInt16Parameter(const std::string & sName, const %s_uint16 nValue)", NameSpace, NameSpace);
	implw.Writeln("{");
	implw.Writeln("  addParameter(sName, \"uint16\", std::to_string(nValue));");
	implw.Writeln("}");
	implw.Writeln("");
	implw.Writeln("void C%sInterfaceJournalEntry::addUInt32Parameter(const std::string & sName, const %s_uint32 nValue)", NameSpace, NameSpace);
	implw.Writeln("{");
	implw.Writeln("  addParameter(sName, \"uint32\", std::to_string(nValue));");
	implw.Writeln("}");
	implw.Writeln("");
	implw.Writeln("void C%sInterfaceJournalEntry::addUInt64Parameter(const std::string & sName, const %s_uint64 nValue)", NameSpace, NameSpace);
	implw.Writeln("{");
	implw.Writeln("  addParameter(sName, \"uint64\", std::to_string(nValue));");
	implw.Writeln("}");
	implw.Writeln("");
	implw.Writeln("void C%sInterfaceJournalEntry::addInt8Parameter(const std::string & sName, const %s_int8 nValue)", NameSpace, NameSpace);
	implw.Writeln("{");
	implw.Writeln("  addParameter(sName, \"int8\", std::to_string(nValue));");
	implw.Writeln("}");
	implw.Writeln("");
	implw.Writeln("void C%sInterfaceJournalEntry::addInt16Parameter(const std::string & sName, const %s_int16 nValue)", NameSpace, NameSpace);
	implw.Writeln("{");
	implw.Writeln("  addParameter(sName, \"int16\", std::to_string(nValue));");
	implw.Writeln("}");
	implw.Writeln("");
	implw.Writeln("void C%sInterfaceJournalEntry::addInt32Parameter(const std::string & sName, const %s_int32 nValue)", NameSpace, NameSpace);
	implw.Writeln("{");
	implw.Writeln("  addParameter(sName, \"uint32\", std::to_string(nValue));");
	implw.Writeln("}");
	implw.Writeln("");
	implw.Writeln("void C%sInterfaceJournalEntry::addInt64Parameter(const std::string & sName, const %s_int64 nValue)", NameSpace, NameSpace);
	implw.Writeln("{");
	implw.Writeln("  addParameter(sName, \"int64\", std::to_string(nValue));");
	implw.Writeln("}");
	implw.Writeln("");
	implw.Writeln("void C%sInterfaceJournalEntry::addSingleParameter(const std::string & sName,  const %s_single fValue)", NameSpace, NameSpace);
	implw.Writeln("{");
	implw.Writeln("  addParameter(sName, \"single\", std::to_string(fValue));");
	implw.Writeln("}");
	implw.Writeln("");
	implw.Writeln("void C%sInterfaceJournalEntry::addDoubleParameter(const std::string & sName, const %s_double dValue)", NameSpace, NameSpace);
	implw.Writeln("{");
	implw.Writeln("  addParameter(sName, \"double\", std::to_string(dValue));");
	implw.Writeln("}");
	implw.Writeln("void C%sInterfaceJournalEntry::addPointerParameter(const std::string & sName, const %s_pvoid pValue)", NameSpace, NameSpace);
	implw.Writeln("{");
	implw.Writeln("  addParameter(sName, \"pointer\", std::to_string(reinterpret_cast<const %s_uint64>(pValue)));", NameSpace);
	implw.Writeln("}");
	implw.Writeln("");
	implw.Writeln("void C%sInterfaceJournalEntry::addStringParameter(const std::string & sName, const char * pValue)", NameSpace);
	implw.Writeln("{");
	implw.Writeln("  if (pValue != nullptr) {");
	implw.Writeln("    addParameter(sName, \"string\", pValue);");
	implw.Writeln("  }");
	implw.Writeln("  else {");
	implw.Writeln("    addParameter(sName, \"nullstring\", \"\");");
	implw.Writeln("  }");
	implw.Writeln("}");
	implw.Writeln("");
	implw.Writeln("void C%sInterfaceJournalEntry::addHandleParameter(const std::string & sName, const %sHandle pHandle)", NameSpace, NameSpace);
	implw.Writeln("{");
	implw.Writeln("  addParameter(sName, \"handle\", %sHandleToHex(pHandle));", NameSpace);
	implw.Writeln("}");
	implw.Writeln("");
	implw.Writeln("void C%sInterfaceJournalEntry::addEnumParameter(const std::string & sName, const std::string & sEnumType, const %s_int32 nValue)", NameSpace, NameSpace);
	implw.Writeln("{");
	implw.Writeln("  addParameter(sName, \"enum\" + sEnumType, std::to_string(nValue));");
	implw.Writeln("}");
	implw.Writeln("");
	implw.Writeln("void C%sInterfaceJournalEntry::addBooleanResult(const std::string & sName, const bool bValue)", NameSpace);
	implw.Writeln("{");
	implw.Writeln("  addResult(sName, \"bool\", std::to_string((int)bValue));");
	implw.Writeln("}");
	implw.Writeln("");
	implw.Writeln("void C%sInterfaceJournalEntry::addUInt8Result(const std::string & sName, const %s_uint8 nValue)", NameSpace, NameSpace);
	implw.Writeln("{");
	implw.Writeln("  addResult(sName, \"uint8\", std::to_string(nValue));");
	implw.Writeln("}");
	implw.Writeln("");
	implw.Writeln("void C%sInterfaceJournalEntry::addUInt16Result(const std::string & sName, const %s_uint16 nValue)", NameSpace, NameSpace);
	implw.Writeln("{");
	implw.Writeln("  addResult(sName, \"uint16\", std::to_string(nValue));");
	implw.Writeln("}");
	implw.Writeln("");
	implw.Writeln("void C%sInterfaceJournalEntry::addUInt32Result(const std::string & sName, const %s_uint32 nValue)", NameSpace, NameSpace);
	implw.Writeln("{");
	implw.Writeln("  addResult(sName, \"uint32\", std::to_string(nValue));");
	implw.Writeln("}");
	implw.Writeln("");
	implw.Writeln("void C%sInterfaceJournalEntry::addUInt64Result(const std::string & sName, const %s_uint64 nValue)", NameSpace, NameSpace);
	implw.Writeln("{");
	implw.Writeln("  addResult(sName, \"uint64\", std::to_string(nValue));");
	implw.Writeln("}");
	implw.Writeln("");
	implw.Writeln("void C%sInterfaceJournalEntry::addInt8Result(const std::string & sName, const %s_int8 nValue)", NameSpace, NameSpace);
	implw.Writeln("{");
	implw.Writeln("  addResult(sName, \"int8\", std::to_string(nValue));");
	implw.Writeln("}");
	implw.Writeln("");
	implw.Writeln("void C%sInterfaceJournalEntry::addInt16Result(const std::string & sName, const %s_int16 nValue)", NameSpace, NameSpace);
	implw.Writeln("{");
	implw.Writeln("  addResult(sName, \"int16\", std::to_string(nValue));");
	implw.Writeln("}");
	implw.Writeln("");
	implw.Writeln("void C%sInterfaceJournalEntry::addInt32Result(const std::string & sName, const %s_int32 nValue)", NameSpace, NameSpace);
	implw.Writeln("{");
	implw.Writeln("  addResult(sName, \"uint32\", std::to_string(nValue));");
	implw.Writeln("}");
	implw.Writeln("");
	implw.Writeln("void C%sInterfaceJournalEntry::addInt64Result(const std::string & sName, const %s_int64 nValue)", NameSpace, NameSpace);
	implw.Writeln("{");
	implw.Writeln("  addResult(sName, \"int64\", std::to_string(nValue));");
	implw.Writeln("}");
	implw.Writeln("");
	implw.Writeln("void C%sInterfaceJournalEntry::addSingleResult(const std::string & sName,  const %s_single fValue)", NameSpace, NameSpace);
	implw.Writeln("{");
	implw.Writeln("  addResult(sName, \"single\", std::to_string(fValue));");
	implw.Writeln("}");
	implw.Writeln("");
	implw.Writeln("void C%sInterfaceJournalEntry::addDoubleResult(const std::string & sName, const %s_double dValue)", NameSpace, NameSpace);
	implw.Writeln("{");
	implw.Writeln("  addResult(sName, \"double\", std::to_string(dValue));");
	implw.Writeln("}");
	implw.Writeln("");
	implw.Writeln("void C%sInterfaceJournalEntry::addPointerResult(const std::string & sName, const %s_pvoid pValue)", NameSpace, NameSpace);
	implw.Writeln("{");
	implw.Writeln("  addResult(sName, \"pointer\", std::to_string(reinterpret_cast<const %s_uint64>(pValue)));", NameSpace);
	implw.Writeln("}");
	implw.Writeln("");
	implw.Writeln("void C%sInterfaceJournalEntry::addStringResult(const std::string & sName, const char * pValue)", NameSpace);
	implw.Writeln("{");
	implw.Writeln("  if (pValue != nullptr) {");
	implw.Writeln("    addResult(sName, \"string\", pValue);");
	implw.Writeln("  }");
	implw.Writeln("  else {");
	implw.Writeln("    addResult(sName, \"nullstring\", \"\");");
	implw.Writeln("  }");
	implw.Writeln("}");
	implw.Writeln("");
	implw.Writeln("void C%sInterfaceJournalEntry::addHandleResult(const std::string & sName, const %sHandle pHandle)", NameSpace, NameSpace);
	implw.Writeln("{");
	implw.Writeln("  addResult(sName, \"handle\", %sHandleToHex(pHandle));", NameSpace);
	implw.Writeln("}");
	implw.Writeln("");
	implw.Writeln("void C%sInterfaceJournalEntry::addEnumResult(const std::string & sName, const std::string & sEnumType, const %s_int32 nValue)", NameSpace, NameSpace);
	implw.Writeln("{");
	implw.Writeln("  addResult(sName, \"enum\" + sEnumType, std::to_string(nValue));");
	implw.Writeln("}");
	implw.Writeln("");
	implw.Writeln("");
	implw.Writeln("C%sInterfaceJournal::C%sInterfaceJournal (const std::string & sFileName)", NameSpace, NameSpace);
	implw.Writeln("  : m_sFileName (sFileName)");
	implw.Writeln("{");
	implw.Writeln("  m_StartTime = std::chrono::high_resolution_clock::now();");
	implw.Writeln("  m_Stream.open (sFileName, std::ios::out);");
	implw.Writeln("  m_Stream << \"<?xml version=\\\"1.0\\\" encoding=\\\"UTF-8\\\" ?>\\n\";");
	implw.Writeln("  m_Stream << \"<journal library=\\\"%s\\\" version=\\\"%s\\\" xmlns=\\\"http://schemas.autodesk.com/components/%s/%s\\\">\\n\";",
		NameSpace, component.Version, NameSpace, component.Version);
	implw.Writeln("  m_Stream << \"\\n\";");
	implw.Writeln("");
	
	
	implw.Writeln("}");
	implw.Writeln("");
	implw.Writeln("C%sInterfaceJournal::~C%sInterfaceJournal ()", NameSpace, NameSpace);
	implw.Writeln("{");
	implw.Writeln("  m_Stream << \"</journal>\\n\";");
	implw.Writeln("}");
	implw.Writeln("");
	implw.Writeln("P%sInterfaceJournalEntry C%sInterfaceJournal::beginClassMethod(const %sHandle pHandle, const std::string & sClassName, const std::string & sMethodName)", NameSpace, NameSpace, NameSpace);
	implw.Writeln("{");
	implw.Writeln("  return std::make_shared<C%sInterfaceJournalEntry>(this, sClassName, sMethodName, pHandle);", NameSpace);
	implw.Writeln("}");
	implw.Writeln("");
	implw.Writeln("P%sInterfaceJournalEntry C%sInterfaceJournal::beginStaticFunction(const std::string & sMethodName)", NameSpace, NameSpace);
	implw.Writeln("{");
	implw.Writeln("  return std::make_shared<C%sInterfaceJournalEntry>(this, \"\", sMethodName, nullptr);", NameSpace);
	implw.Writeln("}");
	implw.Writeln("");
	implw.Writeln("void C%sInterfaceJournal::writeEntry (C%sInterfaceJournalEntry * pEntry)", NameSpace, NameSpace);
	implw.Writeln("{");
	implw.Writeln("  if (pEntry == nullptr)");
	implw.Writeln("    throw E%sInterfaceException (%s_ERROR_INVALIDPARAM);", NameSpace, strings.ToUpper (NameSpace));
	implw.Writeln("");
	implw.Writeln("  std::string sXMLString = pEntry->getXMLString();");
	implw.Writeln("  m_Mutex.lock();");
	implw.Writeln("  try {");
	implw.Writeln("    m_Stream << sXMLString;");
	implw.Writeln("    m_Stream << \"\\n\";");
	implw.Writeln("");
	implw.Writeln("    m_Mutex.unlock();");
	implw.Writeln("  }");
	implw.Writeln("  catch (...) {");
	implw.Writeln("    m_Mutex.unlock();");
	implw.Writeln("  }");
	implw.Writeln("}");
	implw.Writeln("");
	implw.Writeln("%s_uint64 C%sInterfaceJournal::getTimeStamp ()", NameSpace, NameSpace);
	implw.Writeln("{");
	implw.Writeln("  auto currentTime = std::chrono::high_resolution_clock::now();");
	implw.Writeln("  if (m_StartTime < currentTime) {");
	implw.Writeln("    auto duration = currentTime - m_StartTime;");	
	implw.Writeln("    auto milliSeconds = std::chrono::duration_cast<std::chrono::milliseconds> (duration);");
	implw.Writeln("");
	implw.Writeln("    return (%s_uint64) milliSeconds.count();", NameSpace);
	implw.Writeln("  }");
	implw.Writeln("  else {");
	implw.Writeln("    return 0;");
	implw.Writeln("  }");
	implw.Writeln("");
	implw.Writeln("}");
	implw.Writeln("");
	
	return nil;
}

