/*
 * Endpoint - Linux SBP2 Disk Target
 *
 * Copyright (C) 2003 Oracle.  All rights reserved.
 *
 * Author: Manish Singh <manish.singh@oracle.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have recieved a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 021110-1307, USA.
 */

#include <glib.h>

#include <libraw1394/raw1394.h>

#include "sbp2configrom.h"
#include "sbp2csr.h"


#define ROM_BUF_SIZE 256


static guint16 crc16 (guint32 *buffer,
		      gint     length);


gboolean
sbp2_setup_config_rom (raw1394handle_t handle)
{
  quadlet_t rom[ROM_BUF_SIZE];
  gsize     rom_size;
  guchar    rom_version;
  gint      ret, i;

  g_return_val_if_fail (handle != NULL, FALSE);

  ret = raw1394_get_config_rom (handle, rom, ROM_BUF_SIZE,
                                &rom_size, &rom_version);

  if (ret != 0)
    return FALSE;

  if (rom[9] == GUINT32_FROM_BE(0xd1000008))
    return TRUE;

  for (i = 15; i > 8; i--)
    rom[i + 1] = rom[i];

  rom[7] = GUINT32_FROM_BE(0x81000003);
  rom[9] = GUINT32_FROM_BE(0xd1000008);

  rom[5] = GUINT32_FROM_BE(4 << 16 | crc16 (&rom[6], 4));

  rom[18] = GUINT32_FROM_BE(0x1200609e);
  rom[19] = GUINT32_FROM_BE(0x13010483);
  rom[20] = GUINT32_FROM_BE(0x3800609e); /* IEEE/RAC */
  rom[21] = GUINT32_FROM_BE(0x390104d8); /* IEEE/RAC */
  rom[22] = GUINT32_FROM_BE(0x3b000000); /* IEEE/RAC */
  rom[23] = GUINT32_FROM_BE(0x3c000001); /* IEEE/RAC */
  rom[24] = GUINT32_FROM_BE(0x54000000 + SBP2_CSR_MANANGEMENT_OFFSET);
  rom[25] = GUINT32_FROM_BE(0x3a003c08); /* unit characteristics */
  rom[26] = GUINT32_FROM_BE(0x3d000003); /* reconnect timeout */
  rom[27] = GUINT32_FROM_BE(0x144e0000); /* logical unit number */

  rom[17] = GUINT32_FROM_BE(10 << 16 | crc16 (&rom[18], 10));

  rom_size += 12 * 4;

  ret = raw1394_update_config_rom (handle, rom, rom_size, rom_version);

  if (ret == -1)
    return FALSE;

  return TRUE;
}

static guint16
crc16 (guint32 *buffer,
       gint     length)
{
  gint    shift;
  guint32 crc, sum, data;

  g_return_val_if_fail (buffer != NULL, 0);

  crc = 0;

  for (; length > 0; length--)
    {
      data = GUINT32_FROM_BE (*buffer);

      for (shift = 28; shift >= 0; shift -= 4)
	{
	  sum = ((crc >> 12) ^ (data >> shift)) & 0x000f;
	  crc = (crc << 4) ^ (sum << 12) ^ (sum << 5) ^ sum;
	}

      crc &= 0xffff;

      buffer++;
    }

  return crc;
}

