;;; latin-pre.el -- Quail packages for inputting various European characters.

;; Copyright (C) 1997 Electrotechnical Laboratory, JAPAN.
;; Licensed to the Free Software Foundation.

;; Keywords: mule, multilingual, latin, input method

;; This file is part of GNU Emacs.

;; GNU Emacs is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; GNU Emacs is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs; see the file COPYING.  If not, write to the
;; Free Software Foundation, Inc., 59 Temple Place - Suite 330,
;; Boston, MA 02111-1307, USA.

;;; Commentary:

;; All key translation maps are copied from iso-acc.el.

;;; Code:

(require 'quail)

;; If things were nice and serial in 21.4 we could use literals for
;; latin-iso8859-{14,15,16} after this require, as GNU does. But they're
;; not, so we call make-char below, to prevent an invalid read syntax error
;; on byte-compile.

(require 'latin-euro-standards)

(quail-define-package
 "latin-1-prefix" "Latin-1" "1>" t
 "Latin-1 characters input method with prefix modifiers

    effect   | prefix | examples
 ------------+--------+----------
    acute    |   '    | 'a -> ,Aa(B, '' -> ,A4(B
    grave    |   `    | `a -> ,A`(B
  circumflex |   ^    | ^a -> ,Ab(B
  diaeresis  |   \"    | \"a -> ,Ad(B, \"\" -> ,A((B
    tilde    |   ~    | ~a -> ,Ac(B
   cedilla   |   ~    | ~c -> ,Ag(B
    misc     |  ----  | \"s -> ,A_(B, ~d -> ,Ap(B, ~t -> ,A~(B, /a -> ,Ae(B, /e -> ,Af(B, /o -> ,Ax(B
   symbol    |   ~    | ~> -> ,A;(B, ~< -> ,A+(B, ~! -> ,A!(B, ~? -> ,A?(B, ~~ -> ,A8(B
" nil t nil nil nil nil nil nil nil nil t)

(quail-define-rules
 ("'A" ?,AA(B)
 ("'E" ?,AI(B)
 ("'I" ?,AM(B)
 ("'O" ?,AS(B)
 ("'U" ?,AZ(B)
 ("'Y" ?,A](B)
 ("'a" ?,Aa(B)
 ("'e" ?,Ai(B)
 ("'i" ?,Am(B)
 ("'o" ?,As(B)
 ("'u" ?,Az(B)
 ("'y" ?,A}(B)
 ("''" ?,A4(B)
 ("' " ?')
 ("`A" ?,A@(B)
 ("`E" ?,AH(B)
 ("`I" ?,AL(B)
 ("`O" ?,AR(B)
 ("`U" ?,AY(B)
 ("`a" ?,A`(B)
 ("`e" ?,Ah(B)
 ("`i" ?,Al(B)
 ("`o" ?,Ar(B)
 ("`u" ?,Ay(B)
 ("``" ?`)
 ("` " ?`)
 ("^A" ?,AB(B)
 ("^E" ?,AJ(B)
 ("^I" ?,AN(B)
 ("^O" ?,AT(B)
 ("^U" ?,A[(B)
 ("^a" ?,Ab(B)
 ("^e" ?,Aj(B)
 ("^i" ?,An(B)
 ("^o" ?,At(B)
 ("^u" ?,A{(B)
 ("^^" ?^)
 ("^ " ?^)
 ("\"A" ?,AD(B)
 ("\"E" ?,AK(B)
 ("\"I" ?,AO(B)
 ("\"O" ?,AV(B)
 ("\"U" ?,A\(B)
 ("\"a" ?,Ad(B)
 ("\"e" ?,Ak(B)
 ("\"i" ?,Ao(B)
 ("\"o" ?,Av(B)
 ("\"s" ?,A_(B)
 ("\"u" ?,A|(B)
 ("\"y" ?,A(B)
 ("\"\"" ?,A((B)
 ("\" " ?\")
 ("~A" ?,AC(B)
 ("~C" ?,AG(B)
 ("~D" ?,AP(B)
 ("~N" ?,AQ(B)
 ("~O" ?,AU(B)
 ("~T" ?,A^(B)
 ("~a" ?,Ac(B)
 ("~c" ?,Ag(B)
 ("~d" ?,Ap(B)
 ("~n" ?,Aq(B)
 ("~o" ?,Au(B)
 ("~t" ?,A~(B)
 ("~>" ?\,A;(B)
 ("~<" ?\,A+(B)
 ("~!" ?,A!(B)
 ("~?" ?,A?(B)
 ("~~" ?,A8(B)
 ("~ " ?~)
 ("/A" ?,AE(B)
 ("/E" ?,AF(B)
 ("/O" ?,AX(B)
 ("/a" ?,Ae(B)
 ("/e" ?,Af(B)
 ("/o" ?,Ax(B)
 ("//" ?,A0(B)
 ("/ " ?/)
)

(quail-define-package
 "catalan-prefix" "Latin-1" "CA>" t
 "Catalan and Spanish input method with prefix modifiers

    effect   | prefix | examples
 ------------+--------+----------
    acute    |   '    | 'a -> ,Aa(B, '' -> ,A4(B
    grave    |   `    | `a -> ,A`(B
  diaeresis  |   \"    | \"i -> ,Ao(B, \"\" -> ,A((B
    tilde    |   ~    | ~n -> ,Aq(B
   cedilla   |   ~    | ~c -> ,Ag(B
   symbol    |   ~    | ~> -> ,A;(B, ~< -> ,A+(B, ~! -> ,A!(B, ~? -> ,A?(B
" nil t nil nil nil nil nil nil nil nil t)

(quail-define-rules
 ("'A" ?,AA(B)
 ("'E" ?,AI(B)
 ("'I" ?,AM(B)
 ("'O" ?,AS(B)
 ("'U" ?,AZ(B)
 ("'a" ?,Aa(B)
 ("'e" ?,Ai(B)
 ("'i" ?,Am(B)
 ("'o" ?,As(B)
 ("'u" ?,Az(B)
 ("' " ?')
 ("`A" ?,A@(B)
 ("`E" ?,AH(B)
 ("`O" ?,AR(B)
 ("`a" ?,A`(B)
 ("`e" ?,Ah(B)
 ("`o" ?,Ar(B)
 ("` " ?`)
 ("\"I" ?,AO(B)
 ("\"U" ?,A\(B)
 ("\"i" ?,Ao(B)
 ("\"u" ?,A|(B)
 ("\" " ?\")
 ("~C" ?,AG(B)
 ("~N" ?,AQ(B)
 ("~c" ?,Ag(B)
 ("~n" ?,Aq(B)
 ("~>" ?\,A;(B)
 ("~<" ?\,A+(B)
 ("~!" ?,A!(B)
 ("~?" ?,A?(B)
 ("~ " ?~)
)

(quail-define-package
 "esperanto-prefix" "Latin-1" "EO>" t
 "Esperanto input method with prefix modifiers
Key translation rules are:
 ^H -> ?,A&(B, ^J -> ?,A,(B, ^h -> ?,A6(B, ^j -> ?,A<(B, ^C -> ?,AF(B, ^G -> ?,AX(B,
 ^S -> ?,A^(B, ^c -> ?,Af(B, ^g -> ?,Ax(B, ^s -> ?,A~(B, ~U -> ?,A](B, ~u -> ?,A}(B
" nil t nil nil nil nil nil nil nil nil t)

(quail-define-rules
 ("^H" ?,A&(B)
 ("^J" ?,A,(B)
 ("^h" ?,A6(B)
 ("^j" ?,A<(B)
 ("^C" ?,AF(B)
 ("^G" ?,AX(B)
 ("^S" ?,A^(B)
 ("^c" ?,Af(B)
 ("^g" ?,Ax(B)
 ("^s" ?,A~(B)
 ("^^" ?^)
 ("^ " ?^)
 ("~U" ?,A](B)
 ("~u" ?,A}(B)
 ("~ " ?~)
)

(quail-define-package
 "french-prefix" "Latin-1" "FR>" t
 "French (Fran,Ag(Bais) input method with prefix modifiers

Caract,Ah(Bres usuels:

              | pr,Ai(Bfixe | examples
 -------------+---------+----------
  aigu        |   '     | 'E -> ,AI(B
  grave       |   `     | `a -> ,A`(B
  circonflexe |   ^     | ^o -> ,At(B
  tr,Ai(Bma       |   \"     | \"i -> ,Ao(B
  c,Ai(Bdille     |   '     | 'c -> ,Ag(B


Caract,Ah(Bres sp,Ai(Bciaux:

<< et >> donnent ,A+(B et ,A;(B
/o donne ,A0(B
/c donne ,A)(B
/r donne ,A.(B

En tapant un espace apr,Ah(Bs le pr,Ai(Bfixe, celui-ci s'isole de la lettre.
Par exemple: 'e -> ,Ai(B mais ' e -> 'e , /o -> ,A0(B mais / o -> /o

Le ,A+(Be dans l'o,A;(B n'est pas disponible."
 nil t nil nil nil nil nil nil nil nil t)

(quail-define-rules
 ("'E" ?,AI(B)
 ("'e" ?,Ai(B)
 ("'C" ?,AG(B)
 ("'c" ?,Ag(B)
 ("' " ?')
 ("`A" ?,A@(B)
 ("`E" ?,AH(B)
 ("`U" ?,AY(B)
 ("`a" ?,A`(B)
 ("`e" ?,Ah(B)
 ("`u" ?,Ay(B)
 ("` " ?`)
 ("^A" ?,AB(B)
 ("^E" ?,AJ(B)
 ("^I" ?,AN(B)
 ("^O" ?,AT(B)
 ("^U" ?,A[(B)
 ("^a" ?,Ab(B)
 ("^e" ?,Aj(B)
 ("^i" ?,An(B)
 ("^o" ?,At(B)
 ("^u" ?,A{(B)
 ("^ " ?^)
 ("\"E" ?,AK(B)
 ("\"I" ?,AO(B)
 ("\"e" ?,Ak(B)
 ("\"i" ?,Ao(B)
 ("\" " ?\")
 ("<<" ?\,A+(B)
 ("< " ?<)
 (">>" ?\,A;(B)
 ("> " ?>)
 ("/o" ?,A0(B)
 ("/c" ?,A)(B)
 ("/r" ?,A.(B)
 ("/ " ?/)
 )

(quail-define-package
 "german-prefix" "Latin-1" "DE>" t
 "German (Deutsch) input method with prefix modifiers
Key translation rules are:
 \"A -> ,AD(B -> \"O -> ,AV(B, \"U -> ,A\(B, \"s -> ?,A_(B
" nil t nil nil nil nil nil nil nil nil t)

(quail-define-rules
 ("\"A" ?,AD(B)
 ("\"O" ?,AV(B)
 ("\"U" ?,A\(B)
 ("\"a" ?,Ad(B)
 ("\"o" ?,Av(B)
 ("\"u" ?,A|(B)
 ("\"s" ?,A_(B)
 ("\" " ?\")
)

(quail-define-package
 "irish-prefix" "Latin-1" "GA>" t
 "Irish input method with prefix modifiers
Key translation rules are:
 'A -> ,AA(B, 'E -> ,AI(B, 'I -> ,AM(B, 'O -> ,AS(B, 'U -> ,AZ(B
" nil t nil nil nil nil nil nil nil nil t)

(quail-define-rules
 ("'A" ?,AA(B)
 ("'E" ?,AI(B)
 ("'I" ?,AM(B)
 ("'O" ?,AS(B)
 ("'U" ?,AZ(B)
 ("'a" ?,Aa(B)
 ("'e" ?,Ai(B)
 ("'i" ?,Am(B)
 ("'o" ?,As(B)
 ("'u" ?,Az(B)
 ("' " ?')
)

(quail-define-package
 "portuguese-prefix" "Latin-1" "PT>" t
 "Portuguese input method with prefix modifiers

    effect   | prefix | examples
 ------------+--------+----------
    acute    |   '    | 'a -> ,Aa(B, '' -> ,A4(B
    grave    |   `    | `a -> ,A`(B
  circumflex |   ^    | ^a -> ,Ab(B
  diaeresis  |   \"    | \"u -> ,A|(B
    tilde    |   ~    | ~a -> ,Ac(B
   cedilla   | ' or , | 'c -> ,Ag(B, ,c -> ,Ag(B
" nil t nil nil nil nil nil nil nil nil t)

(quail-define-rules
 ("'A" ?,AA(B)
 ("'E" ?,AI(B)
 ("'I" ?,AM(B)
 ("'O" ?,AS(B)
 ("'U" ?,AZ(B)
 ("'C" ?,AG(B)
 ("'a" ?,Aa(B)
 ("'e" ?,Ai(B)
 ("'i" ?,Am(B)
 ("'o" ?,As(B)
 ("'u" ?,Az(B)
 ("'c" ?,Ag(B)
 ("' " ?')
 ("`A" ?,A@(B)
 ("`a" ?,A`(B)
 ("` " ?`)
 ("^A" ?,AB(B)
 ("^E" ?,AJ(B)
 ("^O" ?,AT(B)
 ("^a" ?,Ab(B)
 ("^e" ?,Aj(B)
 ("^o" ?,At(B)
 ("^ " ?^)
 ("\"U" ?,A\(B)
 ("\"u" ?,A|(B)
 ("\" " ?\")
 ("~A" ?,AC(B)
 ("~O" ?,AU(B)
 ("~a" ?,Ac(B)
 ("~o" ?,Au(B)
 ("~ " ?~)
 (",c" ?,Ag(B)
 (",C" ?,AG(B)
 (",," ?,)
)

(quail-define-package
 "spanish-prefix" "Latin-1" "ES>" t
 "Spanish (Espa,Aq(Bol) input method with prefix modifiers

    effect   | prefix | examples
 ------------+--------+----------
    acute    |   '    | 'a -> ,Aa(B
  diaeresis  |   \"    | \"u -> ,A|(B
    tilde    |   ~    | ~n -> ,Aq(B
   symbol    |   ~    | ~> -> ,A;(B, ~< -> ,A+(B, ~! -> ,A!(B, ~? -> ,A?(B
" nil t nil nil nil nil nil nil nil nil t)

(quail-define-rules
 ("'A" ?,AA(B)
 ("'E" ?,AI(B)
 ("'I" ?,AM(B)
 ("'O" ?,AS(B)
 ("'U" ?,AZ(B)
 ("'a" ?,Aa(B)
 ("'e" ?,Ai(B)
 ("'i" ?,Am(B)
 ("'o" ?,As(B)
 ("'u" ?,Az(B)
 ("' " ?')
 ("\"U" ?,A\(B)
 ("\"u" ?,A|(B)
 ("\" " ?\")
 ("~N" ?,AQ(B)
 ("~n" ?,Aq(B)
 ("~>" ?\,A;(B)
 ("~<" ?\,A+(B)
 ("~!" ?,A!(B)
 ("~?" ?,A?(B)
 ("~ " ?~)
)

(quail-define-package
 "latin-2-prefix" "Latin-2" "2>" t
 "Latin-2 characters input method with prefix modifiers

    effect   | prefix | examples
 ------------+--------+----------
    acute    |   '    | 'a -> ,Ba(B, '' -> ?,B4(B
  circumflex |   ^    | ^a -> ,Bb(B
  diaeresis  |   \"    | \"a -> ,Bd(B, \"\" -> ,B((B
    breve    |   ~    | ~a -> ,Bc(B
    caron    |   ~    | ~c -> ,Bh(B
   cedilla   |   `    | `c -> ,Bg(B, `e -> ?,Bj(B
    misc     |  ----  | 'd -> ,Bp(B, `l -> ,B3(B, `z -> ,B?(B, ~o -> ,Bu(B, ~u -> ,B{(B
   symbol    |   ~    | `. -> ,B(B, ~~ -> ,B"(B, ~. -> ?,B8(B
" nil t nil nil nil nil nil nil nil nil t)

(quail-define-rules
 ("'A" ?,BA(B)
 ("'C" ?,BF(B)
 ("'D" ?,BP(B)
 ("'E" ?,BI(B)
 ("'I" ?,BM(B)
 ("'L" ?,BE(B)
 ("'N" ?,BQ(B)
 ("'O" ?,BS(B)
 ("'R" ?,B@(B)
 ("'S" ?,B&(B)
 ("'U" ?,BZ(B)
 ("'Y" ?,B](B)
 ("'Z" ?,B,(B)
 ("'a" ?,Ba(B)
 ("'c" ?,Bf(B)
 ("'d" ?,Bp(B)
 ("'e" ?,Bi(B)
 ("'i" ?,Bm(B)
 ("'l" ?,Be(B)
 ("'n" ?,Bq(B)
 ("'o" ?,Bs(B)
 ("'r" ?,B`(B)
 ("'s" ?,B6(B)
 ("'u" ?,Bz(B)
 ("'y" ?,B}(B)
 ("'z" ?,B<(B)
 ("''" ?,B4(B)
 ("' " ?')
 ("`A" ?,B!(B)
 ("`C" ?,BG(B)
 ("`E" ?,BJ(B)
 ("`L" ?,B#(B)
 ("`S" ?,B*(B)
 ("`T" ?,B^(B)
 ("`Z" ?,B/(B)
 ("`a" ?,B1(B)
 ("`l" ?,B3(B)
 ("`c" ?,Bg(B)
 ("`e" ?,Bj(B)
 ("`s" ?,B:(B)
 ("`t" ?,B~(B)
 ("`z" ?,B?(B)
 ("``" ?,B*(B)
 ("`." ?,B(B)
 ("` " ?`)
 ("^A" ?,BB(B)
 ("^I" ?,BN(B)
 ("^O" ?,BT(B)
 ("^a" ?,Bb(B)
 ("^i" ?,Bn(B)
 ("^o" ?,Bt(B)
 ("^^" ?^)
 ("^ " ?^)
 ("\"A" ?,BD(B)
 ("\"E" ?,BK(B)
 ("\"O" ?,BV(B)
 ("\"U" ?,B\(B)
 ("\"a" ?,Bd(B)
 ("\"e" ?,Bk(B)
 ("\"o" ?,Bv(B)
 ("\"s" ?,B_(B)
 ("\"u" ?,B|(B)
 ("\"\"" ?,B((B)
 ("\" " ?\")
 ("~A" ?,BC(B)
 ("~C" ?,BH(B)
 ("~D" ?,BO(B)
 ("~L" ?,B%(B)
 ("~N" ?,BR(B)
 ("~O" ?,BU(B)
 ("~R" ?,BX(B)
 ("~S" ?,B)(B)
 ("~T" ?,B+(B)
 ("~U" ?,B[(B)
 ("~Z" ?,B.(B)
 ("~a" ?,Bc(B)
 ("~c" ?,Bh(B)
 ("~d" ?,Bo(B)
 ("~l" ?,B5(B)
 ("~n" ?,Br(B)
 ("~o" ?,Bu(B)
 ("~r" ?,Bx(B)
 ("~s" ?,B9(B)
 ("~t" ?,B;(B)
 ("~u" ?,B{(B)
 ("~z" ?,B>(B)
 ("~v" ?,B"(B)
 ("~~" ?,B"(B)
 ("~." ?,B8(B)
 ("~ " ?~)
)

(quail-define-package
 "latin-3-prefix" "Latin-3" "3>" t
 "Latin-3 characters input method with prefix modifiers

    effect   | prefix | examples
 ------------+--------+----------
    acute    |   '    | 'a -> ,Ca(B, '' -> ?,C4(B
    grave    |   `    | `a -> ,C`(B
  circumflex |   ^    | ^a -> ,Cb(B
  diaeresis  |   \"    | \"a -> ,Cd(B, \"\" -> ,C((B
   cedilla   |   ~    | ~c -> ,Cg(B, ~s -> ,C:(B, ~~ -> ,C8(B
  dot above  | ~ / .  | ~o -> ,Cu(B, /o -> ,Cu(B, .o -> ,Cu(B
    misc     |  ----  | \"s -> ,C_(B, ~g -> ,C;(B, ~u -> ,C}(B, /h -> ,C1(B, /i -> ,C9(B
   symbol    |   ~    | ~` -> ,C"(B, /# -> ,C#(B, /$ -> ,C$(B, // -> ,C0(B
" nil t nil nil nil nil nil nil nil nil t)

(quail-define-rules
 ("'A" ?,CA(B)
 ("'E" ?,CI(B)
 ("'I" ?,CM(B)
 ("'O" ?,CS(B)
 ("'U" ?,CZ(B)
 ("'a" ?,Ca(B)
 ("'e" ?,Ci(B)
 ("'i" ?,Cm(B)
 ("'o" ?,Cs(B)
 ("'u" ?,Cz(B)
 ("''" ?,C4(B)
 ("' " ?')
 ("`A" ?,C@(B)
 ("`E" ?,CH(B)
 ("`I" ?,CL(B)
 ("`O" ?,CR(B)
 ("`U" ?,CY(B)
 ("`a" ?,C`(B)
 ("`e" ?,Ch(B)
 ("`i" ?,Cl(B)
 ("`o" ?,Cr(B)
 ("`u" ?,Cy(B)
 ("``" ?`)
 ("` " ?`)
 ("^A" ?,CB(B)
 ("^C" ?,CF(B)
 ("^E" ?,CJ(B)
 ("^G" ?,CX(B)
 ("^H" ?,C&(B)
 ("^I" ?,CN(B)
 ("^J" ?,C,(B)
 ("^O" ?,CT(B)
 ("^S" ?,C^(B)
 ("^U" ?,C[(B)
 ("^a" ?,Cb(B)
 ("^c" ?,Cf(B)
 ("^e" ?,Cj(B)
 ("^g" ?,Cx(B)
 ("^h" ?,C6(B)
 ("^i" ?,Cn(B)
 ("^j" ?,C<(B)
 ("^o" ?,Ct(B)
 ("^s" ?,C~(B)
 ("^u" ?,C{(B)
 ("^^" ?^)
 ("^ " ?^)
 ("\"A" ?,CD(B)
 ("\"E" ?,CK(B)
 ("\"I" ?,CO(B)
 ("\"O" ?,CV(B)
 ("\"U" ?,C\(B)
 ("\"a" ?,Cd(B)
 ("\"e" ?,Ck(B)
 ("\"i" ?,Co(B)
 ("\"o" ?,Cv(B)
 ("\"u" ?,C|(B)
 ("\"s" ?,C_(B)
 ("\"\"" ?,C((B)
 ("\" " ?\")
 ("~A" ?,CC(B)
 ("~C" ?,CG(B)
 ("~D" ?,CP(B)
 ("~N" ?,CQ(B)
 ("~O" ?,CU(B)
 ("~a" ?,Cc(B)
 ("~c" ?,Cg(B)
 ("~d" ?,Cp(B)
 ("~n" ?,Cq(B)
 ("~o" ?,Cu(B)
 ("~$" ?,C%(B)
 ("~S" ?,C*(B)
 ("~s" ?,C:(B)
 ("~G" ?,C+(B)
 ("~g" ?,C;(B)
 ("~U" ?,C](B)
 ("~u" ?,C}(B)
 ("~`" ?,C"(B)
 ("~~" ?,C8(B)
 ("~ " ?~)
 ("/C" ?,CE(B)
 ("/G" ?,CU(B)
 ("/H" ?,C!(B)
 ("/I" ?,C)(B)
 ("/Z" ?,C/(B)
 ("/c" ?,Ce(B)
 ("/g" ?,Cu(B)
 ("/h" ?,C1(B)
 ("/i" ?,C9(B)
 ("/z" ?,C?(B)
 ("/r" ?,C.(B)
 ("/." ?,C(B)
 ("/#" ?,C#(B)
 ("/$" ?,C$(B)
 ("//" ?,C0(B)
 ("/ " ?/)
 (".C" ?,CE(B)
 (".G" ?,CU(B)
 (".I" ?,C)(B)
 (".Z" ?,C/(B)
 (".c" ?,Ce(B)
 (".g" ?,Cu(B)
 (".z" ?,C?(B)
)

(quail-define-package
 "polish-slash" "Polish" "PL>" nil
 "Polish diacritics and slash character are input as `/[acelnosxzACELNOSXZ/]'.
For example, the character named `aogonek' is obtained by `/a'."
 nil t t t nil nil nil nil nil nil t)

(quail-define-rules
 ("//" ?/)
 ("/a" ?,B1(B)
 ("/c" ?,Bf(B)
 ("/e" ?,Bj(B)
 ("/l" ?,B3(B)
 ("/n" ?,Bq(B)
 ("/o" ?,Bs(B)
 ("/s" ?,B6(B)
 ("/x" ?,B<(B)
 ("/z" ?,B?(B)
 ("/A" ?,B!(B)
 ("/C" ?,BF(B)
 ("/E" ?,BJ(B)
 ("/L" ?,B#(B)
 ("/N" ?,BQ(B)
 ("/O" ?,BS(B)
 ("/S" ?,B&(B)
 ("/X" ?,B,(B)
 ("/Z" ?,B/(B))

(quail-define-package
 "latin-9-prefix" "Latin-9" "0>" t
 (apply 'format
 "Latin-9 (ISO-8859-15) characters input method with prefix modifiers

    effect   | prefix | examples
 ------------+--------+----------
    acute    |   '    | 'a -> %c
    grave    |   `    | `a -> %c
  circumflex |   ^    | ^a -> %c
  diaeresis  |   \"    | \"a -> %c, \"Y -> %c
    tilde    |   ~    | ~a -> %c
    caron    |   ~    | ~z -> %c
   cedilla   |   ~    | ~c -> %c
    misc     | \" ~ /  | \"s -> %c  ~d -> %c  ~t -> %c  /a -> %c  /e -> %c  /o -> %c
             | \" ~ /  | /o -> %c
   symbol    |   ~    | ~> -> %c  ~< -> %c  ~! -> %c  ~? -> %c  ~~ -> %c
             |   ~    | ~s -> %c  ~e -> %c  ~. -> %c  ~$ -> %c  ~u -> %c
             |   ~    | ~- -> %c  ~= -> %c
   symbol    |  _ /   | _o -> %c  _a -> %c  // -> %c  /\\ -> %c  _y -> %c
             |  _ /   | _: -> %c  /c -> %c  ~p -> %c
             |  _ /   | /= -> %c
   symbol    |   ^    | ^r -> %c  ^c -> %c  ^1 -> %c  ^2 -> %c  ^3 -> %c  _a -> %c
" (mapcar (lambda (code) (make-char 'latin-iso8859-15 code))

          '(97 96 98 100 62 99 56 103 95 112 126 101 102 120 61 59 43 33 63
            56 39 36 55 35 53 45 47 58 42 48 87 37 119 34 54 44 46 41 57 50 
            51 42)))
 nil t nil nil nil nil nil nil nil nil t)

(loop
  for (string code)
  in '(("'A" #x41)
       ("'E" #x49)
       ("'I" #x4d)
       ("'O" #x53)
       ("'U" #x5a)
       ("'Y" #x5d)
       ("'a" #x61)
       ("'e" #x69)
       ("'i" #x6d)
       ("'o" #x73)
       ("'u" #x7a)
       ("'y" #x7d)
       ("' " #x27)
       ("`A" #x40)
       ("`E" #x48)
       ("`I" #x4c)
       ("`O" #x52)
       ("`U" #x59)
       ("`a" #x60)
       ("`e" #x68)
       ("`i" #x6c)
       ("`o" #x72)
       ("`u" #x79)
       ("``" #x60)
       ("` " #x60)
       ("^A" #x42)
       ("^E" #x4a)
       ("^I" #x4e)
       ("^O" #x54)
       ("^U" #x5b)
       ("^a" #x62)
       ("^e" #x6a)
       ("^i" #x6e)
       ("^o" #x74)
       ("^u" #x7b)
       ("^^" #x5e)
       ("^ " #x5e)
       ("\"A" #x44)
       ("\"E" #x4b)
       ("\"I" #x4f)
       ("\"O" #x56)
       ("\"U" #x5c)
       ("\"a" #x64)
       ("\"e" #x6b)
       ("\"i" #x6f)
       ("\"o" #x76)
       ("\"s" #x5f)
       ("\"u" #x7c)
       ("\"y" #x7f)
       ("\" " #x22)
       ("~A" #x43)
       ("~C" #x47)
       ("~D" #x50)
       ("~N" #x51)
       ("~O" #x55)
       ("~S" #x26)
       ("~T" #x5e)
       ("~Z" #x34)
       ("~a" #x63)
       ("~c" #x67)
       ("~d" #x70)
       ("~n" #x71)
       ("~o" #x75)
       ("~s" #x28)
       ("~t" #x7e)
       ("~z" #x38)
       ("~>" #x3b)
       ("~<" #x2b)
       ("~!" #x21)
       ("~?" #x3f)
       ("~ " #x7e)
       ("/A" #x45)
       ("/E" #x46)
       ("/O" #x58)
       ("/a" #x65)
       ("/e" #x66)
       ("/o" #x78)
       ("//" #x30)
       ("/ " #x2f)
       ("_o" #x3a)
       ("_a" #x2a)
       ("_+" #x31)
       ("_y" #x25)
       ("_:" #x77)
       ("_ " #x20)
       ("__" #x5f)
       ("/c" #x22)
       ("/\\" #x57)
       ("/o" #x3d)
       ("/O" #x3c)
       ("\"Y" #x3e)
       ("~s" #x27)
       ("~p" #x36)
       ("~e" #x24)
       ("~." #x37)
       ("~$" #x23)
       ("~u" #x35)
       ("^r" #x2e)
       ("^c" #x29)
       ("^1" #x39)
       ("^2" #x32)
       ("^3" #x33)
       ("~-" #x2d)
       ("~=" #x2f)
       ("/=" #x2c))
  do (quail-defrule string (make-char 'latin-iso8859-15 code)))

;; Latin-8 was done by an Englishman -- Johnny Celt should take a
;; squint at it.
;;
;;    "Johnny" Celt here; it's fine. Aidan Kehoe, So Mai 20 15:58:45 CEST
;;    2007

(quail-define-package
 "latin-8-prefix" "Latin-8" "8>" t
 (apply 'format
 "Latin-8 (ISO 8859-14) characters input method with prefix modifiers

    effect   | prefix | examples
 ------------+--------+----------
    acute    |   '    | 'a -> %c
    grave    |   `    | `a -> %c
  circumflex |   ^    | ^w -> %c
  diaeresis  |   \"    | \"a -> %c
  dot above  |   .    | .b -> %c
    tilde    |   ~    | ~a -> %c
   cedilla   |   ~    | ~c -> %c
    misc     | \" ~ /  | \"s -> %c   /a -> %c  /e -> %c  /o -> %c
             |   ~    | ~s -> %c  ~$ -> %c  ~p -> %c
   symbol    |   ^    | ^r -> %c  ^c -> %c
"
 (mapcar (lambda (code) (make-char 'latin-iso8859-14 code))
         '(97 96 112 100 34 99 103 95 101 102 120 39 35 54 46 41)))
 nil t nil nil nil nil nil nil nil nil t)

;; Basically following Latin-1, plus dottiness from Latin-3.
(loop
  for (string code)
  in '((".B" #x21)
       (".b" #x22)
       (".c" #x25)
       (".C" #x24)
       (".D" #x26)
       (".d" #x2b)
       (".f" #x31)
       (".F" #x30)
       (".g" #x33)
       (".G" #x32)
       (".m" #x35)
       (".M" #x34)
       (".p" #x39)
       (".P" #x37)
       (".s" #x3f)
       (".S" #x3b)
       (".t" #x77)
       (".T" #x57)
       ("'A" #x41)
       ("'E" #x49)
       ("'I" #x4d)
       ("'O" #x53)
       ("'U" #x5a)
       ("'Y" #x5d)
       ("'W" #x2a)
       ("'a" #x61)
       ("'e" #x69)
       ("'i" #x6d)
       ("'o" #x73)
       ("'u" #x7a)
       ("'w" #x3a)
       ("'y" #x7d)
       ("' " #x27)
       ("`A" #x40)
       ("`E" #x48)
       ("`I" #x4c)
       ("`O" #x52)
       ("`U" #x59)
       ("`W" #x28)
       ("`Y" #x2c)
       ("`a" #x60)
       ("`e" #x68)
       ("`i" #x6c)
       ("`o" #x72)
       ("`u" #x79)
       ("`w" #x38)
       ("`y" #x3c)
       ("``" #x60)
       ("` " #x60)
       ("^A" #x42)
       ("^E" #x4a)
       ("^I" #x4e)
       ("^O" #x54)
       ("^U" #x5b)
       ("^a" #x62)
       ("^e" #x6a)
       ("^i" #x6e)
       ("^o" #x74)
       ("^u" #x7b)
       ("^w" #x70)
       ("^W" #x50)
       ("^y" #x7e)
       ("^Y" #x5e)
       ("^^" #x5e)
       ("^ " #x5e)
       ("\"A" #x44)
       ("\"E" #x4b)
       ("\"I" #x4f)
       ("\"O" #x56)
       ("\"U" #x5c)
       ("\"a" #x64)
       ("\"e" #x6b)
       ("\"i" #x6f)
       ("\"o" #x76)
       ("\"s" #x5f)
       ("\"u" #x7c)
       ("\"w" #x3e)
       ("\"W" #x3d)
       ("\"y" #x7f)
       ("\"Y" #x2f)
       ("\" " #x22)
       ("~A" #x43)
       ("~C" #x47)
       ("~N" #x51)
       ("~O" #x55)
       ("~a" #x63)
       ("~c" #x67)
       ("~n" #x71)
       ("~o" #x75)
       ("~ " #x7e)
       ("/A" #x45)
       ("/E" #x46)
       ("/O" #x58)
       ("/a" #x65)
       ("/e" #x66)
       ("/o" #x78)
       ("/ " #x2f)
       ("~p" #x36)
       ("~s" #x27)
       ("~$" #x23)
       ("^r" #x2e)
       ("^c" #x29))
  do (quail-defrule string (make-char 'latin-iso8859-14 code)))

;; XEmacs change; breves removed from the docstring since we don't actually
;; support them. Also, encode characters using the standard ISO-2022-7
;; escapes instead of GNU's mule-unicode-XXXX-XXXX character sets. 
(quail-define-package
 "latin-prefix" "Latin" "L>" t
 (format "Latin characters input method with prefix modifiers.
This is the union of various input methods originally made for input
of characters from a single Latin-N charset.

    effect   | prefix | examples
 ------------+--------+----------
    acute    |   '    | 'a -> ,Aa(B, '' -> ,A4(B
    grave    |   `    | `a -> ,A`(B
  circumflex |   ^    | ^a -> ,Ab(B
  diaeresis  |   \"    | \"a -> ,Ad(B  \"\" -> ,A((B
    tilde    |   ~    | ~a -> ,Ac(B
   cedilla   |   ~    | ~c -> ,Ag(B
  dot above  |   .    | .g -> %c   .B -> %c
    misc     | \" ~ /  | \"s -> ,A_(B  ~d -> ,Ap(B  ~t -> ,A~(B  /a -> ,Ae(B  /e -> ,Af(B  /o -> ,Ax(B
   symbol    |   ~    | ~> -> ,A;(B  ~< -> ,A+(B  ~! -> ,A!(B  ~? -> ,A?(B  ~~ -> ,A8(B
   symbol    |  _ /   | _o -> ,A:(B  _a -> ,A*(B  // -> ,A0(B  /\\ -> ,AW(B  _y -> ,A%(B
   symbol    |   ^    | ^r -> ,A.(B  ^c -> ,A)(B  ^1 -> ,A9(B  ^2 -> ,A2(B  ^3 -> ,A3(B
" (make-char 'latin-iso8859-3 117) (make-char 'latin-iso8859-14 33))
 nil t nil nil nil nil nil nil nil nil t)

(loop for (string charset code)
  in '(("' " ascii #x27)
       ("''" latin-iso8859-1 #x34)
       ("'A" latin-iso8859-1 #x41)
       ("'E" latin-iso8859-1 #x49)
       ("'I" latin-iso8859-1 #x4d)
       ("'O" latin-iso8859-1 #x53)
       ("'U" latin-iso8859-1 #x5a)
       ("'W" latin-iso8859-14 #x2a)
       ("'Y" latin-iso8859-1 #x5d)
       ("'a" latin-iso8859-1 #x61)
       ("'e" latin-iso8859-1 #x69)
       ("'i" latin-iso8859-1 #x6d)
       ("'o" latin-iso8859-1 #x73)
       ("'u" latin-iso8859-1 #x7a)
       ("'w" latin-iso8859-14 #x3a)
       ("'y" latin-iso8859-1 #x7d)
       (".B" latin-iso8859-14 #x21)
       (".C" latin-iso8859-14 #x24)
       (".D" latin-iso8859-14 #x26)
       (".F" latin-iso8859-14 #x30)
       (".G" latin-iso8859-14 #x32)
       (".I" latin-iso8859-3 #x29)
       (".M" latin-iso8859-14 #x34)
       (".P" latin-iso8859-14 #x37)
       (".S" latin-iso8859-14 #x3b)
       (".T" latin-iso8859-14 #x57)
       (".Z" latin-iso8859-3 #x2f)
       (".b" latin-iso8859-14 #x22)
       (".c" latin-iso8859-14 #x25)
       (".d" latin-iso8859-14 #x2b)
       (".f" latin-iso8859-14 #x31)
       (".g" latin-iso8859-3 #x75)
       (".m" latin-iso8859-14 #x35)
       (".p" latin-iso8859-14 #x39)
       (".s" latin-iso8859-14 #x3f)
       (".t" latin-iso8859-14 #x77)
       (".z" latin-iso8859-3 #x3f)
       ("/ " ascii #x2f)
       ("/#" latin-iso8859-1 #x23)
       ("/$" latin-iso8859-1 #x24)
       ("/." latin-iso8859-3 #x7f)
       ("//" latin-iso8859-1 #x30)
       ("/2" latin-iso8859-1 #x3d)
       ("/3" latin-iso8859-1 #x3e)
       ("/4" latin-iso8859-1 #x3c)
       ("/=" latin-iso8859-1 #x2c)
       ("/A" latin-iso8859-1 #x45)
       ("/C" latin-iso8859-3 #x45)
       ("/E" latin-iso8859-1 #x46)
       ("/G" latin-iso8859-3 #x55)
       ("/H" latin-iso8859-3 #x21)
       ("/I" latin-iso8859-3 #x29)
       ("/O" latin-iso8859-1 #x58)
       ("/Z" latin-iso8859-3 #x2f)
       ("/\\" latin-iso8859-1 #x57)
       ("/a" latin-iso8859-1 #x65)
       ("/c" latin-iso8859-1 #x22)
       ("/c" latin-iso8859-3 #x65)
       ("/e" latin-iso8859-1 #x66)
       ("/g" latin-iso8859-3 #x75)
       ("/h" latin-iso8859-3 #x31)
       ("/i" latin-iso8859-3 #x39)
       ("/o" latin-iso8859-1 #x78)
       ("/o" latin-iso8859-15 #x3d)
       ("/z" latin-iso8859-2 #x3f)
       ("\" " ascii #x22)
       ("\"A" latin-iso8859-1 #x44)
       ("\"E" latin-iso8859-1 #x4b)
       ("\"I" latin-iso8859-1 #x4f)
       ("\"O" latin-iso8859-1 #x56)
       ("\"U" latin-iso8859-1 #x5c)
       ("\"W" latin-iso8859-14 #x3d)
       ("\"Y" latin-iso8859-14 #x2f)
       ("\"\"" latin-iso8859-1 #x28)
       ("\"a" latin-iso8859-1 #x64)
       ("\"e" latin-iso8859-1 #x6b)
       ("\"i" latin-iso8859-1 #x6f)
       ("\"o" latin-iso8859-1 #x76)
       ("\"s" latin-iso8859-1 #x5f)
       ("\"u" latin-iso8859-1 #x7c)
       ("\"w" latin-iso8859-14 #x3e)
       ("\"y" latin-iso8859-1 #x7f)
       ("^ " ascii #x5e)
       ("^1" latin-iso8859-1 #x39)
       ("^2" latin-iso8859-1 #x32)
       ("^3" latin-iso8859-1 #x33)
       ("^A" latin-iso8859-1 #x42)
       ("^C" latin-iso8859-3 #x46)
       ("^E" latin-iso8859-1 #x4a)
       ("^G" latin-iso8859-3 #x58)
       ("^H" latin-iso8859-3 #x26)
       ("^I" latin-iso8859-1 #x4e)
       ("^J" latin-iso8859-3 #x2c)
       ("^O" latin-iso8859-1 #x54)
       ("^S" latin-iso8859-3 #x5e)
       ("^U" latin-iso8859-1 #x5b)
       ("^W" latin-iso8859-14 #x50)
       ("^Y" latin-iso8859-14 #x5e)
       ("^^" ascii #x5e)
       ("^a" latin-iso8859-1 #x62)
       ("^c" latin-iso8859-1 #x29)
       ("^c" latin-iso8859-3 #x66)
       ("^e" latin-iso8859-1 #x6a)
       ("^g" latin-iso8859-3 #x78)
       ("^h" latin-iso8859-3 #x36)
       ("^i" latin-iso8859-1 #x6e)
       ("^j" latin-iso8859-3 #x3c)
       ("^o" latin-iso8859-1 #x74)
       ("^r" latin-iso8859-1 #x2e)
       ("^s" latin-iso8859-3 #x7e)
       ("^u" latin-iso8859-1 #x7b)
       ("^w" latin-iso8859-14 #x70)
       ("^y" latin-iso8859-14 #x7e)
       ("_+" latin-iso8859-1 #x31)
       ("_:" latin-iso8859-1 #x77)
       ("_a" latin-iso8859-1 #x2a)
       ("_o" latin-iso8859-1 #x3a)
       ("_y" latin-iso8859-1 #x25)
       ("_ " latin-iso8859-1 #x20)
       ("` " ascii #x60)
       ("`A" latin-iso8859-1 #x40)
       ("`E" latin-iso8859-1 #x48)
       ("`I" latin-iso8859-1 #x4c)
       ("`O" latin-iso8859-1 #x52)
       ("`U" latin-iso8859-1 #x59)
       ("`W" latin-iso8859-14 #x28)
       ("`Y" latin-iso8859-14 #x2c)
       ("``" ascii #x60)
       ("`a" latin-iso8859-1 #x60)
       ("`e" latin-iso8859-1 #x68)
       ("`i" latin-iso8859-1 #x6c)
       ("`o" latin-iso8859-1 #x72)
       ("`u" latin-iso8859-1 #x79)
       ("`w" latin-iso8859-14 #x38)
       ("`y" latin-iso8859-14 #x3c)
       ("~ " ascii #x7e)
       ("~!" latin-iso8859-1 #x21)
       ("~$" latin-iso8859-1 #x23)
       ("~-" latin-iso8859-1 #x2d)
       ("~." latin-iso8859-1 #x37)
       ("~<" latin-iso8859-1 #x2b)
       ("~=" latin-iso8859-1 #x2f)
       ("~>" latin-iso8859-1 #x3b)
       ("~?" latin-iso8859-1 #x3f)
       ("~A" latin-iso8859-1 #x43)
       ("~C" latin-iso8859-1 #x47)
       ("~D" latin-iso8859-1 #x50)
       ("~G" latin-iso8859-3 #x2b)
       ("~N" latin-iso8859-1 #x51)
       ("~O" latin-iso8859-1 #x55)
       ("~O" latin-iso8859-2 #x55)
       ("~S" latin-iso8859-3 #x2a)
       ("~S" latin-iso8859-2 #x29)
       ("~T" latin-iso8859-1 #x5e)
       ("~U" latin-iso8859-3 #x5d)
       ("~Z" latin-iso8859-2 #x2e)
       ("~`" latin-iso8859-3 #x22)
       ("~a" latin-iso8859-1 #x63)
       ("~c" latin-iso8859-1 #x67)
       ("~d" latin-iso8859-1 #x70)
       ("~e" latin-iso8859-15 #x24)
       ("~g" latin-iso8859-3 #x3b)
       ("~n" latin-iso8859-1 #x71)
       ("~o" latin-iso8859-1 #x75)
       ("~p" latin-iso8859-1 #x36)
       ("~s" latin-iso8859-1 #x27)
       ("~S" latin-iso8859-3 #x2a)
       ("~s" latin-iso8859-2 #x39)
       ("~t" latin-iso8859-1 #x7e)
       ("~u" latin-iso8859-1 #x35)
       ("~u" latin-iso8859-3 #x7d)
       ("~x" latin-iso8859-1 #x24)
       ("~z" latin-iso8859-15 #x38)
       ("~|" latin-iso8859-1 #x26)
       ("~~" latin-iso8859-1 #x38))
  do (quail-defrule string (make-char charset code)))

;; XEmacs change; added. 
(quail-define-package
 "latin-10-prefix" "Latin-10" "10>" t
 (apply 'format
 "Latin-10 (ISO 8859-16) characters input method with prefix modifiers

    effect   | prefix | examples
 ------------+--------+----------
    acute    |   '    | 'a -> %c
    grave    |   `    | `a -> %c
  circumflex |   ^    | ^a -> %c
  diaeresis  |   \"    | \"a -> %c
    breve    |   ~    | ~a -> %c
   cedilla   |   ,    | ~c -> %c
    ogonek   |   ,    | ,a -> %c
  dbl. acute |   :    | :o -> %c
    misc     | ' , ~  | '' -> %c  ,, -> %c  ~< -> %c,  ~D -> %c
             |        | ~s -> %c   ~p -> %c
"
 (mapcar (lambda (code) (make-char 'latin-iso8859-16 code))
         '(97 96 98 100 99 103 34 117 53 37 43 80 39 54)))
 nil t nil nil nil nil nil nil nil nil t)

(loop
  for (string code)
  in '((",A" #x21)
       (",a" #x22)
       ("/L" #x23)
       ("~e" #x24)
       (",," #x25)
       ("~S" #x26)
       ("/S" #x27)
       ("~s" #x28)
       (",S" #x2a)
       ("~<" #x2b)
       ("'Z" #x2c)
       ("'z" #x2e)
       (".Z" #x2f)
       ("~C" #x32)
       ("/l" #x33)
       ("~Z" #x34)
       ("''" #x35)
       ("~p" #x36)
       ("~." #x37)
       ("~z" #x38)
       ("~c" #x39)
       (",s" #x3a)
       ("~>" #x3b)
       ("/O" #x3c)
       ("/o" #x3d)
       ("\"y" #x3e)
       (".z" #x3f)
       ("`A" #x40)
       ("'A" #x41)
       ("^A" #x42)
       ("~A" #x43)
       ("\"A" #x44)
       ("'C" #x45)
       ("/E" #x46)
       (",C" #x47)
       ("`E" #x48)
       ("'E" #x49)
       ("^E" #x4a)
       ("\"E" #x4b)
       ("`I" #x4c)
       ("'I" #x4d)
       ("^I" #x4e)
       ("\"I" #x4f)
       ("~D" #x50)
       ("'N" #x51)
       ("`O" #x52)
       ("'O" #x53)
       ("^O" #x54)
       (":O" #x55)
       ("\"O" #x56)
       ("'S" #x57)
       (":U" #x58)
       ("`U" #x59)
       ("'U" #x5a)
       ("^U" #x5b)
       ("\"U" #x5c)
       (",E" #x5d)
       (",T" #x5e)
       ("/s" #x5f)
       ("`a" #x60)
       ("'a" #x61)
       ("^a" #x62)
       ("~a" #x63)
       ("\"a" #x64)
       ("'c" #x65)
       ("/e" #x66)
       (",c" #x67)
       ("`e" #x68)
       ("'e" #x69)
       ("^e" #x6a)
       ("\"e" #x6b)
       ("`i" #x6c)
       ("'i" #x6d)
       ("^i" #x6e)
       ("\"i" #x6f)
       ("~d" #x70)
       ("'n" #x71)
       ("`o" #x72)
       ("'o" #x73)
       ("^o" #x74)
       (":o" #x75)
       ("\"o" #x76)
       ("'s" #x77)
       (":u" #x78)
       ("`u" #x79)
       ("'u" #x7a)
       ("^u" #x7b)
       ("\"u" #x7c)
       (",e" #x7d)
       (",t" #x7e)
       ("\"y" #x7f))
  do (quail-defrule string (make-char 'latin-iso8859-16 code)))

;; XEmacs change; added. 
(quail-define-package
 "latin-7-prefix" "Latin-7" "7>" t
 (apply 'format 
        "Latin-7 (ISO 8859-13) characters input method with prefix modifiers

    effect   | prefix | examples
 ------------+--------+----------
    acute    |   '    | 'c -> %c,  'o -> %c
    ogonek   |   ,    | ,a -> %c,  ,u -> %c
    macron   |   -    | -o -> %c,  -A -> %c
   diaeresis |   \"    | \"u -> %c,  \"o -> %c
    h,Aa(B%cek    |   ~    | ~c -> %c,  ~s -> %c
   cedilla   |   ,    | ,k -> %c
    misc     | ~ ' ,  | ~+ -> %c,  ~' -> %c [others] 
" (mapcar (lambda (code) (make-char 'latin-iso8859-13 code)) 
          '(99 115 96 120 116 66 124 118 104 104 112 109 49 127)))
 nil t nil nil nil nil nil nil nil nil t)

(loop
  for (string code)
  in '(("''" #x21)
       ("|c" #x22)
       ("~L" #x23)
       (",," #x25)
       ("||" #x26)
       ("/S" #x27)
       ("/O" #x28)
       ("~c" #x29)
       ("~r" #x2a)
       ("~<" #x2b)
       ("~-" #x2c)
       ("~-" #x2d)
       ("~r" #x2e)
       ("/E" #x2f)
       ("_o" #x30)
       ("~+" #x31)
       ("^2" #x32)
       ("^3" #x33)
       ("``" #x34)
       (",u" #x35)
       ("~p" #x36)
       ("~." #x37)
       ("/o" #x38)
       ("^1" #x39)
       (",r" #x3a)
       ("~>" #x3b)
       ("/14" #x3c)
       ("/12" #x3d)
       ("/34" #x3e)
       ("/e" #x3f)
       (",A" #x40)
       (",I" #x41)
       ("-A" #x42)
       ("'C" #x43)
       ("\"A" #x44)
       ("/A" #x45)
       (",E" #x46)
       ("-E" #x47)
       ("~C" #x48)
       ("'E" #x49)
       ("'Z" #x4a)
       (".E" #x4b)
       (",G" #x4c)
       (",K" #x4d)
       ("-I" #x4e)
       (",L" #x4f)
       ("~S" #x50)
       ("'N" #x51)
       (",N" #x52)
       ("'O" #x53)
       ("-O" #x54)
       ("~O" #x55)
       ("\"O" #x56)
       ("~x" #x57)
       (",U" #x58)
       ("/L" #x59)
       ("'S" #x5a)
       ("-U" #x5b)
       ("\"U" #x5c)
       (".Z" #x5d)
       ("~Z" #x5e)
       ("\"s" #x5f)
       (",a" #x60)
       (",i" #x61)
       ("-a" #x62)
       ("'c" #x63)
       ("\"a" #x64)
       ("/a" #x65)
       (",e" #x66)
       ("-e" #x67)
       ("~c" #x68)
       ("'e" #x69)
       ("'z" #x6a)
       (".e" #x6b)
       ("'g" #x6c)
       (",k" #x6d)
       ("-i" #x6e)
       (",l" #x6f)
       ("~s" #x70)
       ("'n" #x71)
       (",n" #x72)
       ("'o" #x73)
       ("-o" #x74)
       ("~o" #x75)
       ("\"o" #x76)
       ("/+" #x77)
       (",u" #x78)
       ("/l" #x79)
       ("'s" #x7a)
       ("-u" #x7b)
       ("\"u" #x7c)
       (".z" #x7d)
       ("~z" #x7e)
       ("~'" #x7f))
  do 
  (quail-defrule string (make-char 'latin-iso8859-13 code)))

;; XEmacs change; added. 
(quail-define-package
 "latin-5-prefix" "Latin-5" "1>" t
 "Latin-5 (ISO 8859-9) characters input method with prefix modifiers.

    effect   | prefix | examples
 ------------+--------+----------
    acute    |   '    | 'a -> ,Ma(B, '' -> ,M4(B
    grave    |   `    | `a -> ,M`(B
  circumflex |   ^    | ^a -> ,Mb(B
  diaeresis  |   \"    | \"a -> ,Md(B, \"\" -> ,M((B
    tilde    |   ~    | ~a -> ,Mc(B
   cedilla   |   ~    | ~c -> ,Mg(B
    misc     |  ----  | \"s -> ,M_(B, ~g -> ,Mp(B, ~s -> ,M~(B, /a -> ,Me(B, /e -> ,Mf(B, /o -> ,Mx(B
   symbol    |   ~    | ~> -> ,M;(B, ~< -> ,M+(B, ~! -> ,M!(B, ~? -> ,M?(B, ~~ -> ,M8(B
" nil t nil nil nil nil nil nil nil nil t)

(quail-define-rules
 ("'A" ?,MA(B)
 ("'E" ?,MI(B)
 ("'I" ?,MM(B)
 ("'O" ?,MS(B)
 ("'U" ?,MZ(B)
 ("'Y" ?,M](B)
 ("'a" ?,Ma(B)
 ("'e" ?,Mi(B)
 ("'i" ?,Mm(B)
 ("'o" ?,Ms(B)
 ("'u" ?,Mz(B)
 ("'y" ?,M}(B)
 ("''" ?,M4(B)
 ("' " ?')
 ("`A" ?,M@(B)
 ("`E" ?,MH(B)
 ("`I" ?,ML(B)
 ("`O" ?,MR(B)
 ("`U" ?,MY(B)
 ("`a" ?,M`(B)
 ("`e" ?,Mh(B)
 ("`i" ?,Ml(B)
 ("`o" ?,Mr(B)
 ("`u" ?,My(B)
 ("``" ?`)
 ("` " ?`)
 ("^A" ?,MB(B)
 ("^E" ?,MJ(B)
 ("^I" ?,MN(B)
 ("^O" ?,MT(B)
 ("^U" ?,M[(B)
 ("^a" ?,Mb(B)
 ("^e" ?,Mj(B)
 ("^i" ?,Mn(B)
 ("^o" ?,Mt(B)
 ("^u" ?,M{(B)
 ("^^" ?^)
 ("^ " ?^)
 ("\"A" ?,MD(B)
 ("\"E" ?,MK(B)
 ("\"I" ?,MO(B)
 ("\"O" ?,MV(B)
 ("\"U" ?,M\(B)
 ("\"a" ?,Md(B)
 ("\"e" ?,Mk(B)
 ("\"i" ?,Mo(B)
 ("\"o" ?,Mv(B)
 ("\"s" ?,M_(B)
 ("\"u" ?,M|(B)
 ("\"y" ?,M(B)
 ("\"\"" ?,M((B)
 ("\" " ?\")
 ("~A" ?,MC(B)
 ("~C" ?,MG(B)
 ("~G" ?,MP(B)
 ("~N" ?,MQ(B)
 ("~O" ?,MU(B)
 ("~S" ?,M^(B)
 ("~a" ?,Mc(B)
 ("~c" ?,Mg(B)
 ("~d" ?,Mp(B)
 ("~n" ?,Mq(B)
 ("~o" ?,Mu(B)
 ("~s" ?,M~(B)
 ("~>" ?\,M;(B)
 ("~<" ?\,M+(B)
 ("~!" ?,M!(B)
 ("~?" ?,M?(B)
 ("~~" ?,M8(B)
 ("~ " ?~)
 ("/A" ?,ME(B)
 ("/E" ?,MF(B)
 ("/O" ?,MX(B)
 ("/a" ?,Me(B)
 ("/e" ?,Mf(B)
 ("/o" ?,Mx(B)
 ("//" ?,M0(B)
 ("/ " ? ))

;; XEmacs change; added. 
(quail-define-package
 "latin-4-prefix" "Latin-4" "4>" t
 "Latin-4 (ISO 8859-4) characters input method with prefix modifiers

    effect   | prefix | examples
 ------------+--------+----------
    acute    |   '    | 'a -> ,Da(B
    macron   |   -    | -a -> ,D`(B
  circumflex |   ^    | ^a -> ,Db(B
  diaeresis  |   \"    | \"a -> ,Dd(B
    tilde    |   ~    | ~a -> ,Dc(B
   cedilla   |   ,    | ~k -> ,Ds(B
    ogonek   |   ,    | ,i -> ,Dg(B
    Nordic   |   /    | /a -> ,De(B, /e -> ,Df(B, /o -> ,Dx(B
    misc     | ' , ~  | '' -> ,D4(B  ,. -> ,D2(B ,, -> ,D8(B
             |        | /S -> ,D'(B,  ~x -> ,DW(B
" nil t nil nil nil nil nil nil nil nil t)

(quail-define-rules
 (",A" ?,D!(B)
 ("~k" ?,D"(B)
 (",R" ?,D#(B)
 ("~I" ?,D%(B)
 (",L" ?,D&(B)
 ("/S" ?,D'(B)
 ("\"\"" ?,D((B)
 ("-E" ?,D*(B)
 (",G" ?,D+(B)
 ("-T" ?,D,(B)
 ("~Z" ?,D.(B)
 ("--" ?,D/(B)
 (",." ?,D2(B)
 (",r" ?,D3(B)
 ("''" ?,D4(B)
 ("~i" ?,D5(B)
 (",l" ?,D6(B)
 ("~~" ?,D7(B)
 (",," ?,D8(B)
 ("~s" ?,D9(B)
 ("-e" ?,D:(B)
 ("'g" ?,D;(B)
 ("-t" ?,D<(B)
 (",N" ?,D=(B)
 ("~z" ?,D>(B)
 (".n" ?,D?(B)
 ("-A" ?,D@(B)
 ("'A" ?,DA(B)
 ("^A" ?,DB(B)
 ("~A" ?,DC(B)
 ("\"A" ?,DD(B)
 ("/A" ?,DE(B)
 ("/E" ?,DF(B)
 (",I" ?,DG(B)
 ("~C" ?,DH(B)
 ("'E" ?,DI(B)
 (",E" ?,DJ(B)
 ("\"E" ?,DK(B)
 (".E" ?,DL(B)
 ("'I" ?,DM(B)
 ("^I" ?,DN(B)
 ("-I" ?,DO(B)
 ("~D" ?,DP(B)
 (",N" ?,DQ(B)
 ("-O" ?,DR(B)
 (",K" ?,DS(B)
 ("^O" ?,DT(B)
 ("~O" ?,DU(B)
 ("\"O" ?,DV(B)
 ("~x" ?,DW(B)
 ("/O" ?,DX(B)
 (",U" ?,DY(B)
 ("'U" ?,DZ(B)
 ("^U" ?,D[(B)
 ("\"U" ?,D\(B)
 ("~U" ?,D](B)
 ("-U" ?,D^(B)
 ("/s" ?,D_(B)
 ("-a" ?,D`(B)
 ("'a" ?,Da(B)
 ("^a" ?,Db(B)
 ("~a" ?,Dc(B)
 ("\"a" ?,Dd(B)
 ("/a" ?,De(B)
 ("/e" ?,Df(B)
 (",i" ?,Dg(B)
 ("~c" ?,Dh(B)
 ("'e" ?,Di(B)
 (",e" ?,Dj(B)
 ("\"e" ?,Dk(B)
 (".e" ?,Dl(B)
 ("'i" ?,Dm(B)
 ("^i" ?,Dn(B)
 ("-i" ?,Do(B)
 ("~d" ?,Dp(B)
 (",n" ?,Dq(B)
 ("-o" ?,Dr(B)
 (",k" ?,Ds(B)
 ("^o" ?,Dt(B)
 ("~o" ?,Du(B)
 ("\"o" ?,Dv(B)
 ("~-" ?,Dw(B)
 ("/o" ?,Dx(B)
 (",u" ?,Dy(B)
 ("'u" ?,Dz(B)
 ("^u" ?,D{(B)
 ("\"u" ?,D|(B)
 ("~u" ?,D}(B)
 ("-u" ?,D~(B)
 (".." ?,D(B))


;;; arch-tag: 83017837-6b84-4366-b183-e0577e3ed838
;;; latin-pre.el ends here