/* gzfile.c
 *
 * gzip file wrapper
 * By Jon Rifkin <jon.rifkin@uconn.edu>
 * Copyright 2005 Jonathan Rifkin
 *
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "gzfile.h"

/*  Open text or gzip'ed file */
void gopen (gfile_t *gp, char *filename, char *mode) {
	/*  Read from standard in  */
	if (!strcmp("-",filename)) {
		gp->filetype = G_STDIN;
		gp->fp      = stdin;

	/*  Read from gzipped file (with .gz suffix)  */
	} else if (!strcmp(".gz",filename+strlen(filename)-3)) {
		gp->filetype = G_GZIP;
		gp->fp  = (void *) gzopen (filename, mode);
		gp->buf = (char *) malloc(G_BUFLEN);
		/*  Initialize cusor, limit pointers to indicate empty buffer */
		gp->cursor = gp->limit = gp->buf;

	/*  Read from normal plain text file */
	} else {
		gp->filetype = G_PLAIN;
		gp->fp = (void *) fopen (filename, mode);
	}

	/*  Test for error  */
	if (NULL==gp->fp) {
			printf ("ERROR:  Cannot open input file <%s>\n", filename);
			exit(1);
	}
}

/*  Close text or gzip'ed file  */
int gclose (gfile_t *gp) {
	if (gp->filetype==G_GZIP) {
		return gzclose( (gzFile)gp->fp);
		if (gp->fp) free (gp->fp);
		gp->fp = NULL;
	} else if (gp->filetype==G_PLAIN) {
		return fclose ( (FILE *)gp->fp);
	} else {
		return 0;
	}
}

/*  Read string from text or gzipped file  */
char *ggets (char *str, int nstr, gfile_t *gp) {
	if (gp->filetype==G_GZIP) {
		char *str_cursor = str;
		char *str_limit  = str + nstr - 1;
		while (1) {
			/*  Load input buffer if empty */
			/*  If gp->cursor equals gp->limit, then the last string from
			 *  the buffer ended at the last legal byte.  Otherwise the
			 *  last string overran the legal buffer  */
			if (gp->cursor>=gp->limit) {
				gp->len    = gzread( (gzFile) gp->fp, gp->buf, G_BUFLEN-1);
				gp->cursor = gp->buf;
				gp->limit  = gp->buf + gp->len;
				/*  Add a '\n' beyond buffer end to indicate buffer overrun */
				*gp->limit = '\n';
			}
			/*  Is buffer empty?  */
			if (gp->cursor==gp->limit) {
				/*  No string stored so return end of file  */
				if (str_cursor==str) {
					return (char *) NULL;
				/*  String is stored so break this loop to return it */
				} else {
					break;
				}
			}
			/*  Read through buffer and copy to string until either
			 *  - we reach '\n' which terminates string
			 *  - we reach '\n\n' which terminates file buffer
			 *  - we reach end of input string
			 *  */
			/*  Read through buffer and copy to str until '\n' */
			while (*gp->cursor!='\n') {
				/*  We've filled up the return string buffer  */
				if (str_cursor==str_limit) break;
				/*  Copy next char  */
				*(str_cursor++) = *(gp->cursor++);
			}
			/*  Not at buffer's end  */
			if (gp->cursor < gp->limit) {
				/*  We stopped at '\n' which terminates input string  */
				if (*gp->cursor=='\n') {
					/*  Copy character  */
					*(str_cursor++) = *(gp->cursor++);
					/*  We're done with this string, except for trailing \0 below */
					break;
				/*  We stopped because we filled return buffer  */
				} else {
					/*  Skip remainder of string in input buffer  */
					gp->cursor = strchr(gp->cursor,'\n') + 1;
					/*  Break from file buffer re-loading loop because we
					 *  have not reached end of file buffer  */
					if (gp->cursor < gp->limit) break;
				}
			}
			/*  If we reach here without break'ing then we need to re-load
			 *  file buffer  */
		}
		/* Terminate str with '\0' (whether we've reached the str or buffer end) */
		*str_cursor     = '\0';
		/*  Return current un-gzip'ed string  */
		return str;
	} else {
		return fgets(str, nstr, (FILE *) gp->fp);
	}
}
