// Copyright 2019 Drone.IO Inc. All rights reserved.
// Use of this source code is governed by the Drone Non-Commercial License
// that can be found in the LICENSE file.

// +build !oss

package global

import (
	"context"
	"database/sql"
	"testing"

	"github.com/drone/drone/core"
	"github.com/drone/drone/store/shared/db/dbtest"
	"github.com/drone/drone/store/shared/encrypt"
)

var noContext = context.TODO()

func TestSecret(t *testing.T) {
	conn, err := dbtest.Connect()
	if err != nil {
		t.Error(err)
		return
	}
	defer func() {
		dbtest.Reset(conn)
		dbtest.Disconnect(conn)
	}()

	store := New(conn, nil).(*secretStore)
	store.enc, _ = encrypt.New("fb4b4d6267c8a5ce8231f8b186dbca92")
	t.Run("Create", testSecretCreate(store))
}

func testSecretCreate(store *secretStore) func(t *testing.T) {
	return func(t *testing.T) {
		item := &core.Secret{
			Namespace: "octocat",
			Name:      "password",
			Data:      "correct-horse-battery-staple",
		}
		err := store.Create(noContext, item)
		if err != nil {
			t.Error(err)
		}
		if item.ID == 0 {
			t.Errorf("Want secret ID assigned, got %d", item.ID)
		}

		t.Run("Find", testSecretFind(store, item))
		t.Run("FindName", testSecretFindName(store))
		t.Run("List", testSecretList(store))
		t.Run("ListAll", testSecretListAll(store))
		t.Run("Update", testSecretUpdate(store))
		t.Run("Delete", testSecretDelete(store))
	}
}

func testSecretFind(store *secretStore, secret *core.Secret) func(t *testing.T) {
	return func(t *testing.T) {
		item, err := store.Find(noContext, secret.ID)
		if err != nil {
			t.Error(err)
		} else {
			t.Run("Fields", testSecret(item))
		}
	}
}

func testSecretFindName(store *secretStore) func(t *testing.T) {
	return func(t *testing.T) {
		item, err := store.FindName(noContext, "octocat", "password")
		if err != nil {
			t.Error(err)
		} else {
			t.Run("Fields", testSecret(item))
		}
	}
}

func testSecretList(store *secretStore) func(t *testing.T) {
	return func(t *testing.T) {
		list, err := store.List(noContext, "octocat")
		if err != nil {
			t.Error(err)
			return
		}
		if got, want := len(list), 1; got != want {
			t.Errorf("Want count %d, got %d", want, got)
		} else {
			t.Run("Fields", testSecret(list[0]))
		}
	}
}

func testSecretListAll(store *secretStore) func(t *testing.T) {
	return func(t *testing.T) {
		list, err := store.ListAll(noContext)
		if err != nil {
			t.Error(err)
			return
		}
		if got, want := len(list), 1; got != want {
			t.Errorf("Want count %d, got %d", want, got)
		} else {
			t.Run("Fields", testSecret(list[0]))
		}
	}
}

func testSecretUpdate(store *secretStore) func(t *testing.T) {
	return func(t *testing.T) {
		before, err := store.FindName(noContext, "octocat", "password")
		if err != nil {
			t.Error(err)
			return
		}
		err = store.Update(noContext, before)
		if err != nil {
			t.Error(err)
			return
		}
		after, err := store.Find(noContext, before.ID)
		if err != nil {
			t.Error(err)
			return
		}
		if after == nil {
			t.Fail()
		}
	}
}

func testSecretDelete(store *secretStore) func(t *testing.T) {
	return func(t *testing.T) {
		secret, err := store.FindName(noContext, "octocat", "password")
		if err != nil {
			t.Error(err)
			return
		}
		err = store.Delete(noContext, secret)
		if err != nil {
			t.Error(err)
			return
		}
		_, err = store.Find(noContext, secret.ID)
		if got, want := sql.ErrNoRows, err; got != want {
			t.Errorf("Want sql.ErrNoRows, got %v", got)
			return
		}
	}
}

func testSecret(item *core.Secret) func(t *testing.T) {
	return func(t *testing.T) {
		if got, want := item.Name, "password"; got != want {
			t.Errorf("Want secret name %q, got %q", want, got)
		}
		if got, want := item.Data, "correct-horse-battery-staple"; got != want {
			t.Errorf("Want secret data %q, got %q", want, got)
		}
	}
}
