// Copyright 2019 Drone.IO Inc. All rights reserved.
// Use of this source code is governed by the Drone Non-Commercial License
// that can be found in the LICENSE file.

// +build !oss

package registry

import (
	"context"
	"testing"

	"github.com/drone/drone/core"
	"github.com/google/go-cmp/cmp"
	"github.com/h2non/gock"
)

var noContext = context.TODO()

func TestEndpointSource(t *testing.T) {
	defer gock.Off()

	gock.New("https://company.com").
		Post("/auths").
		MatchHeader("Accept", "application/vnd.drone.registry.v1\\+json").
		MatchHeader("Accept-Encoding", "identity").
		MatchHeader("Content-Type", "application/json").
		Reply(200).
		BodyString(`[{"address":"index.docker.io","username":"octocat","password":"pa55word"}]`).
		Done()

	service := EndpointSource("https://company.com/auths", "GMEuUHQfmrMRsseWxi9YlIeBtn9lm6im", false)
	got, err := service.List(noContext, &core.RegistryArgs{Repo: &core.Repository{}, Build: &core.Build{}})
	if err != nil {
		t.Error(err)
		return
	}

	want := []*core.Registry{
		{
			Address:  "index.docker.io",
			Username: "octocat",
			Password: "pa55word",
		},
	}
	if diff := cmp.Diff(got, want); diff != "" {
		t.Errorf(diff)
		return
	}

	if gock.IsPending() {
		t.Errorf("Unfinished requests")
		return
	}
}

func TestEndpointSource_Err(t *testing.T) {
	defer gock.Off()

	gock.New("https://company.com").
		Post("/auths").
		MatchHeader("Accept", "application/vnd.drone.registry.v1\\+json").
		MatchHeader("Accept-Encoding", "identity").
		MatchHeader("Content-Type", "application/json").
		Reply(404)

	service := EndpointSource("https://company.com/auths", "GMEuUHQfmrMRsseWxi9YlIeBtn9lm6im", false)
	_, err := service.List(noContext, &core.RegistryArgs{Repo: &core.Repository{}, Build: &core.Build{}})
	if err == nil {
		t.Errorf("Expect http.Response error")
	} else if err.Error() != "Not Found" {
		t.Errorf("Expect Not Found error")
	}

	if gock.IsPending() {
		t.Errorf("Unfinished requests")
	}
}

func TestNotConfigured(t *testing.T) {
	service := EndpointSource("", "", false)
	registry, err := service.List(noContext, &core.RegistryArgs{})
	if err != nil {
		t.Error(err)
	}
	if registry != nil {
		t.Errorf("Expect nil registry")
	}
}
