// Copyright 2019 Drone.IO Inc. All rights reserved.
// Use of this source code is governed by the Drone Non-Commercial License
// that can be found in the LICENSE file.

// +build !oss

package admission

import (
	"errors"
	"testing"
	"time"

	"github.com/drone/drone/core"
	"github.com/drone/drone/mock"
	"github.com/golang/mock/gomock"
)

func TestNobot(t *testing.T) {
	controller := gomock.NewController(t)
	defer controller.Finish()

	localUser := &core.User{Login: "octocat"}
	remoteUser := &core.User{Login: "octocat", Created: time.Now().Unix() - 120} // 120 seconds
	users := mock.NewMockUserService(controller)
	users.EXPECT().Find(gomock.Any(), gomock.Any(), gomock.Any()).Return(remoteUser, nil)

	admission := Nobot(users, time.Minute) // 60 seconds
	err := admission.Admit(noContext, localUser)
	if err != nil {
		t.Error(err)
	}
}

func TestNobot_AccountTooNew(t *testing.T) {
	controller := gomock.NewController(t)
	defer controller.Finish()

	localUser := &core.User{Login: "octocat"}
	remoteUser := &core.User{Login: "octocat", Created: time.Now().Unix()}
	users := mock.NewMockUserService(controller)
	users.EXPECT().Find(gomock.Any(), gomock.Any(), gomock.Any()).Return(remoteUser, nil)

	admission := Nobot(users, time.Hour)
	err := admission.Admit(noContext, localUser)
	if err != ErrCannotVerify {
		t.Errorf("Expect ErrCannotVerify error")
	}
}

func TestNobot_ZeroDate(t *testing.T) {
	controller := gomock.NewController(t)
	defer controller.Finish()

	localUser := &core.User{Login: "octocat"}
	remoteUser := &core.User{Login: "octocat", Created: 0}
	users := mock.NewMockUserService(controller)
	users.EXPECT().Find(gomock.Any(), gomock.Any(), gomock.Any()).Return(remoteUser, nil)

	admission := Nobot(users, time.Minute)
	err := admission.Admit(noContext, localUser)
	if err != nil {
		t.Error(err)
	}
}

func TestNobot_RemoteError(t *testing.T) {
	controller := gomock.NewController(t)
	defer controller.Finish()

	want := errors.New("")
	users := mock.NewMockUserService(controller)
	users.EXPECT().Find(gomock.Any(), gomock.Any(), gomock.Any()).Return(nil, want)

	admission := Nobot(users, time.Minute)
	got := admission.Admit(noContext, new(core.User))
	if got != want {
		t.Errorf("Expect error from source control management system returned")
	}
}

func TestNobot_SkipCheck(t *testing.T) {
	controller := gomock.NewController(t)
	defer controller.Finish()

	dummyUser := &core.User{
		Login: "octocat",
	}

	admission := Nobot(nil, 0)
	err := admission.Admit(noContext, dummyUser)
	if err != nil {
		t.Error(err)
	}
}
