// Copyright 2019 Drone.IO Inc. All rights reserved.
// Use of this source code is governed by the Drone Non-Commercial License
// that can be found in the LICENSE file.

package user

import (
	"encoding/json"
	"net/http/httptest"
	"testing"

	"github.com/drone/drone/handler/api/errors"
	"github.com/drone/drone/handler/api/request"
	"github.com/drone/drone/mock"
	"github.com/drone/drone/core"

	"github.com/golang/mock/gomock"
	"github.com/google/go-cmp/cmp"
	"github.com/google/go-cmp/cmp/cmpopts"
)

func TestToken(t *testing.T) {
	controller := gomock.NewController(t)
	defer controller.Finish()

	mockUser := &core.User{
		ID:    1,
		Login: "octocat",
		Hash:  "MjAxOC0wOC0xMVQxNTo1ODowN1o",
	}

	w := httptest.NewRecorder()
	r := httptest.NewRequest("POST", "/", nil)
	r = r.WithContext(
		request.WithUser(r.Context(), mockUser),
	)

	HandleToken(nil)(w, r)
	if got, want := w.Code, 200; want != got {
		t.Errorf("Want response code %d, got %d", want, got)
	}

	got, want := &userWithToken{}, mockUser
	json.NewDecoder(w.Body).Decode(got)

	if got, want := got.Token, want.Hash; got != want {
		t.Errorf("Expect user secret returned")
	}
}

// the purpose of this unit test is to verify that the token
// is refreshed if the user ?refresh=true query parameter is
// included in the http request.
func TestTokenRotate(t *testing.T) {
	controller := gomock.NewController(t)
	defer controller.Finish()

	mockUser := &core.User{
		ID:    1,
		Login: "octocat",
		Hash:  "MjAxOC0wOC0xMVQxNTo1ODowN1o",
	}

	w := httptest.NewRecorder()
	r := httptest.NewRequest("POST", "/?rotate=true", nil)
	r = r.WithContext(
		request.WithUser(r.Context(), mockUser),
	)

	users := mock.NewMockUserStore(controller)
	users.EXPECT().Update(gomock.Any(), gomock.Any()).Return(nil)

	HandleToken(users)(w, r)
	if got, want := w.Code, 200; want != got {
		t.Errorf("Want response code %d, got %d", want, got)
	}

	got, want := &userWithToken{}, mockUser
	json.NewDecoder(w.Body).Decode(got)

	ignore := cmpopts.IgnoreFields(core.User{}, "Hash")
	if diff := cmp.Diff(got.User, want, ignore); len(diff) != 0 {
		t.Errorf(diff)
	}
	if got.Token == "" {
		t.Errorf("Expect user token returned")
	}
	if got, want := got.Token, "MjAxOC0wOC0xMVQxNTo1ODowN1o"; got == want {
		t.Errorf("Expect user hash updated")
	}
}

// the purpose of this unit test is to verify that an error
// updating the database will result in an internal server
// error returned to the client.
func TestToken_UpdateError(t *testing.T) {
	controller := gomock.NewController(t)
	defer controller.Finish()

	mockUser := &core.User{
		ID:    1,
		Login: "octocat",
	}

	w := httptest.NewRecorder()
	r := httptest.NewRequest("POST", "/?rotate=true", nil)
	r = r.WithContext(
		request.WithUser(r.Context(), mockUser),
	)

	users := mock.NewMockUserStore(controller)
	users.EXPECT().Update(gomock.Any(), gomock.Any()).Return(errors.ErrNotFound)

	HandleToken(users)(w, r)
	if got, want := w.Code, 500; want != got {
		t.Errorf("Want response code %d, got %d", want, got)
	}

	got, want := new(errors.Error), errors.ErrNotFound
	json.NewDecoder(w.Body).Decode(got)
	if diff := cmp.Diff(got, want); len(diff) != 0 {
		t.Errorf(diff)
	}
}
