// Copyright 2019 Drone.IO Inc. All rights reserved.
// Use of this source code is governed by the Drone Non-Commercial License
// that can be found in the LICENSE file.

package repos

import (
	"encoding/json"
	"io"
	"net/http"
	"net/http/httptest"
	"testing"

	"github.com/drone/drone/core"
	"github.com/drone/drone/handler/api/errors"
	"github.com/drone/drone/mock"

	"github.com/go-chi/chi"
	"github.com/golang/mock/gomock"
	"github.com/google/go-cmp/cmp"
)

func TestDisable(t *testing.T) {
	controller := gomock.NewController(t)
	defer controller.Finish()

	repo := &core.Repository{
		ID:        1,
		Namespace: "octocat",
		Name:      "hello-world",
		Slug:      "octocat/hello-world",
		Active:    true,
	}

	repos := mock.NewMockRepositoryStore(controller)
	repos.EXPECT().FindName(gomock.Any(), gomock.Any(), repo.Name).Return(repo, nil)
	repos.EXPECT().Update(gomock.Any(), repo).Return(nil)

	// a failed webhook should result in a warning message in the
	// logs, but should not cause the endpoint to error.
	webhook := mock.NewMockWebhookSender(controller)
	webhook.EXPECT().Send(gomock.Any(), gomock.Any()).Return(io.EOF)

	w := httptest.NewRecorder()
	r := httptest.NewRequest("DELETE", "/api/repos/octocat/hello-world", nil)

	router := chi.NewRouter()
	router.Delete("/api/repos/{owner}/{name}", HandleDisable(repos, webhook))
	router.ServeHTTP(w, r)

	if got, want := w.Code, 200; want != got {
		t.Errorf("Want response code %d, got %d", want, got)
	}

	if got, want := repo.Active, false; got != want {
		t.Errorf("Want repository activate %v, got %v", want, got)
	}

	got, want := new(core.Repository), repo
	json.NewDecoder(w.Body).Decode(got)
	if diff := cmp.Diff(got, want); len(diff) != 0 {
		t.Errorf(diff)
	}
}

func TestDisable_NotFound(t *testing.T) {
	controller := gomock.NewController(t)
	defer controller.Finish()

	repos := mock.NewMockRepositoryStore(controller)
	repos.EXPECT().FindName(gomock.Any(), "octocat", "hello-world").Return(nil, errors.ErrNotFound)

	w := httptest.NewRecorder()
	r := httptest.NewRequest("DELETE", "/api/repos/octocat/hello-world", nil)

	router := chi.NewRouter()
	router.Delete("/api/repos/{owner}/{name}", HandleDisable(repos, nil))
	router.ServeHTTP(w, r)

	if got, want := w.Code, 404; want != got {
		t.Errorf("Want response code %d, got %d", want, got)
	}

	got, want := new(errors.Error), errors.ErrNotFound
	json.NewDecoder(w.Body).Decode(got)
	if diff := cmp.Diff(got, want); len(diff) != 0 {
		t.Errorf(diff)
	}
}

func TestDisable_InternalError(t *testing.T) {
	controller := gomock.NewController(t)
	defer controller.Finish()

	repo := &core.Repository{
		ID:        1,
		Namespace: "octocat",
		Name:      "hello-world",
		Slug:      "octocat/hello-world",
		Active:    false,
	}

	repos := mock.NewMockRepositoryStore(controller)
	repos.EXPECT().FindName(gomock.Any(), gomock.Any(), repo.Name).Return(repo, nil)
	repos.EXPECT().Update(gomock.Any(), repo).Return(errors.ErrNotFound)

	w := httptest.NewRecorder()
	r := httptest.NewRequest("DELETE", "/api/repos/octocat/hello-world", nil)

	router := chi.NewRouter()
	router.Delete("/api/repos/{owner}/{name}", HandleDisable(repos, nil))
	router.ServeHTTP(w, r)

	if got, want := w.Code, http.StatusInternalServerError; want != got {
		t.Errorf("Want response code %d, got %d", want, got)
	}

	got, want := new(errors.Error), errors.ErrNotFound
	json.NewDecoder(w.Body).Decode(got)
	if diff := cmp.Diff(got, want); len(diff) != 0 {
		t.Errorf(diff)
	}
}

func TestDelete(t *testing.T) {
	controller := gomock.NewController(t)
	defer controller.Finish()

	repo := &core.Repository{
		ID:        1,
		Namespace: "octocat",
		Name:      "hello-world",
		Slug:      "octocat/hello-world",
		Active:    true,
	}

	repos := mock.NewMockRepositoryStore(controller)
	repos.EXPECT().FindName(gomock.Any(), gomock.Any(), repo.Name).Return(repo, nil)
	repos.EXPECT().Update(gomock.Any(), repo).Return(nil)
	repos.EXPECT().Delete(gomock.Any(), repo).Return(nil)

	// a failed webhook should result in a warning message in the
	// logs, but should not cause the endpoint to error.
	webhook := mock.NewMockWebhookSender(controller)
	webhook.EXPECT().Send(gomock.Any(), gomock.Any()).Return(io.EOF)

	w := httptest.NewRecorder()
	r := httptest.NewRequest("DELETE", "/api/repos/octocat/hello-world?remove=true", nil)

	router := chi.NewRouter()
	router.Delete("/api/repos/{owner}/{name}", HandleDisable(repos, webhook))
	router.ServeHTTP(w, r)

	if got, want := w.Code, 200; want != got {
		t.Errorf("Want response code %d, got %d", want, got)
	}

	got, want := new(core.Repository), repo
	json.NewDecoder(w.Body).Decode(got)
	if diff := cmp.Diff(got, want); len(diff) != 0 {
		t.Errorf(diff)
	}
}
