/* -*- Mode: C; tab-width: 2; indent-tabs-mode: nil; c-basic-offset: 2 -*- */
/*
 * cpvts.c
 *
 * copy a complete title set from a DVD to disc
 * written by Christian Vogelgsang <chris@lallafa.de>
 * under the GNU Public License V2
 *
 * $Id: cpvts.c,v 1.2 2003/04/13 10:31:28 cnvogelg Exp $
 */

#include <assert.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <math.h>

/* libdvdread stuff */
#include <dvdread/dvd_reader.h>
#include <dvdread/ifo_types.h>
#include <dvdread/ifo_read.h>

/* ----- copy_blocks ----------------------------------------------------------
   copy raw 2048 byte data blocks from a DVD file to a disc file.
   uses different methods if the file is a VOB or an IFO/BUP
   param:
     file    = dvd input file
     out     = file handle of target file
     num_blk = number of blocks to copy (must fit into buf!!)
     buf     = memory for copying
     pos     = current start block
     vobs    = 1=VOB 0=IFO/BUP file
   returns: 0=read/write error, 1=ok
*/

int copy_blocks(dvd_file_t *file,FILE *out,ssize_t num_blk,char *buf,
		ssize_t pos,int vobs,int verbose)
{
  ssize_t bytes = num_blk * 2048;
  ssize_t result;

  if(vobs) {
    result = DVDReadBlocks(file,pos,num_blk,buf);
    if(result!=num_blk) {
      fprintf(stderr,"Read error at block %u from DVDReadBlocks!\n",pos);
      return 0;
    }
  } else {
    result = DVDReadBytes(file,buf,bytes);
    if(result!=bytes) {
      fprintf(stderr,"Read error at block %u from DVDReadBytes!\n",pos);
      return 0;
    }
  }

  if(fwrite(buf,bytes,1,out)!=1) {
    fprintf(stderr,"Write error at block %u (size: %u)\n",pos,num_blk);
    return 0;
  }

  /* print MB news */
  if(verbose) {
    printf("%u MB\r",pos>>9);
    fflush(stdout);
  }
  return 1;
}

/* ----- copy_part ------------------------------------------------------------
   copy a part of the DVD input file into a single output file.
   will split the total count of blocks into parts that fit into the
   copy buffer.
   param:
     file    = DVD input file
     name    = file path of output file
     num_blk = total blocks to copy
     buf_blk = how many blocks fit into the buffer?
     buf     = memory for copy buffer
     vob     = 1=VOB 0=IFO/BUP
     pos     = start pos. will be incremented on copy!
     verbose = 0=no comment >0 be verbose
   returns: 0=read/write error, 1=ok
*/

int copy_part(dvd_file_t *file,const char *name,ssize_t num_blk,
	      ssize_t buf_blk,char *buf,int vobs,ssize_t *pos,
	      int verbose)
{
  ssize_t parts = num_blk / buf_blk;
  ssize_t remainder = num_blk % buf_blk;
  ssize_t i;
  FILE *fh;

  if(verbose)
    printf("    writing '%s' (%u blocks, %u bytes, %u MB)\n",
	   name,num_blk,num_blk*2048,(num_blk+(1<<9)-1)>>9);

  if (name[0] != '-') 
    fh = fopen(name,"w");
  else
    fh = stdout;
  if(fh==NULL) {
    fprintf(stderr,"Can't open file '%s'\n",name);
    return 0;
  }

  /* copy full buffer blocks */
  if(verbose>1)
    printf("  start pos: %10u (%u parts)\n",*pos,parts);
  for(i=0;i<parts;i++) {
    if(verbose>1)
      printf(" part #%4d: %10u\n",i,*pos);
    if(!copy_blocks(file,fh,buf_blk,buf,*pos,vobs,verbose))
      return 0;
    *pos += buf_blk;
  }
  /* copy last remainder (smaller than full buffer */
  if(remainder>0) {
    if(verbose>1)
      printf("  remainder: %10u (%u)\n",*pos,remainder);
    if(!copy_blocks(file,fh,remainder,buf,*pos,vobs,verbose))
      return 0;
    *pos += remainder;
  }
  if(verbose>1)
    printf("   last pos: %10u (%u bytes)\n",*pos,*pos * 2048);

  if (name[0] != '-') 
    fclose(fh);

  return 1;
}

/* ----- copy_file ------------------------------------------------------------
   copy a DVD file.
   can split up the output into multiple files (with max size=max_blk)
   param:
     dvd     = handle to DVD access
     tgt_pat = target file path pattern (we will fill in parameters!)
     domain  = select the file of the title set (IFO,BUP,Menu VOB,Title VOB)
     title_set = the title set you want to clone (0=VIDEO_TS,1...n)
     max_blk = max blocks for each (splitted) output file (0=no split)
     buf_blk = size of copy buffer in blocks
     buf     = pointer to copy buffer
   returns: 0=read/write error, 1=ok
*/

int copy_file(dvd_reader_t *dvd,char *tgt_pat,dvd_read_domain_t domain,
	      int title_set,ssize_t max_blk,ssize_t buf_blk,char *buf,
	      int verbose)
{
  dvd_file_t *file;
  ssize_t total_blk;
  const char *ext,*txt;
  int id = 0;
  int vobs = 0;
  int pos_ext = strlen(tgt_pat)-3;
  int num_ext = pos_ext-5;
  ssize_t parts,remainder,i,pos,tparts;

  /* build current target name string */
  switch(domain) {
  case DVD_READ_INFO_FILE: 
    ext="IFO"; txt="IFO"; break;
  case DVD_READ_INFO_BACKUP_FILE: 
    ext="BUP"; txt="BUP"; break;
  case DVD_READ_MENU_VOBS: 
    ext="VOB"; txt="MenuVOB"; vobs=1; break;
  case DVD_READ_TITLE_VOBS: 
  default:
    ext="VOB"; txt="TitleVOB"; id=1; vobs=1; break;
  }
  strcpy(tgt_pat+pos_ext,ext);
  if(title_set>0) {
    sprintf(tgt_pat+num_ext,"%02d",title_set);
    tgt_pat[num_ext+2] = '_';    
    num_ext+=3;
    tgt_pat[num_ext] = '0'+id;
  }

  /* open DVD file */
  file = DVDOpenFile(dvd,title_set,domain);
  if(file==NULL) {
    fprintf(stderr,"Can't open title set %d\n",title_set);
    return 0;
  }
  
  /* get size of file in blocks */
  total_blk = DVDFileSize(file);

  /* split up source file into target chunks */
  if(max_blk>0) {
    parts     = total_blk / max_blk;
    remainder = total_blk % max_blk;
  } else {
    parts     = 0;
    remainder = total_blk;
  }
  tparts = parts + ((remainder>0)?1:0);
  if(tparts>9) {
    fprintf(stderr,"TOO many parts occured: %u (max 9 allowed)\n"
	   "(increase split size with -s)\n",tparts);
    return 0;
  }

  if(verbose) {
    printf("  %s: %u blocks, %u bytes, %u MB -> %u target file(s)\n",
	   txt,total_blk,total_blk*2048,(total_blk+(1<<9)-1)>>9,tparts);
  }

  /* main block copy */
  pos = 0;
  for(i=0;i<parts;i++) {
    if(!copy_part(file,tgt_pat,max_blk,buf_blk,buf,vobs,&pos,verbose))
      return 0;
    if(title_set>0)
      tgt_pat[num_ext]++;
  }
  if(remainder>0) {
    if(!copy_part(file,tgt_pat,remainder,buf_blk,buf,vobs,&pos,verbose))
      return 0;
  }

  return 1;
}

/* ----- copy_title_set -------------------------------------------------------
   this is the real worker :)
   copy all files of a title set from DVD to an arbitrary disc path
   params:
     dvd         = dvd device handle
     target_dir  = directory where to copy the files
     title_set   = 0=VIDEO_TS or 1...n (the title set you want)
     max_mb      = split output at "max_mb" Mbytes (0=nosplit)
     buf_mb      = size of copy buffer in Mbytes
     copy_menu   = copy menu files? (IFO/BUP/VTS_XX_0.VOB)
     verbose     = 0=no output 1=info 2=debug
   returns:
     0=error 1=ok
*/

int copy_title_set(dvd_reader_t *dvd,const char *target_dir,
		   int title_set,ssize_t max_mb,ssize_t buf_mb,
		   int copy_menu, int verbose)
{
  ssize_t max_blk = max_mb * 512;
  ssize_t buf_blk = buf_mb * 512;
  char *buf,*name;
  int tlen,ok = 0;

  /* alloc copy buffer */
  buf = malloc(buf_mb*1024*1024);
  if(buf==NULL) {
    fprintf(stderr,"No memory!\n");
    return 0;
  }

  /* build name pattern */
  tlen = strlen(target_dir);
  name = malloc(tlen+32);
  if(name==0) {
    free(buf);
    fprintf(stderr,"No memory!\n");
    return 0;
  }
  strcpy(name,target_dir);
  if(name[tlen-1]!='/') {
    name[tlen] = '/';
    tlen++;
  }
  if(title_set==0)
    strcpy(name+tlen,"VIDEO_TS.xxx");
  else
    strcpy(name+tlen,"VTS_xx_x.xxx");


  /* copy files */
  if (copy_menu) {
    ok = copy_file(dvd,name,DVD_READ_INFO_FILE,title_set,
                   0,buf_blk,buf,verbose) 
      && copy_file(dvd,name,DVD_READ_INFO_BACKUP_FILE,title_set,
                   0,buf_blk,buf,verbose) 
      && copy_file(dvd,name,DVD_READ_MENU_VOBS,title_set,
                   0,buf_blk,buf,verbose)
      && (!title_set || /* no TitleVOB in video manager */
          copy_file(dvd,name,DVD_READ_TITLE_VOBS,title_set,
                    max_blk,buf_blk,buf,verbose));
  } else if (title_set) {
	  copy_file(dvd,name,DVD_READ_TITLE_VOBS,title_set,
              max_blk,buf_blk,buf,verbose);
  }

  /* shutdown */
  free(buf);
  free(name);

  /* all ok */
  return ok;
}

/* ----- get_total_sets -------------------------------------------------------
   return the number of total title sets on the DVD
   param:
     dvd = DVD device handle
   returns:
     0=error  otherwise 1...n
*/

int get_total_sets(dvd_reader_t *dvd)
{
  ifo_handle_t *vmg;
  int num;

  /* acces video manager */
  vmg = ifoOpen(dvd,0);
  if(vmg==NULL) {
    fprintf(stderr,"Can't open VIDEO_TS.IFO\n" );
    return 0;
  }
 
  /* acces total titles */
  num = vmg->vmgi_mat->vmg_nr_of_title_sets;

  /* cleanup */
  ifoClose(vmg);
  return num;
}

/* ---------- main --------------------------------------------------------- */

/* usage - print usage information and exit program */
void usage()
{
  fprintf(stderr,
	  "cpvts - copy a complete title set from DVD to disc\n"
	  " written by Christian Vogelgsang <chris@lallafa.de>\n"
	  " $Revision: 1.2 $\n"
	  "\nUsage: cpvts [options] <dir>\n\n"
	  " -d <dev>   input DVD device or image dir (default: /dev/dvd)\n"
	  " -s <MB>    split output in <MB> chunks   (default: 1024 MB)\n"
	  " -b <MB>    use MB copy buffer            (default: 4 MB)\n"
	  "\nSelect Title Set:\n"
	  " -t <num>   select title set number       (default: 1)\n"
	  " -a         copy all title sets           (i.e. whole DVD)\n"
	  " -n         do not copy video manager     (VIDEO_TS)\n"
	  " -i         do not copy IFO/BUP/VTS_XX_0.VOB files\n"
	  "\nVerbose Options:\n"
	  " -q         be quiet\n"
	  " -v         be more verbose\n"
	  );
  exit(1);
}

int main( int argc, char **argv )
{
  dvd_reader_t *dvd;

  /* arguments */
  char *dvd_path = "/dev/dvd";
  int title_set = 1;
  int copy_all = 0;
  int split_mb = 1024;
  int copy_mb = 4;
  int copy_vmg = 1;
  int copy_menu = 1;
  int verbose = 1;
  char *target;
  int arg;

  /* check arguments */
  while((arg = getopt(argc,argv,"d:t:anis:b:qv"))!=-1) {
    switch(arg) {
    /* d - dvd_path */
    case 'd':
    dvd_path = optarg;
    break;
    /* t - title_set */
    case 't':
      title_set = atoi(optarg);
      break;
      /* a - copy all */
    case 'a':
      copy_all = 1;
      break;
      /* n - no video manager */
    case 'n':
      copy_vmg = 0;
      break;
      /* i - no IFO/BUP */
    case 'i':
      copy_menu = 0;
      break;
      /* s = split size */
    case 's':
      split_mb = atoi(optarg);
      break;
      /* b = copy buffer size */
    case 'b':
      copy_mb = atoi(optarg);
      break;
      /* q = quiet */
    case 'q':
      verbose = 0;
      break;
    case 'v':
      verbose = 2;
      break;
      /* ??? */
    default:
      usage();
      break;
    }
    }
  /* last arg is target dir */
  if(optind!=argc-1)
    usage();
  target = argv[optind];

  /* ----- now the fun begins ----- */

  /* welcome msg */
  if(verbose>1) {
    puts("cpvts options:");
    printf("  dvd device : %s\n"
	   "  target dir : %s\n"
	   "  copy buffer: %d MB\n"
	   "  split at   : %d MB\n",
	   dvd_path,target,copy_mb,split_mb);
  }

  /* access DVD */
  dvd = DVDOpen(dvd_path);
  if(!dvd) {
    fprintf(stderr, "Couldn't open DVD device/image: %s\n",dvd_path);
    return 1;
  }

  /* copy video manager VIDEO_TS */
  if(copy_vmg) {
    if(verbose)
      puts("copy video manager VIDEO_TS\n");
    if(!copy_title_set(dvd,target,0,0,copy_mb,1,verbose))
      return 1;
  }

  /* copy all title sets */
  if(copy_all) {
    int i;
    int total = get_total_sets(dvd);
    if(total==0)
      return 1;
    if(verbose)
      printf("copy ALL title sets (1..%d)\n",total);
    for(i=1;i<total;i++) {
      if(!copy_title_set(dvd,target,i,split_mb,copy_mb,copy_menu,verbose))
	return 1;
    }
  } 
  /* copy one title set */
  else {
    if(verbose)
      printf("copy title set %d\n",title_set);
    /* copy a single title set */
    if(!copy_title_set(dvd,target,title_set,split_mb,copy_mb,copy_menu,verbose))
      return 1;
  }

  /* ready! */
  if(verbose)
    printf("all OK, ready!\n");

  DVDClose(dvd);
  return 0;
}

