/*
 * Copyright 2021-2024 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.opentest4j.reporting.events.java;

import org.apiguardian.api.API;
import org.opentest4j.reporting.events.api.ChildElement;
import org.opentest4j.reporting.events.api.Context;
import org.opentest4j.reporting.events.core.Sources;
import org.opentest4j.reporting.schema.Namespace;
import org.opentest4j.reporting.schema.QualifiedName;

import static org.apiguardian.api.API.Status.EXPERIMENTAL;

/**
 * The {@code methodSource} element of the Java namespace.
 *
 * @since 0.1.0
 */
@API(status = EXPERIMENTAL, since = "0.1.0")
public class MethodSource extends ChildElement<Sources, MethodSource> {

	private static final QualifiedName ELEMENT = QualifiedName.of(Namespace.REPORTING_JAVA, "methodSource");
	private static final QualifiedName CLASS_NAME = QualifiedName.of(Namespace.REPORTING_JAVA, "className");
	private static final QualifiedName METHOD_NAME = QualifiedName.of(Namespace.REPORTING_JAVA, "methodName");
	private static final QualifiedName METHOD_PARAMETER_TYPES = QualifiedName.of(Namespace.REPORTING_JAVA,
		"methodParameterTypes");

	MethodSource(Context context) {
		super(context, MethodSource.ELEMENT);
	}

	MethodSource withClassName(String className) {
		withAttribute(CLASS_NAME, className);
		return this;
	}

	MethodSource withMethodName(String methodName) {
		withAttribute(METHOD_NAME, methodName);
		return this;
	}

	/**
	 * Set the {@code methodParameterTypes} attribute of this element.
	 *
	 * @param methodParameterTypes the method parameter types
	 * @return this element
	 */
	public MethodSource withMethodParameterTypes(String methodParameterTypes) {
		withAttribute(METHOD_PARAMETER_TYPES, methodParameterTypes);
		return this;
	}
}
