/*
 * Copyright 2021-2024 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.opentest4j.reporting.events.api;

import org.apiguardian.api.API;
import org.opentest4j.reporting.schema.Namespace;

import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import static org.apiguardian.api.API.Status.EXPERIMENTAL;

/**
 * Registry of namespaces for writing a document.
 *
 * @see Namespace
 * @see DocumentWriter
 * @since 0.1.0
 */
@API(status = EXPERIMENTAL, since = "0.1.0")
public class NamespaceRegistry {

	static final String XSI_PREFIX = "xsi";

	/**
	 * Create a new builder with the supplied default namespace.
	 *
	 * @param defaultNamespace the default namespace
	 * @return a new builder
	 */
	public static Builder builder(Namespace defaultNamespace) {
		return new Builder(defaultNamespace, null);
	}

	/**
	 * Create a new builder with the supplied default namespace and schema location.
	 *
	 * @param defaultNamespace the default namespace
	 * @param schemaLocation   the schema location of the default namespace
	 * @return a new builder
	 */
	public static Builder builder(Namespace defaultNamespace, String schemaLocation) {
		return new Builder(defaultNamespace, schemaLocation);
	}

	private final Namespace defaultNamespace;
	private final Map<Namespace, String> schemaLocations;
	private final Map<Namespace, String> additionalNamespaces;

	private NamespaceRegistry(Namespace defaultNamespace, Map<Namespace, String> schemaLocations,
			Map<Namespace, String> additionalNamespaces) {
		this.defaultNamespace = defaultNamespace;
		this.schemaLocations = Collections.unmodifiableMap(new LinkedHashMap<>(schemaLocations));
		this.additionalNamespaces = Collections.unmodifiableMap(new LinkedHashMap<>(additionalNamespaces));
	}

	/**
	 * Get the default namespace of this registry.
	 *
	 * @return the default namespace
	 */
	public Namespace getDefaultNamespace() {
		return defaultNamespace;
	}

	/**
	 * Get additional namespaces registered with this registry.
	 *
	 * @return additional namespaces
	 */
	public Set<Namespace> getAdditionalNamespaces() {
		return additionalNamespaces.keySet();
	}

	/**
	 * Get the prefix for the supplied namespace registered with this registry.
	 *
	 * @param namespace the namespace to look up the prefix for
	 * @return the prefix for the namespace or empty if the namespace is not as an additional namespace
	 */
	public Optional<String> getPrefix(Namespace namespace) {
		return Optional.ofNullable(additionalNamespaces.get(namespace));
	}

	/**
	 * Get all schema locations registered with this registry.
	 *
	 * @return schema locations
	 */
	public Map<Namespace, String> getSchemaLocations() {
		return schemaLocations;
	}

	/**
	 * Check whether a namespace with the supplied URI is registered with this registry.
	 *
	 * @param uri the namespace URI to look up
	 * @return whether the namespace is registered
	 */
	public boolean containsUri(String uri) {
		return defaultNamespace.getUri().equals(uri) || additionalNamespaces.containsKey(Namespace.of(uri));
	}

	/**
	 * Builder for {@link NamespaceRegistry}.
	 */
	public static class Builder {

		private final Namespace defaultNamespace;
		private final Map<Namespace, String> schemaLocations = new LinkedHashMap<>();
		private final Map<Namespace, String> additionalNamespaces = new LinkedHashMap<>();

		private Builder(Namespace defaultNamespace, String schemaLocation) {
			this.defaultNamespace = defaultNamespace;
			if (schemaLocation != null) {
				schemaLocations.put(defaultNamespace, schemaLocation);
			}
		}

		/**
		 * Add an additional namespace to this registry.
		 *
		 * @param prefix    the prefix for the namespace
		 * @param namespace the namespace
		 * @return this builder
		 */
		public Builder add(String prefix, Namespace namespace) {
			return add(prefix, namespace, null);
		}

		/**
		 * Add an additional namespace to this registry.
		 *
		 * @param prefix         the prefix for the namespace
		 * @param namespace      the namespace
		 * @param schemaLocation the schema location of the namespace
		 * @return this builder
		 */
		public Builder add(String prefix, Namespace namespace, String schemaLocation) {
			if (additionalNamespaces.containsKey(namespace) || defaultNamespace.equals(namespace)) {
				throw new IllegalStateException("Namespace has already been added previously: " + namespace);
			}
			additionalNamespaces.put(namespace, prefix);
			if (schemaLocation != null) {
				schemaLocations.put(namespace, schemaLocation);
			}
			return this;
		}

		/**
		 * Build the {@link NamespaceRegistry}.
		 *
		 * @return the namespace registry
		 */
		public NamespaceRegistry build() {
			if (!schemaLocations.isEmpty()) {
				add(XSI_PREFIX, Namespace.XML_SCHEMA_INSTANCE);
			}
			return new NamespaceRegistry(defaultNamespace, schemaLocations, additionalNamespaces);
		}

	}

}
