/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "image.h"
#include "shared.h"

/*------------------------------------------------------------------------*/

/*  NOAA_Image_Line()
 *
 *  Fills one line of one channel's image by
 *  reading samples from buffer and processing
 */

  void
NOAA_Image_Line( unsigned char *image_line, int *max_gray_val )
{
  /* An amplitude sample of the 2.4kHz sub-carrier */
  short sample_a;

  int
    pixel_val, /* Un-normalized value of pixel */
    line_idx,  /* Index to image line buffer   */
    idx, tmp;  /* Index for loops etc and temp */


  /* Process samples to make one line of image */
  for( line_idx = 0; line_idx < NOAA_IMAGE_WIDTH; line_idx++ )
  {
    pixel_val = 0;

    /*** Summate NOAA_PIXEL_SAMPLES carrier
     * samples for each pixel value ***/
    for( idx = 0; idx < NOAA_PIXEL_SAMPLES; idx++ )
    {
      /* Read a sample of the 2.4kHz sub_carrier */
      sample_a = line_buffer[line_index++];

      /* This is the instantaneous magnitude of the 2.4 Khz */
      /* subcarrier, calculated as the scalar magnitude of */
      /* two consecutive signal samples at 90 deg phase angle */
      Carrier_Amplitude( sample_a, &tmp );
      pixel_val += tmp;

    } /* End of for( idx = 0; idx < NOAA_PIXEL_SAMPLES; idx++ ) */

    /* Normalize pixel values to below 255 */
    tmp = (pixel_val * NOAA_SCALE_FACTOR) / MAX_SUBCARRIER_AMPL;
    if( tmp > MAX_PIXEL_VAL ) tmp = MAX_PIXEL_VAL;
    image_line[line_idx] = (unsigned char)tmp;

    /* Record max gray value of line */
    if( image_line[line_idx] > *max_gray_val )
      *max_gray_val = image_line[line_idx];

  } /* End of for( line_idx = 0; line_idx < NOAA_IMAGE_WIDTH; line_idx++ ) */

} /* End of NOAA_Image_Line() */

/*------------------------------------------------------------------------*/

/*  Meteor_Image_Line()
 *
 *  Fills one line of a Meteor APT image by
 *  reading samples from buffer and processing
 */

  void
Meteor_Image_Line( unsigned char *image_line, int *max_gray_val )
{
  /* An amplitude sample of the 2.4kHz sub-carrier */
  short sample_a;

  int
    pixel_val, /* Un-normalized value of pixel */
    line_idx,  /* Index to image line buffer   */
    idx, tmp;  /* Index for loops etc and temp */


  /* Process samples to make one line of image */
  for( line_idx = 0; line_idx < METEOR_IMAGE_WIDTH; line_idx++ )
  {
    pixel_val = 0;

    /*** Summate METEOR_NSAMPLES carrier samples for each pixel value ***/
    for( idx = 0; idx < METEOR_PIXEL_SAMPLES; idx++ )
    {
      /* Read a sample of the 2.4kHz sub_carrier*/
      sample_a = line_buffer[line_index++];

      /* This is the instantaneous magnitude of the 2.4 Khz  */
      /* subcarrier, calculated as the scalar magnitude of   */
      /* two consecutive signal samples at 90 deg phase angle */
      Carrier_Amplitude( sample_a, &tmp );
      pixel_val += tmp;

    } /* End of for( idx = 0; idx < METEOR_PIXEL_SAMPLES; idx++ ) */

    /* Normalize pixel values to below 255 */
    tmp = (pixel_val * METEOR_SCALE_FACTOR) / MAX_SUBCARRIER_AMPL;
    if( tmp > MAX_PIXEL_VAL ) tmp = MAX_PIXEL_VAL;
    image_line[line_idx] = (unsigned char)tmp;

    /* Record max gray value of line */
    if( image_line[line_idx] > *max_gray_val )
      *max_gray_val = image_line[line_idx];

  } /* End of for( line_idx = 0; line_idx < image_width; line_idx++ ) */

} /* End of Meteor_Image_Line() */

/*------------------------------------------------------------------------*/

/*  Dsp_Process_Image()
 *
 *  Processes APT images direct from DSP signal samples.
 */

  gboolean
Dsp_Process_Image( gpointer data )
{
  /* Number of APT image lines processed */
  static int line_cnt;

  int
    carrier_val, /* Instantaneous value of 2.4Khz sub-carrier */
    carrier_max; /* Maximum value of 2.4KHz sub-carrier       */

  /* Progressbar fraction */
  double pbar_frac;

  /* Index for filling pixbuf */
  gint idx;

  /* Message buffer */
  char mesg[MESG_SIZE];

  /* Pixbuf pixel pointer */
  guchar *pixel;

  /* First idle call */
  static gboolean first_call = TRUE;


  /* Process APT images according to satellite type */
  carrier_max = 0;
  switch( sat_type )
  {
    case NOAA_15: case NOAA_18: case NOAA_19: /* Decode NOAA APT images */
      {
        /* Image file names of NOAA ch A and B images */
        static char
          A_image_file[MAX_FILE_NAME+20],
          B_image_file[MAX_FILE_NAME+20];

        /* Image file names of NOAA ch A color images */
        static char  A_color_file[MAX_FILE_NAME+20];

        static int
          A_max_gray_val,  /* Maximum gray value in channel A PGM image */
          B_max_gray_val,  /* Maximum gray value in channel B PGM image */
          A_image_height,  /* Height in lines of channel A PGM image    */
          B_image_height,  /* Height in lines of channel B PGM image    */
          A_image_size,    /* Current size in bytes of A image buffer   */
          B_image_size;    /* Current size in bytes of B image buffer   */

        /* Buffers for the output images */
        static unsigned char *A_image_buffer;
        static unsigned char *B_image_buffer;
        static unsigned char *A_color_buffer;

        /* NOAA images separator line */
        gint noaa_sep = (NOAA_IMAGE_WIDTH * n_channels) / 2;

        static FILE
          *A_image_fp, /* Channel A image file */
          *B_image_fp; /* Channel B image file */


        if( first_call )
        {
          /* Fill pixbuf with background color */
          gdk_pixbuf_fill( image_pixbuf, 0xaaaaaaff );

          /* Process file name */
          File_Name( image_file );

          /* Prepare image file names */
          size_t s = sizeof( A_image_file );
          Strlcpy( A_image_file, image_file, s );
          Strlcat( A_image_file, "-", s );
          Strlcat( A_image_file, sat_names[sat_type], s );
          Strlcat( A_image_file, "-A.pgm", s );

          s = sizeof( B_image_file );
          Strlcpy( B_image_file, image_file, s );
          Strlcat( B_image_file, "-", s );
          Strlcat( B_image_file, sat_names[sat_type], s );
          Strlcat( B_image_file, "-B.pgm", s );

          /* Prepare color image file names */
          s = sizeof( A_color_file );
          Strlcpy( A_color_file, image_file, s );
          Strlcat( A_color_file, "-", s );
          Strlcat( A_color_file, sat_names[sat_type], s );
          Strlcat( A_color_file, "-A+B-color.ppm", s );

          /* Print some information */
          snprintf( mesg, MESG_SIZE,
              _("Decoding duration: %d sec"), duration );
          Show_Message( mesg, "black" );
          Show_Message( _("Decoding images:"), "black" );
          snprintf( mesg, MESG_SIZE, "%s\n%s",
              Fname(A_image_file), Fname(B_image_file) );
          Show_Message( mesg, "black" );

          /* Clear variables before decoding samples buffer */
          A_image_height = B_image_height = 0;
          A_image_size   = B_image_size   = 0;
          A_max_gray_val = B_max_gray_val = 0;
          A_image_buffer = B_image_buffer = NULL;
          line_cnt       = 0;

          first_call = FALSE;

        } /* if( first_call ) */

        /*** Process samples to make an image ***/

        /* Fill APT samples buffer */
        if( !Fill_Samples_Buffer() )
        {
          first_call = TRUE;
          return( FALSE );
        }

        /* Bypass sync and space/markers of ch A. This
         * is done by searching this zone for carrier max */
        line_index += SYNC_SEARCH_EXTEND;
        for( idx = 0; idx < NOAA_SYNC_CHA_SPACE - SYNC_SEARCH_EXTEND; idx++ )
        {
          Carrier_Amplitude( line_buffer[line_index], &carrier_val );
          line_index++;

          /* Find maximum carrier level */
          if( carrier_val > carrier_max )
            carrier_max = carrier_val;
        }

        /* Re-allocate memory to image buffers, abort on error */
        if( !mem_realloc((void **)&A_image_buffer,
              (size_t)(A_image_size + NOAA_IMAGE_WIDTH)) )
        {
          first_call = TRUE;
          return(FALSE );
        }
        if( !mem_realloc((void **)&B_image_buffer,
              (size_t)(B_image_size + NOAA_IMAGE_WIDTH)) )
        {
          first_call = TRUE;
          return(FALSE );
        }

        /* Decode one image line */
        NOAA_Image_Line( A_image_buffer + A_image_size, &A_max_gray_val );
        A_image_height++;
        A_image_size = NOAA_IMAGE_WIDTH * A_image_height;

        /* Bypass ch A gray scale/ch B sync/ch B space/markers.
         * This is done by searching this zone for carrier max. */
        for( idx = 0; idx < NOAA_CHA_CHB_SPACE; idx++ )
        {
          Carrier_Amplitude( line_buffer[line_index], &carrier_val );
          line_index++;

          /* Find maximum carrier level */
          if( carrier_val > carrier_max )
            carrier_max = carrier_val;
        } /* for( idx = 0; idx < NOAA_CHA_CHB_SPACE; idx++ ) */

        /* Decode an image line */
        NOAA_Image_Line( B_image_buffer + B_image_size, &B_max_gray_val );
        B_image_height++;
        B_image_size = NOAA_IMAGE_WIDTH * B_image_height;

        /* Bypass ch B Telemetry. This is done by
         * searching this zone for carrier max. */
        for( idx = 0; idx < CHB_PADDING; idx++ )
        {
          Carrier_Amplitude( line_buffer[line_index], &carrier_val );
          line_index++;

          /* Find maximum carrier level */
          if( carrier_val > carrier_max )
            carrier_max = carrier_val;
        } /* for( idx = 0; idx < NOAA_CHA_CHB_SPACE; idx++ ) */

        /* Find input signal level error and adjust scale */
        pbar_frac = (gdouble)carrier_max / (gdouble)MAX_SUBCARRIER_AMPL;
        if( pbar_frac > 1.0 ) pbar_frac = 1.0;
        gtk_progress_bar_set_fraction(
            GTK_PROGRESS_BAR(level_pbar), pbar_frac );

        /* Draw images incrementally at half size */
        Halfsize_Row( A_image_buffer,
            NOAA_IMAGE_WIDTH, line_cnt, 0 );
        Halfsize_Row( B_image_buffer,
            NOAA_IMAGE_WIDTH, line_cnt, NOAA_IMAGE_WIDTH / 2 );

        /* Find the B pixel at 0,y and make it white */
        pixel = pixel_buf + (line_cnt / 2) * rowstride + noaa_sep;
        pixel[0] = 0xff;
        pixel[1] = 0xff;
        pixel[2] = 0xff;

        /* Draw image progressively to window */
        if( line_cnt & 0x01 )
          gtk_image_set_from_pixbuf( GTK_IMAGE(apt_image), image_pixbuf );

        /*** Continue if line count less than limit or action enabled ***/
        if( (++line_cnt / 2 < duration) &&
            isFlagSet(ACTION_PROCESS_DSP) )
          return( TRUE );

        /* Finished, show message and cleanup */
        SetFlag( ACTION_PENDING );
        Show_Message( _("Stopping Real Time image decoding"), "black" );
        Cleanup();

        /* Rotate images 180 degrees if enabled */
        if( isFlagSet(IMAGE_ROTATE) )
        {
          Rotate( A_image_buffer, A_image_size );
          Rotate( B_image_buffer, B_image_size );
        }

        /* Carry out histogram normalization if enabled */
        if( isFlagSet(IMAGE_NORMALIZE) )
        {
          Normalize( A_image_buffer, A_image_size );
          Normalize( B_image_buffer, B_image_size );
          A_max_gray_val = B_max_gray_val = 255;
        }

        /* Carry out pseudo-colorization if enabled */
        if( isFlagSet(IMAGE_COLORIZE) )
        {
          /* Re-allocate memory to image buffers, abort on error */
          A_color_buffer = NULL;
          if( !mem_realloc((void **)&A_color_buffer, (size_t)(3 * A_image_size)) )
          {
            first_call = TRUE;
            return(FALSE );
          }

          Colorize( A_image_buffer, B_image_buffer,
              A_image_size, A_color_buffer, NOAA_A_COLORMAP );

          /* Print some information */
          Show_Message( _("Pseudo-colorizing image:"), "black" );
          snprintf( mesg, MESG_SIZE, "%s\n", Fname(A_color_file) );
          Show_Message( mesg, "black" );

          /* Open Channel A colorized image file */
          if( !Open_File(&A_image_fp, A_color_file, "w+") )
          {
            first_call = TRUE;
            return( FALSE );
          }

          /* Write Ch-A colorized buffer to PPM file */
          if( !File_Image(A_image_fp, "P6", NOAA_IMAGE_WIDTH,
                A_image_height, A_max_gray_val,  A_color_buffer) )
          {
            first_call = TRUE;
            return( FALSE );
          }

          /* Close colorized image files */
          Show_Message( _("Closing image file:"), "black" );
          snprintf( mesg, MESG_SIZE, "%s\n", Fname(A_color_file) );
          Show_Message( mesg, "black" );

          fclose( A_image_fp );

        } /* if( isFlagSet(IMAGE_COLORIZE) ) */

        /*** Processing finished, complete image files ***/

        /* Draw images at half size in xwxapt window */
        if( isFlagClear(IMAGE_COLORIZE) )
        {
          Halfsize_Image( A_image_buffer,
              NOAA_IMAGE_WIDTH, line_cnt, 0, FALSE );
          Halfsize_Image( B_image_buffer, NOAA_IMAGE_WIDTH,
              line_cnt, NOAA_IMAGE_WIDTH / 2, FALSE );
        }
        else
        {
          Halfsize_Image( A_color_buffer,
              NOAA_IMAGE_WIDTH, line_cnt, 0, TRUE );
          Halfsize_Image( B_image_buffer, NOAA_IMAGE_WIDTH,
              line_cnt, NOAA_IMAGE_WIDTH / 2, FALSE );
        }

        /* Find the B pixel at 0,y and make it white */
        for( idx = 0; idx < line_cnt/2; idx++ )
        {
          pixel = pixel_buf + idx * rowstride + noaa_sep;
          pixel[0] = 0xff;
          pixel[1] = 0xff;
          pixel[2] = 0xff;
        }

        /* Draw image to window */
        gtk_image_set_from_pixbuf( GTK_IMAGE(apt_image), image_pixbuf );

        /* Open Channel A image file */
        if( !Open_File(&A_image_fp, A_image_file, "w+") )
        {
          first_call = TRUE;
          return( FALSE );
        }

        /* Open Channel B image file */
        if( !Open_File(&B_image_fp, B_image_file, "w+") )
        {
          first_call = TRUE;
          return( FALSE );
        }

        /* Write Ch-A greyscale buffer to PGM file */
        if( !File_Image(A_image_fp, "P5", NOAA_IMAGE_WIDTH,
              A_image_height, A_max_gray_val,  A_image_buffer) )
        {
          first_call = TRUE;
          return( FALSE );
        }

        /* Write Ch-B greyscale buffer to PGM file */
        if( !File_Image(B_image_fp, "P5", NOAA_IMAGE_WIDTH,
              B_image_height, B_max_gray_val,  B_image_buffer) )
        {
          first_call = TRUE;
          return( FALSE );
        }

        /* Free buffers and close image files */
        Show_Message( _("Closing image files:"), "black" );
        snprintf( mesg, MESG_SIZE, "%s\n%s",
            Fname(A_image_file), Fname(B_image_file) );
        Show_Message( mesg, "black" );

        free( A_image_buffer );
        free( B_image_buffer );
        fclose( A_image_fp );
        fclose( B_image_fp );
        if( isFlagSet(IMAGE_COLORIZE) ) free( A_color_buffer );

      } /* End of case NOAA: */
      break;

    case METEOR: /* Decode Meteor APT images */
      {
        /* Image file name of Meteor image */
        static char Met_image_file[MAX_FILE_NAME+20];
        static char Met_color_file[MAX_FILE_NAME+20];

        static int
          max_gray_val,     /* Maximum gray value in Meteor APT image */
          Met_image_height, /* Height in lines of Meteor APT image    */
          Met_image_size;   /* Current size in bytes of Meteor image  */

        /* Buffer for the output image */
        static unsigned char *Met_image_buffer;
        static unsigned char *Met_color_buffer;

        /* Meteor APT image file pointer */
        static FILE *Met_image_fp;


        if( first_call )
        {
          /* Fill pixbuf with background color */
          gdk_pixbuf_fill( image_pixbuf, 0xaaaaaaff );

          /* Process file name */
          File_Name( image_file );

          /* Prepare image file name */
          size_t s = sizeof( Met_image_file );
          Strlcpy( Met_image_file, image_file, s );
          Strlcat( Met_image_file, sat_names[sat_type], s );
          Strlcat( Met_image_file, ".pgm", s );

          s = sizeof( Met_color_file );
          Strlcpy( Met_color_file, image_file, s );
          Strlcat( Met_color_file, sat_names[sat_type], s );
          Strlcat( Met_color_file, "-color.ppm", s );

          /* Print some information */
          if( duration )
          {
            snprintf( mesg, MESG_SIZE,
                _("Decoding duration: %d sec"), duration );
            Show_Message( mesg, "black" );
          }

          Show_Message( _("Decoding image:"), "black" );
          snprintf( mesg, MESG_SIZE, "%s", Fname(Met_image_file) );
          Show_Message( mesg, "black" );

          /* Clear variables before decoding sample buffer */
          Met_image_height = 0;
          max_gray_val     = 0;
          Met_image_size   = 0;
          Met_image_buffer = NULL;
          line_cnt         = 0;

          first_call = FALSE;

        } /* if( first_call ) */

        /*** Process samples to make an image ***/

        /* Fill APT samples buffer */
        if( !Fill_Samples_Buffer() )
        {
          first_call = TRUE;
          return( FALSE );
        }

        /* Bypass sync and space/markers of image */
        /* This is done by searching this zone for carrier max. */
        for( idx = 0; idx < METEOR_SYNC_IMAGE_SPACE; idx++ )
        {
          Carrier_Amplitude( line_buffer[line_index], &carrier_val );
          line_index++;

          /* Find maximum carrier level */
          if( carrier_val > carrier_max )
            carrier_max = carrier_val;
        } /* for( idx = 0; idx < METEOR_SYNC_IMAGE_SPACE; idx++ ) */

        /* Find input signal level and adjust scale */
        pbar_frac = (gdouble)carrier_max / (gdouble)MAX_SUBCARRIER_AMPL;
        if( pbar_frac > 1.0 ) pbar_frac = 1.0;
        gtk_progress_bar_set_fraction( GTK_PROGRESS_BAR(level_pbar), pbar_frac );

        /* Re-allocate memory to image buffer, abort on error */
        if( !mem_realloc((void *)&Met_image_buffer,
              (size_t)(Met_image_size + METEOR_IMAGE_WIDTH)) )
        {
          first_call = TRUE;
          return( FALSE );
        }

        /* Decode an image line */
        Meteor_Image_Line( Met_image_buffer + Met_image_size, &max_gray_val );
        Met_image_height++;
        Met_image_size = METEOR_IMAGE_WIDTH * Met_image_height;

        /* Draw images incrementally */
        Halfsize_Row( Met_image_buffer, METEOR_IMAGE_WIDTH, line_cnt, 0 );

        /* Progressively draw image to window */
        if( line_cnt & 0x01 )
          gtk_image_set_from_pixbuf( GTK_IMAGE(apt_image), image_pixbuf );

        /*** Continue if line count less than limit or action enabled ***/
        if( (++line_cnt/2 < duration) &&
            isFlagSet(ACTION_PROCESS_DSP) )
          return( TRUE );

        /* Finished, show message and cleanup */
        SetFlag( ACTION_PENDING );
        Show_Message( _("Stopping Real Time decoding"), "black" );
        Cleanup();

        /* Rotate images 180 degrees if requested */
        if( isFlagSet(IMAGE_ROTATE) )
          Rotate( Met_image_buffer, Met_image_size );

        /* Carry out histogram normalization if not suppressed */
        if( isFlagSet(IMAGE_NORMALIZE) )
        {
          Normalize( Met_image_buffer, Met_image_size );
          max_gray_val = 255;
        }

        /* Carry out pseudo-colorization if enabled */
        if( isFlagSet(IMAGE_COLORIZE) )
        {
          /* Re-allocate memory to image buffers, abort on error */
          Met_color_buffer = NULL;
          if( !mem_realloc((void *)&Met_color_buffer, (size_t)(3 * Met_image_size)) )
          {
            first_call = TRUE;
            return(FALSE );
          }

          /* The second argument is a dummy */
          Colorize( Met_image_buffer, Met_image_buffer,
              Met_image_size, Met_color_buffer, METEOR_COLORMAP );

          /* Print some information */
          Show_Message( _("Pseudo-colorizing image:"), "black" );
          snprintf( mesg, MESG_SIZE, "%s", Fname(Met_color_file) );
          Show_Message( mesg, "black" );

          /* Open Meteor colorized image file */
          if( !Open_File(&Met_image_fp, Met_color_file, "w+") )
          {
            first_call = TRUE;
            return( FALSE );
          }

          /* Write Meteor colorized buffer to PPM file */
          if( !File_Image(Met_image_fp, "P6", METEOR_IMAGE_WIDTH,
                Met_image_height, max_gray_val,  Met_color_buffer) )
          {
            first_call = TRUE;
            return( FALSE );
          }

          /* Close colorized image files */
          Show_Message( _("Closing image file:"), "black" );
          snprintf( mesg, MESG_SIZE, "%s", Fname(Met_color_file) );
          Show_Message( mesg, "black" );

          fclose( Met_image_fp );

        } /* if( isFlagSet(IMAGE_COLORIZE) ) */

        /*** Processing finished, complete image files ***/

        /* Draw 1/2 size image to xwxapt window */
        if( isFlagClear(IMAGE_COLORIZE) )
          Halfsize_Image( Met_image_buffer,
              METEOR_IMAGE_WIDTH, line_cnt, 0, FALSE );
        else
          Halfsize_Image( Met_color_buffer,
              METEOR_IMAGE_WIDTH, line_cnt, 0, TRUE );

        /* Draw image to window */
        gtk_image_set_from_pixbuf( GTK_IMAGE(apt_image), image_pixbuf );

        /* Open Meteor image file */
        if( !Open_File(&Met_image_fp, Met_image_file, "w+") )
        {
          first_call = TRUE;
          return( FALSE );
        }

        /* Write Meteor image buffer to file */
        if( !File_Image(
              Met_image_fp, "P5",
              METEOR_IMAGE_WIDTH, Met_image_height,
              max_gray_val, Met_image_buffer) )
        {
          first_call = TRUE;
          return( FALSE );
        }

        /* Free image buffer and close image file */
        Show_Message( _("Closing image file:"), "black" );
        snprintf( mesg, MESG_SIZE, "%s", Fname(Met_image_file) );
        Show_Message( mesg, "black" );

        free( Met_image_buffer );
        if( isFlagSet(IMAGE_COLORIZE) )
          free( Met_color_buffer );
        fclose( Met_image_fp );

      } /* End of case METEOR: */

  } /* End of switch( sat_type ) */

  /* Enable re-initialization */
  first_call = TRUE;
  Cleanup();
  ClearFlag( ACTION_PENDING );

  return( FALSE );
} /* End of Dsp_Process_Image() */

/*------------------------------------------------------------------------*/

/*  File_Process_Image()
 *
 *  Processes APT images from signal samples file.
 */

  gboolean
File_Process_Image( gpointer data )
{
  /* Message buffer */
  char mesg[MESG_SIZE];

  /* File suffix to be removed from file name */
  char suffix[20];

  /* Position of suffix in file name */
  char *suffix_pos;

  /* Pixbuf pixel pointer */
  guchar *pixel;


  /* Open APT signal samples file */
  snprintf( mesg, MESG_SIZE, _("Opening samples file:\n%s"),
      Fname(samples_file) );
  Show_Message( mesg, "black" );

  if( !Open_File(&samples_fp, samples_file, "r") )
    return( FALSE );

  /* Read satellite type from file, abort on error */
  if( fread(&sat_type, 4, 1, samples_fp) < 1 )
  {
    perror( "xwxapt: fread()" );
    snprintf( mesg, MESG_SIZE,
        _("Failed to read satellite type from\n%s"),
        Fname(samples_file) );
    Show_Message( mesg, "red" );
    Error_Dialog();
    return( FALSE );
  }

  if( (sat_type != NOAA_15) &&
      (sat_type != NOAA_18) &&
      (sat_type != NOAA_19) &&
      (sat_type != METEOR) )
  {
    snprintf( mesg, MESG_SIZE,
        _("Incorrect satellite type in\n%s"), Fname(samples_file) );
    Show_Message( mesg, "red" );
    Error_Dialog();
    return( FALSE );
  }

  /* Read sync position reference */
  sync_ref = sync_refs[ sat_type ];

  /* Creat the suffix to be removed from sample file name. */
  /* This is to remove the suffix <sat_name>.bin which is  */
  /* added automatically by xwxapt to samples file name.   */
  suffix[0] = '-';
  suffix[1] = '\0';
  size_t s = sizeof(suffix);
  Strlcat( suffix, sat_names[sat_type], s );
  Strlcat( suffix, ".bin", s );

  /* Find position of suffix in file name */
  suffix_pos = strstr( samples_file, suffix );

  /* Cut out suffix if it exists */
  if( suffix_pos != NULL )
    suffix_pos[0] = '\0';

  /* Process APT images according to satellite type */
  switch( sat_type )
  {
    case NOAA_15: case NOAA_18: case NOAA_19: /* Decode NOAA APT images */
      {
        /* Image file names of NOAA ch A and B images */
        char
          A_image_file[MAX_FILE_NAME+20],
          B_image_file[MAX_FILE_NAME+20];

        /* Image file names of NOAA ch A color images */
        static char  A_color_file[MAX_FILE_NAME+20];

        int
          idx,
          line_cnt,        /* Number of APT image lines processed       */
          A_max_gray_val,  /* Maximum gray value in channel A PGM image */
          B_max_gray_val,  /* Maximum gray value in channel B PGM image */
          A_image_height,  /* Height in lines of channel A PGM image    */
          B_image_height,  /* Height in lines of channel B PGM image    */
          A_image_size,    /* Current size in bytes of A image buffer   */
          B_image_size;    /* Current size in bytes of B image buffer   */

        /* Buffers for the output images */
        unsigned char *A_image_buffer = NULL;
        unsigned char *B_image_buffer = NULL;
        unsigned char *A_color_buffer = NULL;

        /* NOAA images separator line */
        gint noaa_sep = (NOAA_IMAGE_WIDTH * n_channels) / 2;

        FILE
          *A_image_fp = NULL,   /* Channel A image file  */
          *B_image_fp = NULL;   /* Channel B image file  */

        /* Process file name for images */
        s = sizeof( image_file );
        Strlcpy( image_file, rc_data.xwxapt_dir, s );
        Strlcat( image_file, "images/", s );
        Strlcat( image_file, Fname(samples_file), s );

        /* Prepare image file names */
        s = sizeof( A_image_file );
        Strlcpy( A_image_file, image_file, s );
        Strlcat( A_image_file, "-", s );
        Strlcat( A_image_file, sat_names[sat_type], s );
        Strlcat( A_image_file, "-A.pgm", s );

        s = sizeof( B_image_file );
        Strlcpy( B_image_file, image_file, s );
        Strlcat( B_image_file, "-", s );
        Strlcat( B_image_file, sat_names[sat_type], s );
        Strlcat( B_image_file, "-B.pgm", s );

        /* Prepare color image file names */
        s = sizeof( A_color_file );
        Strlcpy( A_color_file, image_file, s );
        Strlcat( A_color_file, "-", s );
        Strlcat( A_color_file, sat_names[sat_type], s );
        Strlcat( A_color_file, "-A+B-color.ppm", s );

        /* Print some information */
        Show_Message( _("Decoding images:"), "black" );
        snprintf( mesg, MESG_SIZE, "%s\n%s",
            Fname(A_image_file), Fname(B_image_file) );
        Show_Message( mesg, "black" );

        /* Clear variables before decoding samples buffer */
        A_image_height = B_image_height = 0;
        A_image_size   = B_image_size   = 0;
        A_max_gray_val = B_max_gray_val = 0;
        A_image_buffer = B_image_buffer = NULL;
        line_cnt       = 0;

        /* Process samples from file to make an image */
        while( File_Fill_Buffer() )
        {
          /* Bypass A sync and space/markers of ch A. */
          line_index += NOAA_SYNC_CHA_SPACE;

          /* Re-allocate memory to image buffers, abort on error */
          if( !mem_realloc((void *)&A_image_buffer,
                (size_t)(A_image_size + NOAA_IMAGE_WIDTH)) )
            return(FALSE );
          if( !mem_realloc((void *)&B_image_buffer,
                (size_t)(B_image_size + NOAA_IMAGE_WIDTH)) )
            return(FALSE );

          /* Decode an image line */
          NOAA_Image_Line( A_image_buffer + A_image_size, &A_max_gray_val );
          A_image_height++;
          A_image_size = NOAA_IMAGE_WIDTH * A_image_height;

          /* Bypass ch A gray scale/ch B sync/ch B space/markers */
          line_index += NOAA_CHA_CHB_SPACE;

          /* Decode an image line */
          NOAA_Image_Line( B_image_buffer + B_image_size, &B_max_gray_val );
          B_image_height++;
          B_image_size = NOAA_IMAGE_WIDTH * B_image_height;

          line_cnt++;

        } /* End of while( File_Fill_Buffer() ) */

        /* Rotate images 180 degrees if requested */
        if( isFlagSet(IMAGE_ROTATE ) )
        {
          Rotate( A_image_buffer, A_image_size );
          Rotate( B_image_buffer, B_image_size );
        }

        /* Carry out histogram normalization if not suppressed */
        if( isFlagSet(IMAGE_NORMALIZE) )
        {
          Normalize( A_image_buffer, A_image_size );
          Normalize( B_image_buffer, B_image_size );
          A_max_gray_val = B_max_gray_val = 255;
        }

        /* Carry out pseudo-colorization if enabled */
        if( isFlagSet(IMAGE_COLORIZE) )
        {
          /* Re-allocate memory to image buffers, abort on error */
          if( !mem_realloc((void *)&A_color_buffer, (size_t)(3 * A_image_size)) )
            return(FALSE );

          Colorize( A_image_buffer, B_image_buffer,
              A_image_size, A_color_buffer, NOAA_A_COLORMAP );

          /* Print some information */
          Show_Message( _("Pseudo-colorizing images:"), "black" );
          snprintf( mesg, MESG_SIZE, "%s\n", Fname(A_color_file) );
          Show_Message( mesg, "black" );

          /* Open Channel A colorized image file */
          if( !Open_File(&A_image_fp, A_color_file, "w+") )
            return( FALSE );

          /* Write Ch-A colorized buffer to PPM file */
          if( !File_Image(A_image_fp, "P6", NOAA_IMAGE_WIDTH,
                A_image_height, A_max_gray_val,  A_color_buffer) )
          {
            fclose( A_image_fp );
            return( FALSE );
          }

          /* Free buffers and close image files */
          Show_Message( _("Closing image files:"), "black" );
          snprintf( mesg, MESG_SIZE, "%s\n", Fname(A_color_file) );
          Show_Message( mesg, "black" );

          fclose( A_image_fp );

        } /* if( isFlagSet(IMAGE_COLORIZE) ) */

        /*** Processing finished, complete image files ***/

        /* Resize window as required */
        duration = line_cnt / 2;
        Initialize_Top_Window();

        /* Draw images at half size in xwxapt window */
        if( isFlagClear(IMAGE_COLORIZE) )
        {
          Halfsize_Image( A_image_buffer, NOAA_IMAGE_WIDTH, line_cnt, 0, FALSE );
          Halfsize_Image( B_image_buffer, NOAA_IMAGE_WIDTH,
              line_cnt, NOAA_IMAGE_WIDTH / 2, FALSE );
        }
        else if( A_color_buffer != NULL )
        {
          Halfsize_Image( A_color_buffer, NOAA_IMAGE_WIDTH, line_cnt, 0, TRUE );
          Halfsize_Image( B_image_buffer, NOAA_IMAGE_WIDTH,
              line_cnt, NOAA_IMAGE_WIDTH / 2, FALSE );
        }

        /* Find the B pixel at 0,y and make it white */
        for( idx = 0; idx < line_cnt/2; idx++ )
        {
          pixel = pixel_buf + idx*rowstride + noaa_sep;
          pixel[0] = 0xff;
          pixel[1] = 0xff;
          pixel[2] = 0xff;
        }

        /* Draw finished image */
        gtk_image_set_from_pixbuf( GTK_IMAGE(apt_image), image_pixbuf );

        /* Open Channel A image file */
        if( !Open_File(&A_image_fp, A_image_file, "w+") )
          return( FALSE );

        /* Open Channel B image file */
        if( !Open_File(&B_image_fp, B_image_file, "w+") )
        {
          fclose( A_image_fp );
          return( FALSE );
        }

        /* Write Ch-A greyscale buffer to PGM file */
        if( !File_Image(A_image_fp, "P5", NOAA_IMAGE_WIDTH,
              A_image_height, A_max_gray_val,  A_image_buffer) )
        {
          fclose( A_image_fp );
          fclose( B_image_fp );
          return( FALSE );
        }

        /* Write Ch-B greyscale buffer to PGM file */
        if( !File_Image(B_image_fp, "P5", NOAA_IMAGE_WIDTH,
              B_image_height, B_max_gray_val,  B_image_buffer) )
        {
          fclose( A_image_fp );
          fclose( B_image_fp );
          return( FALSE );
        }

        /* Free image buffers and close image files */
        Show_Message( _("Closing image files:"), "black" );
        snprintf( mesg, MESG_SIZE, "%s\n%s",
            Fname(A_image_file), Fname(B_image_file) );
        Show_Message( mesg, "black" );

        free( A_image_buffer );
        free( B_image_buffer );
        fclose( A_image_fp );
        fclose( B_image_fp );
        if( isFlagSet(IMAGE_COLORIZE) ) free( A_color_buffer );

      } /* End of case NOAA : */

      break;

    case METEOR: /* Decode METEOR APT images */
      {
        /* Meteor image file name */
        char Met_image_file[MAX_FILE_NAME+20];

        int
          line_cnt,     /* Number of APT image lines processed       */
          max_gray_val, /* Maximum gray value of Meteor APT image    */
          Met_image_height, /* Height in lines of Meteor APT image   */
          Met_image_size;   /* Current size in bytes of Meteor image */

        /* Buffer for the output image */
        unsigned char *Met_image_buffer = NULL;
        unsigned char *Met_color_buffer = NULL;

        /* Meteor image file pointer */
        FILE *Met_image_fp;

        /* Process file name for image */
        s = sizeof( image_file );
        Strlcpy( image_file, rc_data.xwxapt_dir, s );
        Strlcat( image_file, "images/", s );
        Strlcat( image_file, Fname(samples_file), s );

        /* Prepare image file name */
        s = sizeof( Met_image_file );
        Strlcpy( Met_image_file, image_file, s );
        Strlcat( Met_image_file, sat_names[sat_type], s );
        Strlcat( Met_image_file, "-.pgm", s );

        /* Print some information */
        Show_Message( _("Decoding image:"), "black" );
        snprintf( mesg, MESG_SIZE, "%s", Fname(Met_image_file) );
        Show_Message( mesg, "black" );

        /* Clear variables before decoding sample buffer */
        Met_image_height = 0;
        max_gray_val     = 0;
        Met_image_size   = 0;
        Met_image_buffer = NULL;
        line_cnt         = 0;

        /* Process samples from file to make an image */
        while( File_Fill_Buffer() )
        {
          /* Bypass sync and space/markers of image */
          line_index += METEOR_SYNC_IMAGE_SPACE;

          /* Re-allocate memory to image buffer, abort on error */
          if( !mem_realloc((void *)&Met_image_buffer,
                (size_t)(Met_image_size + METEOR_IMAGE_WIDTH)) )
            return( FALSE );

          /* Decode an image line */
          Meteor_Image_Line( Met_image_buffer + Met_image_size, &max_gray_val );
          Met_image_height++;
          Met_image_size = METEOR_IMAGE_WIDTH * Met_image_height;

          line_cnt++;

        } /* End of while( File_Fill_Buffer() ) */

        /* Rotate images 180 degrees if requested */
        if( isFlagSet(IMAGE_ROTATE) )
          Rotate( Met_image_buffer, Met_image_size );

        /* Carry out histogram normalization if not suppressed */
        if( isFlagSet(IMAGE_NORMALIZE) )
        {
          Normalize( Met_image_buffer, Met_image_size );
          max_gray_val = 255;
        }

        /* Carry out pseudo-colorization if enabled */
        if( isFlagSet(IMAGE_COLORIZE) )
        {
          /* Colorized image file name */
          char Met_color_file[MAX_FILE_NAME+20];

          /* Re-allocate memory to image buffers, abort on error */
          Met_color_buffer = NULL;
          if( !mem_realloc((void *)&Met_color_buffer, (size_t)(3 * Met_image_size)) )
            return(FALSE );

          /* The second argument is a dummy */
          Colorize( Met_image_buffer, Met_image_buffer,
              Met_image_size, Met_color_buffer, METEOR_COLORMAP );

          /* Prepare image file names */
          s = sizeof( Met_color_file );
          Strlcpy( Met_color_file, image_file, s );
          Strlcat( Met_color_file, "-Meteor.ppm", s );

          /* Print some information */
          Show_Message( _("Pseudo-colorizing image:"), "black" );
          snprintf( mesg, MESG_SIZE, "%s", Fname(Met_color_file) );
          Show_Message( mesg, "black" );

          /* Open Channel A colorized image file */
          if( !Open_File(&Met_image_fp, Met_color_file, "w+") )
            return( FALSE );

          /* Write Ch-A colorized buffer to PPM file */
          if( !File_Image(Met_image_fp, "P6", METEOR_IMAGE_WIDTH,
                Met_image_height, max_gray_val,  Met_color_buffer) )
          {
            fclose( Met_image_fp );
            return( FALSE );
          }

          /* Free buffers and close image files */
          Show_Message( _("Closing image file:"), "black" );
          snprintf( mesg, MESG_SIZE, "%s", Fname(Met_color_file) );
          Show_Message( mesg, "black" );

          fclose( Met_image_fp );

        } /* if( isFlagSet(IMAGE_COLORIZE) ) */

        /*** Processing finished, complete image files ***/

        /* Resize window as required */
        duration = line_cnt / 2;
        Initialize_Top_Window();

        /* Draw 1/2 size image to xwxapt window */
        if( isFlagClear(IMAGE_COLORIZE) )
        {
          Halfsize_Image( Met_image_buffer,
              METEOR_IMAGE_WIDTH, line_cnt, 0, FALSE );
          gtk_image_set_from_pixbuf( GTK_IMAGE(apt_image), image_pixbuf );
        }
        else if( Met_color_buffer != NULL )
        {
          Halfsize_Image( Met_color_buffer,
              METEOR_IMAGE_WIDTH, line_cnt, 0, TRUE );
          gtk_image_set_from_pixbuf( GTK_IMAGE(apt_image), image_pixbuf );
        }

        /* Open Meteor APT image file */
        if( !Open_File(&Met_image_fp, Met_image_file, "w+") )
          return( FALSE );

        /* Write Meteor image buffer to file */
        if( !File_Image(
              Met_image_fp, "P5",
              METEOR_IMAGE_WIDTH, Met_image_height,
              max_gray_val, Met_image_buffer) )
        {
          fclose( Met_image_fp );
          return( FALSE );
        }

        /* Free image buffer and close image file */
        Show_Message( _("Closing image file:"), "black" );
        snprintf( mesg, MESG_SIZE, "%s", Fname(Met_image_file) );
        Show_Message( mesg, "black" );

        free( Met_image_buffer );
        if( isFlagSet(IMAGE_COLORIZE) )
          free( Met_color_buffer );
        fclose( Met_image_fp );

      } /* End of case METEOR : */

  } /* End of switch( sat_type ) */

  Cleanup();
  return( FALSE );
} /* End of File_Process_Image() */

/*------------------------------------------------------------------------*/

/*  Normalize()
 *
 *  Does histogram (linear) normalization of a pgm (P5) image file
 */

  void
Normalize( unsigned char *image_buffer, int image_size )
{
  int
    hist[256],  /* Intensity histogram of pgm image file  */
    blk_cutoff, /* Count of pixels for black cutoff value */
    wht_cutoff, /* Count of pixels for white cutoff value */
    pixel_val,  /* Used for calculating normalized pixels */
    pixel_cnt,  /* Total pixels counter for cut-off point */
    idx;        /* Index for loops etc */

  int
    black_val,  /* Black cut-off pixel intensity value */
    white_val,  /* White cut-off pixel intensity value */
    val_range;  /* Range of intensity values in image  */

  if( image_size <= 0 )
  {
    Show_Message( _("Image file seems empty\n"\
          "Normalization not performed"), "red" );
    Error_Dialog();
    return;
  }

  /* Clear histogram */
  for( idx = 0; idx < 256; idx++ )
    hist[ idx ] = 0;

  /* Build image intensity histogram */
  for( idx = 0; idx < image_size; idx++ )
    ++hist[ image_buffer[idx] ];

  /* Determine black/white cut-off counts */
  blk_cutoff = (image_size * BLACK_CUT_OFF) / 100;
  wht_cutoff = (image_size * WHITE_CUT_OFF) / 100;

  /* Find black cut-off intensity value */
  pixel_cnt = 0;
  for( black_val = 0; black_val < 256; black_val++ )
  {
    pixel_cnt += hist[ black_val ];
    if( pixel_cnt > blk_cutoff ) break;
  }

  /* Find white cut-off intensity value */
  pixel_cnt = 0;
  for( white_val = 255; white_val >= 0; white_val-- )
  {
    pixel_cnt += hist[ white_val ];
    if( pixel_cnt > wht_cutoff ) break;
  }

  /* Rescale pixels in image for full intensity range */
  val_range = white_val - black_val;
  if( (int) val_range <= 0 )
  {
    Show_Message( _("Image seems flat\n"\
        "Normalization not performed"), "red" );
    Error_Dialog();
    return;
  }

  /* Perform histogram normalization on images */
  Show_Message( _("Performing histogram normalization "), "black" );
  for( pixel_cnt = 0; pixel_cnt < image_size; pixel_cnt++ )
  {
    pixel_val = image_buffer[ pixel_cnt ];
    pixel_val = ( (pixel_val - black_val) * 255 ) / val_range;

    pixel_val = ( pixel_val < 0 ? 0 : pixel_val );
    pixel_val = ( pixel_val > 255 ? 255 : pixel_val );
    image_buffer[ pixel_cnt ] = (unsigned char)pixel_val;
  }

} /* End of Normalize() */

/*------------------------------------------------------------------------*/

/*  Rotate()
 *
 *  Rotates a pgm (P5) image by 180 degrees
 */

  void
Rotate( unsigned char *image_buffer, int image_size )
{
  int idx; /* Index for loops etc */

  unsigned char
    *idx_temp,  /* Buffer location to be saved in temp    */
    *idx_swap;  /* Buffer location to be swaped with temp */

  /* Holds a pixel value temporarily */
  unsigned char temp;

  if( image_size <= 0 )
  {
    Show_Message( _("Image file empty\n"\
          "Rotation not performed"), "red" );
    Error_Dialog();
    return;
  }

  /* Rotate image 180 degrees */
  Show_Message( _("Rotating image by 180 degrees "), "black" );

  for( idx = 0; idx < image_size / 2; idx++ )
  {
    idx_temp = image_buffer + idx;
    idx_swap = image_buffer - 1 + image_size - idx;
    temp = *( idx_temp );
    *( idx_temp ) = *( idx_swap );
    *( idx_swap ) = temp;
  }

} /* End of Rotate() */

/*------------------------------------------------------------------------*/

/*  Colorize()
 *
 *  Pseudo-colorizes a greyscale image using tables
 */

  void
Colorize(
    unsigned char *image_buffer_a,
    unsigned char *image_buffer_b,
    int image_size,
    unsigned char *colorized_buffer,
    int map_type )
{
  int range, idg, idc, delta, slope;
  color_map_t *map;

  /* Select color map */
  switch( map_type )
  {
    case NOAA_A_COLORMAP:
      map = &rc_data.noaa_A_map;
      break;

    case METEOR_COLORMAP:
      map = &rc_data.meteor_map;
      break;

    default: return;
  }

  /*** Pseudo-colorize greyscale image ***/
  idc = 0;
  for( idg = 0; idg < image_size; idg++ )
  {
    /* Enter clouds image */
    if( image_buffer_b[idg] >= map->white_thld )
    {
      colorized_buffer[idc++] = image_buffer_b[idg];
      colorized_buffer[idc++] = image_buffer_b[idg];
      colorized_buffer[idc++] = image_buffer_b[idg];
    }
    else /* Colorize */
    {
      /* Find gray scale range of image pixel */
      for( range = 0; range < map->num_ranges; range++ )
        if( (image_buffer_a[idg] >= map->gray_from[range]) &&
            (image_buffer_a[idg] <= map->gray_to[range]) )
          break;
      if( range == map->num_ranges )
      {
        Show_Message( _("Error searching colorization map"), "red" );
        Error_Dialog();
        return;
      }

      /* Colorize image */
      delta = map->gray_to[range] - map->gray_from[range];
      if( delta <= 0 )
      {
        Show_Message( _("Invalid gray range in colorization map"), "red" );
        Error_Dialog();
        return;
      }
      slope = ( map->red_to[range] - map->red_from[range] );
      colorized_buffer[idc++] = (unsigned char)( map->red_from[range] +
        (slope*(image_buffer_a[idg]-map->gray_from[range]))/delta );

      slope = ( map->green_to[range] - map->green_from[range] );
      colorized_buffer[idc++] = (unsigned char)(map->green_from[range] +
        (slope*(image_buffer_a[idg]-map->gray_from[range]))/delta );

      slope = ( map->blue_to[range] - map->blue_from[range] );
      colorized_buffer[idc++] = (unsigned char)(map->blue_from[range] +
        (slope*(image_buffer_a[idg]-map->gray_from[range]))/delta );

    } /* if( image_buffer_b[idg] >= map->white_thld ) */

  } /* for( idg = 0; idg < image_size; idg++ ) */

} /* End of Colorize() */

/*------------------------------------------------------------------------*/

/* Halfsize_Image()
 *
 * Scales an image buffer and enters it to pixel buffer
 */

  void
Halfsize_Image(
    unsigned char *buffer, int width,
    int height, int offset, gboolean color )
{
  int sc, sc2, is, idx, idy, ws, hs, pp;
  guchar *pixel;
  int pixel_val;

  /* Draw images at reduced size/sc in xwxapt window */
  sc = 2;
  sc2 = sc * sc;
  hs = height / sc;
  ws = width / sc;

  /* Greyscale images */
  if( !color )
  {
    /* Go down the vertical */
    for( idy = 0; idy < hs; idy++ )
    {
      /* Summate every 2|3 pixels of 2 rows of the image */
      for( idx = 0; idx < ws; idx++ )
      {
        /* Points to top left corner of a 2 x 2
         * block of pixels in greyscale buffer */
        pp = sc * idx + sc * idy * width;

        /* Average pixel value of above block */
        pixel_val = 0;
        for( is = 0; is < sc; is++ )
        {
          pixel_val += (int)buffer[pp];
          pp++;
        }

        /* Go down one row in image buffer */
        pp = sc * idx + (sc * idy + 1) * width;
        for( is = 0; is < sc; is++ )
        {
          pixel_val += (int)buffer[pp];
          pp++;
        }

        /* Normalize pixel value */
        pixel_val /= sc2;

        /* Find the pixel at x,y */
        pixel = pixel_buf +
          ( idy * rowstride ) +
          ( (idx + offset) * n_channels );

        /* Enter average value of 2x2 pixel blocks from image
         * buffer to pixel buffer of image in xwxapt main window */
        pixel[0] = (guchar)pixel_val;
        pixel[1] = (guchar)pixel_val;
        pixel[2] = (guchar)pixel_val;

      } /* for( idx = 0; idx < ws; idx++ ) */
    } /* for( idy = 0; idy < hs; idy++ ) */
  }
  else /* Colorized image */
  {
    for( idy = 0; idy < hs; idy++ )
    {
      /* Top left y of corner of 2x2 pixel block of colorized buffer */
      for( idx = 0; idx < ws; idx++ )
      {
        /* Find the pixel at x,y */
        pixel = pixel_buf + ( idy * rowstride ) +
          ( (idx + offset) * n_channels );

        /* Summate RGB values of each pixel of colorized
         * buffer into pixel buffer of the xwxapt window */
        pixel[0] = 0;
        pixel[1] = 0;
        pixel[2] = 0;

        /* Points to top left corner of a 2 x 2 block
         * of pixel RGB values  in colorized buffer */
        pp = 3 * sc * idx + 3 * sc * idy * width;
        for( is = 0; is < sc; is++ )
        {
          pixel[0] += buffer[pp++] / sc2;
          pixel[1] += buffer[pp++] / sc2;
          pixel[2] += buffer[pp++] / sc2;
        }

        pp = 3 * sc * idx + (3 * sc * idy + 3) * width;
        for( is = 0; is < sc; is++ )
        {
          pixel[0] += buffer[pp++] / sc2;
          pixel[1] += buffer[pp++] / sc2;
          pixel[2] += buffer[pp++] / sc2;
        }

      } /* for( idx = 0; idx < ws; idx++ ) */
    } /* for( idy = 0; idy < hs; idy++ ) */
  } /* if( isFlagClear(IMAGE_COLORIZE) ) */

} /* End of Halfsize_Image() */

/*------------------------------------------------------------------------*/

/* Halfsize_Row()
 *
 * Produces an image row at half the width of a greyscale image
 * by averaging a 2x2 block of pixels from two consecutive rows
 */

  void
Halfsize_Row( unsigned char *buffer, int width, int line_cnt, int offset )
{
  guchar *pixel;
  static int pixel_val = 0;
  int sc = 2, is, ws, xs, idx, idy, stride;


  /* Move one row down in image buffer */
  stride = line_cnt * width;

  /* Move one row down in 1/2 size image buffer */
  idy = (line_cnt / sc) * rowstride;

  /* Summate pixels of full size image
   * row into pixel of the scaled row */
  ws = width / sc;
  for( idx = 0; idx < ws; idx++ )
  {
    /* Summate pixel values from image buffer */
    xs = idx * sc;
    for( is = 0; is < sc; is++ )
    {
      pixel_val += buffer[xs + stride];
      xs++;
    }

    /* Find the pixel at x,y */
    pixel = pixel_buf + idy + ((idx + offset) * n_channels);

    /* If 2nd line, summate, else, initialize */
    if( idy & 1 )
    {
      pixel_val /= sc;
      pixel[0]  += (guchar)pixel_val;
      pixel[1]  += (guchar)pixel_val;
      pixel[2]  += (guchar)pixel_val;
      pixel_val  = 0;
    } /* if( line_cnt & 0x01 ) */
    else
    {
      pixel_val /= sc;
      pixel[0]   = (guchar)pixel_val;
      pixel[1]   = (guchar)pixel_val;
      pixel[2]   = (guchar)pixel_val;
      pixel_val  = 0;
    }

  } /* for( idx = 0; idx < width/2; idx++ ) */

} /* Halfsize_Row() */

/*------------------------------------------------------------------------*/

