// -*- C++ -*-
#include "Rivet/Analysis.hh"
#include "Rivet/Projections/FinalState.hh"
#include "Rivet/Projections/ChargedFinalState.hh"

namespace Rivet {


  /// @brief Hadronic charged multiplicity measurement between 12 and 31.3 GeV
  class PLUTO_1980_I154270 : public Analysis {
  public:

    /// Constructor
    RIVET_DEFAULT_ANALYSIS_CTOR(PLUTO_1980_I154270);


    /// @name Analysis methods
    /// @{

    /// Book histograms and initialise projections before the run
    void init() {
      const ChargedFinalState cfs;
      declare(cfs, "CFS");
      book(_mult, 1, 1, 1);
      for (const string& en : _mult.binning().edges<0>()) {
        const size_t idx = en.find("-");
        if (idx != string::npos) {
          const double emin = stod(en.substr(0,idx));
          const double emax = stod(en.substr(idx+1, string::npos));
          if (inRange(sqrtS()/GeV, emin, emax)) {
            _sqs = en; break;
          }
        }
        else {
          const double eval = stod(en)*GeV;
          if (isCompatibleWithSqrtS(eval)) {
            _sqs = en; break;
          }
        }
      }
      raiseBeamErrorIf(_sqs.empty());
    }


    /// Perform the per-event analysis
    void analyze(const Event& event) {
      const FinalState& cfs = apply<FinalState>(event, "CFS");
      MSG_DEBUG("Total charged multiplicity = " << cfs.size());
      unsigned int nPart(0);
      for (const Particle& p : cfs.particles()) {
        // check if prompt or not
        ConstGenParticlePtr pmother = p.genParticle();
        ConstGenVertexPtr ivertex = pmother->production_vertex();
        bool prompt = true;
        while (ivertex) {
          vector<ConstGenParticlePtr> inparts = HepMCUtils::particles(ivertex, Relatives::PARENTS);
          int n_inparts = inparts.size();
          if (n_inparts < 1)  break;
          pmother = inparts[0]; // first mother particle
          int mother_pid = abs(pmother->pdg_id());
          if (mother_pid==PID::K0S || mother_pid==PID::LAMBDA) {
            prompt = false; break;
          }
          else if (mother_pid<6)  break;
          ivertex = pmother->production_vertex();
        }
        if (prompt) ++nPart;
      }
      _mult->fill(_sqs, nPart);
    }


    /// Normalise histograms etc., after the run
    void finalize() {
      scale(_mult, 1./sumOfWeights());
    }

    /// @}


  private:

    BinnedProfilePtr<string> _mult;
    string _sqs = "";

  };


  RIVET_DECLARE_PLUGIN(PLUTO_1980_I154270);

}
