// -*- c++ -*-
//------------------------------------------------------------------------------
// $Id: assa-genesis.cpp,v 1.14 2007/05/14 19:19:50 vlg Exp $
//------------------------------------------------------------------------------
//                            assa-genesis.cpp
//------------------------------------------------------------------------------
//  Copyright (C) 2001,2005-2007  Vladislav Grinchenko
//
//  This program is free software; you can redistribute it and/or
//  modify it under the terms of the GNU General Public License
//  as published by the Free Software Foundation; either version
//  2 of the License, or (at your option) any later version.
//
//  Creation date: Nov 1 2001
//------------------------------------------------------------------------------
static const char help_msg[]=
"                                                                            \n"
" NAME:                                                                      \n"
"                                                                            \n"
"   assa-genesis                                                             \n"
"                                                                            \n"
" DESCRIPTION:                                                               \n"
"                                                                            \n"
"   assa-genesis generates skeleton files for rapid application              \n"
"   development with ASSA library.                                           \n"
"                                                                            \n"
"   See ASSA User's Guide for further details.                               \n"
"                                                                            \n"
" USAGE:                                                                     \n"
"                                                                            \n"
"   shell>  assa-genesis [OPTIONS] <app_name>                                \n"
"                                                                            \n"
" Four files are generated by default:                                       \n"
"                                                                            \n"
"   <app_name>-main.h     - Header file with debug tracing masks             \n"
"   <app_name>-main.cpp   - 'main()' function with event loop                \n"
"   <app_name>.h          - Header file for class <app_name>                 \n"
"   <app_name>.cpp        - Implementation stubs for class <app_name>        \n"
"                                                                            \n"
" In addition, if --gtk2-app options was specified, the MainWindow class     \n"
" is generated:                                                              \n"
"                                                                            \n"
"   MainWindow.h          - Class MainWindow declaration                     \n"
"   MainWindow.cpp        - Class MainWindow definition                      \n"
"                                                                            \n"
" If {-t, --one-file} switch is used, everything is put in one file,         \n"
" <app_name>.cpp. This is primarily used for writing test programs.          \n"
"                                                                            \n"
"                                                                            \n"
" OPTIONS:                                                                   \n"
"                                                                            \n"
" --with-example=BOOL      - Add example of processing positional            \n"
"                            command-line arguments.                         \n"
" --with-gpl-license=BOOL  - (Default) Generate files with GPL license       \n"
"                            headers.                                        \n"
" --with-lgpl-license=BOOL - Generate files with LGPL license headers.       \n"
" --with-x11-license=BOOL  - Generate files with X11-type license headers.   \n"
" --with-makefile=BOOL     - Generate makefile.                              \n"
"                                                                            \n"
" --gtk2-app=BOOL          - Generate Gtk2 ready application.                \n"
" -o, --one-file=BOOL      - Put everything in one file                      \n"
" -e, --extension STRING   - Set file extention for C++ code (C, cxx, ...)   \n"
" -h, --help               - print help messag                               \n"
" -v, --version            - print version number                            \n"
"                                                                            \n"
" NOTE:                                                                      \n"
"                                                                            \n"
"  From Merriam-Webster Dictionary:                                          \n"
"    'genesis' (n) - to be born (Etymology: from Greek 'gignesthai')         \n"
"                                                                           \n";
//------------------------------------------------------------------------------

#ifdef HAVE_CONFIG_H
#    include "config.h"
#else
#    define ASSA_MAJOR_VERSION  1
#    define ASSA_MINOR_VERSION  0
#    define ASSA_MICRO_VERSION  0
#endif

#include <iostream>
#include <algorithm>
#include <cctype>
#include <string>
#include <fstream>

#include "assa/Assure.h"   
#include "assa/TimeVal.h"  
#include "assa/GenServer.h"
#include "assa/Singleton.h"

#include <string>
using std::string;
using std::cerr;
using std::endl;
using std::cout;

using namespace ASSA;

static char assa_version [64];
string date_year;
string year;

static string mytab ("\t");

class Genesis :
    public GenServer,
    public Singleton<Genesis>
{
public:                                      
    Genesis ();               

    virtual void init_service   ();          
    virtual void process_events ();       

private:
    virtual void pos_arg (const char* arg_);

    void write_header            (const char* name_, ostream& sink_);
	void write_onefile_header    (ostream& sink_);
    void write_class_declaration (const char* name_, ostream& sink_);
    void write_class_definition  (const char* name_, ostream& sink_);

    void write_mw_class_declaration (ostream& sink_);
    void write_mw_class_definition  (ostream& sink_);

    void write_main     (const char* name_, ostream& msrc_);
    void write_help_msg (ostream& sink_);

	void write_gpl_license  (ostream& sink_);
	void write_lgpl_license (ostream& sink_);
	void write_x11_license  (ostream& sink_);

    void generate_one_file ();
	void generate_makefile ();

private:
    string m_app_name;
    string m_extension;	
    u_int  m_pos_arg_count;
    bool   m_one_file;		
	bool   m_with_example;
	bool   m_with_gpl_license;
	bool   m_with_lgpl_license;
	bool   m_with_x11_license;
	bool   m_gtk2_app;
	bool   m_with_makefile;
};

// Static declarations mandated by Singleton class

ASSA_DECL_SINGLETON(Genesis);

Genesis::
Genesis () : 
    m_extension ("cpp"), 
	m_pos_arg_count (1), 
	m_one_file (false),
	m_with_example (false),
	m_with_gpl_license (true),
	m_with_lgpl_license (false),
	m_with_x11_license (false),
	m_gtk2_app (false),
	m_with_makefile (false)
{
    add_opt ('e', "extension", &m_extension); 

    add_flag_opt ('o', "one-file",        &m_one_file);
	add_flag_opt (0, "with-example",      &m_with_example);
	add_flag_opt (0, "with-gpl-license",  &m_with_gpl_license);
	add_flag_opt (0, "with-lgpl-license", &m_with_lgpl_license);
	add_flag_opt (0, "with-x11-license",  &m_with_x11_license);
	add_flag_opt (0, "gtk2-app",          &m_gtk2_app);
	add_flag_opt (0, "with-makefile",     &m_with_makefile);

    rm_opt ('m', "mask"         );
    rm_opt ('d', "log-stdout"   );
    rm_opt ('D', "debug-file"   );
    rm_opt ('z', "log-size"     );

    rm_opt ('f', "config-file"  ); 
    rm_opt ('n', "instance"     ); 
    rm_opt ('p', "port"         ); 
    rm_opt ('s', "set-name"     ); 
    rm_opt ('t', "comm-timeout" ); 

    rm_opt ('b', "daemon"       ); 
    rm_opt ('l', "pidfile"      ); 
    rm_opt ('L', "ommit-pidfile"); 

    /*---
     * Disable all debugging
     *---*/
    m_mask = 0;
}

void
Genesis::
pos_arg (const char* arg_) 
{
    switch (m_pos_arg_count) {
    case 1:
		m_app_name = arg_; 
		break;
    case 2:
    default:
		cerr << "Error: unexpected arguments!" << endl;
		Assure_exit (false);
    }
    m_pos_arg_count++;
}


void
Genesis::
init_service ()
{
    Log::disable_timestamp ();
}

void
Genesis::
generate_makefile ()
{
	string mkfname ("GNUmakefile." + m_app_name);
	std::ofstream mkfile (mkfname.c_str ());
	if (!mkfile) {
		cerr << "Cannot create \"" << mkfname << "\"" << endl; 
		exit (1);
	}
    cout << "\nassa-genesis: Generating makefile ...\n";

	mkfile 
<< "## -*- makefile -*-\n"
<< "##\n"
<< "## " << mkfname << ": generated by assa-genesis\n"
<< "##\n"
<< '\n';

	if (m_gtk2_app) {
		mkfile
<< "APP_CFLAGS=`pkg-config assa-" << assa_version << " --cflags` `pkg-config gtkmm-2.4 --cflags`\n"
<< "APP_LIBS=`pkg-config assa-"   << assa_version << " --libs` `pkg-config gtkmm-2.4 --libs`\n\n";
	}
	else {
		mkfile
<< "APP_CFLAGS=`pkg-config assa-" << assa_version << " --cflags`\n"
<< "APP_LIBS=`pkg-config assa-"   << assa_version << " --libs`\n\n";
	}

	mkfile
<< "OBJS = \\\n";

	if (m_one_file) {
		mkfile
<<	"\t" << m_app_name << ".o\n";
	}
	else {
		mkfile << " " << m_app_name << ".o " << m_app_name << "-main.o";
		if (m_gtk2_app) {
			mkfile << " MainWindow.o";
		}
		mkfile << '\n';
	}

	mkfile
<< '\n'
<< "all: " << m_app_name << "\n"
<< '\n'
<< m_app_name << ": ${OBJS}\n"
<< "\tg++ -g ${APP_CFLAGS} ${OBJS} -o " << m_app_name << " ${APP_LIBS}\n"
<< '\n'
<< "." << m_extension << ".o:\n"
<< "\tg++ -g ${APP_CFLAGS} -c $<\n"
<< '\n'
<< "clean:\n"
<< "\trm -f *.o *~ core core.*\n"
<< '\n'
<< "distclean: clean\n"
<< "\trm " << m_app_name << "\n"
<< "\trm *.log *.log.0\n"
<< '\n'
<< "dist:\n"
<< "\ttar cvfz " << m_app_name << ".tar.gz *.h *." << m_extension << " makefile\n";

	mkfile
<< '\n'
<< "## Dependencies\n"
<< '\n';

	if (m_one_file) {
		mkfile
<< m_app_name << ".o: " << m_app_name << "." << m_extension << "\n";
	}
	else {
		mkfile
<< m_app_name << ".o: " << m_app_name << "." << m_extension << " " << m_app_name << ".h\n"
<< m_app_name << "-main.o: " << m_app_name << "-main." << m_extension << " " << m_app_name << "-main.h\n";
		if (m_gtk2_app) {
			mkfile
<< "MainWindow.o: MainWindow." << m_extension << " MainWindow.h\n";
		}
	}

	mkfile << endl;
	mkfile.close ();
}

void
Genesis::
generate_one_file ()
{
    string projname (m_app_name);
    string fname (projname);
    fname += '.' + m_extension;

    std::ofstream onefile (fname.c_str ());
    if (!onefile) {
		cerr << "Cannot open input file \"" << fname << "\"" << endl;
		exit (1);
    }
    cout << "\nassa-genesis: Generating skeleton files ...\n";

    write_header (fname.c_str (), onefile);
    write_help_msg (onefile);
	write_onefile_header (onefile);
    write_class_declaration (projname.c_str (), onefile);

	if (m_gtk2_app) {
		write_mw_class_declaration (onefile);
	}

    write_class_definition  (projname.c_str (), onefile);

	if (m_gtk2_app) {
		write_mw_class_definition (onefile);
	}

    write_main (projname.c_str (), onefile);
    
    onefile << endl;
    onefile.close ();

    cout << "\nCreated: \"" << fname << "\"\n\n";
}

void
Genesis::
write_gpl_license (ostream& sink_)
{
	sink_
<< "//  This program is free software; you can redistribute it and/or \n"
<< "//  modify it under the terms of the GNU General Public License   \n"
<< "//  as published by the Free Software Foundation; either version  \n"
<< "//  2 of the License, or (at your option) any later version.      \n";

}
    
void
Genesis::
write_lgpl_license (ostream& sink_)
{
	sink_
<< "//  This library is free software; you can redistribute it and/or    \n"
<< "//  modify it under the terms of the GNU Library General Public      \n"
<< "//  License as published by the Free Software Foundation; either     \n"
<< "//  version 2 of the License, or (at your option) any later version. \n";

}
    
void
Genesis::
write_x11_license (ostream& sink_)
{
	sink_
<< "//  Permission to use, copy, modify, and distribute this software       \n"
<< "//  and its documentation for any purpose and without fee is hereby     \n"
<< "//  granted, provided that the above copyright notice appear in all     \n"
<< "//  copies.  The author makes no representations about the suitability  \n"
<< "//  of this software for any purpose.  It is provided \"as is\" without \n"
<< "//  express or implied warranty.                                        \n";

}
    
void
Genesis::
process_events ()              
{
    string projname (m_app_name);
    string fname;

    if (m_pos_arg_count == 1) {
		/*---
		 * We haven't seen expected position argument
		 *---*/
		cerr << endl << endl
			 << "Missing <app_name> argument" << endl
			 << "Try `assa-genesis --help` for details" << endl 
			 << endl;
		exit (1);
    }

	if (m_with_makefile) {
		generate_makefile ();
	}
	
    if (m_one_file) {
		generate_one_file ();
		return;
    }

    cout << "\nassa-genesis: Generating skeleton files ...\n";

    /*-----------------------------------------------------*/
    /*---              Create 'main.h'                  ---*/
    /*-----------------------------------------------------*/

    string header_name (projname);
    header_name += "-main.h";
    std::ofstream mheader (header_name.c_str ());

    if (!mheader) {
		cerr << "Cannot open \"" << header_name << "\"\n";
		exit (1);
    }

    write_header (header_name.c_str (), mheader);

    mheader 
<< "#ifndef MAIN_H\n"
<< "#define MAIN_H\n";

    mheader << endl

<< "#include <assa/Assure.h>\n";

	if (m_gtk2_app) {
		mheader	<< endl
<< "enum { GUITRACE = ASSA::USR1 };\n";
	}

	mheader	<< endl

<< "#endif /* MAIN_H */\n"

	<< endl;
    
	mheader.close ();
    cout << "\nCreated: \"" << header_name << "\"\n";

    /*-----------------------------------------------------*/
    /*---              Creating 'main.cpp'              ---*/
    /*-----------------------------------------------------*/

    fname = projname + "-main." + m_extension;
    std::ofstream msrc (fname.c_str ());
    if (!msrc) {
		cerr << "Cannot open \"" << fname << "\"\n";
		exit (1);
    }
    
    write_header (fname.c_str (), msrc);
    write_help_msg (msrc);

    msrc << endl

<< "#include \"" << header_name << "\"\n"
<< "#include \"" << projname << ".h\"\n"

		 << endl;

    write_main (projname.c_str (), msrc);

    msrc.close ();
    cout << "\nCreated: \"" << fname << "\"\n";

    /*-----------------------------------------------------*/
    /*---            Create Project's header file       ---*/
    /*-----------------------------------------------------*/

    string hdr_fname = projname + string (".h");
    std::ofstream hdr (hdr_fname.c_str ());
	
    if (!hdr) {
		cerr << "Cannot open file " << hdr_fname << endl;
		exit (1);
    }

    write_header (hdr_fname.c_str (), hdr);

    hdr << endl

<< "#ifndef " << projname << "_H\n"
<< "#define " << projname << "_H\n"

		<< endl;

    write_class_declaration (projname.c_str (), hdr);

    hdr << endl

<< "#endif // " << projname << "_H\n"

		<< endl;

    hdr.close ();
    cout << "\nCreated: \"" << hdr_fname << "\"\n";

    /*-----------------------------------------------------*/
    /*---            Create Project's source file       ---*/
    /*-----------------------------------------------------*/

    fname = projname + string (".") + m_extension;
    std::ofstream source (fname.c_str ());
	
    if (!source) {
		cerr << "Cannot open input file \"" << fname << "\"" << endl;
		exit (1);
    }

    write_header (fname.c_str (), source);

    source << endl

<< "#include \"" << header_name << "\"\n"
<< "#include \"" << projname << ".h\"\n";

	if (m_gtk2_app) {
		source << endl
<< "#include \"MainWindow.h\"\n";
	}

    write_class_definition (projname.c_str (), source);

    source << endl;
    source.close ();
    cout << "\nCreated: \"" << fname << "\"\n\n";


	/** Bail out here if we are not generating GTK2 application
	 */
	if (!m_gtk2_app) {
		return;
	}

    /*-----------------------------------------------------*/
    /*---           Create MainWindow header file       ---*/
    /*-----------------------------------------------------*/

    std::ofstream mwh ("MainWindow.h");
	
    if (!mwh) {
		cerr << "Cannot open file MainWindow.h" << endl;
		exit (1);
    }

    write_header ("MainWindow.h", mwh);

    mwh << endl

<< "#ifndef MAIN_WINDOW_H\n"
<< "#define MAIN_WINDOW_H\n"

		<< endl;

    write_mw_class_declaration (mwh);

    mwh << endl

<< "#endif // MAIN_WINDOW_H\n"

		<< endl;

    mwh.close ();
    cout << "\nCreated: \"MainWindow.h\"\n";

    /*-----------------------------------------------------*/
    /*---            Create MainWindow source file      ---*/
    /*-----------------------------------------------------*/

    fname = string ("MainWindow.") + m_extension;
    std::ofstream mw_source (fname.c_str ());
	
    if (!mw_source) {
		cerr << "Cannot open input file \"" << fname << "\"" << endl;
		exit (1);
    }

    write_header (fname.c_str (), mw_source);
    write_mw_class_definition (mw_source);

    mw_source << endl;
    mw_source.close ();
    cout << "\nCreated: \"" << fname << "\"\n\n";
}


void 
Genesis::
write_header (const char* name_, ostream& sink_)
{
    sink_ << "// -*- c++ -*-\n" 
<< "// Generated by assa-genesis\n" 
<< "//------------------------------------------------------------------------------\n" 
<< "// $" << "Id" << "$\n"
<< "//------------------------------------------------------------------------------\n"
<< "//                            " << name_ << "\n"
<< "//------------------------------------------------------------------------------\n"
		  << "//  Copyright (c) " << year << " by YOUR-NAME \n"
<< "//\n";

	if (m_with_lgpl_license) {
		write_lgpl_license (sink_);
	}
	else if (m_with_x11_license) {
		write_x11_license (sink_);
	}
	else if (m_with_gpl_license) {
		write_gpl_license (sink_);
	}
	else {
		write_gpl_license (sink_);
	}

	sink_
<< "//------------------------------------------------------------------------------\n"
<< "//\n"
<< "// Date   : " << date_year << "\n"
<< "//\n"
<< "//------------------------------------------------------------------------------\n";
	sink_ << endl;
}

void
Genesis::
write_help_msg (ostream& sink_)
{
    sink_ 
<< "static const char help_msg[]=\n"
<< "\"                                                                            \\n\"\n"
<< "\" NAME:                                                                      \\n\"\n"
<< "\"                                                                            \\n\"\n"
<< "\"   Your application program name                                            \\n\"\n"
<< "\"                                                                            \\n\"\n"
<< "\" DESCRIPTION:                                                               \\n\"\n"
<< "\"                                                                            \\n\"\n"
<< "\"   Short description to give general feeling                                \\n\"\n"
<< "\"   of what its main purpose is.                                             \\n\"\n"
<< "\"                                                                            \\n\"\n"
<< "\" USAGE:                                                                     \\n\"\n"
<< "\"                                                                            \\n\"\n"
<< "\"   shell>  app_name [OPTIONS]                                               \\n\"\n"
<< "\"                                                                            \\n\"\n"
<< "\" OPTIONS:                                                                   \\n\"\n"
<< "\"                                                                            \\n\"\n"
<< "\" -b, --daemon BOOL       - Run process as true UNIX daemon                  \\n\"\n"
<< "\" -l, --pidfile PATH      - The process ID is written to the lockfile PATH   \\n\"\n"
<< "\"                           instead of default ~/.{procname}.pid             \\n\"\n"
<< "\" -L, --ommit-pidfile BOOL- Do not create PID lockfile                       \\n\"\n"
<< "\"                                                                            \\n\"\n"
<< "\" -D, --log-file NAME     - Write debug to NAME file                         \\n\"\n"
<< "\" -d, --log-stdout BOOL   - Write debug to standard output                   \\n\"\n"
<< "\" -z, --log-size NUM      - Maximum size debug file can reach (dfl: is 10Mb) \\n\"\n"
<< "\"                                                                            \\n\"\n"
<< "\" -c, --log-level NUM     - Log verbosity                                    \\n\"\n"
<< "\" -s, --with-log-server BOOL - Redirect log messages to the log server       \\n\"\n"
<< "\" -S, --log-server NAME   - Define assa-logd server address                  \\n\"\n"
<< "\"                                                                            \\n\"\n"
<< "\" -m, --mask MASK         - Mask (default: ALL = 0x7fffffff)                 \\n\"\n"
<< "\" -p, --port NAME         - The tcp/ip port NAME (default - procname)        \\n\"\n"
<< "\" -n, --instance NUM      - Process instance NUM (default - none)            \\n\"\n"
<< "\" -f, --config-file NAME  - Alternative configuration file NAME              \\n\"\n"
<< "\"                                                                            \\n\"\n"
<< "\" -h, --help              - Print this messag                                \\n\"\n"
<< "\" -v, --version           - Print version number                            \\n\";\n"
<< "//------------------------------------------------------------------------------\n";
}

void
Genesis::
write_mw_class_declaration (ostream& sink_)
{
	sink_ << endl

<< "#ifdef HAVE_CONFIG_H\n"
<< "#    include \"config.h\"\n"
<< "#else\n"
<< "#    define APP_MAJOR_VERSION  1\n"
<< "#    define APP_MINOR_VERSION  0\n"
<< "#    define APP_MICRO_VERSION  0\n"
<< "#endif\n"
<< "\n"
<< "#include <gtkmm/window.h>\n"
<< "#include <gtkmm/box.h>\n"
<< "#include <gtkmm/menubar.h>\n"
<< "#include <gtkmm/menu.h>\n"
<< "\n"
<< "class MainWindow : public Gtk::Window\n"
<< "{\n"
<< "public:\n"
<< mytab << "MainWindow ();\n"
<< mytab << "~MainWindow ();\n"
<< "\n"
<< "protected:\n"
<< mytab << "void on_menu_file_quit ();\n"
<< mytab << "void on_menu_help ();\n"
<< "\n"
<< "private:\n"
<< mytab << "Gtk::MenuBar m_menu_bar;\n"
<< mytab << "Gtk::Menu    m_menu_file;\n"
<< mytab << "Gtk::Menu    m_menu_help;\n"
<< mytab << "Gtk::VBox    m_box;\n"
<< "};\n"
<< "\n"
		  << endl;
}

void
Genesis::
write_mw_class_definition (ostream& sink_)
{
	/** Included 
	 */
	sink_ << endl
<< "#include <gtkmm/stock.h>\n";

	if (!m_one_file) {
		sink_
<< "#include \"MainWindow.h\"\n"
<< "#include \"" << m_app_name << "-main.h\"\n"
<< "\n";
	}

	/** Constructor
	 */
	sink_ << endl

<< "MainWindow::\n"
<< "MainWindow ()\n"
<< "{\n"
<< mytab << "trace_with_mask(\"MainWindow::MainWindow\",GUITRACE);\n"
<< "\n"
<< mytab << "set_title (\"" << m_app_name << "\");\n"
<< mytab << "set_default_size (200, 300);\n"
<< "\n"
<< mytab << "add (m_box);\n"
<< "\n"
<< mytab << "/** <File> menu\n"
<< mytab << " */\n"
<< mytab << "{\n"
<< mytab << "    Gtk::Menu::MenuList& menulist = m_menu_file.items ();\n"
<< mytab << "    menulist.push_back (\n"
<< mytab << "        Gtk::Menu_Helpers::MenuElem (\"_Quit\",\n"
<< mytab << "            Gtk::AccelKey (\"<control>q\"),\n"
<< mytab << "            sigc::mem_fun (*this, &MainWindow::on_menu_file_quit)));\n"
<< mytab << "}\n"
<< "\n"
<< mytab << "/** <Help> menu\n"
<< mytab << " */\n"
<< mytab << "{\n"
<< mytab << "    Gtk::Menu::MenuList& menulist = m_menu_help.items ();\n"
<< mytab << "    menulist.push_back (\n"
<< mytab << "        Gtk::Menu_Helpers::MenuElem (\"_Help\",\n"
<< mytab << "            Gtk::AccelKey (\"<control>h\"),\n"
<< mytab << "            sigc::mem_fun (*this, &MainWindow::on_menu_help)));\n"
<< mytab << "    menulist.back ().set_right_justified ();\n"
<< mytab << "}\n"
<< "\n"
<< mytab << "/** Add menus to the MenuBar\n"
<< mytab << " */\n"
<< mytab << "m_menu_bar.items ().push_back (\n"
<< mytab << "    Gtk::Menu_Helpers::MenuElem (\"_File\", m_menu_file));\n"
<< "\n"
<< mytab << "m_menu_bar.items ().push_back (\n"
<< mytab << "    Gtk::Menu_Helpers::StockMenuElem (Gtk::Stock::HELP, m_menu_help));\n"
<< "\n"
<< mytab << "/** Add the MenuBar to the Window\n"
<< mytab << " */\n"
<< mytab << "m_box.pack_start (m_menu_bar, Gtk::PACK_SHRINK);\n"
<< "\n"
<< mytab << "show_all ();\n"
<< "}\n"
<< "\n";

	/** Destructor
	 */
	sink_ << "\n"

<< "MainWindow::\n"
<< "~MainWindow ()\n"
<< "{\n"
<< mytab << "trace_with_mask(\"MainWindow::~MainWindow\",GUITRACE);\n"
<< mytab << "/* no-op */\n"
<< "}\n";

	/** Callbacks
	 */
	sink_ << "\n"

<< "void\n"
<< "MainWindow::\n"
<< "on_menu_file_quit ()\n"
<< "{\n"
<< mytab << "trace_with_mask(\"MainWindow::on_menu_file_quit\",GUITRACE);\n"
<< "\n"
<< mytab << "hide ();   // Close the MainWindow to stop Gtk::Main::run ()\n"
<< "}\n";

	sink_ << "\n"

<< "void\n"
<< "MainWindow::\n"
<< "on_menu_help ()\n"
<< "{\n"
<< mytab << "trace_with_mask(\"MainWindow::on_menu_help\",GUITRACE);\n"
<< "\n"
<< mytab << "// Do your help here\n"
<< "}\n"
<< "\n"
		  << endl;
}

void
Genesis::
write_class_declaration (const char* projname_, ostream& sink_)
{
	string PROJNAME (projname_);

	for (int i = 0; i < PROJNAME.length (); i++) {
		PROJNAME [i] = toupper (PROJNAME [i]);
	}

    sink_ <<  endl

<< "#ifdef HAVE_CONFIG_H\n"
<< "#    include \"config.h\"\n"
<< "#else\n"
<< "#    define APP_MAJOR_VERSION  1\n"
<< "#    define APP_MINOR_VERSION  0\n"
<< "#    define APP_MICRO_VERSION  0\n"
<< "#endif"
<< "\n"
<< "#include <string>\n"
<< "using std::string;\n";

	if (m_gtk2_app) {
		sink_
<< "#include <gtkmm.h>\n";
	}

	sink_ << endl

<< "#include <assa/GenServer.h>\n"
<< "#include <assa/Singleton.h>\n"
<< "#include <assa/TimeVal.h>\n";

	if (m_gtk2_app) {
		sink_ << endl
<< "class MainWindow;\n";
	}

	sink_ << endl

/** Class declaration
 */

<< "class " << projname_ << " :\n";

	if (m_gtk2_app) {
		sink_ 
<< mytab << "public virtual sigc::trackable,\n";
	}

	sink_

<< mytab << "public ASSA::GenServer,\n" 
<< mytab << "public ASSA::Singleton<" << projname_  << ">\n"
<< "{\n"
<< "public:\n"
<< mytab << projname_ << " ();\n";

	if (m_gtk2_app) {
		sink_
<< mytab << "~" << projname_ << " ();\n";
	}

	sink_
<< "\n"
<< mytab << "virtual void init_service ();\n" 
<< mytab << "virtual void process_events ();\n"
<< "\n";

	if (m_gtk2_app) {
		sink_ 
<< mytab << "bool timer_cb ();\n";
	}

	if (m_with_example) {
		sink_
<< '\n'
<< "private:\n"
<< mytab << "// An example of processing positional arguments\n"
<< mytab << "virtual void pos_arg (const char* arg_);\n";
	}

	sink_
<< '\n'
<< "private:\n";

	if (m_with_example) {
		sink_
<< mytab << "string  m_pos_arg1;       // An example of positional argument\n"
<< mytab << "float   m_cmd_loop;       // An example of optional argument\n"
<< mytab << "u_int   m_pos_arg_count;  // Number of pos. arguments processed\n";
	}
	
	if (m_gtk2_app) {
		sink_
<< mytab << "static const int IDLE_TIMEOUT = 500;\n"
<< '\n'
<< mytab << "std::string      m_gtk_options;\n"
<< mytab << "Gtk::Main*       m_kit;\n"
<< mytab << "MainWindow*      m_main_window;\n"
<< mytab << "ASSA::TimeVal    m_timeout;\n"
<< mytab << "sigc::connection m_tconn;\n";
	}

	sink_
<< "};\n"
<< '\n'
<< '\n'
<< "/* Useful definitions */\n"
<< '\n'
<< "#define " << PROJNAME << "  " << projname_ << "::get_instance()\n"
<< "#define REACTOR "       << PROJNAME <<    "->get_reactor()\n"
<< '\n';
}

void 
Genesis::
write_class_definition (const char* projname_, ostream& sink_)
{

/** Static declarations
 */
    sink_ << endl

<< "// Static declarations mandated by Singleton class\n"
<< "ASSA_DECL_SINGLETON(" << projname_ << ");\n";

/** Constructor
 */
	sink_ << endl

<< projname_ << "::\n" 
<< projname_ << " ()";

	if (m_with_example || m_gtk2_app) {
		sink_ << " :\n";
	}

	if (m_with_example) {
		sink_
<< mytab << "  m_pos_arg1 (\"VLG\"),\n"
<< mytab << "  m_cmd_loop (34.2),\n"
<< mytab << "  m_pos_arg_count (0)";
		if (m_gtk2_app) {
			sink_ << ",\n";
		}
	}
	else if (m_gtk2_app) {
		sink_
<< mytab << "  m_kit (NULL),\n"
<< mytab << "  m_main_window (NULL)";
	}
	else {
		sink_ << "\n";
	}

	sink_
<< "{\n";

	if (m_with_example) {
		sink_
<< mytab << "// An example of adding a command-line option\n"
<< mytab << "// {-l, --loop}\n"
<< mytab << "//\n"
<< "\n"
<< mytab << "add_opt ('l', \"loop\", &m_cmd_loop);\n";
   }

	if (m_gtk2_app) {
		sink_
<< mytab << "add_opt ('g', \"gtk-options\", &m_gtk_options);\n";
	}

	sink_
<< "\n"
<< mytab << "// Following list removes all predefined options.\n"
<< mytab << "// Remove only those that you won't use.\n"
<< mytab << "// Don't forget to update your help message in 'main.cpp'!\n"
<< mytab << "//\n"
<< "\n"
<< mytab << "// ---General---\n"
<< mytab << "// rm_opt ('h', \"help\"         );\n"
<< mytab << "// rm_opt ('v', \"version\"      );\n"
<< "\n"
<< mytab << "// ---Debugging---\n"
<< mytab << "// rm_opt ('m', \"mask\"         );\n"
<< mytab << "// rm_opt ('d', \"log-stdout\"   );\n"
<< mytab << "// rm_opt ('D', \"log-file\"     );\n"
<< mytab << "// rm_opt ('z', \"log-size\"     );\n"
<< "\n"
<< mytab << "// ---Configuration---\n"
<< mytab << "// rm_opt ('f', \"config-file\"  );\n"
<< mytab << "// rm_opt ('n', \"instance\"     );\n"
<< mytab << "// rm_opt ('p', \"port\"         );\n"
<< "\n"
<< mytab << "// ---Process bookkeeping---\n"
<< mytab << "// rm_opt ('b', \"daemon\"       );\n"
<< mytab << "// rm_opt ('l', \"pidfile\"      );\n"
<< mytab << "// rm_opt ('L', \"ommit-pidfile\");\n"
	       << endl
<< mytab << "/*---\n"
<< mytab << " * By defauil disable all debugging\n"
<< mytab << " *---*/\n";

	if (m_gtk2_app) {
		sink_
<< mytab << "m_mask = ASSA::APP | GUITRACE | ASSA::ASSAERR;\n";
	}
	else {
		sink_
<< mytab << "m_mask = ASSA::APP | ASSA::ASSAERR;\n";
	}

	   sink_
<< mytab << "m_log_file = \"" << projname_ << ".log\";\n"
<< "}\n"

<< endl;

/** Optional (gtk2-app) destructor
 */
	if (m_gtk2_app) {
		sink_

<< projname_ << "::\n" 
<< "~" << projname_ << " ()\n"
<< "{\n"
<< mytab << "trace_with_mask(\"" << projname_ << "::~" << projname_ << "\",GUITRACE);\n"
<< "\n"
<< mytab << "m_tconn.disconnect ();\n"
<< mytab << "delete m_main_window;\n"
<< "}\n"
<< endl;
	}

/** Example of processing positional arguments
 */
	if (m_with_example) {
		sink_
<< "void\n"
<< "" << projname_ << "::\n"
<< "pos_arg (const char* arg_)\n"
<< "{\n"
<< mytab << "trace(\"" << projname_ << "::pos_arg\");\n"
<< '\n'
<< mytab << "switch(m_pos_arg_count) {\n"
<< mytab << "case 0:\n"
<< mytab << "    m_pos_arg1 = arg_;\n"
<< mytab << "    break;\n"
<< "\n"
<< mytab << "case 1:\n"
<< mytab << "default:\n"
<< mytab << "    DL((ASSA::ASSAERR,\"Error: unexpected argument '%s'\\n\", arg_));\n"
<< mytab << "    DL((ASSA::ASSAERR,\"Try `app-name --help` for more information.\\n\"));\n"
<< mytab << "    Assure_exit (false);\n"
<< mytab << "}\n"
<< mytab << "m_pos_arg_count++;\n"
<< "}\n";
	}

/** Idle callback to run ASSA event loop
 */
	if (m_gtk2_app) {
		sink_

<< "bool\n"
<<  projname_ << "::\n"
<< "timer_cb ()\n"
<< "{\n"
<< mytab << "ASSA::TimeVal timeout (m_timeout);\n"
<< mytab << "REACTOR->waitForEvents (&timeout);\n"
<< mytab << "return true;\n"
<< "}\n";
	}

/** Service initialization
 */
	if (m_gtk2_app) {
		sink_
<< "\n"
<< "void\n"
<< "" << projname_ << "::\n"
<< "init_service ()\n"
<< "{\n"
<< mytab << "trace_with_mask(\"" << projname_ << "::init_service\"" << ",GUITRACE);\n"
<< "\n"
<< mytab << "ASSA::Log::disable_timestamp ();\n"
<< mytab << "int gtk_argc = 0;\n"
<< mytab << "char** gtk_argv = NULL;\n"
<< "\n"
<< mytab << "m_gtk_options = \"" << projname_ << " --g-fatal-warnings \" + m_gtk_options;\n"
<< mytab << "CmdLineOpts::str_to_argv (m_gtk_options, gtk_argc, gtk_argv);\n"
<< "\n"
<< mytab << "m_kit = new Gtk::Main (&gtk_argc, &gtk_argv);\n"
<< "\n"
<< mytab << "CmdLineOpts::free_argv (gtk_argv);\n"
<< "\n"
<< mytab << "m_main_window = new MainWindow;\n"
<< "\n"
<< mytab << "sigc::slot0<bool> tslot = sigc::mem_fun (*this, &" << projname_ << "::timer_cb);\n"
<< mytab << "m_tconn = Glib::signal_timeout ().connect (tslot, IDLE_TIMEOUT); \n"
<< "\n"
<< mytab << "DL((ASSA::APP,\"Service has been initialized\\n\"));\n"
<< "}\n"
<< "\n";
	}
	else {
		sink_
<< "\n"
<< "void\n"
<< "" << projname_ << "::\n"
<< "init_service ()\n"
<< "{\n"
<< mytab << "trace(\"" << projname_ << "::init_service\");\n"
<< "\n"
<< mytab << "//\n"
<< mytab << "// Insert initialization code here\n"
<< mytab << "//\n"
<< "\n"
<< mytab << "DL((ASSA::APP,\"Service has been initialized\\n\"));\n"
<< "}\n"
<< "\n";
	}

/** Service processing - main event loop
 */

	if (m_gtk2_app) {
		sink_
<< "void\n"
<< projname_ << "::\n"
<< "process_events ()\n"
<< "{\n"
<< mytab << "trace_with_mask(\"" << projname_ << "::process_events\",GUITRACE);\n"
<< "\n"
<< mytab << "m_kit->run (*m_main_window);\n"
<< "\n"
<< mytab << "DL((ASSA::APP,\"Service stopped!\\n\"));\n"
<< "}\n";
	}
	else {
		sink_ 
<< "void\n"
<< projname_ << "::\n"
<< "process_events ()\n"
<< "{\n"
<< mytab << "trace(\"" << projname_ << "::process_events\");\n"
<< "\n"
<< mytab << "while (service_is_active ()) {\n"
<< mytab << "    m_reactor.waitForEvents ();\n"
<< mytab << "}\n"
<< "\n"
<< mytab << "/** Shut the service down\n"
<< mytab << "*/\n"
<< mytab << "m_reactor.stopReactor ();\n"
<< "\n"
<< mytab << "DL((ASSA::APP,\"Service stopped!\\n\"));\n"
<< "}\n"
	  << endl;
	}
}

void
Genesis::
write_main (const char* projname_, ostream& msrc_)
{
	string PROJNAME (projname_);

	for (int i = 0; i < PROJNAME.length (); i++) {
		PROJNAME [i] = toupper (PROJNAME [i]);
	}

    msrc_ << endl

<< endl
<< "int\n"
<< "main (int argc, char* argv[])\n"
<< "{\n"
<< mytab << "static char release[128];\n"
<< mytab << "int patch_level = 0;\n"
<< "\n"
<< mytab << "snprintf (release, 127, \"%d.%d.%d-%d\",\n"
<< mytab << mytab << mytab << "  APP_MAJOR_VERSION,\n"
<< mytab << mytab << mytab << "  APP_MINOR_VERSION,\n"
<< mytab << mytab << mytab << "  APP_MICRO_VERSION,\n"
<< mytab << mytab << mytab << "  patch_level);\n"
<< "\n"
<< mytab << PROJNAME << "->set_version (release, patch_level);\n"
<< mytab << PROJNAME << "->set_author  (\"YOUR-NAME\");\n"
<< mytab << PROJNAME << "->set_flags   (ASSA::GenServer::RMLOG);\n"
<< "\n"
<< mytab << PROJNAME << "->init (&argc, argv, help_msg);\n"
<< " \n"
<< mytab << PROJNAME << "->init_service ();\n"
<< mytab << PROJNAME << "->process_events ();\n"
<< "\n"
<< mytab << "return " << PROJNAME << "->get_exit_value ();\n"
<< "}\n";

}

/*-----------------------------------------------------------------*/
/*      MAIN                                                       */
/*-----------------------------------------------------------------*/
int                             
main (int argc, char* argv[])   
{ 
    static char release[128];
    int patch_level = 0;

    snprintf (release, 127, "%d.%d.%d-%d",
			  ASSA_MAJOR_VERSION,
			  ASSA_MINOR_VERSION,
			  ASSA_MICRO_VERSION,
			  patch_level);

	sprintf (assa_version, "%d.%d", ASSA_MAJOR_VERSION, ASSA_MINOR_VERSION);

	ASSA::TimeVal tv_dd_yy = ASSA::TimeVal::gettimeofday ();

	date_year = tv_dd_yy.fmtString ("%B %d %Y");
	year = tv_dd_yy.fmtString ("%Y");
	
	Genesis& server = *Genesis::get_instance ();  
	
	server.set_version (release, patch_level);        
	server.set_author  ("Vladislav Grinchenko");    
	server.init        (&argc, argv, help_msg);       
	
	server.init_service   ();                          
	server.process_events ();                          
	
	return 0;                                         
} 

void
Genesis::
write_onefile_header (ostream& sink_)
{
	sink_ << endl

<< "#include <assa/Assure.h>\n";

	if (m_gtk2_app) {
		sink_	<< endl
<< "enum { GUITRACE = ASSA::USR1 };\n";
	}

	sink_ << endl;
}

