/**
 * Copyright (c) Members of the EGEE Collaboration. 2004-2010.
 * See http://www.eu-egee.org/partners/ for details on the copyright
 * holders.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 *
 *  Authors:
 *  2009-
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     Mischa Sall\'e <msalle@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *     <grid-mw-security@nikhef.nl>
 *
 *  2007-2009
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 *  2003-2007
 *     Martijn Steenbakkers <martijn@nikhef.nl>
 *     Gerben Venekamp <venekamp@nikhef.nl>
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 */


/*!
    \file   lcmaps_x509_utils.c
    \brief  This file contains utility functions needed for x509 credential
            handling (openssl)
    \author Martijn Steenbakkers for EGEE
    This file contains utility functions needed for x509 credential
    handling (openssl).
    Contains code contributed by Andrew McNab.

*/

/******************************************************************************
                             Include header files
******************************************************************************/

#if 0
#define DEBUG 0
#define HACK 0
#endif

/* Needed for NULL */
#include <stdio.h>

/* For X509 and STACK_OF(X509) structs (output) */
#include <openssl/x509.h>
#include <openssl/x509v3.h>
#include <openssl/x509_vfy.h>
#include <openssl/pem.h>
#include <openssl/bio.h>

/* LCMAPS includes */
#include "_lcmaps_x509_utils.h"
#include "lcmaps_log.h"

/******************************************************************************
                             Defines
******************************************************************************/



/******************************************************************************
Function:       lcmaps_x509_free_chain()
Description:    Properly free a previously allocated STACK_OF(X509) *chain
                NOTE: we need a **chain, otherwise NULL is not returned...
Parameters:
                chain: pointer to STACK_OF(X509)
******************************************************************************/
void lcmaps_x509_free_chain(
        STACK_OF(X509) **  chain
)
{
     if (chain && *chain) {
         sk_X509_pop_free(*chain, X509_free);
         *chain = NULL;
     }
}

/******************************************************************************
Function:       lcmaps_x509_to_dn()
Description:    Get the name of the X509 cert
Parameters:
                px509: x509 certificate
Returns:        dn (which should be freed)
******************************************************************************/
char * lcmaps_x509_to_dn(
        X509 *  px509
)
{
    if (!px509)
    {
        lcmaps_log(LOG_ERR, "%s(): No certificate found as input.\n", __func__);
        return NULL;
    }
    return X509_NAME_oneline(X509_get_subject_name(px509),NULL,0);
}
/******************************************************************************
Function:       lcmaps_x509_chain_to_dn()
Description:    Get the base DN of the X509 proxy cert
                Watch out! This function changes the subject DN in the X509 structure!
Parameters:
                px509: x509 certificate
                certstack: certificate chain
Returns:        dn (which should be freed)
******************************************************************************/
char * lcmaps_x509_chain_to_dn(
        X509 *  px509, STACK_OF(X509) *certstack
)
{
    return cgul_x509_chain_to_subject_dn(certstack);
}

/******************************************************************************
Function:       lcmaps_pem_string_to_x509
Description:    Reads a X509 certificate from a PEM-encoded string and returns
                the first X509 in the stack, which is the last certificate in
                the chain (EEC or final/leaf certificate/delegation)
Parameters:
                px: pointer to an allocated X509 certificate. (needs free)
                cert_string: PEM-encoded string
Returns:        0 on success, non-zero otherwise
******************************************************************************/
int lcmaps_pem_string_to_x509(X509 **px, char *certstring)
{
    /* int rval = 0; */
    STACK_OF (X509) * chain = NULL;
    X509 *cert = NULL;

    if (!px) {
        lcmaps_log(LOG_ERR, "%s: Error: No output pointer provided.\n", __func__);
        return -1;
    }

    if (lcmaps_pem_string_to_x509_chain(&chain, certstring) == 0) {
        /* Getting the final proxy (the Leaf proxy) */
        cert = sk_X509_value(chain, 0);
        *px = X509_dup(cert);
        if (!(*px)) {
            lcmaps_x509_free_chain(&chain); /* note chain->NULL */
            return -1;
        }
        lcmaps_x509_free_chain(&chain); /* note chain->NULL */
    } else {
        return -1;
    }
    return 0;
}

/******************************************************************************
Function:       lcmaps_pem_string_to_x509_chain
Description:    Creates a dynamically allocated stack of X509 certificate objects
                by walking through the PEM-encoded X509 certificates.
                Copied from gridsite (Andrew McNab), original name:
                    GRSTx509StringToChain()
Parameters:
                certstack: pointer to a stack of X509 certificates
                cert_string: PEM-encoded string
Returns:        0 on success, non-zero otherwise
******************************************************************************/
int lcmaps_pem_string_to_x509_chain(STACK_OF(X509) **certstack, char *certstring)
{
    STACK_OF(X509_INFO) *sk=NULL;
    BIO *certbio = NULL;
    X509_INFO *xi;

    *certstack = sk_X509_new_null();
    if (*certstack == NULL) return -1;

    certbio = BIO_new_mem_buf(certstring, -1);
    sk=PEM_X509_INFO_read_bio(certbio, NULL, NULL, NULL);
    BIO_free(certbio);
    if (sk==NULL)
    {
        lcmaps_x509_free_chain( certstack ); /* *certstack ->NULL*/
        return 1;
    }

    while (sk_X509_INFO_num(sk))
    {
        xi=sk_X509_INFO_shift(sk);
        if (xi->x509 != NULL)
        {
            sk_X509_push(*certstack, xi->x509);
            xi->x509=NULL;
        }
        X509_INFO_free(xi);
    }
    sk_X509_INFO_free(sk);

    if (!sk_X509_num(*certstack))
    {
        lcmaps_x509_free_chain( certstack ); /* *certstack -> NULL */
        return 1;
    }

   return 0;
}


/******************************************************************************
Function:   lcmaps_x509IsCA
Description:
    Tests if the X509 * cert is a CA certificate or not
Parameters:
    A X509 pointer
Returns:
    0      : Not a CA cert
    1      : This is a CA cert
******************************************************************************/
int lcmaps_x509IsCA (X509 *cert)
{
    return cgul_x509IsCA (cert);
}


/******************************************************************************
Function:   cgul_x509IsCA
Description:
    Tests if the X509 * cert is a CA certificate or not
Parameters:
    A X509 pointer
Returns:
    0      : Not a CA cert
    1      : This is a CA cert
******************************************************************************/
int cgul_x509IsCA (X509 *cert)
{
   int purpose_id;

   purpose_id = X509_PURPOSE_get_by_sname("sslclient");

   /* final argument to X509_check_purpose() is whether to check for CAness */

   if (X509_check_purpose(cert, purpose_id + X509_PURPOSE_MIN, 1))
        return 1;
   else return 0;
}


/******************************************************************************
Function:       cgul_x509_select_eec_from_chain()
Description:    Get the End-Entity Certificate from a certificate chain
Parameters:
                certstack: certificate chain
Returns:        pointer to X509 EEC
******************************************************************************/
X509 * cgul_x509_select_eec_from_chain (STACK_OF(X509) * chain)
{
    int amount_of_CAs = 0;
    int i             = 0;
    int depth         = 0;

    /* I must have a certificate chain to work with */
    if (!chain)
    {
        lcmaps_log(LOG_DEBUG, "%s: Empty chain, nothing to do\n", __func__);
        return NULL;
    }
    else
    {
        /* How many elements are there to the chain? */
        depth = sk_X509_num(chain);


        /* How many CA certs are there in the chain? */
        for (i = 0; i < depth; i++)
        {
            if (cgul_x509IsCA(sk_X509_value(chain, i)))
                amount_of_CAs++;
        }

        /* Getting the user cert (filtering the CAs and such) */
        i = depth - (amount_of_CAs + 1);

        /* Next check also catches depth==0 */
        if (i < 0)
        {
            /* Counting the stack went wrong. Errornous stack? */
            return NULL;
        }

        /* The End-Entity Certificate */
        return sk_X509_value(chain, i);
    }
}


/******************************************************************************
Function:       cgul_x509_select_final_cert_from_chain()
Description:    Get the Final Delegation from a certificate chain
Parameters:
                certstack: certificate chain
Returns:        pointer to X509 final delegation
******************************************************************************/
X509 * cgul_x509_select_final_cert_from_chain (STACK_OF(X509) * chain)
{
    int amount_of_CAs = 0;
    int i             = 0;
    int depth         = 0;

    /* I must have a certificate chain to work with */
    if (!chain)
    {
        return NULL;
    }
    else
    {
        /* How many elements are there to the chain? */
        depth = sk_X509_num(chain);

        /* How many CA certs are there in the chain? */
        for (i = 0; i < depth; i++)
        {
            if (cgul_x509IsCA(sk_X509_value(chain, i)))
                amount_of_CAs++;
        }

        /* Must have at least an EEC, possibly a delegation */
        if (depth > amount_of_CAs)
        {
            /* Selecting EEC or final delegation */
            return sk_X509_value(chain, 0);
        }
        else
        {
            /* Only CA files found */
            return NULL;
        }
    }
}

/******************************************************************************
Function:       cgul_x509_chain_to_subject_dn()
Description:    Get the Subject DN of the End-Entity Certificate of the X509 cert
Parameters:
                certstack: certificate chain
Returns:        Subject DN string (which should be freed)
******************************************************************************/
char * cgul_x509_chain_to_subject_dn(STACK_OF(X509) * chain)
{
    X509 * eec_cert = NULL;

    /* I need input */
    if (chain == NULL) {
        return NULL;
    }

    /* Select the End-Entity Certificate from the certificate chain */
    eec_cert = cgul_x509_select_eec_from_chain (chain);
    if (eec_cert == NULL) {
        lcmaps_log(LOG_DEBUG, "%s: No EEC found in the certificate chain.\n", __func__);
        return NULL;
    } else {
        /* You must FREE the DN after use */
        return X509_NAME_oneline(X509_get_subject_name(eec_cert), NULL, 0);
    }
}

/******************************************************************************
Function:       cgul_x509_chain_to_issuer_dn()
Description:    Get the Issuer DN of the End-Entity Certificate of the X509 cert
Parameters:
                certstack: certificate chain
Returns:        Issuer DN string (which should be freed)
******************************************************************************/
char * cgul_x509_chain_to_issuer_dn(STACK_OF(X509) * chain)
{
    X509 * eec_cert = NULL;

    if ((eec_cert = cgul_x509_select_eec_from_chain (chain)) == NULL)
    {
        return NULL;
    }
    else
    {
        /* You must FREE the Issuer DN after use */
        return X509_NAME_oneline(X509_get_issuer_name (eec_cert), NULL, 0);
    }
}

/******************************************************************************
CVS Information:
    $Source: /srv/home/dennisvd/svn/mw-security/lcmaps/src/grid_credential_handling/x509_handling/lcmaps_x509_utils.c,v $
    $Date: 2015-01-13 12:28:19 +0100 (Tue, 13 Jan 2015) $
    $Revision: 18158 $
    $Author: msalle $
******************************************************************************/
