package agentkapp

import (
	"context"
	"errors"
	"net"

	"github.com/ash2k/stager"
	grpc_validator "github.com/grpc-ecosystem/go-grpc-middleware/v2/interceptors/validator"
	"gitlab.com/gitlab-org/cluster-integration/gitlab-agent/v16/internal/module/modshared"
	"gitlab.com/gitlab-org/cluster-integration/gitlab-agent/v16/internal/tool/grpctool"
	"gitlab.com/gitlab-org/cluster-integration/gitlab-agent/v16/internal/tool/logz"
	"go.opentelemetry.io/contrib/instrumentation/google.golang.org/grpc/otelgrpc"
	otelmetric "go.opentelemetry.io/otel/metric"
	"go.opentelemetry.io/otel/propagation"
	"go.opentelemetry.io/otel/trace"
	"go.uber.org/zap"
	"google.golang.org/grpc"
	"google.golang.org/grpc/credentials/insecure"
)

type internalServer struct {
	server        *grpc.Server
	inMemConn     *grpc.ClientConn
	inMemListener net.Listener
}

func newInternalServer(log *zap.Logger, tp trace.TracerProvider, mp otelmetric.MeterProvider, p propagation.TextMapPropagator,
	streamProm grpc.StreamServerInterceptor, unaryProm grpc.UnaryServerInterceptor) (*internalServer, error) {
	// Internal gRPC client->listener pipe
	listener := grpctool.NewDialListener()

	// Construct connection to internal gRPC server
	conn, err := grpc.NewClient("passthrough:pipe",
		grpc.WithSharedWriteBuffer(true),
		grpc.WithContextDialer(listener.DialContext),
		grpc.WithTransportCredentials(insecure.NewCredentials()),
		grpc.WithDefaultCallOptions(grpc.ForceCodec(grpctool.RawCodec{})),
	)
	if err != nil {
		return nil, err
	}
	factory := func(ctx context.Context, method string) modshared.RPCAPI {
		return &agentRPCAPI{
			RPCAPIStub: modshared.RPCAPIStub{
				Logger:    log.With(logz.TraceIDFromContext(ctx)),
				StreamCtx: ctx,
			},
		}
	}
	return &internalServer{
		server: grpc.NewServer(
			grpc.StatsHandler(otelgrpc.NewServerHandler(
				otelgrpc.WithTracerProvider(tp),
				otelgrpc.WithMeterProvider(mp),
				otelgrpc.WithPropagators(p),
				otelgrpc.WithMessageEvents(otelgrpc.ReceivedEvents, otelgrpc.SentEvents),
			)),
			grpc.StatsHandler(grpctool.ServerNoopMaxConnAgeStatsHandler{}),
			grpc.SharedWriteBuffer(true),
			grpc.ChainStreamInterceptor(
				streamProm, // 1. measure all invocations
				modshared.StreamRPCAPIInterceptor(factory), // 2. inject RPC API
				grpc_validator.StreamServerInterceptor(),   // x. wrap with validator
			),
			grpc.ChainUnaryInterceptor(
				unaryProm, // 1. measure all invocations
				modshared.UnaryRPCAPIInterceptor(factory), // 2. inject RPC API
				grpc_validator.UnaryServerInterceptor(),   // x. wrap with validator
			),
		),
		inMemConn:     conn,
		inMemListener: listener,
	}, nil
}

func (s *internalServer) Start(stage stager.Stage) {
	grpctool.StartServer(stage, s.server, func() (net.Listener, error) {
		return s.inMemListener, nil
	}, func() {})
}

func (s *internalServer) Close() error {
	return errors.Join(
		s.inMemConn.Close(),     // first close the client
		s.inMemListener.Close(), // then close the listener (if not closed already)
	)
}
