#! /bin/sh
# test-driver - basic testsuite driver script. Modified for vkd3d tests.

scriptversion=2024-08-05.01; # UTC

# This is a modified version of the test_driver script provided by
# auto-tools, whose licence is as follows:
#
# Copyright (C) 2011-2021 Free Software Foundation, Inc.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2, or (at your option)
# any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <https://www.gnu.org/licenses/>.

# As a special exception to the GNU General Public License, if you
# distribute this file as part of a program that contains a
# configuration script generated by Autoconf, you may include it under
# the same distribution terms that you use for the rest of that program.

# This file is maintained in Automake, please report
# bugs to <bug-automake@gnu.org> or send patches to
# <automake-patches@gnu.org>.

# Make unconditional expansion of undefined variables an error.  This
# helps a lot in preventing typo-related bugs.
set -u

usage_error ()
{
  echo "$0: $*" >&2
  print_usage >&2
  exit 2
}

print_usage ()
{
  cat <<END
Usage:
  test-driver --test-name NAME --log-file PATH --trs-file PATH
              [--expect-failure {yes|no}] [--color-tests {yes|no}]
              [--enable-hard-errors {yes|no}] [--]
              TEST-SCRIPT [TEST-SCRIPT-ARGUMENTS]

The '--test-name', '--log-file' and '--trs-file' options are mandatory.
See the GNU Automake documentation for information.
END
}

test_name= # Used for reporting.
log_file=  # Where to save the output of the test script.
trs_file=  # Where to save the metadata of the test run.
expect_failure=no
color_tests=no
enable_hard_errors=yes
while test $# -gt 0; do
  case $1 in
  --help) print_usage; exit $?;;
  --version) echo "test-driver $scriptversion"; exit $?;;
  --test-name) test_name=$2; shift;;
  --log-file) log_file=$2; shift;;
  --trs-file) trs_file=$2; shift;;
  --color-tests) color_tests=$2; shift;;
  --expect-failure) expect_failure=$2; shift;;
  --enable-hard-errors) enable_hard_errors=$2; shift;;
  --) shift; break;;
  -*) usage_error "invalid option: '$1'";;
   *) break;;
  esac
  shift
done

missing_opts=
test x"$test_name" = x && missing_opts="$missing_opts --test-name"
test x"$log_file"  = x && missing_opts="$missing_opts --log-file"
test x"$trs_file"  = x && missing_opts="$missing_opts --trs-file"
if test x"$missing_opts" != x; then
  usage_error "the following mandatory options are missing:$missing_opts"
fi

if test $# -eq 0; then
  usage_error "missing argument"
fi

if test $color_tests = yes; then
  color_reset='[0m'
  color_dark_red='[31m'
  color_dark_purple='[35m'
  color_green='[32m'
  color_yellow='[33m'
  color_blue='[34m'
  color_bright_red='[1;91m'
  color_bright_purple='[1;95m'
  color_fade='[0;2m'
else
  color_reset=
  color_dark_red=
  color_dark_purple=
  color_green=
  color_yellow=
  color_blue=
  color_bright_red=
  color_bright_purple=
  color_fade=
fi

do_exit='rm -f $log_file $trs_file; (exit $st); exit $st'
trap "st=129; $do_exit" 1
trap "st=130; $do_exit" 2
trap "st=141; $do_exit" 13
trap "st=143; $do_exit" 15

# Test script is run here. We create the file first, then append to it,
# to ameliorate tests themselves also writing to the log file. Our tests
# don't, but others can (automake bug#35762).
: >"$log_file"
"$@" >>"$log_file" 2>&1
estatus=$?

if test $enable_hard_errors = no && test $estatus -eq 99; then
  tweaked_estatus=1
else
  tweaked_estatus=$estatus
fi

case $tweaked_estatus:$expect_failure in
  0:yes) col=$color_dark_red      res=XPASS recheck=yes gcopy=yes;;
  0:*)   col=$color_green         res=PASS  recheck=no  gcopy=no;;
  77:*)  col=$color_blue          res=SKIP  recheck=no  gcopy=yes;;
  99:*)  col=$color_bright_purple res=ERROR recheck=yes gcopy=yes;;
  *:yes) col=$color_yellow        res=XFAIL recheck=no  gcopy=yes;;
  *:*)   col=$color_bright_red    res=FAIL  recheck=yes gcopy=yes;;
esac

# Report the test outcome and exit status in the logs, so that one can
# know whether the test passed or failed simply by looking at the '.log'
# file, without the need of also peaking into the corresponding '.trs'
# file (automake bug#11814).
echo "$res $test_name (exit status: $estatus)" >>"$log_file"

# Give detailed report
awk_program=$(cat <<'EOF'
BEGIN {
    FS = ":"
}

/: Compiling [^[:space:]]+ shaders with [^[:space:]]+ and executing with [^[:space:]]+/ {
    n = split($3, arr, " ")
    shader_models = arr[2]
    backend = arr[n]
    gsub(/\./, "", backend)
    str = "<fade>(" shader_models ")" backend "<reset>"
    printf("#   %-20s\n", str)
    last_entry = ""
    last_tag = ""
}

function print_entry(line, model, tag) {
    if ($5 ~ " Model [^[:space:]]+")
        entry = "<fade>" $4 "+" substr($5, 8) "<reset>"
    else
        entry = "<fade>" $4 "<reset>"

    if (entry != last_entry)
        print entry tag
    else if (tag != last_tag)
        print tag

    last_entry = entry
    last_tag = tag
}

/: Test failed:/ {
    print_entry($4, $5, "[F]")
}

/: Todo:/ {
    print_entry($4, $5, "[XF]")
}

/: Todo succeeded:/ {
    print_entry($4, $5, "[XP]")
}

/: Test skipped:/ {
    print_entry($4, $5, "[SK]")
}

/: Assertion .* failed\./ {
    print "#    [AF] <fade>" $0 "<reset>"
}

/: Aborting, reached unreachable code\./ {
    print "#   [AF] <fade>" $0 "<reset>"
}

EOF
)

details=$(awk "$awk_program" "$log_file")

# In case of SIGABRT or SIGSEGV, add tag and print second to last line, containing the
#   "(core dumped)" message.
nxt_to_last_line=$(tail -n2 "$log_file" | head -n1)
if [ "$tweaked_estatus" -eq 134 ]; then
  details="$details#   [SIGABRT] <fade>$nxt_to_last_line<reset>"
elif [ "$tweaked_estatus" -eq 139 ]; then
  details="$details#   [SIGSEGV] <fade>$nxt_to_last_line<reset>"
fi

# Count number of [XF] tags.
xfcount=$(echo "$details" | awk '/\[XF\]/{count++} END{printf "%d", count}')

# Count number of [SK] tags.
skcount=$(echo "$details" | awk '/\[SK\]/{count++} END{printf "%d", count}')

details=$(echo "$details" |\
    tr '\n' ' ' |\
    tr '#' '\n' |\
    sed "s/\] \[/][/g" |\
    sed "s/\[F\]/$color_bright_red[F]$color_reset/g" |\
    sed "s/\[XF\]/$color_yellow[XF]$color_reset/g" |\
    sed "s/\[XP\]/$color_dark_red[XP]$color_reset/g" |\
    sed "s/\[SK\]/$color_blue[SK]$color_reset/g" |\
    sed "s/\[AF\]/$color_bright_purple[AF]$color_reset/g" |\
    sed "s/\[SIGABRT\]/$color_bright_purple[SIGABRT]$color_reset/g" |\
    sed "s/\[SIGSEGV\]/$color_bright_purple[SIGSEGV]$color_reset/g" |\
    sed "s/<fade>/$color_fade/g" |\
    sed "s/<reset>/$color_reset/g" |\
    awk 'NF != 1' )

# Set VKD3D_TEST_DETAILED to 0 if unset.
VKD3D_TEST_DETAILED=${VKD3D_TEST_DETAILED:-0}

# If the test passes, we will omit details but report number of [XF] and [SK]
if [ "$res" = "PASS" ] && [ "$VKD3D_TEST_DETAILED" != "1" ]; then
  details=""
  if [ "$xfcount" -gt 0 ]; then
    details="$details $color_yellow($xfcount XF)$color_reset"
  fi
  if [ "$skcount" -gt 0 ]; then
    details="$details $color_blue($skcount SK)$color_reset"
  fi
fi

# Report outcome to console.
echo "${col}${res}${color_reset}: $test_name""$details"

# Register the test result, and other relevant metadata.
echo ":test-result: $res" > $trs_file
echo ":global-test-result: $res" >> $trs_file
echo ":recheck: $recheck" >> $trs_file
echo ":copy-in-global-log: $gcopy" >> $trs_file

# Local Variables:
# mode: shell-script
# sh-indentation: 2
# eval: (add-hook 'before-save-hook 'time-stamp)
# time-stamp-start: "scriptversion="
# time-stamp-format: "%:y-%02m-%02d.%02H"
# time-stamp-time-zone: "UTC0"
# time-stamp-end: "; # UTC"
# End:
