const char datetime_lua[] =
"local ffi = require('ffi')\n"
"local buffer = require('buffer')\n"
"local tz = require('timezones')\n"
"\n"
"--[[\n"
"    `c-dt` library functions handles properly both positive and negative `dt`\n"
"    values, where `dt` is a number of dates since Rata Die date (0001-01-01).\n"
"\n"
"    `c-dt` uses 32-bit integer number to store `dt` values, so range of\n"
"    suported dates is limited to dates from -5879610-06-22 (INT32_MIN) to\n"
"    +5879611-07-11 (INT32_MAX).\n"
"\n"
"    For better compactness of our typical data in MessagePack stream we shift\n"
"    root of our time to the Unix Epoch date (1970-01-01), thus our 0 is\n"
"    actually dt = 719163.\n"
"\n"
"    So here is a simple formula how convert our epoch-based seconds to dt values\n"
"        dt = (secs / 86400) + 719163\n"
"    Where 719163 is an offset of Unix Epoch (1970-01-01) since Rata Die\n"
"    (0001-01-01) in dates.\n"
"]]\n"
"\n"
"ffi.cdef[[\n"
"\n"
"/* dt_core.h definitions */\n"
"typedef int dt_t;\n"
"\n"
"typedef enum {\n"
"    DT_MON       = 1,\n"
"    DT_MONDAY    = 1,\n"
"    DT_TUE       = 2,\n"
"    DT_TUESDAY   = 2,\n"
"    DT_WED       = 3,\n"
"    DT_WEDNESDAY = 3,\n"
"    DT_THU       = 4,\n"
"    DT_THURSDAY  = 4,\n"
"    DT_FRI       = 5,\n"
"    DT_FRIDAY    = 5,\n"
"    DT_SAT       = 6,\n"
"    DT_SATURDAY  = 6,\n"
"    DT_SUN       = 7,\n"
"    DT_SUNDAY    = 7,\n"
"} dt_dow_t;\n"
"\n"
"dt_t   tnt_dt_from_rdn     (int n);\n"
"bool   tnt_dt_from_ymd_checked(int y, int m, int d, dt_t *val);\n"
"void   tnt_dt_to_ymd       (dt_t dt, int *y, int *m, int *d);\n"
"\n"
"dt_dow_t tnt_dt_dow        (dt_t dt);\n"
"\n"
"/* dt_util.h */\n"
"int     tnt_dt_days_in_month   (int y, int m);\n"
"\n"
"/* dt_accessor.h */\n"
"int     tnt_dt_year         (dt_t dt);\n"
"int     tnt_dt_month        (dt_t dt);\n"
"int     tnt_dt_doy          (dt_t dt);\n"
"int     tnt_dt_dom          (dt_t dt);\n"
"\n"
"dt_t   tnt_dt_add_months   (dt_t dt, int delta, dt_adjust_t adjust);\n"
"\n"
"/* dt_parse_iso.h definitions */\n"
"size_t tnt_dt_parse_iso_date(const char *str, size_t len, dt_t *dt);\n"
"size_t tnt_dt_parse_iso_zone_lenient(const char *str, size_t len, int *offset);\n"
"\n"
"/* Tarantool datetime functions - datetime.c */\n"
"size_t tnt_datetime_to_string(const struct datetime * date, char *buf,\n"
"                              ssize_t len);\n"
"size_t tnt_datetime_strftime(const struct datetime *date, char *buf,\n"
"                             uint32_t len, const char *fmt);\n"
"ssize_t tnt_datetime_parse_full(struct datetime *date, const char *str,\n"
"                                size_t len, const char *tzsuffix,\n"
"                                int32_t offset);\n"
"ssize_t tnt_datetime_parse_tz(const char *str, size_t len, time_t base,\n"
"                              int16_t *tzoffset, int16_t *tzindex);\n"
"size_t tnt_datetime_strptime(struct datetime *date, const char *buf,\n"
"                             const char *fmt);\n"
"void   tnt_datetime_now(struct datetime *now);\n"
"bool   tnt_datetime_totable(const struct datetime *date,\n"
"                            struct interval *out);\n"
"bool   tnt_datetime_isdst(const struct datetime *date);\n"
"\n"
"/* Tarantool interval support functions */\n"
"size_t tnt_interval_to_string(const struct interval *, char *, ssize_t);\n"
"int    tnt_datetime_increment_by(struct datetime *self, int direction,\n"
"                                 const struct interval *ival);\n"
"int    tnt_datetime_datetime_sub(struct interval *res,\n"
"                                 const struct datetime *lhs,\n"
"                                 const struct datetime *rhs);\n"
"\n"
"int    tnt_interval_interval_sub(struct interval *lhs,\n"
"                                 const struct interval *rhs);\n"
"int    tnt_interval_interval_add(struct interval *lhs,\n"
"                                 const struct interval *rhs);\n"
"\n"
"/* Tarantool timezone support */\n"
"enum {\n"
"    TZ_UTC = 0x01,\n"
"    TZ_RFC = 0x02,\n"
"    TZ_MILITARY = 0x04,\n"
"    TZ_AMBIGUOUS = 0x08,\n"
"    TZ_NYI = 0x10,\n"
"};\n"
"\n"
"]]\n"
"\n"
"local builtin = ffi.C\n"
"local math_modf = math.modf\n"
"local math_floor = math.floor\n"
"\n"
"-- Unix, January 1, 1970, Thursday\n"
"local DAYS_EPOCH_OFFSET = 719163\n"
"local SECS_PER_DAY      = 86400\n"
"local SECS_EPOCH_OFFSET = DAYS_EPOCH_OFFSET * SECS_PER_DAY\n"
"local TOSTRING_BUFSIZE  = 64\n"
"local IVAL_TOSTRING_BUFSIZE = 96\n"
"local STRFTIME_BUFSIZE  = 128\n"
"\n"
"-- minimum supported date - -5879610-06-22\n"
"local MIN_DATE_YEAR = -5879610\n"
"local MIN_DATE_MONTH = 6\n"
"local MIN_DATE_DAY = 22\n"
"-- maximum supported date - 5879611-07-11\n"
"local MAX_DATE_YEAR = 5879611\n"
"local MAX_DATE_MONTH = 7\n"
"local MAX_DATE_DAY = 11\n"
"-- In the Julian calendar, the average year length is\n"
"-- 365 1/4 days = 365.25 days. This gives an error of\n"
"-- about 1 day in 128 years.\n"
"local AVERAGE_DAYS_YEAR = 365.25\n"
"local AVERAGE_WEEK_YEAR = AVERAGE_DAYS_YEAR / 7\n"
"local INT_MAX = 2147483647\n"
"-- -5879610-06-22\n"
"local MIN_DATE_TEXT = ('%d-%02d-%02d'):format(MIN_DATE_YEAR, MIN_DATE_MONTH,\n"
"                                              MIN_DATE_DAY)\n"
"-- 5879611-07-11\n"
"local MAX_DATE_TEXT = ('%d-%02d-%02d'):format(MAX_DATE_YEAR, MAX_DATE_MONTH,\n"
"                                              MAX_DATE_DAY)\n"
"local MAX_YEAR_RANGE = MAX_DATE_YEAR - MIN_DATE_YEAR\n"
"local MAX_MONTH_RANGE = MAX_YEAR_RANGE * 12\n"
"local MAX_WEEK_RANGE = MAX_YEAR_RANGE * AVERAGE_WEEK_YEAR\n"
"local MAX_DAY_RANGE = MAX_YEAR_RANGE * AVERAGE_DAYS_YEAR\n"
"local MAX_HOUR_RANGE = MAX_DAY_RANGE * 24\n"
"local MAX_MIN_RANGE = MAX_HOUR_RANGE * 60\n"
"local MAX_SEC_RANGE = MAX_DAY_RANGE * SECS_PER_DAY\n"
"local MAX_NSEC_RANGE = INT_MAX\n"
"local MAX_USEC_RANGE = math_floor(MAX_NSEC_RANGE / 1e3)\n"
"local MAX_MSEC_RANGE = math_floor(MAX_NSEC_RANGE / 1e6)\n"
"local DEF_DT_ADJUST = builtin.DT_LIMIT\n"
"\n"
"local date_tostr_stash =\n"
"    buffer.ffi_stash_new(string.format('char[%s]', TOSTRING_BUFSIZE))\n"
"local date_tostr_stash_take = date_tostr_stash.take\n"
"local date_tostr_stash_put = date_tostr_stash.put\n"
"\n"
"local ival_tostr_stash =\n"
"    buffer.ffi_stash_new(string.format('char[%s]', IVAL_TOSTRING_BUFSIZE))\n"
"local ival_tostr_stash_take = ival_tostr_stash.take\n"
"local ival_tostr_stash_put = ival_tostr_stash.put\n"
"\n"
"local date_strf_stash =\n"
"    buffer.ffi_stash_new(string.format('char[%s]', STRFTIME_BUFSIZE))\n"
"local date_strf_stash_take = date_strf_stash.take\n"
"local date_strf_stash_put = date_strf_stash.put\n"
"\n"
"local date_dt_stash = buffer.ffi_stash_new('dt_t[1]')\n"
"local date_dt_stash_take = date_dt_stash.take\n"
"local date_dt_stash_put = date_dt_stash.put\n"
"\n"
"local date_int16_stash = buffer.ffi_stash_new('int16_t[1]')\n"
"local date_int16_stash_take = date_int16_stash.take\n"
"local date_int16_stash_put = date_int16_stash.put\n"
"\n"
"local datetime_t = ffi.typeof('struct datetime')\n"
"local interval_t = ffi.typeof('struct interval')\n"
"\n"
"local function is_interval(o)\n"
"    return ffi.istype(interval_t, o)\n"
"end\n"
"\n"
"local function is_datetime(o)\n"
"    return ffi.istype(datetime_t, o)\n"
"end\n"
"\n"
"local function is_table(o)\n"
"    return type(o) == 'table'\n"
"end\n"
"\n"
"local function check_date(o, message)\n"
"    if not is_datetime(o) then\n"
"        return error((\"%s: expected datetime, but received %s\"):\n"
"                     format(message, type(o)), 2)\n"
"    end\n"
"end\n"
"\n"
"local function check_date_interval(o, message)\n"
"    if not is_datetime(o) and not is_interval(o) then\n"
"        return error((\"%s: expected datetime or interval, but received %s\"):\n"
"                     format(message, type(o)), 2)\n"
"    end\n"
"end\n"
"\n"
"local function check_interval(o, message)\n"
"    if not is_interval(o) then\n"
"        return error((\"%s: expected interval, but received %s\"):\n"
"                     format(message, type(o)), 2)\n"
"    end\n"
"end\n"
"\n"
"local function check_interval_table(o, message)\n"
"    if not is_table(o) and not is_interval(o) then\n"
"        return error((\"%s: expected interval or table, but received %s\"):\n"
"                     format(message, type(o)), 2)\n"
"    end\n"
"end\n"
"\n"
"local function check_date_interval_table(o, message)\n"
"    if not is_table(o) and not is_datetime(o) and not is_interval(o) then\n"
"        return error((\"%s: expected datetime, interval or table, but received %s\"):\n"
"                     format(message, type(o)), 2)\n"
"    end\n"
"end\n"
"\n"
"local function check_table(o, message)\n"
"    if not is_table(o) then\n"
"        return error((\"%s: expected table, but received %s\"):\n"
"                     format(message, type(o)), 2)\n"
"    end\n"
"end\n"
"\n"
"local function check_str(s, message)\n"
"    if type(s) ~= 'string' then\n"
"        return error((\"%s: expected string, but received %s\"):\n"
"                     format(message, type(s)), 2)\n"
"    end\n"
"end\n"
"\n"
"local function check_integer(v, message)\n"
"    if v == nil then\n"
"        return\n"
"    end\n"
"    if type(v) ~= 'number' or v % 1 ~= 0 then\n"
"        error(('%s: integer value expected, but received %s'):\n"
"              format(message, type(v)), 4)\n"
"    end\n"
"end\n"
"\n"
"local function check_str_or_nil(s, message)\n"
"    if s ~= nil and type(s) ~= 'string' then\n"
"        return error((\"%s: expected string, but received %s\"):\n"
"                     format(message, type(s)), 2)\n"
"    end\n"
"end\n"
"\n"
"-- range may be of a form of pair {from, to} or\n"
"-- tuple {fom, to, -1 in extra}\n"
"-- -1 is a special value (so far) used for days only\n"
"local function check_range(v, from, to, txt, extra)\n"
"    if type(v) ~= 'number' then\n"
"        error(('numeric value expected, but received %s'):\n"
"              format(type(v)), 3)\n"
"    end\n"
"    if extra == v or (v >= from and v <= to) then\n"
"        return\n"
"    end\n"
"    if extra == nil then\n"
"        error(('value %d of %s is out of allowed range [%d, %d]'):\n"
"              format(v, txt, from, to), 3)\n"
"    else\n"
"        error(('value %d of %s is out of allowed range [%d, %d..%d]'):\n"
"              format(v, txt, extra, from, to), 3)\n"
"    end\n"
"end\n"
"\n"
"local function dt_from_ymd_checked(y, M, d)\n"
"    local pdt = date_dt_stash_take()\n"
"    local is_valid = builtin.tnt_dt_from_ymd_checked(y, M, d, pdt)\n"
"    if not is_valid then\n"
"        date_dt_stash_put(pdt)\n"
"        error(('date %4d-%02d-%02d is invalid'):format(y, M, d))\n"
"    end\n"
"    local dt = pdt[0]\n"
"    date_dt_stash_put(pdt)\n"
"    return dt\n"
"end\n"
"\n"
"-- check v value against maximum/minimum possible values\n"
"-- if there is nil then simply return default 0\n"
"local function checked_max_value(v, max, txt, def)\n"
"    if v == nil then\n"
"        return def\n"
"    end\n"
"    if type(v) ~= 'number' then\n"
"        error(('numeric value expected, but received %s'):\n"
"              format(type(v)), 2)\n"
"    end\n"
"    if v >= -max and v <= max then\n"
"        return v\n"
"    end\n"
"    error(('value %s of %s is out of allowed range [%s, %s]'):\n"
"            format(v, txt, -max, max), 4)\n"
"end\n"
"\n"
"local function bool2int(b)\n"
"    return b and 1 or 0\n"
"end\n"
"\n"
"local adjust_xlat = {\n"
"    none = builtin.DT_LIMIT,\n"
"    last = builtin.DT_SNAP,\n"
"    excess = builtin.DT_EXCESS,\n"
"}\n"
"\n"
"local function interval_init(year, month, week, day, hour, min, sec, nsec,\n"
"                             adjust)\n"
"    return ffi.new(interval_t, sec, min, hour, day, week, month, year, nsec,\n"
"                   adjust)\n"
"end\n"
"\n"
"local function interval_new_copy(obj)\n"
"    return interval_init(obj.year, obj.month, obj.week, obj.day, obj.hour,\n"
"                         obj.min, obj.sec, obj.nsec, obj.adjust)\n"
"end\n"
"\n"
"local function interval_decode_args(obj)\n"
"    if is_interval(obj) then\n"
"        return obj\n"
"    end\n"
"    local year = checked_max_value(obj.year, MAX_YEAR_RANGE, 'year', 0)\n"
"    check_integer(year, 'year')\n"
"    local month = checked_max_value(obj.month, MAX_MONTH_RANGE, 'month', 0)\n"
"    check_integer(month, 'month')\n"
"    local adjust = adjust_xlat[obj.adjust] or DEF_DT_ADJUST\n"
"\n"
"    local weeks = checked_max_value(obj.week, MAX_WEEK_RANGE, 'week', 0)\n"
"    local days = checked_max_value(obj.day, MAX_DAY_RANGE, 'day', 0)\n"
"    check_integer(days, 'day')\n"
"    local hours = checked_max_value(obj.hour, MAX_HOUR_RANGE, 'hour', 0)\n"
"    check_integer(hours, 'hour')\n"
"    local minutes = checked_max_value(obj.min, MAX_MIN_RANGE, 'min', 0)\n"
"    check_integer(minutes, 'min')\n"
"    local secs = checked_max_value(obj.sec, MAX_SEC_RANGE, 'sec', 0)\n"
"    check_integer(secs, 'sec')\n"
"\n"
"    local nsec = checked_max_value(obj.nsec, MAX_NSEC_RANGE, 'nsec')\n"
"    local usec = checked_max_value(obj.usec, MAX_USEC_RANGE, 'usec')\n"
"    local msec = checked_max_value(obj.msec, MAX_MSEC_RANGE, 'msec')\n"
"    local count_usec = bool2int(nsec ~= nil) + bool2int(usec ~= nil) +\n"
"                       bool2int(msec ~= nil)\n"
"    if count_usec > 1 then\n"
"        error('only one of nsec, usec or msecs may be defined '..\n"
"                'simultaneously', 3)\n"
"    end\n"
"    nsec = (msec or 0) * 1e6 + (usec or 0) * 1e3 + (nsec or 0)\n"
"\n"
"    return interval_init(year, month, weeks, days, hours, minutes, secs, nsec,\n"
"                         adjust)\n"
"end\n"
"\n"
"local function interval_new(obj)\n"
"    if obj == nil then\n"
"        return interval_init(0, 0, 0, 0, 0, 0, 0, 0, DEF_DT_ADJUST)\n"
"    end\n"
"    check_table(obj, 'interval.new()')\n"
"    return interval_decode_args(obj)\n"
"end\n"
"\n"
"-- convert from epoch related time to Rata Die related\n"
"local function local_rd(secs)\n"
"    return math_floor((secs + SECS_EPOCH_OFFSET) / SECS_PER_DAY)\n"
"end\n"
"\n"
"-- convert UTC seconds to local seconds, adjusting by timezone\n"
"local function local_secs(obj)\n"
"    return obj.epoch + obj.tzoffset * 60\n"
"end\n"
"\n"
"local function utc_secs(epoch, tzoffset)\n"
"    return epoch - tzoffset * 60\n"
"end\n"
"\n"
"local function time_delocalize(self)\n"
"    self.epoch = local_secs(self)\n"
"    self.tzoffset = 0\n"
"end\n"
"\n"
"local function time_localize(self, offset)\n"
"    self.epoch = utc_secs(self.epoch, offset)\n"
"    self.tzoffset = offset\n"
"end\n"
"\n"
"-- get epoch seconds, shift to the local timezone\n"
"-- adjust from 1970-related to 0000-related time\n"
"-- then return dt in those coordinates (number of days\n"
"-- since Rata Die date)\n"
"local function local_dt(obj)\n"
"    return builtin.tnt_dt_from_rdn(local_rd(local_secs(obj)))\n"
"end\n"
"\n"
"local function datetime_cmp(lhs, rhs, is_raising)\n"
"    if not is_datetime(lhs) or not is_datetime(rhs) then\n"
"        if is_raising then\n"
"            error('incompatible types for datetime comparison', 3)\n"
"        else\n"
"            return nil\n"
"        end\n"
"    end\n"
"    local sdiff = lhs.epoch - rhs.epoch\n"
"    return sdiff ~= 0 and sdiff or (lhs.nsec - rhs.nsec)\n"
"end\n"
"\n"
"local function datetime_eq(lhs, rhs)\n"
"    return datetime_cmp(lhs, rhs, false) == 0\n"
"end\n"
"\n"
"local function datetime_lt(lhs, rhs)\n"
"    return datetime_cmp(lhs, rhs, true) < 0\n"
"end\n"
"\n"
"local function datetime_le(lhs, rhs)\n"
"    return datetime_cmp(lhs, rhs, true) <= 0\n"
"end\n"
"\n"
"--[[\n"
"    parse_tzoffset accepts time-zone strings in both basic\n"
"    and extended iso-8601 formats.\n"
"\n"
"    Basic    Extended\n"
"    Z        N/A\n"
"    +hh      N/A\n"
"    -hh      N/A\n"
"    +hhmm    +hh:mm\n"
"    -hhmm    -hh:mm\n"
"\n"
"    Returns timezone offset in minutes if string was accepted\n"
"    by parser, otherwise raise an error.\n"
"]]\n"
"local function parse_tzoffset(str)\n"
"    local offset = ffi.new('int[1]')\n"
"    local len = builtin.tnt_dt_parse_iso_zone_lenient(str, #str, offset)\n"
"    if len ~= #str then\n"
"        error(('invalid time-zone format %s'):format(str), 3)\n"
"    end\n"
"    return offset[0]\n"
"end\n"
"\n"
"local function epoch_from_dt(dt)\n"
"    return (dt - DAYS_EPOCH_OFFSET) * SECS_PER_DAY\n"
"end\n"
"\n"
"-- Use Olson facilities to determine whether local time in obj is DST\n"
"local function datetime_isdst(obj)\n"
"    return builtin.tnt_datetime_isdst(obj)\n"
"end\n"
"\n"
"--[[\n"
"    Parse timezone name similar way as datetime_parse_full parse\n"
"    full literal.\n"
"]]\n"
"local function parse_tzname(base_epoch, tzname)\n"
"    check_str(tzname, 'parse_tzname()')\n"
"    local ptzindex = date_int16_stash_take()\n"
"    local ptzoffset = date_int16_stash_take()\n"
"    local len = builtin.tnt_datetime_parse_tz(tzname, #tzname, base_epoch,\n"
"                                              ptzoffset, ptzindex)\n"
"    if len > 0 then\n"
"        local tzoffset, tzindex = ptzoffset[0], ptzindex[0]\n"
"        date_int16_stash_put(ptzoffset)\n"
"        date_int16_stash_put(ptzindex)\n"
"        return tzoffset, tzindex\n"
"    end\n"
"    date_int16_stash_put(ptzoffset)\n"
"    date_int16_stash_put(ptzindex)\n"
"    if len == -builtin.TZ_NYI then\n"
"        error((\"could not parse '%s' - nyi timezone\"):format(tzname))\n"
"    elseif len == -builtin.TZ_AMBIGUOUS then\n"
"        error((\"could not parse '%s' - ambiguous timezone\"):format(tzname))\n"
"    else -- len <= 0\n"
"        error((\"could not parse '%s'\"):format(tzname))\n"
"    end\n"
"end\n"
"\n"
"local function datetime_new_raw(epoch, nsec, tzoffset, tzindex)\n"
"    local dt_obj = ffi.new(datetime_t)\n"
"    dt_obj.epoch = epoch\n"
"    dt_obj.nsec = nsec\n"
"    dt_obj.tzoffset = tzoffset\n"
"    dt_obj.tzindex = tzindex\n"
"    return dt_obj\n"
"end\n"
"\n"
"local function datetime_new_copy(obj)\n"
"    return datetime_new_raw(obj.epoch, obj.nsec, obj.tzoffset, obj.tzindex)\n"
"end\n"
"\n"
"local function datetime_new_dt(dt, secs, nanosecs, offset, tzindex)\n"
"    secs = secs or 0\n"
"    nanosecs = nanosecs or 0\n"
"    offset = offset or 0\n"
"    tzindex = tzindex or 0\n"
"    return datetime_new_raw(epoch_from_dt(dt) + secs - offset * 60, nanosecs,\n"
"                            offset, tzindex)\n"
"end\n"
"\n"
"local function get_timezone(offset, msg)\n"
"    if type(offset) == 'number' then\n"
"        return offset\n"
"    elseif type(offset) == 'string' then\n"
"        return parse_tzoffset(offset)\n"
"    else\n"
"        error(('%s: string or number expected, but received %s'):\n"
"              format(msg, offset), 3)\n"
"    end\n"
"end\n"
"\n"
"-- create datetime given attribute values from obj\n"
"local function datetime_new(obj)\n"
"    if obj == nil then\n"
"        return datetime_new_raw(0, 0, 0, 0)\n"
"    end\n"
"    check_table(obj, 'datetime.new()')\n"
"\n"
"    local ymd = false\n"
"    local hms = false\n"
"    local dt = DAYS_EPOCH_OFFSET\n"
"\n"
"    local y = obj.year\n"
"    if y ~= nil then\n"
"        check_range(y, MIN_DATE_YEAR, MAX_DATE_YEAR, 'year')\n"
"        ymd = true\n"
"    end\n"
"    local M = obj.month\n"
"    if M ~= nil then\n"
"        check_range(M, 1, 12, 'month')\n"
"        ymd = true\n"
"    end\n"
"    local d = obj.day\n"
"    if d ~= nil then\n"
"        check_range(d, 1, 31, 'day', -1)\n"
"        ymd = true\n"
"    end\n"
"    local h = obj.hour\n"
"    if h ~= nil then\n"
"        check_range(h, 0, 23, 'hour')\n"
"        hms = true\n"
"    end\n"
"    local m = obj.min\n"
"    if m ~= nil then\n"
"        check_range(m, 0, 59, 'min')\n"
"        hms = true\n"
"    end\n"
"    local s = obj.sec\n"
"    if s ~= nil then\n"
"        check_range(s, 0, 60, 'sec')\n"
"        hms = true\n"
"    end\n"
"\n"
"    local nsec, usec, msec = obj.nsec, obj.usec, obj.msec\n"
"    local count_usec = bool2int(nsec ~= nil) + bool2int(usec ~= nil) +\n"
"                       bool2int(msec ~= nil)\n"
"    if count_usec > 0 then\n"
"        if count_usec > 1 then\n"
"            error('only one of nsec, usec or msecs may be defined '..\n"
"                  'simultaneously', 2)\n"
"        end\n"
"        if usec ~= nil then\n"
"            check_range(usec, 0, 1e6, 'usec')\n"
"            nsec = usec * 1e3\n"
"        elseif msec ~= nil then\n"
"            check_range(msec, 0, 1e3, 'msec')\n"
"            nsec = msec * 1e6\n"
"        else\n"
"            check_range(nsec, 0, 1e9, 'nsec')\n"
"        end\n"
"    else\n"
"        nsec = 0\n"
"    end\n"
"    local ts = obj.timestamp\n"
"    if ts ~= nil then\n"
"        if ymd then\n"
"            error('timestamp is not allowed if year/month/day provided', 2)\n"
"        end\n"
"        if hms then\n"
"            error('timestamp is not allowed if hour/min/sec provided', 2)\n"
"        end\n"
"        if type(ts) ~= 'number' then\n"
"            error((\"bad timestamp ('number' expected, got '%s')\"):format(type(ts)))\n"
"        end\n"
"        local fraction\n"
"        s, fraction = math_modf(ts)\n"
"        -- In case of negative fraction part we should\n"
"        -- make it positive at the expense of the integer part.\n"
"        -- Code below expects that \"nsec\" value is always positive.\n"
"        if fraction < 0 then\n"
"            s = s - 1\n"
"            fraction = fraction + 1\n"
"        end\n"
"        -- if there are separate nsec, usec, or msec provided then\n"
"        -- timestamp should be integer\n"
"        if count_usec == 0 then\n"
"            nsec = fraction * 1e9\n"
"        elseif fraction ~= 0 then\n"
"            error('only integer values allowed in timestamp '..\n"
"                  'if nsec, usec, or msecs provided', 2)\n"
"        end\n"
"        hms = true\n"
"    end\n"
"\n"
"    local offset = obj.tzoffset\n"
"    if offset ~= nil then\n"
"        offset = get_timezone(offset, 'tzoffset')\n"
"        -- at the moment the range of known timezones is UTC-12:00..UTC+14:00\n"
"        -- https://en.wikipedia.org/wiki/List_of_UTC_time_offsets\n"
"        check_range(offset, -720, 840, 'tzoffset')\n"
"    end\n"
"\n"
"    -- .year, .month, .day\n"
"    if ymd then\n"
"        y = y or 1970\n"
"        M = M or 1\n"
"        d = d or 1\n"
"        if d < 0 then\n"
"            d = builtin.tnt_dt_days_in_month(y, M)\n"
"        elseif d > 28 then\n"
"            local day_in_month = builtin.tnt_dt_days_in_month(y, M)\n"
"            if d > day_in_month then\n"
"                error(('invalid number of days %d in month %d for %d'):\n"
"                    format(d, M, y), 3)\n"
"            end\n"
"        end\n"
"        dt = dt_from_ymd_checked(y, M, d)\n"
"    end\n"
"\n"
"    local tzindex = 0\n"
"    local tzname = obj.tz\n"
"    if tzname ~= nil then\n"
"        offset, tzindex = parse_tzname(epoch_from_dt(dt), tzname)\n"
"    end\n"
"\n"
"    -- .hour, .minute, .second\n"
"    local secs = 0\n"
"    if hms then\n"
"        secs = (h or 0) * 3600 + (m or 0) * 60 + (s or 0)\n"
"    end\n"
"\n"
"    return datetime_new_dt(dt, secs, nsec, offset or 0, tzindex)\n"
"end\n"
"\n"
"--[[\n"
"    Convert to text datetime values\n"
"\n"
"    - datetime will use ISO-8601 format:\n"
"        1970-01-01T00:00Z\n"
"        2021-08-18T16:57:08.981725+03:00\n"
"]]\n"
"local function datetime_tostring(self)\n"
"    local buff = date_tostr_stash_take()\n"
"    local len = builtin.tnt_datetime_to_string(self, buff, TOSTRING_BUFSIZE)\n"
"    assert(len < TOSTRING_BUFSIZE)\n"
"    local s = ffi.string(buff)\n"
"    date_tostr_stash_put(buff)\n"
"    return s\n"
"end\n"
"\n"
"--[[\n"
"    Convert to text interval values of different types\n"
"\n"
"    - depending on a values stored there generic interval\n"
"      values may be displayed in the following format:\n"
"        +12 secs\n"
"        -23 minutes, 0 seconds\n"
"        +12 hours, 23 minutes, 1 seconds\n"
"        -7 days, -23 hours, -23 minutes, -1 seconds\n"
"    - years will be displayed as\n"
"        +10 years\n"
"    - months will be displayed as:\n"
"        +2 months\n"
"]]\n"
"local function interval_tostring(self)\n"
"    check_interval(self, 'datetime.interval.tostring')\n"
"    local buff = ival_tostr_stash_take()\n"
"    local len = builtin.tnt_interval_to_string(self, buff, IVAL_TOSTRING_BUFSIZE)\n"
"    if len < IVAL_TOSTRING_BUFSIZE then\n"
"        local s = ffi.string(buff)\n"
"        ival_tostr_stash_put(buff)\n"
"        return s\n"
"    end\n"
"    -- slow path - reallocate for a fuller size, and then restart interval_to_string\n"
"    ival_tostr_stash_put(buff)\n"
"    buff = ffi.new('char[\?]', len + 1)\n"
"    builtin.tnt_interval_to_string(self, buff, len + 1)\n"
"    return ffi.string(buff)\n"
"end\n"
"\n"
"-- subtract/addition operation for date object and interval\n"
"local function datetime_increment_by(self, direction, ival)\n"
"    local rc = builtin.tnt_datetime_increment_by(self, direction, ival)\n"
"    if rc == 0 then\n"
"        return self\n"
"    end\n"
"    local operation = direction >= 0 and 'addition' or 'subtraction'\n"
"    if rc < 0 then\n"
"        error(('%s makes date less than minimum allowed %s'):\n"
"                format(operation, MIN_DATE_TEXT), 3)\n"
"    else -- rc > 0\n"
"        error(('%s makes date greater than maximum allowed %s'):\n"
"                format(operation, MAX_DATE_TEXT), 3)\n"
"    end\n"
"end\n"
"\n"
"local check_ranges = {\n"
"    [1] = {'year', MAX_YEAR_RANGE},\n"
"    [2] = {'month', MAX_MONTH_RANGE},\n"
"    [3] = {'week', MAX_WEEK_RANGE},\n"
"    [4] = {'day', MAX_DAY_RANGE},\n"
"    [5] = {'hour', MAX_HOUR_RANGE},\n"
"    [6] = {'min', MAX_MIN_RANGE},\n"
"    [7] = {'sec', MAX_SEC_RANGE},\n"
"    [8] = {'nsec', MAX_NSEC_RANGE},\n"
"}\n"
"\n"
"local function check_rc(rc, operation, obj)\n"
"    -- fast path\n"
"    if rc == 0 then\n"
"        return obj\n"
"    end\n"
"\n"
"    -- slow, error reporting path\n"
"    local index = rc < 0 and -rc or rc\n"
"    assert(index >= 1 and index <= 8)\n"
"    local txt, max = unpack(check_ranges[index])\n"
"    local v = obj[txt]\n"
"    error(('%s moves value %s of %s out of allowed range [%s, %s]'):\n"
"            format(operation, v, txt, -max, max), 3)\n"
"end\n"
"\n"
"-- subtract operation when left is date, and right is date\n"
"local function datetime_datetime_sub(lhs, rhs)\n"
"    local obj = interval_new()\n"
"    local rc = builtin.tnt_datetime_datetime_sub(obj, lhs, rhs)\n"
"    return check_rc(rc, 'subtraction', obj)\n"
"end\n"
"\n"
"-- subtract operation for both left and right operands are intervals\n"
"local function interval_interval_sub(lhs, rhs)\n"
"    local lobj = interval_decode_args(interval_new_copy(lhs))\n"
"    local robj = interval_decode_args(rhs)\n"
"    local rc = builtin.tnt_interval_interval_sub(lobj, robj)\n"
"    return check_rc(rc, 'subtraction', lobj)\n"
"end\n"
"\n"
"-- addition operation for both left and right operands are intervals\n"
"local function interval_interval_add(lhs, rhs)\n"
"    local lobj = interval_decode_args(interval_new_copy(lhs))\n"
"    local robj = interval_decode_args(rhs)\n"
"    local rc = builtin.tnt_interval_interval_add(lobj, robj)\n"
"    return check_rc(rc, 'addition', lobj)\n"
"end\n"
"\n"
"local function date_first(lhs, rhs)\n"
"    if is_datetime(rhs) then\n"
"        return rhs, lhs\n"
"    else\n"
"        return lhs, rhs\n"
"    end\n"
"end\n"
"\n"
"local function error_incompatible(name)\n"
"    error((\"datetime:%s() - incompatible type of arguments\"):\n"
"          format(name), 3)\n"
"end\n"
"\n"
"--[[\n"
"Matrix of subtraction operands eligibility and their result type\n"
"\n"
"|                 | datetime | interval | table    |\n"
"+-----------------+----------+----------+----------+\n"
"| datetime        | interval | datetime | datetime |\n"
"| interval        |          | interval | interval |\n"
"| table           |          |          |          |\n"
"]]\n"
"local function datetime_interval_sub(lhs, rhs)\n"
"    check_date_interval(lhs, \"operator -\")\n"
"    check_date_interval_table(rhs, \"operator -\")\n"
"    local left_is_interval = is_table(lhs) or is_interval(lhs)\n"
"    local right_is_interval = is_table(rhs) or is_interval(rhs)\n"
"\n"
"    -- left is date, right is interval\n"
"    if not left_is_interval and right_is_interval then\n"
"        return datetime_increment_by(datetime_new_copy(lhs), -1,\n"
"                                     interval_decode_args(rhs))\n"
"    -- left is date, right is date\n"
"    elseif not left_is_interval and not right_is_interval then\n"
"        return datetime_datetime_sub(lhs, rhs)\n"
"    -- both left and right are intervals\n"
"    elseif left_is_interval and right_is_interval then\n"
"        return interval_interval_sub(lhs, rhs)\n"
"    else\n"
"        error_incompatible(\"operator -\")\n"
"    end\n"
"end\n"
"\n"
"--[[\n"
"Matrix of addition operands eligibility and their result type\n"
"\n"
"|                 | datetime | interval | table    |\n"
"+-----------------+----------+----------+----------+\n"
"| datetime        |          | datetime | datetime |\n"
"| interval        | datetime | interval | interval |\n"
"| table           |          |          |          |\n"
"]]\n"
"local function datetime_interval_add(lhs, rhs)\n"
"    lhs, rhs = date_first(lhs, rhs)\n"
"\n"
"    check_date_interval(lhs, \"operator +\")\n"
"    check_interval_table(rhs, \"operator +\")\n"
"    local left_is_interval = is_table(lhs) or is_interval(lhs)\n"
"    local right_is_interval = is_table(rhs) or is_interval(rhs)\n"
"\n"
"    -- left is date, right is interval\n"
"    if not left_is_interval and right_is_interval then\n"
"        local obj = datetime_new_copy(lhs)\n"
"        return datetime_increment_by(obj, 1, interval_decode_args(rhs))\n"
"    -- both left and right are intervals\n"
"    elseif left_is_interval and right_is_interval then\n"
"        return interval_interval_add(lhs, rhs)\n"
"    else\n"
"        error_incompatible(\"operator +\")\n"
"    end\n"
"end\n"
"\n"
"--[[\n"
"    Parse partial ISO-8601 date string\n"
"\n"
"    Accepted formats are:\n"
"\n"
"    Basic      Extended\n"
"    20121224   2012-12-24   Calendar date   (ISO 8601)\n"
"    2012359    2012-359     Ordinal date    (ISO 8601)\n"
"    2012W521   2012-W52-1   Week date       (ISO 8601)\n"
"    2012Q485   2012-Q4-85   Quarter date\n"
"\n"
"    Returns pair of constructed datetime object, and length of string\n"
"    which has been accepted by parser.\n"
"]]\n"
"local function datetime_parse_date(str)\n"
"    check_str(str, 'datetime.parse_date()')\n"
"    local dt = date_dt_stash_take()\n"
"    local len = tonumber(builtin.tnt_dt_parse_iso_date(str, #str, dt))\n"
"    if len == 0 then\n"
"        date_dt_stash_put(dt)\n"
"        error(('invalid date format %s'):format(str), 2)\n"
"    end\n"
"    local d = datetime_new_dt(dt[0])\n"
"    date_dt_stash_put(dt)\n"
"    return d, tonumber(len)\n"
"end\n"
"\n"
"--[[\n"
"    datetime parse function for strings in extended iso-8601 format\n"
"    assumes to deal with date T time time_zone at once\n"
"       date [T] time [ ] time_zone\n"
"    Returns constructed datetime object and length of accepted string.\n"
"]]\n"
"local function datetime_parse_full(str, tzname, offset)\n"
"    check_str(str, 'datetime.parse()')\n"
"    local date = ffi.new(datetime_t)\n"
"    local len = builtin.tnt_datetime_parse_full(date, str, #str, tzname, offset)\n"
"    if len > 0 then\n"
"        return date, tonumber(len)\n"
"    elseif len == -builtin.TZ_NYI then\n"
"        error((\"could not parse '%s' - nyi timezone\"):format(str))\n"
"    elseif len == -builtin.TZ_AMBIGUOUS then\n"
"        error((\"could not parse '%s' - ambiguous timezone\"):format(str))\n"
"    else -- len <= 0\n"
"        error((\"could not parse '%s'\"):format(str))\n"
"    end\n"
"end\n"
"\n"
"--[[\n"
"    Parse datetime string given `strptime` like format.\n"
"    Returns constructed datetime object and length of accepted string.\n"
"]]\n"
"local function datetime_parse_format(str, fmt)\n"
"    local date = ffi.new(datetime_t)\n"
"    local len = builtin.tnt_datetime_strptime(date, str, fmt)\n"
"    if len == 0 then\n"
"        error((\"could not parse '%s' using '%s' format\"):format(str, fmt))\n"
"    end\n"
"    return date, tonumber(len)\n"
"end\n"
"\n"
"local function datetime_parse_from(str, obj)\n"
"    check_str(str, 'datetime.parse()')\n"
"    local fmt = ''\n"
"    local tzoffset\n"
"    local tzname\n"
"\n"
"    if obj ~= nil then\n"
"        check_table(obj, 'datetime.parse()')\n"
"        fmt = obj.format\n"
"        tzoffset = obj.tzoffset\n"
"        tzname = obj.tz\n"
"    end\n"
"    check_str_or_nil(fmt, 'datetime.parse()')\n"
"\n"
"    local offset = 0\n"
"    if tzoffset ~= nil then\n"
"        offset = get_timezone(tzoffset, 'tzoffset')\n"
"        check_range(offset, -720, 840, 'tzoffset')\n"
"    end\n"
"\n"
"    if tzname ~= nil then\n"
"        check_str(tzname, 'datetime.parse()')\n"
"    end\n"
"\n"
"    if not fmt or fmt == '' or fmt == 'iso8601' or fmt == 'rfc3339' then\n"
"        -- Effect of .tz overrides .tzoffset\n"
"        return datetime_parse_full(str, tzname, offset)\n"
"    else\n"
"        return datetime_parse_format(str, fmt)\n"
"    end\n"
"end\n"
"\n"
"--[[\n"
"    Create datetime object representing current time using microseconds\n"
"    platform timer and local timezone information.\n"
"]]\n"
"local function datetime_now()\n"
"    local d = datetime_new_raw(0, 0, 0, 0)\n"
"    builtin.tnt_datetime_now(d)\n"
"    return d\n"
"end\n"
"\n"
"-- addition or subtraction from date/time of a given interval\n"
"-- described via table direction should be +1 or -1\n"
"local function datetime_shift(self, o, direction)\n"
"    assert(direction == -1 or direction == 1)\n"
"    local title = direction > 0 and \"datetime.add\" or \"datetime.sub\"\n"
"    check_interval_table(o, title)\n"
"\n"
"    return datetime_increment_by(self, direction, interval_decode_args(o))\n"
"end\n"
"\n"
"--[[\n"
"    dt_dow() returns days of week in range: 1=Monday .. 7=Sunday\n"
"    convert it to os.date() wday which is in range: 1=Sunday .. 7=Saturday\n"
"]]\n"
"local function dow_to_wday(dow)\n"
"    return tonumber(dow) % 7 + 1\n"
"end\n"
"\n"
"--[[\n"
"    Return table in os.date('*t') format, but with timezone\n"
"    and nanoseconds\n"
"]]\n"
"local function datetime_totable(self)\n"
"    check_date(self, 'datetime.totable()')\n"
"    local dt = local_dt(self)\n"
"    local tmp_ival = interval_new()\n"
"    local rc = builtin.tnt_datetime_totable(self, tmp_ival)\n"
"    assert(rc == true)\n"
"\n"
"    return {\n"
"        year = tmp_ival.year,\n"
"        month = tmp_ival.month,\n"
"        yday = builtin.tnt_dt_doy(dt),\n"
"        wday = dow_to_wday(builtin.tnt_dt_dow(dt)),\n"
"        day = tmp_ival.day,\n"
"        hour = tmp_ival.hour,\n"
"        min = tmp_ival.min,\n"
"        sec = tmp_ival.sec,\n"
"        isdst = datetime_isdst(self),\n"
"        nsec = self.nsec,\n"
"        tzoffset = self.tzoffset,\n"
"    }\n"
"end\n"
"\n"
"local function datetime_update_dt(self, dt)\n"
"    local epoch = self.epoch\n"
"    local secs_day = epoch % SECS_PER_DAY\n"
"    self.epoch = (dt - DAYS_EPOCH_OFFSET) * SECS_PER_DAY + secs_day\n"
"end\n"
"\n"
"local function datetime_ymd_update(self, y, M, d)\n"
"    if d < 0 then\n"
"        d = builtin.tnt_dt_days_in_month(y, M)\n"
"    elseif d > 28 then\n"
"        local day_in_month = builtin.tnt_dt_days_in_month(y, M)\n"
"        if d > day_in_month then\n"
"            error(('invalid number of days %d in month %d for %d'):\n"
"                  format(d, M, y), 3)\n"
"        end\n"
"    end\n"
"    local dt = dt_from_ymd_checked(y, M, d)\n"
"    datetime_update_dt(self, dt)\n"
"end\n"
"\n"
"local function datetime_hms_update(self, h, m, s)\n"
"    local epoch = self.epoch\n"
"    local secs_day = epoch - (epoch % SECS_PER_DAY)\n"
"    self.epoch = secs_day + h * 3600 + m * 60 + s\n"
"end\n"
"\n"
"local function datetime_set(self, obj)\n"
"    check_date(self, 'datetime.set()')\n"
"    check_table(obj, \"datetime.set()\")\n"
"\n"
"    local ymd = false\n"
"    local hms = false\n"
"\n"
"    local dt = local_dt(self)\n"
"    local y0 = ffi.new('int[1]')\n"
"    local M0 = ffi.new('int[1]')\n"
"    local d0 = ffi.new('int[1]')\n"
"    builtin.tnt_dt_to_ymd(dt, y0, M0, d0)\n"
"    y0, M0, d0 = y0[0], M0[0], d0[0]\n"
"\n"
"    local y = obj.year\n"
"    if y ~= nil then\n"
"        check_range(y, MIN_DATE_YEAR, MAX_DATE_YEAR, 'year')\n"
"        ymd = true\n"
"    end\n"
"    local M = obj.month\n"
"    if M ~= nil then\n"
"        check_range(M, 1, 12, 'month')\n"
"        ymd = true\n"
"    end\n"
"    local d = obj.day\n"
"    if d ~= nil then\n"
"        check_range(d, 1, 31, 'day', -1)\n"
"        ymd = true\n"
"    end\n"
"\n"
"    local lsecs = local_secs(self)\n"
"    local h0 = math_floor(lsecs / (60 * 60)) % 24\n"
"    local m0 = math_floor(lsecs / 60) % 60\n"
"    local sec0 = lsecs % 60\n"
"\n"
"    local h = obj.hour\n"
"    if h ~= nil then\n"
"        check_range(h, 0, 23, 'hour')\n"
"        hms = true\n"
"    end\n"
"    local m = obj.min\n"
"    if m ~= nil then\n"
"        check_range(m, 0, 59, 'min')\n"
"        hms = true\n"
"    end\n"
"    local sec = obj.sec\n"
"    if sec ~= nil then\n"
"        check_range(sec, 0, 60, 'sec')\n"
"        hms = true\n"
"    end\n"
"\n"
"    local nsec, usec, msec = obj.nsec, obj.usec, obj.msec\n"
"    local count_usec = bool2int(nsec ~= nil) + bool2int(usec ~= nil) +\n"
"                       bool2int(msec ~= nil)\n"
"    if count_usec > 0 then\n"
"        if count_usec > 1 then\n"
"            error('only one of nsec, usec or msecs may be defined '..\n"
"                  'simultaneously', 2)\n"
"        end\n"
"        if usec ~= nil then\n"
"            check_range(usec, 0, 1e6, 'usec')\n"
"            self.nsec = usec * 1e3\n"
"        elseif msec ~= nil then\n"
"            check_range(msec, 0, 1e3, 'msec')\n"
"            self.nsec = msec * 1e6\n"
"        elseif nsec ~= nil then\n"
"            check_range(nsec, 0, 1e9, 'nsec')\n"
"            self.nsec = nsec\n"
"        end\n"
"    end\n"
"\n"
"    local offset = obj.tzoffset\n"
"    if offset ~= nil then\n"
"        offset = get_timezone(offset, 'tzoffset')\n"
"        check_range(offset, -720, 840, 'tzoffset')\n"
"    end\n"
"    offset = offset or self.tzoffset\n"
"\n"
"    local tzname = obj.tz\n"
"\n"
"    local ts = obj.timestamp\n"
"    if ts ~= nil then\n"
"        if ymd then\n"
"            error('timestamp is not allowed if year/month/day provided', 2)\n"
"        end\n"
"        if hms then\n"
"            error('timestamp is not allowed if hour/min/sec provided', 2)\n"
"        end\n"
"        local sec_int, fraction\n"
"        sec_int, fraction = math_modf(ts)\n"
"        -- In case of negative fraction part we should\n"
"        -- make it positive at the expense of the integer part.\n"
"        -- Code below expects that \"nsec\" value is always positive.\n"
"        if fraction < 0 then\n"
"            sec_int = sec_int - 1\n"
"            fraction = fraction + 1\n"
"        end\n"
"        -- if there is one of nsec, usec, msec provided\n"
"        -- then ignore fraction in timestamp\n"
"        -- otherwise - use nsec, usec, or msec\n"
"        if count_usec == 0 then\n"
"            nsec = fraction * 1e9\n"
"        elseif fraction ~= 0 then\n"
"            error('only integer values allowed in timestamp '..\n"
"                  'if nsec, usec, or msecs provided', 2)\n"
"        end\n"
"\n"
"        if msec ~= nil then\n"
"            nsec = msec * 1e6\n"
"        end\n"
"        if usec ~= nil then\n"
"            nsec = usec * 1e3\n"
"        end\n"
"        if tzname ~= nil then\n"
"            offset, self.tzindex = parse_tzname(sec_int, tzname)\n"
"        end\n"
"        self.epoch = utc_secs(sec_int, offset)\n"
"        self.nsec = nsec\n"
"        self.tzoffset = offset\n"
"\n"
"        return self\n"
"    end\n"
"\n"
"    -- normalize time to UTC from current timezone\n"
"    time_delocalize(self)\n"
"\n"
"    -- .year, .month, .day\n"
"    if ymd then\n"
"        y = y or y0\n"
"        M = M or M0\n"
"        d = d or d0\n"
"        datetime_ymd_update(self, y, M, d)\n"
"    end\n"
"\n"
"    if tzname ~= nil then\n"
"        offset, self.tzindex = parse_tzname(self.epoch, tzname)\n"
"    end\n"
"\n"
"    -- .hour, .minute, .second\n"
"    if hms then\n"
"        datetime_hms_update(self, h or h0, m or m0, sec or sec0)\n"
"    end\n"
"\n"
"    -- denormalize back to local timezone\n"
"    time_localize(self, offset)\n"
"\n"
"    return self\n"
"end\n"
"\n"
"local function datetime_strftime(self, fmt)\n"
"    check_str(fmt, \"datetime.strftime()\")\n"
"    local buff = date_strf_stash_take()\n"
"    local len = builtin.tnt_datetime_strftime(self, buff, STRFTIME_BUFSIZE, fmt)\n"
"    if len < STRFTIME_BUFSIZE then\n"
"        local s = ffi.string(buff)\n"
"        date_strf_stash_put(buff)\n"
"        return s\n"
"    end\n"
"    -- slow path - reallocate for a fuller size, and then restart strftime\n"
"    date_strf_stash_put(buff)\n"
"    buff = ffi.new('char[\?]', len + 1)\n"
"    builtin.tnt_datetime_strftime(self, buff, len + 1, fmt)\n"
"    return ffi.string(buff)\n"
"end\n"
"\n"
"local function datetime_format(self, fmt)\n"
"    check_date(self, 'datetime.format()')\n"
"    if fmt ~= nil then\n"
"        return datetime_strftime(self, fmt)\n"
"    else\n"
"        return datetime_tostring(self)\n"
"    end\n"
"end\n"
"\n"
"local datetime_index_fields = {\n"
"    timestamp = function(self) return self.epoch + self.nsec / 1e9 end,\n"
"\n"
"    year = function(self) return builtin.tnt_dt_year(local_dt(self)) end,\n"
"    yday = function(self) return builtin.tnt_dt_doy(local_dt(self)) end,\n"
"    month = function(self) return builtin.tnt_dt_month(local_dt(self)) end,\n"
"    day = function(self)\n"
"        return builtin.tnt_dt_dom(local_dt(self))\n"
"    end,\n"
"    wday = function(self)\n"
"        return dow_to_wday(builtin.tnt_dt_dow(local_dt(self)))\n"
"    end,\n"
"    hour = function(self) return math_floor((local_secs(self) / 3600) % 24) end,\n"
"    min = function(self) return math_floor((local_secs(self) / 60) % 60) end,\n"
"    sec = function(self) return self.epoch % 60 end,\n"
"    usec = function(self) return self.nsec / 1e3 end,\n"
"    msec = function(self) return self.nsec / 1e6 end,\n"
"    isdst = function(self) return datetime_isdst(self) end,\n"
"    tz = function(self) return self:format('%Z') end,\n"
"}\n"
"\n"
"local datetime_index_functions = {\n"
"    format = datetime_format,\n"
"    totable = datetime_totable,\n"
"    set = datetime_set,\n"
"    add = function(self, obj) return datetime_shift(self, obj, 1) end,\n"
"    sub = function(self, obj) return datetime_shift(self, obj, -1) end,\n"
"}\n"
"\n"
"local function datetime_index(self, key)\n"
"    local handler_field = datetime_index_fields[key]\n"
"    if handler_field ~= nil then\n"
"        return handler_field(self)\n"
"    end\n"
"    return datetime_index_functions[key]\n"
"end\n"
"\n"
"ffi.metatype(datetime_t, {\n"
"    __tostring = datetime_tostring,\n"
"    __eq = datetime_eq,\n"
"    __lt = datetime_lt,\n"
"    __le = datetime_le,\n"
"    __sub = datetime_interval_sub,\n"
"    __add = datetime_interval_add,\n"
"    __index = datetime_index,\n"
"})\n"
"\n"
"local function interval_totable(self)\n"
"    if not is_interval(self) then\n"
"        return error((\"interval.totable(): expected interval, but received \"..\n"
"                     type(self)), 2)\n"
"    end\n"
"    local adjust = {'excess', 'none', 'last'}\n"
"    return {\n"
"        year = self.year,\n"
"        month = self.month,\n"
"        week = self.week,\n"
"        day = self.day,\n"
"        hour = self.hour,\n"
"        min = self.min,\n"
"        sec = self.sec,\n"
"        nsec = self.nsec,\n"
"        adjust = adjust[tonumber(self.adjust) + 1],\n"
"    }\n"
"end\n"
"\n"
"local function interval_cmp(lhs, rhs, is_raising)\n"
"    if not is_interval(lhs) or not is_interval(rhs) then\n"
"        if is_raising then\n"
"            error('incompatible types for interval comparison', 3)\n"
"        else\n"
"            return nil\n"
"        end\n"
"    end\n"
"    local tags = {\n"
"        'year', 'month', 'week', 'day',\n"
"        'hour', 'min', 'sec', 'nsec'\n"
"    }\n"
"    for _, key in pairs(tags) do\n"
"        local diff = lhs[key] - rhs[key]\n"
"        if diff ~= 0 then\n"
"            return diff\n"
"        end\n"
"    end\n"
"    return 0\n"
"end\n"
"\n"
"local function interval_eq(lhs, rhs)\n"
"    return interval_cmp(lhs, rhs, false) == 0\n"
"end\n"
"\n"
"local function interval_lt(lhs, rhs)\n"
"    return interval_cmp(lhs, rhs, true) < 0\n"
"end\n"
"\n"
"local function interval_le(lhs, rhs)\n"
"    return interval_cmp(lhs, rhs, true) <= 0\n"
"end\n"
"\n"
"local interval_index_fields = {\n"
"    usec = function(self) return math_floor(self.nsec / 1e3) end,\n"
"    msec = function(self) return math_floor(self.nsec / 1e6) end,\n"
"}\n"
"\n"
"local interval_index_functions = {\n"
"    totable = interval_totable,\n"
"    __serialize = interval_tostring,\n"
"}\n"
"\n"
"local function interval_index(self, key)\n"
"    local handler_field = interval_index_fields[key]\n"
"    return handler_field ~= nil and handler_field(self) or\n"
"           interval_index_functions[key]\n"
"end\n"
"\n"
"ffi.metatype(interval_t, {\n"
"    __tostring = interval_tostring,\n"
"    __eq = interval_eq,\n"
"    __lt = interval_lt,\n"
"    __le = interval_le,\n"
"    __sub = datetime_interval_sub,\n"
"    __add = datetime_interval_add,\n"
"    __index = interval_index,\n"
"})\n"
"\n"
"local interval_mt = {\n"
"    new     = interval_new,\n"
"}\n"
"\n"
"return setmetatable(\n"
"    {\n"
"        new         = datetime_new,\n"
"        interval    = setmetatable(interval_mt, interval_mt),\n"
"        now         = datetime_now,\n"
"        parse       = datetime_parse_from,\n"
"        parse_date  = datetime_parse_date,\n"
"        is_datetime = is_datetime,\n"
"        TZ          = tz,\n"
"    }, {}\n"
")\n"
""
;
