// Copyright © 2023-2025 Advanced Micro Devices, Inc.
// SPDX-License-Identifier: MIT

// clang-format off
#include "shim.attn_fwd.h"
#include <aotriton/util.h>
#include <tuple>
#include "iface.op_attn_fwd.h"

namespace AOTRITON_NS::v3::flash {

#if 1
using AOTRITON_NS::v3::flash::OpAttnFwdParams;
#endif

#define CAST(x) const_cast<void*>(static_cast<const void*>(x))
typedef std::vector<void*>(*PP_FUNC)(const OpAttnFwdParams& context, hipDeviceptr_t*);

namespace {
extern PP_FUNC prepare_arguments[ 12 ];
}

int64_t AttnFwdContext::godel_number() const
{
    int64_t sum = 0;
    const auto& args = *params;
    {
        int64_t number = 0;
        if (args.Q->dtype() == DType::kFloat16) number = 0 ;
        if (args.Q->dtype() == DType::kBFloat16) number = 1 ;
        if (args.Q->dtype() == DType::kFloat32) number = 2 ;
        sum += number * 192;
    }
    {
        int64_t number = 0;
        if (args.BLOCK_DMODEL == 16) number = 0 ;
        if (args.BLOCK_DMODEL == 32) number = 1 ;
        if (args.BLOCK_DMODEL == 48) number = 2 ;
        if (args.BLOCK_DMODEL == 64) number = 3 ;
        if (args.BLOCK_DMODEL == 80) number = 4 ;
        if (args.BLOCK_DMODEL == 96) number = 5 ;
        if (args.BLOCK_DMODEL == 128) number = 6 ;
        if (args.BLOCK_DMODEL == 160) number = 7 ;
        if (args.BLOCK_DMODEL == 192) number = 8 ;
        if (args.BLOCK_DMODEL == 224) number = 9 ;
        if (args.BLOCK_DMODEL == 256) number = 10 ;
        if (args.BLOCK_DMODEL == 512) number = 11 ;
        sum += number * 16;
    }
    {
        int64_t number = 0;
        if (args.PADDED_HEAD == false) number = 0 ;
        if (args.PADDED_HEAD == true) number = 1 ;
        sum += number * 8;
    }
    {
        int64_t number = 0;
        if (args.ENABLE_DROPOUT == false) number = 0 ;
        if (args.ENABLE_DROPOUT == true) number = 1 ;
        sum += number * 4;
    }
    {
        int64_t number = 0;
        if (args.CAUSAL_TYPE == 0) number = 0 ;
        if (args.CAUSAL_TYPE == 3) number = 1 ;
        sum += number * 2;
    }
    {
        int64_t number = 0;
        if (args.BIAS_TYPE == 0) number = 0 ;
        if (args.BIAS_TYPE == 1) number = 1 ;
        sum += number * 1;
    }

    return sum;
}

hipError_t
AttnFwdContext::lookup_optimal(Gpu gpu) {
    auto [arch_number, mod_number] = get_archmod_number(gpu);
    if (arch_number < 0) {
        return hipErrorNoBinaryForGpu;
    }
    kernel_on_device = nullptr;
    auto tune_func = autotune_table[arch_number][godel_number()];
    if (!tune_func)
        return hipErrorProfilerNotInitialized;
    tune_func(*this, mod_number);
    if (!kernel_on_device)
        return hipErrorSharedObjectSymbolNotFound;
    return hipSuccess;
}

hipError_t
AttnFwdContext::launch(hipStream_t stream) const {
    constexpr std::string_view triton_kernel_name { "attn_fwd" };
    hipDeviceptr_t global_scratch = 0;
    auto args = prepare_arguments[pp_args_index](*this->params, &global_scratch);
    dim3 grid;
    if (custom_grid_calculator) {
        grid = custom_grid_calculator(*this);
    } else {
        grid = grid_calculator();
    }
#if AOTRITON_BUILD_FOR_TUNING
    return kernel_on_device->invoke(triton_kernel_name,
                                    package_path,
                                    func_name,
                                    arch_name,
                                    grid,
                                    args,
                                    peek_kernel_image,
                                    stream);
#else
    return kernel_on_device->invoke(triton_kernel_name,
                                    package_path,
                                    func_name,
                                    arch_name,
                                    grid,
                                    args,
                                    stream);
#endif
}

std::tuple<int, int>
AttnFwdContext::get_archmod_number(Gpu gpu) {
    if (gpu == GPU_AMD_ARCH_GFX950_MOD0) return { 0, 0 };
    if (gpu == GPU_AMD_ARCH_GFX1151_MOD0) return { 1, 0 };
    if (gpu == GPU_AMD_ARCH_GFX1150_MOD0) return { 2, 0 };
    if (gpu == GPU_AMD_ARCH_GFX1201_MOD0) return { 3, 0 };
    if (gpu == GPU_AMD_ARCH_GFX1200_MOD0) return { 4, 0 };
    // TODO: print warning about tuning for this GPU mod is not built.
    // Note: if some mod does not have tuning info in the database at all, the
    //       getGpuFromStream should not return that mod from beginning.
    return std::make_tuple(-1, 0);
}


static std::vector<void*>
attn_fwd_pp_args_0(const OpAttnFwdParams& params,
                   hipDeviceptr_t* global_scratch) {
  return { params.Q->kparam_data_ptr(), // Q
           params.K->kparam_data_ptr(), // K
           params.V->kparam_data_ptr(), // V
           // params.B->kparam_data_ptr(), // B as constexpr nullptr
           // params.A->kparam_data_ptr(), // A as constexpr nullptr
           CAST(&params.Sm_scale), // Sm_scale
           params.L->kparam_data_ptr(), // L
           params.Out->kparam_data_ptr(), // Out
           params.Q->kparam_stride(0), // stride_qz
           params.Q->kparam_stride(1), // stride_qh
           params.Q->kparam_stride(2), // stride_qm
           params.K->kparam_stride(0), // stride_kz
           params.K->kparam_stride(1), // stride_kh
           params.K->kparam_stride(2), // stride_kn
           params.V->kparam_stride(0), // stride_vz
           params.V->kparam_stride(1), // stride_vh
           params.V->kparam_stride(2), // stride_vk
           params.Out->kparam_stride(0), // stride_oz
           params.Out->kparam_stride(1), // stride_oh
           params.Out->kparam_stride(2), // stride_om
           // params.B->kparam_stride(0), // stride_bz as constexpr 0
           // params.B->kparam_stride(1), // stride_bh as constexpr 0
           // params.B->kparam_stride(2), // stride_bm as constexpr 0
           // params.A->kparam_stride(0), // stride_az as constexpr 0
           CAST(&params.Num_head_q), // Num_head_q
           CAST(&params.Num_head_k), // Num_head_k
           CAST(&params.Num_seqlens), // Num_seqlens
           params.cu_seqlens_q->kparam_data_ptr(), // cu_seqlens_q
           params.cu_seqlens_k->kparam_data_ptr(), // cu_seqlens_k
           CAST(&params.Max_seqlen_q), // Max_seqlen_q
           CAST(&params.Max_seqlen_k), // Max_seqlen_k
           // CAST(&params.Head_dim), // Head_dim as constexpr 16/32/48/64/80/96/128/160/192/224/256/512
           // CAST(&params.dropout_p), // dropout_p as constexpr 0
           // params.philox_seed_ptr->kparam_data_ptr(), // philox_seed_ptr as constexpr 0
           // params.philox_offset1->kparam_data_ptr(), // philox_offset1 as constexpr 0
           // CAST(&params.philox_offset2), // philox_offset2 as constexpr 0
           // params.philox_seed_output->kparam_data_ptr(), // philox_seed_output as constexpr 0
           // params.philox_offset_output->kparam_data_ptr(), // philox_offset_output as constexpr 0
           // params.encoded_softmax->kparam_data_ptr(), // encoded_softmax as constexpr nullptr
           // CAST(&params.Window_left), // Window_left as constexpr 0
           // CAST(&params.Window_right), // Window_right as constexpr 0
           // params.persistent_atomic_counter->kparam_data_ptr(), // persistent_atomic_counter as constexpr 0
           CAST(&params.Num_CU), // Num_CU
           CAST(&params.Batch), // Batch
           CAST(global_scratch)
         };
}
static std::vector<void*>
attn_fwd_pp_args_1(const OpAttnFwdParams& params,
                   hipDeviceptr_t* global_scratch) {
  return { params.Q->kparam_data_ptr(), // Q
           params.K->kparam_data_ptr(), // K
           params.V->kparam_data_ptr(), // V
           params.B->kparam_data_ptr(), // B
           // params.A->kparam_data_ptr(), // A as constexpr nullptr
           CAST(&params.Sm_scale), // Sm_scale
           params.L->kparam_data_ptr(), // L
           params.Out->kparam_data_ptr(), // Out
           params.Q->kparam_stride(0), // stride_qz
           params.Q->kparam_stride(1), // stride_qh
           params.Q->kparam_stride(2), // stride_qm
           params.K->kparam_stride(0), // stride_kz
           params.K->kparam_stride(1), // stride_kh
           params.K->kparam_stride(2), // stride_kn
           params.V->kparam_stride(0), // stride_vz
           params.V->kparam_stride(1), // stride_vh
           params.V->kparam_stride(2), // stride_vk
           params.Out->kparam_stride(0), // stride_oz
           params.Out->kparam_stride(1), // stride_oh
           params.Out->kparam_stride(2), // stride_om
           params.B->kparam_stride(0), // stride_bz
           params.B->kparam_stride(1), // stride_bh
           params.B->kparam_stride(2), // stride_bm
           // params.A->kparam_stride(0), // stride_az as constexpr 0
           CAST(&params.Num_head_q), // Num_head_q
           CAST(&params.Num_head_k), // Num_head_k
           CAST(&params.Num_seqlens), // Num_seqlens
           params.cu_seqlens_q->kparam_data_ptr(), // cu_seqlens_q
           params.cu_seqlens_k->kparam_data_ptr(), // cu_seqlens_k
           CAST(&params.Max_seqlen_q), // Max_seqlen_q
           CAST(&params.Max_seqlen_k), // Max_seqlen_k
           // CAST(&params.Head_dim), // Head_dim as constexpr 16/32/48/64/80/96/128/160/192/224/256/512
           // CAST(&params.dropout_p), // dropout_p as constexpr 0
           // params.philox_seed_ptr->kparam_data_ptr(), // philox_seed_ptr as constexpr 0
           // params.philox_offset1->kparam_data_ptr(), // philox_offset1 as constexpr 0
           // CAST(&params.philox_offset2), // philox_offset2 as constexpr 0
           // params.philox_seed_output->kparam_data_ptr(), // philox_seed_output as constexpr 0
           // params.philox_offset_output->kparam_data_ptr(), // philox_offset_output as constexpr 0
           // params.encoded_softmax->kparam_data_ptr(), // encoded_softmax as constexpr nullptr
           // CAST(&params.Window_left), // Window_left as constexpr 0
           // CAST(&params.Window_right), // Window_right as constexpr 0
           // params.persistent_atomic_counter->kparam_data_ptr(), // persistent_atomic_counter as constexpr 0
           CAST(&params.Num_CU), // Num_CU
           CAST(&params.Batch), // Batch
           CAST(global_scratch)
         };
}
static std::vector<void*>
attn_fwd_pp_args_2(const OpAttnFwdParams& params,
                   hipDeviceptr_t* global_scratch) {
  return { params.Q->kparam_data_ptr(), // Q
           params.K->kparam_data_ptr(), // K
           params.V->kparam_data_ptr(), // V
           // params.B->kparam_data_ptr(), // B as constexpr nullptr
           // params.A->kparam_data_ptr(), // A as constexpr nullptr
           CAST(&params.Sm_scale), // Sm_scale
           params.L->kparam_data_ptr(), // L
           params.Out->kparam_data_ptr(), // Out
           params.Q->kparam_stride(0), // stride_qz
           params.Q->kparam_stride(1), // stride_qh
           params.Q->kparam_stride(2), // stride_qm
           params.K->kparam_stride(0), // stride_kz
           params.K->kparam_stride(1), // stride_kh
           params.K->kparam_stride(2), // stride_kn
           params.V->kparam_stride(0), // stride_vz
           params.V->kparam_stride(1), // stride_vh
           params.V->kparam_stride(2), // stride_vk
           params.Out->kparam_stride(0), // stride_oz
           params.Out->kparam_stride(1), // stride_oh
           params.Out->kparam_stride(2), // stride_om
           // params.B->kparam_stride(0), // stride_bz as constexpr 0
           // params.B->kparam_stride(1), // stride_bh as constexpr 0
           // params.B->kparam_stride(2), // stride_bm as constexpr 0
           // params.A->kparam_stride(0), // stride_az as constexpr 0
           CAST(&params.Num_head_q), // Num_head_q
           CAST(&params.Num_head_k), // Num_head_k
           CAST(&params.Num_seqlens), // Num_seqlens
           params.cu_seqlens_q->kparam_data_ptr(), // cu_seqlens_q
           params.cu_seqlens_k->kparam_data_ptr(), // cu_seqlens_k
           CAST(&params.Max_seqlen_q), // Max_seqlen_q
           CAST(&params.Max_seqlen_k), // Max_seqlen_k
           // CAST(&params.Head_dim), // Head_dim as constexpr 16/32/48/64/80/96/128/160/192/224/256/512
           // CAST(&params.dropout_p), // dropout_p as constexpr 0
           // params.philox_seed_ptr->kparam_data_ptr(), // philox_seed_ptr as constexpr 0
           // params.philox_offset1->kparam_data_ptr(), // philox_offset1 as constexpr 0
           // CAST(&params.philox_offset2), // philox_offset2 as constexpr 0
           // params.philox_seed_output->kparam_data_ptr(), // philox_seed_output as constexpr 0
           // params.philox_offset_output->kparam_data_ptr(), // philox_offset_output as constexpr 0
           // params.encoded_softmax->kparam_data_ptr(), // encoded_softmax as constexpr nullptr
           CAST(&params.Window_left), // Window_left
           CAST(&params.Window_right), // Window_right
           params.persistent_atomic_counter->kparam_data_ptr(), // persistent_atomic_counter
           CAST(&params.Num_CU), // Num_CU
           CAST(&params.Batch), // Batch
           CAST(global_scratch)
         };
}
static std::vector<void*>
attn_fwd_pp_args_3(const OpAttnFwdParams& params,
                   hipDeviceptr_t* global_scratch) {
  return { params.Q->kparam_data_ptr(), // Q
           params.K->kparam_data_ptr(), // K
           params.V->kparam_data_ptr(), // V
           // params.B->kparam_data_ptr(), // B as constexpr nullptr
           // params.A->kparam_data_ptr(), // A as constexpr nullptr
           CAST(&params.Sm_scale), // Sm_scale
           params.L->kparam_data_ptr(), // L
           params.Out->kparam_data_ptr(), // Out
           params.Q->kparam_stride(0), // stride_qz
           params.Q->kparam_stride(1), // stride_qh
           params.Q->kparam_stride(2), // stride_qm
           params.K->kparam_stride(0), // stride_kz
           params.K->kparam_stride(1), // stride_kh
           params.K->kparam_stride(2), // stride_kn
           params.V->kparam_stride(0), // stride_vz
           params.V->kparam_stride(1), // stride_vh
           params.V->kparam_stride(2), // stride_vk
           params.Out->kparam_stride(0), // stride_oz
           params.Out->kparam_stride(1), // stride_oh
           params.Out->kparam_stride(2), // stride_om
           // params.B->kparam_stride(0), // stride_bz as constexpr 0
           // params.B->kparam_stride(1), // stride_bh as constexpr 0
           // params.B->kparam_stride(2), // stride_bm as constexpr 0
           // params.A->kparam_stride(0), // stride_az as constexpr 0
           CAST(&params.Num_head_q), // Num_head_q
           CAST(&params.Num_head_k), // Num_head_k
           CAST(&params.Num_seqlens), // Num_seqlens
           params.cu_seqlens_q->kparam_data_ptr(), // cu_seqlens_q
           params.cu_seqlens_k->kparam_data_ptr(), // cu_seqlens_k
           CAST(&params.Max_seqlen_q), // Max_seqlen_q
           CAST(&params.Max_seqlen_k), // Max_seqlen_k
           // CAST(&params.Head_dim), // Head_dim as constexpr 16/32/48/64/80/96/128/160/192/224/256/512
           CAST(&params.dropout_p), // dropout_p
           params.philox_seed_ptr->kparam_data_ptr(), // philox_seed_ptr
           params.philox_offset1->kparam_data_ptr(), // philox_offset1
           CAST(&params.philox_offset2), // philox_offset2
           params.philox_seed_output->kparam_data_ptr(), // philox_seed_output
           params.philox_offset_output->kparam_data_ptr(), // philox_offset_output
           // params.encoded_softmax->kparam_data_ptr(), // encoded_softmax as constexpr nullptr
           // CAST(&params.Window_left), // Window_left as constexpr 0
           // CAST(&params.Window_right), // Window_right as constexpr 0
           // params.persistent_atomic_counter->kparam_data_ptr(), // persistent_atomic_counter as constexpr 0
           CAST(&params.Num_CU), // Num_CU
           CAST(&params.Batch), // Batch
           CAST(global_scratch)
         };
}
static std::vector<void*>
attn_fwd_pp_args_4(const OpAttnFwdParams& params,
                   hipDeviceptr_t* global_scratch) {
  return { params.Q->kparam_data_ptr(), // Q
           params.K->kparam_data_ptr(), // K
           params.V->kparam_data_ptr(), // V
           params.B->kparam_data_ptr(), // B
           // params.A->kparam_data_ptr(), // A as constexpr nullptr
           CAST(&params.Sm_scale), // Sm_scale
           params.L->kparam_data_ptr(), // L
           params.Out->kparam_data_ptr(), // Out
           params.Q->kparam_stride(0), // stride_qz
           params.Q->kparam_stride(1), // stride_qh
           params.Q->kparam_stride(2), // stride_qm
           params.K->kparam_stride(0), // stride_kz
           params.K->kparam_stride(1), // stride_kh
           params.K->kparam_stride(2), // stride_kn
           params.V->kparam_stride(0), // stride_vz
           params.V->kparam_stride(1), // stride_vh
           params.V->kparam_stride(2), // stride_vk
           params.Out->kparam_stride(0), // stride_oz
           params.Out->kparam_stride(1), // stride_oh
           params.Out->kparam_stride(2), // stride_om
           params.B->kparam_stride(0), // stride_bz
           params.B->kparam_stride(1), // stride_bh
           params.B->kparam_stride(2), // stride_bm
           // params.A->kparam_stride(0), // stride_az as constexpr 0
           CAST(&params.Num_head_q), // Num_head_q
           CAST(&params.Num_head_k), // Num_head_k
           CAST(&params.Num_seqlens), // Num_seqlens
           params.cu_seqlens_q->kparam_data_ptr(), // cu_seqlens_q
           params.cu_seqlens_k->kparam_data_ptr(), // cu_seqlens_k
           CAST(&params.Max_seqlen_q), // Max_seqlen_q
           CAST(&params.Max_seqlen_k), // Max_seqlen_k
           // CAST(&params.Head_dim), // Head_dim as constexpr 16/32/48/64/80/96/128/160/192/224/256/512
           CAST(&params.dropout_p), // dropout_p
           params.philox_seed_ptr->kparam_data_ptr(), // philox_seed_ptr
           params.philox_offset1->kparam_data_ptr(), // philox_offset1
           CAST(&params.philox_offset2), // philox_offset2
           params.philox_seed_output->kparam_data_ptr(), // philox_seed_output
           params.philox_offset_output->kparam_data_ptr(), // philox_offset_output
           // params.encoded_softmax->kparam_data_ptr(), // encoded_softmax as constexpr nullptr
           // CAST(&params.Window_left), // Window_left as constexpr 0
           // CAST(&params.Window_right), // Window_right as constexpr 0
           // params.persistent_atomic_counter->kparam_data_ptr(), // persistent_atomic_counter as constexpr 0
           CAST(&params.Num_CU), // Num_CU
           CAST(&params.Batch), // Batch
           CAST(global_scratch)
         };
}
static std::vector<void*>
attn_fwd_pp_args_5(const OpAttnFwdParams& params,
                   hipDeviceptr_t* global_scratch) {
  return { params.Q->kparam_data_ptr(), // Q
           params.K->kparam_data_ptr(), // K
           params.V->kparam_data_ptr(), // V
           // params.B->kparam_data_ptr(), // B as constexpr nullptr
           // params.A->kparam_data_ptr(), // A as constexpr nullptr
           CAST(&params.Sm_scale), // Sm_scale
           params.L->kparam_data_ptr(), // L
           params.Out->kparam_data_ptr(), // Out
           params.Q->kparam_stride(0), // stride_qz
           params.Q->kparam_stride(1), // stride_qh
           params.Q->kparam_stride(2), // stride_qm
           params.K->kparam_stride(0), // stride_kz
           params.K->kparam_stride(1), // stride_kh
           params.K->kparam_stride(2), // stride_kn
           params.V->kparam_stride(0), // stride_vz
           params.V->kparam_stride(1), // stride_vh
           params.V->kparam_stride(2), // stride_vk
           params.Out->kparam_stride(0), // stride_oz
           params.Out->kparam_stride(1), // stride_oh
           params.Out->kparam_stride(2), // stride_om
           // params.B->kparam_stride(0), // stride_bz as constexpr 0
           // params.B->kparam_stride(1), // stride_bh as constexpr 0
           // params.B->kparam_stride(2), // stride_bm as constexpr 0
           // params.A->kparam_stride(0), // stride_az as constexpr 0
           CAST(&params.Num_head_q), // Num_head_q
           CAST(&params.Num_head_k), // Num_head_k
           CAST(&params.Num_seqlens), // Num_seqlens
           params.cu_seqlens_q->kparam_data_ptr(), // cu_seqlens_q
           params.cu_seqlens_k->kparam_data_ptr(), // cu_seqlens_k
           CAST(&params.Max_seqlen_q), // Max_seqlen_q
           CAST(&params.Max_seqlen_k), // Max_seqlen_k
           // CAST(&params.Head_dim), // Head_dim as constexpr 16/32/48/64/80/96/128/160/192/224/256/512
           CAST(&params.dropout_p), // dropout_p
           params.philox_seed_ptr->kparam_data_ptr(), // philox_seed_ptr
           params.philox_offset1->kparam_data_ptr(), // philox_offset1
           CAST(&params.philox_offset2), // philox_offset2
           params.philox_seed_output->kparam_data_ptr(), // philox_seed_output
           params.philox_offset_output->kparam_data_ptr(), // philox_offset_output
           // params.encoded_softmax->kparam_data_ptr(), // encoded_softmax as constexpr nullptr
           CAST(&params.Window_left), // Window_left
           CAST(&params.Window_right), // Window_right
           params.persistent_atomic_counter->kparam_data_ptr(), // persistent_atomic_counter
           CAST(&params.Num_CU), // Num_CU
           CAST(&params.Batch), // Batch
           CAST(global_scratch)
         };
}
static std::vector<void*>
attn_fwd_pp_args_6(const OpAttnFwdParams& params,
                   hipDeviceptr_t* global_scratch) {
  return { params.Q->kparam_data_ptr(), // Q
           params.K->kparam_data_ptr(), // K
           params.V->kparam_data_ptr(), // V
           // params.B->kparam_data_ptr(), // B as constexpr nullptr
           // params.A->kparam_data_ptr(), // A as constexpr nullptr
           CAST(&params.Sm_scale), // Sm_scale
           params.L->kparam_data_ptr(), // L
           params.Out->kparam_data_ptr(), // Out
           params.Q->kparam_stride(0), // stride_qz
           params.Q->kparam_stride(1), // stride_qh
           params.Q->kparam_stride(2), // stride_qm
           params.K->kparam_stride(0), // stride_kz
           params.K->kparam_stride(1), // stride_kh
           params.K->kparam_stride(2), // stride_kn
           params.V->kparam_stride(0), // stride_vz
           params.V->kparam_stride(1), // stride_vh
           params.V->kparam_stride(2), // stride_vk
           params.Out->kparam_stride(0), // stride_oz
           params.Out->kparam_stride(1), // stride_oh
           params.Out->kparam_stride(2), // stride_om
           // params.B->kparam_stride(0), // stride_bz as constexpr 0
           // params.B->kparam_stride(1), // stride_bh as constexpr 0
           // params.B->kparam_stride(2), // stride_bm as constexpr 0
           // params.A->kparam_stride(0), // stride_az as constexpr 0
           CAST(&params.Num_head_q), // Num_head_q
           CAST(&params.Num_head_k), // Num_head_k
           CAST(&params.Num_seqlens), // Num_seqlens
           params.cu_seqlens_q->kparam_data_ptr(), // cu_seqlens_q
           params.cu_seqlens_k->kparam_data_ptr(), // cu_seqlens_k
           CAST(&params.Max_seqlen_q), // Max_seqlen_q
           CAST(&params.Max_seqlen_k), // Max_seqlen_k
           CAST(&params.Head_dim), // Head_dim
           // CAST(&params.dropout_p), // dropout_p as constexpr 0
           // params.philox_seed_ptr->kparam_data_ptr(), // philox_seed_ptr as constexpr 0
           // params.philox_offset1->kparam_data_ptr(), // philox_offset1 as constexpr 0
           // CAST(&params.philox_offset2), // philox_offset2 as constexpr 0
           // params.philox_seed_output->kparam_data_ptr(), // philox_seed_output as constexpr 0
           // params.philox_offset_output->kparam_data_ptr(), // philox_offset_output as constexpr 0
           // params.encoded_softmax->kparam_data_ptr(), // encoded_softmax as constexpr nullptr
           // CAST(&params.Window_left), // Window_left as constexpr 0
           // CAST(&params.Window_right), // Window_right as constexpr 0
           // params.persistent_atomic_counter->kparam_data_ptr(), // persistent_atomic_counter as constexpr 0
           CAST(&params.Num_CU), // Num_CU
           CAST(&params.Batch), // Batch
           CAST(global_scratch)
         };
}
static std::vector<void*>
attn_fwd_pp_args_7(const OpAttnFwdParams& params,
                   hipDeviceptr_t* global_scratch) {
  return { params.Q->kparam_data_ptr(), // Q
           params.K->kparam_data_ptr(), // K
           params.V->kparam_data_ptr(), // V
           params.B->kparam_data_ptr(), // B
           // params.A->kparam_data_ptr(), // A as constexpr nullptr
           CAST(&params.Sm_scale), // Sm_scale
           params.L->kparam_data_ptr(), // L
           params.Out->kparam_data_ptr(), // Out
           params.Q->kparam_stride(0), // stride_qz
           params.Q->kparam_stride(1), // stride_qh
           params.Q->kparam_stride(2), // stride_qm
           params.K->kparam_stride(0), // stride_kz
           params.K->kparam_stride(1), // stride_kh
           params.K->kparam_stride(2), // stride_kn
           params.V->kparam_stride(0), // stride_vz
           params.V->kparam_stride(1), // stride_vh
           params.V->kparam_stride(2), // stride_vk
           params.Out->kparam_stride(0), // stride_oz
           params.Out->kparam_stride(1), // stride_oh
           params.Out->kparam_stride(2), // stride_om
           params.B->kparam_stride(0), // stride_bz
           params.B->kparam_stride(1), // stride_bh
           params.B->kparam_stride(2), // stride_bm
           // params.A->kparam_stride(0), // stride_az as constexpr 0
           CAST(&params.Num_head_q), // Num_head_q
           CAST(&params.Num_head_k), // Num_head_k
           CAST(&params.Num_seqlens), // Num_seqlens
           params.cu_seqlens_q->kparam_data_ptr(), // cu_seqlens_q
           params.cu_seqlens_k->kparam_data_ptr(), // cu_seqlens_k
           CAST(&params.Max_seqlen_q), // Max_seqlen_q
           CAST(&params.Max_seqlen_k), // Max_seqlen_k
           CAST(&params.Head_dim), // Head_dim
           // CAST(&params.dropout_p), // dropout_p as constexpr 0
           // params.philox_seed_ptr->kparam_data_ptr(), // philox_seed_ptr as constexpr 0
           // params.philox_offset1->kparam_data_ptr(), // philox_offset1 as constexpr 0
           // CAST(&params.philox_offset2), // philox_offset2 as constexpr 0
           // params.philox_seed_output->kparam_data_ptr(), // philox_seed_output as constexpr 0
           // params.philox_offset_output->kparam_data_ptr(), // philox_offset_output as constexpr 0
           // params.encoded_softmax->kparam_data_ptr(), // encoded_softmax as constexpr nullptr
           // CAST(&params.Window_left), // Window_left as constexpr 0
           // CAST(&params.Window_right), // Window_right as constexpr 0
           // params.persistent_atomic_counter->kparam_data_ptr(), // persistent_atomic_counter as constexpr 0
           CAST(&params.Num_CU), // Num_CU
           CAST(&params.Batch), // Batch
           CAST(global_scratch)
         };
}
static std::vector<void*>
attn_fwd_pp_args_8(const OpAttnFwdParams& params,
                   hipDeviceptr_t* global_scratch) {
  return { params.Q->kparam_data_ptr(), // Q
           params.K->kparam_data_ptr(), // K
           params.V->kparam_data_ptr(), // V
           // params.B->kparam_data_ptr(), // B as constexpr nullptr
           // params.A->kparam_data_ptr(), // A as constexpr nullptr
           CAST(&params.Sm_scale), // Sm_scale
           params.L->kparam_data_ptr(), // L
           params.Out->kparam_data_ptr(), // Out
           params.Q->kparam_stride(0), // stride_qz
           params.Q->kparam_stride(1), // stride_qh
           params.Q->kparam_stride(2), // stride_qm
           params.K->kparam_stride(0), // stride_kz
           params.K->kparam_stride(1), // stride_kh
           params.K->kparam_stride(2), // stride_kn
           params.V->kparam_stride(0), // stride_vz
           params.V->kparam_stride(1), // stride_vh
           params.V->kparam_stride(2), // stride_vk
           params.Out->kparam_stride(0), // stride_oz
           params.Out->kparam_stride(1), // stride_oh
           params.Out->kparam_stride(2), // stride_om
           // params.B->kparam_stride(0), // stride_bz as constexpr 0
           // params.B->kparam_stride(1), // stride_bh as constexpr 0
           // params.B->kparam_stride(2), // stride_bm as constexpr 0
           // params.A->kparam_stride(0), // stride_az as constexpr 0
           CAST(&params.Num_head_q), // Num_head_q
           CAST(&params.Num_head_k), // Num_head_k
           CAST(&params.Num_seqlens), // Num_seqlens
           params.cu_seqlens_q->kparam_data_ptr(), // cu_seqlens_q
           params.cu_seqlens_k->kparam_data_ptr(), // cu_seqlens_k
           CAST(&params.Max_seqlen_q), // Max_seqlen_q
           CAST(&params.Max_seqlen_k), // Max_seqlen_k
           CAST(&params.Head_dim), // Head_dim
           // CAST(&params.dropout_p), // dropout_p as constexpr 0
           // params.philox_seed_ptr->kparam_data_ptr(), // philox_seed_ptr as constexpr 0
           // params.philox_offset1->kparam_data_ptr(), // philox_offset1 as constexpr 0
           // CAST(&params.philox_offset2), // philox_offset2 as constexpr 0
           // params.philox_seed_output->kparam_data_ptr(), // philox_seed_output as constexpr 0
           // params.philox_offset_output->kparam_data_ptr(), // philox_offset_output as constexpr 0
           // params.encoded_softmax->kparam_data_ptr(), // encoded_softmax as constexpr nullptr
           CAST(&params.Window_left), // Window_left
           CAST(&params.Window_right), // Window_right
           params.persistent_atomic_counter->kparam_data_ptr(), // persistent_atomic_counter
           CAST(&params.Num_CU), // Num_CU
           CAST(&params.Batch), // Batch
           CAST(global_scratch)
         };
}
static std::vector<void*>
attn_fwd_pp_args_9(const OpAttnFwdParams& params,
                   hipDeviceptr_t* global_scratch) {
  return { params.Q->kparam_data_ptr(), // Q
           params.K->kparam_data_ptr(), // K
           params.V->kparam_data_ptr(), // V
           // params.B->kparam_data_ptr(), // B as constexpr nullptr
           // params.A->kparam_data_ptr(), // A as constexpr nullptr
           CAST(&params.Sm_scale), // Sm_scale
           params.L->kparam_data_ptr(), // L
           params.Out->kparam_data_ptr(), // Out
           params.Q->kparam_stride(0), // stride_qz
           params.Q->kparam_stride(1), // stride_qh
           params.Q->kparam_stride(2), // stride_qm
           params.K->kparam_stride(0), // stride_kz
           params.K->kparam_stride(1), // stride_kh
           params.K->kparam_stride(2), // stride_kn
           params.V->kparam_stride(0), // stride_vz
           params.V->kparam_stride(1), // stride_vh
           params.V->kparam_stride(2), // stride_vk
           params.Out->kparam_stride(0), // stride_oz
           params.Out->kparam_stride(1), // stride_oh
           params.Out->kparam_stride(2), // stride_om
           // params.B->kparam_stride(0), // stride_bz as constexpr 0
           // params.B->kparam_stride(1), // stride_bh as constexpr 0
           // params.B->kparam_stride(2), // stride_bm as constexpr 0
           // params.A->kparam_stride(0), // stride_az as constexpr 0
           CAST(&params.Num_head_q), // Num_head_q
           CAST(&params.Num_head_k), // Num_head_k
           CAST(&params.Num_seqlens), // Num_seqlens
           params.cu_seqlens_q->kparam_data_ptr(), // cu_seqlens_q
           params.cu_seqlens_k->kparam_data_ptr(), // cu_seqlens_k
           CAST(&params.Max_seqlen_q), // Max_seqlen_q
           CAST(&params.Max_seqlen_k), // Max_seqlen_k
           CAST(&params.Head_dim), // Head_dim
           CAST(&params.dropout_p), // dropout_p
           params.philox_seed_ptr->kparam_data_ptr(), // philox_seed_ptr
           params.philox_offset1->kparam_data_ptr(), // philox_offset1
           CAST(&params.philox_offset2), // philox_offset2
           params.philox_seed_output->kparam_data_ptr(), // philox_seed_output
           params.philox_offset_output->kparam_data_ptr(), // philox_offset_output
           // params.encoded_softmax->kparam_data_ptr(), // encoded_softmax as constexpr nullptr
           // CAST(&params.Window_left), // Window_left as constexpr 0
           // CAST(&params.Window_right), // Window_right as constexpr 0
           // params.persistent_atomic_counter->kparam_data_ptr(), // persistent_atomic_counter as constexpr 0
           CAST(&params.Num_CU), // Num_CU
           CAST(&params.Batch), // Batch
           CAST(global_scratch)
         };
}
static std::vector<void*>
attn_fwd_pp_args_10(const OpAttnFwdParams& params,
                    hipDeviceptr_t* global_scratch) {
  return { params.Q->kparam_data_ptr(), // Q
           params.K->kparam_data_ptr(), // K
           params.V->kparam_data_ptr(), // V
           params.B->kparam_data_ptr(), // B
           // params.A->kparam_data_ptr(), // A as constexpr nullptr
           CAST(&params.Sm_scale), // Sm_scale
           params.L->kparam_data_ptr(), // L
           params.Out->kparam_data_ptr(), // Out
           params.Q->kparam_stride(0), // stride_qz
           params.Q->kparam_stride(1), // stride_qh
           params.Q->kparam_stride(2), // stride_qm
           params.K->kparam_stride(0), // stride_kz
           params.K->kparam_stride(1), // stride_kh
           params.K->kparam_stride(2), // stride_kn
           params.V->kparam_stride(0), // stride_vz
           params.V->kparam_stride(1), // stride_vh
           params.V->kparam_stride(2), // stride_vk
           params.Out->kparam_stride(0), // stride_oz
           params.Out->kparam_stride(1), // stride_oh
           params.Out->kparam_stride(2), // stride_om
           params.B->kparam_stride(0), // stride_bz
           params.B->kparam_stride(1), // stride_bh
           params.B->kparam_stride(2), // stride_bm
           // params.A->kparam_stride(0), // stride_az as constexpr 0
           CAST(&params.Num_head_q), // Num_head_q
           CAST(&params.Num_head_k), // Num_head_k
           CAST(&params.Num_seqlens), // Num_seqlens
           params.cu_seqlens_q->kparam_data_ptr(), // cu_seqlens_q
           params.cu_seqlens_k->kparam_data_ptr(), // cu_seqlens_k
           CAST(&params.Max_seqlen_q), // Max_seqlen_q
           CAST(&params.Max_seqlen_k), // Max_seqlen_k
           CAST(&params.Head_dim), // Head_dim
           CAST(&params.dropout_p), // dropout_p
           params.philox_seed_ptr->kparam_data_ptr(), // philox_seed_ptr
           params.philox_offset1->kparam_data_ptr(), // philox_offset1
           CAST(&params.philox_offset2), // philox_offset2
           params.philox_seed_output->kparam_data_ptr(), // philox_seed_output
           params.philox_offset_output->kparam_data_ptr(), // philox_offset_output
           // params.encoded_softmax->kparam_data_ptr(), // encoded_softmax as constexpr nullptr
           // CAST(&params.Window_left), // Window_left as constexpr 0
           // CAST(&params.Window_right), // Window_right as constexpr 0
           // params.persistent_atomic_counter->kparam_data_ptr(), // persistent_atomic_counter as constexpr 0
           CAST(&params.Num_CU), // Num_CU
           CAST(&params.Batch), // Batch
           CAST(global_scratch)
         };
}
static std::vector<void*>
attn_fwd_pp_args_11(const OpAttnFwdParams& params,
                    hipDeviceptr_t* global_scratch) {
  return { params.Q->kparam_data_ptr(), // Q
           params.K->kparam_data_ptr(), // K
           params.V->kparam_data_ptr(), // V
           // params.B->kparam_data_ptr(), // B as constexpr nullptr
           // params.A->kparam_data_ptr(), // A as constexpr nullptr
           CAST(&params.Sm_scale), // Sm_scale
           params.L->kparam_data_ptr(), // L
           params.Out->kparam_data_ptr(), // Out
           params.Q->kparam_stride(0), // stride_qz
           params.Q->kparam_stride(1), // stride_qh
           params.Q->kparam_stride(2), // stride_qm
           params.K->kparam_stride(0), // stride_kz
           params.K->kparam_stride(1), // stride_kh
           params.K->kparam_stride(2), // stride_kn
           params.V->kparam_stride(0), // stride_vz
           params.V->kparam_stride(1), // stride_vh
           params.V->kparam_stride(2), // stride_vk
           params.Out->kparam_stride(0), // stride_oz
           params.Out->kparam_stride(1), // stride_oh
           params.Out->kparam_stride(2), // stride_om
           // params.B->kparam_stride(0), // stride_bz as constexpr 0
           // params.B->kparam_stride(1), // stride_bh as constexpr 0
           // params.B->kparam_stride(2), // stride_bm as constexpr 0
           // params.A->kparam_stride(0), // stride_az as constexpr 0
           CAST(&params.Num_head_q), // Num_head_q
           CAST(&params.Num_head_k), // Num_head_k
           CAST(&params.Num_seqlens), // Num_seqlens
           params.cu_seqlens_q->kparam_data_ptr(), // cu_seqlens_q
           params.cu_seqlens_k->kparam_data_ptr(), // cu_seqlens_k
           CAST(&params.Max_seqlen_q), // Max_seqlen_q
           CAST(&params.Max_seqlen_k), // Max_seqlen_k
           CAST(&params.Head_dim), // Head_dim
           CAST(&params.dropout_p), // dropout_p
           params.philox_seed_ptr->kparam_data_ptr(), // philox_seed_ptr
           params.philox_offset1->kparam_data_ptr(), // philox_offset1
           CAST(&params.philox_offset2), // philox_offset2
           params.philox_seed_output->kparam_data_ptr(), // philox_seed_output
           params.philox_offset_output->kparam_data_ptr(), // philox_offset_output
           // params.encoded_softmax->kparam_data_ptr(), // encoded_softmax as constexpr nullptr
           CAST(&params.Window_left), // Window_left
           CAST(&params.Window_right), // Window_right
           params.persistent_atomic_counter->kparam_data_ptr(), // persistent_atomic_counter
           CAST(&params.Num_CU), // Num_CU
           CAST(&params.Batch), // Batch
           CAST(global_scratch)
         };
}

namespace {
PP_FUNC prepare_arguments[ 12 ] = {
  attn_fwd_pp_args_0,
  attn_fwd_pp_args_1,
  attn_fwd_pp_args_2,
  attn_fwd_pp_args_3,
  attn_fwd_pp_args_4,
  attn_fwd_pp_args_5,
  attn_fwd_pp_args_6,
  attn_fwd_pp_args_7,
  attn_fwd_pp_args_8,
  attn_fwd_pp_args_9,
  attn_fwd_pp_args_10,
  attn_fwd_pp_args_11
};
}


const std::vector<std::string>& AttnFwdMetadata::get_Q_choices()
{
    static const std::vector<std::string> choices = { "*fp16:16", "*bf16:16", "*fp32:16" };
    return choices;
}

const std::vector<std::string>& AttnFwdMetadata::get_Sm_scale_choices()
{
    static const std::vector<std::string> choices = { "fp32" };
    return choices;
}

const std::vector<std::string>& AttnFwdMetadata::get_L_choices()
{
    static const std::vector<std::string> choices = { "*fp32:16" };
    return choices;
}

const std::vector<int>& AttnFwdMetadata::get_Q_descale_choices()
{
    static const std::vector<int> choices = { 0 };
    return choices;
}

const std::vector<std::string>& AttnFwdMetadata::get_Num_head_q_choices()
{
    static const std::vector<std::string> choices = { "i32" };
    return choices;
}

const std::vector<std::string>& AttnFwdMetadata::get_cu_seqlens_q_choices()
{
    static const std::vector<std::string> choices = { "*i32:16" };
    return choices;
}

const std::vector<int>& AttnFwdMetadata::get_BLOCK_DMODEL_choices()
{
    static const std::vector<int> choices = { 16, 32, 48, 64, 80, 96, 128, 160, 192, 224, 256, 512 };
    return choices;
}

const std::vector<bool>& AttnFwdMetadata::get_PADDED_HEAD_choices()
{
    static const std::vector<bool> choices = { false, true };
    return choices;
}

const std::vector<bool>& AttnFwdMetadata::get_ENABLE_DROPOUT_choices()
{
    static const std::vector<bool> choices = { false, true };
    return choices;
}

const std::vector<bool>& AttnFwdMetadata::get_RETURN_ENCODED_SOFTMAX_choices()
{
    static const std::vector<bool> choices = { false };
    return choices;
}

const std::vector<int>& AttnFwdMetadata::get_CAUSAL_TYPE_choices()
{
    static const std::vector<int> choices = { 0, 3 };
    return choices;
}

const std::vector<int>& AttnFwdMetadata::get_BIAS_TYPE_choices()
{
    static const std::vector<int> choices = { 0, 1 };
    return choices;
}

const std::vector<bool>& AttnFwdMetadata::get_USE_ALIBI_choices()
{
    static const std::vector<bool> choices = { false };
    return choices;
}

const std::vector<bool>& AttnFwdMetadata::get_INT8_choices()
{
    static const std::vector<bool> choices = { false };
    return choices;
}

const std::vector<std::string>& AttnFwdMetadata::get_Num_CU_choices()
{
    static const std::vector<std::string> choices = { "i32" };
    return choices;
}

namespace autotune {

const char attn_fwd_packed_string[] =
"0_2_32_16_False\0"
"wave4_warp2_stg1\0"
"0_2_32_16_True\0"
"wave1_warp2_stg1\0"
"wave2_warp2_stg1\0"
"wave3_warp2_stg1\0"
"0_2_64_32_False\0"
"0_2_64_32_True\0"
"0_2_64_64_False\0"
"wave3_warp4_stg1\0"
"0_2_64_64_True\0"
"wave2_warp4_stg1\0"
"wave4_warp4_stg1\0"
"0_2_128_64_False\0"
"0_2_128_64_True\0"
"0_2_128_128_False\0"
"wave1_warp4_stg1\0"
"0_2_128_128_True\0"
"2_2_32_16_True\0"
"2_2_64_32_False\0"
"2_2_64_32_True\0"
"2_2_64_64_False\0"
"2_2_64_64_True\0"
"2_2_128_64_False\0"
"2_2_128_64_True\0"
"2_2_128_128_False\0"
"2_2_128_128_True\0"
"2_2_32_16_False\0"
"0_2_16_16_False\0"
"0_2_16_16_True\0"
"0_2_32_32_False\0"
"0_2_32_32_True\0"
"2_2_16_16_True\0"
"2_2_32_32_True\0"
"2_2_32_32_False\0"
"2_2_16_16_False\0";

int attn_fwd__lut_lambda__0 (const OpAttnFwdParams& params, int mod_number, int8_t lut[1][10][10]) {
    auto Max_seqlen_q_binned_index = [] (int x) {
        if (x <= 16) return 0;
        if (x <= 32) return 1;
        if (x <= 64) return 2;
        if (x <= 128) return 3;
        if (x <= 256) return 4;
        if (x <= 512) return 5;
        if (x <= 1024) return 6;
        if (x <= 2048) return 7;
        if (x <= 4096) return 8;
        if (x <= 8192) return 9;
        return 9;
    }(params.Max_seqlen_q);
    auto Max_seqlen_k_binned_index = [] (int x) {
        if (x <= 16) return 0;
        if (x <= 32) return 1;
        if (x <= 64) return 2;
        if (x <= 128) return 3;
        if (x <= 256) return 4;
        if (x <= 512) return 5;
        if (x <= 1024) return 6;
        if (x <= 2048) return 7;
        if (x <= 4096) return 8;
        if (x <= 8192) return 9;
        return 9;
    }(params.Max_seqlen_k);
    return lut[mod_number][Max_seqlen_q_binned_index][Max_seqlen_k_binned_index];
};
int attn_fwd__lut_lambda__1 (const OpAttnFwdParams& params, int mod_number, int8_t lut[1][1]) {
    
    return lut[mod_number][0];
};

} // namespace autotune

AttnFwdContext::AutoTuneTableEntry
AttnFwdContext::autotune_table[][ 576 ] = {
    {
        &autotune::Autotune_attn_fwd__A0__F0,
        &autotune::Autotune_attn_fwd__A0__F1,
        &autotune::Autotune_attn_fwd__A0__F2,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F4,
        &autotune::Autotune_attn_fwd__A0__F5,
        &autotune::Autotune_attn_fwd__A0__F6,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F8,
        &autotune::Autotune_attn_fwd__A0__F9,
        &autotune::Autotune_attn_fwd__A0__F10,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F12,
        &autotune::Autotune_attn_fwd__A0__F13,
        &autotune::Autotune_attn_fwd__A0__F14,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F16,
        &autotune::Autotune_attn_fwd__A0__F17,
        &autotune::Autotune_attn_fwd__A0__F18,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F20,
        &autotune::Autotune_attn_fwd__A0__F21,
        &autotune::Autotune_attn_fwd__A0__F22,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F24,
        &autotune::Autotune_attn_fwd__A0__F25,
        &autotune::Autotune_attn_fwd__A0__F26,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F28,
        &autotune::Autotune_attn_fwd__A0__F29,
        &autotune::Autotune_attn_fwd__A0__F30,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F32,
        &autotune::Autotune_attn_fwd__A0__F33,
        &autotune::Autotune_attn_fwd__A0__F34,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F36,
        &autotune::Autotune_attn_fwd__A0__F37,
        &autotune::Autotune_attn_fwd__A0__F38,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F40,
        &autotune::Autotune_attn_fwd__A0__F41,
        &autotune::Autotune_attn_fwd__A0__F42,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F44,
        &autotune::Autotune_attn_fwd__A0__F45,
        &autotune::Autotune_attn_fwd__A0__F46,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F48,
        &autotune::Autotune_attn_fwd__A0__F49,
        &autotune::Autotune_attn_fwd__A0__F50,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F52,
        &autotune::Autotune_attn_fwd__A0__F53,
        &autotune::Autotune_attn_fwd__A0__F54,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F56,
        &autotune::Autotune_attn_fwd__A0__F57,
        &autotune::Autotune_attn_fwd__A0__F58,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F60,
        &autotune::Autotune_attn_fwd__A0__F61,
        &autotune::Autotune_attn_fwd__A0__F62,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F64,
        &autotune::Autotune_attn_fwd__A0__F65,
        &autotune::Autotune_attn_fwd__A0__F66,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F68,
        &autotune::Autotune_attn_fwd__A0__F69,
        &autotune::Autotune_attn_fwd__A0__F70,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F72,
        &autotune::Autotune_attn_fwd__A0__F73,
        &autotune::Autotune_attn_fwd__A0__F74,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F76,
        &autotune::Autotune_attn_fwd__A0__F77,
        &autotune::Autotune_attn_fwd__A0__F78,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F80,
        &autotune::Autotune_attn_fwd__A0__F81,
        &autotune::Autotune_attn_fwd__A0__F82,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F84,
        &autotune::Autotune_attn_fwd__A0__F85,
        &autotune::Autotune_attn_fwd__A0__F86,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F88,
        &autotune::Autotune_attn_fwd__A0__F89,
        &autotune::Autotune_attn_fwd__A0__F90,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F92,
        &autotune::Autotune_attn_fwd__A0__F93,
        &autotune::Autotune_attn_fwd__A0__F94,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F96,
        &autotune::Autotune_attn_fwd__A0__F97,
        &autotune::Autotune_attn_fwd__A0__F98,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F100,
        &autotune::Autotune_attn_fwd__A0__F101,
        &autotune::Autotune_attn_fwd__A0__F102,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F104,
        &autotune::Autotune_attn_fwd__A0__F105,
        &autotune::Autotune_attn_fwd__A0__F106,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F108,
        &autotune::Autotune_attn_fwd__A0__F109,
        &autotune::Autotune_attn_fwd__A0__F110,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F112,
        &autotune::Autotune_attn_fwd__A0__F113,
        &autotune::Autotune_attn_fwd__A0__F114,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F116,
        &autotune::Autotune_attn_fwd__A0__F117,
        &autotune::Autotune_attn_fwd__A0__F118,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F120,
        &autotune::Autotune_attn_fwd__A0__F121,
        &autotune::Autotune_attn_fwd__A0__F122,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F124,
        &autotune::Autotune_attn_fwd__A0__F125,
        &autotune::Autotune_attn_fwd__A0__F126,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F128,
        &autotune::Autotune_attn_fwd__A0__F129,
        &autotune::Autotune_attn_fwd__A0__F130,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F132,
        &autotune::Autotune_attn_fwd__A0__F133,
        &autotune::Autotune_attn_fwd__A0__F134,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F136,
        &autotune::Autotune_attn_fwd__A0__F137,
        &autotune::Autotune_attn_fwd__A0__F138,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F140,
        &autotune::Autotune_attn_fwd__A0__F141,
        &autotune::Autotune_attn_fwd__A0__F142,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F144,
        &autotune::Autotune_attn_fwd__A0__F145,
        &autotune::Autotune_attn_fwd__A0__F146,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F148,
        &autotune::Autotune_attn_fwd__A0__F149,
        &autotune::Autotune_attn_fwd__A0__F150,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F152,
        &autotune::Autotune_attn_fwd__A0__F153,
        &autotune::Autotune_attn_fwd__A0__F154,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F156,
        &autotune::Autotune_attn_fwd__A0__F157,
        &autotune::Autotune_attn_fwd__A0__F158,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F160,
        &autotune::Autotune_attn_fwd__A0__F161,
        &autotune::Autotune_attn_fwd__A0__F162,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F164,
        &autotune::Autotune_attn_fwd__A0__F165,
        &autotune::Autotune_attn_fwd__A0__F166,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F168,
        &autotune::Autotune_attn_fwd__A0__F169,
        &autotune::Autotune_attn_fwd__A0__F170,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F172,
        &autotune::Autotune_attn_fwd__A0__F173,
        &autotune::Autotune_attn_fwd__A0__F174,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F176,
        &autotune::Autotune_attn_fwd__A0__F177,
        &autotune::Autotune_attn_fwd__A0__F178,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F180,
        &autotune::Autotune_attn_fwd__A0__F181,
        &autotune::Autotune_attn_fwd__A0__F182,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F184,
        &autotune::Autotune_attn_fwd__A0__F185,
        &autotune::Autotune_attn_fwd__A0__F186,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F188,
        &autotune::Autotune_attn_fwd__A0__F189,
        &autotune::Autotune_attn_fwd__A0__F190,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F192,
        &autotune::Autotune_attn_fwd__A0__F193,
        &autotune::Autotune_attn_fwd__A0__F194,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F196,
        &autotune::Autotune_attn_fwd__A0__F197,
        &autotune::Autotune_attn_fwd__A0__F198,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F200,
        &autotune::Autotune_attn_fwd__A0__F201,
        &autotune::Autotune_attn_fwd__A0__F202,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F204,
        &autotune::Autotune_attn_fwd__A0__F205,
        &autotune::Autotune_attn_fwd__A0__F206,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F208,
        &autotune::Autotune_attn_fwd__A0__F209,
        &autotune::Autotune_attn_fwd__A0__F210,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F212,
        &autotune::Autotune_attn_fwd__A0__F213,
        &autotune::Autotune_attn_fwd__A0__F214,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F216,
        &autotune::Autotune_attn_fwd__A0__F217,
        &autotune::Autotune_attn_fwd__A0__F218,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F220,
        &autotune::Autotune_attn_fwd__A0__F221,
        &autotune::Autotune_attn_fwd__A0__F222,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F224,
        &autotune::Autotune_attn_fwd__A0__F225,
        &autotune::Autotune_attn_fwd__A0__F226,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F228,
        &autotune::Autotune_attn_fwd__A0__F229,
        &autotune::Autotune_attn_fwd__A0__F230,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F232,
        &autotune::Autotune_attn_fwd__A0__F233,
        &autotune::Autotune_attn_fwd__A0__F234,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F236,
        &autotune::Autotune_attn_fwd__A0__F237,
        &autotune::Autotune_attn_fwd__A0__F238,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F240,
        &autotune::Autotune_attn_fwd__A0__F241,
        &autotune::Autotune_attn_fwd__A0__F242,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F244,
        &autotune::Autotune_attn_fwd__A0__F245,
        &autotune::Autotune_attn_fwd__A0__F246,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F248,
        &autotune::Autotune_attn_fwd__A0__F249,
        &autotune::Autotune_attn_fwd__A0__F250,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F252,
        &autotune::Autotune_attn_fwd__A0__F253,
        &autotune::Autotune_attn_fwd__A0__F254,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F256,
        &autotune::Autotune_attn_fwd__A0__F257,
        &autotune::Autotune_attn_fwd__A0__F258,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F260,
        &autotune::Autotune_attn_fwd__A0__F261,
        &autotune::Autotune_attn_fwd__A0__F262,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F264,
        &autotune::Autotune_attn_fwd__A0__F265,
        &autotune::Autotune_attn_fwd__A0__F266,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F268,
        &autotune::Autotune_attn_fwd__A0__F269,
        &autotune::Autotune_attn_fwd__A0__F270,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F272,
        &autotune::Autotune_attn_fwd__A0__F273,
        &autotune::Autotune_attn_fwd__A0__F274,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F276,
        &autotune::Autotune_attn_fwd__A0__F277,
        &autotune::Autotune_attn_fwd__A0__F278,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F280,
        &autotune::Autotune_attn_fwd__A0__F281,
        &autotune::Autotune_attn_fwd__A0__F282,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F284,
        &autotune::Autotune_attn_fwd__A0__F285,
        &autotune::Autotune_attn_fwd__A0__F286,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F288,
        &autotune::Autotune_attn_fwd__A0__F289,
        &autotune::Autotune_attn_fwd__A0__F290,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F292,
        &autotune::Autotune_attn_fwd__A0__F293,
        &autotune::Autotune_attn_fwd__A0__F294,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F296,
        &autotune::Autotune_attn_fwd__A0__F297,
        &autotune::Autotune_attn_fwd__A0__F298,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F300,
        &autotune::Autotune_attn_fwd__A0__F301,
        &autotune::Autotune_attn_fwd__A0__F302,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F304,
        &autotune::Autotune_attn_fwd__A0__F305,
        &autotune::Autotune_attn_fwd__A0__F306,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F308,
        &autotune::Autotune_attn_fwd__A0__F309,
        &autotune::Autotune_attn_fwd__A0__F310,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F312,
        &autotune::Autotune_attn_fwd__A0__F313,
        &autotune::Autotune_attn_fwd__A0__F314,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F316,
        &autotune::Autotune_attn_fwd__A0__F317,
        &autotune::Autotune_attn_fwd__A0__F318,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F320,
        &autotune::Autotune_attn_fwd__A0__F321,
        &autotune::Autotune_attn_fwd__A0__F322,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F324,
        &autotune::Autotune_attn_fwd__A0__F325,
        &autotune::Autotune_attn_fwd__A0__F326,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F328,
        &autotune::Autotune_attn_fwd__A0__F329,
        &autotune::Autotune_attn_fwd__A0__F330,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F332,
        &autotune::Autotune_attn_fwd__A0__F333,
        &autotune::Autotune_attn_fwd__A0__F334,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F336,
        &autotune::Autotune_attn_fwd__A0__F337,
        &autotune::Autotune_attn_fwd__A0__F338,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F340,
        &autotune::Autotune_attn_fwd__A0__F341,
        &autotune::Autotune_attn_fwd__A0__F342,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F344,
        &autotune::Autotune_attn_fwd__A0__F345,
        &autotune::Autotune_attn_fwd__A0__F346,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F348,
        &autotune::Autotune_attn_fwd__A0__F349,
        &autotune::Autotune_attn_fwd__A0__F350,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F352,
        &autotune::Autotune_attn_fwd__A0__F353,
        &autotune::Autotune_attn_fwd__A0__F354,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F356,
        &autotune::Autotune_attn_fwd__A0__F357,
        &autotune::Autotune_attn_fwd__A0__F358,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F360,
        &autotune::Autotune_attn_fwd__A0__F361,
        &autotune::Autotune_attn_fwd__A0__F362,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F364,
        &autotune::Autotune_attn_fwd__A0__F365,
        &autotune::Autotune_attn_fwd__A0__F366,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F368,
        &autotune::Autotune_attn_fwd__A0__F369,
        &autotune::Autotune_attn_fwd__A0__F370,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F372,
        &autotune::Autotune_attn_fwd__A0__F373,
        &autotune::Autotune_attn_fwd__A0__F374,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F376,
        &autotune::Autotune_attn_fwd__A0__F377,
        &autotune::Autotune_attn_fwd__A0__F378,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F380,
        &autotune::Autotune_attn_fwd__A0__F381,
        &autotune::Autotune_attn_fwd__A0__F382,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F384,
        &autotune::Autotune_attn_fwd__A0__F385,
        &autotune::Autotune_attn_fwd__A0__F386,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F388,
        &autotune::Autotune_attn_fwd__A0__F389,
        &autotune::Autotune_attn_fwd__A0__F390,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F392,
        &autotune::Autotune_attn_fwd__A0__F393,
        &autotune::Autotune_attn_fwd__A0__F394,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F396,
        &autotune::Autotune_attn_fwd__A0__F397,
        &autotune::Autotune_attn_fwd__A0__F398,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F400,
        &autotune::Autotune_attn_fwd__A0__F401,
        &autotune::Autotune_attn_fwd__A0__F402,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F404,
        &autotune::Autotune_attn_fwd__A0__F405,
        &autotune::Autotune_attn_fwd__A0__F406,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F408,
        &autotune::Autotune_attn_fwd__A0__F409,
        &autotune::Autotune_attn_fwd__A0__F410,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F412,
        &autotune::Autotune_attn_fwd__A0__F413,
        &autotune::Autotune_attn_fwd__A0__F414,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F416,
        &autotune::Autotune_attn_fwd__A0__F417,
        &autotune::Autotune_attn_fwd__A0__F418,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F420,
        &autotune::Autotune_attn_fwd__A0__F421,
        &autotune::Autotune_attn_fwd__A0__F422,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F424,
        &autotune::Autotune_attn_fwd__A0__F425,
        &autotune::Autotune_attn_fwd__A0__F426,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F428,
        &autotune::Autotune_attn_fwd__A0__F429,
        &autotune::Autotune_attn_fwd__A0__F430,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F432,
        &autotune::Autotune_attn_fwd__A0__F433,
        &autotune::Autotune_attn_fwd__A0__F434,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F436,
        &autotune::Autotune_attn_fwd__A0__F437,
        &autotune::Autotune_attn_fwd__A0__F438,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F440,
        &autotune::Autotune_attn_fwd__A0__F441,
        &autotune::Autotune_attn_fwd__A0__F442,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F444,
        &autotune::Autotune_attn_fwd__A0__F445,
        &autotune::Autotune_attn_fwd__A0__F446,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F448,
        &autotune::Autotune_attn_fwd__A0__F449,
        &autotune::Autotune_attn_fwd__A0__F450,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F452,
        &autotune::Autotune_attn_fwd__A0__F453,
        &autotune::Autotune_attn_fwd__A0__F454,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F456,
        &autotune::Autotune_attn_fwd__A0__F457,
        &autotune::Autotune_attn_fwd__A0__F458,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F460,
        &autotune::Autotune_attn_fwd__A0__F461,
        &autotune::Autotune_attn_fwd__A0__F462,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F464,
        &autotune::Autotune_attn_fwd__A0__F465,
        &autotune::Autotune_attn_fwd__A0__F466,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F468,
        &autotune::Autotune_attn_fwd__A0__F469,
        &autotune::Autotune_attn_fwd__A0__F470,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F472,
        &autotune::Autotune_attn_fwd__A0__F473,
        &autotune::Autotune_attn_fwd__A0__F474,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F476,
        &autotune::Autotune_attn_fwd__A0__F477,
        &autotune::Autotune_attn_fwd__A0__F478,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F480,
        &autotune::Autotune_attn_fwd__A0__F481,
        &autotune::Autotune_attn_fwd__A0__F482,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F484,
        &autotune::Autotune_attn_fwd__A0__F485,
        &autotune::Autotune_attn_fwd__A0__F486,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F488,
        &autotune::Autotune_attn_fwd__A0__F489,
        &autotune::Autotune_attn_fwd__A0__F490,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F492,
        &autotune::Autotune_attn_fwd__A0__F493,
        &autotune::Autotune_attn_fwd__A0__F494,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F496,
        &autotune::Autotune_attn_fwd__A0__F497,
        &autotune::Autotune_attn_fwd__A0__F498,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F500,
        &autotune::Autotune_attn_fwd__A0__F501,
        &autotune::Autotune_attn_fwd__A0__F502,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F504,
        &autotune::Autotune_attn_fwd__A0__F505,
        &autotune::Autotune_attn_fwd__A0__F506,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F508,
        &autotune::Autotune_attn_fwd__A0__F509,
        &autotune::Autotune_attn_fwd__A0__F510,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F512,
        &autotune::Autotune_attn_fwd__A0__F513,
        &autotune::Autotune_attn_fwd__A0__F514,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F516,
        &autotune::Autotune_attn_fwd__A0__F517,
        &autotune::Autotune_attn_fwd__A0__F518,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F520,
        &autotune::Autotune_attn_fwd__A0__F521,
        &autotune::Autotune_attn_fwd__A0__F522,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F524,
        &autotune::Autotune_attn_fwd__A0__F525,
        &autotune::Autotune_attn_fwd__A0__F526,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F528,
        &autotune::Autotune_attn_fwd__A0__F529,
        &autotune::Autotune_attn_fwd__A0__F530,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F532,
        &autotune::Autotune_attn_fwd__A0__F533,
        &autotune::Autotune_attn_fwd__A0__F534,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F536,
        &autotune::Autotune_attn_fwd__A0__F537,
        &autotune::Autotune_attn_fwd__A0__F538,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F540,
        &autotune::Autotune_attn_fwd__A0__F541,
        &autotune::Autotune_attn_fwd__A0__F542,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F544,
        &autotune::Autotune_attn_fwd__A0__F545,
        &autotune::Autotune_attn_fwd__A0__F546,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F548,
        &autotune::Autotune_attn_fwd__A0__F549,
        &autotune::Autotune_attn_fwd__A0__F550,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F552,
        &autotune::Autotune_attn_fwd__A0__F553,
        &autotune::Autotune_attn_fwd__A0__F554,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F556,
        &autotune::Autotune_attn_fwd__A0__F557,
        &autotune::Autotune_attn_fwd__A0__F558,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F560,
        &autotune::Autotune_attn_fwd__A0__F561,
        &autotune::Autotune_attn_fwd__A0__F562,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F564,
        &autotune::Autotune_attn_fwd__A0__F565,
        &autotune::Autotune_attn_fwd__A0__F566,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F568,
        &autotune::Autotune_attn_fwd__A0__F569,
        &autotune::Autotune_attn_fwd__A0__F570,
        nullptr,
        &autotune::Autotune_attn_fwd__A0__F572,
        &autotune::Autotune_attn_fwd__A0__F573,
        &autotune::Autotune_attn_fwd__A0__F574,
        nullptr,
    },
    {
        &autotune::Autotune_attn_fwd__A1__F0,
        &autotune::Autotune_attn_fwd__A1__F1,
        &autotune::Autotune_attn_fwd__A1__F2,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F4,
        &autotune::Autotune_attn_fwd__A1__F5,
        &autotune::Autotune_attn_fwd__A1__F6,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F8,
        &autotune::Autotune_attn_fwd__A1__F9,
        &autotune::Autotune_attn_fwd__A1__F10,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F12,
        &autotune::Autotune_attn_fwd__A1__F13,
        &autotune::Autotune_attn_fwd__A1__F14,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F16,
        &autotune::Autotune_attn_fwd__A1__F17,
        &autotune::Autotune_attn_fwd__A1__F18,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F20,
        &autotune::Autotune_attn_fwd__A1__F21,
        &autotune::Autotune_attn_fwd__A1__F22,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F24,
        &autotune::Autotune_attn_fwd__A1__F25,
        &autotune::Autotune_attn_fwd__A1__F26,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F28,
        &autotune::Autotune_attn_fwd__A1__F29,
        &autotune::Autotune_attn_fwd__A1__F30,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F32,
        &autotune::Autotune_attn_fwd__A1__F33,
        &autotune::Autotune_attn_fwd__A1__F34,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F36,
        &autotune::Autotune_attn_fwd__A1__F37,
        &autotune::Autotune_attn_fwd__A1__F38,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F40,
        &autotune::Autotune_attn_fwd__A1__F41,
        &autotune::Autotune_attn_fwd__A1__F42,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F44,
        &autotune::Autotune_attn_fwd__A1__F45,
        &autotune::Autotune_attn_fwd__A1__F46,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F48,
        &autotune::Autotune_attn_fwd__A1__F49,
        &autotune::Autotune_attn_fwd__A1__F50,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F52,
        &autotune::Autotune_attn_fwd__A1__F53,
        &autotune::Autotune_attn_fwd__A1__F54,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F56,
        &autotune::Autotune_attn_fwd__A1__F57,
        &autotune::Autotune_attn_fwd__A1__F58,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F60,
        &autotune::Autotune_attn_fwd__A1__F61,
        &autotune::Autotune_attn_fwd__A1__F62,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F64,
        &autotune::Autotune_attn_fwd__A1__F65,
        &autotune::Autotune_attn_fwd__A1__F66,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F68,
        &autotune::Autotune_attn_fwd__A1__F69,
        &autotune::Autotune_attn_fwd__A1__F70,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F72,
        &autotune::Autotune_attn_fwd__A1__F73,
        &autotune::Autotune_attn_fwd__A1__F74,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F76,
        &autotune::Autotune_attn_fwd__A1__F77,
        &autotune::Autotune_attn_fwd__A1__F78,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F80,
        &autotune::Autotune_attn_fwd__A1__F81,
        &autotune::Autotune_attn_fwd__A1__F82,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F84,
        &autotune::Autotune_attn_fwd__A1__F85,
        &autotune::Autotune_attn_fwd__A1__F86,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F88,
        &autotune::Autotune_attn_fwd__A1__F89,
        &autotune::Autotune_attn_fwd__A1__F90,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F92,
        &autotune::Autotune_attn_fwd__A1__F93,
        &autotune::Autotune_attn_fwd__A1__F94,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F96,
        &autotune::Autotune_attn_fwd__A1__F97,
        &autotune::Autotune_attn_fwd__A1__F98,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F100,
        &autotune::Autotune_attn_fwd__A1__F101,
        &autotune::Autotune_attn_fwd__A1__F102,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F104,
        &autotune::Autotune_attn_fwd__A1__F105,
        &autotune::Autotune_attn_fwd__A1__F106,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F108,
        &autotune::Autotune_attn_fwd__A1__F109,
        &autotune::Autotune_attn_fwd__A1__F110,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F112,
        &autotune::Autotune_attn_fwd__A1__F113,
        &autotune::Autotune_attn_fwd__A1__F114,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F116,
        &autotune::Autotune_attn_fwd__A1__F117,
        &autotune::Autotune_attn_fwd__A1__F118,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F120,
        &autotune::Autotune_attn_fwd__A1__F121,
        &autotune::Autotune_attn_fwd__A1__F122,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F124,
        &autotune::Autotune_attn_fwd__A1__F125,
        &autotune::Autotune_attn_fwd__A1__F126,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F128,
        &autotune::Autotune_attn_fwd__A1__F129,
        &autotune::Autotune_attn_fwd__A1__F130,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F132,
        &autotune::Autotune_attn_fwd__A1__F133,
        &autotune::Autotune_attn_fwd__A1__F134,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F136,
        &autotune::Autotune_attn_fwd__A1__F137,
        &autotune::Autotune_attn_fwd__A1__F138,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F140,
        &autotune::Autotune_attn_fwd__A1__F141,
        &autotune::Autotune_attn_fwd__A1__F142,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F144,
        &autotune::Autotune_attn_fwd__A1__F145,
        &autotune::Autotune_attn_fwd__A1__F146,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F148,
        &autotune::Autotune_attn_fwd__A1__F149,
        &autotune::Autotune_attn_fwd__A1__F150,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F152,
        &autotune::Autotune_attn_fwd__A1__F153,
        &autotune::Autotune_attn_fwd__A1__F154,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F156,
        &autotune::Autotune_attn_fwd__A1__F157,
        &autotune::Autotune_attn_fwd__A1__F158,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F160,
        &autotune::Autotune_attn_fwd__A1__F161,
        &autotune::Autotune_attn_fwd__A1__F162,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F164,
        &autotune::Autotune_attn_fwd__A1__F165,
        &autotune::Autotune_attn_fwd__A1__F166,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F168,
        &autotune::Autotune_attn_fwd__A1__F169,
        &autotune::Autotune_attn_fwd__A1__F170,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F172,
        &autotune::Autotune_attn_fwd__A1__F173,
        &autotune::Autotune_attn_fwd__A1__F174,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F176,
        &autotune::Autotune_attn_fwd__A1__F177,
        &autotune::Autotune_attn_fwd__A1__F178,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F180,
        &autotune::Autotune_attn_fwd__A1__F181,
        &autotune::Autotune_attn_fwd__A1__F182,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F184,
        &autotune::Autotune_attn_fwd__A1__F185,
        &autotune::Autotune_attn_fwd__A1__F186,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F188,
        &autotune::Autotune_attn_fwd__A1__F189,
        &autotune::Autotune_attn_fwd__A1__F190,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F192,
        &autotune::Autotune_attn_fwd__A1__F193,
        &autotune::Autotune_attn_fwd__A1__F194,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F196,
        &autotune::Autotune_attn_fwd__A1__F197,
        &autotune::Autotune_attn_fwd__A1__F198,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F200,
        &autotune::Autotune_attn_fwd__A1__F201,
        &autotune::Autotune_attn_fwd__A1__F202,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F204,
        &autotune::Autotune_attn_fwd__A1__F205,
        &autotune::Autotune_attn_fwd__A1__F206,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F208,
        &autotune::Autotune_attn_fwd__A1__F209,
        &autotune::Autotune_attn_fwd__A1__F210,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F212,
        &autotune::Autotune_attn_fwd__A1__F213,
        &autotune::Autotune_attn_fwd__A1__F214,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F216,
        &autotune::Autotune_attn_fwd__A1__F217,
        &autotune::Autotune_attn_fwd__A1__F218,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F220,
        &autotune::Autotune_attn_fwd__A1__F221,
        &autotune::Autotune_attn_fwd__A1__F222,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F224,
        &autotune::Autotune_attn_fwd__A1__F225,
        &autotune::Autotune_attn_fwd__A1__F226,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F228,
        &autotune::Autotune_attn_fwd__A1__F229,
        &autotune::Autotune_attn_fwd__A1__F230,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F232,
        &autotune::Autotune_attn_fwd__A1__F233,
        &autotune::Autotune_attn_fwd__A1__F234,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F236,
        &autotune::Autotune_attn_fwd__A1__F237,
        &autotune::Autotune_attn_fwd__A1__F238,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F240,
        &autotune::Autotune_attn_fwd__A1__F241,
        &autotune::Autotune_attn_fwd__A1__F242,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F244,
        &autotune::Autotune_attn_fwd__A1__F245,
        &autotune::Autotune_attn_fwd__A1__F246,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F248,
        &autotune::Autotune_attn_fwd__A1__F249,
        &autotune::Autotune_attn_fwd__A1__F250,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F252,
        &autotune::Autotune_attn_fwd__A1__F253,
        &autotune::Autotune_attn_fwd__A1__F254,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F256,
        &autotune::Autotune_attn_fwd__A1__F257,
        &autotune::Autotune_attn_fwd__A1__F258,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F260,
        &autotune::Autotune_attn_fwd__A1__F261,
        &autotune::Autotune_attn_fwd__A1__F262,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F264,
        &autotune::Autotune_attn_fwd__A1__F265,
        &autotune::Autotune_attn_fwd__A1__F266,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F268,
        &autotune::Autotune_attn_fwd__A1__F269,
        &autotune::Autotune_attn_fwd__A1__F270,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F272,
        &autotune::Autotune_attn_fwd__A1__F273,
        &autotune::Autotune_attn_fwd__A1__F274,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F276,
        &autotune::Autotune_attn_fwd__A1__F277,
        &autotune::Autotune_attn_fwd__A1__F278,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F280,
        &autotune::Autotune_attn_fwd__A1__F281,
        &autotune::Autotune_attn_fwd__A1__F282,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F284,
        &autotune::Autotune_attn_fwd__A1__F285,
        &autotune::Autotune_attn_fwd__A1__F286,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F288,
        &autotune::Autotune_attn_fwd__A1__F289,
        &autotune::Autotune_attn_fwd__A1__F290,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F292,
        &autotune::Autotune_attn_fwd__A1__F293,
        &autotune::Autotune_attn_fwd__A1__F294,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F296,
        &autotune::Autotune_attn_fwd__A1__F297,
        &autotune::Autotune_attn_fwd__A1__F298,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F300,
        &autotune::Autotune_attn_fwd__A1__F301,
        &autotune::Autotune_attn_fwd__A1__F302,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F304,
        &autotune::Autotune_attn_fwd__A1__F305,
        &autotune::Autotune_attn_fwd__A1__F306,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F308,
        &autotune::Autotune_attn_fwd__A1__F309,
        &autotune::Autotune_attn_fwd__A1__F310,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F312,
        &autotune::Autotune_attn_fwd__A1__F313,
        &autotune::Autotune_attn_fwd__A1__F314,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F316,
        &autotune::Autotune_attn_fwd__A1__F317,
        &autotune::Autotune_attn_fwd__A1__F318,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F320,
        &autotune::Autotune_attn_fwd__A1__F321,
        &autotune::Autotune_attn_fwd__A1__F322,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F324,
        &autotune::Autotune_attn_fwd__A1__F325,
        &autotune::Autotune_attn_fwd__A1__F326,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F328,
        &autotune::Autotune_attn_fwd__A1__F329,
        &autotune::Autotune_attn_fwd__A1__F330,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F332,
        &autotune::Autotune_attn_fwd__A1__F333,
        &autotune::Autotune_attn_fwd__A1__F334,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F336,
        &autotune::Autotune_attn_fwd__A1__F337,
        &autotune::Autotune_attn_fwd__A1__F338,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F340,
        &autotune::Autotune_attn_fwd__A1__F341,
        &autotune::Autotune_attn_fwd__A1__F342,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F344,
        &autotune::Autotune_attn_fwd__A1__F345,
        &autotune::Autotune_attn_fwd__A1__F346,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F348,
        &autotune::Autotune_attn_fwd__A1__F349,
        &autotune::Autotune_attn_fwd__A1__F350,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F352,
        &autotune::Autotune_attn_fwd__A1__F353,
        &autotune::Autotune_attn_fwd__A1__F354,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F356,
        &autotune::Autotune_attn_fwd__A1__F357,
        &autotune::Autotune_attn_fwd__A1__F358,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F360,
        &autotune::Autotune_attn_fwd__A1__F361,
        &autotune::Autotune_attn_fwd__A1__F362,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F364,
        &autotune::Autotune_attn_fwd__A1__F365,
        &autotune::Autotune_attn_fwd__A1__F366,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F368,
        &autotune::Autotune_attn_fwd__A1__F369,
        &autotune::Autotune_attn_fwd__A1__F370,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F372,
        &autotune::Autotune_attn_fwd__A1__F373,
        &autotune::Autotune_attn_fwd__A1__F374,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F376,
        &autotune::Autotune_attn_fwd__A1__F377,
        &autotune::Autotune_attn_fwd__A1__F378,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F380,
        &autotune::Autotune_attn_fwd__A1__F381,
        &autotune::Autotune_attn_fwd__A1__F382,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F384,
        &autotune::Autotune_attn_fwd__A1__F385,
        &autotune::Autotune_attn_fwd__A1__F386,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F388,
        &autotune::Autotune_attn_fwd__A1__F389,
        &autotune::Autotune_attn_fwd__A1__F390,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F392,
        &autotune::Autotune_attn_fwd__A1__F393,
        &autotune::Autotune_attn_fwd__A1__F394,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F396,
        &autotune::Autotune_attn_fwd__A1__F397,
        &autotune::Autotune_attn_fwd__A1__F398,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F400,
        &autotune::Autotune_attn_fwd__A1__F401,
        &autotune::Autotune_attn_fwd__A1__F402,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F404,
        &autotune::Autotune_attn_fwd__A1__F405,
        &autotune::Autotune_attn_fwd__A1__F406,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F408,
        &autotune::Autotune_attn_fwd__A1__F409,
        &autotune::Autotune_attn_fwd__A1__F410,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F412,
        &autotune::Autotune_attn_fwd__A1__F413,
        &autotune::Autotune_attn_fwd__A1__F414,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F416,
        &autotune::Autotune_attn_fwd__A1__F417,
        &autotune::Autotune_attn_fwd__A1__F418,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F420,
        &autotune::Autotune_attn_fwd__A1__F421,
        &autotune::Autotune_attn_fwd__A1__F422,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F424,
        &autotune::Autotune_attn_fwd__A1__F425,
        &autotune::Autotune_attn_fwd__A1__F426,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F428,
        &autotune::Autotune_attn_fwd__A1__F429,
        &autotune::Autotune_attn_fwd__A1__F430,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F432,
        &autotune::Autotune_attn_fwd__A1__F433,
        &autotune::Autotune_attn_fwd__A1__F434,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F436,
        &autotune::Autotune_attn_fwd__A1__F437,
        &autotune::Autotune_attn_fwd__A1__F438,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F440,
        &autotune::Autotune_attn_fwd__A1__F441,
        &autotune::Autotune_attn_fwd__A1__F442,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F444,
        &autotune::Autotune_attn_fwd__A1__F445,
        &autotune::Autotune_attn_fwd__A1__F446,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F448,
        &autotune::Autotune_attn_fwd__A1__F449,
        &autotune::Autotune_attn_fwd__A1__F450,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F452,
        &autotune::Autotune_attn_fwd__A1__F453,
        &autotune::Autotune_attn_fwd__A1__F454,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F456,
        &autotune::Autotune_attn_fwd__A1__F457,
        &autotune::Autotune_attn_fwd__A1__F458,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F460,
        &autotune::Autotune_attn_fwd__A1__F461,
        &autotune::Autotune_attn_fwd__A1__F462,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F464,
        &autotune::Autotune_attn_fwd__A1__F465,
        &autotune::Autotune_attn_fwd__A1__F466,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F468,
        &autotune::Autotune_attn_fwd__A1__F469,
        &autotune::Autotune_attn_fwd__A1__F470,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F472,
        &autotune::Autotune_attn_fwd__A1__F473,
        &autotune::Autotune_attn_fwd__A1__F474,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F476,
        &autotune::Autotune_attn_fwd__A1__F477,
        &autotune::Autotune_attn_fwd__A1__F478,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F480,
        &autotune::Autotune_attn_fwd__A1__F481,
        &autotune::Autotune_attn_fwd__A1__F482,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F484,
        &autotune::Autotune_attn_fwd__A1__F485,
        &autotune::Autotune_attn_fwd__A1__F486,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F488,
        &autotune::Autotune_attn_fwd__A1__F489,
        &autotune::Autotune_attn_fwd__A1__F490,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F492,
        &autotune::Autotune_attn_fwd__A1__F493,
        &autotune::Autotune_attn_fwd__A1__F494,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F496,
        &autotune::Autotune_attn_fwd__A1__F497,
        &autotune::Autotune_attn_fwd__A1__F498,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F500,
        &autotune::Autotune_attn_fwd__A1__F501,
        &autotune::Autotune_attn_fwd__A1__F502,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F504,
        &autotune::Autotune_attn_fwd__A1__F505,
        &autotune::Autotune_attn_fwd__A1__F506,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F508,
        &autotune::Autotune_attn_fwd__A1__F509,
        &autotune::Autotune_attn_fwd__A1__F510,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F512,
        &autotune::Autotune_attn_fwd__A1__F513,
        &autotune::Autotune_attn_fwd__A1__F514,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F516,
        &autotune::Autotune_attn_fwd__A1__F517,
        &autotune::Autotune_attn_fwd__A1__F518,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F520,
        &autotune::Autotune_attn_fwd__A1__F521,
        &autotune::Autotune_attn_fwd__A1__F522,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F524,
        &autotune::Autotune_attn_fwd__A1__F525,
        &autotune::Autotune_attn_fwd__A1__F526,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F528,
        &autotune::Autotune_attn_fwd__A1__F529,
        &autotune::Autotune_attn_fwd__A1__F530,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F532,
        &autotune::Autotune_attn_fwd__A1__F533,
        &autotune::Autotune_attn_fwd__A1__F534,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F536,
        &autotune::Autotune_attn_fwd__A1__F537,
        &autotune::Autotune_attn_fwd__A1__F538,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F540,
        &autotune::Autotune_attn_fwd__A1__F541,
        &autotune::Autotune_attn_fwd__A1__F542,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F544,
        &autotune::Autotune_attn_fwd__A1__F545,
        &autotune::Autotune_attn_fwd__A1__F546,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F548,
        &autotune::Autotune_attn_fwd__A1__F549,
        &autotune::Autotune_attn_fwd__A1__F550,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F552,
        &autotune::Autotune_attn_fwd__A1__F553,
        &autotune::Autotune_attn_fwd__A1__F554,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F556,
        &autotune::Autotune_attn_fwd__A1__F557,
        &autotune::Autotune_attn_fwd__A1__F558,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F560,
        &autotune::Autotune_attn_fwd__A1__F561,
        &autotune::Autotune_attn_fwd__A1__F562,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F564,
        &autotune::Autotune_attn_fwd__A1__F565,
        &autotune::Autotune_attn_fwd__A1__F566,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F568,
        &autotune::Autotune_attn_fwd__A1__F569,
        &autotune::Autotune_attn_fwd__A1__F570,
        nullptr,
        &autotune::Autotune_attn_fwd__A1__F572,
        &autotune::Autotune_attn_fwd__A1__F573,
        &autotune::Autotune_attn_fwd__A1__F574,
        nullptr,
    },
    {
        &autotune::Autotune_attn_fwd__A2__F0,
        &autotune::Autotune_attn_fwd__A2__F1,
        &autotune::Autotune_attn_fwd__A2__F2,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F4,
        &autotune::Autotune_attn_fwd__A2__F5,
        &autotune::Autotune_attn_fwd__A2__F6,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F8,
        &autotune::Autotune_attn_fwd__A2__F9,
        &autotune::Autotune_attn_fwd__A2__F10,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F12,
        &autotune::Autotune_attn_fwd__A2__F13,
        &autotune::Autotune_attn_fwd__A2__F14,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F16,
        &autotune::Autotune_attn_fwd__A2__F17,
        &autotune::Autotune_attn_fwd__A2__F18,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F20,
        &autotune::Autotune_attn_fwd__A2__F21,
        &autotune::Autotune_attn_fwd__A2__F22,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F24,
        &autotune::Autotune_attn_fwd__A2__F25,
        &autotune::Autotune_attn_fwd__A2__F26,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F28,
        &autotune::Autotune_attn_fwd__A2__F29,
        &autotune::Autotune_attn_fwd__A2__F30,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F32,
        &autotune::Autotune_attn_fwd__A2__F33,
        &autotune::Autotune_attn_fwd__A2__F34,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F36,
        &autotune::Autotune_attn_fwd__A2__F37,
        &autotune::Autotune_attn_fwd__A2__F38,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F40,
        &autotune::Autotune_attn_fwd__A2__F41,
        &autotune::Autotune_attn_fwd__A2__F42,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F44,
        &autotune::Autotune_attn_fwd__A2__F45,
        &autotune::Autotune_attn_fwd__A2__F46,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F48,
        &autotune::Autotune_attn_fwd__A2__F49,
        &autotune::Autotune_attn_fwd__A2__F50,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F52,
        &autotune::Autotune_attn_fwd__A2__F53,
        &autotune::Autotune_attn_fwd__A2__F54,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F56,
        &autotune::Autotune_attn_fwd__A2__F57,
        &autotune::Autotune_attn_fwd__A2__F58,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F60,
        &autotune::Autotune_attn_fwd__A2__F61,
        &autotune::Autotune_attn_fwd__A2__F62,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F64,
        &autotune::Autotune_attn_fwd__A2__F65,
        &autotune::Autotune_attn_fwd__A2__F66,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F68,
        &autotune::Autotune_attn_fwd__A2__F69,
        &autotune::Autotune_attn_fwd__A2__F70,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F72,
        &autotune::Autotune_attn_fwd__A2__F73,
        &autotune::Autotune_attn_fwd__A2__F74,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F76,
        &autotune::Autotune_attn_fwd__A2__F77,
        &autotune::Autotune_attn_fwd__A2__F78,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F80,
        &autotune::Autotune_attn_fwd__A2__F81,
        &autotune::Autotune_attn_fwd__A2__F82,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F84,
        &autotune::Autotune_attn_fwd__A2__F85,
        &autotune::Autotune_attn_fwd__A2__F86,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F88,
        &autotune::Autotune_attn_fwd__A2__F89,
        &autotune::Autotune_attn_fwd__A2__F90,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F92,
        &autotune::Autotune_attn_fwd__A2__F93,
        &autotune::Autotune_attn_fwd__A2__F94,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F96,
        &autotune::Autotune_attn_fwd__A2__F97,
        &autotune::Autotune_attn_fwd__A2__F98,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F100,
        &autotune::Autotune_attn_fwd__A2__F101,
        &autotune::Autotune_attn_fwd__A2__F102,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F104,
        &autotune::Autotune_attn_fwd__A2__F105,
        &autotune::Autotune_attn_fwd__A2__F106,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F108,
        &autotune::Autotune_attn_fwd__A2__F109,
        &autotune::Autotune_attn_fwd__A2__F110,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F112,
        &autotune::Autotune_attn_fwd__A2__F113,
        &autotune::Autotune_attn_fwd__A2__F114,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F116,
        &autotune::Autotune_attn_fwd__A2__F117,
        &autotune::Autotune_attn_fwd__A2__F118,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F120,
        &autotune::Autotune_attn_fwd__A2__F121,
        &autotune::Autotune_attn_fwd__A2__F122,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F124,
        &autotune::Autotune_attn_fwd__A2__F125,
        &autotune::Autotune_attn_fwd__A2__F126,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F128,
        &autotune::Autotune_attn_fwd__A2__F129,
        &autotune::Autotune_attn_fwd__A2__F130,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F132,
        &autotune::Autotune_attn_fwd__A2__F133,
        &autotune::Autotune_attn_fwd__A2__F134,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F136,
        &autotune::Autotune_attn_fwd__A2__F137,
        &autotune::Autotune_attn_fwd__A2__F138,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F140,
        &autotune::Autotune_attn_fwd__A2__F141,
        &autotune::Autotune_attn_fwd__A2__F142,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F144,
        &autotune::Autotune_attn_fwd__A2__F145,
        &autotune::Autotune_attn_fwd__A2__F146,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F148,
        &autotune::Autotune_attn_fwd__A2__F149,
        &autotune::Autotune_attn_fwd__A2__F150,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F152,
        &autotune::Autotune_attn_fwd__A2__F153,
        &autotune::Autotune_attn_fwd__A2__F154,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F156,
        &autotune::Autotune_attn_fwd__A2__F157,
        &autotune::Autotune_attn_fwd__A2__F158,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F160,
        &autotune::Autotune_attn_fwd__A2__F161,
        &autotune::Autotune_attn_fwd__A2__F162,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F164,
        &autotune::Autotune_attn_fwd__A2__F165,
        &autotune::Autotune_attn_fwd__A2__F166,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F168,
        &autotune::Autotune_attn_fwd__A2__F169,
        &autotune::Autotune_attn_fwd__A2__F170,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F172,
        &autotune::Autotune_attn_fwd__A2__F173,
        &autotune::Autotune_attn_fwd__A2__F174,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F176,
        &autotune::Autotune_attn_fwd__A2__F177,
        &autotune::Autotune_attn_fwd__A2__F178,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F180,
        &autotune::Autotune_attn_fwd__A2__F181,
        &autotune::Autotune_attn_fwd__A2__F182,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F184,
        &autotune::Autotune_attn_fwd__A2__F185,
        &autotune::Autotune_attn_fwd__A2__F186,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F188,
        &autotune::Autotune_attn_fwd__A2__F189,
        &autotune::Autotune_attn_fwd__A2__F190,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F192,
        &autotune::Autotune_attn_fwd__A2__F193,
        &autotune::Autotune_attn_fwd__A2__F194,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F196,
        &autotune::Autotune_attn_fwd__A2__F197,
        &autotune::Autotune_attn_fwd__A2__F198,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F200,
        &autotune::Autotune_attn_fwd__A2__F201,
        &autotune::Autotune_attn_fwd__A2__F202,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F204,
        &autotune::Autotune_attn_fwd__A2__F205,
        &autotune::Autotune_attn_fwd__A2__F206,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F208,
        &autotune::Autotune_attn_fwd__A2__F209,
        &autotune::Autotune_attn_fwd__A2__F210,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F212,
        &autotune::Autotune_attn_fwd__A2__F213,
        &autotune::Autotune_attn_fwd__A2__F214,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F216,
        &autotune::Autotune_attn_fwd__A2__F217,
        &autotune::Autotune_attn_fwd__A2__F218,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F220,
        &autotune::Autotune_attn_fwd__A2__F221,
        &autotune::Autotune_attn_fwd__A2__F222,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F224,
        &autotune::Autotune_attn_fwd__A2__F225,
        &autotune::Autotune_attn_fwd__A2__F226,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F228,
        &autotune::Autotune_attn_fwd__A2__F229,
        &autotune::Autotune_attn_fwd__A2__F230,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F232,
        &autotune::Autotune_attn_fwd__A2__F233,
        &autotune::Autotune_attn_fwd__A2__F234,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F236,
        &autotune::Autotune_attn_fwd__A2__F237,
        &autotune::Autotune_attn_fwd__A2__F238,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F240,
        &autotune::Autotune_attn_fwd__A2__F241,
        &autotune::Autotune_attn_fwd__A2__F242,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F244,
        &autotune::Autotune_attn_fwd__A2__F245,
        &autotune::Autotune_attn_fwd__A2__F246,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F248,
        &autotune::Autotune_attn_fwd__A2__F249,
        &autotune::Autotune_attn_fwd__A2__F250,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F252,
        &autotune::Autotune_attn_fwd__A2__F253,
        &autotune::Autotune_attn_fwd__A2__F254,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F256,
        &autotune::Autotune_attn_fwd__A2__F257,
        &autotune::Autotune_attn_fwd__A2__F258,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F260,
        &autotune::Autotune_attn_fwd__A2__F261,
        &autotune::Autotune_attn_fwd__A2__F262,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F264,
        &autotune::Autotune_attn_fwd__A2__F265,
        &autotune::Autotune_attn_fwd__A2__F266,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F268,
        &autotune::Autotune_attn_fwd__A2__F269,
        &autotune::Autotune_attn_fwd__A2__F270,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F272,
        &autotune::Autotune_attn_fwd__A2__F273,
        &autotune::Autotune_attn_fwd__A2__F274,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F276,
        &autotune::Autotune_attn_fwd__A2__F277,
        &autotune::Autotune_attn_fwd__A2__F278,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F280,
        &autotune::Autotune_attn_fwd__A2__F281,
        &autotune::Autotune_attn_fwd__A2__F282,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F284,
        &autotune::Autotune_attn_fwd__A2__F285,
        &autotune::Autotune_attn_fwd__A2__F286,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F288,
        &autotune::Autotune_attn_fwd__A2__F289,
        &autotune::Autotune_attn_fwd__A2__F290,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F292,
        &autotune::Autotune_attn_fwd__A2__F293,
        &autotune::Autotune_attn_fwd__A2__F294,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F296,
        &autotune::Autotune_attn_fwd__A2__F297,
        &autotune::Autotune_attn_fwd__A2__F298,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F300,
        &autotune::Autotune_attn_fwd__A2__F301,
        &autotune::Autotune_attn_fwd__A2__F302,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F304,
        &autotune::Autotune_attn_fwd__A2__F305,
        &autotune::Autotune_attn_fwd__A2__F306,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F308,
        &autotune::Autotune_attn_fwd__A2__F309,
        &autotune::Autotune_attn_fwd__A2__F310,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F312,
        &autotune::Autotune_attn_fwd__A2__F313,
        &autotune::Autotune_attn_fwd__A2__F314,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F316,
        &autotune::Autotune_attn_fwd__A2__F317,
        &autotune::Autotune_attn_fwd__A2__F318,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F320,
        &autotune::Autotune_attn_fwd__A2__F321,
        &autotune::Autotune_attn_fwd__A2__F322,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F324,
        &autotune::Autotune_attn_fwd__A2__F325,
        &autotune::Autotune_attn_fwd__A2__F326,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F328,
        &autotune::Autotune_attn_fwd__A2__F329,
        &autotune::Autotune_attn_fwd__A2__F330,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F332,
        &autotune::Autotune_attn_fwd__A2__F333,
        &autotune::Autotune_attn_fwd__A2__F334,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F336,
        &autotune::Autotune_attn_fwd__A2__F337,
        &autotune::Autotune_attn_fwd__A2__F338,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F340,
        &autotune::Autotune_attn_fwd__A2__F341,
        &autotune::Autotune_attn_fwd__A2__F342,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F344,
        &autotune::Autotune_attn_fwd__A2__F345,
        &autotune::Autotune_attn_fwd__A2__F346,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F348,
        &autotune::Autotune_attn_fwd__A2__F349,
        &autotune::Autotune_attn_fwd__A2__F350,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F352,
        &autotune::Autotune_attn_fwd__A2__F353,
        &autotune::Autotune_attn_fwd__A2__F354,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F356,
        &autotune::Autotune_attn_fwd__A2__F357,
        &autotune::Autotune_attn_fwd__A2__F358,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F360,
        &autotune::Autotune_attn_fwd__A2__F361,
        &autotune::Autotune_attn_fwd__A2__F362,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F364,
        &autotune::Autotune_attn_fwd__A2__F365,
        &autotune::Autotune_attn_fwd__A2__F366,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F368,
        &autotune::Autotune_attn_fwd__A2__F369,
        &autotune::Autotune_attn_fwd__A2__F370,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F372,
        &autotune::Autotune_attn_fwd__A2__F373,
        &autotune::Autotune_attn_fwd__A2__F374,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F376,
        &autotune::Autotune_attn_fwd__A2__F377,
        &autotune::Autotune_attn_fwd__A2__F378,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F380,
        &autotune::Autotune_attn_fwd__A2__F381,
        &autotune::Autotune_attn_fwd__A2__F382,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F384,
        &autotune::Autotune_attn_fwd__A2__F385,
        &autotune::Autotune_attn_fwd__A2__F386,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F388,
        &autotune::Autotune_attn_fwd__A2__F389,
        &autotune::Autotune_attn_fwd__A2__F390,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F392,
        &autotune::Autotune_attn_fwd__A2__F393,
        &autotune::Autotune_attn_fwd__A2__F394,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F396,
        &autotune::Autotune_attn_fwd__A2__F397,
        &autotune::Autotune_attn_fwd__A2__F398,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F400,
        &autotune::Autotune_attn_fwd__A2__F401,
        &autotune::Autotune_attn_fwd__A2__F402,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F404,
        &autotune::Autotune_attn_fwd__A2__F405,
        &autotune::Autotune_attn_fwd__A2__F406,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F408,
        &autotune::Autotune_attn_fwd__A2__F409,
        &autotune::Autotune_attn_fwd__A2__F410,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F412,
        &autotune::Autotune_attn_fwd__A2__F413,
        &autotune::Autotune_attn_fwd__A2__F414,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F416,
        &autotune::Autotune_attn_fwd__A2__F417,
        &autotune::Autotune_attn_fwd__A2__F418,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F420,
        &autotune::Autotune_attn_fwd__A2__F421,
        &autotune::Autotune_attn_fwd__A2__F422,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F424,
        &autotune::Autotune_attn_fwd__A2__F425,
        &autotune::Autotune_attn_fwd__A2__F426,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F428,
        &autotune::Autotune_attn_fwd__A2__F429,
        &autotune::Autotune_attn_fwd__A2__F430,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F432,
        &autotune::Autotune_attn_fwd__A2__F433,
        &autotune::Autotune_attn_fwd__A2__F434,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F436,
        &autotune::Autotune_attn_fwd__A2__F437,
        &autotune::Autotune_attn_fwd__A2__F438,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F440,
        &autotune::Autotune_attn_fwd__A2__F441,
        &autotune::Autotune_attn_fwd__A2__F442,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F444,
        &autotune::Autotune_attn_fwd__A2__F445,
        &autotune::Autotune_attn_fwd__A2__F446,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F448,
        &autotune::Autotune_attn_fwd__A2__F449,
        &autotune::Autotune_attn_fwd__A2__F450,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F452,
        &autotune::Autotune_attn_fwd__A2__F453,
        &autotune::Autotune_attn_fwd__A2__F454,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F456,
        &autotune::Autotune_attn_fwd__A2__F457,
        &autotune::Autotune_attn_fwd__A2__F458,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F460,
        &autotune::Autotune_attn_fwd__A2__F461,
        &autotune::Autotune_attn_fwd__A2__F462,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F464,
        &autotune::Autotune_attn_fwd__A2__F465,
        &autotune::Autotune_attn_fwd__A2__F466,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F468,
        &autotune::Autotune_attn_fwd__A2__F469,
        &autotune::Autotune_attn_fwd__A2__F470,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F472,
        &autotune::Autotune_attn_fwd__A2__F473,
        &autotune::Autotune_attn_fwd__A2__F474,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F476,
        &autotune::Autotune_attn_fwd__A2__F477,
        &autotune::Autotune_attn_fwd__A2__F478,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F480,
        &autotune::Autotune_attn_fwd__A2__F481,
        &autotune::Autotune_attn_fwd__A2__F482,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F484,
        &autotune::Autotune_attn_fwd__A2__F485,
        &autotune::Autotune_attn_fwd__A2__F486,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F488,
        &autotune::Autotune_attn_fwd__A2__F489,
        &autotune::Autotune_attn_fwd__A2__F490,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F492,
        &autotune::Autotune_attn_fwd__A2__F493,
        &autotune::Autotune_attn_fwd__A2__F494,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F496,
        &autotune::Autotune_attn_fwd__A2__F497,
        &autotune::Autotune_attn_fwd__A2__F498,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F500,
        &autotune::Autotune_attn_fwd__A2__F501,
        &autotune::Autotune_attn_fwd__A2__F502,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F504,
        &autotune::Autotune_attn_fwd__A2__F505,
        &autotune::Autotune_attn_fwd__A2__F506,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F508,
        &autotune::Autotune_attn_fwd__A2__F509,
        &autotune::Autotune_attn_fwd__A2__F510,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F512,
        &autotune::Autotune_attn_fwd__A2__F513,
        &autotune::Autotune_attn_fwd__A2__F514,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F516,
        &autotune::Autotune_attn_fwd__A2__F517,
        &autotune::Autotune_attn_fwd__A2__F518,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F520,
        &autotune::Autotune_attn_fwd__A2__F521,
        &autotune::Autotune_attn_fwd__A2__F522,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F524,
        &autotune::Autotune_attn_fwd__A2__F525,
        &autotune::Autotune_attn_fwd__A2__F526,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F528,
        &autotune::Autotune_attn_fwd__A2__F529,
        &autotune::Autotune_attn_fwd__A2__F530,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F532,
        &autotune::Autotune_attn_fwd__A2__F533,
        &autotune::Autotune_attn_fwd__A2__F534,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F536,
        &autotune::Autotune_attn_fwd__A2__F537,
        &autotune::Autotune_attn_fwd__A2__F538,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F540,
        &autotune::Autotune_attn_fwd__A2__F541,
        &autotune::Autotune_attn_fwd__A2__F542,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F544,
        &autotune::Autotune_attn_fwd__A2__F545,
        &autotune::Autotune_attn_fwd__A2__F546,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F548,
        &autotune::Autotune_attn_fwd__A2__F549,
        &autotune::Autotune_attn_fwd__A2__F550,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F552,
        &autotune::Autotune_attn_fwd__A2__F553,
        &autotune::Autotune_attn_fwd__A2__F554,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F556,
        &autotune::Autotune_attn_fwd__A2__F557,
        &autotune::Autotune_attn_fwd__A2__F558,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F560,
        &autotune::Autotune_attn_fwd__A2__F561,
        &autotune::Autotune_attn_fwd__A2__F562,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F564,
        &autotune::Autotune_attn_fwd__A2__F565,
        &autotune::Autotune_attn_fwd__A2__F566,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F568,
        &autotune::Autotune_attn_fwd__A2__F569,
        &autotune::Autotune_attn_fwd__A2__F570,
        nullptr,
        &autotune::Autotune_attn_fwd__A2__F572,
        &autotune::Autotune_attn_fwd__A2__F573,
        &autotune::Autotune_attn_fwd__A2__F574,
        nullptr,
    },
    {
        &autotune::Autotune_attn_fwd__A3__F0,
        &autotune::Autotune_attn_fwd__A3__F1,
        &autotune::Autotune_attn_fwd__A3__F2,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F4,
        &autotune::Autotune_attn_fwd__A3__F5,
        &autotune::Autotune_attn_fwd__A3__F6,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F8,
        &autotune::Autotune_attn_fwd__A3__F9,
        &autotune::Autotune_attn_fwd__A3__F10,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F12,
        &autotune::Autotune_attn_fwd__A3__F13,
        &autotune::Autotune_attn_fwd__A3__F14,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F16,
        &autotune::Autotune_attn_fwd__A3__F17,
        &autotune::Autotune_attn_fwd__A3__F18,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F20,
        &autotune::Autotune_attn_fwd__A3__F21,
        &autotune::Autotune_attn_fwd__A3__F22,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F24,
        &autotune::Autotune_attn_fwd__A3__F25,
        &autotune::Autotune_attn_fwd__A3__F26,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F28,
        &autotune::Autotune_attn_fwd__A3__F29,
        &autotune::Autotune_attn_fwd__A3__F30,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F32,
        &autotune::Autotune_attn_fwd__A3__F33,
        &autotune::Autotune_attn_fwd__A3__F34,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F36,
        &autotune::Autotune_attn_fwd__A3__F37,
        &autotune::Autotune_attn_fwd__A3__F38,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F40,
        &autotune::Autotune_attn_fwd__A3__F41,
        &autotune::Autotune_attn_fwd__A3__F42,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F44,
        &autotune::Autotune_attn_fwd__A3__F45,
        &autotune::Autotune_attn_fwd__A3__F46,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F48,
        &autotune::Autotune_attn_fwd__A3__F49,
        &autotune::Autotune_attn_fwd__A3__F50,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F52,
        &autotune::Autotune_attn_fwd__A3__F53,
        &autotune::Autotune_attn_fwd__A3__F54,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F56,
        &autotune::Autotune_attn_fwd__A3__F57,
        &autotune::Autotune_attn_fwd__A3__F58,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F60,
        &autotune::Autotune_attn_fwd__A3__F61,
        &autotune::Autotune_attn_fwd__A3__F62,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F64,
        &autotune::Autotune_attn_fwd__A3__F65,
        &autotune::Autotune_attn_fwd__A3__F66,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F68,
        &autotune::Autotune_attn_fwd__A3__F69,
        &autotune::Autotune_attn_fwd__A3__F70,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F72,
        &autotune::Autotune_attn_fwd__A3__F73,
        &autotune::Autotune_attn_fwd__A3__F74,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F76,
        &autotune::Autotune_attn_fwd__A3__F77,
        &autotune::Autotune_attn_fwd__A3__F78,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F80,
        &autotune::Autotune_attn_fwd__A3__F81,
        &autotune::Autotune_attn_fwd__A3__F82,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F84,
        &autotune::Autotune_attn_fwd__A3__F85,
        &autotune::Autotune_attn_fwd__A3__F86,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F88,
        &autotune::Autotune_attn_fwd__A3__F89,
        &autotune::Autotune_attn_fwd__A3__F90,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F92,
        &autotune::Autotune_attn_fwd__A3__F93,
        &autotune::Autotune_attn_fwd__A3__F94,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F96,
        &autotune::Autotune_attn_fwd__A3__F97,
        &autotune::Autotune_attn_fwd__A3__F98,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F100,
        &autotune::Autotune_attn_fwd__A3__F101,
        &autotune::Autotune_attn_fwd__A3__F102,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F104,
        &autotune::Autotune_attn_fwd__A3__F105,
        &autotune::Autotune_attn_fwd__A3__F106,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F108,
        &autotune::Autotune_attn_fwd__A3__F109,
        &autotune::Autotune_attn_fwd__A3__F110,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F112,
        &autotune::Autotune_attn_fwd__A3__F113,
        &autotune::Autotune_attn_fwd__A3__F114,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F116,
        &autotune::Autotune_attn_fwd__A3__F117,
        &autotune::Autotune_attn_fwd__A3__F118,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F120,
        &autotune::Autotune_attn_fwd__A3__F121,
        &autotune::Autotune_attn_fwd__A3__F122,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F124,
        &autotune::Autotune_attn_fwd__A3__F125,
        &autotune::Autotune_attn_fwd__A3__F126,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F128,
        &autotune::Autotune_attn_fwd__A3__F129,
        &autotune::Autotune_attn_fwd__A3__F130,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F132,
        &autotune::Autotune_attn_fwd__A3__F133,
        &autotune::Autotune_attn_fwd__A3__F134,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F136,
        &autotune::Autotune_attn_fwd__A3__F137,
        &autotune::Autotune_attn_fwd__A3__F138,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F140,
        &autotune::Autotune_attn_fwd__A3__F141,
        &autotune::Autotune_attn_fwd__A3__F142,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F144,
        &autotune::Autotune_attn_fwd__A3__F145,
        &autotune::Autotune_attn_fwd__A3__F146,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F148,
        &autotune::Autotune_attn_fwd__A3__F149,
        &autotune::Autotune_attn_fwd__A3__F150,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F152,
        &autotune::Autotune_attn_fwd__A3__F153,
        &autotune::Autotune_attn_fwd__A3__F154,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F156,
        &autotune::Autotune_attn_fwd__A3__F157,
        &autotune::Autotune_attn_fwd__A3__F158,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F160,
        &autotune::Autotune_attn_fwd__A3__F161,
        &autotune::Autotune_attn_fwd__A3__F162,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F164,
        &autotune::Autotune_attn_fwd__A3__F165,
        &autotune::Autotune_attn_fwd__A3__F166,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F168,
        &autotune::Autotune_attn_fwd__A3__F169,
        &autotune::Autotune_attn_fwd__A3__F170,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F172,
        &autotune::Autotune_attn_fwd__A3__F173,
        &autotune::Autotune_attn_fwd__A3__F174,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F176,
        &autotune::Autotune_attn_fwd__A3__F177,
        &autotune::Autotune_attn_fwd__A3__F178,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F180,
        &autotune::Autotune_attn_fwd__A3__F181,
        &autotune::Autotune_attn_fwd__A3__F182,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F184,
        &autotune::Autotune_attn_fwd__A3__F185,
        &autotune::Autotune_attn_fwd__A3__F186,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F188,
        &autotune::Autotune_attn_fwd__A3__F189,
        &autotune::Autotune_attn_fwd__A3__F190,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F192,
        &autotune::Autotune_attn_fwd__A3__F193,
        &autotune::Autotune_attn_fwd__A3__F194,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F196,
        &autotune::Autotune_attn_fwd__A3__F197,
        &autotune::Autotune_attn_fwd__A3__F198,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F200,
        &autotune::Autotune_attn_fwd__A3__F201,
        &autotune::Autotune_attn_fwd__A3__F202,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F204,
        &autotune::Autotune_attn_fwd__A3__F205,
        &autotune::Autotune_attn_fwd__A3__F206,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F208,
        &autotune::Autotune_attn_fwd__A3__F209,
        &autotune::Autotune_attn_fwd__A3__F210,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F212,
        &autotune::Autotune_attn_fwd__A3__F213,
        &autotune::Autotune_attn_fwd__A3__F214,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F216,
        &autotune::Autotune_attn_fwd__A3__F217,
        &autotune::Autotune_attn_fwd__A3__F218,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F220,
        &autotune::Autotune_attn_fwd__A3__F221,
        &autotune::Autotune_attn_fwd__A3__F222,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F224,
        &autotune::Autotune_attn_fwd__A3__F225,
        &autotune::Autotune_attn_fwd__A3__F226,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F228,
        &autotune::Autotune_attn_fwd__A3__F229,
        &autotune::Autotune_attn_fwd__A3__F230,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F232,
        &autotune::Autotune_attn_fwd__A3__F233,
        &autotune::Autotune_attn_fwd__A3__F234,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F236,
        &autotune::Autotune_attn_fwd__A3__F237,
        &autotune::Autotune_attn_fwd__A3__F238,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F240,
        &autotune::Autotune_attn_fwd__A3__F241,
        &autotune::Autotune_attn_fwd__A3__F242,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F244,
        &autotune::Autotune_attn_fwd__A3__F245,
        &autotune::Autotune_attn_fwd__A3__F246,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F248,
        &autotune::Autotune_attn_fwd__A3__F249,
        &autotune::Autotune_attn_fwd__A3__F250,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F252,
        &autotune::Autotune_attn_fwd__A3__F253,
        &autotune::Autotune_attn_fwd__A3__F254,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F256,
        &autotune::Autotune_attn_fwd__A3__F257,
        &autotune::Autotune_attn_fwd__A3__F258,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F260,
        &autotune::Autotune_attn_fwd__A3__F261,
        &autotune::Autotune_attn_fwd__A3__F262,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F264,
        &autotune::Autotune_attn_fwd__A3__F265,
        &autotune::Autotune_attn_fwd__A3__F266,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F268,
        &autotune::Autotune_attn_fwd__A3__F269,
        &autotune::Autotune_attn_fwd__A3__F270,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F272,
        &autotune::Autotune_attn_fwd__A3__F273,
        &autotune::Autotune_attn_fwd__A3__F274,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F276,
        &autotune::Autotune_attn_fwd__A3__F277,
        &autotune::Autotune_attn_fwd__A3__F278,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F280,
        &autotune::Autotune_attn_fwd__A3__F281,
        &autotune::Autotune_attn_fwd__A3__F282,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F284,
        &autotune::Autotune_attn_fwd__A3__F285,
        &autotune::Autotune_attn_fwd__A3__F286,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F288,
        &autotune::Autotune_attn_fwd__A3__F289,
        &autotune::Autotune_attn_fwd__A3__F290,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F292,
        &autotune::Autotune_attn_fwd__A3__F293,
        &autotune::Autotune_attn_fwd__A3__F294,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F296,
        &autotune::Autotune_attn_fwd__A3__F297,
        &autotune::Autotune_attn_fwd__A3__F298,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F300,
        &autotune::Autotune_attn_fwd__A3__F301,
        &autotune::Autotune_attn_fwd__A3__F302,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F304,
        &autotune::Autotune_attn_fwd__A3__F305,
        &autotune::Autotune_attn_fwd__A3__F306,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F308,
        &autotune::Autotune_attn_fwd__A3__F309,
        &autotune::Autotune_attn_fwd__A3__F310,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F312,
        &autotune::Autotune_attn_fwd__A3__F313,
        &autotune::Autotune_attn_fwd__A3__F314,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F316,
        &autotune::Autotune_attn_fwd__A3__F317,
        &autotune::Autotune_attn_fwd__A3__F318,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F320,
        &autotune::Autotune_attn_fwd__A3__F321,
        &autotune::Autotune_attn_fwd__A3__F322,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F324,
        &autotune::Autotune_attn_fwd__A3__F325,
        &autotune::Autotune_attn_fwd__A3__F326,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F328,
        &autotune::Autotune_attn_fwd__A3__F329,
        &autotune::Autotune_attn_fwd__A3__F330,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F332,
        &autotune::Autotune_attn_fwd__A3__F333,
        &autotune::Autotune_attn_fwd__A3__F334,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F336,
        &autotune::Autotune_attn_fwd__A3__F337,
        &autotune::Autotune_attn_fwd__A3__F338,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F340,
        &autotune::Autotune_attn_fwd__A3__F341,
        &autotune::Autotune_attn_fwd__A3__F342,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F344,
        &autotune::Autotune_attn_fwd__A3__F345,
        &autotune::Autotune_attn_fwd__A3__F346,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F348,
        &autotune::Autotune_attn_fwd__A3__F349,
        &autotune::Autotune_attn_fwd__A3__F350,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F352,
        &autotune::Autotune_attn_fwd__A3__F353,
        &autotune::Autotune_attn_fwd__A3__F354,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F356,
        &autotune::Autotune_attn_fwd__A3__F357,
        &autotune::Autotune_attn_fwd__A3__F358,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F360,
        &autotune::Autotune_attn_fwd__A3__F361,
        &autotune::Autotune_attn_fwd__A3__F362,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F364,
        &autotune::Autotune_attn_fwd__A3__F365,
        &autotune::Autotune_attn_fwd__A3__F366,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F368,
        &autotune::Autotune_attn_fwd__A3__F369,
        &autotune::Autotune_attn_fwd__A3__F370,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F372,
        &autotune::Autotune_attn_fwd__A3__F373,
        &autotune::Autotune_attn_fwd__A3__F374,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F376,
        &autotune::Autotune_attn_fwd__A3__F377,
        &autotune::Autotune_attn_fwd__A3__F378,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F380,
        &autotune::Autotune_attn_fwd__A3__F381,
        &autotune::Autotune_attn_fwd__A3__F382,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F384,
        &autotune::Autotune_attn_fwd__A3__F385,
        &autotune::Autotune_attn_fwd__A3__F386,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F388,
        &autotune::Autotune_attn_fwd__A3__F389,
        &autotune::Autotune_attn_fwd__A3__F390,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F392,
        &autotune::Autotune_attn_fwd__A3__F393,
        &autotune::Autotune_attn_fwd__A3__F394,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F396,
        &autotune::Autotune_attn_fwd__A3__F397,
        &autotune::Autotune_attn_fwd__A3__F398,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F400,
        &autotune::Autotune_attn_fwd__A3__F401,
        &autotune::Autotune_attn_fwd__A3__F402,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F404,
        &autotune::Autotune_attn_fwd__A3__F405,
        &autotune::Autotune_attn_fwd__A3__F406,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F408,
        &autotune::Autotune_attn_fwd__A3__F409,
        &autotune::Autotune_attn_fwd__A3__F410,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F412,
        &autotune::Autotune_attn_fwd__A3__F413,
        &autotune::Autotune_attn_fwd__A3__F414,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F416,
        &autotune::Autotune_attn_fwd__A3__F417,
        &autotune::Autotune_attn_fwd__A3__F418,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F420,
        &autotune::Autotune_attn_fwd__A3__F421,
        &autotune::Autotune_attn_fwd__A3__F422,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F424,
        &autotune::Autotune_attn_fwd__A3__F425,
        &autotune::Autotune_attn_fwd__A3__F426,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F428,
        &autotune::Autotune_attn_fwd__A3__F429,
        &autotune::Autotune_attn_fwd__A3__F430,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F432,
        &autotune::Autotune_attn_fwd__A3__F433,
        &autotune::Autotune_attn_fwd__A3__F434,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F436,
        &autotune::Autotune_attn_fwd__A3__F437,
        &autotune::Autotune_attn_fwd__A3__F438,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F440,
        &autotune::Autotune_attn_fwd__A3__F441,
        &autotune::Autotune_attn_fwd__A3__F442,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F444,
        &autotune::Autotune_attn_fwd__A3__F445,
        &autotune::Autotune_attn_fwd__A3__F446,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F448,
        &autotune::Autotune_attn_fwd__A3__F449,
        &autotune::Autotune_attn_fwd__A3__F450,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F452,
        &autotune::Autotune_attn_fwd__A3__F453,
        &autotune::Autotune_attn_fwd__A3__F454,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F456,
        &autotune::Autotune_attn_fwd__A3__F457,
        &autotune::Autotune_attn_fwd__A3__F458,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F460,
        &autotune::Autotune_attn_fwd__A3__F461,
        &autotune::Autotune_attn_fwd__A3__F462,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F464,
        &autotune::Autotune_attn_fwd__A3__F465,
        &autotune::Autotune_attn_fwd__A3__F466,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F468,
        &autotune::Autotune_attn_fwd__A3__F469,
        &autotune::Autotune_attn_fwd__A3__F470,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F472,
        &autotune::Autotune_attn_fwd__A3__F473,
        &autotune::Autotune_attn_fwd__A3__F474,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F476,
        &autotune::Autotune_attn_fwd__A3__F477,
        &autotune::Autotune_attn_fwd__A3__F478,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F480,
        &autotune::Autotune_attn_fwd__A3__F481,
        &autotune::Autotune_attn_fwd__A3__F482,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F484,
        &autotune::Autotune_attn_fwd__A3__F485,
        &autotune::Autotune_attn_fwd__A3__F486,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F488,
        &autotune::Autotune_attn_fwd__A3__F489,
        &autotune::Autotune_attn_fwd__A3__F490,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F492,
        &autotune::Autotune_attn_fwd__A3__F493,
        &autotune::Autotune_attn_fwd__A3__F494,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F496,
        &autotune::Autotune_attn_fwd__A3__F497,
        &autotune::Autotune_attn_fwd__A3__F498,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F500,
        &autotune::Autotune_attn_fwd__A3__F501,
        &autotune::Autotune_attn_fwd__A3__F502,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F504,
        &autotune::Autotune_attn_fwd__A3__F505,
        &autotune::Autotune_attn_fwd__A3__F506,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F508,
        &autotune::Autotune_attn_fwd__A3__F509,
        &autotune::Autotune_attn_fwd__A3__F510,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F512,
        &autotune::Autotune_attn_fwd__A3__F513,
        &autotune::Autotune_attn_fwd__A3__F514,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F516,
        &autotune::Autotune_attn_fwd__A3__F517,
        &autotune::Autotune_attn_fwd__A3__F518,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F520,
        &autotune::Autotune_attn_fwd__A3__F521,
        &autotune::Autotune_attn_fwd__A3__F522,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F524,
        &autotune::Autotune_attn_fwd__A3__F525,
        &autotune::Autotune_attn_fwd__A3__F526,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F528,
        &autotune::Autotune_attn_fwd__A3__F529,
        &autotune::Autotune_attn_fwd__A3__F530,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F532,
        &autotune::Autotune_attn_fwd__A3__F533,
        &autotune::Autotune_attn_fwd__A3__F534,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F536,
        &autotune::Autotune_attn_fwd__A3__F537,
        &autotune::Autotune_attn_fwd__A3__F538,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F540,
        &autotune::Autotune_attn_fwd__A3__F541,
        &autotune::Autotune_attn_fwd__A3__F542,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F544,
        &autotune::Autotune_attn_fwd__A3__F545,
        &autotune::Autotune_attn_fwd__A3__F546,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F548,
        &autotune::Autotune_attn_fwd__A3__F549,
        &autotune::Autotune_attn_fwd__A3__F550,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F552,
        &autotune::Autotune_attn_fwd__A3__F553,
        &autotune::Autotune_attn_fwd__A3__F554,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F556,
        &autotune::Autotune_attn_fwd__A3__F557,
        &autotune::Autotune_attn_fwd__A3__F558,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F560,
        &autotune::Autotune_attn_fwd__A3__F561,
        &autotune::Autotune_attn_fwd__A3__F562,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F564,
        &autotune::Autotune_attn_fwd__A3__F565,
        &autotune::Autotune_attn_fwd__A3__F566,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F568,
        &autotune::Autotune_attn_fwd__A3__F569,
        &autotune::Autotune_attn_fwd__A3__F570,
        nullptr,
        &autotune::Autotune_attn_fwd__A3__F572,
        &autotune::Autotune_attn_fwd__A3__F573,
        &autotune::Autotune_attn_fwd__A3__F574,
        nullptr,
    },
    {
        &autotune::Autotune_attn_fwd__A4__F0,
        &autotune::Autotune_attn_fwd__A4__F1,
        &autotune::Autotune_attn_fwd__A4__F2,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F4,
        &autotune::Autotune_attn_fwd__A4__F5,
        &autotune::Autotune_attn_fwd__A4__F6,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F8,
        &autotune::Autotune_attn_fwd__A4__F9,
        &autotune::Autotune_attn_fwd__A4__F10,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F12,
        &autotune::Autotune_attn_fwd__A4__F13,
        &autotune::Autotune_attn_fwd__A4__F14,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F16,
        &autotune::Autotune_attn_fwd__A4__F17,
        &autotune::Autotune_attn_fwd__A4__F18,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F20,
        &autotune::Autotune_attn_fwd__A4__F21,
        &autotune::Autotune_attn_fwd__A4__F22,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F24,
        &autotune::Autotune_attn_fwd__A4__F25,
        &autotune::Autotune_attn_fwd__A4__F26,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F28,
        &autotune::Autotune_attn_fwd__A4__F29,
        &autotune::Autotune_attn_fwd__A4__F30,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F32,
        &autotune::Autotune_attn_fwd__A4__F33,
        &autotune::Autotune_attn_fwd__A4__F34,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F36,
        &autotune::Autotune_attn_fwd__A4__F37,
        &autotune::Autotune_attn_fwd__A4__F38,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F40,
        &autotune::Autotune_attn_fwd__A4__F41,
        &autotune::Autotune_attn_fwd__A4__F42,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F44,
        &autotune::Autotune_attn_fwd__A4__F45,
        &autotune::Autotune_attn_fwd__A4__F46,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F48,
        &autotune::Autotune_attn_fwd__A4__F49,
        &autotune::Autotune_attn_fwd__A4__F50,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F52,
        &autotune::Autotune_attn_fwd__A4__F53,
        &autotune::Autotune_attn_fwd__A4__F54,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F56,
        &autotune::Autotune_attn_fwd__A4__F57,
        &autotune::Autotune_attn_fwd__A4__F58,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F60,
        &autotune::Autotune_attn_fwd__A4__F61,
        &autotune::Autotune_attn_fwd__A4__F62,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F64,
        &autotune::Autotune_attn_fwd__A4__F65,
        &autotune::Autotune_attn_fwd__A4__F66,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F68,
        &autotune::Autotune_attn_fwd__A4__F69,
        &autotune::Autotune_attn_fwd__A4__F70,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F72,
        &autotune::Autotune_attn_fwd__A4__F73,
        &autotune::Autotune_attn_fwd__A4__F74,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F76,
        &autotune::Autotune_attn_fwd__A4__F77,
        &autotune::Autotune_attn_fwd__A4__F78,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F80,
        &autotune::Autotune_attn_fwd__A4__F81,
        &autotune::Autotune_attn_fwd__A4__F82,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F84,
        &autotune::Autotune_attn_fwd__A4__F85,
        &autotune::Autotune_attn_fwd__A4__F86,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F88,
        &autotune::Autotune_attn_fwd__A4__F89,
        &autotune::Autotune_attn_fwd__A4__F90,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F92,
        &autotune::Autotune_attn_fwd__A4__F93,
        &autotune::Autotune_attn_fwd__A4__F94,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F96,
        &autotune::Autotune_attn_fwd__A4__F97,
        &autotune::Autotune_attn_fwd__A4__F98,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F100,
        &autotune::Autotune_attn_fwd__A4__F101,
        &autotune::Autotune_attn_fwd__A4__F102,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F104,
        &autotune::Autotune_attn_fwd__A4__F105,
        &autotune::Autotune_attn_fwd__A4__F106,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F108,
        &autotune::Autotune_attn_fwd__A4__F109,
        &autotune::Autotune_attn_fwd__A4__F110,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F112,
        &autotune::Autotune_attn_fwd__A4__F113,
        &autotune::Autotune_attn_fwd__A4__F114,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F116,
        &autotune::Autotune_attn_fwd__A4__F117,
        &autotune::Autotune_attn_fwd__A4__F118,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F120,
        &autotune::Autotune_attn_fwd__A4__F121,
        &autotune::Autotune_attn_fwd__A4__F122,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F124,
        &autotune::Autotune_attn_fwd__A4__F125,
        &autotune::Autotune_attn_fwd__A4__F126,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F128,
        &autotune::Autotune_attn_fwd__A4__F129,
        &autotune::Autotune_attn_fwd__A4__F130,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F132,
        &autotune::Autotune_attn_fwd__A4__F133,
        &autotune::Autotune_attn_fwd__A4__F134,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F136,
        &autotune::Autotune_attn_fwd__A4__F137,
        &autotune::Autotune_attn_fwd__A4__F138,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F140,
        &autotune::Autotune_attn_fwd__A4__F141,
        &autotune::Autotune_attn_fwd__A4__F142,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F144,
        &autotune::Autotune_attn_fwd__A4__F145,
        &autotune::Autotune_attn_fwd__A4__F146,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F148,
        &autotune::Autotune_attn_fwd__A4__F149,
        &autotune::Autotune_attn_fwd__A4__F150,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F152,
        &autotune::Autotune_attn_fwd__A4__F153,
        &autotune::Autotune_attn_fwd__A4__F154,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F156,
        &autotune::Autotune_attn_fwd__A4__F157,
        &autotune::Autotune_attn_fwd__A4__F158,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F160,
        &autotune::Autotune_attn_fwd__A4__F161,
        &autotune::Autotune_attn_fwd__A4__F162,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F164,
        &autotune::Autotune_attn_fwd__A4__F165,
        &autotune::Autotune_attn_fwd__A4__F166,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F168,
        &autotune::Autotune_attn_fwd__A4__F169,
        &autotune::Autotune_attn_fwd__A4__F170,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F172,
        &autotune::Autotune_attn_fwd__A4__F173,
        &autotune::Autotune_attn_fwd__A4__F174,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F176,
        &autotune::Autotune_attn_fwd__A4__F177,
        &autotune::Autotune_attn_fwd__A4__F178,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F180,
        &autotune::Autotune_attn_fwd__A4__F181,
        &autotune::Autotune_attn_fwd__A4__F182,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F184,
        &autotune::Autotune_attn_fwd__A4__F185,
        &autotune::Autotune_attn_fwd__A4__F186,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F188,
        &autotune::Autotune_attn_fwd__A4__F189,
        &autotune::Autotune_attn_fwd__A4__F190,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F192,
        &autotune::Autotune_attn_fwd__A4__F193,
        &autotune::Autotune_attn_fwd__A4__F194,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F196,
        &autotune::Autotune_attn_fwd__A4__F197,
        &autotune::Autotune_attn_fwd__A4__F198,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F200,
        &autotune::Autotune_attn_fwd__A4__F201,
        &autotune::Autotune_attn_fwd__A4__F202,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F204,
        &autotune::Autotune_attn_fwd__A4__F205,
        &autotune::Autotune_attn_fwd__A4__F206,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F208,
        &autotune::Autotune_attn_fwd__A4__F209,
        &autotune::Autotune_attn_fwd__A4__F210,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F212,
        &autotune::Autotune_attn_fwd__A4__F213,
        &autotune::Autotune_attn_fwd__A4__F214,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F216,
        &autotune::Autotune_attn_fwd__A4__F217,
        &autotune::Autotune_attn_fwd__A4__F218,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F220,
        &autotune::Autotune_attn_fwd__A4__F221,
        &autotune::Autotune_attn_fwd__A4__F222,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F224,
        &autotune::Autotune_attn_fwd__A4__F225,
        &autotune::Autotune_attn_fwd__A4__F226,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F228,
        &autotune::Autotune_attn_fwd__A4__F229,
        &autotune::Autotune_attn_fwd__A4__F230,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F232,
        &autotune::Autotune_attn_fwd__A4__F233,
        &autotune::Autotune_attn_fwd__A4__F234,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F236,
        &autotune::Autotune_attn_fwd__A4__F237,
        &autotune::Autotune_attn_fwd__A4__F238,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F240,
        &autotune::Autotune_attn_fwd__A4__F241,
        &autotune::Autotune_attn_fwd__A4__F242,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F244,
        &autotune::Autotune_attn_fwd__A4__F245,
        &autotune::Autotune_attn_fwd__A4__F246,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F248,
        &autotune::Autotune_attn_fwd__A4__F249,
        &autotune::Autotune_attn_fwd__A4__F250,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F252,
        &autotune::Autotune_attn_fwd__A4__F253,
        &autotune::Autotune_attn_fwd__A4__F254,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F256,
        &autotune::Autotune_attn_fwd__A4__F257,
        &autotune::Autotune_attn_fwd__A4__F258,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F260,
        &autotune::Autotune_attn_fwd__A4__F261,
        &autotune::Autotune_attn_fwd__A4__F262,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F264,
        &autotune::Autotune_attn_fwd__A4__F265,
        &autotune::Autotune_attn_fwd__A4__F266,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F268,
        &autotune::Autotune_attn_fwd__A4__F269,
        &autotune::Autotune_attn_fwd__A4__F270,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F272,
        &autotune::Autotune_attn_fwd__A4__F273,
        &autotune::Autotune_attn_fwd__A4__F274,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F276,
        &autotune::Autotune_attn_fwd__A4__F277,
        &autotune::Autotune_attn_fwd__A4__F278,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F280,
        &autotune::Autotune_attn_fwd__A4__F281,
        &autotune::Autotune_attn_fwd__A4__F282,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F284,
        &autotune::Autotune_attn_fwd__A4__F285,
        &autotune::Autotune_attn_fwd__A4__F286,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F288,
        &autotune::Autotune_attn_fwd__A4__F289,
        &autotune::Autotune_attn_fwd__A4__F290,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F292,
        &autotune::Autotune_attn_fwd__A4__F293,
        &autotune::Autotune_attn_fwd__A4__F294,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F296,
        &autotune::Autotune_attn_fwd__A4__F297,
        &autotune::Autotune_attn_fwd__A4__F298,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F300,
        &autotune::Autotune_attn_fwd__A4__F301,
        &autotune::Autotune_attn_fwd__A4__F302,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F304,
        &autotune::Autotune_attn_fwd__A4__F305,
        &autotune::Autotune_attn_fwd__A4__F306,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F308,
        &autotune::Autotune_attn_fwd__A4__F309,
        &autotune::Autotune_attn_fwd__A4__F310,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F312,
        &autotune::Autotune_attn_fwd__A4__F313,
        &autotune::Autotune_attn_fwd__A4__F314,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F316,
        &autotune::Autotune_attn_fwd__A4__F317,
        &autotune::Autotune_attn_fwd__A4__F318,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F320,
        &autotune::Autotune_attn_fwd__A4__F321,
        &autotune::Autotune_attn_fwd__A4__F322,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F324,
        &autotune::Autotune_attn_fwd__A4__F325,
        &autotune::Autotune_attn_fwd__A4__F326,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F328,
        &autotune::Autotune_attn_fwd__A4__F329,
        &autotune::Autotune_attn_fwd__A4__F330,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F332,
        &autotune::Autotune_attn_fwd__A4__F333,
        &autotune::Autotune_attn_fwd__A4__F334,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F336,
        &autotune::Autotune_attn_fwd__A4__F337,
        &autotune::Autotune_attn_fwd__A4__F338,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F340,
        &autotune::Autotune_attn_fwd__A4__F341,
        &autotune::Autotune_attn_fwd__A4__F342,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F344,
        &autotune::Autotune_attn_fwd__A4__F345,
        &autotune::Autotune_attn_fwd__A4__F346,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F348,
        &autotune::Autotune_attn_fwd__A4__F349,
        &autotune::Autotune_attn_fwd__A4__F350,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F352,
        &autotune::Autotune_attn_fwd__A4__F353,
        &autotune::Autotune_attn_fwd__A4__F354,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F356,
        &autotune::Autotune_attn_fwd__A4__F357,
        &autotune::Autotune_attn_fwd__A4__F358,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F360,
        &autotune::Autotune_attn_fwd__A4__F361,
        &autotune::Autotune_attn_fwd__A4__F362,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F364,
        &autotune::Autotune_attn_fwd__A4__F365,
        &autotune::Autotune_attn_fwd__A4__F366,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F368,
        &autotune::Autotune_attn_fwd__A4__F369,
        &autotune::Autotune_attn_fwd__A4__F370,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F372,
        &autotune::Autotune_attn_fwd__A4__F373,
        &autotune::Autotune_attn_fwd__A4__F374,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F376,
        &autotune::Autotune_attn_fwd__A4__F377,
        &autotune::Autotune_attn_fwd__A4__F378,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F380,
        &autotune::Autotune_attn_fwd__A4__F381,
        &autotune::Autotune_attn_fwd__A4__F382,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F384,
        &autotune::Autotune_attn_fwd__A4__F385,
        &autotune::Autotune_attn_fwd__A4__F386,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F388,
        &autotune::Autotune_attn_fwd__A4__F389,
        &autotune::Autotune_attn_fwd__A4__F390,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F392,
        &autotune::Autotune_attn_fwd__A4__F393,
        &autotune::Autotune_attn_fwd__A4__F394,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F396,
        &autotune::Autotune_attn_fwd__A4__F397,
        &autotune::Autotune_attn_fwd__A4__F398,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F400,
        &autotune::Autotune_attn_fwd__A4__F401,
        &autotune::Autotune_attn_fwd__A4__F402,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F404,
        &autotune::Autotune_attn_fwd__A4__F405,
        &autotune::Autotune_attn_fwd__A4__F406,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F408,
        &autotune::Autotune_attn_fwd__A4__F409,
        &autotune::Autotune_attn_fwd__A4__F410,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F412,
        &autotune::Autotune_attn_fwd__A4__F413,
        &autotune::Autotune_attn_fwd__A4__F414,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F416,
        &autotune::Autotune_attn_fwd__A4__F417,
        &autotune::Autotune_attn_fwd__A4__F418,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F420,
        &autotune::Autotune_attn_fwd__A4__F421,
        &autotune::Autotune_attn_fwd__A4__F422,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F424,
        &autotune::Autotune_attn_fwd__A4__F425,
        &autotune::Autotune_attn_fwd__A4__F426,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F428,
        &autotune::Autotune_attn_fwd__A4__F429,
        &autotune::Autotune_attn_fwd__A4__F430,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F432,
        &autotune::Autotune_attn_fwd__A4__F433,
        &autotune::Autotune_attn_fwd__A4__F434,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F436,
        &autotune::Autotune_attn_fwd__A4__F437,
        &autotune::Autotune_attn_fwd__A4__F438,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F440,
        &autotune::Autotune_attn_fwd__A4__F441,
        &autotune::Autotune_attn_fwd__A4__F442,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F444,
        &autotune::Autotune_attn_fwd__A4__F445,
        &autotune::Autotune_attn_fwd__A4__F446,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F448,
        &autotune::Autotune_attn_fwd__A4__F449,
        &autotune::Autotune_attn_fwd__A4__F450,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F452,
        &autotune::Autotune_attn_fwd__A4__F453,
        &autotune::Autotune_attn_fwd__A4__F454,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F456,
        &autotune::Autotune_attn_fwd__A4__F457,
        &autotune::Autotune_attn_fwd__A4__F458,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F460,
        &autotune::Autotune_attn_fwd__A4__F461,
        &autotune::Autotune_attn_fwd__A4__F462,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F464,
        &autotune::Autotune_attn_fwd__A4__F465,
        &autotune::Autotune_attn_fwd__A4__F466,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F468,
        &autotune::Autotune_attn_fwd__A4__F469,
        &autotune::Autotune_attn_fwd__A4__F470,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F472,
        &autotune::Autotune_attn_fwd__A4__F473,
        &autotune::Autotune_attn_fwd__A4__F474,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F476,
        &autotune::Autotune_attn_fwd__A4__F477,
        &autotune::Autotune_attn_fwd__A4__F478,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F480,
        &autotune::Autotune_attn_fwd__A4__F481,
        &autotune::Autotune_attn_fwd__A4__F482,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F484,
        &autotune::Autotune_attn_fwd__A4__F485,
        &autotune::Autotune_attn_fwd__A4__F486,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F488,
        &autotune::Autotune_attn_fwd__A4__F489,
        &autotune::Autotune_attn_fwd__A4__F490,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F492,
        &autotune::Autotune_attn_fwd__A4__F493,
        &autotune::Autotune_attn_fwd__A4__F494,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F496,
        &autotune::Autotune_attn_fwd__A4__F497,
        &autotune::Autotune_attn_fwd__A4__F498,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F500,
        &autotune::Autotune_attn_fwd__A4__F501,
        &autotune::Autotune_attn_fwd__A4__F502,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F504,
        &autotune::Autotune_attn_fwd__A4__F505,
        &autotune::Autotune_attn_fwd__A4__F506,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F508,
        &autotune::Autotune_attn_fwd__A4__F509,
        &autotune::Autotune_attn_fwd__A4__F510,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F512,
        &autotune::Autotune_attn_fwd__A4__F513,
        &autotune::Autotune_attn_fwd__A4__F514,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F516,
        &autotune::Autotune_attn_fwd__A4__F517,
        &autotune::Autotune_attn_fwd__A4__F518,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F520,
        &autotune::Autotune_attn_fwd__A4__F521,
        &autotune::Autotune_attn_fwd__A4__F522,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F524,
        &autotune::Autotune_attn_fwd__A4__F525,
        &autotune::Autotune_attn_fwd__A4__F526,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F528,
        &autotune::Autotune_attn_fwd__A4__F529,
        &autotune::Autotune_attn_fwd__A4__F530,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F532,
        &autotune::Autotune_attn_fwd__A4__F533,
        &autotune::Autotune_attn_fwd__A4__F534,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F536,
        &autotune::Autotune_attn_fwd__A4__F537,
        &autotune::Autotune_attn_fwd__A4__F538,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F540,
        &autotune::Autotune_attn_fwd__A4__F541,
        &autotune::Autotune_attn_fwd__A4__F542,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F544,
        &autotune::Autotune_attn_fwd__A4__F545,
        &autotune::Autotune_attn_fwd__A4__F546,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F548,
        &autotune::Autotune_attn_fwd__A4__F549,
        &autotune::Autotune_attn_fwd__A4__F550,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F552,
        &autotune::Autotune_attn_fwd__A4__F553,
        &autotune::Autotune_attn_fwd__A4__F554,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F556,
        &autotune::Autotune_attn_fwd__A4__F557,
        &autotune::Autotune_attn_fwd__A4__F558,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F560,
        &autotune::Autotune_attn_fwd__A4__F561,
        &autotune::Autotune_attn_fwd__A4__F562,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F564,
        &autotune::Autotune_attn_fwd__A4__F565,
        &autotune::Autotune_attn_fwd__A4__F566,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F568,
        &autotune::Autotune_attn_fwd__A4__F569,
        &autotune::Autotune_attn_fwd__A4__F570,
        nullptr,
        &autotune::Autotune_attn_fwd__A4__F572,
        &autotune::Autotune_attn_fwd__A4__F573,
        &autotune::Autotune_attn_fwd__A4__F574,
        nullptr,
    },
};

}

// vim: set fileencoding=utf-8

