/*
    Copyright (C) 2013-2022 Nicola L.C. Talbot
    www.dickimaw-books.com

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/
package com.dickimawbooks.texparserlib.latex.stix;

import java.io.IOException;

import com.dickimawbooks.texparserlib.*;
import com.dickimawbooks.texparserlib.latex.*;
import com.dickimawbooks.texparserlib.primitives.Undefined;
import com.dickimawbooks.texparserlib.generic.*;

public class StixSty extends LaTeXSty
{
   public StixSty(KeyValList options, LaTeXParserListener listener, 
     boolean loadParentOptions)
    throws IOException
   {
      super(options, "stix", listener, loadParentOptions);
   }

   @Override
   public void addDefinitions()
   {
      LaTeXParserListener listener = getListener();

      for (int i = 0; i < GREEK.length; i++)
      {
         registerControlSequence(listener.createGreekSymbol(
           (String)GREEK[i][0], 
           ((Integer)GREEK[i][1]).intValue()));
      }

      for (int i = 0; i < ALPHABETICS.length; i++)
      {
         registerControlSequence(listener.createMathSymbol(
           (String)ALPHABETICS[i][0], 
           ((Integer)ALPHABETICS[i][1]).intValue()));
      }

      for (int i = 0; i < ORDINARY.length; i++)
      {
         registerControlSequence(listener.createMathSymbol(
           (String)ORDINARY[i][0], 
           ((Integer)ORDINARY[i][1]).intValue()));
      }

      for (int i = 0; i < BINARY.length; i++)
      {
         registerControlSequence(listener.createBinarySymbol(
           (String)BINARY[i][0], 
           ((Integer)BINARY[i][1]).intValue()));
      }

      for (int i = 0; i < PUNCTUATION.length; i++)
      {
         registerControlSequence(listener.createMathSymbol(
           (String)PUNCTUATION[i][0], 
           ((Integer)PUNCTUATION[i][1]).intValue()));
      }

      for (int i = 0; i < SMALL_INTEGRALS.length; i++)
      {
         registerControlSequence(listener.createMathSymbol(
           (String)SMALL_INTEGRALS[i][0], 
           ((Integer)SMALL_INTEGRALS[i][1]).intValue()));
      }

      for (int i = 0; i < INTEGRALS.length; i++)
      {
         registerControlSequence(listener.createBigOperator(
           (String)INTEGRALS[i][0], 
           ((Integer)INTEGRALS[i][1]).intValue()));
      }

      for (int i = 0; i < BIG_OPERATORS.length; i++)
      {
         registerControlSequence(listener.createBigOperator(
           (String)BIG_OPERATORS[i][0], 
           ((Integer)BIG_OPERATORS[i][1]).intValue()));
      }

      for (int i = 0; i < DELIMITERS.length; i++)
      {
         registerControlSequence(listener.createBigOperator(
           (String)DELIMITERS[i][0], 
           ((Integer)DELIMITERS[i][1]).intValue()));
      }

      for (int i = 0; i < OTHER_BRACES.length; i++)
      {
         registerControlSequence(listener.createMathSymbol(
           (String)OTHER_BRACES[i][0], 
           ((Integer)OTHER_BRACES[i][1]).intValue()));
      }

      for (int i = 0; i < ACCENTS.length; i++)
      {
         registerControlSequence(new MathAccent(
           (String)ACCENTS[i][0], ((Integer)ACCENTS[i][1]).intValue()));
      }

      for (int i = 0; i < WIDE_ACCENTS.length; i++)
      {
         registerControlSequence(new MathAccent(
           (String)WIDE_ACCENTS[i][0],
           ((Integer)WIDE_ACCENTS[i][1]).intValue()));
      }

      for (int i = 0; i < WIDE_BRACKETS.length; i++)
      {
         registerControlSequence(new MathAccent(
           (String)WIDE_BRACKETS[i][0],
           ((Integer)WIDE_BRACKETS[i][1]).intValue()));
      }

      for (int i = 0; i < RADICALS.length; i++)
      {
         registerControlSequence(new MathAccent(
           (String)RADICALS[i][0],
           ((Integer)RADICALS[i][1]).intValue()));
      }
   }

   private static final Object[][] GREEK = new Object[][]
   {
      // Greek lower case
      new Object[]{"alpha", Integer.valueOf(0x1D6FC)},
      new Object[]{"beta", Integer.valueOf(0x1D6FD)},
      new Object[]{"gamma", Integer.valueOf(0x1D6FE)},
      new Object[]{"delta", Integer.valueOf(0x1D6FF)},
      new Object[]{"varepsilon", Integer.valueOf(0x1D700)},
      new Object[]{"zeta", Integer.valueOf(0x1D701)},
      new Object[]{"eta", Integer.valueOf(0x1D702)},
      new Object[]{"theta", Integer.valueOf(0x1D703)},
      new Object[]{"iota", Integer.valueOf(0x1D704)},
      new Object[]{"kappa", Integer.valueOf(0x1D705)},
      new Object[]{"lambda", Integer.valueOf(0x1D706)},
      new Object[]{"mu", Integer.valueOf(0x1D707)},
      new Object[]{"nu", Integer.valueOf(0x1D708)},
      new Object[]{"xi", Integer.valueOf(0x1D709)},
      new Object[]{"omicron", Integer.valueOf(0x1D70A)},
      new Object[]{"pi", Integer.valueOf(0x1D70B)},
      new Object[]{"rho", Integer.valueOf(0x1D70C)},
      new Object[]{"varsigma", Integer.valueOf(0x1D70D)},
      new Object[]{"sigma", Integer.valueOf(0x1D70E)},
      new Object[]{"tau", Integer.valueOf(0x1D70F)},
      new Object[]{"upsilon", Integer.valueOf(0x1D710)},
      new Object[]{"varphi", Integer.valueOf(0x1D711)},
      new Object[]{"chi", Integer.valueOf(0x1D712)},
      new Object[]{"psi", Integer.valueOf(0x1D713)},
      new Object[]{"omega", Integer.valueOf(0x1D714)},
      new Object[]{"epsilon", Integer.valueOf(0x1D716)},
      new Object[]{"vartheta", Integer.valueOf(0x1D717)},
      new Object[]{"varkappa", Integer.valueOf(0x1D718)},
      new Object[]{"phi", Integer.valueOf(0x1D719)},
      new Object[]{"varrho", Integer.valueOf(0x1D71A)},
      new Object[]{"varpi", Integer.valueOf(0x1D71B)},
      // Greek upper case
      new Object[]{"Alpha", Integer.valueOf(0x1D6E2)},
      new Object[]{"Beta", Integer.valueOf(0x1D6E3)},
      new Object[]{"Gamma", Integer.valueOf(0x1D6E4)},
      new Object[]{"Delta", Integer.valueOf(0x1D6E5)},
      new Object[]{"Epsilon", Integer.valueOf(0x1D6E6)},
      new Object[]{"Zeta", Integer.valueOf(0x1D6E7)},
      new Object[]{"Eta", Integer.valueOf(0x1D6E8)},
      new Object[]{"Theta", Integer.valueOf(0x1D6E9)},
      new Object[]{"Iota", Integer.valueOf(0x1D6EA)},
      new Object[]{"Kappa", Integer.valueOf(0x1D6EB)},
      new Object[]{"Lambda", Integer.valueOf(0x1D6EC)},
      new Object[]{"Mu", Integer.valueOf(0x1D6ED)},
      new Object[]{"Nu", Integer.valueOf(0x1D6EE)},
      new Object[]{"Xi", Integer.valueOf(0x1D6EF)},
      new Object[]{"Omicron", Integer.valueOf(0x1D6F0)},
      new Object[]{"Pi", Integer.valueOf(0x1D6F1)},
      new Object[]{"Rho", Integer.valueOf(0x1D6F2)},
      new Object[]{"Theta", Integer.valueOf(0x1D6F3)},
      new Object[]{"Sigma", Integer.valueOf(0x1D6F4)},
      new Object[]{"Tau", Integer.valueOf(0x1D6F5)},
      new Object[]{"Upsilon", Integer.valueOf(0x1D6F6)},
      new Object[]{"Phi", Integer.valueOf(0x1D6F7)},
      new Object[]{"Chi", Integer.valueOf(0x1D6F8)},
      new Object[]{"Psi", Integer.valueOf(0x1D6F9)},
      new Object[]{"Omega", Integer.valueOf(0x1D6FA)},
      new Object[]{"nabla", Integer.valueOf(0x1D6FB)},
   };

   private static final Object[][] ALPHABETICS = new Object[][]
   {
      // other alphabetics
      new Object[]{"nabla", Integer.valueOf(0x2207)},
      new Object[]{"partial", Integer.valueOf(0x2202)},
      new Object[]{"imath", Integer.valueOf(0x1D6A4)},
      new Object[]{"jmath", Integer.valueOf(0x1D6A5)},
   };

   private static final Object[][] ORDINARY = new Object[][]
   {
      // ordinary symbols
      new Object[]{"mathdollar", Integer.valueOf(0x0024)},
      new Object[]{"mathsterling", Integer.valueOf(0x00A3)},
      new Object[]{"mathsection", Integer.valueOf(0x00A7)},
      new Object[]{"neg", Integer.valueOf(0x00AC)},
      new Object[]{"lnot", Integer.valueOf(0x00AC)},
      new Object[]{"mathparagraph", Integer.valueOf(0x00B6)},
      new Object[]{"eth", Integer.valueOf(0x00F0)},
      new Object[]{"Zbar", Integer.valueOf(0x01B5)},
      new Object[]{"digamma", Integer.valueOf(0x03DD)},
      new Object[]{"backepsilon", Integer.valueOf(0x03F6)},
      new Object[]{"upbackepsilon", Integer.valueOf(0x03F6)},
      new Object[]{"enleadertwodots", Integer.valueOf(0x2025)},
      new Object[]{"mathellipsis", Integer.valueOf(0x2026)},
      new Object[]{"prime", Integer.valueOf(0x2032)},
      new Object[]{"dprime", Integer.valueOf(0x2033)},
      new Object[]{"trprime", Integer.valueOf(0x2034)},
      new Object[]{"backprime", Integer.valueOf(0x2035)},
      new Object[]{"backdprime", Integer.valueOf(0x2036)},
      new Object[]{"backtrprime", Integer.valueOf(0x2037)},
      new Object[]{"caretinsert", Integer.valueOf(0x2038)},
      new Object[]{"Exclam", Integer.valueOf(0x203C)},
      new Object[]{"hyphenbullet", Integer.valueOf(0x2043)},
      new Object[]{"Question", Integer.valueOf(0x2047)},
      new Object[]{"qprime", Integer.valueOf(0x2057)},
      new Object[]{"enclosecircle", Integer.valueOf(0x20DD)},
      new Object[]{"enclosesquare", Integer.valueOf(0x20DE)},
      new Object[]{"enclosediamond", Integer.valueOf(0x20DF)},
      new Object[]{"enclosetriangle", Integer.valueOf(0x20E4)},
      new Object[]{"Eulerconst", Integer.valueOf(0x2107)},
      new Object[]{"hbar", Integer.valueOf(0x210F)},
      new Object[]{"hslash", Integer.valueOf(0x210F)},
      new Object[]{"Im", Integer.valueOf(0x2111)},
      new Object[]{"ell", Integer.valueOf(0x2113)},
      new Object[]{"wp", Integer.valueOf(0x2118)},
      new Object[]{"Re", Integer.valueOf(0x211C)},
      new Object[]{"mho", Integer.valueOf(0x2127)},
      new Object[]{"turnediota", Integer.valueOf(0x2129)},
      new Object[]{"Angstrom", Integer.valueOf(0x212B)},
      new Object[]{"Finv", Integer.valueOf(0x2132)},
      new Object[]{"aleph", Integer.valueOf(0x2135)},
      new Object[]{"beth", Integer.valueOf(0x2136)},
      new Object[]{"gimel", Integer.valueOf(0x2137)},
      new Object[]{"daleth", Integer.valueOf(0x2138)},
      new Object[]{"Game", Integer.valueOf(0x2141)},
      new Object[]{"sansLturned", Integer.valueOf(0x2142)},
      new Object[]{"sansLmirroed", Integer.valueOf(0x2143)},
      new Object[]{"Yup", Integer.valueOf(0x2144)},
      new Object[]{"PropertyLine", Integer.valueOf(0x214A)},
      new Object[]{"updownarrowbar", Integer.valueOf(0x21A8)},
      new Object[]{"linefeed", Integer.valueOf(0x21B4)},
      new Object[]{"carriagereturn", Integer.valueOf(0x21B5)},
      new Object[]{"barovernorthwestarrow", Integer.valueOf(0x21B8)},
      new Object[]{"barleftarrowrightarrowbar", Integer.valueOf(0x21B9)},
      new Object[]{"acwopencirclearrow", Integer.valueOf(0x21BA)},
      new Object[]{"cwopencirclearrow", Integer.valueOf(0x21BB)},
      new Object[]{"nHuparrow", Integer.valueOf(0x21DE)},
      new Object[]{"nHdownarrow", Integer.valueOf(0x21DF)},
      new Object[]{"leftdasharrow", Integer.valueOf(0x21E0)},
      new Object[]{"updasharrow", Integer.valueOf(0x21E1)},
      new Object[]{"rightdasharrow", Integer.valueOf(0x21E2)},
      new Object[]{"downdasharrow", Integer.valueOf(0x21E3)},
      new Object[]{"leftwhitearrow", Integer.valueOf(0x21E6)},
      new Object[]{"upwhitearrow", Integer.valueOf(0x21E7)},
      new Object[]{"rightwhitearrow", Integer.valueOf(0x21E8)},
      new Object[]{"downwhitearrow", Integer.valueOf(0x21E9)},
      new Object[]{"whitearrowupfrombar", Integer.valueOf(0x21EA)},
      new Object[]{"forall", Integer.valueOf(0x2200)},
      new Object[]{"complement", Integer.valueOf(0x2201)},
      new Object[]{"exists", Integer.valueOf(0x2203)},
      new Object[]{"nexists", Integer.valueOf(0x2204)},
      new Object[]{"varnothing", Integer.valueOf(0x2205)},
      new Object[]{"emptyset", Integer.valueOf(0x2205)},
      new Object[]{"increment", Integer.valueOf(0x2206)},
      new Object[]{"QED", Integer.valueOf(0x220E)},
      new Object[]{"infty", Integer.valueOf(0x221E)},
      new Object[]{"rightangle", Integer.valueOf(0x221F)},
      new Object[]{"angle", Integer.valueOf(0x2220)},
      new Object[]{"measuredangle", Integer.valueOf(0x2221)},
      new Object[]{"sphericalangle", Integer.valueOf(0x2222)},
      new Object[]{"therefore", Integer.valueOf(0x2234)},
      new Object[]{"because", Integer.valueOf(0x2235)},
      new Object[]{"sinewave", Integer.valueOf(0x223F)},
      new Object[]{"top", Integer.valueOf(0x22A4)},
      new Object[]{"bot", Integer.valueOf(0x22A5)},
      new Object[]{"hermitmatrix", Integer.valueOf(0x22B9)},
      new Object[]{"measuredrightangle", Integer.valueOf(0x22BE)},
      new Object[]{"varlrtriangle", Integer.valueOf(0x22BF)},
      new Object[]{"cdots", Integer.valueOf(0x22EF)},
      new Object[]{"diameter", Integer.valueOf(0x2300)},
      new Object[]{"house", Integer.valueOf(0x2302)},
      new Object[]{"invnot", Integer.valueOf(0x2310)},
      new Object[]{"sqlozenge", Integer.valueOf(0x2311)},
      new Object[]{"profline", Integer.valueOf(0x2312)},
      new Object[]{"profsurf", Integer.valueOf(0x2313)},
      new Object[]{"viewdata", Integer.valueOf(0x2317)},
      new Object[]{"turnednot", Integer.valueOf(0x2319)},
      new Object[]{"varhexagonlrbonds", Integer.valueOf(0x232C)},
      new Object[]{"conictaper", Integer.valueOf(0x2332)},
      new Object[]{"topbot", Integer.valueOf(0x2336)},
      new Object[]{"APLnotbackslash", Integer.valueOf(0x2340)},
      new Object[]{"APLboxquestion", Integer.valueOf(0x2370)},
      new Object[]{"rangledownzigzagarrow", Integer.valueOf(0x237C)},
      new Object[]{"hexagon", Integer.valueOf(0x2394)},
      new Object[]{"bbrktbrk", Integer.valueOf(0x23B6)},
      new Object[]{"varcarriagereturn", Integer.valueOf(0x23CE)},
      new Object[]{"obrbrak", Integer.valueOf(0x23E0)},
      new Object[]{"ubrbrak", Integer.valueOf(0x23E1)},
      new Object[]{"trapezium", Integer.valueOf(0x23E2)},
      new Object[]{"benzenr", Integer.valueOf(0x23E3)},
      new Object[]{"strns", Integer.valueOf(0x23E4)},
      new Object[]{"fltns", Integer.valueOf(0x23E5)},
      new Object[]{"accurrent", Integer.valueOf(0x23E6)},
      new Object[]{"elinters", Integer.valueOf(0x23E7)},
      new Object[]{"mathvisiblespace", Integer.valueOf(0x2423)},
      new Object[]{"circledR", Integer.valueOf(0x24C7)},
      new Object[]{"circledS", Integer.valueOf(0x24C8)},
      new Object[]{"mdlgblksquare", Integer.valueOf(0x25A0)},
      new Object[]{"mdlgwhtsquare", Integer.valueOf(0x25A1)},
      new Object[]{"square", Integer.valueOf(0x25A1)},
      new Object[]{"Box", Integer.valueOf(0x25A1)},
      new Object[]{"squoval", Integer.valueOf(0x25A2)},
      new Object[]{"blackinwhitesquare", Integer.valueOf(0x25A3)},
      new Object[]{"squarehfill", Integer.valueOf(0x25A4)},
      new Object[]{"squarevfill", Integer.valueOf(0x25A5)},
      new Object[]{"squarehvfill", Integer.valueOf(0x25A6)},
      new Object[]{"squarenwsefill", Integer.valueOf(0x25A7)},
      new Object[]{"squareneswfill", Integer.valueOf(0x25A8)},
      new Object[]{"squarecrossfill", Integer.valueOf(0x25A9)},
      new Object[]{"smblksquare", Integer.valueOf(0x25AA)},
      new Object[]{"smwhtsquare", Integer.valueOf(0x25AB)},
      new Object[]{"hrectangleblack", Integer.valueOf(0x25AC)},
      new Object[]{"hrectangle", Integer.valueOf(0x25AD)},
      new Object[]{"vrectangleblack", Integer.valueOf(0x25AE)},
      new Object[]{"vrectangle", Integer.valueOf(0x25AF)},
      new Object[]{"parallelogramblack", Integer.valueOf(0x25B0)},
      new Object[]{"parallelogram", Integer.valueOf(0x25B1)},
      new Object[]{"bigblacktriangleup", Integer.valueOf(0x25B2)},
      new Object[]{"blacktriangle", Integer.valueOf(0x25B4)},
      new Object[]{"blacktriangleright", Integer.valueOf(0x25B6)},
      new Object[]{"smallblacktriangleright", Integer.valueOf(0x25B8)},
      new Object[]{"smalltriangleright", Integer.valueOf(0x25B9)},
      new Object[]{"blackpointerright", Integer.valueOf(0x25BA)},
      new Object[]{"whitepointerright", Integer.valueOf(0x25BB)},
      new Object[]{"bigblacktriangledown", Integer.valueOf(0x25BC)},
      new Object[]{"bigtriangledown", Integer.valueOf(0x25BD)},
      new Object[]{"blacktriangledown", Integer.valueOf(0x25BE)},
      new Object[]{"triangledown", Integer.valueOf(0x25BF)},
      new Object[]{"blacktriangleleft", Integer.valueOf(0x25C0)},
      new Object[]{"smallblacktriangleleft", Integer.valueOf(0x25C2)},
      new Object[]{"smalltriangleleft", Integer.valueOf(0x25C3)},
      new Object[]{"blackpointerleft", Integer.valueOf(0x25C4)},
      new Object[]{"whitepointerleft", Integer.valueOf(0x25C5)},
      new Object[]{"mdlgblkdiamond", Integer.valueOf(0x25C6)},
      new Object[]{"mdlgwhtdiamond", Integer.valueOf(0x25C7)},
      new Object[]{"blackinwhitediamond", Integer.valueOf(0x25C8)},
      new Object[]{"fisheye", Integer.valueOf(0x25C9)},
      new Object[]{"mdlgwhtlozenge", Integer.valueOf(0x25CA)},
      new Object[]{"lozenge", Integer.valueOf(0x25CA)},
      new Object[]{"Diamond", Integer.valueOf(0x25CA)},
      new Object[]{"dottedcircle", Integer.valueOf(0x25CC)},
      new Object[]{"circlevertfill", Integer.valueOf(0x25CD)},
      new Object[]{"bullseye", Integer.valueOf(0x25CE)},
      new Object[]{"mdlgblkcircle", Integer.valueOf(0x25CF)},
      new Object[]{"circlelefthalfblack", Integer.valueOf(0x25D0)},
      new Object[]{"circlerighthalfblack", Integer.valueOf(0x25D1)},
      new Object[]{"circlebottomhalfblack", Integer.valueOf(0x25D2)},
      new Object[]{"circletophalfblack", Integer.valueOf(0x25D3)},
      new Object[]{"circleurquadblack", Integer.valueOf(0x25D4)},
      new Object[]{"blackcircleulquadwhite", Integer.valueOf(0x25D5)},
      new Object[]{"blacklefthalfcircle", Integer.valueOf(0x25D6)},
      new Object[]{"blackrighthalfcircle", Integer.valueOf(0x25D7)},
      new Object[]{"inversebullet", Integer.valueOf(0x25D8)},
      new Object[]{"inversewhitecircle", Integer.valueOf(0x25D9)},
      new Object[]{"invwhiteupperhalfcircle", Integer.valueOf(0x25DA)},
      new Object[]{"invwhitelowerhalfcircle", Integer.valueOf(0x25DB)},
      new Object[]{"ularc", Integer.valueOf(0x25DC)},
      new Object[]{"urarc", Integer.valueOf(0x25DD)},
      new Object[]{"lrarc", Integer.valueOf(0x25DE)},
      new Object[]{"llarc", Integer.valueOf(0x25DF)},
      new Object[]{"topsemicircle", Integer.valueOf(0x25E0)},
      new Object[]{"botsemicircle", Integer.valueOf(0x25E1)},
      new Object[]{"lrblacktriangle", Integer.valueOf(0x25E2)},
      new Object[]{"llblacktriangle", Integer.valueOf(0x25E3)},
      new Object[]{"ulblacktriangle", Integer.valueOf(0x25E4)},
      new Object[]{"urblacktriangle", Integer.valueOf(0x25E5)},
      new Object[]{"circ", Integer.valueOf(0x25E6)},
      new Object[]{"smwhtcircle", Integer.valueOf(0x25E6)},
      new Object[]{"squareleftblack", Integer.valueOf(0x25E7)},
      new Object[]{"squarerightblack", Integer.valueOf(0x25E8)},
      new Object[]{"squareulblack", Integer.valueOf(0x25E9)},
      new Object[]{"squarelrblack", Integer.valueOf(0x25EA)},
      new Object[]{"trianglecdot", Integer.valueOf(0x25EC)},
      new Object[]{"triangleleftblack", Integer.valueOf(0x25ED)},
      new Object[]{"trianglerightblack", Integer.valueOf(0x25EE)},
      new Object[]{"lgwhtcircle", Integer.valueOf(0x25EF)},
      new Object[]{"squareulquad", Integer.valueOf(0x25F0)},
      new Object[]{"squarellquad", Integer.valueOf(0x25F1)},
      new Object[]{"squarelrquad", Integer.valueOf(0x25F2)},
      new Object[]{"squareurquad", Integer.valueOf(0x25F3)},
      new Object[]{"circleulquad", Integer.valueOf(0x25F4)},
      new Object[]{"circlellquad", Integer.valueOf(0x25F5)},
      new Object[]{"circlelrquad", Integer.valueOf(0x25F6)},
      new Object[]{"circleurquad", Integer.valueOf(0x25F7)},
      new Object[]{"ultriangle", Integer.valueOf(0x25F8)},
      new Object[]{"urtriangle", Integer.valueOf(0x25F9)},
      new Object[]{"lltriangle", Integer.valueOf(0x25FA)},
      new Object[]{"mdwhtsquare", Integer.valueOf(0x25FB)},
      new Object[]{"mdblksquare", Integer.valueOf(0x25FC)},
      new Object[]{"mdsmwhtsquare", Integer.valueOf(0x25FD)},
      new Object[]{"mdsmblksquare", Integer.valueOf(0x25FE)},
      new Object[]{"lrtriangle", Integer.valueOf(0x25FF)},
      new Object[]{"bigstar", Integer.valueOf(0x2605)},
      new Object[]{"bigwhitestar", Integer.valueOf(0x2606)},
      new Object[]{"astrosun", Integer.valueOf(0x2609)},
      new Object[]{"danger", Integer.valueOf(0x2621)},
      new Object[]{"blacksmiley", Integer.valueOf(0x263B)},
      new Object[]{"sun", Integer.valueOf(0x263C)},
      new Object[]{"rightmoon", Integer.valueOf(0x263D)},
      new Object[]{"leftmoon", Integer.valueOf(0x263E)},
      new Object[]{"female", Integer.valueOf(0x2640)},
      new Object[]{"male", Integer.valueOf(0x2642)},
      new Object[]{"spadesuit", Integer.valueOf(0x2660)},
      new Object[]{"heartsuit", Integer.valueOf(0x2661)},
      new Object[]{"diamondsuit", Integer.valueOf(0x2662)},
      new Object[]{"clubsuit", Integer.valueOf(0x2663)},
      new Object[]{"varspadesuit", Integer.valueOf(0x2664)},
      new Object[]{"varheartsuit", Integer.valueOf(0x2665)},
      new Object[]{"vardiamondsuit", Integer.valueOf(0x2666)},
      new Object[]{"varclubsuit", Integer.valueOf(0x2667)},
      new Object[]{"quarternote", Integer.valueOf(0x2669)},
      new Object[]{"eighthnote", Integer.valueOf(0x266A)},
      new Object[]{"twonotes", Integer.valueOf(0x266B)},
      new Object[]{"flat", Integer.valueOf(0x266D)},
      new Object[]{"natural", Integer.valueOf(0x266E)},
      new Object[]{"sharp", Integer.valueOf(0x266F)},
      new Object[]{"acidfree", Integer.valueOf(0x267E)},
      new Object[]{"dicei", Integer.valueOf(0x2680)},
      new Object[]{"diceii", Integer.valueOf(0x2681)},
      new Object[]{"diceiii", Integer.valueOf(0x2682)},
      new Object[]{"diceiv", Integer.valueOf(0x2683)},
      new Object[]{"dicev", Integer.valueOf(0x2684)},
      new Object[]{"dicevi", Integer.valueOf(0x2685)},
      new Object[]{"circledrightdot", Integer.valueOf(0x2686)},
      new Object[]{"circledtwodots", Integer.valueOf(0x2687)},
      new Object[]{"blackcircledrightdot", Integer.valueOf(0x2688)},
      new Object[]{"blackcircledtwodots", Integer.valueOf(0x2689)},
      new Object[]{"Hermaphrodite", Integer.valueOf(0x26A5)},
      new Object[]{"mdwhtcircle", Integer.valueOf(0x26AA)},
      new Object[]{"mdblkcircle", Integer.valueOf(0x26AB)},
      new Object[]{"mdsmwhtcircle", Integer.valueOf(0x26AC)},
      new Object[]{"neuter", Integer.valueOf(0x26B2)},
      new Object[]{"checkmark", Integer.valueOf(0x2713)},
      new Object[]{"maltese", Integer.valueOf(0x2720)},
      new Object[]{"circledstar", Integer.valueOf(0x272A)},
      new Object[]{"varstar", Integer.valueOf(0x2736)},
      new Object[]{"dingasterisk", Integer.valueOf(0x273D)},
      new Object[]{"draftingarrow", Integer.valueOf(0x279B)},
      new Object[]{"threedangle", Integer.valueOf(0x27C0)},
      new Object[]{"whiteinwhitetriangle", Integer.valueOf(0x27C1)},
      new Object[]{"subsetcirc", Integer.valueOf(0x27C3)},
      new Object[]{"supsetcirc", Integer.valueOf(0x27C4)},
      new Object[]{"diagup", Integer.valueOf(0x27CB)},
      new Object[]{"diagdown", Integer.valueOf(0x27CD)},
      new Object[]{"diamondcdot", Integer.valueOf(0x27D0)},
      new Object[]{"rdiagovfdiag", Integer.valueOf(0x292B)},
      new Object[]{"fdiagovrdiag", Integer.valueOf(0x292C)},
      new Object[]{"seovnearrow", Integer.valueOf(0x292D)},
      new Object[]{"neovsearrow", Integer.valueOf(0x292E)},
      new Object[]{"fdiagovnearrow", Integer.valueOf(0x292F)},
      new Object[]{"rdiagovsearrow", Integer.valueOf(0x2930)},
      new Object[]{"neovnwarrow", Integer.valueOf(0x2931)},
      new Object[]{"nwovnearrow", Integer.valueOf(0x2932)},
      new Object[]{"uprightcurvearrow", Integer.valueOf(0x2934)},
      new Object[]{"downrightcurvedarrow", Integer.valueOf(0x2935)},
      new Object[]{"mdsmblkcircle", Integer.valueOf(0x2981)},
      new Object[]{"fourvdots", Integer.valueOf(0x2999)},
      new Object[]{"vzigzag", Integer.valueOf(0x299A)},
      new Object[]{"measuredangleleft", Integer.valueOf(0x299B)},
      new Object[]{"rightanglesqr", Integer.valueOf(0x299C)},
      new Object[]{"rightanglemdot", Integer.valueOf(0x299D)},
      new Object[]{"angles", Integer.valueOf(0x299E)},
      new Object[]{"angdnr", Integer.valueOf(0x299F)},
      new Object[]{"gtlpar", Integer.valueOf(0x29A0)},
      new Object[]{"sphericalangleup", Integer.valueOf(0x29A1)},
      new Object[]{"turnangle", Integer.valueOf(0x29A2)},
      new Object[]{"revangle", Integer.valueOf(0x29A3)},
      new Object[]{"angleubar", Integer.valueOf(0x29A4)},
      new Object[]{"revangleubar", Integer.valueOf(0x29A5)},
      new Object[]{"wideangledown", Integer.valueOf(0x29A6)},
      new Object[]{"wideangleup", Integer.valueOf(0x29A7)},
      new Object[]{"measanglerutone", Integer.valueOf(0x29A8)},
      new Object[]{"measanglelutonw", Integer.valueOf(0x29A9)},
      new Object[]{"measanglerdtose", Integer.valueOf(0x29AA)},
      new Object[]{"measangleldtosw", Integer.valueOf(0x29AB)},
      new Object[]{"measangleurtone", Integer.valueOf(0x29AC)},
      new Object[]{"measangleultonw", Integer.valueOf(0x29AD)},
      new Object[]{"measangledrtose", Integer.valueOf(0x29AE)},
      new Object[]{"measangledltosw", Integer.valueOf(0x29AF)},
      new Object[]{"revemptyset", Integer.valueOf(0x29B0)},
      new Object[]{"emptysetobar", Integer.valueOf(0x29B1)},
      new Object[]{"emptysetocirc", Integer.valueOf(0x29B2)},
      new Object[]{"emptysetoarr", Integer.valueOf(0x29B3)},
      new Object[]{"emptysetoarrl", Integer.valueOf(0x29B4)},
      new Object[]{"obot", Integer.valueOf(0x29BA)},
      new Object[]{"olcross", Integer.valueOf(0x29BB)},
      new Object[]{"odotslashdot", Integer.valueOf(0x29BC)},
      new Object[]{"uparrowoncircle", Integer.valueOf(0x29BD)},
      new Object[]{"circledwhitebullet", Integer.valueOf(0x29BE)},
      new Object[]{"circledbullet", Integer.valueOf(0x29BF)},
      new Object[]{"cirscir", Integer.valueOf(0x29C2)},
      new Object[]{"cirE", Integer.valueOf(0x29C3)},
      new Object[]{"boxonbox", Integer.valueOf(0x29C9)},
      new Object[]{"triangleodot", Integer.valueOf(0x29CA)},
      new Object[]{"triangleubar", Integer.valueOf(0x29CB)},
      new Object[]{"triangles", Integer.valueOf(0x29CC)},
      new Object[]{"iinfin", Integer.valueOf(0x29DC)},
      new Object[]{"tieinfty", Integer.valueOf(0x29DD)},
      new Object[]{"nvinfty", Integer.valueOf(0x29DE)},
      new Object[]{"laplac", Integer.valueOf(0x29E0)},
      new Object[]{"thermod", Integer.valueOf(0x29E7)},
      new Object[]{"downtriangleleftblack", Integer.valueOf(0x29E8)},
      new Object[]{"downtrianglerightblack", Integer.valueOf(0x29E9)},
      new Object[]{"blackdiamonddownarrow", Integer.valueOf(0x29EA)},
      new Object[]{"blacklozenge", Integer.valueOf(0x29EB)},
      new Object[]{"circledownarrow", Integer.valueOf(0x29EC)},
      new Object[]{"blackcircledownarrow", Integer.valueOf(0x29ED)},
      new Object[]{"errbarsquare", Integer.valueOf(0x29EE)},
      new Object[]{"errbarblacksquare", Integer.valueOf(0x29EF)},
      new Object[]{"errbardiamond", Integer.valueOf(0x29F0)},
      new Object[]{"errbarblackdiamond", Integer.valueOf(0x29F1)},
      new Object[]{"errbarcircle", Integer.valueOf(0x29F2)},
      new Object[]{"errbarblackcircle", Integer.valueOf(0x29F3)},
      new Object[]{"perps", Integer.valueOf(0x2AE1)},
      new Object[]{"topcir", Integer.valueOf(0x2AF1)},
      new Object[]{"squaretopblack", Integer.valueOf(0x2B12)},
      new Object[]{"squarebotblack", Integer.valueOf(0x2B13)},
      new Object[]{"squareurblack", Integer.valueOf(0x2B14)},
      new Object[]{"squarellblack", Integer.valueOf(0x2B15)},
      new Object[]{"diamondleftblack", Integer.valueOf(0x2B16)},
      new Object[]{"diamondrightblack", Integer.valueOf(0x2B17)},
      new Object[]{"diamondtopblack", Integer.valueOf(0x2B18)},
      new Object[]{"diamondbotblack", Integer.valueOf(0x2B19)},
      new Object[]{"dottedsquare", Integer.valueOf(0x2B1A)},
      new Object[]{"lgblksquare", Integer.valueOf(0x2B1B)},
      new Object[]{"lgwhtsquare", Integer.valueOf(0x2B1C)},
      new Object[]{"vysmblksquare", Integer.valueOf(0x2B1D)},
      new Object[]{"vysmwhtsquare", Integer.valueOf(0x2B1E)},
      new Object[]{"pentagonblack", Integer.valueOf(0x2B1F)},
      new Object[]{"pentagon", Integer.valueOf(0x2B20)},
      new Object[]{"varhexagon", Integer.valueOf(0x2B21)},
      new Object[]{"varhexagonblack", Integer.valueOf(0x2B22)},
      new Object[]{"hexagonblack", Integer.valueOf(0x2B23)},
      new Object[]{"lgblkcircle", Integer.valueOf(0x2B24)},
      new Object[]{"mdblkdiamond", Integer.valueOf(0x2B25)},
      new Object[]{"mdwhtdiamond", Integer.valueOf(0x2B26)},
      new Object[]{"mdblklozenge", Integer.valueOf(0x2B27)},
      new Object[]{"mdwhtlozenge", Integer.valueOf(0x2B28)},
      new Object[]{"smblkdiamond", Integer.valueOf(0x2B29)},
      new Object[]{"smblklozenge", Integer.valueOf(0x2B2A)},
      new Object[]{"smwhtlozenge", Integer.valueOf(0x2B2B)},
      new Object[]{"blkhorzoval", Integer.valueOf(0x2B2C)},
      new Object[]{"whthorzoval", Integer.valueOf(0x2B2D)},
      new Object[]{"blkvertoval", Integer.valueOf(0x2B2E)},
      new Object[]{"whtvertoval", Integer.valueOf(0x2B2F)},
      new Object[]{"medwhitestar", Integer.valueOf(0x2B50)},
      new Object[]{"medblackstar", Integer.valueOf(0x2B51)},
      new Object[]{"smwhitestar", Integer.valueOf(0x2B52)},
      new Object[]{"rightpentagonblack", Integer.valueOf(0x2B53)},
      new Object[]{"rightpentagon", Integer.valueOf(0x2B54)},
      new Object[]{"postalmark", Integer.valueOf(0x3012)},
      new Object[]{"hzigzag", Integer.valueOf(0x3030)},
      new Object[]{"Bbbk", Integer.valueOf(0x1D55C)},
   };

   private static final Object[][] BINARY = new Object[][]
   {
      // binary operators
      new Object[]{"pm", Integer.valueOf(0x00B1)},
      new Object[]{"cdotp", Integer.valueOf(0x00B7)},
      new Object[]{"centerdot", Integer.valueOf(0x00B7)},
      new Object[]{"times", Integer.valueOf(0x00D7)},
      new Object[]{"div", Integer.valueOf(0x00F7)},
      new Object[]{"dagger", Integer.valueOf(0x2020)},
      new Object[]{"ddagger", Integer.valueOf(0x2021)},
      new Object[]{"smblkcircle", Integer.valueOf(0x2022)},
      new Object[]{"fracslash", Integer.valueOf(0x2044)},
      new Object[]{"upand", Integer.valueOf(0x214B)},
      new Object[]{"mp", Integer.valueOf(0x2213)},
      new Object[]{"dotplus", Integer.valueOf(0x2214)},
      new Object[]{"smallsetminus", Integer.valueOf(0x2216)},
      new Object[]{"ast", Integer.valueOf(0x2217)},
      new Object[]{"vysmwhtcircle", Integer.valueOf(0x2218)},
      new Object[]{"vysmblkcircle", Integer.valueOf(0x2219)},
      new Object[]{"wedge", Integer.valueOf(0x2227)},
      new Object[]{"land", Integer.valueOf(0x2227)},
      new Object[]{"vee", Integer.valueOf(0x2228)},
      new Object[]{"lor", Integer.valueOf(0x2228)},
      new Object[]{"cap", Integer.valueOf(0x2229)},
      new Object[]{"cup", Integer.valueOf(0x222A)},
      new Object[]{"dotminus", Integer.valueOf(0x2238)},
      new Object[]{"invlazys", Integer.valueOf(0x223E)},
      new Object[]{"wr", Integer.valueOf(0x2240)},
      new Object[]{"cupleftarrow", Integer.valueOf(0x228C)},
      new Object[]{"cupdot", Integer.valueOf(0x228D)},
      new Object[]{"uplus", Integer.valueOf(0x228E)},
      new Object[]{"sqcap", Integer.valueOf(0x2293)},
      new Object[]{"sqcup", Integer.valueOf(0x2294)},
      new Object[]{"oplus", Integer.valueOf(0x2295)},
      new Object[]{"ominus", Integer.valueOf(0x2296)},
      new Object[]{"otimes", Integer.valueOf(0x2297)},
      new Object[]{"oslash", Integer.valueOf(0x2298)},
      new Object[]{"odot", Integer.valueOf(0x2299)},
      new Object[]{"circledcirc", Integer.valueOf(0x229A)},
      new Object[]{"circledast", Integer.valueOf(0x229B)},
      new Object[]{"circledequal", Integer.valueOf(0x229C)},
      new Object[]{"circleddash", Integer.valueOf(0x229D)},
      new Object[]{"boxplus", Integer.valueOf(0x229E)},
      new Object[]{"boxminus", Integer.valueOf(0x229F)},
      new Object[]{"boxtimes", Integer.valueOf(0x22A0)},
      new Object[]{"boxdot", Integer.valueOf(0x22A1)},
      new Object[]{"intercal", Integer.valueOf(0x22BA)},
      new Object[]{"veebar", Integer.valueOf(0x22BB)},
      new Object[]{"barwedge", Integer.valueOf(0x22BC)},
      new Object[]{"barvee", Integer.valueOf(0x22BD)},
      new Object[]{"diamond", Integer.valueOf(0x22C4)},
      new Object[]{"smwhtdiamond", Integer.valueOf(0x22C4)},
      new Object[]{"cdot", Integer.valueOf(0x22C5)},
      new Object[]{"star", Integer.valueOf(0x22C6)},
      new Object[]{"divideontimes", Integer.valueOf(0x22C7)},
      new Object[]{"ltimes", Integer.valueOf(0x22C9)},
      new Object[]{"rtimes", Integer.valueOf(0x22CA)},
      new Object[]{"leftthreetimes", Integer.valueOf(0x22CB)},
      new Object[]{"rightthreetimes", Integer.valueOf(0x22CC)},
      new Object[]{"curlyvee", Integer.valueOf(0x22CE)},
      new Object[]{"curlywedge", Integer.valueOf(0x22CF)},
      new Object[]{"Cap", Integer.valueOf(0x22D2)},
      new Object[]{"doublecap", Integer.valueOf(0x22D2)},
      new Object[]{"Cup", Integer.valueOf(0x22D3)},
      new Object[]{"doublecup", Integer.valueOf(0x22D3)},
      new Object[]{"varbarwedge", Integer.valueOf(0x2305)},
      new Object[]{"vardoublebarwedge", Integer.valueOf(0x2306)},
      new Object[]{"obar", Integer.valueOf(0x233D)},
      new Object[]{"triangle", Integer.valueOf(0x25B3)},
      new Object[]{"bigtriangleup", Integer.valueOf(0x25B3)},
      new Object[]{"lhd", Integer.valueOf(0x22B2)},
      new Object[]{"rhd", Integer.valueOf(0x22B3)},
      new Object[]{"unlhd", Integer.valueOf(0x22B4)},
      new Object[]{"unrhd", Integer.valueOf(0x22B5)},
      new Object[]{"mdlgwhtcircle", Integer.valueOf(0x25CB)},
      new Object[]{"boxbar", Integer.valueOf(0x25EB)},
      new Object[]{"veedot", Integer.valueOf(0x27C7)},
      new Object[]{"wedgedot", Integer.valueOf(0x27D1)},
      new Object[]{"lozengeminus", Integer.valueOf(0x27E0)},
      new Object[]{"concavediamond", Integer.valueOf(0x27E1)},
      new Object[]{"concavediamondtickleft", Integer.valueOf(0x27E2)},
      new Object[]{"concavediamondtickright", Integer.valueOf(0x27E3)},
      new Object[]{"whitesquaretickleft", Integer.valueOf(0x27E4)},
      new Object[]{"whitesquaretickright", Integer.valueOf(0x27E5)},
      new Object[]{"typecolon", Integer.valueOf(0x2982)},
      new Object[]{"circlehbar", Integer.valueOf(0x29B5)},
      new Object[]{"circledvert", Integer.valueOf(0x29B6)},
      new Object[]{"circledparallel", Integer.valueOf(0x29B7)},
      new Object[]{"obslash", Integer.valueOf(0x29B8)},
      new Object[]{"operp", Integer.valueOf(0x29B9)},
      new Object[]{"olessthan", Integer.valueOf(0x29C0)},
      new Object[]{"ogreaterthan", Integer.valueOf(0x29C1)},
      new Object[]{"boxdiag", Integer.valueOf(0x29C4)},
      new Object[]{"boxbslash", Integer.valueOf(0x29C5)},
      new Object[]{"boxast", Integer.valueOf(0x29C6)},
      new Object[]{"boxcircle", Integer.valueOf(0x29C7)},
      new Object[]{"boxbox", Integer.valueOf(0x29C8)},
      new Object[]{"triangleserifs", Integer.valueOf(0x29CD)},
      new Object[]{"hourglass", Integer.valueOf(0x29D6)},
      new Object[]{"blackhourglass", Integer.valueOf(0x29D7)},
      new Object[]{"shuffle", Integer.valueOf(0x29E2)},
      new Object[]{"mdlgblklozenge", Integer.valueOf(0x29EB)},
      new Object[]{"setminus", Integer.valueOf(0x29F5)},
      new Object[]{"dsol", Integer.valueOf(0x29F6)},
      new Object[]{"rsolbar", Integer.valueOf(0x29F7)},
      new Object[]{"doubleplus", Integer.valueOf(0x29FA)},
      new Object[]{"tripleplus", Integer.valueOf(0x29FB)},
      new Object[]{"tplus", Integer.valueOf(0x29FE)},
      new Object[]{"tminus", Integer.valueOf(0x29FF)},
      new Object[]{"ringplus", Integer.valueOf(0x2A22)},
      new Object[]{"plushat", Integer.valueOf(0x2A23)},
      new Object[]{"simplus", Integer.valueOf(0x2A24)},
      new Object[]{"plusdot", Integer.valueOf(0x2A25)},
      new Object[]{"plussim", Integer.valueOf(0x2A26)},
      new Object[]{"plussubtwo", Integer.valueOf(0x2A27)},
      new Object[]{"plustrif", Integer.valueOf(0x2A28)},
      new Object[]{"commaminus", Integer.valueOf(0x2A29)},
      new Object[]{"minusdot", Integer.valueOf(0x2A2A)},
      new Object[]{"minusfdots", Integer.valueOf(0x2A2B)},
      new Object[]{"minusrdots", Integer.valueOf(0x2A2C)},
      new Object[]{"opluslhrim", Integer.valueOf(0x2A2D)},
      new Object[]{"oplusrhrim", Integer.valueOf(0x2A2E)},
      new Object[]{"vectimes", Integer.valueOf(0x2A2F)},
      new Object[]{"dottimes", Integer.valueOf(0x2A30)},
      new Object[]{"timesbar", Integer.valueOf(0x2A31)},
      new Object[]{"btimes", Integer.valueOf(0x2A32)},
      new Object[]{"smashtimes", Integer.valueOf(0x2A33)},
      new Object[]{"otimeslhrim", Integer.valueOf(0x2A34)},
      new Object[]{"otimesrhrim", Integer.valueOf(0x2A35)},
      new Object[]{"otimeshat", Integer.valueOf(0x2A36)},
      new Object[]{"Otimes", Integer.valueOf(0x2A37)},
      new Object[]{"odiv", Integer.valueOf(0x2A38)},
      new Object[]{"triangleplus", Integer.valueOf(0x2A39)},
      new Object[]{"triangleminus", Integer.valueOf(0x2A3A)},
      new Object[]{"triangletimes", Integer.valueOf(0x2A3B)},
      new Object[]{"intprod", Integer.valueOf(0x2A3C)},
      new Object[]{"intprodr", Integer.valueOf(0x2A3D)},
      new Object[]{"fcmp", Integer.valueOf(0x2A3E)},
      new Object[]{"amalg", Integer.valueOf(0x2A3F)},
      new Object[]{"capdot", Integer.valueOf(0x2A40)},
      new Object[]{"uminus", Integer.valueOf(0x2A41)},
      new Object[]{"barcup", Integer.valueOf(0x2A42)},
      new Object[]{"barcap", Integer.valueOf(0x2A43)},
      new Object[]{"capwedge", Integer.valueOf(0x2A44)},
      new Object[]{"cupvee", Integer.valueOf(0x2A45)},
      new Object[]{"cupovercap", Integer.valueOf(0x2A46)},
      new Object[]{"capovercup", Integer.valueOf(0x2A47)},
      new Object[]{"cupbarcap", Integer.valueOf(0x2A48)},
      new Object[]{"capbarcup", Integer.valueOf(0x2A49)},
      new Object[]{"twocups", Integer.valueOf(0x2A4A)},
      new Object[]{"twocaps", Integer.valueOf(0x2A4B)},
      new Object[]{"closedvarcup", Integer.valueOf(0x2A4C)},
      new Object[]{"closedvarcap", Integer.valueOf(0x2A4D)},
      new Object[]{"Sqcap", Integer.valueOf(0x2A4E)},
      new Object[]{"Sqcup", Integer.valueOf(0x2A4F)},
      new Object[]{"closedvarcupsmashprod", Integer.valueOf(0x2A50)},
      new Object[]{"wedgeodot", Integer.valueOf(0x2A51)},
      new Object[]{"veeodot", Integer.valueOf(0x2A52)},
      new Object[]{"Wedge", Integer.valueOf(0x2A53)},
      new Object[]{"Vee", Integer.valueOf(0x2A54)},
      new Object[]{"wedgeonwedge", Integer.valueOf(0x2A55)},
      new Object[]{"veeonvee", Integer.valueOf(0x2A56)},
      new Object[]{"bigslopedvee", Integer.valueOf(0x2A57)},
      new Object[]{"bigslopedwedge", Integer.valueOf(0x2A58)},
      new Object[]{"wedgemidvert", Integer.valueOf(0x2A5A)},
      new Object[]{"veemidvert", Integer.valueOf(0x2A5B)},
      new Object[]{"midbarwedge", Integer.valueOf(0x2A5C)},
      new Object[]{"midbarvee", Integer.valueOf(0x2A5D)},
      new Object[]{"doublebarwedge", Integer.valueOf(0x2A5E)},
      new Object[]{"wedgebar", Integer.valueOf(0x2A5F)},
      new Object[]{"wedgedoublebar", Integer.valueOf(0x2A60)},
      new Object[]{"varveebar", Integer.valueOf(0x2A61)},
      new Object[]{"doublebarvee", Integer.valueOf(0x2A62)},
      new Object[]{"veedoublebar", Integer.valueOf(0x2A63)},
      new Object[]{"dsub", Integer.valueOf(0x2A64)},
      new Object[]{"rsub", Integer.valueOf(0x2A65)},
      new Object[]{"eqqplus", Integer.valueOf(0x2A71)},
      new Object[]{"pluseqq", Integer.valueOf(0x2A72)},
      new Object[]{"interleave", Integer.valueOf(0x2AF4)},
      new Object[]{"nhVvert", Integer.valueOf(0x2AF5)},
      new Object[]{"threedotcolon", Integer.valueOf(0x2AF6)},
      new Object[]{"trslash", Integer.valueOf(0x2AFB)},
      new Object[]{"sslash", Integer.valueOf(0x2AFD)},
      new Object[]{"talloblong", Integer.valueOf(0x2AFE)},
      // relations
      new Object[]{"ast", Integer.valueOf(0x002A)},
      new Object[]{"less", Integer.valueOf(0x003C)},
      new Object[]{"equal", Integer.valueOf(0x003D)},
      new Object[]{"greater", Integer.valueOf(0x003E)},
      new Object[]{"closure", Integer.valueOf(0x2050)},
      new Object[]{"vertoverlay", Integer.valueOf(0x20D2)},
      new Object[]{"leftarrow", Integer.valueOf(0x2190)},
      new Object[]{"gets", Integer.valueOf(0x2190)},
      new Object[]{"uparrow", Integer.valueOf(0x2191)},
      new Object[]{"rightarrow", Integer.valueOf(0x2192)},
      new Object[]{"to", Integer.valueOf(0x2192)},
      new Object[]{"downarrow", Integer.valueOf(0x2193)},
      new Object[]{"leftrightarrow", Integer.valueOf(0x2194)},
      new Object[]{"updownarrow", Integer.valueOf(0x2195)},
      new Object[]{"nwarrow", Integer.valueOf(0x2196)},
      new Object[]{"nearrow", Integer.valueOf(0x2197)},
      new Object[]{"searrow", Integer.valueOf(0x2198)},
      new Object[]{"swarrow", Integer.valueOf(0x2199)},
      new Object[]{"nleftarrow", Integer.valueOf(0x219A)},
      new Object[]{"nrightarrow", Integer.valueOf(0x219B)},
      new Object[]{"leftwavearrow", Integer.valueOf(0x219C)},
      new Object[]{"rightwavearrow", Integer.valueOf(0x219D)},
      new Object[]{"twoheadleftarrow", Integer.valueOf(0x219E)},
      new Object[]{"twoheaduparrow", Integer.valueOf(0x219F)},
      new Object[]{"twoheadrightarrow", Integer.valueOf(0x21A0)},
      new Object[]{"twoheaddownarrow", Integer.valueOf(0x21A1)},
      new Object[]{"leftarrowtail", Integer.valueOf(0x21A2)},
      new Object[]{"rightarrowtail", Integer.valueOf(0x21A3)},
      new Object[]{"mapsfrom", Integer.valueOf(0x21A4)},
      new Object[]{"mapsup", Integer.valueOf(0x21A5)},
      new Object[]{"mapsto", Integer.valueOf(0x21A6)},
      new Object[]{"mapsdown", Integer.valueOf(0x21A7)},
      new Object[]{"hookleftarrow", Integer.valueOf(0x21A9)},
      new Object[]{"hookrightarrow", Integer.valueOf(0x21AA)},
      new Object[]{"looparrowleft", Integer.valueOf(0x21AB)},
      new Object[]{"looparrowright", Integer.valueOf(0x21AC)},
      new Object[]{"leftrightsquigarrow", Integer.valueOf(0x21AD)},
      new Object[]{"nleftrightarrow", Integer.valueOf(0x21AE)},
      new Object[]{"downzigzagarrow", Integer.valueOf(0x21AF)},
      new Object[]{"Lsh", Integer.valueOf(0x21B0)},
      new Object[]{"Rsh", Integer.valueOf(0x21B1)},
      new Object[]{"Ldsh", Integer.valueOf(0x21B2)},
      new Object[]{"Rdsh", Integer.valueOf(0x21B3)},
      new Object[]{"curvearrowleft", Integer.valueOf(0x21B6)},
      new Object[]{"curvearrowright", Integer.valueOf(0x21B7)},
      new Object[]{"circlearrowleft", Integer.valueOf(0x21BA)},
      new Object[]{"circlearrowright", Integer.valueOf(0x21BB)},
      new Object[]{"leftharpoonup", Integer.valueOf(0x21BC)},
      new Object[]{"leftharpoondown", Integer.valueOf(0x21BD)},
      new Object[]{"upharpoonright", Integer.valueOf(0x21BE)},
      new Object[]{"restriction", Integer.valueOf(0x21BE)},
      new Object[]{"upharpoonleft", Integer.valueOf(0x21BF)},
      new Object[]{"rightharpoonup", Integer.valueOf(0x21C0)},
      new Object[]{"rightharpoondown", Integer.valueOf(0x21C1)},
      new Object[]{"downharpoonright", Integer.valueOf(0x21C2)},
      new Object[]{"downharpoonleft", Integer.valueOf(0x21C3)},
      new Object[]{"rightleftarrows", Integer.valueOf(0x21C4)},
      new Object[]{"updownarrows", Integer.valueOf(0x21C5)},
      new Object[]{"leftrightarrows", Integer.valueOf(0x21C6)},
      new Object[]{"leftleftarrows", Integer.valueOf(0x21C7)},
      new Object[]{"upuparrows", Integer.valueOf(0x21C8)},
      new Object[]{"rightrightarrows", Integer.valueOf(0x21C9)},
      new Object[]{"downdownarrows", Integer.valueOf(0x21CA)},
      new Object[]{"leftrightharpoons", Integer.valueOf(0x21CB)},
      new Object[]{"rightleftharpoons", Integer.valueOf(0x21CC)},
      new Object[]{"nLeftarrow", Integer.valueOf(0x21CD)},
      new Object[]{"nLeftrightarrow", Integer.valueOf(0x21CE)},
      new Object[]{"nRightarrow", Integer.valueOf(0x21CF)},
      new Object[]{"Leftarrow", Integer.valueOf(0x21D0)},
      new Object[]{"Uparrow", Integer.valueOf(0x21D1)},
      new Object[]{"Rightarrow", Integer.valueOf(0x21D2)},
      new Object[]{"Downarrow", Integer.valueOf(0x21D3)},
      new Object[]{"Leftrightarrow", Integer.valueOf(0x21D4)},
      new Object[]{"Updownarrow", Integer.valueOf(0x21D5)},
      new Object[]{"Nwarrow", Integer.valueOf(0x21D6)},
      new Object[]{"Nearrow", Integer.valueOf(0x21D7)},
      new Object[]{"Searrow", Integer.valueOf(0x21D8)},
      new Object[]{"Swarrow", Integer.valueOf(0x21D9)},
      new Object[]{"Lleftarrow", Integer.valueOf(0x21DA)},
      new Object[]{"Rrightarrow", Integer.valueOf(0x21DB)},
      new Object[]{"leftsquigarrow", Integer.valueOf(0x21DC)},
      new Object[]{"rightsquigarrow", Integer.valueOf(0x21DD)},
      new Object[]{"leadsto", Integer.valueOf(0x21DD)},
      new Object[]{"barleftarrow", Integer.valueOf(0x21E4)},
      new Object[]{"rightarrowbar", Integer.valueOf(0x21E5)},
      new Object[]{"circleonrightarrow", Integer.valueOf(0x21F4)},
      new Object[]{"downuparrows", Integer.valueOf(0x21F5)},
      new Object[]{"rightthreearrows", Integer.valueOf(0x21F6)},
      new Object[]{"nvleftarrow", Integer.valueOf(0x21F7)},
      new Object[]{"nvrightarrow", Integer.valueOf(0x21F8)},
      new Object[]{"nvleftrightarrow", Integer.valueOf(0x21F9)},
      new Object[]{"nVleftarrow", Integer.valueOf(0x21FA)},
      new Object[]{"nVrightarrow", Integer.valueOf(0x21FB)},
      new Object[]{"nVleftrightarrow", Integer.valueOf(0x21FC)},
      new Object[]{"leftarrowtriangle", Integer.valueOf(0x21FD)},
      new Object[]{"rightarrowtriangle", Integer.valueOf(0x21FE)},
      new Object[]{"leftrightarrowtriangle", Integer.valueOf(0x21FF)},
      new Object[]{"in", Integer.valueOf(0x2208)},
      new Object[]{"notin", Integer.valueOf(0x2209)},
      new Object[]{"smallin", Integer.valueOf(0x220A)},
      new Object[]{"ni", Integer.valueOf(0x220B)},
      new Object[]{"owns", Integer.valueOf(0x220B)},
      new Object[]{"nni", Integer.valueOf(0x220C)},
      new Object[]{"smallni", Integer.valueOf(0x220D)},
      new Object[]{"propto", Integer.valueOf(0x221D)},
      new Object[]{"varpropto", Integer.valueOf(0x221D)},
      new Object[]{"mid", Integer.valueOf(0x2223)},
      new Object[]{"shortmid", Integer.valueOf(0x2223)},
      new Object[]{"nmid", Integer.valueOf(0x2224)},
      new Object[]{"nshortmid", Integer.valueOf(0x2224)},
      new Object[]{"parallel", Integer.valueOf(0x2225)},
      new Object[]{"shortparallel", Integer.valueOf(0x2225)},
      new Object[]{"nparallel", Integer.valueOf(0x2226)},
      new Object[]{"nshortparallel", Integer.valueOf(0x2226)},
      new Object[]{"Colon", Integer.valueOf(0x2237)},
      new Object[]{"dashcolon", Integer.valueOf(0x2239)},
      new Object[]{"dotsminusdots", Integer.valueOf(0x223A)},
      new Object[]{"kernelcontraction", Integer.valueOf(0x223B)},
      new Object[]{"sim", Integer.valueOf(0x223C)},
      new Object[]{"thicksim", Integer.valueOf(0x223C)},
      new Object[]{"backsim", Integer.valueOf(0x223D)},
      new Object[]{"nsim", Integer.valueOf(0x2241)},
      new Object[]{"eqsim", Integer.valueOf(0x2242)},
      new Object[]{"simeq", Integer.valueOf(0x2243)},
      new Object[]{"nsime", Integer.valueOf(0x2244)},
      new Object[]{"cong", Integer.valueOf(0x2245)},
      new Object[]{"simneqq", Integer.valueOf(0x2246)},
      new Object[]{"ncong", Integer.valueOf(0x2247)},
      new Object[]{"approx", Integer.valueOf(0x2248)},
      new Object[]{"thickapprox", Integer.valueOf(0x2248)},
      new Object[]{"napprox", Integer.valueOf(0x2249)},
      new Object[]{"approxeq", Integer.valueOf(0x224A)},
      new Object[]{"approxident", Integer.valueOf(0x224B)},
      new Object[]{"backcong", Integer.valueOf(0x224C)},
      new Object[]{"asymp", Integer.valueOf(0x224D)},
      new Object[]{"Bumpeq", Integer.valueOf(0x224E)},
      new Object[]{"bumpeq", Integer.valueOf(0x224F)},
      new Object[]{"doteq", Integer.valueOf(0x2250)},
      new Object[]{"Doteq", Integer.valueOf(0x2251)},
      new Object[]{"doteqdot", Integer.valueOf(0x2251)},
      new Object[]{"fallingdotseq", Integer.valueOf(0x2252)},
      new Object[]{"risingdotseq", Integer.valueOf(0x2253)},
      new Object[]{"coloneq", Integer.valueOf(0x2254)},
      new Object[]{"eqcolon", Integer.valueOf(0x2255)},
      new Object[]{"eqcirc", Integer.valueOf(0x2256)},
      new Object[]{"circeq", Integer.valueOf(0x2257)},
      new Object[]{"arceq", Integer.valueOf(0x2258)},
      new Object[]{"wedgeq", Integer.valueOf(0x2259)},
      new Object[]{"veeeq", Integer.valueOf(0x225A)},
      new Object[]{"stareq", Integer.valueOf(0x225B)},
      new Object[]{"triangleq", Integer.valueOf(0x225C)},
      new Object[]{"eqdef", Integer.valueOf(0x225D)},
      new Object[]{"measeq", Integer.valueOf(0x225E)},
      new Object[]{"questeq", Integer.valueOf(0x225F)},
      new Object[]{"ne", Integer.valueOf(0x2260)},
      new Object[]{"neq", Integer.valueOf(0x2260)},
      new Object[]{"equiv", Integer.valueOf(0x2261)},
      new Object[]{"nequiv", Integer.valueOf(0x2262)},
      new Object[]{"Equiv", Integer.valueOf(0x2263)},
      new Object[]{"leq", Integer.valueOf(0x2264)},
      new Object[]{"le", Integer.valueOf(0x2264)},
      new Object[]{"geq", Integer.valueOf(0x2265)},
      new Object[]{"ge", Integer.valueOf(0x2265)},
      new Object[]{"leqq", Integer.valueOf(0x2266)},
      new Object[]{"geqq", Integer.valueOf(0x2267)},
      new Object[]{"lneqq", Integer.valueOf(0x2268)},
      new Object[]{"lvertneqq", Integer.valueOf(0x2268)},
      new Object[]{"gneqq", Integer.valueOf(0x2269)},
      new Object[]{"gvertneqq", Integer.valueOf(0x2269)},
      new Object[]{"ll", Integer.valueOf(0x226A)},
      new Object[]{"gg", Integer.valueOf(0x226B)},
      new Object[]{"between", Integer.valueOf(0x226C)},
      new Object[]{"nasymp", Integer.valueOf(0x226D)},
      new Object[]{"nless", Integer.valueOf(0x226E)},
      new Object[]{"ngtr", Integer.valueOf(0x226F)},
      new Object[]{"nleq", Integer.valueOf(0x2270)},
      new Object[]{"ngeq", Integer.valueOf(0x2271)},
      new Object[]{"lesssim", Integer.valueOf(0x2272)},
      new Object[]{"gtrsim", Integer.valueOf(0x2273)},
      new Object[]{"nlesssim", Integer.valueOf(0x2274)},
      new Object[]{"ngtrsim", Integer.valueOf(0x2275)},
      new Object[]{"lessgtr", Integer.valueOf(0x2276)},
      new Object[]{"gtrless", Integer.valueOf(0x2277)},
      new Object[]{"nlessgtr", Integer.valueOf(0x2278)},
      new Object[]{"ngtrless", Integer.valueOf(0x2279)},
      new Object[]{"prec", Integer.valueOf(0x227A)},
      new Object[]{"succ", Integer.valueOf(0x227B)},
      new Object[]{"preccurlyeq", Integer.valueOf(0x227C)},
      new Object[]{"succcurlyeq", Integer.valueOf(0x227D)},
      new Object[]{"precsim", Integer.valueOf(0x227E)},
      new Object[]{"succsim", Integer.valueOf(0x227F)},
      new Object[]{"nprec", Integer.valueOf(0x2280)},
      new Object[]{"nsucc", Integer.valueOf(0x2281)},
      new Object[]{"subset", Integer.valueOf(0x2282)},
      new Object[]{"supset", Integer.valueOf(0x2283)},
      new Object[]{"nsubset", Integer.valueOf(0x2284)},
      new Object[]{"nsupset", Integer.valueOf(0x2285)},
      new Object[]{"subseteq", Integer.valueOf(0x2286)},
      new Object[]{"supseteq", Integer.valueOf(0x2287)},
      new Object[]{"nsubseteq", Integer.valueOf(0x2288)},
      new Object[]{"nsupseteq", Integer.valueOf(0x2289)},
      new Object[]{"subsetneq", Integer.valueOf(0x228A)},
      new Object[]{"varsubsetneq", Integer.valueOf(0x228A)},
      new Object[]{"supsetneq", Integer.valueOf(0x228B)},
      new Object[]{"varsupsetneq", Integer.valueOf(0x228B)},
      new Object[]{"sqsubset", Integer.valueOf(0x228F)},
      new Object[]{"sqsupset", Integer.valueOf(0x2290)},
      new Object[]{"sqsubseteq", Integer.valueOf(0x2291)},
      new Object[]{"sqsupseteq", Integer.valueOf(0x2292)},
      new Object[]{"vdash", Integer.valueOf(0x22A2)},
      new Object[]{"dashv", Integer.valueOf(0x22A3)},
      new Object[]{"assert", Integer.valueOf(0x22A6)},
      new Object[]{"models", Integer.valueOf(0x22A7)},
      new Object[]{"vDash", Integer.valueOf(0x22A8)},
      new Object[]{"Vdash", Integer.valueOf(0x22A9)},
      new Object[]{"Vvdash", Integer.valueOf(0x22AA)},
      new Object[]{"VDash", Integer.valueOf(0x22AB)},
      new Object[]{"nvdash", Integer.valueOf(0x22AC)},
      new Object[]{"nvDash", Integer.valueOf(0x22AD)},
      new Object[]{"nVdash", Integer.valueOf(0x22AE)},
      new Object[]{"nVDash", Integer.valueOf(0x22AF)},
      new Object[]{"prurel", Integer.valueOf(0x22B0)},
      new Object[]{"scurel", Integer.valueOf(0x22B1)},
      new Object[]{"vartriangleleft", Integer.valueOf(0x22B2)},
      new Object[]{"vartriangleright", Integer.valueOf(0x22B3)},
      new Object[]{"trianglelefteq", Integer.valueOf(0x22B4)},
      new Object[]{"trianglerighteq", Integer.valueOf(0x22B5)},
      new Object[]{"origof", Integer.valueOf(0x22B6)},
      new Object[]{"imageof", Integer.valueOf(0x22B7)},
      new Object[]{"multimap", Integer.valueOf(0x22B8)},
      new Object[]{"bowtie", Integer.valueOf(0x22C8)},
      new Object[]{"backsimeq", Integer.valueOf(0x22CD)},
      new Object[]{"Subset", Integer.valueOf(0x22D0)},
      new Object[]{"Supset", Integer.valueOf(0x22D1)},
      new Object[]{"pitchfork", Integer.valueOf(0x22D4)},
      new Object[]{"equalparallel", Integer.valueOf(0x22D5)},
      new Object[]{"lessdot", Integer.valueOf(0x22D6)},
      new Object[]{"gtrdot", Integer.valueOf(0x22D7)},
      new Object[]{"lll", Integer.valueOf(0x22D8)},
      new Object[]{"llless", Integer.valueOf(0x22D8)},
      new Object[]{"ggg", Integer.valueOf(0x22D9)},
      new Object[]{"gggtr", Integer.valueOf(0x22D9)},
      new Object[]{"lesseqgtr", Integer.valueOf(0x22DA)},
      new Object[]{"gtreqless", Integer.valueOf(0x22DB)},
      new Object[]{"eqless", Integer.valueOf(0x22DC)},
      new Object[]{"eqgtr", Integer.valueOf(0x22DD)},
      new Object[]{"curlyeqprec", Integer.valueOf(0x22DE)},
      new Object[]{"curlyeqsucc", Integer.valueOf(0x22DF)},
      new Object[]{"npreccurlyeq", Integer.valueOf(0x22E0)},
      new Object[]{"nsucccurlyeq", Integer.valueOf(0x22E1)},
      new Object[]{"nsqsubseteq", Integer.valueOf(0x22E2)},
      new Object[]{"nsqsupseteq", Integer.valueOf(0x22E3)},
      new Object[]{"sqsubsetneq", Integer.valueOf(0x22E4)},
      new Object[]{"sqsupsetneq", Integer.valueOf(0x22E5)},
      new Object[]{"lnsim", Integer.valueOf(0x22E6)},
      new Object[]{"gnsim", Integer.valueOf(0x22E7)},
      new Object[]{"precnsim", Integer.valueOf(0x22E8)},
      new Object[]{"succnsim", Integer.valueOf(0x22E9)},
      new Object[]{"nvartriangleleft", Integer.valueOf(0x22EA)},
      new Object[]{"nvartriangleright", Integer.valueOf(0x22EB)},
      new Object[]{"ntrianglelefteq", Integer.valueOf(0x22EC)},
      new Object[]{"ntrianglerighteq", Integer.valueOf(0x22ED)},
      new Object[]{"vdots", Integer.valueOf(0x22EE)},
      new Object[]{"adots", Integer.valueOf(0x22F0)},
      new Object[]{"ddots", Integer.valueOf(0x22F1)},
      new Object[]{"disin", Integer.valueOf(0x22F2)},
      new Object[]{"varisins", Integer.valueOf(0x22F3)},
      new Object[]{"isins", Integer.valueOf(0x22F4)},
      new Object[]{"isindot", Integer.valueOf(0x22F5)},
      new Object[]{"varisinobar", Integer.valueOf(0x22F6)},
      new Object[]{"isinobar", Integer.valueOf(0x22F7)},
      new Object[]{"isinvb", Integer.valueOf(0x22F8)},
      new Object[]{"isinE", Integer.valueOf(0x22F9)},
      new Object[]{"nisd", Integer.valueOf(0x22FA)},
      new Object[]{"varnis", Integer.valueOf(0x22FB)},
      new Object[]{"nis", Integer.valueOf(0x22FC)},
      new Object[]{"varniobar", Integer.valueOf(0x22FD)},
      new Object[]{"niobar", Integer.valueOf(0x22FE)},
      new Object[]{"bagmember", Integer.valueOf(0x22FF)},
      new Object[]{"frown", Integer.valueOf(0x2322)},
      new Object[]{"smallfrown", Integer.valueOf(0x2322)},
      new Object[]{"smile", Integer.valueOf(0x2323)},
      new Object[]{"smallsmile", Integer.valueOf(0x2323)},
      new Object[]{"APLnotslash", Integer.valueOf(0x233F)},
      new Object[]{"vartriangle", Integer.valueOf(0x25B5)},
      new Object[]{"perp", Integer.valueOf(0x27C2)},
      new Object[]{"bsolhsub", Integer.valueOf(0x27C8)},
      new Object[]{"suphsol", Integer.valueOf(0x27C9)},
      new Object[]{"upin", Integer.valueOf(0x27D2)},
      new Object[]{"pullback", Integer.valueOf(0x27D3)},
      new Object[]{"pushout", Integer.valueOf(0x27D4)},
      new Object[]{"DashVDash", Integer.valueOf(0x27DA)},
      new Object[]{"dashVdash", Integer.valueOf(0x27DB)},
      new Object[]{"multimapinv", Integer.valueOf(0x27DC)},
      new Object[]{"vlongdash", Integer.valueOf(0x27DD)},
      new Object[]{"longdashv", Integer.valueOf(0x27DE)},
      new Object[]{"cirbot", Integer.valueOf(0x27DF)},
      new Object[]{"UUparrow", Integer.valueOf(0x27F0)},
      new Object[]{"DDownarrow", Integer.valueOf(0x27F1)},
      new Object[]{"acwgapcirclearrow", Integer.valueOf(0x27F2)},
      new Object[]{"cwgapcirclearrow", Integer.valueOf(0x27F3)},
      new Object[]{"rightarrowonoplus", Integer.valueOf(0x27F4)},
      new Object[]{"longleftarrow", Integer.valueOf(0x27F5)},
      new Object[]{"longrightarrow", Integer.valueOf(0x27F6)},
      new Object[]{"longleftrightarrow", Integer.valueOf(0x27F7)},
      new Object[]{"Longleftarrow", Integer.valueOf(0x27F8)},
      new Object[]{"Longrightarrow", Integer.valueOf(0x27F9)},
      new Object[]{"Longleftrightarrow", Integer.valueOf(0x27FA)},
      new Object[]{"longmapsfrom", Integer.valueOf(0x27FB)},
      new Object[]{"longmapsto", Integer.valueOf(0x27FC)},
      new Object[]{"Longmapsfrom", Integer.valueOf(0x27FD)},
      new Object[]{"Longmapsto", Integer.valueOf(0x27FE)},
      new Object[]{"longrightsquigarrow", Integer.valueOf(0x27FF)},
      new Object[]{"nvtwoheadrightarrow", Integer.valueOf(0x2900)},
      new Object[]{"nVtwoheadrightarrow", Integer.valueOf(0x2901)},
      new Object[]{"nvLeftarrow", Integer.valueOf(0x2902)},
      new Object[]{"nvRightarrow", Integer.valueOf(0x2903)},
      new Object[]{"nvLeftrightarrow", Integer.valueOf(0x2904)},
      new Object[]{"twoheadmapsto", Integer.valueOf(0x2905)},
      new Object[]{"Mapsfrom", Integer.valueOf(0x2906)},
      new Object[]{"Mapsto", Integer.valueOf(0x2907)},
      new Object[]{"downarrowbarred", Integer.valueOf(0x2908)},
      new Object[]{"uparrowbarred", Integer.valueOf(0x2909)},
      new Object[]{"Uuparrow", Integer.valueOf(0x290A)},
      new Object[]{"Ddownarrow", Integer.valueOf(0x290B)},
      new Object[]{"leftbkarrow", Integer.valueOf(0x290C)},
      new Object[]{"rightbkarrow", Integer.valueOf(0x290D)},
      new Object[]{"leftdbkarrow", Integer.valueOf(0x290E)},
      new Object[]{"dashleftarrow", Integer.valueOf(0x290E)},
      new Object[]{"dbkarow", Integer.valueOf(0x290F)},
      new Object[]{"dashrightarrow", Integer.valueOf(0x290F)},
      new Object[]{"drbkarow", Integer.valueOf(0x2910)},
      new Object[]{"rightdotarrow", Integer.valueOf(0x2911)},
      new Object[]{"baruparrow", Integer.valueOf(0x2912)},
      new Object[]{"downarrowbar", Integer.valueOf(0x2913)},
      new Object[]{"nvrightarrowtail", Integer.valueOf(0x2914)},
      new Object[]{"nVrightarrowtail", Integer.valueOf(0x2915)},
      new Object[]{"twoheadrightarrowtail", Integer.valueOf(0x2916)},
      new Object[]{"nvtwoheadrightarrowtail", Integer.valueOf(0x2917)},
      new Object[]{"nVtwoheadrightarrowtail", Integer.valueOf(0x2918)},
      new Object[]{"lefttail", Integer.valueOf(0x2919)},
      new Object[]{"righttail", Integer.valueOf(0x291A)},
      new Object[]{"leftdbltail", Integer.valueOf(0x291B)},
      new Object[]{"rightdbltail", Integer.valueOf(0x291C)},
      new Object[]{"diamondleftarrow", Integer.valueOf(0x291D)},
      new Object[]{"rightarrowdiamond", Integer.valueOf(0x291E)},
      new Object[]{"diamondleftarrowbar", Integer.valueOf(0x291F)},
      new Object[]{"barrightarrowdiamond", Integer.valueOf(0x2920)},
      new Object[]{"nwsearrow", Integer.valueOf(0x2921)},
      new Object[]{"neswarrow", Integer.valueOf(0x2922)},
      new Object[]{"hknwarrow", Integer.valueOf(0x2923)},
      new Object[]{"hknearrow", Integer.valueOf(0x2924)},
      new Object[]{"hksearow", Integer.valueOf(0x2925)},
      new Object[]{"hkswarow", Integer.valueOf(0x2926)},
      new Object[]{"tona", Integer.valueOf(0x2927)},
      new Object[]{"toea", Integer.valueOf(0x2928)},
      new Object[]{"tosa", Integer.valueOf(0x2929)},
      new Object[]{"towa", Integer.valueOf(0x292A)},
      new Object[]{"rightcurvedarrow", Integer.valueOf(0x2933)},
      new Object[]{"leftdowncurvedarrow", Integer.valueOf(0x2936)},
      new Object[]{"rightdowncurvedarrow", Integer.valueOf(0x2937)},
      new Object[]{"cwrightarcarrow", Integer.valueOf(0x2938)},
      new Object[]{"acwleftarcarrow", Integer.valueOf(0x2939)},
      new Object[]{"acwoverarcarrow", Integer.valueOf(0x293A)},
      new Object[]{"acwunderarcarrow", Integer.valueOf(0x293B)},
      new Object[]{"curvearrowrightminus", Integer.valueOf(0x293C)},
      new Object[]{"curvearrowleftplus", Integer.valueOf(0x293D)},
      new Object[]{"cwundercurvearrow", Integer.valueOf(0x293E)},
      new Object[]{"ccwundercurvearrow", Integer.valueOf(0x293F)},
      new Object[]{"acwcirclearrow", Integer.valueOf(0x2940)},
      new Object[]{"cwcirclearrow", Integer.valueOf(0x2941)},
      new Object[]{"rightarrowshortleftarrow", Integer.valueOf(0x2942)},
      new Object[]{"leftarrowshortrightarrow", Integer.valueOf(0x2943)},
      new Object[]{"shortrightarrowleftarrow", Integer.valueOf(0x2944)},
      new Object[]{"rightarrowplus", Integer.valueOf(0x2945)},
      new Object[]{"leftarrowplus", Integer.valueOf(0x2946)},
      new Object[]{"rightarrowx", Integer.valueOf(0x2947)},
      new Object[]{"leftrightarrowcircle", Integer.valueOf(0x2948)},
      new Object[]{"twoheaduparrowcircle", Integer.valueOf(0x2949)},
      new Object[]{"leftrightharpoonupdown", Integer.valueOf(0x294A)},
      new Object[]{"leftrightharpoondownup", Integer.valueOf(0x294B)},
      new Object[]{"updownharpoonrightleft", Integer.valueOf(0x294C)},
      new Object[]{"updownharpoonleftright", Integer.valueOf(0x294D)},
      new Object[]{"leftrightharpoonupup", Integer.valueOf(0x294E)},
      new Object[]{"updownharpoonrightright", Integer.valueOf(0x294F)},
      new Object[]{"leftrightharpoondowndown", Integer.valueOf(0x2950)},
      new Object[]{"updownharpoonleftleft", Integer.valueOf(0x2951)},
      new Object[]{"barleftharpoonup", Integer.valueOf(0x2952)},
      new Object[]{"rightharpoonupbar", Integer.valueOf(0x2953)},
      new Object[]{"barupharpoonright", Integer.valueOf(0x2954)},
      new Object[]{"downharpoonrightbar", Integer.valueOf(0x2955)},
      new Object[]{"barleftharpoondown", Integer.valueOf(0x2956)},
      new Object[]{"rightharpoondownbar", Integer.valueOf(0x2957)},
      new Object[]{"barupharpoonleft", Integer.valueOf(0x2958)},
      new Object[]{"downharpoonleftbar", Integer.valueOf(0x2959)},
      new Object[]{"leftharpoonupbar", Integer.valueOf(0x295A)},
      new Object[]{"barrightharpoonup", Integer.valueOf(0x295B)},
      new Object[]{"upharpoonrightbar", Integer.valueOf(0x295C)},
      new Object[]{"bardownharpoonright", Integer.valueOf(0x295D)},
      new Object[]{"leftharpoondownbar", Integer.valueOf(0x295E)},
      new Object[]{"barrightharpoondown", Integer.valueOf(0x295F)},
      new Object[]{"upharpoonleftbar", Integer.valueOf(0x2960)},
      new Object[]{"bardownharpoonleft", Integer.valueOf(0x2961)},
      new Object[]{"leftharpoonsupdown", Integer.valueOf(0x2962)},
      new Object[]{"upharpoonsleftright", Integer.valueOf(0x2963)},
      new Object[]{"rightharpoonsupdown", Integer.valueOf(0x2964)},
      new Object[]{"downharpoonsleftright", Integer.valueOf(0x2965)},
      new Object[]{"leftrightharpoonsup", Integer.valueOf(0x2966)},
      new Object[]{"leftrightharpoonsdown", Integer.valueOf(0x2967)},
      new Object[]{"rightleftharpoonsup", Integer.valueOf(0x2968)},
      new Object[]{"rightleftharpoonsdown", Integer.valueOf(0x2969)},
      new Object[]{"leftharpoonupdash", Integer.valueOf(0x296A)},
      new Object[]{"dashleftharpoondown", Integer.valueOf(0x296B)},
      new Object[]{"rightharpoonupdash", Integer.valueOf(0x296C)},
      new Object[]{"dashrightharpoondown", Integer.valueOf(0x296D)},
      new Object[]{"updownharpoonsleftright", Integer.valueOf(0x296E)},
      new Object[]{"downupharpoonsleftright", Integer.valueOf(0x296F)},
      new Object[]{"rightimply", Integer.valueOf(0x2970)},
      new Object[]{"equalrightarrow", Integer.valueOf(0x2971)},
      new Object[]{"similarrightarrow", Integer.valueOf(0x2972)},
      new Object[]{"leftarrowsimilar", Integer.valueOf(0x2973)},
      new Object[]{"rightarrowsimilar", Integer.valueOf(0x2974)},
      new Object[]{"rightarrowapprox", Integer.valueOf(0x2975)},
      new Object[]{"ltlarr", Integer.valueOf(0x2976)},
      new Object[]{"leftarrowless", Integer.valueOf(0x2977)},
      new Object[]{"gtrarr", Integer.valueOf(0x2978)},
      new Object[]{"subrarr", Integer.valueOf(0x2979)},
      new Object[]{"leftarrowsubset", Integer.valueOf(0x297A)},
      new Object[]{"suplarr", Integer.valueOf(0x297B)},
      new Object[]{"leftfishtail", Integer.valueOf(0x297C)},
      new Object[]{"rightfishtail", Integer.valueOf(0x297D)},
      new Object[]{"upfishtail", Integer.valueOf(0x297E)},
      new Object[]{"downfishtail", Integer.valueOf(0x297F)},
      new Object[]{"rtriltri", Integer.valueOf(0x29CE)},
      new Object[]{"ltrivb", Integer.valueOf(0x29CF)},
      new Object[]{"vbrtri", Integer.valueOf(0x29D0)},
      new Object[]{"lfbowtie", Integer.valueOf(0x29D1)},
      new Object[]{"rfbowtie", Integer.valueOf(0x29D2)},
      new Object[]{"fbowtie", Integer.valueOf(0x29D3)},
      new Object[]{"lftimes", Integer.valueOf(0x29D4)},
      new Object[]{"rftimes", Integer.valueOf(0x29D5)},
      new Object[]{"dualmap", Integer.valueOf(0x29DF)},
      new Object[]{"lrtriangleeq", Integer.valueOf(0x29E1)},
      new Object[]{"eparsl", Integer.valueOf(0x29E3)},
      new Object[]{"smeparsl", Integer.valueOf(0x29E4)},
      new Object[]{"eqvparsl", Integer.valueOf(0x29E5)},
      new Object[]{"gleichstark", Integer.valueOf(0x29E6)},
      new Object[]{"ruledelayed", Integer.valueOf(0x29F4)},
      new Object[]{"veeonwedge", Integer.valueOf(0x2A59)},
      new Object[]{"eqdot", Integer.valueOf(0x2A66)},
      new Object[]{"dotequiv", Integer.valueOf(0x2A67)},
      new Object[]{"equivVert", Integer.valueOf(0x2A68)},
      new Object[]{"equivVvert", Integer.valueOf(0x2A69)},
      new Object[]{"dotsim", Integer.valueOf(0x2A6A)},
      new Object[]{"simrdots", Integer.valueOf(0x2A6B)},
      new Object[]{"simminussim", Integer.valueOf(0x2A6C)},
      new Object[]{"congdot", Integer.valueOf(0x2A6D)},
      new Object[]{"asteq", Integer.valueOf(0x2A6E)},
      new Object[]{"hatapprox", Integer.valueOf(0x2A6F)},
      new Object[]{"approxeqq", Integer.valueOf(0x2A70)},
      new Object[]{"eqqsim", Integer.valueOf(0x2A73)},
      new Object[]{"Coloneq", Integer.valueOf(0x2A74)},
      new Object[]{"eqeq", Integer.valueOf(0x2A75)},
      new Object[]{"eqeqeq", Integer.valueOf(0x2A76)},
      new Object[]{"ddotseq", Integer.valueOf(0x2A77)},
      new Object[]{"equivDD", Integer.valueOf(0x2A78)},
      new Object[]{"ltcir", Integer.valueOf(0x2A79)},
      new Object[]{"gtcir", Integer.valueOf(0x2A7A)},
      new Object[]{"ltquest", Integer.valueOf(0x2A7B)},
      new Object[]{"gtquest", Integer.valueOf(0x2A7C)},
      new Object[]{"leqslant", Integer.valueOf(0x2A7D)},
      new Object[]{"geqslant", Integer.valueOf(0x2A7E)},
      new Object[]{"lesdot", Integer.valueOf(0x2A7F)},
      new Object[]{"gesdot", Integer.valueOf(0x2A80)},
      new Object[]{"lesdoto", Integer.valueOf(0x2A81)},
      new Object[]{"gesdoto", Integer.valueOf(0x2A82)},
      new Object[]{"lesdotor", Integer.valueOf(0x2A83)},
      new Object[]{"gesdotol", Integer.valueOf(0x2A84)},
      new Object[]{"lessapprox", Integer.valueOf(0x2A85)},
      new Object[]{"gtrapprox", Integer.valueOf(0x2A86)},
      new Object[]{"lneq", Integer.valueOf(0x2A87)},
      new Object[]{"gneq", Integer.valueOf(0x2A88)},
      new Object[]{"lnapprox", Integer.valueOf(0x2A89)},
      new Object[]{"gnapprox", Integer.valueOf(0x2A8A)},
      new Object[]{"lesseqqgtr", Integer.valueOf(0x2A8B)},
      new Object[]{"gtreqqless", Integer.valueOf(0x2A8C)},
      new Object[]{"lsime", Integer.valueOf(0x2A8D)},
      new Object[]{"gsime", Integer.valueOf(0x2A8E)},
      new Object[]{"lsimg", Integer.valueOf(0x2A8F)},
      new Object[]{"gsiml", Integer.valueOf(0x2A90)},
      new Object[]{"lgE", Integer.valueOf(0x2A91)},
      new Object[]{"glE", Integer.valueOf(0x2A92)},
      new Object[]{"lesges", Integer.valueOf(0x2A93)},
      new Object[]{"gesles", Integer.valueOf(0x2A94)},
      new Object[]{"eqslantless", Integer.valueOf(0x2A95)},
      new Object[]{"eqslantgtr", Integer.valueOf(0x2A96)},
      new Object[]{"elsdot", Integer.valueOf(0x2A97)},
      new Object[]{"egsdot", Integer.valueOf(0x2A98)},
      new Object[]{"eqqless", Integer.valueOf(0x2A99)},
      new Object[]{"eqqgtr", Integer.valueOf(0x2A9A)},
      new Object[]{"eqqslantless", Integer.valueOf(0x2A9B)},
      new Object[]{"eqqslantgtr", Integer.valueOf(0x2A9C)},
      new Object[]{"simless", Integer.valueOf(0x2A9D)},
      new Object[]{"simgtr", Integer.valueOf(0x2A9E)},
      new Object[]{"simlE", Integer.valueOf(0x2A9F)},
      new Object[]{"simgE", Integer.valueOf(0x2AA0)},
      new Object[]{"Lt", Integer.valueOf(0x2AA1)},
      new Object[]{"Gt", Integer.valueOf(0x2AA2)},
      new Object[]{"partialmeetcontraction", Integer.valueOf(0x2AA3)},
      new Object[]{"glj", Integer.valueOf(0x2AA4)},
      new Object[]{"gla", Integer.valueOf(0x2AA5)},
      new Object[]{"ltcc", Integer.valueOf(0x2AA6)},
      new Object[]{"gtcc", Integer.valueOf(0x2AA7)},
      new Object[]{"lescc", Integer.valueOf(0x2AA8)},
      new Object[]{"gescc", Integer.valueOf(0x2AA9)},
      new Object[]{"smt", Integer.valueOf(0x2AAA)},
      new Object[]{"lat", Integer.valueOf(0x2AAB)},
      new Object[]{"smte", Integer.valueOf(0x2AAC)},
      new Object[]{"late", Integer.valueOf(0x2AAD)},
      new Object[]{"bumpeqq", Integer.valueOf(0x2AAE)},
      new Object[]{"preceq", Integer.valueOf(0x2AAF)},
      new Object[]{"succeq", Integer.valueOf(0x2AB0)},
      new Object[]{"precneq", Integer.valueOf(0x2AB1)},
      new Object[]{"succneq", Integer.valueOf(0x2AB2)},
      new Object[]{"preceqq", Integer.valueOf(0x2AB3)},
      new Object[]{"succeqq", Integer.valueOf(0x2AB4)},
      new Object[]{"precneqq", Integer.valueOf(0x2AB5)},
      new Object[]{"succneqq", Integer.valueOf(0x2AB6)},
      new Object[]{"precapprox", Integer.valueOf(0x2AB7)},
      new Object[]{"succapprox", Integer.valueOf(0x2AB8)},
      new Object[]{"precnapprox", Integer.valueOf(0x2AB9)},
      new Object[]{"succnapprox", Integer.valueOf(0x2ABA)},
      new Object[]{"Prec", Integer.valueOf(0x2ABB)},
      new Object[]{"Succ", Integer.valueOf(0x2ABC)},
      new Object[]{"subsetdot", Integer.valueOf(0x2ABD)},
      new Object[]{"supsetdot", Integer.valueOf(0x2ABE)},
      new Object[]{"subsetplus", Integer.valueOf(0x2ABF)},
      new Object[]{"supsetplus", Integer.valueOf(0x2AC0)},
      new Object[]{"submult", Integer.valueOf(0x2AC1)},
      new Object[]{"supmult", Integer.valueOf(0x2AC2)},
      new Object[]{"subedot", Integer.valueOf(0x2AC3)},
      new Object[]{"supedot", Integer.valueOf(0x2AC4)},
      new Object[]{"subseteqq", Integer.valueOf(0x2AC5)},
      new Object[]{"supseteqq", Integer.valueOf(0x2AC6)},
      new Object[]{"subsim", Integer.valueOf(0x2AC7)},
      new Object[]{"supsim", Integer.valueOf(0x2AC8)},
      new Object[]{"subsetapprox", Integer.valueOf(0x2AC9)},
      new Object[]{"supsetapprox", Integer.valueOf(0x2ACA)},
      new Object[]{"subsetneqq", Integer.valueOf(0x2ACB)},
      new Object[]{"varsubsetneqq", Integer.valueOf(0x2ACB)},
      new Object[]{"supsetneqq", Integer.valueOf(0x2ACC)},
      new Object[]{"varsupsetneqq", Integer.valueOf(0x2ACC)},
      new Object[]{"lsqhook", Integer.valueOf(0x2ACD)},
      new Object[]{"rsqhook", Integer.valueOf(0x2ACE)},
      new Object[]{"csub", Integer.valueOf(0x2ACF)},
      new Object[]{"csup", Integer.valueOf(0x2AD0)},
      new Object[]{"csube", Integer.valueOf(0x2AD1)},
      new Object[]{"csupe", Integer.valueOf(0x2AD2)},
      new Object[]{"subsup", Integer.valueOf(0x2AD3)},
      new Object[]{"supsub", Integer.valueOf(0x2AD4)},
      new Object[]{"subsub", Integer.valueOf(0x2AD5)},
      new Object[]{"supsup", Integer.valueOf(0x2AD6)},
      new Object[]{"suphsub", Integer.valueOf(0x2AD7)},
      new Object[]{"supdsub", Integer.valueOf(0x2AD8)},
      new Object[]{"forkv", Integer.valueOf(0x2AD9)},
      new Object[]{"topfork", Integer.valueOf(0x2ADA)},
      new Object[]{"mlcp", Integer.valueOf(0x2ADB)},
      new Object[]{"forks", Integer.valueOf(0x2ADC)},
      new Object[]{"forksnot", Integer.valueOf(0x2ADD)},
      new Object[]{"shortlefttack", Integer.valueOf(0x2ADE)},
      new Object[]{"shortdowntack", Integer.valueOf(0x2ADF)},
      new Object[]{"shortuptack", Integer.valueOf(0x2AE0)},
      new Object[]{"vDdash", Integer.valueOf(0x2AE2)},
      new Object[]{"dashV", Integer.valueOf(0x2AE3)},
      new Object[]{"Dashv", Integer.valueOf(0x2AE4)},
      new Object[]{"DashV", Integer.valueOf(0x2AE5)},
      new Object[]{"varVdash", Integer.valueOf(0x2AE6)},
      new Object[]{"Barv", Integer.valueOf(0x2AE7)},
      new Object[]{"vBar", Integer.valueOf(0x2AE8)},
      new Object[]{"vBarv", Integer.valueOf(0x2AE9)},
      new Object[]{"barV", Integer.valueOf(0x2AEA)},
      new Object[]{"Vbar", Integer.valueOf(0x2AEB)},
      new Object[]{"Not", Integer.valueOf(0x2AEC)},
      new Object[]{"bNot", Integer.valueOf(0x2AED)},
      new Object[]{"revnmid", Integer.valueOf(0x2AEE)},
      new Object[]{"cirmid", Integer.valueOf(0x2AEF)},
      new Object[]{"midcir", Integer.valueOf(0x2AF0)},
      new Object[]{"nhpar", Integer.valueOf(0x2AF2)},
      new Object[]{"parsim", Integer.valueOf(0x2AF3)},
      new Object[]{"lllnest", Integer.valueOf(0x2AF7)},
      new Object[]{"gggnest", Integer.valueOf(0x2AF8)},
      new Object[]{"leqqslant", Integer.valueOf(0x2AF9)},
      new Object[]{"geqqslant", Integer.valueOf(0x2AFA)},
      new Object[]{"circleonleftarrow", Integer.valueOf(0x2B30)},
      new Object[]{"leftthreearrows", Integer.valueOf(0x2B31)},
      new Object[]{"leftarrowonoplus", Integer.valueOf(0x2B32)},
      new Object[]{"longleftsquigarrow", Integer.valueOf(0x2B33)},
      new Object[]{"nvtwoheadleftarrow", Integer.valueOf(0x2B34)},
      new Object[]{"nVtwoheadleftarrow", Integer.valueOf(0x2B35)},
      new Object[]{"twoheadmapsfrom", Integer.valueOf(0x2B36)},
      new Object[]{"twoheadleftdbkarrow", Integer.valueOf(0x2B37)},
      new Object[]{"leftdotarrow", Integer.valueOf(0x2B38)},
      new Object[]{"nvleftarrowtail", Integer.valueOf(0x2B39)},
      new Object[]{"nVleftarrowtail", Integer.valueOf(0x2B3A)},
      new Object[]{"twoheadleftarrowtail", Integer.valueOf(0x2B3B)},
      new Object[]{"nvtwoheadleftarrowtail", Integer.valueOf(0x2B3C)},
      new Object[]{"nVtwoheadleftarrowtail", Integer.valueOf(0x2B3D)},
      new Object[]{"leftarrowx", Integer.valueOf(0x2B3E)},
      new Object[]{"leftcurvedarrow", Integer.valueOf(0x2B3F)},
      new Object[]{"equalleftarrow", Integer.valueOf(0x2B40)},
      new Object[]{"bsimilarleftarrow", Integer.valueOf(0x2B41)},
      new Object[]{"leftarrowbackapprox", Integer.valueOf(0x2B42)},
      new Object[]{"rightarrowgtr", Integer.valueOf(0x2B43)},
      new Object[]{"rightarrowsupset", Integer.valueOf(0x2B44)},
      new Object[]{"LLeftarrow", Integer.valueOf(0x2B45)},
      new Object[]{"RRightarrow", Integer.valueOf(0x2B46)},
      new Object[]{"bsimilarrightarrow", Integer.valueOf(0x2B47)},
      new Object[]{"rightarrowbackapprox", Integer.valueOf(0x2B48)},
      new Object[]{"similarleftarrow", Integer.valueOf(0x2B49)},
      new Object[]{"leftarrowapprox", Integer.valueOf(0x2B4A)},
      new Object[]{"leftarrowbsimilar", Integer.valueOf(0x2B4B)},
      new Object[]{"rightarrowbsimilar", Integer.valueOf(0x2B4C)},
   };

   private static final Object[][] PUNCTUATION = new Object[][]
   {
      // punctuation
      new Object[]{"ldotp", Integer.valueOf(0x002E)},
      new Object[]{"colon", Integer.valueOf(0x003A)},
   };

   private static final Object[][] SMALL_INTEGRALS = new Object[][]
   {
      // integrals
      new Object[]{"smallint", Integer.valueOf(0x222B)},
      new Object[]{"smalliint", Integer.valueOf(0x222C)},
      new Object[]{"smalliiint", Integer.valueOf(0x222D)},
      new Object[]{"smalloint", Integer.valueOf(0x222E)},
      new Object[]{"smalloiint", Integer.valueOf(0x222F)},
      new Object[]{"smalloiiint", Integer.valueOf(0x2230)},
      new Object[]{"smallintclockwise", Integer.valueOf(0x2231)},
      new Object[]{"smallvarointclockwise", Integer.valueOf(0x2232)},
      new Object[]{"smallointctrclockwise", Integer.valueOf(0x2233)},
      new Object[]{"smallsumint", Integer.valueOf(0x2A0B)},
      new Object[]{"smalliiiint", Integer.valueOf(0x2A0C)},
      new Object[]{"smallintbar", Integer.valueOf(0x2A0D)},
      new Object[]{"smallintBar", Integer.valueOf(0x2A0E)},
      new Object[]{"smallfint", Integer.valueOf(0x2A0F)},
      new Object[]{"smallcirfnint", Integer.valueOf(0x2A10)},
      new Object[]{"smallawint", Integer.valueOf(0x2A11)},
      new Object[]{"smallrppolint", Integer.valueOf(0x2A12)},
      new Object[]{"smallscpolint", Integer.valueOf(0x2A13)},
      new Object[]{"smallnpolint", Integer.valueOf(0x2A14)},
      new Object[]{"smallpointint", Integer.valueOf(0x2A15)},
      new Object[]{"smallsqint", Integer.valueOf(0x2A16)},
      new Object[]{"smallintlarhk", Integer.valueOf(0x2A17)},
      new Object[]{"smallintx", Integer.valueOf(0x2A18)},
      new Object[]{"smallintcap", Integer.valueOf(0x2A19)},
      new Object[]{"smallintcup", Integer.valueOf(0x2A1A)},
      new Object[]{"smallupint", Integer.valueOf(0x2A1B)},
      new Object[]{"smalllowint", Integer.valueOf(0x2A1C)},
   };

   private static final Object[][] INTEGRALS = new Object[][]
   {
      new Object[]{"int", Integer.valueOf(0x222B)},
      new Object[]{"iint", Integer.valueOf(0x222C)},
      new Object[]{"iiint", Integer.valueOf(0x222D)},
      new Object[]{"oint", Integer.valueOf(0x222E)},
      new Object[]{"oiint", Integer.valueOf(0x222F)},
      new Object[]{"oiiint", Integer.valueOf(0x2230)},
      new Object[]{"intclockwise", Integer.valueOf(0x2231)},
      new Object[]{"varointclockwise", Integer.valueOf(0x2232)},
      new Object[]{"ointctrclockwise", Integer.valueOf(0x2233)},
      new Object[]{"sumint", Integer.valueOf(0x2A0B)},
      new Object[]{"iiiint", Integer.valueOf(0x2A0C)},
      new Object[]{"intbar", Integer.valueOf(0x2A0D)},
      new Object[]{"intBar", Integer.valueOf(0x2A0E)},
      new Object[]{"fint", Integer.valueOf(0x2A0F)},
      new Object[]{"cirfnint", Integer.valueOf(0x2A10)},
      new Object[]{"awint", Integer.valueOf(0x2A11)},
      new Object[]{"rppolint", Integer.valueOf(0x2A12)},
      new Object[]{"scpolint", Integer.valueOf(0x2A13)},
      new Object[]{"npolint", Integer.valueOf(0x2A14)},
      new Object[]{"pointint", Integer.valueOf(0x2A15)},
      new Object[]{"sqint", Integer.valueOf(0x2A16)},
      new Object[]{"intlarhk", Integer.valueOf(0x2A17)},
      new Object[]{"intx", Integer.valueOf(0x2A18)},
      new Object[]{"intcap", Integer.valueOf(0x2A19)},
      new Object[]{"intcup", Integer.valueOf(0x2A1A)},
      new Object[]{"upint", Integer.valueOf(0x2A1B)},
      new Object[]{"lowint", Integer.valueOf(0x2A1C)},
   };

   private static final Object[][] BIG_OPERATORS = new Object[][]
   {
      // big operators
      new Object[]{"Bbbsum", Integer.valueOf(0x2140)},
      new Object[]{"prod", Integer.valueOf(0x220F)},
      new Object[]{"coprod", Integer.valueOf(0x2210)},
      new Object[]{"sum", Integer.valueOf(0x2211)},
      new Object[]{"bigwedge", Integer.valueOf(0x22C0)},
      new Object[]{"bigvee", Integer.valueOf(0x22C1)},
      new Object[]{"bigcap", Integer.valueOf(0x22C2)},
      new Object[]{"bigcup", Integer.valueOf(0x22C3)},
      new Object[]{"leftouterjoin", Integer.valueOf(0x27D5)},
      new Object[]{"rightouterjoin", Integer.valueOf(0x27D6)},
      new Object[]{"fullouterjoin", Integer.valueOf(0x27D7)},
      new Object[]{"bigbot", Integer.valueOf(0x27D8)},
      new Object[]{"bigtop", Integer.valueOf(0x27D9)},
      new Object[]{"xsol", Integer.valueOf(0x29F8)},
      new Object[]{"xbsol", Integer.valueOf(0x29F9)},
      new Object[]{"bigodot", Integer.valueOf(0x2A00)},
      new Object[]{"bigoplus", Integer.valueOf(0x2A01)},
      new Object[]{"bigotimes", Integer.valueOf(0x2A02)},
      new Object[]{"bigcupdot", Integer.valueOf(0x2A03)},
      new Object[]{"biguplus", Integer.valueOf(0x2A04)},
      new Object[]{"bigsqcap", Integer.valueOf(0x2A05)},
      new Object[]{"bigsqcup", Integer.valueOf(0x2A06)},
      new Object[]{"conjquant", Integer.valueOf(0x2A07)},
      new Object[]{"disjquant", Integer.valueOf(0x2A08)},
      new Object[]{"bigtimes", Integer.valueOf(0x2A09)},
      new Object[]{"modtwosum", Integer.valueOf(0x2A0A)},
      new Object[]{"Join", Integer.valueOf(0x2A1D)},
      new Object[]{"bigtriangleleft", Integer.valueOf(0x2A1E)},
      new Object[]{"zcmp", Integer.valueOf(0x2A1F)},
      new Object[]{"zpipe", Integer.valueOf(0x2A20)},
      new Object[]{"zproject", Integer.valueOf(0x2A21)},
      new Object[]{"biginterleave", Integer.valueOf(0x2AFC)},
      new Object[]{"bigtalloblong", Integer.valueOf(0x2AFF)},
   };

   private static final Object[][] DELIMITERS = new Object[][]
   {
      // delimiters
      new Object[]{"lbrace", Integer.valueOf(0x007B)},
      new Object[]{"backslash", Integer.valueOf(0x005C)},
      new Object[]{"rbrace", Integer.valueOf(0x007D)},
      new Object[]{"lceil", Integer.valueOf(0x2308)},
      new Object[]{"lfloor", Integer.valueOf(0x230A)},
      new Object[]{"lmoustache", Integer.valueOf(0x23B0)},
      new Object[]{"lbrbrak", Integer.valueOf(0x2772)},
      new Object[]{"lBrack", Integer.valueOf(0x27E6)},
      new Object[]{"langle", Integer.valueOf(0x27E8)},
      new Object[]{"lAngle", Integer.valueOf(0x27EA)},
      new Object[]{"lgroup", Integer.valueOf(0x27EE)},
      new Object[]{"lBrace", Integer.valueOf(0x2983)},
      new Object[]{"lParen", Integer.valueOf(0x2985)},
      new Object[]{"rceil", Integer.valueOf(0x2309)},
      new Object[]{"rfloor", Integer.valueOf(0x230B)},
      new Object[]{"rmoustache", Integer.valueOf(0x23B1)},
      new Object[]{"rbrbrak", Integer.valueOf(0x2773)},
      new Object[]{"rBrack", Integer.valueOf(0x27E7)},
      new Object[]{"rangle", Integer.valueOf(0x27E9)},
      new Object[]{"rAngle", Integer.valueOf(0x27EB)},
      new Object[]{"rgroup", Integer.valueOf(0x27EF)},
      new Object[]{"rBrace", Integer.valueOf(0x2984)},
      new Object[]{"rParen", Integer.valueOf(0x2986)},
      new Object[]{"vert", Integer.valueOf(0x007C)},
      new Object[]{"Vert", Integer.valueOf(0x2016)},
      new Object[]{"|", Integer.valueOf(0x2016)},
      new Object[]{"Vvert", Integer.valueOf(0x2980)},
      new Object[]{"uparrow", Integer.valueOf(0x2191)},
      new Object[]{"downarrow", Integer.valueOf(0x2193)},
      new Object[]{"updownarrow", Integer.valueOf(0x2195)},
      new Object[]{"Uparrow", Integer.valueOf(0x21D1)},
      new Object[]{"Downarrow", Integer.valueOf(0x21D3)},
      new Object[]{"Updownarrow", Integer.valueOf(0x21D5)},
      new Object[]{"Uuparrow", Integer.valueOf(0x290A)},
      new Object[]{"Ddownarrow", Integer.valueOf(0x290B)},
      new Object[]{"UUparrow", Integer.valueOf(0x27F0)},
      new Object[]{"DDownarrow", Integer.valueOf(0x27F1)},
   };

   private static final Object[][] OTHER_BRACES = new Object[][]
   {
      new Object[]{"ulcorner", Integer.valueOf(0x231C)},
      new Object[]{"urcorner", Integer.valueOf(0x231D)},
      new Object[]{"llcorner", Integer.valueOf(0x231E)},
      new Object[]{"lrcorner", Integer.valueOf(0x231F)},
      new Object[]{"Lbrbrak", Integer.valueOf(0x27EC)},
      new Object[]{"Rbrbrak", Integer.valueOf(0x27ED)},
      new Object[]{"llparenthesis", Integer.valueOf(0x2987)},
      new Object[]{"rrparenthesis", Integer.valueOf(0x2988)},
      new Object[]{"llangle", Integer.valueOf(0x2989)},
      new Object[]{"rrangle", Integer.valueOf(0x298A)},
      new Object[]{"lbrackubar", Integer.valueOf(0x298B)},
      new Object[]{"rbrackubar", Integer.valueOf(0x298C)},
      new Object[]{"lbrackultick", Integer.valueOf(0x298D)},
      new Object[]{"rbracklrtick", Integer.valueOf(0x298E)},
      new Object[]{"lbracklltick", Integer.valueOf(0x298F)},
      new Object[]{"rbrackurtick", Integer.valueOf(0x2990)},
      new Object[]{"langledot", Integer.valueOf(0x2991)},
      new Object[]{"rangledot", Integer.valueOf(0x2992)},
      new Object[]{"lparenless", Integer.valueOf(0x2993)},
      new Object[]{"rparengtr", Integer.valueOf(0x2994)},
      new Object[]{"Lparengtr", Integer.valueOf(0x2995)},
      new Object[]{"Rparenless", Integer.valueOf(0x2996)},
      new Object[]{"lblkbrbrak", Integer.valueOf(0x2997)},
      new Object[]{"rblkbrbrak", Integer.valueOf(0x2998)},
      new Object[]{"lvzigzag", Integer.valueOf(0x29D8)},
      new Object[]{"rvzigzag", Integer.valueOf(0x29D9)},
      new Object[]{"Lvzigzag", Integer.valueOf(0x29DA)},
      new Object[]{"Rvzigzag", Integer.valueOf(0x29DB)},
      new Object[]{"lcurvyangle", Integer.valueOf(0x29FC)},
      new Object[]{"rcurvyangle", Integer.valueOf(0x29FD)},
      new Object[]{"lbrbrak", Integer.valueOf(0x2772)},
      new Object[]{"rbrbrak", Integer.valueOf(0x2773)},
      new Object[]{"lbag", Integer.valueOf(0x27C5)},
      new Object[]{"rbag", Integer.valueOf(0x27C6)},
      new Object[]{"Lbrbrak", Integer.valueOf(0x27EC)},
      new Object[]{"Rbrbrak", Integer.valueOf(0x27ED)},
   };

   private static final Object[][] ACCENTS = new Object[][]
   {
      new Object[]{"grave", Integer.valueOf(0x0300)},
      new Object[]{"acute", Integer.valueOf(0x0301)},
      new Object[]{"hat", Integer.valueOf(0x0302)},
      new Object[]{"tilde", Integer.valueOf(0x0303)},
      new Object[]{"bar", Integer.valueOf(0x0304)},
      new Object[]{"breve", Integer.valueOf(0x0306)},
      new Object[]{"dot", Integer.valueOf(0x0307)},
      new Object[]{"ddot", Integer.valueOf(0x0308)},
      new Object[]{"ovhook", Integer.valueOf(0x0309)},
      new Object[]{"mathring", Integer.valueOf(0x030A)},
      new Object[]{"check", Integer.valueOf(0x030C)},
      new Object[]{"candra", Integer.valueOf(0x0310)},
      new Object[]{"oturnedcomma", Integer.valueOf(0x0312)},
      new Object[]{"ocommatopright", Integer.valueOf(0x0315)},
      new Object[]{"droang", Integer.valueOf(0x031A)},
      new Object[]{"leftharpoonaccent", Integer.valueOf(0x20D0)},
      new Object[]{"rightharpoonaccent", Integer.valueOf(0x20D1)},
      new Object[]{"leftarrowaccent", Integer.valueOf(0x20D6)},
      new Object[]{"vec", Integer.valueOf(0x20D7)},
      new Object[]{"rightarrowaccent", Integer.valueOf(0x20D7)},
      new Object[]{"leftrightarrowaccent", Integer.valueOf(0x20E1)},
      new Object[]{"dddot", Integer.valueOf(0x20DB)},
      new Object[]{"ddddot", Integer.valueOf(0x20DC)},
      new Object[]{"annuity", Integer.valueOf(0x20E7)},
      new Object[]{"widebridgeabove", Integer.valueOf(0x20E9)},
      new Object[]{"asteraccent", Integer.valueOf(0x20F0)},
   };

   private static final Object[][] WIDE_ACCENTS = new Object[][]
   {
      new Object[]{"widehat", Integer.valueOf(0x0302)},
      new Object[]{"widetilde", Integer.valueOf(0x0303)},
      new Object[]{"widecheck", Integer.valueOf(0x030C)},
      new Object[]{"overleftarrow", Integer.valueOf(0x20D6)},
      new Object[]{"overrightarrow", Integer.valueOf(0x20D7)},
      new Object[]{"underrightarrow", Integer.valueOf(0x20EF)},
      new Object[]{"underleftarrow", Integer.valueOf(0x20EE)},
      new Object[]{"overleftrightarrow", Integer.valueOf(0x20E1)},
      new Object[]{"underleftrightarrow", Integer.valueOf(0x034D)},
      new Object[]{"overleftharpoon", Integer.valueOf(0x20D0)},
      new Object[]{"overrightharpoon", Integer.valueOf(0x20D1)},
      new Object[]{"underleftharpoon", Integer.valueOf(0x20EC)},
      new Object[]{"underrightharpoon", Integer.valueOf(0x20ED)},
   };

   private static final Object[][] WIDE_BRACKETS = new Object[][]
   {
      new Object[]{"overbracket", Integer.valueOf(0x23B4)},
      new Object[]{"overparen", Integer.valueOf(0x23DC)},
      new Object[]{"overbrace", Integer.valueOf(0x23DE)},
      new Object[]{"underbracket", Integer.valueOf(0x23B5)},
      new Object[]{"underparen", Integer.valueOf(0x23DD)},
      new Object[]{"underbrace", Integer.valueOf(0x23DF)},
   };

   private static final Object[][] RADICALS = new Object[][]
   {
      new Object[]{"sqrt", Integer.valueOf(0x221A)},
      new Object[]{"longdivision", Integer.valueOf(0x27CC)},
   };
}
