% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sparseGFM.R
\name{sparseGFM}
\alias{sparseGFM}
\title{Sparse Generalized Factor Model with Multiple Penalty Functions}
\usage{
sparseGFM(
  x,
  type = c("continuous", "count", "binary"),
  q = 2,
  penalty = c("lasso", "SCAD", "MCP", "group", "agroup", "gSCAD", "agSCAD", "gMCP",
    "agMCP", "alasso", "glasso", "aglasso"),
  lambda = 1,
  gam = 1,
  tau = NULL,
  mat_sd = 1,
  delta = 1e-04,
  maxiter = 30,
  C = 5,
  verbose = TRUE
)
}
\arguments{
\item{x}{A numeric matrix of observations (n x p), where n is the number of observations
and p is the number of variables}

\item{type}{Character string specifying the data type. Options are:
\itemize{
  \item "continuous": Gaussian family for continuous data
  \item "count": Poisson family for count data
  \item "binary": Binomial family for binary data
}}

\item{q}{Integer specifying the number of latent factors (default = 2)}

\item{penalty}{Character string specifying the penalty type for sparsity. Available options:
\itemize{
  \item "lasso": L1 penalty (adaptive version: "alasso")
  \item "SCAD": Smoothly clipped absolute deviation penalty (adaptive: "agSCAD")
  \item "MCP": Minimax concave penalty (adaptive: "agMCP")
  \item "group"/"glasso": Group lasso penalty (adaptive: "agroup"/"aglasso")
  \item "gSCAD": Group SCAD penalty (adaptive: "agSCAD")
  \item "gMCP": Group MCP penalty (adaptive: "agMCP")
}}

\item{lambda}{Numeric value for the penalty tuning parameter (default = 1)}

\item{gam}{Numeric value for the adaptive weight parameter in adaptive penalties (default = 1)}

\item{tau}{Numeric value for the shape parameter in SCAD/MCP penalties.
Default is 3.7 for SCAD and 3 for MCP if not specified}

\item{mat_sd}{Standard deviation for continuous data (default = 1)}

\item{delta}{Convergence tolerance for the iterative algorithm (default = 1e-4)}

\item{maxiter}{Maximum number of iterations (default = 30)}

\item{C}{Numeric value for the constraint bound to ensure stability (default = 5)}

\item{verbose}{Logical indicating whether to print iteration progress (default = TRUE)}
}
\value{
A list containing:
\itemize{
  \item FF_hat: Estimated factor matrix (n x q)
  \item BB_hat: Estimated loading matrix (p x q)
  \item alpha_hat: Estimated intercept vector (p x 1)
  \item obj_loglik: Log-likelihood value
  \item obj_pen: Penalized objective function value
  \item index: Indices of variables with zero loadings (selected out)
  \item df_est: Estimated degrees of freedom
  \item iter: Number of iterations performed
}
}
\description{
Implements sparse generalized factor models with 12 different penalty functions for dimension reduction
and variable selection in high-dimensional data. The method is designed to handle row-sparse loading
structures and can adapt to weak factor scenarios where factors have relatively small eigenvalues.
Missing values are automatically handled through imputation.
}
\details{
The algorithm employs alternating minimization with the following steps:
1. Initialization using the GFM package for initial estimates
2. Iteratively updating factors (F) and loadings (B)
3. Applying penalty functions to achieve variable selection
4. Ensuring identifiability through SVD-based constraints
5. Monitoring convergence through objective function changes

The adaptive group lasso (aglasso) penalty is particularly effective for row-sparse
loading matrices as it can select entire rows (variables) rather than individual elements.
}
\examples{
# Generate data with sparse loading matrix
library(sparseGFM)
set.seed(123)
n <- 200; p <- 200; q <- 2
a_param <- 0.9
s <- ceiling(p^a_param)

# Generate factors and sparse loadings
FF <- matrix(runif(n * q, min = -3, max = 3), nrow = n, ncol = q)
BB <- rbind(matrix(runif(s * q, min = 1, max = 2), nrow = s, ncol = q),
            matrix(0, nrow = (p - s), ncol = q))
alpha_true <- runif(p, min = -1, max = 1)

# Apply identifiability constraints
ident_res <- add_identifiability(FF, BB, alpha_true)
FF0 <- ident_res$H; BB0 <- ident_res$B; alpha0 <- ident_res$mu

# Generate data matrix
mat_para <- FF0 \%*\% t(BB0) + as.matrix(rep(1, n)) \%*\% t(as.matrix(alpha0))
x <- matrix(nrow = n, ncol = p)
for (i in 1:n) {
  for (j in 1:p) {
    x[i, j] <- rnorm(1, mean = mat_para[i, j])
  }
}

# Fit sparse GFM with adaptive group lasso
result <- sparseGFM(x, type = "continuous", q = 2,
                    penalty = "aglasso", lambda = 0.1, C = 5)

# View results
print(paste("Selected variables:", length(setdiff(1:p, result$index))))


}
