% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/integration.R
\name{runINMF}
\alias{runINMF}
\alias{runINMF.liger}
\alias{runINMF.Seurat}
\title{Perform iNMF on scaled datasets}
\usage{
runINMF(object, k = 20, lambda = 5, ...)

\method{runINMF}{liger}(
  object,
  k = 20,
  lambda = 5,
  nIteration = 30,
  nRandomStarts = 1,
  HInit = NULL,
  WInit = NULL,
  VInit = NULL,
  seed = 1,
  nCores = 2L,
  verbose = getOption("ligerVerbose", TRUE),
  ...
)

\method{runINMF}{Seurat}(
  object,
  k = 20,
  lambda = 5,
  datasetVar = "orig.ident",
  layer = "ligerScaleData",
  assay = NULL,
  reduction = "inmf",
  nIteration = 30,
  nRandomStarts = 1,
  HInit = NULL,
  WInit = NULL,
  VInit = NULL,
  seed = 1,
  nCores = 2L,
  verbose = getOption("ligerVerbose", TRUE),
  ...
)
}
\arguments{
\item{object}{A \linkS4class{liger} object or a Seurat object with
non-negative scaled data of variable features (Done with
\code{\link{scaleNotCenter}}).}

\item{k}{Inner dimension of factorization (number of factors). Generally, a
higher \code{k} will be needed for datasets with more sub-structure. Default
\code{20}.}

\item{lambda}{Regularization parameter. Larger values penalize
dataset-specific effects more strongly (i.e. alignment should increase as
\code{lambda} increases). Default \code{5}.}

\item{...}{Arguments passed to methods.}

\item{nIteration}{Total number of block coordinate descent iterations to
perform. Default \code{30}.}

\item{nRandomStarts}{Number of restarts to perform (iNMF objective function
is non-convex, so taking the best objective from multiple successive
initialization is recommended). For easier reproducibility, this increments
the random seed by 1 for each consecutive restart, so future factorization
of the same dataset can be run with one rep if necessary. Default \code{1}.}

\item{HInit}{Initial values to use for \eqn{H} matrices. A list object where
each element is the initial \eqn{H} matrix of each dataset. Default
\code{NULL}.}

\item{WInit}{Initial values to use for \eqn{W} matrix. A matrix object.
Default \code{NULL}.}

\item{VInit}{Initial values to use for \eqn{V} matrices. A list object where
each element is the initial \eqn{V} matrix of each dataset. Default
\code{NULL}.}

\item{seed}{Random seed to allow reproducible results. Default \code{1}.}

\item{nCores}{The number of parallel tasks to speed up the computation.
Default \code{2L}. Only supported for platform with OpenMP support.}

\item{verbose}{Logical. Whether to show information of the progress. Default
\code{getOption("ligerVerbose")} or \code{TRUE} if users have not set.}

\item{datasetVar}{Metadata variable name that stores the dataset source
annotation. Default \code{"orig.ident"}.}

\item{layer}{For Seurat>=4.9.9, the name of layer to retrieve input
non-negative scaled data. Default \code{"ligerScaleData"}. For older Seurat,
always retrieve from \code{scale.data} slot.}

\item{assay}{Name of assay to use. Default \code{NULL} uses current active
assay.}

\item{reduction}{Name of the reduction to store result. Also used as the
feature key. Default \code{"inmf"}.}
}
\value{
\itemize{
\item{liger method - Returns updated input \linkS4class{liger} object
\itemize{
\item{A list of all \eqn{H} matrices can be accessed with
\code{getMatrix(object, "H")}}
\item{A list of all \eqn{V} matrices can be accessed with
\code{getMatrix(object, "V")}}
\item{The \eqn{W} matrix can be accessed with
\code{getMatrix(object, "W")}}
}}
\item{Seurat method - Returns updated input Seurat object
\itemize{
\item{\eqn{H} matrices for all datasets will be concatenated and
transposed (all cells by k), and form a DimReduc object in the
\code{reductions} slot named by argument \code{reduction}.}
\item{\eqn{W} matrix will be presented as \code{feature.loadings} in the
same DimReduc object.}
\item{\eqn{V} matrices, an objective error value and the dataset
variable used for the factorization is currently stored in
\code{misc} slot of the same DimReduc object.}
}}
}
}
\description{
Performs integrative non-negative matrix factorization (iNMF) (J.D. Welch,
2019) using block coordinate descent (alternating non-negative
least squares, ANLS) to return factorized \eqn{H}, \eqn{W}, and \eqn{V}
matrices. The objective function is stated as

\deqn{\arg\min_{H\ge0,W\ge0,V\ge0}\sum_{i}^{d}||E_i-(W+V_i)Hi||^2_F+
\lambda\sum_{i}^{d}||V_iH_i||_F^2}

where \eqn{E_i} is the input non-negative matrix of the i'th dataset, \eqn{d}
is the total number of datasets. \eqn{E_i} is of size \eqn{m \times n_i} for
\eqn{m} variable genes and \eqn{n_i} cells, \eqn{H_i} is of size
\eqn{n_i \times k}, \eqn{V_i} is of size \eqn{m \times k}, and \eqn{W} is of
size \eqn{m \times k}.

The factorization produces a shared \eqn{W} matrix (genes by k), and for each
dataset, an \eqn{H} matrix (k by cells) and a \eqn{V} matrix (genes by k).
The \eqn{H} matrices represent the cell factor loadings. \eqn{W} is held
consistent among all datasets, as it represents the shared components of the
metagenes across datasets. The \eqn{V} matrices represent the
dataset-specific components of the metagenes.

This function adopts highly optimized fast and memory efficient
implementation extended from Planc (Kannan, 2016). Pre-installation of
extension package \code{RcppPlanc} is required. The underlying algorithm
adopts the identical ANLS strategy as \code{\link{optimizeALS}} in the old
version of LIGER.
}
\section{Difference from optimizeALS()}{

In the old version implementation, we compute the objective error at the end
of each iteration, and then compares if the algorithm is reaching a
convergence, using an argument \code{thresh}. Now, since the computation of
objective error is indeed expensive, we canceled this feature and directly
runs a default of 30 (\code{nIteration}) iterations, which empirically leads
to a convergence most of the time. Given that the new version is highly
optimized, running this many iteration should be acceptable.
}

\examples{
pbmc <- normalize(pbmc)
pbmc <- selectGenes(pbmc)
pbmc <- scaleNotCenter(pbmc)
if (requireNamespace("RcppPlanc", quietly = TRUE)) {
    pbmc <- runINMF(pbmc)
}
}
\references{
Joshua D. Welch and et al., Single-Cell Multi-omic Integration
Compares and Contrasts Features of Brain Cell Identity, Cell, 2019
}
