% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/thr_twocorner.R
\name{thr_twocorner}
\alias{thr_twocorner}
\title{Compute two-corner thresholds}
\usage{
thr_twocorner(
  x,
  sigma = 2,
  slope_reduction = TRUE,
  method = "prominence",
  diagnose = FALSE,
  adjust_par = TRUE
)
}
\arguments{
\item{x}{numeric vector or a single-column \code{matrix} or \code{data.frame} able to
be coerced to numeric.}

\item{sigma}{numeric vector of length one. Standard deviation (DN) of the
Gaussian kernel used to smooth the histogram prior to peak detection and
Rosin’s construction.}

\item{slope_reduction}{logical vector of length one. If \code{TRUE}, apply
Macfarlane’s slope-reduction before Rosin’s construction on each side.}

\item{method}{character vector of length one. Peak detection strategy. One of
\code{"prominence"} (default) or \code{"macfarlane"}.}

\item{diagnose}{logical vector of length one. If \code{TRUE}, plot the geometric
construction.}

\item{adjust_par}{logical vector of length one. If \code{TRUE} and
\code{diagnose = TRUE}, temporarily adjust and then restore graphical
parameters.}
}
\value{
A list with:
\describe{
\item{lp}{Lower peak DN.}
\item{lc}{Lower corner DN (Rosin on the left mode).}
\item{tl}{Low threshold derived from \code{lc} and \code{uc}.}
\item{tm}{Mid threshold derived from \code{lc} and \code{uc}.}
\item{th}{High threshold derived from \code{lc} and \code{uc}.}
\item{uc}{Upper corner DN (Rosin on the right mode).}
\item{up}{Upper peak DN.}
}
}
\description{
Apply Rosin’s geometric corner detector for unimodal histograms
\insertCite{Rosin2001;textual}{rcaiman} to both sides of a bimodal canopy
histogram as in Macfarlane’s two-corner approach
\insertCite{Macfarlane2011;textual}{rcaiman}. Optional slope-reduction as in
Macfarlane is supported. Peak detection can use a prominence-based method or
Macfarlane’s original windowed maxima.
}
\details{
Runs the following pipeline:
\enumerate{
\item Build an 8-bit histogram of \code{x} after min–max normalization to
\verb{[0,255]}.
\item Smooth the histogram with a discrete Gaussian kernel of standard
deviation \code{sigma} (in DN), using reflective padding to mitigate edge
bias.
\item Detect the two mode peaks according to \code{method}:
\itemize{
\item \code{method = "prominence"}: find local maxima via discrete
derivatives with plateau handling; find nearest left/right minima; compute
peak prominence as \eqn{\min(y[p]-y[L],\,y[p]-y[R])}; filter by minimum
prominence and minimum peak separation; select the pair that maximizes
\eqn{\min(\mathrm{prom}_\mathrm{left},\,\mathrm{prom}_\mathrm{right})}.
\item \code{method = "macfarlane"}: search peaks within fixed DN windows as in
\insertCite{Macfarlane2011;textual}{rcaiman}. Peak search is performed on the
\strong{unsmoothed} histogram, as originally proposed.
}
\item Apply Rosin’s corner construction on each mode. The line end at the
“first empty bin after the last filled bin” is determined on the
\strong{unsmoothed} histogram \insertCite{Rosin2001;textual}{rcaiman}. If
\code{slope_reduction = TRUE} and the peak height exceeds the mean of the
smoothed histogram, the peak height is reduced to that mean before the
geometric construction (Macfarlane’s variant).
\item Derive thresholds:
\deqn{T_l = DN_{lc} + (DN_{uc} - DN_{lc}) \times 0.25}
\deqn{T_m = DN_{lc} + (DN_{uc} - DN_{lc}) \times 0.50}
\deqn{T_h = DN_{lc} + (DN_{uc} - DN_{lc}) \times 0.75}
where \eqn{DN_{lc}} and \eqn{DN_{uc}} are the lower and upper corners.
}

When \code{diagnose = TRUE}, a geometric construction like the one shown below is
sent to the active graphics device.
\if{html}{\figure{twocorner_method.png}{options: style="display:block;margin:0 auto;max-width:40\%;"}}
\if{latex}{\figure{twocorner_method.png}}
When \code{diagnose = TRUE} and \code{adjust_par = TRUE}, the function temporarily
adjusts margins to draw the geometric construction in a large square format
and restores the previous settings on exit. If \code{adjust_par = FALSE}, no
parameters are changed and the plot respects the current device/layout.
}
\examples{
caim <- conventional_lens_image()
# Prominence-based peak detection, Gaussian smoothing with sigma = 2 DN
thr <- thr_twocorner(caim$Blue[], sigma = 2, slope_reduction = FALSE,
                     method = "prominence")
# Original Macfarlane peak search (for comparison)
thr2 <- thr_twocorner(caim$Blue[], sigma = 2, slope_reduction = TRUE,
                      method = "macfarlane")
data.frame(unlist(thr), unlist(thr2))
}
\references{
\insertAllCited{}
}
