# This file is licensed under MIT by Zhengjia Wang

#' @name read-write-yaml
#' @title Read write \code{'YAML'} format
#' @description supports reading data into a map object, and write the map
#' to files with names sorted for consistency
#' @param file file to read from or write to
#' @param x list or map to write
#' @param ... passed to \code{\link{as.list}}
#' @param sorted whether to sort the list by name; default is false
#' @param map a \code{fastmap} object; can be generated by \code{fastmap}
#' or \code{dipsaus} package; default is to create a new map internally
#' @returns A map object
#' @examples
#'
#' tfile <- tempfile(fileext = ".yml")
#'
#' save_yaml(list(b = 2, a = 1), tfile, sorted = TRUE)
#'
#' cat(readLines(tfile), sep = "\n")
#'
#' load_yaml(tfile)
#'
#' unlink(tfile)
#'
#'
#' @export
load_yaml <- function(file, ..., map = NULL){
  re <- yaml::read_yaml(file = file, ...)
  if(!inherits(map, 'fastmap2')){
    if(inherits(map, 'fastmap')) {
      class(map) <- c("fastmap2", class(map))
    } else {
      map <- fastmap2()
    }
  }
  for(nm in names(re)){
    if(nm != ''){
      map[[nm]] <- re[[nm]]
    }
  }
  map
}

read_yaml <- function(file, ...) {
  yaml::read_yaml(file = file, ...)
}

#' @rdname read-write-yaml
#' @export
save_yaml <- function(x, file, ..., sorted = FALSE) {
  if (inherits(x, "fastmap")) {
    x <- x$as_list(sort = sorted)
  } else if (inherits(x, "fastmap2")) {
    x <- x[["@as_list"]](sort = sorted)
  } else if (inherits(x, c("fastqueue", "fastqueue2"))) {
    x <- x$as_list()
  } else if (sorted) {
    x <- as.list(x, sorted = sorted, ...)
    nms <- names(x)
    if(length(nms) && is.unsorted(nms)) {
      x <- x[order(nms)]
    }
  } else {
    x <- as.list(x, ...)
  }
  yaml::write_yaml(x, file = file, ...)
  if (!inherits(file, "connection")) {
    file <- normalizePath(file)
  }
  invisible(file)
}


