% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/diary_functions.R
\name{auto_by}
\alias{auto_by}
\title{Autoregressive Coefficient by Group}
\usage{
auto_by(
  x,
  grp,
  n = -1L,
  how = "cor",
  cw = TRUE,
  method = "pearson",
  use = "na.or.complete",
  REML = TRUE,
  control = NULL,
  sep = "."
)
}
\arguments{
\item{x}{numeric vector.}

\item{grp}{list of atomic vector(s) and/or factor(s) (e.g., data.frame),
which each have same length as \code{x}. It can also be an atomic vector or
factor, which will then be made the first element of a list.}

\item{n}{integer vector with length 1. Specifies the direction and magnitude
of the shift. See \code{shift} for details. The default is -1L, which is a
one-lag autoregressive coefficient' +2L would be a two-lead autoregressive
coefficient. The sign of \code{n} only affects the results for \code{how} =
"lm", "lme", or "lmer".}

\item{how}{character vector of length 1 specifying how to compute the
autoregressive coefficients. The options are 1) "cor" for correlation with
the \code{cor} function, 2) "cov" for covariance with the \code{cov}
function, 3) "lm" for the linear regression slope with the \code{lm}
function, 4) "lme" for empirical Bayes estimates from a linear mixed
effects model with the \code{\link[nlme]{lme}} function, 5) "lmer" for
empirical Bayes estimates from a linear mixed effects model with the
\code{\link[lme4]{lmer}} function.}

\item{cw}{logical vector of length 1 specifying whether the shifted vector
should be group-mean centered (TRUE) or not (FALSE). This only affects the
results for \code{how} = "lme" or "lmer".}

\item{method}{character vector of length 1 specifying the type of correlation
or covariance to compute. Only used when \code{how} = "cor" or "cov". See
\code{\link[stats]{cor}} for details.}

\item{use}{character vector of length 1 specifying how to handle missing
data. Only used when \code{how} = "cor" or "cov". See
\code{\link[stats]{cor}} for details.}

\item{REML}{logical vector of length 1 specifying whether to use restricted
estimated maximum liklihood (TRUE) rather than traditional maximum
likelihood (FALSE). Only used when \code{how} = "lme" or "lmer".}

\item{control}{list of control parameters for \code{lme} or \code{lmer} when
\code{how} = "lme" or "lmer", respectively. See
\code{\link[nlme]{lmeControl}} and \code{\link[lme4]{lmerControl}} for
details.}

\item{sep}{character vector of length 1 specifying what string should
separate different group values when naming the return object. This
argument is only used if \code{grp} is a list of atomic vectors (e.g.,
data.frame).}
}
\value{
numeric vector of autoregressive coefficients with length =
  \code{length(levels(interaction(grp)))} and names = pasteing of the
  grouping value(s) together separated by \code{sep}.
}
\description{
\code{auto_by} computes the autoregressive coefficient by group for
longitudinal data where each observation within the group represents a
different timepoint. The function assumes the data are already sorted by
time.
}
\details{
There are several different ways to estimate the autoregressive parameter.
This function offers a variety of ways with the \code{how} and \code{cw}
arguments. Note, that a recent simulation suggests that group-mean centering
via \code{cw} is the best approach when using linear mixed effects modeling
via \code{how} = "lme" or "lmer" (Hamaker  & Grasman, 2015).
}
\examples{

# cor
auto_by(x = airquality$"Ozone", grp = airquality$"Month", how = "cor")
auto_by(x = airquality$"Ozone", grp = airquality$"Month",
   n = -2L, how = "cor") # lag across 2 timepoints
auto_by(x = airquality$"Ozone", grp = airquality$"Month",
   n = +1L, how = "cor") # lag and lead identical for cor
auto_by(x = airquality$"Ozone", grp = airquality$"Month", how = "cor",
   cw = FALSE) # centering within-person identical for cor

# cov
auto_by(x = airquality$"Ozone", grp = airquality$"Month", how = "cov")
auto_by(x = airquality$"Ozone", grp = airquality$"Month",
   n = -2L, how = "cov") # lag across 2 timepoints
auto_by(x = airquality$"Ozone", grp = airquality$"Month",
   n = +1L, how = "cov") # lag and lead identical for cov
auto_by(x = airquality$"Ozone", grp = airquality$"Month", how = "cov",
   cw = FALSE) # centering within-person identical for cov

# lm
auto_by(x = airquality$"Ozone", grp = airquality$"Month", how = "lm")
auto_by(x = airquality$"Ozone", grp = airquality$"Month",
   n = -2L, how = "lm") # lag across 2 timepoints
auto_by(x = airquality$"Ozone", grp = airquality$"Month",
   n = +1L, how = "lm") # lag and lead NOT identical for lm
auto_by(x = airquality$"Ozone", grp = airquality$"Month", how = "lm",
   cw = FALSE) # centering within-person identical for lm

# lme
chick_weight <- as.data.frame(ChickWeight)
auto_by(x = chick_weight$"weight", grp = chick_weight$"Chick", how = "lme")
control_lme <- nlme::lmeControl(maxIter = 250L, msMaxIter = 250L,
   tolerance = 1e-3, msTol = 1e-3) # custom controls
auto_by(x = chick_weight$"weight", grp = chick_weight$"Chick", how = "lme",
   control = control_lme)
auto_by(x = chick_weight$"weight", grp = chick_weight$"Chick",
   n = -2L, how = "lme") # lag across 2 timepoints
auto_by(x = chick_weight$"weight", grp = chick_weight$"Chick",
   n = +1L, how = "lme") # lag and lead NOT identical for lme
auto_by(x = chick_weight$"weight", grp = chick_weight$"Chick", how = "lme",
   cw = FALSE) # centering within-person NOT identical for lme

# lmer
bryant_2016 <- as.data.frame(lmeInfo::Bryant2016)
\dontrun{
auto_by(x = bryant_2016$"outcome", grp = bryant_2016$"case", how = "lmer")
control_lmer <- lme4::lmerControl(check.conv.grad = lme4::.makeCC("stop",
   tol = 2e-3, relTol = NULL), check.conv.singular = lme4::.makeCC("stop",
   tol = formals(lme4::isSingular)$"tol"), check.conv.hess = lme4::.makeCC(action = "stop",
   tol = 1e-6)) # custom controls
auto_by(x = bryant_2016$"outcome", grp = bryant_2016$"case", how = "lmer",
   control = control_lmer) # TODO: for some reason lmer doesn't like this
   # and is not taking into account the custom controls
auto_by(x = bryant_2016$"outcome", grp = bryant_2016$"case",
   n = -2L, how = "lmer") # lag across 2 timepoints
auto_by(x = bryant_2016$"outcome", grp = bryant_2016$"case",
   n = +1L, how = "lmer") # lag and lead NOT identical for lmer
auto_by(x = bryant_2016$"outcome", grp = bryant_2016$"case", how = "lmer",
   cw = FALSE) # centering within-person NOT identical for lmer
}

}
\references{
Hamaker, E. L., & Grasman, R. P. (2015). To center or not to center?  Investigating
inertia with a multilevel autoregressive model. Frontiers in Psychology, 5, 1492.
}
