% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/refine.R
\name{Refine}
\alias{Refine}
\title{Refine or extend the result of PowerGrid}
\usage{
Refine(old, n_iter_add = 1, pars = NULL, ...)
}
\arguments{
\item{old}{the object of class \code{power_array} to extend}

\item{n_iter_add}{the number of iterations to \emph{add} to old}

\item{pars}{the new parameter grid to evaluate across}

\item{...}{further arguments passed on to PowerGrid internally.}
}
\value{
object of class \code{power_array}, containing old, extended by
\code{pars} and/or \code{n_iter_add}.
}
\description{
Add further results to an existing power_array (created by
PowerGrid or by another call of Refine), adding further values in
\code{pars} and/or larger \code{n_iter}.
}
\details{
If \code{pars == NULL}, update extends \code{old} by adding
iterations \code{n_iter_add} to the existing power_array. If \code{pars}
is given, the function that was evaluated in \code{old} (attribute
\code{sim_function}) is evaluated at the crossings of \code{pars}. If
argument \code{pars} is different from \code{attr(old, which = 'pars')},
this means that the function is evaluated additional crossings of
parameters.

Note that certain combinations of \code{pars} and \code{n_iter_add} result
in arrays where some crossings of parameters include more iterations than
others. This is a feature, not a bug. May result in less aesthetic
plotting, however.

For details about handling the random seed, see \code{\link{PowerGrid}}.
}
\examples{
## ============================================
## very simple example with one parameter
## ============================================
pars = list(x = 1:2)
fun = function(x){round(x+runif(1, 0, .2), 3)} # nonsense function
set.seed(1)
original = PowerGrid(pars = pars,
                     fun = fun,
                     n_iter = 3,
                     summarize = FALSE)
refined = Refine(original, n_iter_add = 2, pars = list(x = 2:3))
## note that refined does not have each parameter sampled in each iteration

## ============================================
## a realistic example, simply increasing n_iter
## ============================================
PowFun <- function(n, delta){
  x1 = rnorm(n = n/2, sd = 1)
  x2 = rnorm(n = n/2, mean = delta, sd = 1)
  t.test(x1, x2)$p.value < .05
}
sse_pars = list(
  n = seq(10, 100, 5),
  delta = seq(.5, 1.5, .1))
##
n_iter = 20
set.seed(1)
power_array = PowerGrid(pars = sse_pars,
                        fun = PowFun,
                        n_iter = n_iter,
                        summarize = FALSE)
summary(power_array)
## add iterations
power_array_up = Refine(power_array, n_iter_add = 30)
summary(power_array_up)

## ============================================
## Starting coarsely, then zooming in
## ============================================
sse_pars = list(
  n = c(10, 50, 100, 200), # finding n "ballpark"
  delta = c(.5,  1, 1.5)) # finding delta "ballpark"
n_iter = 60
power_array = PowerGrid(pars = sse_pars,
                        fun = PowFun,
                        n_iter = n_iter,
                        summarize = FALSE)
summary(power_array)
PowerPlot(power_array)
## Based on figure above, let's look at n between 50 and 100, delta around .9
\donttest{
sse_pars = list(
  n = seq(50, 100, 5),
  delta = seq(.7, 1.1, .05))
set.seed(1)
power_array_up = Refine(power_array, n_iter_add = 555, pars = sse_pars)
summary(power_array_up)
PowerPlot(power_array_up) # that looks funny! It's because the default summary
                          # mean does not deal with the empty value in the
                          # grid. Solution is in illustration below.

## A visual illustration of this zooming in, in three figures
layout(t(1:3))
PowerPlot(power_array, title = 'Course grid to start with')
PowerPlot(power_array_up, summary_function = function(x)mean(x, na.rm = TRUE),
          title = 'Extra samples at finer parameter grid (does not look good)')
PowerPlot(power_array_up,
          slicer = list(n = seq(50, 100, 5),
                        delta = seq(.7, 1.1, .05)),
          summary_function = function(x)mean(x, na.rm = TRUE),
          title = 'Zoomed in')
layout(1)
}
}
\seealso{
\code{\link{PowerGrid}}
}
\author{
Gilles Dutilh
}
