#' Computes Stuart's Q test of marginal homogeneity.
#'
#' Stuart, A. (1955). A test for homogeneity of the marginal distributions
#' in a two-way classification.  Biometrika, 42(3/4), 412-416.
#' @importFrom stats pchisq
#' @param n matrix of observed counts
#' @returns a list containing
#'    q: value of q test-statistic
#'    df: degrees of freedom
#'    p: upper tail p-value of q
#' @export
#' @examples
#' Stuart_marginal_homogeneity(vision_data)
Stuart_marginal_homogeneity <- function(n) {
  k <- nrow(n) - 1
  v <- matrix(nrow = k, ncol = k)
  d <- rep(0.0, k)
  v_i <- rowSums(n)
  v_j <- colSums(n)
  for (i in 1:k) {
    for (j in 1: k) {
      v[i, j] <- -(n[i, j] + n[j, i])
    }
    v[i, i] <-  v[i, i] + v_i[i] + v_j[i]
    d[i] <- v_i[i] - v_j[i]
  }

  epsilon <- 1.0e-10
  epsilon2 <- .Machine$double.eps * 10
  if (det(v) < epsilon || rcond(v) < epsilon2) {
    warning("Near singular matrix")
    q <- NA
  } else {
    q <- t(d) %*% solve(v, d)
  }
  p <- pchisq(q, k, lower.tail = FALSE)
  list(chisq=q, df=k, p=p)
}
