#' Agresti's simple diganal quasi-symmetry model.
#'
#' This parameterization does not match equation (2.2) in the paper, but it yields
#' results that are identical to those in the paper.
#' Agresti, A. (1983), A simple diagonals-parameter symmetry and quasi-symmetry model.
#' Statistics and Probability Letters I, 313-316.
#' @param n the matrix of observed counts
#' @returns a list containing
#'    expected: matrix of expected cell frequencies,
#'    chisq: Pearson X^2
#'    g_squared: likelihood ratio G^2
#'    df: degrees of freedom
#'    beta: the parameter estimated
#'    sigma_beta: standard error of beta
#'    z: z-score for beta
#'    delta: transformation of the the parameter into the model formulation
#' @export
#' @examples
#' Agresti_simple_diagonals_parameter_quasi_symmetry(vision_data)
Agresti_simple_diagonals_parameter_quasi_symmetry <- function(n) {
  agresti_design <- Agresti_create_design_matrix(nrow(n))
  agresti_result <- log_linear_fit(n, agresti_design)
  agresti <- Agresti_extract_delta(agresti_result)
  list(expected=agresti_result$expected,
       chisq=agresti_result$chisq, g_squared=agresti_result$g_squared,
       df=agresti_result$df, beta=agresti$beta, sigma_beta=agresti$sigma_beta,
       z=agresti$z, delta=agresti$delta)
}

#' Creates the design matrix for Agresti's simple diagonal quasi-symmetry model.
#'
#' This parameterization does not match equation (2.2) in the paper, but it yields
#' results that are identical to those in the paper.
#' Agresti, A. (1983), A simple diagonals-parameter symmetry and quasi-symmetry model.
#' Statistics and Probability Letters I, 313-316.
#' @param n_dim the size of the date matrix
#' @return the design matrix for the model, that can bee used with ml_for_log_linear
#' @export
Agresti_create_design_matrix <- function(n_dim) {
  n_rows <- n_dim^2
  n_columns <- 1 + n_dim * (n_dim + 1) / 2
  x <- matrix(0.0, nrow=n_rows, ncol=n_columns)

  column <- 1
  final_column <- ncol(x)
  for (i in 1:n_dim) {
    for (j in 1:n_dim) {
      index <- (i - 1) * n_dim + j
      if (i == j) {
        x[index, column] <- 1
        column <- column + 1
      }
      x[index, final_column] <- j - i
    }
  }

  for (i in 1:n_dim) {
    for (j in i:n_dim) {
      if (i == j) {
        next
      }
      index <- (i - 1) * n_dim + j
      x[index, column] <- 1
      index <- (j - 1) * n_dim + i
      x[index, column] <- 1
      column <- column + 1
    }
  }
  x
}


#' Extracts the quasi-symmetry information from the result provided.
#'
#' @param result result of call to log_linear_fit()
#' @returns list consisting of
#'    beta: the beta coefficient
#'    se: the standard error of beta
#'    z: the ratio beta / se
#'    delta: the delta coefficient = exp(2.0 * beta)
#' @export
Agresti_extract_delta <- function(result) {
  beta <- result$beta[length(result$beta)]
  se <- result$se[length(result$se)]
  z <- beta / se
  delta <- exp(2.0 * beta)
  list(beta=beta, sigma_beta=se, delta=delta, z=z)
}
