% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mspline.R
\name{make_knots}
\alias{make_knots}
\title{Knot locations for M-spline baseline hazard models}
\usage{
make_knots(
  network,
  n_knots = 7,
  type = c("quantile", "quantile_common", "quantile_lumped", "quantile_longest", "equal",
    "equal_common")
)
}
\arguments{
\item{network}{A network object, containing survival outcomes}

\item{n_knots}{Non-negative integer giving the number of internal knots
(default \code{7})}

\item{type}{String specifying the knot location algorithm to use (see
details). The default used by \code{\link[=nma]{nma()}} is \code{"quantile"}, except when a
regression model is specified (using \code{aux_regression}) in which case the
default is \code{"quantile_common"}.}
}
\value{
A named list of vectors giving the knot locations in each study.
}
\description{
Several different algorithms are provided to calculate knot locations for
M-spline baseline hazard models. This function is called internally within
the \code{\link[=nma]{nma()}} function, but may be called directly by the user for more
control.
}
\details{
The \code{type} argument can be used to choose between different
algorithms for placing the knots:

\describe{
\item{\code{"quantile"}}{Creates separate knot locations for each study,
internal knots are placed at evenly-spaced quantiles of the observed event
times within each study.}
\item{\code{"quantile_lumped"}}{Creates a common set of knots for all studies,
calculated as evenly-spaced quantiles of the observed event times from all
studies lumped together.}
\item{\code{"quantile_common"}}{Creates a common set of knots for all studies,
taking quantiles of the quantiles of the observed event times within each
study. This often seems to result in a more even knot spacing than
\code{"quantile_lumped"}, particularly when follow-up is uneven across studies,
and may handle differing behaviour in the baseline hazard across studies
better than \code{"quantile_longest"}.}
\item{\code{"quantile_longest"}}{Creates a common set of knots for all studies,
using evenly-spaced quantiles of the observed event times in the longest
study.}
\item{\code{"equal"}}{Creates separate knot locations for each study, at
evenly-spaced times between the boundary knots in each study.}
\item{\code{"equal_common"}}{Creates a common set of knots for all studies, at
evenly-spaced times between the earliest entry time and last
event/censoring time in the network.}
}

Boundary knots are calculated as follows:
\itemize{
\item For separate knot locations in each study, boundary knots are placed at the
earliest entry time and last event/censoring time in each study.
\item For a common set of knots across all studies, boundary knots are placed at
the earliest entry time and last event/censoring time across all studies.
}

Models with regression on the spline coefficients (i.e. with \code{aux_regression}
specified) require a common set of knots across all studies.

Provided that a sufficient number of knots are used, model fit should be
largely unaffected by the knot locations. However, sampling difficulties can
sometimes occur if knot placement is poor, for example if a knot is placed
just before the last follow-up time in a study.
}
\examples{
# Set up newly-diagnosed multiple myeloma network

head(ndmm_ipd)
head(ndmm_agd)

ndmm_net <- combine_network(
  set_ipd(ndmm_ipd,
          study, trt,
          Surv = Surv(eventtime / 12, status)),
  set_agd_surv(ndmm_agd,
               study, trt,
               Surv = Surv(eventtime / 12, status),
               covariates = ndmm_agd_covs))

# The default knot locations
make_knots(ndmm_net, type = "quantile")

# Increasing the number of knots
make_knots(ndmm_net, n_knots = 10)

# Comparing alternative knot positioning algorithms
# Visualise these with a quick function
plot_knots <- function(network, knots) {
  ggplot2::ggplot() +
    geom_km(network) +
    ggplot2::geom_vline(ggplot2::aes(xintercept = .data$knot),
                        data = tidyr::pivot_longer(as.data.frame(knots), cols = dplyr::everything(),
                                                   names_to = "Study", values_to = "knot"),
                        linetype = 2, colour = "grey60") +
    ggplot2::facet_wrap(~Study) +
    theme_multinma()
}

plot_knots(ndmm_net, make_knots(ndmm_net, type = "quantile"))
plot_knots(ndmm_net, make_knots(ndmm_net, type = "quantile_common"))
plot_knots(ndmm_net, make_knots(ndmm_net, type = "quantile_lumped"))
plot_knots(ndmm_net, make_knots(ndmm_net, type = "quantile_longest"))
plot_knots(ndmm_net, make_knots(ndmm_net, type = "equal"))
plot_knots(ndmm_net, make_knots(ndmm_net, type = "equal_common"))

}
