% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/esp-get-prov.R
\encoding{UTF-8}
\name{esp_get_prov}
\alias{esp_get_prov}
\title{Provinces of Spain - GISCO}
\source{
\url{https://gisco-services.ec.europa.eu/distribution/v2/}.

Copyright:
\url{https://ec.europa.eu/eurostat/web/gisco/geodata/administrative-units}.
}
\usage{
esp_get_prov(prov = NULL, moveCAN = TRUE, ...)
}
\arguments{
\item{prov}{A vector of names and/or codes for provinces or \code{NULL} to get all
the provinces. See \strong{Details}.}

\item{moveCAN}{A logical \code{TRUE/FALSE} or a vector of coordinates
\code{c(lat, lon)}. It places the Canary Islands close to Spain's mainland.
Initial position can be adjusted using the vector of coordinates. See
\strong{Displacing the Canary Islands} in \code{\link[=esp_move_can]{esp_move_can()}}.}

\item{...}{
  Arguments passed on to \code{\link[=esp_get_nuts]{esp_get_nuts}}, \code{\link[=esp_get_nuts]{esp_get_nuts}}
  \describe{
    \item{\code{year}}{year character string or number. Release year of the file. See
\code{\link[giscoR:gisco_get_nuts]{giscoR::gisco_get_nuts()}} for valid values.}
    \item{\code{epsg}}{character string or number. Projection of the map: 4-digit
\href{https://epsg.io/}{EPSG code}. One of:
\itemize{
\item \code{"4258"}: \href{https://epsg.io/4258}{ETRS89}
\item \code{"4326"}: \href{https://epsg.io/4326}{WGS84}.
\item \code{"3035"}: \href{https://epsg.io/3035}{ETRS89 / ETRS-LAEA}.
\item \code{"3857"}: \href{https://epsg.io/3857}{Pseudo-Mercator}.
}}
    \item{\code{cache}}{logical. Whether to do caching. Default is \code{TRUE}. See
\strong{Caching strategies} section in \code{\link[=esp_set_cache_dir]{esp_set_cache_dir()}}.}
    \item{\code{update_cache}}{logical. Should the cached file be refreshed? Default
is \code{FALSE}. When set to \code{TRUE}, it will force a new download.}
    \item{\code{cache_dir}}{character string. A path to a cache directory. See
\strong{Caching strategies} section in \code{\link[=esp_set_cache_dir]{esp_set_cache_dir()}}.}
    \item{\code{spatialtype}}{character string. Type of geometry to be returned.
Options available are:
\itemize{
\item "RG": Regions - \code{MULTIPOLYGON/POLYGON} object.
\item "LB": Labels - \code{POINT} object.
}}
    \item{\code{ext}}{character. Extension of the file (default \code{"gpkg"}). See
\code{\link[giscoR:gisco_get_nuts]{giscoR::gisco_get_nuts()}}.}
    \item{\code{verbose}}{logical. If \code{TRUE} displays informational messages.}
    \item{\code{resolution}}{character string or number. Resolution of the geospatial
data. One of:
\itemize{
\item \code{"60"}: 1:60 million.
\item \code{"20"}: 1:20 million.
\item \code{"10"}: 1:10 million.
\item \code{"03"}: 1:3 million.
\item \code{"01"}: 1:1 million.
}}
  }}
}
\value{
A \code{\link[sf:sf]{sf}} object.
}
\description{
Returns
\href{https://en.wikipedia.org/wiki/Provinces_of_Spain}{provinces of Spain}
at a specified scale.
}
\details{
When using \code{prov} you can use and mix names and NUTS codes (levels 1, 2 or
3), ISO codes (corresponding to level 2 or 3) or "cpro" (see
\link{esp_codelist}).

Ceuta and Melilla are considered as provinces in this dataset.

When calling a higher level (Autonomous Community or NUTS1), all the
provinces of that level will be added.
}
\section{Note}{

Please check the download and usage provisions on \code{\link[giscoR:gisco_attributions]{gisco_attributions()}}.

}

\examples{
prov <- esp_get_prov()

library(ggplot2)

ggplot(prov) +
  geom_sf() +
  theme_minimal()

\donttest{
# Random Provinces
random <- esp_get_prov(prov = c(
  "Zamora", "Palencia", "ES-GR",
  "ES521", "01"
))


ggplot(random) +
  geom_sf(aes(fill = codauto), show.legend = FALSE, alpha = 0.5) +
  scale_fill_manual(values = hcl.colors(nrow(random), "Spectral")) +
  theme_minimal()


# All Provinces of a Zone plus an addition
mix <- esp_get_prov(prov = c(
  "Noroeste",
  "Castilla y Leon", "La Rioja"
))

mix$ccaa <- esp_dict_region_code(
  mix$codauto,
  origin = "codauto"
)

ggplot(mix) +
  geom_sf(aes(fill = ccaa), alpha = 0.5) +
  scale_fill_discrete(type = hcl.colors(5, "Temps")) +
  theme_classic()

# ISO codes available

allprovs <- esp_get_prov()

ggplot(allprovs) +
  geom_sf(fill = NA) +
  geom_sf_text(aes(label = iso2.prov.code),
    check_overlap = TRUE,
    fontface = "bold"
  ) +
  coord_sf(crs = 3857) +
  theme_void()
}
}
\seealso{
Other datasets representing political borders:
\code{\link{esp_get_capimun}()},
\code{\link{esp_get_ccaa}()},
\code{\link{esp_get_ccaa_siane}()},
\code{\link{esp_get_comarca}()},
\code{\link{esp_get_countries_siane}()},
\code{\link{esp_get_gridmap}},
\code{\link{esp_get_munic}()},
\code{\link{esp_get_munic_siane}()},
\code{\link{esp_get_nuts}()},
\code{\link{esp_get_prov_siane}()},
\code{\link{esp_get_simpl}},
\code{\link{esp_get_spain}()},
\code{\link{esp_get_spain_siane}()},
\code{\link{esp_siane_bulk_download}()}

Datasets provided by GISCO:
\code{\link{esp_get_ccaa}()},
\code{\link{esp_get_munic}()},
\code{\link{esp_get_nuts}()},
\code{\link{esp_get_spain}()}
}
\concept{gisco}
\concept{political}
