% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/micro_read.R
\name{read_ipums_micro}
\alias{read_ipums_micro}
\alias{read_ipums_micro_list}
\title{Read data from an IPUMS microdata extract}
\usage{
read_ipums_micro(
  ddi,
  vars = NULL,
  n_max = Inf,
  data_file = NULL,
  verbose = TRUE,
  var_attrs = c("val_labels", "var_label", "var_desc"),
  lower_vars = FALSE
)

read_ipums_micro_list(
  ddi,
  vars = NULL,
  n_max = Inf,
  data_file = NULL,
  verbose = TRUE,
  var_attrs = c("val_labels", "var_label", "var_desc"),
  lower_vars = FALSE
)
}
\arguments{
\item{ddi}{Either a path to a DDI .xml file downloaded from
\href{https://www.ipums.org/}{IPUMS}, or an
\link{ipums_ddi} object parsed by \code{\link[=read_ipums_ddi]{read_ipums_ddi()}}. See
\emph{Downloading IPUMS files} below.}

\item{vars}{Names of variables to include in the output. Accepts a
vector of names or a \link[=selection_language]{tidyselect selection}.
If \code{NULL}, includes all variables in the file.

For hierarchical data, the \code{RECTYPE} variable is always included even if
unspecified.}

\item{n_max}{The maximum number of lines to read. For
\code{read_ipums_micro_list()}, this applies before splitting records into
list components.}

\item{data_file}{Path to the data (.gz) file associated with
the provided \code{ddi} file. By default, looks for the data file in the same
directory as the DDI file. If the data file has been moved, specify
its location here.}

\item{verbose}{Logical indicating whether to display IPUMS conditions and
progress information.}

\item{var_attrs}{Variable attributes from the DDI to add to the columns of
the output data. Defaults to all available attributes.
See \code{\link[=set_ipums_var_attributes]{set_ipums_var_attributes()}} for more details.}

\item{lower_vars}{If reading a DDI from a file,
a logical indicating whether to convert variable names to lowercase.
Defaults to \code{FALSE} for consistency with IPUMS conventions.

This argument will be ignored if argument \code{ddi} is
an \link{ipums_ddi} object. Use \code{\link[=read_ipums_ddi]{read_ipums_ddi()}} to convert variable
names to lowercase when reading a DDI file.

If \code{lower_vars = TRUE} and \code{vars} is specified, \code{vars} should reference the
lowercase column names.}
}
\value{
\code{read_ipums_micro()} returns a single
\code{\link[tibble:tbl_df-class]{tibble}} object.

\code{read_ipums_micro_list()} returns a list of \code{tibble} objects with one
entry for each record type.
}
\description{
Read a microdata dataset downloaded from the IPUMS extract system.

Two files are required to load IPUMS microdata extracts:
\itemize{
\item A \href{https://ddialliance.org/introduction-to-ddi}{DDI codebook} file
(.xml) used to parse the extract's data file
\item A data file (either .dat.gz or .csv.gz)
}

See \emph{Downloading IPUMS files} below for more information about
downloading these files.

\code{read_ipums_micro()} and \code{read_ipums_micro_list()} differ in their handling
of extracts that contain multiple record types. See \emph{Data structures}
below.

Note that Stata, SAS, and SPSS file formats are not supported by
ipumsr readers. Convert your extract to fixed-width or CSV format, or see
\href{https://haven.tidyverse.org/index.html}{haven} for help
loading these files.
}
\section{Data structures}{


Files from IPUMS projects that contain data for multiple types of records
(e.g. household records and person records) may be either rectangular
or hierarchical.

Rectangular data are transformed such that each row of data
represents only one type of record. For instance, each row will represent
a person record, and all household-level information for that person will
be included in the same row.

Hierarchical data have records of
different types interspersed in a single file. For instance, a household
record will be included in its own row followed by the person records
associated with that household.

Hierarchical data can be read in two different formats:
\itemize{
\item \code{read_ipums_micro()} reads data into a \code{\link[tibble:tbl_df-class]{tibble}}
where each row represents a single record, regardless of record type.
Variables that do not apply to a particular record type will be filled with
\code{NA} in rows of that record type. For instance, a person-specific variable
will be missing in all rows associated with household records.
\item \code{read_ipums_micro_list()} reads data into a list of
\code{tibble} objects, where each list element contains
only one record type. Each list element is named with its corresponding
record type.
}
}

\section{Downloading IPUMS files}{


You must download both the DDI codebook and the data file from the IPUMS
extract system to load the data into R. \verb{read_ipums_micro_*()} functions
assume that the data file and codebook share a common base file name and
are present in the same directory. If this is not the case, provide a
separate path to the data file with the \code{data_file} argument.

If using the IPUMS extract interface:
\itemize{
\item Download the data file by clicking \strong{Download .dat} under
\strong{Download Data}.
\item Download the DDI codebook by right clicking on the \strong{DDI} link in the
\strong{Codebook} column of the extract interface and selecting \strong{Save as...}
(on Safari, you may have to select \strong{Download Linked File as...}).
Be sure that the codebook is downloaded in .xml format.
}

If using the IPUMS API:
\itemize{
\item For supported collections, use \code{\link[=download_extract]{download_extract()}} to download a completed
extract via the IPUMS API. This automatically downloads both the DDI
codebook and the data file from the extract and
returns the path to the codebook file.
}
}

\examples{
# Codebook for rectangular example file
cps_rect_ddi_file <- ipums_example("cps_00157.xml")

# Load data based on codebook file info
cps <- read_ipums_micro(cps_rect_ddi_file)

head(cps)

# Can also load data from a pre-existing `ipums_ddi` object
# (This may be useful to retain codebook metadata even if lost from data
# during processing)
ddi <- read_ipums_ddi(cps_rect_ddi_file)
cps <- read_ipums_micro(ddi, verbose = FALSE)

# Codebook for hierarchical example file
cps_hier_ddi_file <- ipums_example("cps_00159.xml")

# Read in "long" format to get a single data frame
read_ipums_micro(cps_hier_ddi_file, verbose = FALSE)

# Read in "list" format and you get a list of multiple data frames
cps_list <- read_ipums_micro_list(cps_hier_ddi_file)

head(cps_list$PERSON)

head(cps_list$HOUSEHOLD)

# Use the `\%<-\%` operator from zeallot to unpack into separate objects
c(household, person) \%<-\% read_ipums_micro_list(cps_hier_ddi_file)

head(person)

head(household)
}
\seealso{
\code{\link[=read_ipums_micro_chunked]{read_ipums_micro_chunked()}} and
\code{\link[=read_ipums_micro_yield]{read_ipums_micro_yield()}} to read data from large IPUMS
microdata extracts in chunks.

\code{\link[=read_ipums_ddi]{read_ipums_ddi()}} to read metadata associated with an IPUMS microdata
extract.

\code{\link[=read_ipums_sf]{read_ipums_sf()}} to read spatial data from an IPUMS extract.

\code{\link[=ipums_list_files]{ipums_list_files()}} to list files in an IPUMS extract.
}
