/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5FDmodule.h" 

#include "H5private.h"   
#include "H5Eprivate.h"  
#include "H5Fprivate.h"  
#include "H5FDcore.h"    
#include "H5FDpkg.h"     
#include "H5FLprivate.h" 
#include "H5Iprivate.h"  
#include "H5MMprivate.h" 
#include "H5Pprivate.h"  
#include "H5SLprivate.h" 

hid_t H5FD_CORE_id_g = H5I_INVALID_HID;

typedef struct H5FD_core_region_t {
    haddr_t start; 
    haddr_t end;   
} H5FD_core_region_t;

typedef struct H5FD_core_t {
    H5FD_t         pub;              
    char          *name;             
    unsigned char *mem;              
    haddr_t        eoa;              
    haddr_t        eof;              
    size_t         increment;        
    bool           backing_store;    
    bool           write_tracking;   
    size_t         bstore_page_size; 
    bool           ignore_disabled_file_locks;
    int            fd; 
    
#ifndef H5_HAVE_WIN32_API
    
    dev_t device; 
    ino_t inode;  
#else
    
    DWORD nFileIndexLow;
    DWORD nFileIndexHigh;
    DWORD dwVolumeSerialNumber;

    HANDLE hFile; 
#endif                                        
    bool                        dirty;        
    H5FD_file_image_callbacks_t fi_callbacks; 
    H5SL_t                     *dirty_list;   
} H5FD_core_t;

typedef struct H5FD_core_fapl_t {
    size_t increment;      
    bool   backing_store;  
    bool   write_tracking; 
    size_t page_size;      
} H5FD_core_fapl_t;

#define H5FD_CORE_INCREMENT                8192
#define H5FD_CORE_WRITE_TRACKING_FLAG      false
#define H5FD_CORE_WRITE_TRACKING_PAGE_SIZE 524288

#define CORE_MAXADDR          ((haddr_t)((~(size_t)0) - 1))
#define CORE_ADDR_OVERFLOW(A) (HADDR_UNDEF == (A) || (A) > (haddr_t)CORE_MAXADDR)
#define CORE_SIZE_OVERFLOW(Z) ((Z) > (hsize_t)CORE_MAXADDR)
#define CORE_REGION_OVERFLOW(A, Z)                                                                           \
    (CORE_ADDR_OVERFLOW(A) || CORE_SIZE_OVERFLOW(Z) || HADDR_UNDEF == (A) + (Z) ||                           \
     (size_t)((A) + (Z)) < (size_t)(A))

static herr_t  H5FD__core_add_dirty_region(H5FD_core_t *file, haddr_t start, haddr_t end);
static herr_t  H5FD__core_destroy_dirty_list(H5FD_core_t *file);
static herr_t  H5FD__core_write_to_bstore(H5FD_core_t *file, haddr_t addr, size_t size);
static void   *H5FD__core_fapl_get(H5FD_t *_file);
static H5FD_t *H5FD__core_open(const char *name, unsigned flags, hid_t fapl_id, haddr_t maxaddr);
static herr_t  H5FD__core_close(H5FD_t *_file);
static int     H5FD__core_cmp(const H5FD_t *_f1, const H5FD_t *_f2);
static herr_t  H5FD__core_query(const H5FD_t *_f1, unsigned long *flags);
static haddr_t H5FD__core_get_eoa(const H5FD_t *_file, H5FD_mem_t type);
static herr_t  H5FD__core_set_eoa(H5FD_t *_file, H5FD_mem_t type, haddr_t addr);
static haddr_t H5FD__core_get_eof(const H5FD_t *_file, H5FD_mem_t type);
static herr_t  H5FD__core_get_handle(H5FD_t *_file, hid_t fapl, void **file_handle);
static herr_t  H5FD__core_read(H5FD_t *_file, H5FD_mem_t type, hid_t fapl_id, haddr_t addr, size_t size,
                               void *buf);
static herr_t  H5FD__core_write(H5FD_t *_file, H5FD_mem_t type, hid_t fapl_id, haddr_t addr, size_t size,
                                const void *buf);
static herr_t  H5FD__core_flush(H5FD_t *_file, hid_t dxpl_id, bool closing);
static herr_t  H5FD__core_truncate(H5FD_t *_file, hid_t dxpl_id, bool closing);
static herr_t  H5FD__core_lock(H5FD_t *_file, bool rw);
static herr_t  H5FD__core_unlock(H5FD_t *_file);
static herr_t  H5FD__core_delete(const char *filename, hid_t fapl_id);
static inline const H5FD_core_fapl_t *H5FD__core_get_default_config(void);

static const H5FD_class_t H5FD_core_g = {
    H5FD_CLASS_VERSION,       
    H5FD_CORE_VALUE,          
    "core",                   
    CORE_MAXADDR,             
    H5F_CLOSE_WEAK,           
    NULL,                     
    NULL,                     
    NULL,                     
    NULL,                     
    sizeof(H5FD_core_fapl_t), 
    H5FD__core_fapl_get,      
    NULL,                     
    NULL,                     
    0,                        
    NULL,                     
    NULL,                     
    H5FD__core_open,          
    H5FD__core_close,         
    H5FD__core_cmp,           
    H5FD__core_query,         
    NULL,                     
    NULL,                     
    NULL,                     
    H5FD__core_get_eoa,       
    H5FD__core_set_eoa,       
    H5FD__core_get_eof,       
    H5FD__core_get_handle,    
    H5FD__core_read,          
    H5FD__core_write,         
    NULL,                     
    NULL,                     
    NULL,                     
    NULL,                     
    H5FD__core_flush,         
    H5FD__core_truncate,      
    H5FD__core_lock,          
    H5FD__core_unlock,        
    H5FD__core_delete,        
    NULL,                     
    H5FD_FLMAP_DICHOTOMY      
};

static const H5FD_core_fapl_t H5FD_core_default_config_g = {
    (size_t)H5_MB, true, H5FD_CORE_WRITE_TRACKING_FLAG, H5FD_CORE_WRITE_TRACKING_PAGE_SIZE};
static const H5FD_core_fapl_t H5FD_core_default_paged_config_g = {(size_t)H5_MB, true, true, (size_t)4096};

H5FL_DEFINE(H5FD_core_region_t);

static herr_t
H5FD__core_add_dirty_region(H5FD_core_t *file, haddr_t start, haddr_t end)
{
    H5FD_core_region_t *b_item          = NULL;
    H5FD_core_region_t *a_item          = NULL;
    H5FD_core_region_t *item            = NULL;
    haddr_t             b_addr          = 0;
    haddr_t             a_addr          = 0;
    bool                create_new_node = true;
    herr_t              ret_value       = SUCCEED;

    FUNC_ENTER_PACKAGE

    assert(file);
    assert(file->dirty_list);
    assert(start <= end);

    
    if (start % file->bstore_page_size != 0)
        start = (start / file->bstore_page_size) * file->bstore_page_size;

    if (end % file->bstore_page_size != (file->bstore_page_size - 1)) {
        end = (((end / file->bstore_page_size) + 1) * file->bstore_page_size) - 1;
        if (end > file->eof)
            end = file->eof - 1;
    } 

    
    b_addr = start + 1;
    a_addr = end + 2;
    b_item = (H5FD_core_region_t *)H5SL_less(file->dirty_list, &b_addr);
    a_item = (H5FD_core_region_t *)H5SL_less(file->dirty_list, &a_addr);

    
    if (a_item)
        if (start < a_item->start && end < a_item->end) {
            
            end = a_item->end;
        } 
    
    if (b_item)
        if (start <= b_item->end + 1) {

            
            start = b_item->start;

            
            create_new_node = false;
        } 

    
    while (a_item && a_item->start > start) {

        H5FD_core_region_t *less;
        haddr_t             key = a_item->start - 1;

        
        less = (H5FD_core_region_t *)H5SL_less(file->dirty_list, &key);

        
        a_item = (H5FD_core_region_t *)H5SL_remove(file->dirty_list, &a_item->start);
        a_item = H5FL_FREE(H5FD_core_region_t, a_item);

        
        if (less)
            a_item = less;
    } 

    
    if (create_new_node) {
        if (NULL == (item = (H5FD_core_region_t *)H5SL_search(file->dirty_list, &start))) {
            
            item        = (H5FD_core_region_t *)H5FL_CALLOC(H5FD_core_region_t);
            item->start = start;
            item->end   = end;
            if (H5SL_insert(file->dirty_list, item, &item->start) < 0)
                HGOTO_ERROR(H5E_SLIST, H5E_CANTINSERT, FAIL, "can't insert new dirty region: (%llu, %llu)\n",
                            (unsigned long long)start, (unsigned long long)end);
        } 
        else {
            
            item->end = (item->end < end) ? end : item->end;
        }
    }
    else {
        
        if (b_item->end < end)
            b_item->end = end;
    }

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5FD__core_destroy_dirty_list(H5FD_core_t *file)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    assert(file);

    
    if (file->dirty_list) {
        H5FD_core_region_t *region = NULL;

        while (NULL != (region = (H5FD_core_region_t *)H5SL_remove_first(file->dirty_list)))
            region = H5FL_FREE(H5FD_core_region_t, region);

        if (H5SL_close(file->dirty_list) < 0)
            HGOTO_ERROR(H5E_SLIST, H5E_CLOSEERROR, FAIL, "can't close core vfd dirty list");
        file->dirty_list = NULL;
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5FD__core_write_to_bstore(H5FD_core_t *file, haddr_t addr, size_t size)
{
    unsigned char *ptr = file->mem + addr; 
    HDoff_t offset    = (HDoff_t)addr;     
    herr_t  ret_value = SUCCEED;           

    FUNC_ENTER_PACKAGE

    assert(file);

#ifndef H5_HAVE_PREADWRITE
    
    if ((HDoff_t)addr != HDlseek(file->fd, (HDoff_t)addr, SEEK_SET))
        HGOTO_ERROR(H5E_IO, H5E_SEEKERROR, FAIL, "error seeking in backing store");
#endif 

    while (size > 0) {

        h5_posix_io_t     bytes_in    = 0;  
        h5_posix_io_ret_t bytes_wrote = -1; 

        
        if (size > H5_POSIX_MAX_IO_BYTES)
            bytes_in = H5_POSIX_MAX_IO_BYTES;
        else
            bytes_in = (h5_posix_io_t)size;

        do {
#ifdef H5_HAVE_PREADWRITE
            bytes_wrote = HDpwrite(file->fd, ptr, bytes_in, offset);
            if (bytes_wrote > 0)
                offset += bytes_wrote;
#else
            bytes_wrote = HDwrite(file->fd, ptr, bytes_in);
#endif 
        } while (-1 == bytes_wrote && EINTR == errno);

        if (-1 == bytes_wrote) { 
            int    myerrno = errno;
            time_t mytime  = time(NULL);

            offset = HDlseek(file->fd, 0, SEEK_CUR);

            HGOTO_ERROR(H5E_IO, H5E_WRITEERROR, FAIL,
                        "write to backing store failed: time = %s, filename = '%s', file descriptor = %d, "
                        "errno = %d, error message = '%s', ptr = %p, total write size = %llu, bytes this "
                        "sub-write = %llu, bytes actually written = %llu, offset = %llu",
                        ctime(&mytime), file->name, file->fd, myerrno, strerror(myerrno), (void *)ptr,
                        (unsigned long long)size, (unsigned long long)bytes_in,
                        (unsigned long long)bytes_wrote, (unsigned long long)offset);
        } 

        assert(bytes_wrote > 0);
        assert((size_t)bytes_wrote <= size);

        size -= (size_t)bytes_wrote;
        ptr = (unsigned char *)ptr + bytes_wrote;

    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static inline const H5FD_core_fapl_t *
H5FD__core_get_default_config(void)
{
    char *driver = getenv(HDF5_DRIVER);

    if (driver) {
        if (!strcmp(driver, "core"))
            return &H5FD_core_default_config_g;
        else if (!strcmp(driver, "core_paged"))
            return &H5FD_core_default_paged_config_g;
    }

    return &H5FD_core_default_config_g;
} 

herr_t
H5FD__core_register(void)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    if (H5I_VFL != H5I_get_type(H5FD_CORE_id_g))
        if ((H5FD_CORE_id_g = H5FD_register(&H5FD_core_g, sizeof(H5FD_class_t), false)) < 0)
            HGOTO_ERROR(H5E_VFL, H5E_CANTREGISTER, FAIL, "unable to register core driver");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5FD__core_unregister(void)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    H5FD_CORE_id_g = H5I_INVALID_HID;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

herr_t
H5Pset_core_write_tracking(hid_t plist_id, bool is_enabled, size_t page_size)
{
    H5P_genplist_t         *plist;               
    H5FD_core_fapl_t        fa;                  
    const H5FD_core_fapl_t *old_fa;              
    herr_t                  ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (page_size == 0)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "page_size cannot be zero");

    
    if (NULL == (plist = H5P_object_verify(plist_id, H5P_FILE_ACCESS, false)))
        HGOTO_ERROR(H5E_PLIST, H5E_BADID, FAIL, "can't find object for ID");
    if (H5FD_CORE != H5P_peek_driver(plist))
        HGOTO_ERROR(H5E_PLIST, H5E_BADVALUE, FAIL, "incorrect VFL driver");
    if (NULL == (old_fa = (const H5FD_core_fapl_t *)H5P_peek_driver_info(plist)))
        old_fa = H5FD__core_get_default_config();

    
    memset(&fa, 0, sizeof(H5FD_core_fapl_t));
    fa.increment      = old_fa->increment;
    fa.backing_store  = old_fa->backing_store;
    fa.write_tracking = is_enabled;
    fa.page_size      = page_size;

    
    if (H5P_set_driver(plist, H5FD_CORE, &fa, NULL) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTSET, FAIL, "can't set core VFD as driver");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Pget_core_write_tracking(hid_t plist_id, bool *is_enabled , size_t *page_size )
{
    H5P_genplist_t         *plist;               
    const H5FD_core_fapl_t *fa;                  
    herr_t                  ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (plist = H5P_object_verify(plist_id, H5P_FILE_ACCESS, true)))
        HGOTO_ERROR(H5E_PLIST, H5E_BADID, FAIL, "can't find object for ID");
    if (H5FD_CORE != H5P_peek_driver(plist))
        HGOTO_ERROR(H5E_PLIST, H5E_BADVALUE, FAIL, "incorrect VFL driver");
    if (NULL == (fa = (const H5FD_core_fapl_t *)H5P_peek_driver_info(plist)))
        HGOTO_ERROR(H5E_PLIST, H5E_BADVALUE, FAIL, "bad VFL driver info");

    
    if (is_enabled)
        *is_enabled = fa->write_tracking;
    if (page_size)
        *page_size = fa->page_size;

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Pset_fapl_core(hid_t fapl_id, size_t increment, bool backing_store)
{
    H5P_genplist_t  *plist;               
    H5FD_core_fapl_t fa;                  
    herr_t           ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (plist = H5P_object_verify(fapl_id, H5P_FILE_ACCESS, false)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a file access property list");

    
    memset(&fa, 0, sizeof(H5FD_core_fapl_t));
    fa.increment      = increment;
    fa.backing_store  = backing_store;
    fa.write_tracking = H5FD_CORE_WRITE_TRACKING_FLAG;
    fa.page_size      = H5FD_CORE_WRITE_TRACKING_PAGE_SIZE;

    
    if (H5P_set_driver(plist, H5FD_CORE, &fa, NULL) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTSET, FAIL, "can't set core VFD as driver");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Pget_fapl_core(hid_t fapl_id, size_t *increment , bool *backing_store )
{
    H5P_genplist_t         *plist;               
    const H5FD_core_fapl_t *fa;                  
    herr_t                  ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    if (NULL == (plist = H5P_object_verify(fapl_id, H5P_FILE_ACCESS, true)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a file access property list");
    if (H5FD_CORE != H5P_peek_driver(plist))
        HGOTO_ERROR(H5E_PLIST, H5E_BADVALUE, FAIL, "incorrect VFL driver");
    if (NULL == (fa = (const H5FD_core_fapl_t *)H5P_peek_driver_info(plist)))
        HGOTO_ERROR(H5E_PLIST, H5E_BADVALUE, FAIL, "bad VFL driver info");

    if (increment)
        *increment = fa->increment;
    if (backing_store)
        *backing_store = fa->backing_store;

done:
    FUNC_LEAVE_API(ret_value)
} 

static void *
H5FD__core_fapl_get(H5FD_t *_file)
{
    H5FD_core_t      *file = (H5FD_core_t *)_file;
    H5FD_core_fapl_t *fa;               
    void             *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    if (NULL == (fa = (H5FD_core_fapl_t *)H5MM_calloc(sizeof(H5FD_core_fapl_t))))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed");

    fa->increment      = file->increment;
    fa->backing_store  = (bool)(file->fd >= 0);
    fa->write_tracking = file->write_tracking;
    fa->page_size      = file->bstore_page_size;

    
    ret_value = fa;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static H5FD_t *
H5FD__core_open(const char *name, unsigned flags, hid_t fapl_id, haddr_t maxaddr)
{
    int                     o_flags;
    H5FD_core_t            *file = NULL;
    const H5FD_core_fapl_t *fa   = NULL;
    H5P_genplist_t         *plist; 
#ifdef H5_HAVE_WIN32_API
    struct _BY_HANDLE_FILE_INFORMATION fileinfo;
#endif
    h5_stat_t              sb;
    int                    fd = -1;
    H5FD_file_image_info_t file_image_info;
    H5FD_t                *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    
    if (!name || !*name)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, NULL, "invalid file name");
    if (0 == maxaddr || HADDR_UNDEF == maxaddr)
        HGOTO_ERROR(H5E_ARGS, H5E_BADRANGE, NULL, "bogus maxaddr");
    if (CORE_ADDR_OVERFLOW(maxaddr))
        HGOTO_ERROR(H5E_ARGS, H5E_OVERFLOW, NULL, "maxaddr overflow");
    assert(H5P_DEFAULT != fapl_id);
    if (NULL == (plist = (H5P_genplist_t *)H5I_object(fapl_id)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "not a file access property list");
    if (NULL == (fa = (const H5FD_core_fapl_t *)H5P_peek_driver_info(plist)))
        fa = H5FD__core_get_default_config();

    
    o_flags = (H5F_ACC_RDWR & flags) ? O_RDWR : O_RDONLY;
    if (H5F_ACC_TRUNC & flags)
        o_flags |= O_TRUNC;
    if (H5F_ACC_CREAT & flags)
        o_flags |= O_CREAT;
    if (H5F_ACC_EXCL & flags)
        o_flags |= O_EXCL;

    
    if (H5P_peek(plist, H5F_ACS_FILE_IMAGE_INFO_NAME, &file_image_info) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, NULL, "can't get initial file image info");

    
    assert(((file_image_info.buffer != NULL) && (file_image_info.size > 0)) ||
           ((file_image_info.buffer == NULL) && (file_image_info.size == 0)));
    memset(&sb, 0, sizeof(sb));
    if ((file_image_info.buffer != NULL) && !(H5F_ACC_CREAT & flags)) {
        if ((fd = HDopen(name, o_flags, H5_POSIX_CREATE_MODE_RW)) >= 0) {
            HDclose(fd);
            HGOTO_ERROR(H5E_FILE, H5E_FILEEXISTS, NULL, "file already exists");
        }

        
        if (fa->backing_store) {
            if ((fd = HDopen(name, o_flags | O_CREAT, H5_POSIX_CREATE_MODE_RW)) < 0)
                HGOTO_ERROR(H5E_FILE, H5E_CANTOPENFILE, NULL, "unable to create file");
            memset(&sb, 0, sizeof(h5_stat_t));
            if (HDfstat(fd, &sb) < 0)
                HSYS_GOTO_ERROR(H5E_FILE, H5E_BADFILE, NULL, "unable to fstat file");
        } 
    }     
    
    else if (fa->backing_store || !(H5F_ACC_CREAT & flags)) {
        if ((fd = HDopen(name, o_flags, H5_POSIX_CREATE_MODE_RW)) < 0)
            HGOTO_ERROR(H5E_FILE, H5E_CANTOPENFILE, NULL, "unable to open file");
        memset(&sb, 0, sizeof(h5_stat_t));
        if (HDfstat(fd, &sb) < 0)
            HSYS_GOTO_ERROR(H5E_FILE, H5E_BADFILE, NULL, "unable to fstat file");
    } 

    
    if (NULL == (file = (H5FD_core_t *)H5MM_calloc(sizeof(H5FD_core_t))))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "unable to allocate file struct");
    file->fd = fd;
    if (name && *name)
        file->name = H5MM_xstrdup(name);

    
    file->increment = (fa->increment > 0) ? fa->increment : H5FD_CORE_INCREMENT;

    
    file->backing_store = fa->backing_store;

    
    file->fi_callbacks = file_image_info.callbacks;

    
    if (H5FD_ignore_disabled_file_locks_p != FAIL)
        
        file->ignore_disabled_file_locks = H5FD_ignore_disabled_file_locks_p;
    else {
        
        if (H5P_get(plist, H5F_ACS_IGNORE_DISABLED_FILE_LOCKS_NAME, &file->ignore_disabled_file_locks) < 0)
            HGOTO_ERROR(H5E_VFL, H5E_CANTGET, NULL, "can't get ignore disabled file locks property");
    }

    if (fd >= 0) {
        
#ifdef H5_HAVE_WIN32_API
        file->hFile = (HANDLE)_get_osfhandle(fd);
        if (INVALID_HANDLE_VALUE == file->hFile)
            HGOTO_ERROR(H5E_FILE, H5E_CANTOPENFILE, NULL, "unable to get Windows file handle");

        if (!GetFileInformationByHandle((HANDLE)file->hFile, &fileinfo))
            HGOTO_ERROR(H5E_FILE, H5E_CANTOPENFILE, NULL, "unable to get Windows file information");

        file->nFileIndexHigh       = fileinfo.nFileIndexHigh;
        file->nFileIndexLow        = fileinfo.nFileIndexLow;
        file->dwVolumeSerialNumber = fileinfo.dwVolumeSerialNumber;
#else  
        file->device = sb.st_dev;
        file->inode  = sb.st_ino;
#endif 
    }  

    
    if (!(H5F_ACC_CREAT & flags)) {
        size_t size;

        
        if (file_image_info.buffer && file_image_info.size > 0)
            size = file_image_info.size;
        else
            size = (size_t)sb.st_size;

        
        if (size) {
            
            if (file->fi_callbacks.image_malloc) {
                
                H5_BEFORE_USER_CB(NULL)
                    {
                        file->mem = file->fi_callbacks.image_malloc(size, H5FD_FILE_IMAGE_OP_FILE_OPEN,
                                                                    file->fi_callbacks.udata);
                    }
                H5_AFTER_USER_CB(NULL)
                if (NULL == file->mem)
                    HGOTO_ERROR(H5E_VFL, H5E_CANTALLOC, NULL, "image malloc callback failed");
            } 
            else {
                if (NULL == (file->mem = H5MM_malloc(size)))
                    HGOTO_ERROR(H5E_VFL, H5E_CANTALLOC, NULL, "unable to allocate memory block");
            } 

            
            file->eof = size;

            
            if (file_image_info.buffer && file_image_info.size > 0) {
                if (file->fi_callbacks.image_memcpy) {
                    void *tmp;

                    
                    H5_BEFORE_USER_CB(NULL)
                        {
                            tmp = file->fi_callbacks.image_memcpy(file->mem, file_image_info.buffer, size,
                                                                  H5FD_FILE_IMAGE_OP_FILE_OPEN,
                                                                  file->fi_callbacks.udata);
                        }
                    H5_AFTER_USER_CB(NULL)
                    if (file->mem != tmp)
                        HGOTO_ERROR(H5E_VFL, H5E_CANTCOPY, NULL, "image_memcpy callback failed");
                } 
                else
                    H5MM_memcpy(file->mem, file_image_info.buffer, size);
            } 
            
            else {
                

                uint8_t *mem    = file->mem; 
                HDoff_t  offset = 0;         

                while (size > 0) {
                    h5_posix_io_t     bytes_in   = 0;  
                    h5_posix_io_ret_t bytes_read = -1; 

                    
                    if (size > H5_POSIX_MAX_IO_BYTES)
                        bytes_in = H5_POSIX_MAX_IO_BYTES;
                    else
                        bytes_in = (h5_posix_io_t)size;

                    do {
#ifdef H5_HAVE_PREADWRITE
                        bytes_read = HDpread(file->fd, mem, bytes_in, offset);
                        if (bytes_read > 0)
                            offset += bytes_read;
#else
                        bytes_read = HDread(file->fd, mem, bytes_in);
#endif 
                    } while (-1 == bytes_read && EINTR == errno);

                    if (-1 == bytes_read) { 
                        int    myerrno = errno;
                        time_t mytime  = time(NULL);

                        offset = HDlseek(file->fd, 0, SEEK_CUR);

                        HGOTO_ERROR(
                            H5E_IO, H5E_READERROR, NULL,
                            "file read failed: time = %s, filename = '%s', file descriptor = %d, errno = %d, "
                            "error message = '%s', file->mem = %p, total read size = %llu, bytes this "
                            "sub-read = %llu, bytes actually read = %llu, offset = %llu",
                            ctime(&mytime), file->name, file->fd, myerrno, strerror(myerrno),
                            (void *)file->mem, (unsigned long long)size, (unsigned long long)bytes_in,
                            (unsigned long long)bytes_read, (unsigned long long)offset);
                    } 

                    assert(bytes_read >= 0);
                    assert((size_t)bytes_read <= size);

                    mem += bytes_read;
                    size -= (size_t)bytes_read;
                } 
            }     
        }         
    }             

    
    file->write_tracking   = fa->write_tracking;
    file->bstore_page_size = fa->page_size;

    
    file->dirty_list = NULL;
    if (fa->backing_store) {
        bool use_write_tracking = false; 

        
        
        use_write_tracking = (true == fa->write_tracking) 
                             && !(o_flags & O_RDONLY)     
                             && (file->bstore_page_size != 0); 

        
        if (use_write_tracking)
            if (NULL == (file->dirty_list = H5SL_create(H5SL_TYPE_HADDR, NULL)))
                HGOTO_ERROR(H5E_SLIST, H5E_CANTCREATE, NULL, "can't create core vfd dirty region list");
    } 

    
    ret_value = (H5FD_t *)file;

done:
    if (!ret_value && file) {
        if (file->fd >= 0)
            HDclose(file->fd);
        H5MM_xfree(file->name);
        H5MM_xfree(file->mem);
        H5MM_xfree(file);
    } 

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5FD__core_close(H5FD_t *_file)
{
    H5FD_core_t *file      = (H5FD_core_t *)_file;
    herr_t       ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    if (H5FD__core_flush(_file, (hid_t)-1, true) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTFLUSH, FAIL, "unable to flush core vfd backing store");

    
    if (file->dirty_list)
        if (H5FD__core_destroy_dirty_list(file) != SUCCEED)
            HGOTO_ERROR(H5E_VFL, H5E_CANTFREE, FAIL, "unable to free core vfd dirty region list");

    
    if (file->fd >= 0)
        HDclose(file->fd);
    if (file->name)
        H5MM_xfree(file->name);
    if (file->mem) {
        
        if (file->fi_callbacks.image_free) {
            
            H5_BEFORE_USER_CB(FAIL)
                {
                    ret_value = file->fi_callbacks.image_free(file->mem, H5FD_FILE_IMAGE_OP_FILE_CLOSE,
                                                              file->fi_callbacks.udata);
                }
            H5_AFTER_USER_CB(FAIL)
            if (ret_value < 0)
                HGOTO_ERROR(H5E_VFL, H5E_CANTFREE, FAIL, "image_free callback failed");
        } 
        else
            H5MM_xfree(file->mem);
    } 
    memset(file, 0, sizeof(H5FD_core_t));
    H5MM_xfree(file);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static int
H5FD__core_cmp(const H5FD_t *_f1, const H5FD_t *_f2)
{
    const H5FD_core_t *f1        = (const H5FD_core_t *)_f1;
    const H5FD_core_t *f2        = (const H5FD_core_t *)_f2;
    int                ret_value = 0;

    FUNC_ENTER_PACKAGE_NOERR

    if (f1->fd >= 0 && f2->fd >= 0) {
        
#ifdef H5_HAVE_WIN32_API
        if (f1->dwVolumeSerialNumber < f2->dwVolumeSerialNumber)
            HGOTO_DONE(-1);
        if (f1->dwVolumeSerialNumber > f2->dwVolumeSerialNumber)
            HGOTO_DONE(1);

        if (f1->nFileIndexHigh < f2->nFileIndexHigh)
            HGOTO_DONE(-1);
        if (f1->nFileIndexHigh > f2->nFileIndexHigh)
            HGOTO_DONE(1);

        if (f1->nFileIndexLow < f2->nFileIndexLow)
            HGOTO_DONE(-1);
        if (f1->nFileIndexLow > f2->nFileIndexLow)
            HGOTO_DONE(1);

#else
        if (f1->device < f2->device)
            HGOTO_DONE(-1);
        if (f1->device > f2->device)
            HGOTO_DONE(1);

        if (f1->inode < f2->inode)
            HGOTO_DONE(-1);
        if (f1->inode > f2->inode)
            HGOTO_DONE(1);

#endif 
    }  
    else {
        if (NULL == f1->name && NULL == f2->name) {
            if (f1 < f2)
                HGOTO_DONE(-1);
            if (f1 > f2)
                HGOTO_DONE(1);
            HGOTO_DONE(0);
        } 

        if (NULL == f1->name)
            HGOTO_DONE(-1);
        if (NULL == f2->name)
            HGOTO_DONE(1);

        ret_value = strcmp(f1->name, f2->name);
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5FD__core_query(const H5FD_t *_file, unsigned long *flags )
{
    const H5FD_core_t *file = (const H5FD_core_t *)_file;

    FUNC_ENTER_PACKAGE_NOERR

    
    
    if(flags) {
        *flags = 0;
        *flags |= H5FD_FEAT_AGGREGATE_METADATA;             
        *flags |= H5FD_FEAT_ACCUMULATE_METADATA;            
        *flags |= H5FD_FEAT_DATA_SIEVE;                     
        *flags |= H5FD_FEAT_AGGREGATE_SMALLDATA;            
        *flags |= H5FD_FEAT_ALLOW_FILE_IMAGE;               
        *flags |= H5FD_FEAT_CAN_USE_FILE_IMAGE_CALLBACKS;   

        
        if(file && file->fd >= 0 && file->backing_store) {
            *flags |= H5FD_FEAT_POSIX_COMPAT_HANDLE;        
            *flags |= H5FD_FEAT_DEFAULT_VFD_COMPATIBLE;     
        }
    } 
    

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static haddr_t
H5FD__core_get_eoa(const H5FD_t *_file, H5FD_mem_t H5_ATTR_UNUSED type)
{
    const H5FD_core_t *file = (const H5FD_core_t *)_file;

    FUNC_ENTER_PACKAGE_NOERR

    FUNC_LEAVE_NOAPI(file->eoa)
} 

static herr_t
H5FD__core_set_eoa(H5FD_t *_file, H5FD_mem_t H5_ATTR_UNUSED type, haddr_t addr)
{
    H5FD_core_t *file      = (H5FD_core_t *)_file;
    herr_t       ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    if (CORE_ADDR_OVERFLOW(addr))
        HGOTO_ERROR(H5E_ARGS, H5E_OVERFLOW, FAIL, "address overflow");

    file->eoa = addr;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static haddr_t
H5FD__core_get_eof(const H5FD_t *_file, H5FD_mem_t H5_ATTR_UNUSED type)
{
    const H5FD_core_t *file = (const H5FD_core_t *)_file;

    FUNC_ENTER_PACKAGE_NOERR

    FUNC_LEAVE_NOAPI(file->eof)
} 

static herr_t
H5FD__core_get_handle(H5FD_t *_file, hid_t fapl, void **file_handle)
{
    H5FD_core_t *file      = (H5FD_core_t *)_file; 
    herr_t       ret_value = SUCCEED;              

    FUNC_ENTER_PACKAGE

    
    if (!file_handle)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "file handle not valid");

    
    if (H5P_FILE_ACCESS_DEFAULT != fapl && H5P_DEFAULT != fapl) {
        H5P_genplist_t *plist; 

        
        if (NULL == (plist = (H5P_genplist_t *)H5I_object(fapl)))
            HGOTO_ERROR(H5E_VFL, H5E_BADTYPE, FAIL, "not a file access property list");

        
        if (H5P_exist_plist(plist, H5F_ACS_WANT_POSIX_FD_NAME) > 0) {
            bool want_posix_fd; 

            
            if (H5P_get(plist, H5F_ACS_WANT_POSIX_FD_NAME, &want_posix_fd) < 0)
                HGOTO_ERROR(H5E_VFL, H5E_CANTGET, FAIL, "can't get property of retrieving file descriptor");

            
            if (want_posix_fd)
                *file_handle = &(file->fd);
            else
                *file_handle = &(file->mem);
        } 
        else
            *file_handle = &(file->mem);
    } 
    else
        *file_handle = &(file->mem);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5FD__core_read(H5FD_t *_file, H5FD_mem_t H5_ATTR_UNUSED type, hid_t H5_ATTR_UNUSED dxpl_id, haddr_t addr,
                size_t size, void *buf )
{
    H5FD_core_t *file      = (H5FD_core_t *)_file;
    herr_t       ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    assert(file && file->pub.cls);
    assert(buf);

    
    if (HADDR_UNDEF == addr)
        HGOTO_ERROR(H5E_IO, H5E_OVERFLOW, FAIL, "file address overflowed");
    if (CORE_REGION_OVERFLOW(addr, size))
        HGOTO_ERROR(H5E_IO, H5E_OVERFLOW, FAIL, "file address overflowed");

    
    if (addr < file->eof) {
        size_t nbytes;
#ifndef NDEBUG
        hsize_t temp_nbytes;

        temp_nbytes = file->eof - addr;
        H5_CHECK_OVERFLOW(temp_nbytes, hsize_t, size_t);
        nbytes = MIN(size, (size_t)temp_nbytes);
#else  
        nbytes = MIN(size, (size_t)(file->eof - addr));
#endif 

        H5MM_memcpy(buf, file->mem + addr, nbytes);
        size -= nbytes;
        addr += nbytes;
        buf = (char *)buf + nbytes;
    }

    
    if (size > 0)
        memset(buf, 0, size);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5FD__core_write(H5FD_t *_file, H5FD_mem_t H5_ATTR_UNUSED type, hid_t H5_ATTR_UNUSED dxpl_id, haddr_t addr,
                 size_t size, const void *buf)
{
    H5FD_core_t *file      = (H5FD_core_t *)_file;
    herr_t       ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    assert(file && file->pub.cls);
    assert(buf);

    
    if (CORE_REGION_OVERFLOW(addr, size))
        HGOTO_ERROR(H5E_IO, H5E_OVERFLOW, FAIL, "file address overflowed");

    
    if (addr + size > file->eof) {
        unsigned char *x;
        size_t         new_eof;

        
        H5_CHECKED_ASSIGN(new_eof, size_t, file->increment * ((addr + size) / file->increment), hsize_t);
        if ((addr + size) % file->increment)
            new_eof += file->increment;

        
        if (file->fi_callbacks.image_realloc) {
            
            H5_BEFORE_USER_CB(FAIL)
                {
                    x = file->fi_callbacks.image_realloc(file->mem, new_eof, H5FD_FILE_IMAGE_OP_FILE_RESIZE,
                                                         file->fi_callbacks.udata);
                }
            H5_AFTER_USER_CB(FAIL)
            if (NULL == x)
                HGOTO_ERROR(H5E_VFL, H5E_CANTALLOC, FAIL,
                            "unable to allocate memory block of %llu bytes with callback",
                            (unsigned long long)new_eof);
        } 
        else {
            if (NULL == (x = H5MM_realloc(file->mem, new_eof)))
                HGOTO_ERROR(H5E_VFL, H5E_CANTALLOC, FAIL, "unable to allocate memory block of %llu bytes",
                            (unsigned long long)new_eof);
        } 

        memset(x + file->eof, 0, (size_t)(new_eof - file->eof));
        file->mem = x;

        file->eof = new_eof;
    } 

    
    if (file->dirty_list) {
        haddr_t start = addr;
        haddr_t end   = addr + (haddr_t)size - 1;

        if (H5FD__core_add_dirty_region(file, start, end) != SUCCEED)
            HGOTO_ERROR(
                H5E_VFL, H5E_CANTINSERT, FAIL,
                "unable to add core VFD dirty region during write call - addresses: start=%llu end=%llu",
                (unsigned long long)start, (unsigned long long)end);
    }

    
    H5MM_memcpy(file->mem + addr, buf, size);

    
    file->dirty = true;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5FD__core_flush(H5FD_t *_file, hid_t H5_ATTR_UNUSED dxpl_id, bool H5_ATTR_UNUSED closing)
{
    H5FD_core_t *file      = (H5FD_core_t *)_file;
    herr_t       ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    if (file->dirty && file->fd >= 0 && file->backing_store) {

        
        if (file->dirty_list) {
            H5FD_core_region_t *item = NULL;
            size_t              size;

            while (NULL != (item = (H5FD_core_region_t *)H5SL_remove_first(file->dirty_list))) {

                
                if (item->start < file->eof) {
                    if (item->end >= file->eof)
                        item->end = file->eof - 1;

                    size = (size_t)((item->end - item->start) + 1);

                    if (H5FD__core_write_to_bstore(file, item->start, size) != SUCCEED)
                        HGOTO_ERROR(H5E_VFL, H5E_WRITEERROR, FAIL, "unable to write to backing store");
                } 

                item = H5FL_FREE(H5FD_core_region_t, item);
            } 

        } 
        
        else {
            if (H5FD__core_write_to_bstore(file, (haddr_t)0, (size_t)file->eof) != SUCCEED)
                HGOTO_ERROR(H5E_VFL, H5E_WRITEERROR, FAIL, "unable to write to backing store");
        } 

        file->dirty = false;
    }

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5FD__core_truncate(H5FD_t *_file, hid_t H5_ATTR_UNUSED dxpl_id, bool closing)
{
    H5FD_core_t *file = (H5FD_core_t *)_file;
    size_t       new_eof;             
    herr_t       ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    assert(file);

    
    if (!closing || file->backing_store) {
        if (closing) 
            new_eof = file->eoa;
        else { 
            
            H5_CHECKED_ASSIGN(new_eof, size_t, file->increment * (file->eoa / file->increment), hsize_t);
            if (file->eoa % file->increment)
                new_eof += file->increment;
        } 

        
        if (!H5_addr_eq(file->eof, (haddr_t)new_eof)) {
            unsigned char *x; 

            
            if (file->fi_callbacks.image_realloc) {
                
                H5_BEFORE_USER_CB(FAIL)
                    {
                        x = file->fi_callbacks.image_realloc(
                            file->mem, new_eof, H5FD_FILE_IMAGE_OP_FILE_RESIZE, file->fi_callbacks.udata);
                    }
                H5_AFTER_USER_CB(FAIL)
                if (NULL == x)
                    HGOTO_ERROR(H5E_VFL, H5E_CANTALLOC, FAIL,
                                "unable to allocate memory block with callback");
            } 
            else {
                if (NULL == (x = H5MM_realloc(file->mem, new_eof)))
                    HGOTO_ERROR(H5E_VFL, H5E_CANTALLOC, FAIL, "unable to allocate memory block");
            } 

            if (file->eof < new_eof)
                memset(x + file->eof, 0, (size_t)(new_eof - file->eof));
            file->mem = x;

            
            if (closing && (file->fd >= 0) && file->backing_store) {
#ifdef H5_HAVE_WIN32_API
                LARGE_INTEGER li;       
                DWORD         dwPtrLow; 
                DWORD dwError;          
                BOOL  bError;           

                
                li.QuadPart = (LONGLONG)file->eoa;

                
                dwPtrLow = SetFilePointer(file->hFile, li.LowPart, &li.HighPart, FILE_BEGIN);
                if (INVALID_SET_FILE_POINTER == dwPtrLow) {
                    dwError = GetLastError();
                    if (dwError != NO_ERROR)
                        HGOTO_ERROR(H5E_FILE, H5E_FILEOPEN, FAIL, "unable to set file pointer");
                }

                bError = SetEndOfFile(file->hFile);
                if (0 == bError)
                    HGOTO_ERROR(H5E_IO, H5E_SEEKERROR, FAIL, "unable to extend file properly");
#else  
                if (-1 == HDftruncate(file->fd, (HDoff_t)new_eof))
                    HSYS_GOTO_ERROR(H5E_IO, H5E_SEEKERROR, FAIL, "unable to extend file properly");
#endif 

            } 

            
            file->eof = new_eof;
        } 
    }     

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5FD__core_lock(H5FD_t *_file, bool rw)
{
    H5FD_core_t *file = (H5FD_core_t *)_file; 
    int          lock_flags;                  
    herr_t       ret_value = SUCCEED;         

    FUNC_ENTER_PACKAGE

    assert(file);

    
    if (file->fd >= 0) {
        
        lock_flags = rw ? LOCK_EX : LOCK_SH;

        
        if (HDflock(file->fd, lock_flags | LOCK_NB) < 0) {
            if (file->ignore_disabled_file_locks && ENOSYS == errno) {
                
                errno = 0;
            }
            else
                HSYS_GOTO_ERROR(H5E_FILE, H5E_BADFILE, FAIL, "unable to lock file");
        } 
    }     

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5FD__core_unlock(H5FD_t *_file)
{
    H5FD_core_t *file      = (H5FD_core_t *)_file; 
    herr_t       ret_value = SUCCEED;              

    FUNC_ENTER_PACKAGE

    assert(file);

    if (file->fd >= 0)
        if (HDflock(file->fd, LOCK_UN) < 0) {
            if (file->ignore_disabled_file_locks && ENOSYS == errno) {
                
                errno = 0;
            }
            else
                HSYS_GOTO_ERROR(H5E_FILE, H5E_BADFILE, FAIL, "unable to unlock file");
        }

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5FD__core_delete(const char *filename, hid_t fapl_id)
{
    const H5FD_core_fapl_t *fa = NULL;
    H5P_genplist_t         *plist;               
    herr_t                  ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    assert(filename);

    if (NULL == (plist = (H5P_genplist_t *)H5I_object(fapl_id)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a file access property list");
    if (NULL == (fa = (const H5FD_core_fapl_t *)H5P_peek_driver_info(plist)))
        fa = H5FD__core_get_default_config();

    if (fa->backing_store)
        if (HDremove(filename) < 0)
            HSYS_GOTO_ERROR(H5E_VFL, H5E_CANTDELETEFILE, FAIL, "unable to delete file");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 
