% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/model-class.R
\name{set_priors}
\alias{set_priors}
\title{Set up Prior Distributions}
\usage{
set_priors(p_prior, l_prior = NULL, s_prior = NULL)
}
\arguments{
\item{p_prior}{A list specifying prior distributions for subject-level
parameters (or the likelihood function for the population-level model).
Each element in the list should contain:
  \itemize{
    \item \code{p0}: The first parameter of the distribution.
    \item \code{p1}: The second parameter of the distribution.
    \item \code{lower}: The lower bound of the distribution.
    \item \code{upper}: The upper bound of the distribution.
    \item \code{dist}: A numeric code representing the distribution type.
    \item \code{log_p}: Logical indicating whether to compute in log space.
  }}

\item{l_prior}{Optional list specifying prior distributions for
population-level location parameters. Should have the same structure as
\code{p_prior}. Defaults to \code{NULL}.}

\item{s_prior}{Optional list specifying prior distributions for
population-level scale parameters. Should have the same structure as
\code{p_prior}. Defaults to \code{NULL}.}
}
\value{
An S4 object of class \code{"prior"} with the following slots:
\itemize{
  \item \code{nparameter}: Integer; number of parameters in the joint prior.
  \item \code{pnames}: Character vector of parameter names.
  \item \code{p_prior}: List containing prior specifications for
subject-level parameters.
  \item \code{h_prior}: List containing merged prior specifications for
\code{l_prior} and \code{s_prior}.
}
}
\description{
Configures a set of joint prior distributions for:
\itemize{
  \item Subject-level parameters (\code{p_prior}), which also serve as the
likelihood function in population-level models.
  \item Population-level location parameters (\code{l_prior}).
  \item Population-level scale parameters (\code{s_prior}).
}
}
\details{
This function performs the following:
\itemize{
  \item Validates the structure of all prior specifications.
  \item Ensures required distribution parameters are present and bounds are
 valid.
  \item Merges \code{l_prior} and \code{s_prior} into a single \code{h_prior}
 using \code{.merge_prior}.
  \item Returns a structured \code{prior} object for use in model fitting
and simulation.
}

The argument \code{log_p} should be set to \code{TRUE} for density
evaluation and \code{FALSE} when generating samples (e.g., for initial
parameter values).
}
\examples{
if (requireNamespace("ggdmcModel", quietly = TRUE)) {
  BuildModel <- getFromNamespace("BuildModel", "ggdmcModel")

  model <- BuildModel(
    p_map = list(
      A = "1", B = "1", t0 = "1", mean_v = "M", sd_v = "M",
      st0 = "1"
    ),
    match_map = list(M = list(s1 = "r1", s2 = "r2")),
    factors = list(S = c("s1", "s2")),
    constants = c(sd_v.false = 1, st0 = 0),
    accumulators = c("r1", "r2"),
    type = "lba"
  )

  ####################################
  # priors for subject-level modelling
  ####################################
  p0 <- rep(0, model@npar)
  names(p0) <- model@pnames
  p_prior <- BuildPrior(
    p0 = p0,
    p1 = rep(10, model@npar),
    lower = rep(0, model@npar),
    upper = rep(NA, model@npar),
    dist = rep("unif", model@npar),
    log_p = rep(TRUE, model@npar)
  )
  sub_priors <- set_priors(p_prior = p_prior)

  ####################################
  # priors for hierarchical modelling
  ####################################
  p0 <- runif(model@npar)
  names(p0) <- model@pnames
  model_likelihood <- BuildPrior(
    p0 = p0,
    p1 = rep(10, model@npar),
    lower = rep(0, model@npar),
    upper = rep(NA, model@npar),
    dist = rep("tnorm", model@npar),
    log_p = rep(TRUE, model@npar)
  )

  p0 <- rep(0, model@npar)
  names(p0) <- model@pnames
  l_prior <- BuildPrior(
    p0 = p0,
    p1 = rep(10, model@npar),
    lower = rep(0, model@npar),
    upper = rep(NA, model@npar),
    dist = rep("unif", model@npar),
    log_p = rep(TRUE, model@npar)
  )
  s_prior <- BuildPrior(
    p0 = p0,
    p1 = rep(10, model@npar),
    lower = rep(NA, model@npar),
    upper = rep(NA, model@npar),
    dist = rep("unif", model@npar),
    log_p = rep(TRUE, model@npar)
  )

  pop_priors <- set_priors(
    p_prior = model_likelihood,
    l_prior = l_prior, s_prior = s_prior
  )
}
}
