#include <RcppArmadillo.h>
#include <algorithm>
#include <cmath>
using namespace Rcpp;
using arma::vec;
using arma::mat;
// [[Rcpp::depends(RcppArmadillo)]]
//' The Distribution of Univariate Factor Copula Model
//'
//' Density, distribution function, quantile function and random generation
//' for the distribution of univariate factor copula model with rate parameter
//' equal to `lambda`.
//'
//' @param w A numeric value representing the spatial process.
//' @param lambda A numeric value representing rate parameter \eqn{\lambda}.
//' @param u a numeric vector of probabilities, with values in the interval from 0 to 1, at which the quantile function is to be computed.
//' @param tol a scalar indicating the desired level of numerical accuracy for the algorithm; default is 1e-9.
//' @param niter a scalar indicating the maximum number of iterations.
//' @param n an integer value specifying the number of samples to generate
//' @return
//' `dfcm` gives a numeric value representing the density of the factor copula model evaluated at \code{w},
//' `pfcm` gives a numeric value representing the CDF evaluated at \code{w},
//' `qfcm` gives the quantile function (QF) of the factor copula model.
//' and `rfcm` generate a numeric vector of random samples drawn.
//' @details
//' The univariate eFCM distribution is \deqn{F(w;\lambda)=\Phi(w)-exp(\lambda^2/2-\lambda w)\Phi(w-\lambda),}
//' where \eqn{\lambda} is the rate parameter.
//' @export
// [[Rcpp::export(rng = false)]]
double pfcm(const double& w, const double& lambda){
   return(R::pnorm(w, 0, 1, TRUE, FALSE) - exp(R::pnorm(w - lambda, 0, 1, TRUE, TRUE) + pow(lambda, 2)/2 - lambda*w));
}
//' @examples
//' pfcm(w = 1, lambda = 0.5)
//' dfcm(w = 1, lambda = 0.5)
//' qfcm(u = 0.5, lambda = 0.5)
//' rfcm(n = 1000, lambda = 0.5)
//'
//' @export
//' @rdname pfcm
// [[Rcpp::export(rng = false)]]
double dfcm(const double &w, const double &lambda){
  return(R::dnorm(w, 0, 1, FALSE) -
        R::dnorm(w - lambda, 0, 1, FALSE) * exp(pow(lambda, 2)/2 - lambda*w) +
        R::pnorm(w - lambda, 0, 1, TRUE, FALSE) * exp(pow(lambda, 2)/2 - lambda * w) * lambda);
}

double dummy(const double &w, const double &u, const double &lambda){
  return(pfcm(w, lambda) - u);
}

//' @export
//' @rdname pfcm
// [[Rcpp::export(rng = false)]]
double qfcm(const double& u, const double& lambda, const double& tol = 1e-8, const int& niter = 1000){
  double x0 = R::qnorm(u, 0, 1, TRUE, FALSE) - 0.5;
  double x1 = R::qnorm(u, 0, 1, TRUE, FALSE);
  double x2 = 0;
  try{
    for(int i = 0; i <= niter; i += 1){
      double temp0 = dummy(x0, u, lambda);
      double temp1 = dummy(x1, u, lambda);
      x2 = x1 - temp1*(x1 - x0)/(temp1 - temp0);
      double temp2 = dummy(x2, u, lambda);
      double aux2 = abs(NumericVector::create(temp2))[0];
      if(aux2 < tol){
        return(x2);
      }else{
        x0 = x1;
        x1 = x2;
      }
    }
  }catch(const char* error){
    x2 = 1000000000;
  }
  return(x2);
}
//' @export
//' @rdname pfcm
// [[Rcpp::export(rng = false)]]
NumericVector rfcm(const int& n, const double& lambda){
  NumericVector x(n);
  for(int i = 0; i < n; i+=1){
    x[i] = qfcm(R::runif(0, 1), lambda);
  }
  return x;
}

 arma::uvec setdiff(const arma::uvec& x, const arma::uvec& y) {
   std::vector<int> a = arma::conv_to< std::vector<int> >::from(arma::sort(x));
   std::vector<int> b = arma::conv_to< std::vector<int> >::from(arma::sort(y));
   std::vector<int> out;
   std::set_difference(a.begin(), a.end(), b.begin(), b.end(),
                       std::inserter(out, out.end()));
   return arma::conv_to<arma::uvec>::from(out);
 }


 arma::vec qF1(const arma::vec& u, const arma::vec& theta){
   double lambda = theta[0];
   arma::vec qf(u.size());
   for(int i = 0; i < u.size(); i+=1){
     qf(i) = qfcm(u(i), lambda);
   }
   return(qf);
 }

 // [[Rcpp::export(rng = false)]]
 double ecGeneral(const IntegerVector& ranks, const IntegerVector& d, const NumericVector& x, const int& n, const int& k, const double& result) {
   int i, j, count;
   double sum;

   sum = 0;
   for (i = 0; i < n; i++) {
     count = 0;
     for (j = i; j < ((d[0] - 1) * n + i + 1); j = j + n) {
       if (ranks[j] >= (n + 0.5 - k * x[count])) {
         sum += 1;
         break;
       }
       count = count + 1;
     }
   }
   return sum/k;
 }

// [[Rcpp::export]]
 List mi_condMVN (const arma::vec& mean, const arma::mat& sigma, const int& given_ind){
   int n = mean.size();
   arma::vec d_ind = arma::regspace(0, n-1);
   d_ind.shed_row(given_ind);
   arma::uvec d_ind_u = arma::conv_to<arma::uvec>::from(d_ind);

   arma::mat B = sigma.cols(d_ind_u);
   B = B.rows(d_ind_u);
   arma::mat C = sigma.rows(d_ind_u);
   C = C.col(given_ind);
   double D = sigma(given_ind, given_ind);
   arma::mat cVar = B - C / D * C.t();
   List L = List::create(_["C"] = C, _["condVar"] = cVar);
   return L;
 }

 arma::mat Matern(const arma::mat& dist_mat, const double& range, const double& smooth){
   Environment fields = Environment::namespace_env("fields");
   Function f = fields["Matern"];
   NumericMatrix sigma = f(_["d"] = dist_mat, _["range"] = range, _["smoothness"] = smooth);
   arma::mat sigma0 = as<arma::mat>(sigma);
   return(sigma0);
 }

// [[Rcpp::export]]
double pmvnorm(const arma::vec&  bound,
               const arma::vec&  mean,
               const arma::mat&  cormat,
               double abseps = 1e-5,
               double releps = 1e-5,
               int    maxpts = 25000)
{
  Environment mnormt = Environment::namespace_env("mnormt");
  Function    sadmvn = mnormt["sadmvn"];

  NumericVector upper = wrap(bound);
  NumericVector lower(upper.size(), R_NegInf);
  NumericVector rmean = wrap(mean);

  NumericVector res = sadmvn(_["lower"]  = lower,
                             _["upper"]  = upper,
                             _["mean"]   = rmean,
                             _["varcov"] = wrap(cormat),
                             _["abseps"] = abseps,
                             _["releps"] = releps,
                             _["maxpts"] = maxpts);
  return res[0];
}
static double pj_cpp(const arma::vec& w,
                     int        j,          // 0-based
                     const arma::mat& sigma,
                     double     lambda,
                     double abseps,
                     double releps,
                     int    maxpts)
{
  int d = w.n_elem;

  // σ_{·j}
  arma::vec sigj = sigma.col(j);
  sigj.shed_row(j);                       //  d-1

  // w without j
  arma::vec w_omit = w;
  w_omit.shed_row(j);

  // w0 = ...
  arma::vec first  = arma::join_vert(w_omit - sigj * w(j), vec{0});
  arma::vec second = arma::join_vert((w(j) - lambda) * (1 - sigj),
                               vec{lambda - w(j)});
  arma::vec upper  = first - second;            //  d

  //  s
  arma::mat sigma_omit = sigma;
  sigma_omit.shed_row(j);
  sigma_omit.shed_col(j);

  arma::mat s11 = sigma_omit - sigj * sigj.t()
    + (1 - sigj) * (1 - sigj).t();

  arma::vec  s12 = sigj - 1;                    // column
  arma::mat  s   = arma::join_cols(
    arma::join_rows(s11, s12            ),
    arma::join_rows(s12.t(), mat(1,1,arma::fill::ones))
  );                           // d × d
  arma::vec mu0 = arma::zeros<vec>(d);
  return pmvnorm(upper, mu0, s, abseps, releps, maxpts);
}

// [[Rcpp::export]]
double pmfcm_rcpp(const arma::vec& w,
                  double lambda, double delta,
                  const arma::mat& dist,
                  double smooth = 0.5,
                  double abseps = 1e-5,
                  double releps = 1e-5,
                  int    maxpts = 25000)
{
  int d = w.n_elem;
  arma::mat sigma = Matern(dist, /*range =*/ delta, /*nu =*/ smooth);
  sigma.diag().ones();

  arma::vec mu0(d, arma::fill::zeros);
  double p1 = pmvnorm(w, mu0, sigma, abseps, releps, maxpts);

  double p3 = 0;
  for (int j = 0; j < d; ++j)
    p3 += std::exp(lambda*lambda/2 - lambda*w(j) +
      std::log(pj_cpp(w, j, sigma, lambda,
                      abseps, releps, maxpts)));

  double p = std::max(0.0, std::min(1.0, p1 - p3));
  return p;
}

// [[Rcpp::export]]
arma::mat rmfcm_rcpp(double lambda,
                     double delta,
                     const arma::mat& dist,
                     double nu    = 0.5,
                     int    n     = 500000)
{
  int d = dist.n_rows;
  arma::mat M = Matern(dist, /*range =*/ delta, /*nu =*/ nu);
  arma::mat L = arma::chol(M);
  arma::mat Z = arma::randn<arma::mat>(d, n);
  arma::mat X = L * Z;      // d × n

  arma::vec V(n);
  for (int i = 0; i < n; ++i) {
    V[i] = R::rexp(lambda);
  }
  arma::mat sim = X.t();    // n × d
  sim.each_col() += V;
  return sim;
}

// [[Rcpp::export]]
double chi_fcm(double  h,
               double  u,
               double  lambda,
               double  delta,
               double  nu = 0.5,
               double  abseps = 1e-5,
               double  releps = 1e-5,
               int     maxpts = 25000)
{
  if (h <= 0)           stop("`h` must be positive.");
  if (u <= 0 || u >= 1) stop("`u` must be in (0, 1).");

  double z_single = qfcm(u, lambda);
  arma::vec z(2); z.fill(z_single);

  arma::mat dist(2, 2, arma::fill::ones);
  dist *= h;
  dist.diag().zeros();

  double cu = pmfcm_rcpp(z, lambda, delta, dist, nu,
                         abseps, releps, maxpts);

  return (1 - 2*u + cu) / (1 - u);
}

// [[Rcpp::export]]
NumericMatrix chiv(const NumericVector& u,
                       const NumericMatrix& b,
                       double h,
                       double nu = 0.5,
                       double abseps = 1e-5,
                       double releps = 1e-5,
                       int    maxpts = 25000)
{
  const int n = u.size();
  const int k = b.nrow();
  NumericMatrix out(n, k);

  for (int i = 0; i < n; ++i) {
    for (int j = 0; j < k; ++j) {
      double lambda  = b(j, 0);
      double delta = b(j, 1);

      out(i, j) = chi_fcm(h, u[i], lambda, delta, nu, abseps, releps, maxpts);
    }
  }
  return out;
}

// [[Rcpp::export]]
 double log_Cn(const arma::vec& theta, const arma::vec& u_star, const arma::mat& dist_mat, const double& nu) {
   double lambda = theta[0];
   double range = theta[1];
   double smooth = nu;
   arma::vec z = qF1(u_star, theta);
   z.replace(R_NaN, 10000);
   arma::mat sigma = Matern(dist_mat, range, smooth);
   int n = sigma.n_cols;
   arma::vec mean = arma::zeros(n);
   arma::vec probs(n);
   arma::vec unos = arma::ones(n-1);
   arma::mat sigma0(n, n);
   sigma0(n-1, n-1) = 1;

   for(int i=0; i< n; i += 1){
     List temp = mi_condMVN(mean, sigma, i);
     arma::vec C = temp["C"];
     arma::mat condVar = temp["condVar"];
     arma::mat D = unos - C;
     arma::mat Dt = D.t();
     sigma0(arma::span(0, n-2), arma::span(0, n-2)) = condVar + D * Dt;
     sigma0(arma::span(0, n-2), n-1) = -D;
     sigma0(n-1, arma::span(0, n-2)) = -Dt;
     arma::vec z2 = z;
     z2.shed_row(i);
     arma::vec upper = z2 - z(i) * unos + D * lambda;
     upper.resize(n);
     upper(n-1) = z(i)-lambda;
     double p1 = pow(lambda, 2)/2 - lambda*z(i);
     probs[i] = exp(p1 + log(pmvnorm(upper, arma::zeros(upper.size()), sigma0)));
   }
   double value = pmvnorm(z, arma::zeros(z.size()), sigma) - sum(probs);
   return log(value);
 }

// [[Rcpp::export]]
 double log_fn(const arma::rowvec& zi, const double& lambda, const arma::mat& sigma_inv, const arma::mat& C_inv){
   arma::rowvec zn = zi;
   arma::uvec idx = arma::find_finite(zn);
   arma::mat sigma_inv_c = sigma_inv(idx, idx);
   arma::mat C_inv_c = C_inv(idx, idx);
   if(!zn.is_finite()){
     zn = arma::conv_to<arma::rowvec>::from(zn.elem(idx));
   }
   int n = zn.size();
   arma::mat m1_1 = zn * C_inv_c.t();
   double m1 = arma::as_scalar(zn * sigma_inv_c * zn.as_col());
   double m2 = arma::as_scalar(arma::ones(1, n) * sigma_inv_c * zn.as_col());
   double m3 = arma::accu(sigma_inv_c);
   double m1_star = (m2 - lambda)/sqrt(m3);
   arma::mat C = C_inv_c.i();
   double out = log(lambda) - (n-1)* log(2*acos(-1))*0.5 -
     sum(log(C.diag())) - 0.5 * log(m3) +
     (pow(m1_star, 2) - m1)*0.5 +
     R::pnorm(m1_star, 0, 1, TRUE, TRUE);
   return(out);
 }

// [[Rcpp::export]]
 double log_f1(const double& zij, const double& lambda){
   if(is_na(NumericVector::create(zij))[0]){
     return(R_NaN);
   }else{
     double out = log(lambda) + pow(lambda, 2)/2 - lambda * zij + R::pnorm(zij-lambda, 0, 1, TRUE, TRUE);
     return(out);
   }
 }

// [[Rcpp::export]]
 arma::vec dF1(const arma::vec& z, const arma::vec& theta, const bool& islog = false){
   double lambda = theta(0);
   arma::vec logdF1(z.size());
   if(z.size() > 0){
     for(int i =0; i<z.size();i+=1){
       logdF1(i) = log(lambda) - lambda * z(i) + 0.5*pow(lambda, 2) + R::pnorm(z(i)-lambda, 0, 1, TRUE, TRUE);
     }
   }else{
     logdF1 = log(lambda) - lambda * z(0) + 0.5*pow(lambda,2) +R::pnorm(z(0)-lambda, 0, 1, TRUE, TRUE);
   }
   if(islog){
     return(logdF1);
   }else{
     return(exp(logdF1));
   }
 }

// [[Rcpp::export]]
 double dFI(const arma::vec& z, const arma::uvec& I, const arma::mat& dist_mat, const arma::vec& theta, const double& nu, const bool& islog = false){
   double lambda = theta[0];
   double range = theta[1];
   double smooth = nu;
   arma::mat sigma = Matern(dist_mat, range, smooth);
   int k = I.size();
   arma::mat sigmaII = sigma(I, I);
   arma::mat C = arma::chol(sigmaII).t();
   arma::mat Cinv = arma::inv(C);
   arma::mat sigmaIIinv = Cinv.t() * Cinv;
   arma::mat CinvzI = Cinv * z(I);
   double m1 = arma::as_scalar(CinvzI.t() * CinvzI);
   double m2 = arma::as_scalar(arma::ones<arma::rowvec>(k) * sigmaIIinv * z(I));
   double m3 = arma::accu(sigmaIIinv);
   double m31 = 1/m3;

   double logdFI = 0;
   if(k == sigma.n_cols){
     logdFI = log(lambda) - 0.5 * log(m3) - (k-1)*0.5*log(2*arma::datum::pi) -
       sum(log(C.diag())) - 0.5*(m1-pow(m2-lambda, 2)/m3) + R::pnorm(0, -1*(m2 - lambda) *m31, sqrt(m31), true, true);
     if(islog){
       return(logdFI);
     }else{
       return(exp(logdFI));
     }
   }else{
     arma::uvec nI = setdiff(arma::regspace<arma::uvec>(0, z.size()-1), I);
     arma::vec zi = z(nI);

     arma::vec z_I(nI.size()+1);z_I(0) = 0;
     arma::uvec nidx = arma::regspace<arma::uvec>(1, nI.size());
     z_I(nidx) = zi - sigma(nI, I) * sigmaIIinv * z(I);
     arma::vec mu_I(nI.size()+1);mu_I(0) = -1;
     arma::vec cvec = sigma(nI, I) * sigmaIIinv * arma::ones<arma::colvec>(k) - 1;
     mu_I(nidx) = -cvec;
     mu_I = (m2 - lambda) * m31 * mu_I;

     arma::mat sigma_I(nI.size() + 1, nI.size() + 1);
     sigma_I(0,0) = 1;
     sigma_I(nidx, arma::zeros<arma::uvec>(1)) = cvec.as_col();
     sigma_I(arma::zeros<arma::uvec>(1), nidx) = cvec.as_row();
     sigma_I(nidx, nidx) = sigma(nI, nI) - sigma(nI,I) * sigmaIIinv * sigma(I,nI) +
       cvec.as_col() * cvec.as_row();
     sigma_I = sigma_I * m31;

     if (!sigma_I.is_symmetric()) {
       arma::mat sigma_I_trans = sigma_I.t();
       for (int i = 0; i < sigma_I.n_rows; i += 1) {
         for (int j = 0; j < sigma_I.n_cols; j+= 1) {
           sigma_I(i, j) = std::min(sigma_I(i, j), sigma_I_trans(i, j));
         }
       }
     }
     arma::vec upper = z_I - mu_I;
     double logdFI = log(lambda) - 0.5 * log(m3) - (k-1)*0.5*log(2*arma::datum::pi) -
       sum(log(C.diag())) - 0.5*(m1-pow(m2-lambda, 2)*m31) + log(pmvnorm(z_I, mu_I, sigma_I));
     if(islog){
       return(logdFI);
     }else{
       return(exp(logdFI));
     }
   }
 }

// [[Rcpp::export]]
 double dCI(const arma::vec& u, const arma::uvec& I, const arma::mat& dist_mat, const arma::vec& theta, const double& nu, const bool& islog = false){
   if(I.size() > 0){
     arma::vec z = qF1(u, theta);
     double logdCI = dFI(z, I, dist_mat, theta, nu, true) - sum(dF1(z(I), theta, true));
     if(islog){
       return(logdCI);
     }else{
       return(exp(logdCI));
     }
   }else{
     return 0;
   }
 }

// [[Rcpp::export]]
 double log_partialCn(const arma::vec& theta, const arma::mat& data_u, const arma::vec& u_star, const arma::mat& dist_mat, const double& nu){
   arma::vec value(data_u.n_rows);
   for(int i = 0; i < data_u.n_rows; i += 1){
     arma::vec u = data_u.row(i).as_col();
     arma::uvec idx = arma::find_finite(u);
     if(!idx.empty()){
       arma::vec u_star1 = u_star(idx);
       arma::vec u1 = u(idx);
       arma::uvec I = find(u1 - u_star1 > 0);
       arma::uvec nI = find(u1 - u_star1 <= 0);
       u1(nI) = u_star1(nI);
       if(I.n_elem == 0){
         value(i) = 0;
       }else{
         value(i) = dCI(u1, I, dist_mat(idx, idx), theta, nu, true);
       }
     }else{
       arma::uvec I = arma::find(u - u_star > 0);
       arma::uvec nI = arma::find(u - u_star <= 0);
       u(nI) = u_star(nI);
       value(i) = dCI(u, I, dist_mat, theta, nu, true);
     }
   }
   return(sum(value));
 }

// [[Rcpp::export]]
 double Log_Lik_Non(const arma::vec& theta, const arma::mat& data_u, const arma::mat& dist_mat, const double& nu){
   double lambda = theta[0];
   double range = theta[1];
   double smooth = nu;
   arma::mat z(data_u.n_rows, data_u.n_cols);
   for(int i = 0; i < data_u.n_rows; i += 1){
     for(int j = 0; j < data_u.n_cols;j += 1){
       z(i,j) = qfcm(data_u(i,j), lambda);
     }
   }

   arma::mat sigma = Matern(dist_mat, range, smooth);
   arma::mat C = arma::chol(sigma).t();
   arma::mat C_inv = inv(C);
   arma::mat sigma_inv = C_inv.t() * C_inv;
   arma::vec lfn(z.n_rows);
   arma::mat lf1(z.n_rows, z.n_cols);

   for(int i = 0; i < z.n_rows; i += 1){
     lfn(i) = log_fn(z.row(i).as_row(), lambda, sigma_inv, C_inv);
     for(int j = 0; j < z.n_cols; j += 1){
       lf1(i,j) = log_f1(z(i,j), lambda);
     }
   }

   double value = sum(lfn) - arma::accu(lf1.elem(arma::find_finite(lf1)));
   return(value);
 }
 // [[Rcpp::export(rng = false)]]
 arma::vec u_np(const arma::vec& Xvec){
   arma::uvec xnan = arma::find_finite(Xvec);
   int n = xnan.size()+1;
   Environment base = Environment::namespace_env("base");
   Function rank = base["rank"];

   arma::vec y = as<arma::vec>(rank(_["x"] = Xvec,  _["na.last"] = "keep", _["ties.method"] = "average"));
   return y/n;
 }
// [[Rcpp::export(rng = false)]]
double model_likelihood(const arma::vec& theta0, const arma::mat& data_u, const arma::mat& coord, const double& thres, const double& nu, const bool& censorL){
     arma::vec theta = arma::exp(theta0);
     Environment fields = Environment::namespace_env("fields");
     Function f = fields["rdist.earth"];
     arma::mat dist_mat = as<arma::mat>(f(coord, Rcpp::_["miles"] = false));
     arma::vec u_star(data_u.n_cols);
     for(int i = 0; i < data_u.n_cols; i += 1){
       arma::vec y = data_u.col(i);
       Environment stats = Environment::namespace_env("stats");
       Function quantileR = stats["quantile"];

       NumericVector q = quantileR(wrap(y), thres, _["na.rm"] = true);
       u_star[i] = q[0];
     }
     int N1 = data_u.n_rows;
     int n1 = data_u.n_cols;
     arma::uvec G1 = arma::zeros<arma::uvec>(N1);
     arma::uvec G2 = arma::zeros<arma::uvec>(N1);
     arma::uvec G3 = arma::zeros<arma::uvec>(N1);
     G1(0) = 1; G2(0) = 1; G3(0) = 1;
     int count1 = 0;
     int count2 = 0;
     int count3 = 0;
     for(int i = 0; i < N1; i++){
       arma::uvec bvec = arma::find(data_u.row(i) > u_star.as_row());
       if(bvec.size() == 0){
         G1(count1) = i;
         count1 += 1;
       }else{
         if(bvec.size() == n1){
           G2(count2) = i;
           count2 += 1;
         }else{
           G3(count3) = i;
           count3 += 1;
         }
       }
     }
     G1 = G1.head(count1);
     G2 = G2.head(count2);
     G3 = G3.head(count3);

     double value1 = 0, value2 = 0, value3 = 0;
     if(G1.size() > 0){


       if(censorL){
         if(!data_u.rows(G1).has_nan()){
           value1 = -log_Cn(theta, u_star, dist_mat, nu) * G1.size();
         }else{
           arma::mat data_u_G1 = data_u.rows(G1);
           arma::uvec u_na = arma::find_nonfinite(data_u_G1);
           arma::uvec u_row_na = u_na - floor(u_na/data_u_G1.n_rows) * data_u_G1.n_rows;
           u_row_na.elem(arma::find_unique(u_row_na));
           std::sort(u_row_na.begin(), u_row_na.end());
           NumericVector  valueG1(u_row_na.size());
           for(int i : u_row_na){
             arma::uvec idx = arma::find_finite(data_u_G1.row(i));
             valueG1[i]  = - log_Cn(theta, u_star(idx), dist_mat(idx, idx), nu);
           }

           u_row_na = setdiff(arma::regspace<arma::uvec>(0, G1.size()-1), u_row_na);
           if(u_row_na.size() > 0){
             value1 = sum(log(valueG1))- u_row_na.size() * log_Cn(theta, u_star, dist_mat, nu);
           }else{
             value1 = sum(valueG1);
           }
         }
       }
       else{
         value1 = -Log_Lik_Non(theta, data_u.rows(G1), dist_mat, nu);
       }
     }
     if(G2.size() > 0){
       value2 = -Log_Lik_Non(theta, data_u.rows(G2), dist_mat, nu);
     }
     if(G3.size() > 0){
       if(censorL){
         value3 = -log_partialCn(theta, data_u.rows(G3), u_star, dist_mat, nu);
       }else{
         value3 = -Log_Lik_Non(theta, data_u.rows(G3), dist_mat, nu);
       }
     }

     double v = value1 + value2  + value3;
     return v;

 }

// [[Rcpp::export(rng = false)]]

 List fit_initial(arma::mat coord, arma::mat data, const arma::uvec& neigh){
   coord = coord.rows(neigh);
   data = data.cols(neigh);
   data.replace(0, R_NaN);
   arma::uvec idx(data.n_rows);
   int count = 0;
   for(int i = 0; i < data.n_rows; i += 1){
     arma::uvec rowi = arma::find_finite(data.row(i));
     if(rowi.size() > 1){
       idx(count) = i;
       count += 1;
     }
   }
   idx = idx.head(count);
   data = data.rows(idx);

   for(int i = 0; i < data.n_cols; i += 1){
     data.col(i) = u_np(data.col(i));
   }
   List temp = List::create(_["data_u"] = data,
                            _["coord"] = coord);
   return(temp);
 }


